/*
 *	linux/mm/mmap.c
 *
 * Written by obz.
 *
 * NO_MM
 *  Copyright (c) 2001 Lineo, Inc. David McCullough <davidm@lineo.com>
 *  Copyright (c) 2000-2001 D Jeff Dionne <jeff@uClinux.org> ref uClinux 2.0
 */
#include <linux/slab.h>
#include <linux/shm.h>
#include <linux/mman.h>
#include <linux/pagemap.h>
#include <linux/swap.h>
#include <linux/swapctl.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/personality.h>

#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <asm/brecis/BrecisMemCtl.h>

/* Flush data and/or instruction caches */
/* extern int icache_size, dcache_size; */
extern int ic_lsize, dc_lsize;
extern unsigned int scache_size, sc_lsize;
#include <asm/mips32_cache.h>

#define WORKING			/* if defined speed up switch_mm */

/* Debugging messages */
/* #define DEBUG */
/* #define DEBUG_MM_RB */
/* Specific file was loaded, M4DEBUG set in binfmt_elf.c */
/* #define M4_DEBUG */
#ifdef DEBUG
  #ifdef M4_DEBUG
    extern int M4DEBUG;		/* in linux/fs/binfmt_elf.c */
    #define DBG_NOMM(a...)	{if (M4DEBUG != 0) {printk(a);}}
  #else					/* !M4_DEBUG */
    #define DBG_NOMM(a...)	printk(a)
  #endif
#else
  #define DBG_NOMM(a...)
#endif


/* ------------------------------------------------------------------------ */
/* sys_brk() for the most part doesn't need the global kernel
   lock, except when an application is doing something nasty
   like trying to un-brk an area that has already been mapped
   to a regular file.  in this case, the unmapping will need
   to invoke file system routines that need the global lock.  */

asmlinkage unsigned long sys_brk(unsigned long brk)
{
  struct mm_struct *mm = current->mm;

/* NOTDONEYET: this does no checking against running stack. */
  if (brk < mm->start_brk || brk > mm->end_brk)
    return mm->brk;

  if (mm->brk == brk)
    return mm->brk;

/* Always allow shrinking brk.  */
  if (brk <= mm->brk) {
    mm->brk = brk;
    return brk;
  }
/* Ok, looks good - let it rip.  */
  return mm->brk = brk;
}

/* ------------------------------------------------------------------------ */
/* Combine the mmap "prot" and "flags" argument into one "vm_flags" used
   internally. Essentially, translate the "PROT_xxx" and "MAP_xxx" bits
   into "VM_xxx".  */
static inline unsigned long calc_vm_flags(unsigned long prot, unsigned long flags)
{
#define _trans(x,bit1,bit2) \
((bit1==bit2)?(x&bit1):(x&bit1)?bit2:0)

  unsigned long prot_bits,
		flag_bits;
  prot_bits =
      _trans(prot, PROT_READ, VM_READ) |
      _trans(prot, PROT_WRITE, VM_WRITE) |
      _trans(prot, PROT_EXEC, VM_EXEC);
  flag_bits =
      _trans(flags, MAP_GROWSDOWN, VM_GROWSDOWN) |
      _trans(flags, MAP_DENYWRITE, VM_DENYWRITE) |
      _trans(flags, MAP_EXECUTABLE, VM_EXECUTABLE);
  return prot_bits | flag_bits;
#undef _trans
}

/* ------------------------------------------------------------------------ */
unsigned long get_unmapped_area(struct file *file, unsigned long addr, unsigned long len, unsigned long pgoff, unsigned long flags)
{
  if (flags & MAP_FIXED) {
    if (addr > TASK_SIZE - len) {
      return -EINVAL;
    }
    if (addr & ~PAGE_MASK) {
      return -EINVAL;
    }
    return addr;
  }
  if (file && file->f_op && file->f_op->get_unmapped_area) {
    return file->f_op->get_unmapped_area(file, addr, len, pgoff, flags);
  }
  return 0;
}

/* ------------------------------------------------------------------------ */
#if 0
#ifdef DEBUG
void	show_process_blocks(struct mm_struct * mm)
{
  struct mm_tblock_struct *tmp;

  if (mm == NULL) {
/* 	DBG_NOMM("mm==NULL\n"); */
    return;
  }
  DBG_NOMM("mm=%p users=%d, count=%d ", mm, mm->mm_users.counter, mm->mm_count.counter);
  DBG_NOMM("blocks:\n");

  tmp = mm->tblock;
  while (tmp) {
    if (tmp->save_kblock != NULL && current->mm == mm) {
      DBG_NOMM("show_process_blocks, save_kblock != NULL for running process, error!\n");
    }
    DBG_NOMM("\t%p: %p {%p}", tmp, tmp->rblock, tmp->save_kblock);
    if (tmp->rblock) {
      DBG_NOMM(" (%d @%p #%d {%p})", ksize(tmp->rblock->kblock), tmp->rblock->kblock, tmp->rblock->refcount, tmp->rblock->loaded_tblock);
      if (tmp->rblock->loaded_tblock) {
	DBG_NOMM(" save_kblock=%p", tmp->rblock->loaded_tblock->save_kblock);
      }
    }
/*     if (tmp->next) { */
/*       DBG_NOMM(" ->\n"); */
/*     } else { */
/*       DBG_NOMM(".\n"); */
/*     } */
    DBG_NOMM("\n");
    tmp = tmp->next;
  }
}
#endif	/* DEBUG */
#endif	/* 0 */

/* ------------------------------------------------------------------------ */
extern unsigned long askedalloc;
extern unsigned long realalloc;
/* ------------------------------------------------------------------------ */
unsigned long do_mmap_pgoff(
	struct file * file,
	unsigned long addr,
	unsigned long len,
	unsigned long prot,
	unsigned long flags,
	unsigned long pgoff)
{
  void 		*result;
  struct mm_tblock_struct *tblock;
  unsigned int vm_flags;

/* Get the NO_MM specific checks done first */
  if ((flags & MAP_SHARED) && (prot & PROT_WRITE) && (file)) {
    printk("ERROR, MAP_SHARED not supported (cannot write mappings to disk)\n");
    return -EINVAL;
  }
/* This is not relevant.  Since nothing is private, go ahead and pretend. */
/*   if ((prot & PROT_WRITE) && (flags & MAP_PRIVATE)) { */
/*     printk("Private writable mappings not supported\n"); */
/*     return -EINVAL; */
/*   } */
  
  /*
   *	now all the standard checks
   */
  if (file && (!file->f_op || !file->f_op->mmap))
	  return -ENODEV;

  if (PAGE_ALIGN(len) == 0)
	  return addr;

  if (len > TASK_SIZE)
	  return -EINVAL;

  /* offset overflow? */
  if ((pgoff + (len >> PAGE_SHIFT)) < pgoff)
	  return -EINVAL;

#if 0
  /* Too many mappings? */
  if (mm->map_count > MAX_MAP_COUNT)
	  return -ENOMEM;
#endif

  /* Obtain the address to map to. we verify (or select) it and ensure
   * that it represents a valid section of the address space.
   */
  addr = get_unmapped_area(file, addr, len, pgoff, flags);
  if (addr & ~PAGE_MASK)
	  return addr;

  /* Do simple checking here so the lower-level routines won't have
   * to. we assume access permissions have been handled by the open
   * of the memory object, so we don't do any here.
   */
  vm_flags = calc_vm_flags(prot,flags) /* | mm->def_flags */ | VM_MAYREAD | VM_MAYWRITE | VM_MAYEXEC;

#if 0
  /* mlock MCL_FUTURE? */
  if (vm_flags & VM_LOCKED) {
	  unsigned long locked = mm->locked_vm << PAGE_SHIFT;
	  locked += len;
	  if (locked > current->rlim[RLIMIT_MEMLOCK].rlim_cur)
		  return -EAGAIN;
  }
#endif

  /*
   * determine the object being mapped and call the appropriate
   * specific mapper. 
   */

  if (file) {
    struct vm_area_struct vma;
    int             error;


    if (!file->f_op)
      return -ENODEV;

    vma.vm_start = addr;
    vma.vm_end = addr + len;
    vma.vm_flags = vm_flags;
    vma.vm_offset = pgoff << PAGE_SHIFT;

/* Then try full mmap routine, which might return a RAM pointer,
   or do something truly complicated. */
    if (file->f_op->mmap) {
      error = file->f_op->mmap(file, &vma);
/* DBG_NOMM("mmap mmap returned %d /%x\n", error, vma.vm_start); */
      if (!error)
	return vma.vm_start;
      else if (error != -ENOSYS)
	return error;
    } else
      return -ENODEV;			/* No mapping operations defined */

/* An ENOSYS error indicates that mmap isn't possible (as opposed to
   tried but failed) so we'll fall through to the copy. */
  }
  tblock = (struct mm_tblock_struct *) kmalloc(sizeof(struct mm_tblock_struct), GFP_KERNEL);
/* #ifdef DEBUG */
/*   DBG_NOMM("do_mmap_pgoff kmalloc tblock %p\n", tblock); */
/* #endif */
  if (!tblock) {
#ifdef DEBUG
    printk("ERROR, Allocation of tblock for %lu byte allocation from process %d failed\n", len, current->pid);
    show_buffers();
    show_free_areas();
#endif
    return -ENOMEM;
  }
  tblock->save_kblock = NULL;		/* no saved data. */
  tblock->rblock = (struct mm_rblock_struct *) kmalloc(sizeof(struct mm_rblock_struct), GFP_KERNEL);
/* #ifdef DEBUG */
/*   DBG_NOMM("do_mmap_pgoff kmalloc rblock %p\n", tblock->rblock); */
/* #endif */

  if (!tblock->rblock) {
#ifdef DEBUG
    printk("ERROR, Allocation of rblock for %lu byte allocation from process %d failed\n", len, current->pid);
    show_buffers();
    show_free_areas();
#endif
    kfree(tblock);
    return -ENOMEM;
  }
  result = kmalloc(len, GFP_KERNEL);
  if (!result) {
#ifdef DEBUG
    printk("ERROR, Allocation of length %lu from process %d failed\n", len, current->pid);
    show_buffers();
    show_free_areas();
#endif
    kfree(tblock->rblock);
    kfree(tblock);
    return -ENOMEM;
  }
  tblock->rblock->loaded_tblock = tblock;
/* #ifdef DEBUG */
/*   printk("do_mmap_pgoff, kmalloc kblock=%p\n", result); */
/* #endif */
  tblock->rblock->refcount = 1;
  tblock->rblock->kblock = result;
  tblock->rblock->size = len;

  realalloc += ksize(result);
  askedalloc += len;

#ifdef WARN_ON_SLACK
  if ((len + WARN_ON_SLACK) <= ksize(result))
	  printk("ERROR, Allocation of %lu bytes from process %d has %lu bytes of slack\n", len, current->pid, ksize(result) - len);
#endif	/* WARN_ON_SLACK */

  if (file) {
    int             error;
    mm_segment_t    old_fs = get_fs();
    set_fs(KERNEL_DS);
    error = file->f_op->read(file, (char *) result, len, &file->f_pos);
    set_fs(old_fs);
    if (error < 0) {
      kfree(result);
      kfree(tblock->rblock);
      kfree(tblock);
      return error;
    }
    if (error < len)
      memset(result + error, '\0', len - error);
  } else {
    memset(result, '\0', len);
  }
        
  realalloc += ksize(tblock);
  askedalloc += sizeof(struct mm_tblock_struct);

  realalloc += ksize(tblock->rblock);
  askedalloc += sizeof(struct mm_rblock_struct);

  tblock->next = current->mm->tblock;
  current->mm->tblock = tblock;

/* #ifdef DEBUG */
/*   DBG_NOMM("do_mmap_pgoff Process %d\n", current->pid); */
/*   show_process_blocks(current->mm); */
/* #endif */

  blast_hit_idcache((unsigned long)result, ksize(result));
  return ((unsigned long) (KERN_2_USER(result)));
}

/* ------------------------------------------------------------------------ */
int             do_munmap(struct mm_struct * mm, unsigned long addr, size_t len)
{
  struct mm_tblock_struct *tblock,
			 **tmp;
  unsigned long kaddr;

  if (((unsigned long)addr & (unsigned long)0x80000000) != 0) {
    printk("ERROR, do_munmap of address that is in kernel! (0x%lx)\n", addr);
  }

  kaddr = USER_2_KERN(addr);
/* #ifdef DEBUG */
/*   DBG_NOMM("do_munmap:\n"); */
/* #endif */

  tmp = &mm->tblock;		/* dummy head */
  tblock = mm->tblock;
  while ((tblock != NULL) && (tblock->rblock) && (tblock->rblock->kblock != (void *) kaddr)) { 
    tmp = &tblock->next;
    tblock = tblock->next;
  }

  if (!tblock) {
    printk("ERROR, munmap of non-mmaped memory by process %d (%s): %p\n", current->pid, current->comm, (void *) addr);
    return -EINVAL;
  }
  blast_hit_idcache(addr, ksize((void *)kaddr));
  if (tblock->rblock) {
    if (!--tblock->rblock->refcount) {
#ifndef WORKING
      if (tblock->save_kblock != NULL) {
	DBG_NOMM("ERROR, do_munmap, save_kblock != NULL??\n");
      }
#endif
      if (tblock->rblock->kblock) {
	realalloc -= ksize(tblock->rblock->kblock);
	askedalloc -= tblock->rblock->size;
/* #ifdef DEBUG */
/* 	DBG_NOMM("do_munmap kfree block %p\n", tblock->rblock->kblock); */
/* #endif */
	kfree(tblock->rblock->kblock);
#ifdef DEBUG
	tblock->rblock->kblock = NULL;
#endif
      }
      realalloc -= ksize(tblock->rblock);
      askedalloc -= sizeof(struct mm_rblock_struct);
/* #ifdef DEBUG */
/*       DBG_NOMM("do_munmap kfree rblock %p\n", tblock->rblock); */
/* #endif */
      kfree(tblock->rblock);
#ifdef DEBUG
      tblock->rblock = NULL;
#endif
    } else {
      tblock->rblock->loaded_tblock = NULL;
    }
  }
  *tmp = tblock->next;
#ifdef DEBUG
  tblock->next = NULL;
#endif
  realalloc -= ksize(tblock);
  askedalloc -= sizeof(struct mm_tblock_struct);
/* #ifdef DEBUG */
/*   DBG_NOMM("do_munmap kfree tblock %p\n", tblock); */
/* #endif */
  kfree(tblock);

/* #ifdef DEBUG */
/*   DBG_NOMM("do_munmap Process %d\n", current->pid); */
/*   show_process_blocks(current->mm); */
/* #endif */

  return (0);				/* ok */
}

/* ------------------------------------------------------------------------ */
/* Release all mmaps. */

void            exit_mmap(struct mm_struct * mm)
{
  struct mm_tblock_struct *tmp;

  if (!mm)
    return;

#ifdef DEBUG
  DBG_NOMM("exit_mmap:\n");
/*   show_process_blocks(mm); */
#endif

  while ((tmp = mm->tblock) != NULL) {
#ifndef WORKING
    if (tmp->save_kblock != NULL) {
      DBG_NOMM("exit_mmap, save_kblock != NULL??\n");
    }
    if (!tmp->rblock) {
      DBG_NOMM("exit_mmap, rblock == NULL??\n");
    } else if (tmp->rblock->loaded_tblock != tmp) {
      DBG_NOMM("exit_mmap, loaded_tblock != me??\n");
    }
#endif
    if (tmp->rblock) {
      if (!--tmp->rblock->refcount) {
	if (tmp->rblock->kblock) {
	  realalloc -= ksize(tmp->rblock->kblock);
	  askedalloc -= tmp->rblock->size;
/* #ifdef DEBUG */
/* 	  DBG_NOMM("exit_mmap kfree kblock %p\n", tmp->rblock->kblock); */
/* #endif */
	  blast_hit_idcache((unsigned long)(KERN_2_USER(tmp->rblock->kblock)), tmp->rblock->size);
	  kfree(tmp->rblock->kblock);
#ifdef DEBUG
	  tmp->rblock->kblock = NULL;	/* no longer loaded */
#endif
	}
	realalloc -= ksize(tmp->rblock);
	askedalloc -= sizeof(struct mm_rblock_struct);
/* #ifdef DEBUG */
/* 	DBG_NOMM("exit_mmap kfree rblock %p\n", tmp->rblock); */
/* #endif */
	kfree(tmp->rblock);
#ifdef DEBUG
	tmp->rblock = NULL;
#endif
      } else {
	tmp->rblock->loaded_tblock = NULL;
      }
    }
    mm->tblock = tmp->next;
    realalloc -= ksize(tmp);
    askedalloc -= sizeof(struct mm_tblock_struct);
/* #ifdef DEBUG */
/*     DBG_NOMM("exit_mmap kfree tblock %p\n", tmp); */
/* #endif */
    kfree(tmp);
  }
/* #ifdef DEBUG */
/*   DBG_NOMM("exit_mmap Process %d mm:\n", current->pid); */
/*   show_process_blocks(current->mm); */
/* #endif */
}

/* ------------------------------------------------------------------------ */
asmlinkage long sys_munmap(unsigned long addr, size_t len)
{
  int             ret;
  struct mm_struct *mm = current->mm;

  down_write(&mm->mmap_sem);
  ret = do_munmap(mm, addr, len);
  up_write(&mm->mmap_sem);
  return ret;
}

/* ------------------------------------------------------------------------ */
/* This routine will save and restore pid's mm structure. */
/* I.e. if fork has been done and both parent and child are
   still running, must put all our mmap entries into place.
   This is everything, stack included -- becareful of stack. */

void            switch_mm(struct mm_struct * prevmm, struct mm_struct * newmm,
			  struct task_struct * nexttask, unsigned cpu)
{
  struct mm_tblock_struct *tmp;

#ifdef DEBUG
  DBG_NOMM("switch_mm %d->%d\n", current->pid, nexttask->pid);
/*   DBG_NOMM("prevmm:\n"); */
/*   show_process_blocks(prevmm); */
/*   DBG_NOMM("newmm:\n"); */
/*   show_process_blocks(newmm); */
#endif
  tmp = newmm->tblock;
/* #ifdef DEBUG */
/*   DBG_NOMM("\t%p: ", tmp); */
/* #endif */
  while (tmp != NULL) {
/* if current kblock != this process, save old one, move new one in. */
/* #ifdef DEBUG */
/*     DBG_NOMM("%p {%p}", tmp->rblock, tmp->save_kblock); */
/*     if (tmp->rblock) { */
/*       DBG_NOMM(" loaded_tblock:%p", tmp->rblock->loaded_tblock); */
/*       if (tmp->rblock->loaded_tblock) { */
/* 	DBG_NOMM(" save_kblock:%p", tmp->rblock->loaded_tblock->save_kblock); */
/*       } */
/*     } */
/*     DBG_NOMM("\n"); */
/* #endif */
    if (tmp->save_kblock != NULL) {
#ifndef WORKING
      if (tmp->rblock->loaded_tblock == tmp) {
	DBG_NOMM("ERROR, switch_mm, save_kblock not NULL, but tmp->rblock is me?\n");
      }
      if (tmp->rblock->loaded_tblock != NULL && tmp->rblock->loaded_tblock->save_kblock != NULL) {
	DBG_NOMM("ERROR, switch_mm, can't save kblock, it isn't NULL?\n");
      }
#endif
      if (tmp->rblock->loaded_tblock != NULL) {
	tmp->rblock->loaded_tblock->save_kblock = kmalloc(tmp->rblock->size, GFP_KERNEL);
	if (tmp->rblock->loaded_tblock->save_kblock == NULL) {
	  panic("switch_mm, out of memory -- tough failure area to fix.\n");
	}
/* #ifdef DEBUG */
/* 	printk("switch_mm, kmalloc save_kblock=%p\n", tmp->rblock->loaded_tblock->save_kblock); */
/* #endif */
	memcpy(tmp->rblock->loaded_tblock->save_kblock, tmp->rblock->kblock, tmp->rblock->size);
      }
      memcpy(tmp->rblock->kblock, tmp->save_kblock, tmp->rblock->size);
      blast_hit_idcache((unsigned long)tmp->rblock->kblock, tmp->rblock->size);
/* #ifdef DEBUG */
/*       DBG_NOMM("switch_mm kfree save_kblock %p\n", tmp->save_kblock); */
/* #endif */
      kfree(tmp->save_kblock);
      tmp->save_kblock = NULL;
      tmp->rblock->loaded_tblock = tmp;
#ifndef WORKING
    } else {
      if (tmp->rblock->loaded_tblock != tmp) {
	DBG_NOMM("ERROR, switch_mm, save_kblock NULL, but loaded_tblock not me?\n");
      }
#endif
    }
    tmp = tmp->next;
/* #ifdef DEBUG */
/*     DBG_NOMM("\t%p: ", tmp); */
/* #endif */
  }
/* #ifdef DEBUG */
/*   DBG_NOMM("\n"); */
/* #endif */
}

/* ------------------------------------------------------------------------ */
