/*
** Definitions for Brecis security engine.  User API portion.
**
** Copyright 2001 Brecis Communications
**
*/

#ifndef BRECIS_MSP_SEC_H
#define BRECIS_MSP_SEC_H

#define SECDEV "/dev/secdev"



/*
** This file should eventually become
** linux/include/asm-mipsnommu/brecis/brecis_sec.h, and be included
** in other files with "#include <asm/brecis/brecis_sec.h>".
*/

/*
**  The security engine's DMA descriptors allow scatter/gather.
**  But each source and destination address pair shares a length.
**  If you have source and destination buffer chains that have
**  differing lengths, you must "merge" the two into operations
**  with like lengths.
**
**  I.E. if you have two soucre buffers of 100 and 300 bytes, and
**  two destination buffers of 200 bytes each, you must split them up
**  like this:
**
**    src            dest          len
**  -------------------------------------
**    srcA           destA         100
**    srcB           destA + 100   100
**    srcB + 100     destB         200
**
*/

/*
** The IOCTL number for IOCTL calls
*/
#define MSP_SEC_CTL ('B' << 8 | 0x0001)


/*
**  The first buffer is specified in the sec_ioctl struct.
**
**  Fields marked with RESV are reserved for driver use, and should not
**  be touched by other software.
*/

typedef struct sec_ioctl
{
	/*
	** The begining of this structure maps directly to the
	** hardware interface.
	*/

	unsigned int     nda ;           /* RESV next descriptor address */

	unsigned int     buflen ;        /* size of first buffer */
	unsigned char    *srcaddr ;      /* source address first buffer */
	unsigned char    *dstaddr ;      /* destination address first buffer */
	unsigned int     mode ;          /* DES or HASH mode register */
	unsigned int     id ;	         /* caller's reference, not touched */

	/* ----- initial vector used by CBC and CFB DES */
	unsigned int	 ivhigh ;
	unsigned int     ivlow ;

	/* ----- DES keys and HASH chainvars share space   */
	/*       DES  uses chainvar[0-1]                   */
	/*       3DES uses chainvar[0-5]                   */
	/*       MD5  uses chainvar[0-3]                   */
	/*       SHA1 uses chainvar[0-4]                   */
	/*	 HASH with pad uses chainvar[5] for length */
	/*             if chaining is used                 */
	
	unsigned int     chainvar[6] ;


	/*
	** The hardware descriptor ends here; remaining fields are
	** driver related.
	*/

	unsigned int     type ;		 /* DES or HASH */
	unsigned int     options ;       /* options, such as padding */
	unsigned int     cbufcount ;     /* number of continue buffers */
	struct sec_cbuf  *cbuflist ;     /* array of continue buffer areas */
	unsigned int     status ;        /* return of status information */

} sec_ioctl ;

/*
**  any additional "continuation buffers" are specified with this
**  sec_cbuf structure:
*/

typedef struct sec_cbuf
{
	/*
	** This structure maps directly to the hardware interface.
	** Fields marked with RESV are reserved for driver use, and
	** should not be touched by other software.
	*/

	unsigned int     nda ;		/* RESV next descriptor address */

	unsigned int     buflen ;
	unsigned char    *srcaddr ;
	/*
	** For hash, dstaddr in sec_ioctl struct is where result will
	** be placed; all others are ignored.  (this is accomplished
	** by driver "magic"; expect that this field will be
	** overwritten with the dstaddr from the sec_ioctl struct)
	*/
	unsigned char    *dstaddr ;
} sec_cbuf ;


/*
** values for the ioctl TYPE field
*/

#define             MSP_SEC_DES     0
#define             MSP_SEC_HASH    1
#define             MSP_SEC_HMAC    2	/* to do HMAC hash, point ioctl */
					/* struct at key, and use cbufs */
					/* for the contents */
#define             MSP_SEC_HMAC2   3	/* HMAC with pre-calculated */
					/* IPAD and OPAD hashes. */
					/* ioctl.srcaddr points to */
					/* IPAD precalc; */
					/* ioctl.dstaddr points to */
					/* OPAD precalc; cbufs for */
					/* all data. */


/*
** values for the ioctl options field
*/

#define             MSP_SEC_DO_PADDING 0x0008
	
/*--------------------------------------------------------------------------+
|
| DES Mode Control Register:
|
|     Bit 0    =  Operation to perform with Key1
|                 0 = Encryption
|                 1 = Decryption
|
|     Bit 1    =  Operation to perform with Key2
|                 0 = Encryption
|                 1 = Decryption
|
|     Bit 2    =  Operation to perform with Key3
|                 0 = Encryption
|                 1 = Decryption
|
|     Bit 3    =  Tripple DES Enable
|                 0 = Single DES
|                 1 = Tripple DES
|
|     Bit 5:4  =  DES Mode bit
|                 0 = ECB (Electronic Codebook Mode)
|                 1 = CBC (Cipher Block Chaining Mode)
|                 2 = CFB (Cipher FeedBack Mode)
|                 3 = PT (Pass Thru Mode)
|
|     Bit 6    =  Decrypt mode (used for CBC and CFB modes only)
|                 0 = Mode is Encrypt
|                 1 = Mode is Decrypt
|
+--------------------------------------------------------------------------*/
#define DMC_K1_ENC	0x00		/* Encryption with Key1            */
#define	DMC_K1_DEC	0x01            /* Decryption with Key1            */
#define DMC_K2_ENC	0x00		/* Encryption with Key2            */
#define	DMC_K2_DEC	0x02            /* Decryption with Key2            */
#define DMC_K3_ENC	0x00		/* Encryption with Key3            */
#define	DMC_K3_DEC	0x04            /* Decryption with Key3            */
#define	DMC_3DES	0x08            /* Tripple DES Enable              */

#define DMC_MOD		0x30		/* DES Mode Mask                   */
#define DMC_MOD_ECB	0x00		/* ECB - Electronic Codebook Mode  */
#define DMC_MOD_CBC	0x10		/* CBC - Cipher Block Chaining Mod */
#define DMC_MOD_CFB	0x20		/* CFB - Cipher FeedBack Mode      */
#define DMC_MOD_PT 	0x30		/* PT  - Pass Thru Mode            */

#define DMC_MOD_ENC 	0x00		/* Encrypt mode                    */
#define DMC_MOD_DEC 	0x40		/* Decrypt mode                    */

/*--------------------------------------------------------------------------+
|
| HASH Mode Control Register:
|
|     Bit 0    =  Hashing Mode
|                 0 = MD5 hashing
|                 1 = SHA-1 hashing
|
|     Bit 1    =  Programmed Chaining variables
|                 0 = Use Hard-wire value (default value)
|                 1 = Use Chaining register value
|
+--------------------------------------------------------------------------*/
#define HMC_MD5		0x00		/* Use MD5 hashing algorithm       */
#define	HMC_SHA1	0x01            /* Use SHA1 hashing algorithm      */
#define	HMC_CRV	        0x02            /* Use Chaining Register Value     */
#define HMC_CHAIN_DEFAULT   0x00

/*
** defines for accessing the chainvar array, so user code isn't
** quite so ugly.  key1[high|low] only for Single DES;
** All three for 3DES.
*/

#define    desc_key1high    chainvar[0]
#define    desc_key1low     chainvar[1]
#define    desc_key2high    chainvar[2]
#define    desc_key2low     chainvar[3]
#define    desc_key3high    chainvar[4]
#define    desc_key3low     chainvar[5]

#define    HSH_LEN_MD5      16
#define    HSH_LEN_SHA1     20
#define    HSH_LEN_MAX      20
#define    HSH_BLK          64

/*

  For user programs, all that's necessary is to fill in the ioctl structure,
  and ioctl the driver.  The ioctl will block until the operation is complete.

  A future, non-blocking version of the code could be based on writing
  and reading sec_ioctl_t structures from the driver.

*/

/*

  Accessing the security engine from the kernel is another kettle of fish.

  You create the same sec_ioctl and sec_cbuf structures as used
  from userland.  You then call the function
  sec_kernel_entry(&your_ioctl_struct, intlevel) .

  If you are calling from interrupt level, intlevel should be 1; else
  0.  If you are calling from code that executes at both interrupt and
  task level, you should either A) figure out whether you're at
  interrupt level, and set intlevel accordingly, or B) block any
  interrupts that would also call sec_kernel_entry before calling and
  set intlevel to 1.  The second option busy-waits for the hardware
  with interrupts masked, and any right thinking programmer will avoid
  this at all costs.

  The interaction between task level calls and interrupt level calls
  works this way: The task level software ensures that only one task
  level request is given to the hardware at any one time.  When an
  interrupt level request comes in, the request is handed to the
  hardware, either directly or queued behind the currently executing
  task level request.  Then a busy-wait loop is entered that waits for
  the hardware to complete the request(s).  Thus, interrupt level
  calls may wait for two security engine requests.  It is assumed that
  these two requests will be completed by hardware in less time than
  it would have taken software to complete the single request from
  interrupt level.  This is the best we can do, because we don't
  restrict the user from using the same memory for source and
  destination buffers (encrypt/decrypt in place) and thus we can't
  reset the security engine, have it process the interrupt level
  request, and then have it re-start the task level request.

*/

/*------------------------------------------------------------
** Security API availability information.
**
** The following function is called to determine what hardware
** is present, and which API(s) are available (if any).
*/

#ifdef __KERNEL__
/* entry point within the kernel */
void msp_sec_enquire(int * info) ;
#endif

/* ioctl number for user level */
#define MSP_SEC_ENQ ('B' << 8 | 0x0002)

/*
** Whether called from kernel space or user space, this call takes a
** pointer to two ints.
**
** The first int describes the hardware on this system.
**
** Note: This represents which hardware interface is being used by the driver,
** which only uses one hardware interface, even if two are available.  For
** example, "POLO" family chips have a new hardware interface, plus the
** old hardware interface in the interests of being backwards compatible.
** If the driver is compiled to take advantage of "POLO" style hardware,
** the value returned will be 2 (for "SECENQ_HW2"), not 3 (which would
** represent "SECENQ_HW1 | SECENQ_HW2").
*/

#define SECENQ_HW1	1	/* "DUET" compatible hardware */
#define SECENQ_HW2	2	/* "POLO" compatible hardware */

/*
** The second int gives info about the driver; which APIs are available,
** and other options.
**
** Note that it is most likely more efficient to use the native API for
** the harware that is present.  I.E. use the newer API if SECENQ_HW2 is
** present, and use the older API if SECENQ_HW1 is present.
**
** Also, even if SECENQ_API2 is available, AES may not be available
** (e.g. if API2 is being simulated on hardware version 1).  AES is only
** available if SECENQ_API2_AES is set.
*/

#define SECENQ_API1	1	/* "DUET" style api is available */
#define SECENQ_API2	2	/* "POLO" style api is available */

#define SECENQ_API2_AES	0x100	/* AES is availalbe in API2. */


#endif /* BRECIS_MSP_SEC_H */
