/*
 * linux/fs/binfmt_elf.c
 *
 * These are the functions used to load ELF format executables as used
 * on SVr4 machines.  Information on the format may be found in the book
 * "UNIX SYSTEM V RELEASE 4 Programmers Guide: Ansi C and Programming Support
 * Tools".
 *
 * CONFIG_BRECIS sections are for the mipsnommu per Brecis changes.
 *
 * Copyright 1993, 1994: Eric Youngdale (ericy@cais.com).
 */

#include <linux/module.h>

#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/sched.h>
#ifndef CONFIG_BRECIS
#include <linux/mm.h>
#endif /* not CONFIG_BRECIS */
#include <linux/mman.h>
#ifndef CONFIG_BRECIS
#include <linux/a.out.h>
#endif /* not CONFIG_BRECIS */
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/binfmts.h>
#include <linux/string.h>
#include <linux/file.h>
#ifndef CONFIG_BRECIS
#include <linux/fcntl.h>
#endif /* not CONFIG_BRECIS */
#include <linux/ptrace.h>
#include <linux/slab.h>
#ifndef CONFIG_BRECIS
#include <linux/shm.h>
#endif /* not CONFIG_BRECIS */
#include <linux/personality.h>
#ifndef CONFIG_BRECIS
#include <linux/elfcore.h>
#endif /* not CONFIG_BRECIS */
#include <linux/init.h>
#ifndef CONFIG_BRECIS
#include <linux/highuid.h>
#include <linux/smp_lock.h>
#include <linux/compiler.h>
#include <linux/highmem.h>
#endif /* not CONFIG_BRECIS */

#include <asm/uaccess.h>
#ifndef CONFIG_BRECIS
#include <asm/param.h>
#include <asm/pgalloc.h>
#endif /* not CONFIG_BRECIS */

#define DLINFO_ITEMS 13

#include <linux/elf.h>

#ifdef CONFIG_BRECIS
#include <asm/brecis/BrecisMemCtl.h>
/* Flush data and/or instruction caches */
/* extern int icache_size, dcache_size; */
extern int ic_lsize, dc_lsize;
extern unsigned int scache_size, sc_lsize;
#include <asm/mips32_cache.h>

/* The following prints out minimal information for each program */
/* #define DEBUG_WHICH */

/* The following is for debugging loader for specific test program. */
/* #define M4_DEBUG */

#ifdef M4_DEBUG
int M4DEBUG = 0;
#endif

/* DEBUG_LOTS prints out lots of information, all relocation tables, etc. */
/* #define DEBUG_LOTS */

/* DEBUG turns on printing of debug messages. */
/* #define DEBUG y */

/* DBG_ELF prints out normal messages needed for debugging. */
/* DBG_ELF_SECTION will print out the PT sections, the SHT sections, */
/* and the program loading. */

#ifdef DEBUG
  #ifdef M4_DEBUG
    #define DBG_ELF(a...)      {if (M4DEBUG != 0) {printk(a);}}
    #define DBG_ELF_SECTION(a...)      {if (M4DEBUG != 0) {printk(a);}}
/*    #define DBG_ELF_SECTION(a...)      printk(a) */
  #else	/* M4_DEBUG */
    #define DBG_ELF(a...)      printk(a)
    #define DBG_ELF_SECTION(a...)      printk(a)
  #endif /* M4_DEBUG */
#else
  #define DBG_ELF(a...)
  #define DBG_ELF_SECTION(a...)
#endif

/* PARTIAL_SIZE is a check for a SHT section size being "reasonable" */
#define PARTIAL_SIZE (1*1024*1024)
/* MAX_ELF_GZIP is the maximum size of a elf executable, after gunzip-ping. */
#define MAX_ELF_GZIP (32*1024*1024)
/* The size of an ELF stack for a process. */
/* #define ELF_STACK_SIZE	(8*1024) */	/* date, insmod don't work */
#define ELF_STACK_SIZE	(64*1024)
/* #define ELF_STACK_SIZE	(128*1024) */
#define	MIPS_HI16_TABLE_SIZE	64	/* start of relocation "hi_mips" table */

#ifdef CONFIG_BINFMT_ZELF
extern int      decompress_exec(struct linux_binprm *, unsigned long, char *, long);
#endif	/* CONFIG_BINFMT_ZELF */

#endif /* CONFIG_BRECIS */
static int load_elf_binary(struct linux_binprm * bprm, struct pt_regs * regs);
#ifndef CONFIG_BRECIS
static int load_elf_library(struct file*);
static unsigned long elf_map (struct file *, unsigned long, struct elf_phdr *, int, int);
extern int dump_fpu (struct pt_regs *, elf_fpregset_t *);
extern void dump_thread(struct pt_regs *, struct user *);
#else /* CONFIG_BRECIS */
#define load_elf_library	NULL
extern void     bzero(char *, size_t);
#endif /* CONFIG_BRECIS */

#ifndef elf_addr_t
#define elf_addr_t unsigned long
#define elf_caddr_t char *
#endif

/*
 * If we don't support core dumping, then supply a NULL so we
 * don't even try.
 */
#ifdef USE_ELF_CORE_DUMP
static int elf_core_dump(long signr, struct pt_regs * regs, struct file * file);
#else
#define elf_core_dump	NULL
#endif

#if ELF_EXEC_PAGESIZE > PAGE_SIZE
# define ELF_MIN_ALIGN	ELF_EXEC_PAGESIZE
#else
# define ELF_MIN_ALIGN	PAGE_SIZE
#endif

#define ELF_PAGESTART(_v) ((_v) & ~(unsigned long)(ELF_MIN_ALIGN-1))
#define ELF_PAGEOFFSET(_v) ((_v) & (ELF_MIN_ALIGN-1))
#ifndef CONFIG_BRECIS
#define ELF_PAGEALIGN(_v) (((_v) + ELF_MIN_ALIGN - 1) & ~(ELF_MIN_ALIGN - 1))
#else /* CONFIG_BRECIS */

#define TOP_OF_ARGS (PAGE_SIZE * MAX_ARG_PAGES - sizeof(void *))

#if R_MIPS_NUM > 38
#error R_MIPS_NUM has changed
#endif

#ifdef DEBUG_LOTS
static char *relocatename[] = {
  "R_MIPS_NONE         ", "R_MIPS_16           ", "R_MIPS_32           ",
  "R_MIPS_REL32        ", "R_MIPS_26           ", "R_MIPS_HI16         ",
  "R_MIPS_LO16         ", "R_MIPS_GPREL16      ", "R_MIPS_LITERAL      ",
  "R_MIPS_GOT16        ", "R_MIPS_PC16         ", "R_MIPS_CALL16       ",
  "R_MIPS_GPREL32      ", "R_MIPS_SHIFT5       ", "R_MIPS_SHIFT6       ",
  "R_MIPS_64           ", "R_MIPS_GOT_DISP     ", "R_MIPS_GOT_PAGE     ",
  "R_MIPS_GOT_OFST     ", "R_MIPS_GOT_HI16     ", "R_MIPS_GOT_LO16     ",
  "R_MIPS_SUB          ", "R_MIPS_INSERT_A     ", "R_MIPS_INSERT_B     ",
  "R_MIPS_DELETE       ", "R_MIPS_HIGHER       ", "R_MIPS_HIGHEST      ",
  "R_MIPS_CALL_HI16    ", "R_MIPS_CALL_LO16    ", "R_MIPS_SCN_DISP     ",
  "R_MIPS_REL16        ", "R_MIPS_ADD_IMMEDIATE", "R_MIPS_PJUMP        ",
  "R_MIPS_RELGOT       ", "R_MIPS_JALR         ",
};
#endif	/* DEBUG_LOTS */
#endif /* CONFIG_BRECIS */

static struct linux_binfmt elf_format = {
	NULL, THIS_MODULE, load_elf_binary, load_elf_library, elf_core_dump, ELF_EXEC_PAGESIZE
};

#ifndef CONFIG_BRECIS
#define BAD_ADDR(x)	((unsigned long)(x) > TASK_SIZE)

static void set_brk(unsigned long start, unsigned long end)
{
	start = ELF_PAGEALIGN(start);
	end = ELF_PAGEALIGN(end);
	if (end <= start)
		return;
	do_brk(start, end - start);
}


/* We need to explicitly zero any fractional pages
   after the data section (i.e. bss).  This would
   contain the junk from the file that should not
   be in memory */


static void padzero(unsigned long elf_bss)
{
	unsigned long nbyte;

	nbyte = ELF_PAGEOFFSET(elf_bss);
	if (nbyte) {
		nbyte = ELF_MIN_ALIGN - nbyte;
		clear_user((void *) elf_bss, nbyte);
	}
}

static unsigned long create_elf_tables(
		  char *p, int argc, int envc,
		  struct elfhdr * exec,
		  unsigned long load_addr,
		  unsigned long load_bias,
		  unsigned long interp_load_addr, int ibcs)
{
	elf_caddr_t *argv;
	elf_caddr_t *envp;
	elf_addr_t *sp, *csp;
	char *k_platform, *u_platform;
	long hwcap;
	size_t platform_len = 0;
	size_t len;

	/*
	 * Get hold of platform and hardware capabilities masks for
	 * the machine we are running on.  In some cases (Sparc), 
	 * this info is impossible to get, in others (i386) it is
	 * merely difficult.
	 */

	hwcap = ELF_HWCAP;
	k_platform = ELF_PLATFORM;

	if (k_platform) {
		platform_len = strlen(k_platform) + 1;
		u_platform = p - platform_len;
		__copy_to_user(u_platform, k_platform, platform_len);
	} else
		u_platform = p;

#if defined(__i386__) && defined(CONFIG_SMP)
	/*
	 * In some cases (e.g. Hyper-Threading), we want to avoid L1 evictions
	 * by the processes running on the same package. One thing we can do
	 * is to shuffle the initial stack for them.
	 *
	 * The conditionals here are unneeded, but kept in to make the
	 * code behaviour the same as pre change unless we have hyperthreaded
	 * processors. This keeps Mr Marcelo Person happier but should be
	 * removed for 2.5
	 */
	 
	if(smp_num_siblings > 1)
		u_platform = u_platform - ((current->pid % 64) << 7);
#endif	

	/*
	 * Force 16 byte _final_ alignment here for generality.
	 */
	sp = (elf_addr_t *)(~15UL & (unsigned long)(u_platform));
	csp = sp;
	csp -= (1+DLINFO_ITEMS)*2 + (k_platform ? 2 : 0);
#ifdef DLINFO_ARCH_ITEMS
	csp -= DLINFO_ARCH_ITEMS*2;
#endif
	csp -= envc+1;
	csp -= argc+1;
	csp -= (!ibcs ? 3 : 1);	/* argc itself */
	if ((unsigned long)csp & 15UL)
		sp -= ((unsigned long)csp & 15UL) / sizeof(*sp);

	/*
	 * Put the ELF interpreter info on the stack
	 */
#define NEW_AUX_ENT(nr, id, val) \
	  __put_user ((id), sp+(nr*2)); \
	  __put_user ((val), sp+(nr*2+1)); \

	sp -= 2;
	NEW_AUX_ENT(0, AT_NULL, 0);
	if (k_platform) {
		sp -= 2;
		NEW_AUX_ENT(0, AT_PLATFORM, (elf_addr_t)(unsigned long) u_platform);
	}
	sp -= DLINFO_ITEMS*2;
	NEW_AUX_ENT( 0, AT_HWCAP, hwcap);
	NEW_AUX_ENT( 1, AT_PAGESZ, ELF_EXEC_PAGESIZE);
	NEW_AUX_ENT( 2, AT_CLKTCK, CLOCKS_PER_SEC);
	NEW_AUX_ENT( 3, AT_PHDR, load_addr + exec->e_phoff);
	NEW_AUX_ENT( 4, AT_PHENT, sizeof (struct elf_phdr));
	NEW_AUX_ENT( 5, AT_PHNUM, exec->e_phnum);
	NEW_AUX_ENT( 6, AT_BASE, interp_load_addr);
	NEW_AUX_ENT( 7, AT_FLAGS, 0);
	NEW_AUX_ENT( 8, AT_ENTRY, load_bias + exec->e_entry);
	NEW_AUX_ENT( 9, AT_UID, (elf_addr_t) current->uid);
	NEW_AUX_ENT(10, AT_EUID, (elf_addr_t) current->euid);
	NEW_AUX_ENT(11, AT_GID, (elf_addr_t) current->gid);
	NEW_AUX_ENT(12, AT_EGID, (elf_addr_t) current->egid);
#ifdef ARCH_DLINFO
	/* 
	 * ARCH_DLINFO must come last so platform specific code can enforce
	 * special alignment requirements on the AUXV if necessary (eg. PPC).
	 */
	ARCH_DLINFO;
#endif
#undef NEW_AUX_ENT

	sp -= envc+1;
	envp = (elf_caddr_t *) sp;
	sp -= argc+1;
	argv = (elf_caddr_t *) sp;
	if (!ibcs) {
		__put_user((elf_addr_t)(unsigned long) envp,--sp);
		__put_user((elf_addr_t)(unsigned long) argv,--sp);
	}

	__put_user((elf_addr_t)argc,--sp);
	current->mm->arg_start = (unsigned long) p;
	while (argc-->0) {
		__put_user((elf_caddr_t)(unsigned long)p,argv++);
		len = strnlen_user(p, PAGE_SIZE*MAX_ARG_PAGES);
		if (!len || len > PAGE_SIZE*MAX_ARG_PAGES)
			return (unsigned long)NULL;
		p += len;
	}
	__put_user(NULL, argv);
	current->mm->arg_end = current->mm->env_start = (unsigned long) p;
	while (envc-->0) {
		__put_user((elf_caddr_t)(unsigned long)p,envp++);
		len = strnlen_user(p, PAGE_SIZE*MAX_ARG_PAGES);
		if (!len || len > PAGE_SIZE*MAX_ARG_PAGES)
			return (unsigned long)NULL;
		p += len;
	}
	__put_user(NULL, envp);
	current->mm->env_end = (unsigned long) p;
	return (unsigned long)sp;
}

#ifndef elf_map

static inline unsigned long
elf_map (struct file *filep, unsigned long addr, struct elf_phdr *eppnt, int prot, int type)
{
	unsigned long map_addr;

	down_write(&current->mm->mmap_sem);
	map_addr = do_mmap(filep, ELF_PAGESTART(addr),
			   eppnt->p_filesz + ELF_PAGEOFFSET(eppnt->p_vaddr), prot, type,
			   eppnt->p_offset - ELF_PAGEOFFSET(eppnt->p_vaddr));
	up_write(&current->mm->mmap_sem);
	return(map_addr);
}

#endif /* !elf_map */

/* This is much more generalized than the library routine read function,
   so we keep this separate.  Technically the library read function
   is only provided so that we can read a.out libraries that have
   an ELF header */

static unsigned long load_elf_interp(struct elfhdr * interp_elf_ex,
				     struct file * interpreter,
				     unsigned long *interp_load_addr)
{
	struct elf_phdr *elf_phdata;
	struct elf_phdr *eppnt;
	unsigned long load_addr = 0;
	int load_addr_set = 0;
	unsigned long last_bss = 0, elf_bss = 0;
	unsigned long error = ~0UL;
	int retval, i, size;

	/* First of all, some simple consistency checks */
	if (interp_elf_ex->e_type != ET_EXEC &&
	    interp_elf_ex->e_type != ET_DYN)
		goto out;
	if (!elf_check_arch(interp_elf_ex))
		goto out;
	if (!interpreter->f_op || !interpreter->f_op->mmap)
		goto out;

	/*
	 * If the size of this structure has changed, then punt, since
	 * we will be doing the wrong thing.
	 */
	if (interp_elf_ex->e_phentsize != sizeof(struct elf_phdr))
		goto out;
	if (interp_elf_ex->e_phnum > 65536U / sizeof(struct elf_phdr))
		goto out;

	/* Now read in all of the header information */

	size = sizeof(struct elf_phdr) * interp_elf_ex->e_phnum;
	if (size > ELF_MIN_ALIGN)
		goto out;
	elf_phdata = (struct elf_phdr *) kmalloc(size, GFP_KERNEL);
	if (!elf_phdata)
		goto out;

	retval = kernel_read(interpreter,interp_elf_ex->e_phoff,(char *)elf_phdata,size);
	error = retval;
	if (retval < 0)
		goto out_close;

	eppnt = elf_phdata;
	for (i=0; i<interp_elf_ex->e_phnum; i++, eppnt++) {
	  if (eppnt->p_type == PT_LOAD) {
	    int elf_type = MAP_PRIVATE | MAP_DENYWRITE;
	    int elf_prot = 0;
	    unsigned long vaddr = 0;
	    unsigned long k, map_addr;

	    if (eppnt->p_flags & PF_R) elf_prot =  PROT_READ;
	    if (eppnt->p_flags & PF_W) elf_prot |= PROT_WRITE;
	    if (eppnt->p_flags & PF_X) elf_prot |= PROT_EXEC;
	    vaddr = eppnt->p_vaddr;
	    if (interp_elf_ex->e_type == ET_EXEC || load_addr_set)
	    	elf_type |= MAP_FIXED;

	    map_addr = elf_map(interpreter, load_addr + vaddr, eppnt, elf_prot, elf_type);
	    if (BAD_ADDR(map_addr))
	    	goto out_close;

	    if (!load_addr_set && interp_elf_ex->e_type == ET_DYN) {
		load_addr = map_addr - ELF_PAGESTART(vaddr);
		load_addr_set = 1;
	    }

	    /*
	     * Find the end of the file mapping for this phdr, and keep
	     * track of the largest address we see for this.
	     */
	    k = load_addr + eppnt->p_vaddr + eppnt->p_filesz;
	    if (k > elf_bss)
		elf_bss = k;

	    /*
	     * Do the same thing for the memory mapping - between
	     * elf_bss and last_bss is the bss section.
	     */
	    k = load_addr + eppnt->p_memsz + eppnt->p_vaddr;
	    if (k > last_bss)
		last_bss = k;
	  }
	}

	/* Now use mmap to map the library into memory. */

	/*
	 * Now fill out the bss section.  First pad the last page up
	 * to the page boundary, and then perform a mmap to make sure
	 * that there are zero-mapped pages up to and including the 
	 * last bss page.
	 */
	padzero(elf_bss);
	elf_bss = ELF_PAGESTART(elf_bss + ELF_MIN_ALIGN - 1);	/* What we have mapped so far */

	/* Map the last of the bss segment */
	if (last_bss > elf_bss)
		do_brk(elf_bss, last_bss - elf_bss);

	*interp_load_addr = load_addr;
	error = ((unsigned long) interp_elf_ex->e_entry) + load_addr;

out_close:
	kfree(elf_phdata);
out:
	return error;
}

static unsigned long load_aout_interp(struct exec * interp_ex,
			     struct file * interpreter)
{
	unsigned long text_data, elf_entry = ~0UL;
	char * addr;
	loff_t offset;
	int retval;

	current->mm->end_code = interp_ex->a_text;
	text_data = interp_ex->a_text + interp_ex->a_data;
	current->mm->end_data = text_data;
	current->mm->brk = interp_ex->a_bss + text_data;

	switch (N_MAGIC(*interp_ex)) {
	case OMAGIC:
		offset = 32;
		addr = (char *) 0;
		break;
	case ZMAGIC:
	case QMAGIC:
		offset = N_TXTOFF(*interp_ex);
		addr = (char *) N_TXTADDR(*interp_ex);
		break;
	default:
		goto out;
	}

	do_brk(0, text_data);
	retval = -ENOEXEC;
	if (!interpreter->f_op || !interpreter->f_op->read)
		goto out;
	retval = interpreter->f_op->read(interpreter, addr, text_data, &offset);
	if (retval < 0)
		goto out;
	flush_icache_range((unsigned long)addr,
	                   (unsigned long)addr + text_data);

	do_brk(ELF_PAGESTART(text_data + ELF_MIN_ALIGN - 1),
		interp_ex->a_bss);
	elf_entry = interp_ex->a_entry;

out:
	return elf_entry;
}

/*
 * These are the functions used to load ELF style executables and shared
 * libraries.  There is no binary dependent code anywhere else.
 */

#define INTERPRETER_NONE 0
#define INTERPRETER_AOUT 1
#define INTERPRETER_ELF 2


static int load_elf_binary(struct linux_binprm * bprm, struct pt_regs * regs)
{
	struct file *interpreter = NULL; /* to shut gcc up */
 	unsigned long load_addr = 0, load_bias = 0;
	int load_addr_set = 0;
	char * elf_interpreter = NULL;
	unsigned int interpreter_type = INTERPRETER_NONE;
	unsigned char ibcs2_interpreter = 0;
	unsigned long error;
	struct elf_phdr * elf_ppnt, *elf_phdata;
	unsigned long elf_bss, k, elf_brk;
	int elf_exec_fileno;
	int retval, i;
	unsigned int size;
	unsigned long elf_entry, interp_load_addr = 0;
	unsigned long start_code, end_code, start_data, end_data;
	struct elfhdr elf_ex;
	struct elfhdr interp_elf_ex;
  	struct exec interp_ex;
	char passed_fileno[6];
	
	/* Get the exec-header */
	elf_ex = *((struct elfhdr *) bprm->buf);

	retval = -ENOEXEC;
	/* First of all, some simple consistency checks */
	if (memcmp(elf_ex.e_ident, ELFMAG, SELFMAG) != 0)
		goto out;

	if (elf_ex.e_type != ET_EXEC && elf_ex.e_type != ET_DYN)
		goto out;
	if (!elf_check_arch(&elf_ex))
		goto out;
	if (!bprm->file->f_op||!bprm->file->f_op->mmap)
		goto out;

	/* Now read in all of the header information */

	retval = -ENOMEM;
	if (elf_ex.e_phentsize != sizeof(struct elf_phdr))
		goto out;
	if (elf_ex.e_phnum > 65536U / sizeof(struct elf_phdr))
		goto out;
	size = elf_ex.e_phnum * sizeof(struct elf_phdr);
	elf_phdata = (struct elf_phdr *) kmalloc(size, GFP_KERNEL);
	if (!elf_phdata)
		goto out;

	retval = kernel_read(bprm->file, elf_ex.e_phoff, (char *) elf_phdata, size);
	if (retval < 0)
		goto out_free_ph;

	retval = get_unused_fd();
	if (retval < 0)
		goto out_free_ph;
	get_file(bprm->file);
	fd_install(elf_exec_fileno = retval, bprm->file);

	elf_ppnt = elf_phdata;
	elf_bss = 0;
	elf_brk = 0;

	start_code = ~0UL;
	end_code = 0;
	start_data = 0;
	end_data = 0;

	for (i = 0; i < elf_ex.e_phnum; i++) {
		if (elf_ppnt->p_type == PT_INTERP) {
			/* This is the program interpreter used for
			 * shared libraries - for now assume that this
			 * is an a.out format binary
			 */

			retval = -ENOMEM;
			if (elf_ppnt->p_filesz > PATH_MAX)
				goto out_free_file;
			elf_interpreter = (char *) kmalloc(elf_ppnt->p_filesz,
							   GFP_KERNEL);
			if (!elf_interpreter)
				goto out_free_file;

			retval = kernel_read(bprm->file, elf_ppnt->p_offset,
					   elf_interpreter,
					   elf_ppnt->p_filesz);
			if (retval < 0)
				goto out_free_interp;
			/* If the program interpreter is one of these two,
			 * then assume an iBCS2 image. Otherwise assume
			 * a native linux image.
			 */
			if (strcmp(elf_interpreter,"/usr/lib/libc.so.1") == 0 ||
			    strcmp(elf_interpreter,"/usr/lib/ld.so.1") == 0)
				ibcs2_interpreter = 1;
#if 0
			printk("Using ELF interpreter %s\n", elf_interpreter);
#endif

			SET_PERSONALITY(elf_ex, ibcs2_interpreter);

			interpreter = open_exec(elf_interpreter);
			retval = PTR_ERR(interpreter);
			if (IS_ERR(interpreter))
				goto out_free_interp;
			retval = kernel_read(interpreter, 0, bprm->buf, BINPRM_BUF_SIZE);
			if (retval < 0)
				goto out_free_dentry;

			/* Get the exec headers */
			interp_ex = *((struct exec *) bprm->buf);
			interp_elf_ex = *((struct elfhdr *) bprm->buf);
			break;
		}
		elf_ppnt++;
	}

	/* Some simple consistency checks for the interpreter */
	if (elf_interpreter) {
		interpreter_type = INTERPRETER_ELF | INTERPRETER_AOUT;

		/* Now figure out which format our binary is */
		if ((N_MAGIC(interp_ex) != OMAGIC) &&
		    (N_MAGIC(interp_ex) != ZMAGIC) &&
		    (N_MAGIC(interp_ex) != QMAGIC))
			interpreter_type = INTERPRETER_ELF;

		if (memcmp(interp_elf_ex.e_ident, ELFMAG, SELFMAG) != 0)
			interpreter_type &= ~INTERPRETER_ELF;

		retval = -ELIBBAD;
		if (!interpreter_type)
			goto out_free_dentry;

		/* Make sure only one type was selected */
		if ((interpreter_type & INTERPRETER_ELF) &&
		     interpreter_type != INTERPRETER_ELF) {
	     		// FIXME - ratelimit this before re-enabling
			// printk(KERN_WARNING "ELF: Ambiguous type, using ELF\n");
			interpreter_type = INTERPRETER_ELF;
		}
	} else {
		/* Executables without an interpreter also need a personality  */
		SET_PERSONALITY(elf_ex, ibcs2_interpreter);
	}

	/* OK, we are done with that, now set up the arg stuff,
	   and then start this sucker up */

	if (!bprm->sh_bang) {
		char * passed_p;

		if (interpreter_type == INTERPRETER_AOUT) {
		  sprintf(passed_fileno, "%d", elf_exec_fileno);
		  passed_p = passed_fileno;

		  if (elf_interpreter) {
		    retval = copy_strings_kernel(1,&passed_p,bprm);
			if (retval)
				goto out_free_dentry; 
		    bprm->argc++;
		  }
		}
	}

	/* Flush all traces of the currently running executable */
	retval = flush_old_exec(bprm);
	if (retval)
		goto out_free_dentry;

	/* OK, This is the point of no return */
	current->mm->start_data = 0;
	current->mm->end_data = 0;
	current->mm->end_code = 0;
	current->mm->mmap = NULL;
	current->flags &= ~PF_FORKNOEXEC;
	elf_entry = (unsigned long) elf_ex.e_entry;

	/* Do this so that we can load the interpreter, if need be.  We will
	   change some of these later */
	current->mm->rss = 0;
	retval = setup_arg_pages(bprm);
	if (retval < 0) {
		send_sig(SIGKILL, current, 0);
		return retval;
	}
	
	current->mm->start_stack = bprm->p;

	/* Now we do a little grungy work by mmaping the ELF image into
	   the correct location in memory.  At this point, we assume that
	   the image should be loaded at fixed address, not at a variable
	   address. */

	for(i = 0, elf_ppnt = elf_phdata; i < elf_ex.e_phnum; i++, elf_ppnt++) {
		int elf_prot = 0, elf_flags;
		unsigned long vaddr;

		if (elf_ppnt->p_type != PT_LOAD)
			continue;

		if (unlikely (elf_brk > elf_bss)) {
			unsigned long nbyte;
	            
			/* There was a PT_LOAD segment with p_memsz > p_filesz
			   before this one. Map anonymous pages, if needed,
			   and clear the area.  */
			set_brk (elf_bss + load_bias, elf_brk + load_bias);
			nbyte = ELF_PAGEOFFSET(elf_bss);
			if (nbyte) {
				nbyte = ELF_MIN_ALIGN - nbyte;
				if (nbyte > elf_brk - elf_bss)
					nbyte = elf_brk - elf_bss;
				clear_user((void *) elf_bss + load_bias, nbyte);
			}
		}

		if (elf_ppnt->p_flags & PF_R) elf_prot |= PROT_READ;
		if (elf_ppnt->p_flags & PF_W) elf_prot |= PROT_WRITE;
		if (elf_ppnt->p_flags & PF_X) elf_prot |= PROT_EXEC;

		elf_flags = MAP_PRIVATE|MAP_DENYWRITE|MAP_EXECUTABLE;

		vaddr = elf_ppnt->p_vaddr;
		if (elf_ex.e_type == ET_EXEC || load_addr_set) {
			elf_flags |= MAP_FIXED;
		} else if (elf_ex.e_type == ET_DYN) {
			/* Try and get dynamic programs out of the way of the default mmap
			   base, as well as whatever program they might try to exec.  This
		           is because the brk will follow the loader, and is not movable.  */
			load_bias = ELF_PAGESTART(ELF_ET_DYN_BASE - vaddr);
		}

		error = elf_map(bprm->file, load_bias + vaddr, elf_ppnt, elf_prot, elf_flags);
		if (BAD_ADDR(error))
			continue;

		if (!load_addr_set) {
			load_addr_set = 1;
			load_addr = (elf_ppnt->p_vaddr - elf_ppnt->p_offset);
			if (elf_ex.e_type == ET_DYN) {
				load_bias += error -
				             ELF_PAGESTART(load_bias + vaddr);
				load_addr += load_bias;
			}
		}
		k = elf_ppnt->p_vaddr;
		if (k < start_code) start_code = k;
		if (start_data < k) start_data = k;

		k = elf_ppnt->p_vaddr + elf_ppnt->p_filesz;

		if (k > elf_bss)
			elf_bss = k;
		if ((elf_ppnt->p_flags & PF_X) && end_code <  k)
			end_code = k;
		if (end_data < k)
			end_data = k;
		k = elf_ppnt->p_vaddr + elf_ppnt->p_memsz;
		if (k > elf_brk)
			elf_brk = k;
	}

	elf_entry += load_bias;
	elf_bss += load_bias;
	elf_brk += load_bias;
	start_code += load_bias;
	end_code += load_bias;
	start_data += load_bias;
	end_data += load_bias;

	if (elf_interpreter) {
		if (interpreter_type == INTERPRETER_AOUT)
			elf_entry = load_aout_interp(&interp_ex,
						     interpreter);
		else
			elf_entry = load_elf_interp(&interp_elf_ex,
						    interpreter,
						    &interp_load_addr);

		allow_write_access(interpreter);
		fput(interpreter);
		kfree(elf_interpreter);

		if (BAD_ADDR(elf_entry)) {
			printk(KERN_ERR "Unable to load interpreter\n");
			kfree(elf_phdata);
			send_sig(SIGSEGV, current, 0);
			return 0;
		}
	}

	kfree(elf_phdata);

	if (interpreter_type != INTERPRETER_AOUT)
		sys_close(elf_exec_fileno);

	set_binfmt(&elf_format);

	compute_creds(bprm);
	current->flags &= ~PF_FORKNOEXEC;
	bprm->p = create_elf_tables((char *)bprm->p,
			bprm->argc,
			bprm->envc,
			&elf_ex,
			load_addr, load_bias,
			interp_load_addr,
			(interpreter_type == INTERPRETER_AOUT ? 0 : 1));
	/* N.B. passed_fileno might not be initialized? */
	if (interpreter_type == INTERPRETER_AOUT)
		current->mm->arg_start += strlen(passed_fileno) + 1;
	current->mm->start_brk = current->mm->brk = elf_brk;
	current->mm->end_code = end_code;
	current->mm->start_code = start_code;
	current->mm->start_data = start_data;
	current->mm->end_data = end_data;
	current->mm->start_stack = bprm->p;

	/* Calling set_brk effectively mmaps the pages that we need
	 * for the bss and break sections
	 */
	set_brk(elf_bss, elf_brk);

	padzero(elf_bss);

#if 0
	printk("(start_brk) %lx\n" , (long) current->mm->start_brk);
	printk("(end_code) %lx\n" , (long) current->mm->end_code);
	printk("(start_code) %lx\n" , (long) current->mm->start_code);
	printk("(start_data) %lx\n" , (long) current->mm->start_data);
	printk("(end_data) %lx\n" , (long) current->mm->end_data);
	printk("(start_stack) %lx\n" , (long) current->mm->start_stack);
	printk("(brk) %lx\n" , (long) current->mm->brk);
#endif

	if (current->personality & MMAP_PAGE_ZERO) {
		/* Why this, you ask???  Well SVr4 maps page 0 as read-only,
		   and some applications "depend" upon this behavior.
		   Since we do not have the power to recompile these, we
		   emulate the SVr4 behavior.  Sigh.  */
		/* N.B. Shouldn't the size here be PAGE_SIZE?? */
		down_write(&current->mm->mmap_sem);
		error = do_mmap(NULL, 0, 4096, PROT_READ | PROT_EXEC,
				MAP_FIXED | MAP_PRIVATE, 0);
		up_write(&current->mm->mmap_sem);
	}

#ifdef ELF_PLAT_INIT
	/*
	 * The ABI may specify that certain registers be set up in special
	 * ways (on i386 %edx is the address of a DT_FINI function, for
	 * example.  This macro performs whatever initialization to
	 * the regs structure is required.
	 */
	ELF_PLAT_INIT(regs);
#endif

	start_thread(regs, elf_entry, bprm->p);
	if (current->ptrace & PT_PTRACED)
		send_sig(SIGTRAP, current, 0);
	retval = 0;
out:
	return retval;

	/* error cleanup */
out_free_dentry:
	allow_write_access(interpreter);
	fput(interpreter);
out_free_interp:
	if (elf_interpreter)
		kfree(elf_interpreter);
out_free_file:
	sys_close(elf_exec_fileno);
out_free_ph:
	kfree(elf_phdata);
	goto out;
}

/* This is really simpleminded and specialized - we are loading an
   a.out library that is given an ELF header. */

static int load_elf_library(struct file *file)
{
	struct elf_phdr *elf_phdata;
	unsigned long elf_bss, bss, len;
	int retval, error, i, j;
	struct elfhdr elf_ex;

	error = -ENOEXEC;
	retval = kernel_read(file, 0, (char *) &elf_ex, sizeof(elf_ex));
	if (retval != sizeof(elf_ex))
		goto out;

	if (memcmp(elf_ex.e_ident, ELFMAG, SELFMAG) != 0)
		goto out;

	/* First of all, some simple consistency checks */
	if (elf_ex.e_type != ET_EXEC || elf_ex.e_phnum > 2 ||
	   !elf_check_arch(&elf_ex) || !file->f_op || !file->f_op->mmap)
		goto out;

	/* Now read in all of the header information */

	j = sizeof(struct elf_phdr) * elf_ex.e_phnum;
	/* j < ELF_MIN_ALIGN because elf_ex.e_phnum <= 2 */

	error = -ENOMEM;
	elf_phdata = (struct elf_phdr *) kmalloc(j, GFP_KERNEL);
	if (!elf_phdata)
		goto out;

	error = -ENOEXEC;
	retval = kernel_read(file, elf_ex.e_phoff, (char *) elf_phdata, j);
	if (retval != j)
		goto out_free_ph;

	for (j = 0, i = 0; i<elf_ex.e_phnum; i++)
		if ((elf_phdata + i)->p_type == PT_LOAD) j++;
	if (j != 1)
		goto out_free_ph;

	while (elf_phdata->p_type != PT_LOAD) elf_phdata++;

	/* Now use mmap to map the library into memory. */
	down_write(&current->mm->mmap_sem);
	error = do_mmap(file,
			ELF_PAGESTART(elf_phdata->p_vaddr),
			(elf_phdata->p_filesz +
			 ELF_PAGEOFFSET(elf_phdata->p_vaddr)),
			PROT_READ | PROT_WRITE | PROT_EXEC,
			MAP_FIXED | MAP_PRIVATE | MAP_DENYWRITE,
			(elf_phdata->p_offset -
			 ELF_PAGEOFFSET(elf_phdata->p_vaddr)));
	up_write(&current->mm->mmap_sem);
	if (error != ELF_PAGESTART(elf_phdata->p_vaddr))
		goto out_free_ph;

	elf_bss = elf_phdata->p_vaddr + elf_phdata->p_filesz;
	padzero(elf_bss);

	len = ELF_PAGESTART(elf_phdata->p_filesz + elf_phdata->p_vaddr + ELF_MIN_ALIGN - 1);
	bss = elf_phdata->p_memsz + elf_phdata->p_vaddr;
	if (bss > len)
		do_brk(len, bss - len);
	error = 0;

out_free_ph:
	kfree(elf_phdata);
out:
	return error;
}

/*
 * Note that some platforms still use traditional core dumps and not
 * the ELF core dump.  Each platform can select it as appropriate.
 */
#ifdef USE_ELF_CORE_DUMP

/*
 * ELF core dumper
 *
 * Modelled on fs/exec.c:aout_core_dump()
 * Jeremy Fitzhardinge <jeremy@sw.oz.au>
 */
/*
 * These are the only things you should do on a core-file: use only these
 * functions to write out all the necessary info.
 */
static int dump_write(struct file *file, const void *addr, int nr)
{
	return file->f_op->write(file, addr, nr, &file->f_pos) == nr;
}

static int dump_seek(struct file *file, off_t off)
{
	if (file->f_op->llseek) {
		if (file->f_op->llseek(file, off, 0) != off)
			return 0;
	} else
		file->f_pos = off;
	return 1;
}

/*
 * Decide whether a segment is worth dumping; default is yes to be
 * sure (missing info is worse than too much; etc).
 * Personally I'd include everything, and use the coredump limit...
 *
 * I think we should skip something. But I am not sure how. H.J.
 */
static inline int maydump(struct vm_area_struct *vma)
{
	/*
	 * If we may not read the contents, don't allow us to dump
	 * them either. "dump_write()" can't handle it anyway.
	 */
	if (!(vma->vm_flags & VM_READ))
		return 0;

	/* Do not dump I/O mapped devices! -DaveM */
	if (vma->vm_flags & VM_IO)
		return 0;
#if 1
	if (vma->vm_flags & (VM_WRITE|VM_GROWSUP|VM_GROWSDOWN))
		return 1;
	if (vma->vm_flags & (VM_READ|VM_EXEC|VM_EXECUTABLE|VM_SHARED))
		return 0;
#endif
	return 1;
}

#define roundup(x, y)  ((((x)+((y)-1))/(y))*(y))

/* An ELF note in memory */
struct memelfnote
{
	const char *name;
	int type;
	unsigned int datasz;
	void *data;
};

static int notesize(struct memelfnote *en)
{
	int sz;

	sz = sizeof(struct elf_note);
	sz += roundup(strlen(en->name), 4);
	sz += roundup(en->datasz, 4);

	return sz;
}

/* #define DEBUG */

#ifdef DEBUG
static void dump_regs(const char *str, elf_greg_t *r)
{
	int i;
	static const char *regs[] = { "ebx", "ecx", "edx", "esi", "edi", "ebp",
					      "eax", "ds", "es", "fs", "gs",
					      "orig_eax", "eip", "cs",
					      "efl", "uesp", "ss"};
	printk("Registers: %s\n", str);

	for(i = 0; i < ELF_NGREG; i++)
	{
		unsigned long val = r[i];
		printk("   %-2d %-5s=%08lx %lu\n", i, regs[i], val, val);
	}
}
#endif

#define DUMP_WRITE(addr, nr)	\
	do { if (!dump_write(file, (addr), (nr))) return 0; } while(0)
#define DUMP_SEEK(off)	\
	do { if (!dump_seek(file, (off))) return 0; } while(0)

static int writenote(struct memelfnote *men, struct file *file)
{
	struct elf_note en;

	en.n_namesz = strlen(men->name);
	en.n_descsz = men->datasz;
	en.n_type = men->type;

	DUMP_WRITE(&en, sizeof(en));
	DUMP_WRITE(men->name, en.n_namesz);
	/* XXX - cast from long long to long to avoid need for libgcc.a */
	DUMP_SEEK(roundup((unsigned long)file->f_pos, 4));	/* XXX */
	DUMP_WRITE(men->data, men->datasz);
	DUMP_SEEK(roundup((unsigned long)file->f_pos, 4));	/* XXX */

	return 1;
}
#undef DUMP_WRITE
#undef DUMP_SEEK

#define DUMP_WRITE(addr, nr)	\
	if ((size += (nr)) > limit || !dump_write(file, (addr), (nr))) \
		goto end_coredump;
#define DUMP_SEEK(off)	\
	if (!dump_seek(file, (off))) \
		goto end_coredump;
/*
 * Actual dumper
 *
 * This is a two-pass process; first we find the offsets of the bits,
 * and then they are actually written out.  If we run out of core limit
 * we just truncate.
 */
static int elf_core_dump(long signr, struct pt_regs * regs, struct file * file)
{
	int has_dumped = 0;
	mm_segment_t fs;
	int segs;
	size_t size = 0;
	int i;
	struct vm_area_struct *vma;
	struct elfhdr elf;
	off_t offset = 0, dataoff;
	unsigned long limit = current->rlim[RLIMIT_CORE].rlim_cur;
	int numnote = 4;
	struct memelfnote notes[4];
	struct elf_prstatus prstatus;	/* NT_PRSTATUS */
	elf_fpregset_t fpu;		/* NT_PRFPREG */
	struct elf_prpsinfo psinfo;	/* NT_PRPSINFO */

	/* first copy the parameters from user space */
	memset(&psinfo, 0, sizeof(psinfo));
	{
		int i, len;

		len = current->mm->arg_end - current->mm->arg_start;
		if (len >= ELF_PRARGSZ)
			len = ELF_PRARGSZ-1;
		copy_from_user(&psinfo.pr_psargs,
			      (const char *)current->mm->arg_start, len);
		for(i = 0; i < len; i++)
			if (psinfo.pr_psargs[i] == 0)
				psinfo.pr_psargs[i] = ' ';
		psinfo.pr_psargs[len] = 0;

	}

	memset(&prstatus, 0, sizeof(prstatus));
	/*
	 * This transfers the registers from regs into the standard
	 * coredump arrangement, whatever that is.
	 */
#ifdef ELF_CORE_COPY_REGS
	ELF_CORE_COPY_REGS(prstatus.pr_reg, regs)
#else
	if (sizeof(elf_gregset_t) != sizeof(struct pt_regs))
	{
		printk("sizeof(elf_gregset_t) (%ld) != sizeof(struct pt_regs) (%ld)\n",
			(long)sizeof(elf_gregset_t), (long)sizeof(struct pt_regs));
	}
	else
		*(struct pt_regs *)&prstatus.pr_reg = *regs;
#endif

	/* now stop all vm operations */
	down_write(&current->mm->mmap_sem);
	segs = current->mm->map_count;

#ifdef DEBUG
	printk("elf_core_dump: %d segs %lu limit\n", segs, limit);
#endif

	/* Set up header */
	memcpy(elf.e_ident, ELFMAG, SELFMAG);
	elf.e_ident[EI_CLASS] = ELF_CLASS;
	elf.e_ident[EI_DATA] = ELF_DATA;
	elf.e_ident[EI_VERSION] = EV_CURRENT;
	memset(elf.e_ident+EI_PAD, 0, EI_NIDENT-EI_PAD);

	elf.e_type = ET_CORE;
	elf.e_machine = ELF_ARCH;
	elf.e_version = EV_CURRENT;
	elf.e_entry = 0;
	elf.e_phoff = sizeof(elf);
	elf.e_shoff = 0;
	elf.e_flags = 0;
	elf.e_ehsize = sizeof(elf);
	elf.e_phentsize = sizeof(struct elf_phdr);
	elf.e_phnum = segs+1;		/* Include notes */
	elf.e_shentsize = 0;
	elf.e_shnum = 0;
	elf.e_shstrndx = 0;

	fs = get_fs();
	set_fs(KERNEL_DS);

	has_dumped = 1;
	current->flags |= PF_DUMPCORE;

	DUMP_WRITE(&elf, sizeof(elf));
	offset += sizeof(elf);				/* Elf header */
	offset += (segs+1) * sizeof(struct elf_phdr);	/* Program headers */

	/*
	 * Set up the notes in similar form to SVR4 core dumps made
	 * with info from their /proc.
	 */

	notes[0].name = "CORE";
	notes[0].type = NT_PRSTATUS;
	notes[0].datasz = sizeof(prstatus);
	notes[0].data = &prstatus;
	prstatus.pr_info.si_signo = prstatus.pr_cursig = signr;
	prstatus.pr_sigpend = current->pending.signal.sig[0];
	prstatus.pr_sighold = current->blocked.sig[0];
	psinfo.pr_pid = prstatus.pr_pid = current->pid;
	psinfo.pr_ppid = prstatus.pr_ppid = current->p_pptr->pid;
	psinfo.pr_pgrp = prstatus.pr_pgrp = current->pgrp;
	psinfo.pr_sid = prstatus.pr_sid = current->session;
	prstatus.pr_utime.tv_sec = CT_TO_SECS(current->times.tms_utime);
	prstatus.pr_utime.tv_usec = CT_TO_USECS(current->times.tms_utime);
	prstatus.pr_stime.tv_sec = CT_TO_SECS(current->times.tms_stime);
	prstatus.pr_stime.tv_usec = CT_TO_USECS(current->times.tms_stime);
	prstatus.pr_cutime.tv_sec = CT_TO_SECS(current->times.tms_cutime);
	prstatus.pr_cutime.tv_usec = CT_TO_USECS(current->times.tms_cutime);
	prstatus.pr_cstime.tv_sec = CT_TO_SECS(current->times.tms_cstime);
	prstatus.pr_cstime.tv_usec = CT_TO_USECS(current->times.tms_cstime);

#ifdef DEBUG
	dump_regs("Passed in regs", (elf_greg_t *)regs);
	dump_regs("prstatus regs", (elf_greg_t *)&prstatus.pr_reg);
#endif

	notes[1].name = "CORE";
	notes[1].type = NT_PRPSINFO;
	notes[1].datasz = sizeof(psinfo);
	notes[1].data = &psinfo;
	i = current->state ? ffz(~current->state) + 1 : 0;
	psinfo.pr_state = i;
	psinfo.pr_sname = (i < 0 || i > 5) ? '.' : "RSDZTD"[i];
	psinfo.pr_zomb = psinfo.pr_sname == 'Z';
	psinfo.pr_nice = current->nice;
	psinfo.pr_flag = current->flags;
	psinfo.pr_uid = NEW_TO_OLD_UID(current->uid);
	psinfo.pr_gid = NEW_TO_OLD_GID(current->gid);
	strncpy(psinfo.pr_fname, current->comm, sizeof(psinfo.pr_fname));

	notes[2].name = "CORE";
	notes[2].type = NT_TASKSTRUCT;
	notes[2].datasz = sizeof(*current);
	notes[2].data = current;

	/* Try to dump the FPU. */
	prstatus.pr_fpvalid = dump_fpu (regs, &fpu);
	if (!prstatus.pr_fpvalid)
	{
		numnote--;
	}
	else
	{
		notes[3].name = "CORE";
		notes[3].type = NT_PRFPREG;
		notes[3].datasz = sizeof(fpu);
		notes[3].data = &fpu;
	}
	
	/* Write notes phdr entry */
	{
		struct elf_phdr phdr;
		int sz = 0;

		for(i = 0; i < numnote; i++)
			sz += notesize(&notes[i]);

		phdr.p_type = PT_NOTE;
		phdr.p_offset = offset;
		phdr.p_vaddr = 0;
		phdr.p_paddr = 0;
		phdr.p_filesz = sz;
		phdr.p_memsz = 0;
		phdr.p_flags = 0;
		phdr.p_align = 0;

		offset += phdr.p_filesz;
		DUMP_WRITE(&phdr, sizeof(phdr));
	}

	/* Page-align dumped data */
	dataoff = offset = roundup(offset, ELF_EXEC_PAGESIZE);

	/* Write program headers for segments dump */
	for(vma = current->mm->mmap; vma != NULL; vma = vma->vm_next) {
		struct elf_phdr phdr;
		size_t sz;

		sz = vma->vm_end - vma->vm_start;

		phdr.p_type = PT_LOAD;
		phdr.p_offset = offset;
		phdr.p_vaddr = vma->vm_start;
		phdr.p_paddr = 0;
		phdr.p_filesz = maydump(vma) ? sz : 0;
		phdr.p_memsz = sz;
		offset += phdr.p_filesz;
		phdr.p_flags = vma->vm_flags & VM_READ ? PF_R : 0;
		if (vma->vm_flags & VM_WRITE) phdr.p_flags |= PF_W;
		if (vma->vm_flags & VM_EXEC) phdr.p_flags |= PF_X;
		phdr.p_align = ELF_EXEC_PAGESIZE;

		DUMP_WRITE(&phdr, sizeof(phdr));
	}

	for(i = 0; i < numnote; i++)
		if (!writenote(&notes[i], file))
			goto end_coredump;

	DUMP_SEEK(dataoff);

	for(vma = current->mm->mmap; vma != NULL; vma = vma->vm_next) {
		unsigned long addr;

		if (!maydump(vma))
			continue;

#ifdef DEBUG
		printk("elf_core_dump: writing %08lx-%08lx\n", vma->vm_start, vma->vm_end);
#endif

		for (addr = vma->vm_start;
		     addr < vma->vm_end;
		     addr += PAGE_SIZE) {
			struct page* page;
			struct vm_area_struct *vma;

			if (get_user_pages(current, current->mm, addr, 1, 0, 1,
						&page, &vma) <= 0) {
				DUMP_SEEK (file->f_pos + PAGE_SIZE);
			} else {
				if (page == ZERO_PAGE(addr)) {
					DUMP_SEEK (file->f_pos + PAGE_SIZE);
				} else {
					void *kaddr;
					flush_cache_page(vma, addr);
					kaddr = kmap(page);
					DUMP_WRITE(kaddr, PAGE_SIZE);
					flush_page_to_ram(page);
					kunmap(page);
				}
				put_page(page);
			}
		}
	}

	if ((off_t) file->f_pos != offset) {
		/* Sanity check */
		printk("elf_core_dump: file->f_pos (%ld) != offset (%ld)\n",
		       (off_t) file->f_pos, offset);
	}

 end_coredump:
	set_fs(fs);
	up_write(&current->mm->mmap_sem);
	return has_dumped;
}
#endif		/* USE_ELF_CORE_DUMP */

#else	/* CONFIG_BRECIS */

/* create_elf_tables() parses the env- and arg-strings in new user
 * memory and creates the pointer tables from them, and puts their
 * addresses on the "stack", returning the new stack pointer value. */

static unsigned long create_elf_tables(
				       unsigned long pp,
				       struct linux_binprm * bprm)
{
  unsigned long  *argv,
                 *envp;
  unsigned long  *sp;
  char           *pc = (char *) pp;
  int             argc = bprm->argc;
  int             envc = bprm->envc;
  char            dummy = 0;

  DBG_ELF("elf: Starting to create stack, and clear BSS...\n");
  sp = (unsigned long *) ((-(unsigned long) sizeof(char *)) & (unsigned long) pc);
  DBG_ELF("elf: sp = %p, pp = %08lx\n", (void *) sp, pp);
  sp -= envc + 1;
  envp = sp;
  sp -= argc + 1;
  argv = sp;
  __put_user(argc, --sp);
  current->mm->arg_start = (unsigned long) pc;
  while (argc-- > 0) {
    __put_user((unsigned long) pc, argv++);
    do {
      __get_user(dummy, pc);
      pc++;
    } while (dummy);
  }
  __put_user((unsigned long) NULL, argv);
  current->mm->arg_end = current->mm->env_start = (unsigned long) pc;
  while (envc-- > 0) {
    __put_user((unsigned long) pc, envp);
    envp++;
    do {
      __get_user(dummy, pc);
      pc++;
    } while (dummy);
  }
  __put_user((unsigned long) NULL, envp);
  current->mm->env_end = (unsigned long) pc;
  return ((unsigned long) sp);
}

/* These are the functions used to load ELF style executables. */
/*  For elf created via "mips-ld -q" (not -fpic -mabicalls): */
/*     a) Read the section table into memory. [elf_shdata] */
/*        1) string tables (debugging) */
/*     b) Read program header structures into memory. [elf_phdata] */
/*        1) get size of memory needed, add in "64k" for stack. */
/*        2) page align size, kmalloc memory. */
/*        3) read in program. */
/*     c) Go through section table. */
/*        1) reginfo section for gp register. */
/*        2) relocation sections. */
/* 	  A) process as read. */
/*        3) Search symtab for _fdata. */
/* 	  A) read table. */
/* 	  B) go through looking for _fdata, and it's value. */
/* 	     Add in load_bias. */
/*     c) Get the symbols read into memory. */
/*     d) Get "_fdata" value. */
/*     e) Do relocations. */

static int      load_elf_binary(struct linux_binprm * bprm, struct pt_regs * regs)
{
#ifdef DEBUG_LOTS
  char           *ptmp;
#endif	/* DEBUG_LOTS */
  unsigned long   load_bias = 0;
  unsigned long   l_size = 0;
  mm_segment_t    old_fs = {0};
  unsigned long   error;
  struct elf_phdr *elf_ppnt;
  struct elf_phdr *elf_phdata = NULL;
  Elf32_Shdr     *elf_shdata = NULL;
  Elf32_Shdr     *elf_psh;
  Elf32_RegInfo  *elf_reginfo = NULL;
  unsigned long   elf_brk;
  unsigned long   elf_gp = 0;
  int             retval, i;
  int             size;
  int             fsize;
  struct elfhdr   elf_ex;
  unsigned long   start_code = ~0UL;
  unsigned long   end_code;
  unsigned long   start_data;
  unsigned long   top_memory = 0;
  unsigned long   p = bprm->p;
  unsigned long   args_at;
  loff_t          fpos;
  int             flags = 0;		/* used to make sure certain sections are present. */
  char           *stmp = NULL;
  char           *strtab = NULL;	/* need string table for _fdata */
  char           *shstrtab = NULL;	/* section names */
  int             strtab_size = 0;
  int             shstrtab_size = 0;
  int		hi_mips_ix = 0;		/* Index into hi_mips table */
  unsigned long  *ltmp;
  unsigned long  **hi_mips = NULL;
  unsigned long   hi_mips_lth = 0;
  unsigned long   offset;
  unsigned long   opcode = 0;
  unsigned long   value_lo;
  Elf32_Rel *relocs = NULL;
  Elf32_Sym *elf_symbols = NULL;
  long            size_elf_symbols = 0;
  int cnt;
  int tmp;
#ifdef CONFIG_BINFMT_ZELF
  char *zflag = NULL;			/* non-zero if gzip-ed executable memory. */
  unsigned long zsize = 0; 
#endif	/* CONFIG_BINFMT_ZELF */

  retval = -ENOEXEC;

#ifdef M4_DEBUG
  if (strcmp(bprm->filename, CONFIG_DEBUG_PROGRAM ) == 0) {
    M4DEBUG |= 1;
  } else {
    M4DEBUG = 0;
  }
#endif
  DBG_ELF_SECTION("load_elf_binary(%s)\n", bprm->filename);
  DBG_ELF("struct elfhdr=%x(%d), struct elf_phdr=%x(%d), Elf32_Shdr=%x(%d), Elf32_RegInfo=%x(%d)\n", sizeof(struct elfhdr), sizeof(struct elfhdr), sizeof(struct elf_phdr), sizeof(struct elf_phdr), sizeof(Elf32_Shdr), sizeof(Elf32_Shdr), sizeof(Elf32_RegInfo), sizeof(Elf32_RegInfo));

/* Get the exec-header */
#ifdef CONFIG_BINFMT_ZELF
  if (strncmp(bprm->buf, "ELF.mips32.gzip", 16) == 0) {
    zsize = ntohl(*(long *)(bprm->buf + 16));	/* size of file after uncompressed */
/* printk("ELF.mips32.gzip(%s), zsize=%ld\n", bprm->filename, zsize); */
    if (zsize < 0 || zsize > MAX_ELF_GZIP) {
      printk("ELF.mips32.gzip, zsize [0x%lx](%ld) bad\n", zsize, zsize);
      goto out_error;
    }
    zflag = (char *) kmalloc(zsize, GFP_KERNEL);
    if (zflag == NULL) {
      DBG_ELF("kmalloc failed, zflag=%x, zsize=0x%lx (%ld)\n", (unsigned int)zflag, zsize, zsize);
      goto out_error;
    }

    fsize = bprm->file->f_dentry->d_inode->i_size;
/* printk("current length=%d\n", fsize); */
    tmp = decompress_exec(bprm, 20UL, zflag, fsize);
    if (tmp >= (unsigned long) -4096) {
      printk("Unable to decompress, errno %d\n", (int) -tmp);
      goto out_error;
    }
/* printk("decompress_exec(, 20, %x, 0x%x (%d)) returned 0x%x (%d)\n", (int)zflag, fsize, fsize, tmp, tmp); */
#if 0
/* Print out first longwords of what was loaded into memory. */
  {
    int nsize;

    for (nsize = 0; nsize < 8800; nsize++) {
      if (nsize % 4 == 0) {
	printk("\n");
      }
      printk("%p:%p ", (void *) (u_long *) (zflag) + nsize, (void *) *((u_long *) (zflag) + nsize));
    }
    printk("\n");
  }
#endif	/* 0 */
    elf_ex = *((struct elfhdr *) zflag);
    fsize = zsize;			/* max file size */
  } else {
#endif	/* CONFIG_BINFMT_ZELF */
    fsize = bprm->file->f_dentry->d_inode->i_size;	/* max file size */
    elf_ex = *((struct elfhdr *) bprm->buf);
#ifdef CONFIG_BINFMT_ZELF
  }
#endif	/* CONFIG_BINFMT_ZELF */

/* printk("elf_ex: e_type=0x%04x e_machine=0x%04x e_version=0x%08x e_entry=0x%08x e_phoff=0x%08x e_shoff=0x%08x e_flags=0x%08x e_ehsize=0x%04x e_phentsize=0x%04x e_phnum=0x%04x e_shentsize=0x%04x e_shnum=0x%04x e_shstrndx=0x%04x\n", ntohs(elf_ex.e_type), ntohs(elf_ex.e_machine), ntohl(elf_ex.e_version), ntohl(elf_ex.e_entry), ntohl(elf_ex.e_phoff), ntohl(elf_ex.e_shoff), ntohl(elf_ex.e_flags), ntohs(elf_ex.e_ehsize), ntohs(elf_ex.e_phentsize), ntohs(elf_ex.e_phnum), ntohs(elf_ex.e_shentsize), ntohs(elf_ex.e_shnum), ntohs(elf_ex.e_shstrndx)); */

/* First of all, some simple consistency checks */
  if (memcmp(elf_ex.e_ident, ELFMAG, SELFMAG) != 0) {
    DBG_ELF("e_ident: check failed\n");
    goto out_error;
  }
  switch (ntohs(elf_ex.e_type)) {
    case ET_REL:
      printk("e_type: ET_REL (relocatable file) is not allowed, must be executable with -q\n");
      goto out_error;
    case  ET_EXEC:
      break;
    case ET_DYN:
      printk("e_type: ET_DYN (shared object file -fpic) is not allowed, must be executable with -q\n");
      goto out_error;
    case ET_CORE:
      printk("e_type: ET_CORE (core file) is not allowed, must be executable with -q\n");
      goto out_error;
    default:
      printk("e_type: check failed %x (%d) != (%d || %d)\n", ntohs(elf_ex.e_type), ntohs(elf_ex.e_type), ET_EXEC, ET_DYN);
      goto out_error;
  }
  if (!elf_check_arch(&elf_ex)) {
    DBG_ELF("elf_check_arch: check failed\n");
/* 		goto out_error; */
/* Above commented out to allow any type through, might crash system. */
  }
  if (!bprm->file->f_op || !bprm->file->f_op->mmap) {
    DBG_ELF("bprm->file->f_op||!bprm->file->f_op->mmap: check failed\n");
    goto out_error;
  }
  old_fs = get_fs();
  set_fs(get_ds());
/* Read in section table entries.  For relocation information, and reginfo for gp register. */
  retval = -ENOMEM;
  size = ntohs(elf_ex.e_shentsize) * ntohs(elf_ex.e_shnum);
  if (size > PARTIAL_SIZE || size < 4) {
    DBG_ELF("elf_shdata size=%d (%d * %d)\n", size, ntohs(elf_ex.e_shentsize), ntohs(elf_ex.e_shnum));
    goto out_error;
  }
  fpos = ntohl(elf_ex.e_shoff);
#ifdef CONFIG_BINFMT_ZELF
  if (zflag == NULL) {	/* if not compressed and thus not in memory */
#endif	/* CONFIG_BINFMT_ZELF */
    elf_shdata = (Elf32_Shdr *) kmalloc(size, GFP_KERNEL);
    if (elf_shdata == NULL) {
      DBG_ELF("elf_shdata size=%d (%d * %d)\n", size, ntohs(elf_ex.e_shentsize), ntohs(elf_ex.e_shnum));
      goto out_error;
    }
/* printk("kmalloc (elf_shdata)=%x\n", (unsigned int)elf_shdata); */
    DBG_ELF("elf: read0(,e_shoff=%x, elf_shdata=%x, size=0x%x (%d))\n", ntohl(elf_ex.e_shoff), (unsigned int) elf_shdata, size, size);
    retval = bprm->file->f_op->read(bprm->file, (char *) elf_shdata, size, &fpos);
    if (retval < 0) {
      DBG_ELF("error, retval from read=%d\n", retval);
      goto out_error;
    }
#ifdef CONFIG_BINFMT_ZELF
  } else {
    elf_shdata = (Elf32_Shdr *)(zflag + fpos);
  }
#endif	/* CONFIG_BINFMT_ZELF */
/* Read the string tables before doing anything. */
  for (i = 0, elf_psh = elf_shdata; i < ntohs(elf_ex.e_shnum); i++, elf_psh++) {
/* printk("sh_type=0x%x (%d)\n", ntohl(elf_psh->sh_type),ntohl(elf_psh->sh_type)); */
    switch (ntohl(elf_psh->sh_type)) {
      case SHT_STRTAB:		/* A string table */
	size = ntohl(elf_psh->sh_size);
	fpos = ntohl(elf_psh->sh_offset);
	retval = -ENOEXEC;
	if (size > PARTIAL_SIZE || size < 4) {
	  printk("SHT_STRTAB bad size=%d\n", size);
	  goto out_error;
	}
#ifdef CONFIG_BINFMT_ZELF
	if (zflag == NULL) {	/* if not compressed and thus not in memory */
#endif	/* CONFIG_BINFMT_ZELF */
	  stmp = (char *) kmalloc(size, GFP_KERNEL);
	  if (!stmp) {
	    printk("strtab kmalloc failure, size=%d\n", size);
	    retval = -ENOMEM;
	    goto out_error;
	  }
/* printk("strtab: read(,sh_offset=%x, stmp=%x, size=0x%x (%d))\n", ntohl(elf_psh->sh_offset), (unsigned int) stmp, size, size); */
	  retval = bprm->file->f_op->read(bprm->file, (char *) stmp, size, &fpos);
	  if (retval < 0) {
	    if (shstrtab == NULL) {
	      printk("can not read shstrtab table, size=%d\n", size);
	      goto out_error;
	    }
	    kfree(stmp);
	    break;	/* strtab missing is ok */
	  }
#ifdef CONFIG_BINFMT_ZELF
	} else {
	  if (fpos >= fsize) {	/* strtab missing is ok */
/* printk("fpos >= fsize\n"); */
	    break;
	  }
	  stmp = (char *)(zflag + fpos);
	}
#endif	/* CONFIG_BINFMT_ZELF */
	if (i == ntohs(elf_ex.e_shstrndx)) {
	  shstrtab = stmp;
	  shstrtab_size = size;
/* printk("kmalloc (shstrtab)=%x\n", (unsigned int)shstrtab); */
/* printk("shstrtab_size=%x\n", shstrtab_size); */
	} else if (strtab == NULL) {
	  strtab = stmp;
	  strtab_size = size;
/* printk("kmalloc (strtab)=%x\n", (unsigned int)strtab); */
/* printk("strtab_size=%x\n", strtab_size); */
	} else {
/* 	  printk("Too many strtabs??\n"); */
#ifdef CONFIG_BINFMT_ZELF
	  if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	    kfree(stmp);
/* printk("kfree (stmp)=%x\n", (unsigned int)stmp); */
#ifdef CONFIG_BINFMT_ZELF
	  }
#endif	/* CONFIG_BINFMT_ZELF */
	}
	stmp = NULL;
      default:
	break;	/* don't do anything else */
    }
  }
/* Read in all of the Program header information */
  retval = -ENOMEM;
  size = ntohs(elf_ex.e_phentsize) * ntohs(elf_ex.e_phnum);
  if (size > PARTIAL_SIZE || size < 4) {
    DBG_ELF("elf_phdata size=%d (%d * %d)\n", size, ntohs(elf_ex.e_phentsize), ntohs(elf_ex.e_phnum));
    goto out_error;
  }
  fpos = ntohl(elf_ex.e_phoff);		/* location in file */
#ifdef CONFIG_BINFMT_ZELF
  if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
    elf_phdata = (struct elf_phdr *) kmalloc(size, GFP_KERNEL);
    if (!elf_phdata) {
      DBG_ELF("elf_phdata size=%d (%d * %d)\n", size, ntohs(elf_ex.e_phentsize), ntohs(elf_ex.e_phnum));
      goto out_error;
    }
/* printk("kmalloc (elf_phdata)=%x\n", (unsigned int)elf_phdata); */
    DBG_ELF("elf: read1(,e_phoff=%x, elf_phdata=%x, size=0x%x (%d))\n", ntohl(elf_ex.e_phoff), (unsigned int) elf_phdata, size, size);
    retval = bprm->file->f_op->read(bprm->file, (char *) elf_phdata, size, &fpos);
    if (retval < 0) {
      DBG_ELF("error retval=%d\n", retval);
      goto out_error;
    }
#ifdef CONFIG_BINFMT_ZELF
  } else {
    elf_phdata = (struct elf_phdr *)(zflag + fpos);
  }
#endif	/* CONFIG_BINFMT_ZELF */
  elf_brk = 0;
  start_code = (unsigned long) ntohl(elf_ex.e_entry);
  end_code = 0;
  start_data = 0;
  elf_gp = 0;				/* assume not used, error */
/* Flush all traces of the currently running executable */
  retval = flush_old_exec(bprm);
  if (retval) {
    DBG_ELF("error, from flush_old_exec retval=%d\n", retval);
    goto out_error;
  }
/* OK, This is the point of no return */
  current->mm->start_data = 0;
  current->mm->end_data = 0;
  current->mm->end_code = 0;
  current->mm->rss = 0;
  current->flags &= ~PF_FORKNOEXEC;
/* Do this immediately, since STACK_TOP as used in setup_arg_pages may depend on the personality.  */
  set_personality(PER_LINUX);
  for (i = 0, elf_ppnt = elf_phdata; i < ntohs(elf_ex.e_phnum); i++, elf_ppnt++) {
    int tmps;

    if (ntohl(elf_ppnt->p_type) != PT_LOAD) {
      continue;
    }
    tmps = ntohl(elf_ppnt->p_memsz) + ntohl(elf_ppnt->p_vaddr);
    if (tmps > l_size) {
      l_size = tmps;			/* largest address. */
    }
  }
  if (l_size == 0) {
    printk("elf: Unable to determine RAM size needed for process.\n");
    goto out_error;
  }
  elf_brk = l_size;
/*   l_size =  l_size + ELF_STACK_SIZE; */
/* We have to add the size of our arguments to our stack size otherwise it's
   too easy for users to create stack overflows by passing in a huge argument
   list.  And yes,  we have to be pedantic and include space for the argv/envp
   array as it may have a lot of entries.  */
  i = TOP_OF_ARGS - bprm->p;			/* the strings */
  i += (bprm->argc + 1) * sizeof(char *);	/* the argv array */
  i += (bprm->envc + 1) * sizeof(char *);	/* the envp array */
  i += ELF_STACK_SIZE;
DBG_ELF("stack_len=%d (0x%x)\n", i, i);
  l_size += i;
  l_size = ELF_PAGESTART(l_size + ELF_MIN_ALIGN - 1);
/* Get memory needed to hold loadable image. */
  down_write(&current->mm->mmap_sem);
  load_bias = do_mmap(0, 0, l_size, PROT_READ | PROT_WRITE | PROT_EXEC | MAP_ANON | MAP_PRIVATE, 0, 0);
  up_write(&current->mm->mmap_sem);
DBG_ELF("do_mmap(, 0, %lx (%ld) returned 0x%lx (%ld)\n", l_size, l_size, load_bias, load_bias);
  if (load_bias >= (unsigned long) -4096) {
     printk("Unable to allocate RAM for process, errno %d\n", (int) -load_bias);
     load_bias = 0;
    goto out_error;
  }
  elf_brk += load_bias;
  start_code += load_bias;
  DBG_ELF("elf_ex.e_phnum=%d\n", ntohs(elf_ex.e_phnum));
  for (i = 0, elf_ppnt = elf_phdata; i < ntohs(elf_ex.e_phnum); i++, elf_ppnt++) {
    DBG_ELF_SECTION("i=%d of (%d), type=%x, offset=%x, vaddr=%x, paddr=%x, filesz=%x, memsz=%x, flags=%x, align=%x\n",
		i, ntohs(elf_ex.e_phnum), ntohl(elf_ppnt->p_type), ntohl(elf_ppnt->p_offset),
		ntohl(elf_ppnt->p_vaddr), ntohl(elf_ppnt->p_paddr), ntohl(elf_ppnt->p_filesz),
		ntohl(elf_ppnt->p_memsz), ntohl(elf_ppnt->p_flags), ntohl(elf_ppnt->p_align));
    if (ntohl(elf_ppnt->p_type) != PT_LOAD) {
      continue;
    }
    if (ntohl(elf_ppnt->p_vaddr) != ntohl(elf_ppnt->p_paddr)) {
      if (ntohl(elf_ppnt->p_filesz) != 0) {	/* if nothing in file, ignore p_paddr */
	printk("elf: %s PT_LOAD p_vaddr(%x) != p_paddr(%x)\n", bprm->filename, ntohl(elf_ppnt->p_vaddr), ntohl(elf_ppnt->p_paddr));
      }
    }
    fpos = ntohl(elf_ppnt->p_offset);
    if (ntohl(elf_ppnt->p_filesz) != 0) {
#ifdef CONFIG_BINFMT_ZELF
      if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	DBG_ELF("elf: read4(,p_offset=%x, loading at=%x, size=0x%x (%d))\n", ntohl(elf_ppnt->p_offset), (unsigned int) load_bias + ntohl(elf_ppnt->p_vaddr), ntohl(elf_ppnt->p_filesz), ntohl(elf_ppnt->p_filesz));
	error = bprm->file->f_op->read(bprm->file, (char *) load_bias + ntohl(elf_ppnt->p_vaddr), ntohl(elf_ppnt->p_filesz), &fpos);
	if (error < 0) {
	  goto out_error;
	}
#ifdef CONFIG_BINFMT_ZELF
      } else {
	memcpy((char *) load_bias + ntohl(elf_ppnt->p_vaddr), zflag + fpos, ntohl(elf_ppnt->p_filesz));
      }
#endif	/* CONFIG_BINFMT_ZELF */
    }
    if (ntohl(elf_ppnt->p_filesz) < ntohl(elf_ppnt->p_memsz)) {	/* bss area to clear */
      char *start = (char *) load_bias + ntohl(elf_ppnt->p_vaddr) + ntohl(elf_ppnt->p_filesz);
      int lth = ntohl(elf_ppnt->p_memsz) - ntohl(elf_ppnt->p_filesz);

      bzero(start, lth);
    }
  }
  hi_mips_lth = MIPS_HI16_TABLE_SIZE;
  hi_mips = (unsigned long **) kmalloc(hi_mips_lth * sizeof(unsigned long), GFP_KERNEL);
  if (hi_mips == NULL) {
    DBG_ELF("kmalloc failed, hi_mips=%x, size=0x%lx (%ld)\n", (unsigned int)hi_mips, hi_mips_lth * sizeof(unsigned long), hi_mips_lth * sizeof(unsigned long));
    goto out_error;
  }
/* Go through section tables, get reginfo and relocatable symbols. */
  for (i = 0, elf_psh = elf_shdata; i < ntohs(elf_ex.e_shnum); i++, elf_psh++) {
    if (shstrtab != NULL) {
      DBG_ELF_SECTION("---%s: ", shstrtab+ntohl(elf_psh->sh_name));
    }
    DBG_ELF_SECTION("name=%x, type=%x, flags=%x, addr=%x, off=%x, size=%x, link=%x, info=%x, align=%x, entsize=%x\n",
	ntohl(elf_psh->sh_name), ntohl(elf_psh->sh_type),
	ntohl(elf_psh->sh_flags), ntohl(elf_psh->sh_addr), ntohl(elf_psh->sh_offset),
	ntohl(elf_psh->sh_size), ntohl(elf_psh->sh_link), ntohl(elf_psh->sh_info),
	ntohl(elf_psh->sh_addralign), ntohl(elf_psh->sh_entsize));
    switch (ntohl(elf_psh->sh_type)) {

      case SHT_MIPS_REGINFO:		/* register usage information */
/* 	DBG_ELF("SHT_MIPS_REGINFO\n"); */
	size = sizeof(Elf32_RegInfo);
	fpos = ntohl(elf_psh->sh_offset);
#ifdef CONFIG_BINFMT_ZELF
	if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	  elf_reginfo = (Elf32_RegInfo *) kmalloc(size, GFP_KERNEL);
/* printk("kmalloc (elf_reginfo)=%x\n", (unsigned int)elf_reginfo); */
	  if (!elf_reginfo) {
	    goto out_error;
	  }
	  error = bprm->file->f_op->read(bprm->file, (char *) elf_reginfo, size, &fpos);
	  if (error < 0) {
	    goto out_error;
	  }
#ifdef CONFIG_BINFMT_ZELF
	} else {
	  elf_reginfo = (Elf32_RegInfo *)(zflag + fpos);
	}
#endif	/* CONFIG_BINFMT_ZELF */
	elf_gp = ntohl(elf_reginfo->ri_gp_value);
	elf_gp += load_bias;
#ifdef CONFIG_BINFMT_ZELF
	if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	  kfree(elf_reginfo);
/* printk("kfree (elf_reginfo)=%x\n", (unsigned int)elf_reginfo); */
#ifdef CONFIG_BINFMT_ZELF
	}
#endif	/* CONFIG_BINFMT_ZELF */
	elf_reginfo = NULL;
	flags |= 1;			/* mark reginfo found. */
	break;

      case SHT_NULL:			/* Section header table entry unused */
/* 	DBG_ELF("SHT_NULL\n"); */
	break;

      case SHT_PROGBITS:		/* private */
/* 	DBG_ELF("SHT_PROGBITS "); */
	switch (ntohl(elf_psh->sh_flags)) {
	  case 0:
/* DBG_ELF("No memory, read-only\n"); */
	    break;
	  case 2:
/* DBG_ELF(".rodata\n"); */
	    break;
	  case 3:
/* DBG_ELF(".data\n"); */
	    break;
	  case 6:			/* 6 = relocatable .text (read-only) */
	  case 7:			/* 7 = executable .text (writable) */
/* DBG_ELF(".text\n"); */
	    break;
	  case 0x10000003:		/* GP-rel, alloc, writable */
/* DBG_ELF(".sdata\n"); */
	    break;
	  default:
            printk("SHT_PROGBITS, unknown flag types 0x%x\n", ntohl(elf_psh->sh_flags));
	    break;
	}
	break;

      case SHT_REL:			/* Relocation entries, no addends */
/* 	DBG_ELF("SHT_REL\n"); */
	if (shstrtab != NULL) {		/* only do known .rel tables. */
	  if (
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.init", 10) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.text", 10) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.rodata", 12) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.data", 10) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.fini", 10) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.ctors", 11) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.dtors", 11) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.got", 9) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.sdata", 11) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.sbss", 10) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.sdata2", 12) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.sbss2", 11) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.bss", 9) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.plt", 9) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.eh_frame", 14) != 0 &&
	      strncmp(shstrtab+ntohl(elf_psh->sh_name), ".rel.gcc_except_table", 22) != 0
	      ) {
#ifdef DEBUG
	    DBG_ELF("not doing it (%s).\n", shstrtab+ntohl(elf_psh->sh_name));
#endif	/* DEBUG */
	    break;
	  }
	}
	size = ntohl(elf_psh->sh_size);
	if (size > PARTIAL_SIZE || size < 4) {
	  printk( "SHT_REL bad size=%d\n", size);
	  goto out_error;
	}
	fpos = ntohl(elf_psh->sh_offset);
#ifdef CONFIG_BINFMT_ZELF
	if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	  relocs = (Elf32_Rel *) kmalloc(size, GFP_KERNEL);
	  if (relocs == NULL) {
	    printk( "REL kmalloc error, size=%d\n", size);
	    goto out_error;
	  }
/* printk("kmalloc (relocs)=%x\n", (unsigned int)relocs); */
	  DBG_ELF("REL: read(,sh_offset=%x, relocs=%x, size=0x%x (%d)  sh_entsize=%d)\n", ntohl(elf_psh->sh_offset), (unsigned int) relocs, size, size, ntohl(elf_psh->sh_entsize));
	  error = bprm->file->f_op->read(bprm->file, (char *) relocs, size, &fpos);
	  if (error < 0) {
	    printk("elf: error reading relocation table\n");
	    goto out_error;
	  }
#ifdef CONFIG_BINFMT_ZELF
	} else {
	  relocs = (Elf32_Rel *)(zflag + fpos);
	}
#endif	/* CONFIG_BINFMT_ZELF */
	size = size / ntohl(elf_psh->sh_entsize);
#ifdef DEBUG_LOTS
	DBG_ELF("size=%x (%d)\n", size, size);
#endif	/* DEBUG_LOTS */
	hi_mips_ix = 0;
	for (cnt = 0; cnt < size; cnt++) {
#ifdef DEBUG_LOTS
	  ptmp = (char *)(relocs + cnt);
	  for (tmp = 0; tmp < ntohl(elf_psh->sh_entsize); tmp++) {
	    DBG_ELF("%-2.2x", *((unsigned char *)ptmp + tmp));
	    if ((tmp % 4) == 3) {
	      DBG_ELF(" ");
	    }
	  }
#endif	/* DEBUG_LOTS */
	  tmp = ntohl((relocs+cnt)->r_info) & 0xff;
	  if (tmp >= R_MIPS_NUM) {
	    printk("relocation 0x%x (%d) out of range \n", tmp, tmp);
	  } else {
#ifdef DEBUG_LOTS
	    DBG_ELF("%s ", relocatename[tmp]);
#endif	/* DEBUG_LOTS */
	    ltmp = (unsigned long *)(ntohl((relocs+cnt)->r_offset) + load_bias);
#ifdef DEBUG_LOTS
	    DBG_ELF("ltmp=%x\n", (unsigned int)ltmp);
#endif	/* DEBUG_LOTS */
	    switch (tmp) {
	      case R_MIPS_32:
if (ntohl(*ltmp) == 0) {
#ifdef DEBUG_LOTS
  printk("  doing R_MIPS_32, contents=0, assuming undefined (maybe weak)\n");
#endif	/* DEBUG_LOTS */
} else {
		tmp = ntohl(*ltmp) + load_bias;
#ifdef DEBUG_LOTS
		DBG_ELF("R_MIPS_32, value=%x, load_bias=%lx, new=%x\n",
			ntohl(*ltmp), load_bias, tmp);
#endif	/* DEBUG_LOTS */
		*ltmp = htonl(tmp);
}
		break;

	      case R_MIPS_26:
		tmp = ntohl(*ltmp);
		offset = tmp & 0x03FFFFFFL;
		offset <<= 2;
		opcode = tmp & 0xFC000000L;
if (offset == 0) {
#ifdef DEBUG_LOTS
  printk("  doing R_MIPS_26, contents=0, assuming undefined (maybe weak)\n");
#endif	/* DEBUG_LOTS */
} else {
		offset += load_bias;
		offset &= 0x0FFFFFFFL;
		if (offset < (load_bias & 0x0FFFFFFFL)) {
		  printk("Error with R_MIPS_26 relocation, offset (0x%lx) < load_bias (0x%lx)\n",
			offset, load_bias & 0x0FFFFFFFL);
		  break;
		}
		offset >>= 2;
		tmp = opcode | (offset & 0x03FFFFFFL);
#ifdef DEBUG_LOTS
		DBG_ELF("R_MIPS_26, value=%x, load_bias=%lx, new=%x\n",
			ntohl(*ltmp), load_bias >> 2, tmp);
#endif	/* DEBUG_LOTS */
		*ltmp = htonl(tmp);
}
		break;

	      case R_MIPS_HI16:
		if (hi_mips_ix >= hi_mips_lth) {
		  unsigned long  **new_hi_mips;
		  hi_mips_lth *= 2;		/* increase size of table */
		  new_hi_mips = (unsigned long **) kmalloc(hi_mips_lth * sizeof(unsigned long), GFP_KERNEL);
		  if (new_hi_mips == NULL) {
		    DBG_ELF("kmalloc failed, new_hi_mips=%x, size=0x%lx (%ld)\n", (unsigned int)new_hi_mips, hi_mips_lth * sizeof(unsigned long), hi_mips_lth * sizeof(unsigned long));
		    goto out_error;
		  }
		  (void) memcpy((void *)new_hi_mips, hi_mips, hi_mips_lth * sizeof(unsigned long));
		  kfree(hi_mips);
		  hi_mips = new_hi_mips;
		}
		hi_mips[hi_mips_ix++] = ltmp;
		break;

	      case R_MIPS_LO16:
{ int flag = 0;
		tmp = ntohl(*ltmp);
		value_lo = ((tmp & 0x0000FFFFL) ^ 0x8000) - 0x8000;
		offset = load_bias + 0x18;
#ifdef	DEBUG_LOTS
		if (hi_mips_ix != 1)
			DBG_ELF("R_MIPS_LO16, numhigh=%d\n", hi_mips_ix);
#endif	/* DEBUG_LOTS */
		while (--hi_mips_ix >= 0) {
		  opcode = ntohl(*hi_mips[hi_mips_ix]);
		  offset = ((opcode & 0x0000FFFFL) << 16) + value_lo;
if (offset == 0) {
  if (flag == 0 || flag == 1) {
#ifdef DEBUG_LOTS
    printk("  doing R_MIPS_HI16, offset=0 (high & low!), assuming undefined (maybe weak)\n");
#endif	/* DEBUG_LOTS */
    flag = 1;
  } else {
    printk("  Drats. Assumption about weak and undefined being zero messed up -- contact BRECIS.\n");
  }
} else {
  if (flag == 1) {
    printk("  previous R_MIPS_HI16 was 0, this is not -- Drats. -- contact BRECIS.\n");
    flag = 2;
  }
		  offset += load_bias;
		  offset += 0x8000;	/* This corrects the high part */
		  if (offset < load_bias + 0x18)
		  	break;
		  opcode &= 0xFFFF0000L;
		  opcode |= (offset >> 16) & 0x0000FFFFL;
		  *hi_mips[hi_mips_ix] = htonl(opcode);
}
		}
if (flag == 1) {
#ifdef DEBUG_LOTS
  printk("  not doing R_MIPS_LO16, offset=0 (high & low!), assuming undefined (maybe weak)\n");
#endif	/* DEBUG_LOTS */
} else {
		if (offset < load_bias) {
		  printk("Error with R_MIPS_LO16 relocation, offset (0x%lx) < load_bias (0x%lx)\n",
			  offset, load_bias);
		  break;
		}
		offset = value_lo + load_bias;
#ifdef DEBUG_LOTS
		DBG_ELF("R_MIPS_LO16, value=%lx, load_bias=%lx\n",
			offset, load_bias);
#endif	/* DEBUG_LOTS */
		opcode = tmp & 0xFFFF0000L;
		offset &= 0x0000FFFFL;
		tmp = opcode | offset;
		*ltmp = htonl(tmp);
}
		hi_mips_ix = 0;
		break;
}

	      case R_MIPS_LITERAL:
	      case R_MIPS_GPREL16:
		break;			/* I believe these are already ok. */

	      default:
		printk("elf: unexpected relocation type 0x%x\n",
			ntohl((relocs+cnt)->r_info) & 0xff);
		break;
	    }
	  }
	}
	if (relocs != NULL) {
#ifdef CONFIG_BINFMT_ZELF
	  if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	    kfree(relocs);
/* printk("kfree (relocs)=%x\n", (unsigned int)relocs); */
#ifdef CONFIG_BINFMT_ZELF
	  }
#endif	/* CONFIG_BINFMT_ZELF */
	  relocs = NULL;
	}
	break;

      case SHT_RELA:			/* Relocation entries, with addends */
/* 	DBG_ELF("SHT_RELA\n"); */
	break;

      case SHT_NOBITS:			/* Section occupies no space in file */
/* 	DBG_ELF("SHT_NOBITS "); */
	switch (ntohl(elf_psh->sh_flags)) {
	  case 3:
/* DBG_ELF(".bss\n"); */
	    break;
	  case 0x10000003:
/* DBG_ELF(".sbss\n"); */
	    break;
	  default:
	    printk("elf: SHT_NOBITS unknown flag types (%x)\n", ntohl(elf_psh->sh_flags));
	    break;
	}
	break;

      case SHT_STRTAB:			/* A string table */
/* DBG_ELF("SHT_STRTAB\n"); */
	break;

      case SHT_SYMTAB:			/* A symbol table */
	if (strtab == NULL) {
	  break;			/* if no names */
	}
/* Need to find symbol _fdata value. */
/* #ifdef DEBUG_LOTS */
/* 	DBG_ELF("SHT_SYMTAB\n"); */
/* #endif */	/* DEBUG_LOTS */
	size_elf_symbols = ntohl(elf_psh->sh_size);
	if (size_elf_symbols > PARTIAL_SIZE || size_elf_symbols < 4) {
	  DBG_ELF("SHT_SYMTAB size_elf_symbols=%ld\n", size_elf_symbols);
	  goto out_error;
	}
/* NOTDONEYET */
	fpos = ntohl(elf_psh->sh_offset);
#ifdef CONFIG_BINFMT_ZELF
	if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	  elf_symbols = (Elf32_Sym *) kmalloc(size_elf_symbols, GFP_KERNEL);
	  if (!elf_symbols) {
	    DBG_ELF("elf_symbols size_elf_symbols=%ld\n", size_elf_symbols);
	    goto out_error;
	  }
/* printk("kmalloc (elf_symbols)=%x\n", (unsigned int)elf_symbols); */
#ifdef DEBUG_LOTS
	  DBG_ELF("elf_symbols: read(,sh_offset=%x, elf_symbols=%x, size_elf_symbols=0x%lx (%ld))\n", ntohl(elf_psh->sh_offset), (unsigned int) elf_symbols, size_elf_symbols, size_elf_symbols);
#endif	/* DEBUG_LOTS */
	  error = bprm->file->f_op->read(bprm->file, (char *) elf_symbols, size_elf_symbols, &fpos);
	  if (error < 0) {
	    printk("elf: error reading SHT_SYMTAB table");
	    goto out_error;
	  }
#ifdef CONFIG_BINFMT_ZELF
	} else {
	  elf_symbols = (Elf32_Sym *)(zflag + fpos);
	}
#endif	/* CONFIG_BINFMT_ZELF */
	size = size_elf_symbols / ntohl(elf_psh->sh_entsize);
	for (cnt = 0; cnt < size; cnt++) {
#ifdef DEBUG_LOTS
	  ptmp = (char *)(elf_symbols + cnt);
	  for (tmp = 0; tmp < ntohl(elf_psh->sh_entsize); tmp++) {	/* print in hex */
	    DBG_ELF("%-2.2x", *((unsigned char *)ptmp + tmp));
	    if ((tmp % 4) == 3) {
	      DBG_ELF(" ");
	    }
	  }
#endif	/* DEBUG_LOTS */
	  tmp = ntohl((elf_symbols+cnt)->st_name);
	  if (tmp < strtab_size) {
#ifdef DEBUG_LOTS
	    DBG_ELF(" %s", strtab + tmp);
#endif	/* DEBUG_LOTS */
	    if (strcmp(strtab + tmp, "_fdata") == 0) {
	      end_code = ntohl((elf_symbols+cnt)->st_value);
	      DBG_ELF("FOUND _fdata, value=%lx\n", end_code);
#ifndef DEBUG_LOTS
	      break;
#endif	/* !DEBUG_LOTS */
	    }
	  } else {
	    DBG_ELF("string not found, #=%x > strtab_size(%x)\n", ntohl((elf_symbols+cnt)->st_name),strtab_size);
	    break;
	  }
#ifdef DEBUG_LOTS
	  DBG_ELF("\n");
#endif	/* DEBUG_LOTS */
	}
	if (elf_symbols != NULL) {
#ifdef CONFIG_BINFMT_ZELF
	  if (zflag == NULL) {
#endif	/* CONFIG_BINFMT_ZELF */
	    kfree(elf_symbols);
/* printk("kfree (elf_symbols)=%x\n", (unsigned int)elf_symbols); */
#ifdef CONFIG_BINFMT_ZELF
	  }
#endif	/* CONFIG_BINFMT_ZELF */
	  elf_symbols = NULL;
	}
	break;

      case SHT_DYNAMIC:			/* Dynamic linking information */
/* DBG_ELF("SHT_DYNAMIC\n"); */
	break;

      case SHT_HASH:			/* Section header table entry unused */
/* DBG_ELF("SHT_HASH\n"); */
	printk("SHT_HASH table not expected, ignored\n");
	break;

      case SHT_DYNSYM:			/* Section header table entry unused */
/* DBG_ELF("SHT_DYNSYM\n"); */
	printk("SHT_DYNSYM table not expected, ignored\n");
	break;

      case SHT_MIPS_DWARF:		/* Section header table entry unused */
/* DBG_ELF("SHT_MIPS_DWARF\n"); */
	break;

      default:
	printk("unknown SHT_  =0x%x (%d)\n", ntohl(elf_psh->sh_type), ntohl(elf_psh->sh_type));
	break;
    }
  }
  if ((flags & 1) != 1) {
    printk("elf: error, did not find .reginfo section in binary.\n");
  }
  if (elf_gp == 0) {
    printk("elf: Error, gp register will be set to zero.\n");
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      elf_shdata != NULL) {
    kfree(elf_shdata);
/* printk("kfree (elf_shdata)=%x\n", (unsigned int)elf_shdata); */
    elf_shdata = NULL;
  }
  set_fs(old_fs);			/* done reading things */
  old_fs.seg = 0;
/* pic loading used to be here. */
  end_code += load_bias;
  start_data = end_code;

  DBG_ELF("start_code=%lx, elf_brk=%lx, end_code=%lx, start_data=%lx\n", start_code,elf_brk,end_code,start_data);

  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      elf_phdata != NULL) {
    kfree(elf_phdata);
/* printk("kfree (elf_phdata)=%x\n", (unsigned int)elf_phdata); */
    elf_phdata = NULL;
  }

  set_binfmt(&elf_format);

  compute_creds(bprm);
  current->flags &= ~PF_FORKNOEXEC;
/* N.B. passed_fileno might not be initialized? */
/*	if (interpreter_type == INTERPRETER_AOUT) */
/*		current->mm->arg_start += strlen(passed_fileno) + 1; */
  current->mm->start_brk = current->mm->brk = elf_brk;	/* used in mmnommu/mmap.c */
  top_memory = load_bias + l_size;		/* Always make run with the same amount of memory. */
  current->mm->end_brk = top_memory;		/* used in mmnommu/mmap.c */
  current->mm->end_code = end_code;		/* used in mmnommu/mmap.c, */
  current->mm->start_code = start_code;
  current->mm->start_data = start_data;		/* used in mmnommu/mmap.c */

  p = top_memory - 4;				/* note stack and brk overlap! */
  DBG_ELF("p=%x\n", (unsigned int) p);

/* copy the arg pages onto the stack, this could be more efficient :-) */
  for (i = TOP_OF_ARGS - 1; i >= bprm->p; i--) {
    unsigned char c;
    c = ((char *) page_address(bprm->page[i/PAGE_SIZE]))[i % PAGE_SIZE];
    *(char *)--p = c;
  }

  p = (unsigned long) create_elf_tables(p, bprm);
  DBG_ELF("SFA: p = %08lx\n", p);
  args_at = p & (~3UL);
  p = p & (~7UL);
  DBG_ELF("SFA: p = %08lx\n", p);
  DBG_ELF("SFA: args_at = %08lx\n", args_at);
  current->mm->start_stack = p;

  blast_hit_idcache(load_bias, ksize((void *) USER_2_KERN(load_bias)));

#if 0
  printk("stack:\n");
  for (i = 0; i < (top_memory-p) ; i += sizeof(unsigned int)) {
    if ((i % (8*sizeof(long))) == 0) {
      printk("\n%p:", (void *) (unsigned int *) (p + i));
    }
    printk(" %-8.8x", *(unsigned int *)(p+i));
  }
  printk("\n");
#endif	/* 0 */

#if 1
#ifndef DEBUG_WHICH
#define __LOAD_PRINT	DBG_ELF
#else	/* DEBUG_WHICH */
#define __LOAD_PRINT	printk
#endif	/* DEBUG_WHICH */
  __LOAD_PRINT("ELF (pid=%d) Load %s: TEXT=%x BSS=%x SP=%lx\n",
	 current->pid, bprm->filename,
	 (int) current->mm->start_code,
	 (int) current->mm->brk, current->mm->start_stack);
#endif
#if 1
  DBG_ELF("(start_code) %lx\n", (long) current->mm->start_code);
  DBG_ELF("*start_code=%p\n", (void *) (*(int *) current->mm->start_code));
  DBG_ELF("(end_code) %lx\n", (long) current->mm->end_code);
  DBG_ELF("(start_data) %lx\n", (long) current->mm->start_data);
  DBG_ELF("(start_brk) %lx\n", (long) current->mm->start_brk);
  DBG_ELF("(end_brk) %lx\n", (long) current->mm->end_brk);
  DBG_ELF("(start_stack) %lx\n", (long) current->mm->start_stack);
  DBG_ELF("(brk) %lx\n", (long) current->mm->brk);
  DBG_ELF("elf_gp = %p\n", (void *) elf_gp);
#endif

#ifdef ELF_PLAT_INIT
/*
 * The ABI may specify that certain registers be set up in special
 * ways (on i386 %edx is the address of a DT_FINI function, for
 * example).  This macro performs whatever initialization to
 * the regs structure is required.
 */
  ELF_PLAT_INIT(regs);
#endif

  regs->regs[4] = args_at;		/* A0 = SP, address of arguments in crt0 */
  regs->regs[25] = start_code;		/* T9 */
  regs->regs[28] = elf_gp;		/* GP */
  DBG_ELF("start_thread(regs=%p, start_code=0x%lx, start_stack=0x%lx)\n",
	  (void *) regs, current->mm->start_code, current->mm->start_stack);
  start_thread(regs, start_code, p);	/* status, epc, sp */
  DBG_ELF("A0=%lx, T9=%lx, GP=%lx, STATUS=%lx, EPC=%lx, SP=%lx\n", regs->regs[4], regs->regs[25], regs->regs[28], regs->cp0_status, regs->cp0_epc, regs->regs[29]);
  if (current->ptrace & PT_PTRACED)
    send_sig(SIGTRAP, current, 0);
  DBG_ELF("start_code=%p, *start_code=%p\n", (void *) start_code, (void *) (*(int *) start_code));
#if 0
/* Print out first longwords of what is in memory after load and relocate. */
  printk("program:\n");
  for (i = 0; i < ((unsigned long)current->mm->start_brk - load_bias); i += sizeof(long)) {
    if ((i % (8*sizeof(long))) == 0) {
      printk("\n%p:", (void *) (u_long *) (load_bias + i));
    }
    printk(" %p", (void *) *((u_long *) (load_bias + i)));
  }
  printk("\n");
#endif	/* 0 */
  retval = 0;
out:
/* Cleanup allocated memory. */
  if (old_fs.seg != 0) {
    set_fs(old_fs);			/* done reading things */
  }
  if (hi_mips != NULL) {
    kfree(hi_mips);
/* printk("kfree (hi_mips)=%x\n", (unsigned int)hi_mips); */
    hi_mips = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      elf_shdata != NULL) {
    kfree(elf_shdata);
/* printk("kfree (elf_shdata)=%x\n", (unsigned int)elf_shdata); */
    elf_shdata = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      stmp != NULL) {
    kfree(stmp);
/* printk("kfree (stmp)=%x\n", (unsigned int)stmp); */
    stmp = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      shstrtab != NULL) {
    kfree(shstrtab);
/* printk("kfree (shstrtab)=%x\n", (unsigned int)shstrtab); */
    shstrtab = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      strtab != NULL) {
    kfree(strtab);
/* printk("kfree (strtab)=%x\n", (unsigned int)strtab); */
    strtab = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      elf_phdata != NULL) {
    kfree(elf_phdata);
/* printk("kfree (elf_phdata)=%x\n", (unsigned int)elf_phdata); */
    elf_phdata = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      elf_reginfo != NULL) {
    kfree(elf_reginfo);
/* printk("kfree (elf_reginfo)=%x\n", (unsigned int)elf_reginfo); */
    elf_reginfo = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      relocs != NULL) {
    kfree(relocs);
/* printk("kfree (relocs)=%x\n", (unsigned int)relocs); */
    relocs = NULL;
  }
  if (
#ifdef CONFIG_BINFMT_ZELF
      zflag == NULL &&
#endif	/* CONFIG_BINFMT_ZELF */
      elf_symbols != NULL) {
    kfree(elf_symbols);
/* printk("kfree (elf_symbols)=%x\n", (unsigned int)elf_symbols); */
    elf_symbols = NULL;
  }
#ifdef CONFIG_BINFMT_ZELF
  if (zflag != NULL) {
/*     do_munmap(current->mm, (unsigned long)zflag, zsize); */
/* printk("do_munmap (zflag)=%x\n", (unsigned int)zflag); */
    kfree(zflag);
/* printk("kfree (zflag)=%x\n", (unsigned int)zflag); */
    zflag = NULL;
  }
#endif	/* CONFIG_BINFMT_ZELF */
  return (retval);

/* Error, cleanup loaded program memory. */
out_error:
  if (load_bias != 0) {
    do_munmap(current->mm, load_bias, l_size);
/* printk("do_munmap (load_bias)=%x\n", (unsigned int)load_bias); */
    load_bias = 0;
  }
  goto out;
}

#endif	/* CONFIG_BRECIS */

static int __init init_elf_binfmt(void)
{
	return register_binfmt(&elf_format);
}

static void __exit exit_elf_binfmt(void)
{
	/* Remove the COFF and ELF loaders. */
	unregister_binfmt(&elf_format);
}

module_init(init_elf_binfmt)
module_exit(exit_elf_binfmt)
MODULE_LICENSE("GPL");
