/****************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ****************************************************************************
 
    Module Name:
    mlme.c
 
    Abstract:
    Major MLME state machiones here
 
    Revision History:
    Who         When          What
    --------    ----------    ----------------------------------------------
    John Chang  08-04-2003    created for 11g soft-AP
 */

#include "rt_config.h"
#include <stdarg.h>

//
// RSSI threshold for TX rate switching - 
//                                        1   2   5.5   11   6    9    12   18   24   36   48   54  72   100 Mbps
CHAR RssiThresholdToDowngradeTxRate[]={ -100, -75, -70, -65, -40, -40, -40, -40, -40, -40, -40, -40, -40, -40}; // RATE_1 << -84 dB << RATE_2 << -80dB << RATE_5_5 << -75 dB << RATE_11
CHAR RssiThresholdToUpgradeTxRate[]=  {  -84, -84, -84, -84, -80, -80, -80, -75, -70, -70, -70, -70, -70, -70}; // RATE_1 >> -84 dB >> RATE_2 >> -84dB >> RATE_5_5 >> -80dB >> RATE_11

                                  //  1      2       5.5      11  
UCHAR Phy11BNextRateDownward[] = {RATE_1, RATE_1,   RATE_2,  RATE_5_5};
UCHAR Phy11BNextRateUpward[]   = {RATE_2, RATE_5_5, RATE_11, RATE_11};

                                  //  1      2       5.5      11        6        9        12      18       24       36       48       54
UCHAR Phy11BGNextRateDownward[]= {RATE_1, RATE_1,   RATE_2,  RATE_5_5,RATE_11,  RATE_6,  RATE_11, RATE_12, RATE_18, RATE_24, RATE_36, RATE_48};
UCHAR Phy11BGNextRateUpward[]  = {RATE_2, RATE_5_5, RATE_11, RATE_12, RATE_9,   RATE_12, RATE_18, RATE_24, RATE_36, RATE_48, RATE_54, RATE_54};

// 2560D and after has implemented ASIC-based OFDM rate switching, but not
// 2560C and before. thus software use different PER for rate switching
//                          RATE_1,  2, 5.5, 11,  6,  9, 12, 18, 24, 36, 48, 54
USHORT NewRateUpPER[]   = {    40,  40,  35, 20, 20, 20, 20, 16, 10, 16,  8,  6 }; // in percentage
USHORT NewRateDownPER[] = {    50,  50,  45, 45, 35, 35, 35, 35, 25, 25, 25, 12 }; // in percentage

USHORT OldRateUpPER[]   = {    40,  40,  40, 40, 30, 30, 30, 30, 20, 20, 10, 10 }; // in percentage
USHORT OldRateDownPER[] = {    45,  45,  45, 45, 35, 35, 35, 35, 25, 25, 25, 12 }; // in percentage


UCHAR RateIdToMbps[]     = { 1, 2, 5, 11, 6, 9, 12, 18, 24, 36, 48, 54, 72, 100};
USHORT RateIdTo500Kbps[] = {2, 4, 11, 22, 12, 18, 24, 36, 48, 72, 96, 108, 144, 200};

RTMP_RF_REGS RF2522RegTable[] = {
//      ch   R1          R2          R3(TX0~4=0) R4
        {1,  0x94002050, 0x940c1fda, 0x94000101, 0},
        {2,  0x94002050, 0x940c1fee, 0x94000101, 0},
        {3,  0x94002050, 0x940c2002, 0x94000101, 0},
        {4,  0x94002050, 0x940c2016, 0x94000101, 0},
        {5,  0x94002050, 0x940c202a, 0x94000101, 0},
        {6,  0x94002050, 0x940c203e, 0x94000101, 0},
        {7,  0x94002050, 0x940c2052, 0x94000101, 0},
        {8,  0x94002050, 0x940c2066, 0x94000101, 0},
        {9,  0x94002050, 0x940c207a, 0x94000101, 0},
        {10, 0x94002050, 0x940c208e, 0x94000101, 0},
        {11, 0x94002050, 0x940c20a2, 0x94000101, 0},
        {12, 0x94002050, 0x940c20b6, 0x94000101, 0},
        {13, 0x94002050, 0x940c20ca, 0x94000101, 0},
        {14, 0x94002050, 0x940c20fa, 0x94000101, 0}
};
#define	NUM_OF_2522_CHNL	(sizeof(RF2522RegTable) / sizeof(RTMP_RF_REGS))

RTMP_RF_REGS RF2523RegTable[] = {
//      ch   R1          R2          R3(TX0~4=0) R4
        {1,  0x94022010, 0x94000c9e, 0x940e0111, 0x94000a1b},
        {2,  0x94022010, 0x94000ca2, 0x940e0111, 0x94000a1b},
        {3,  0x94022010, 0x94000ca6, 0x940e0111, 0x94000a1b},
        {4,  0x94022010, 0x94000caa, 0x940e0111, 0x94000a1b},
        {5,  0x94022010, 0x94000cae, 0x940e0111, 0x94000a1b},
        {6,  0x94022010, 0x94000cb2, 0x940e0111, 0x94000a1b},
        {7,  0x94022010, 0x94000cb6, 0x940e0111, 0x94000a1b},
        {8,  0x94022010, 0x94000cba, 0x940e0111, 0x94000a1b},
        {9,  0x94022010, 0x94000cbe, 0x940e0111, 0x94000a1b},
        {10, 0x94022010, 0x94000d02, 0x940e0111, 0x94000a1b},
        {11, 0x94022010, 0x94000d06, 0x940e0111, 0x94000a1b},
        {12, 0x94022010, 0x94000d0a, 0x940e0111, 0x94000a1b},
        {13, 0x94022010, 0x94000d0e, 0x940e0111, 0x94000a1b},
        {14, 0x94022010, 0x94000d1a, 0x940e0111, 0x94000a03}
#if 0
        {1,  0x94022050, 0x940c1fda, 0x940e8101, 0},
        {2,  0x94022050, 0x940c1fee, 0x940e8101, 0},
        {3,  0x94022050, 0x940c2002, 0x940e8101, 0},
        {4,  0x94022050, 0x940c2016, 0x940e8101, 0},
        {5,  0x94022050, 0x940c202a, 0x940e8101, 0},
        {6,  0x94022050, 0x940c203e, 0x940e8101, 0},
        {7,  0x94022050, 0x940c2052, 0x940e8101, 0},
        {8,  0x94022050, 0x940c2066, 0x940e8101, 0},
        {9,  0x94022050, 0x940c207a, 0x940e8101, 0},
        {10, 0x94022050, 0x940c208e, 0x940e8101, 0},
        {11, 0x94022050, 0x940c20a2, 0x940e8101, 0},
        {12, 0x94022050, 0x940c20b6, 0x940e8101, 0},
        {13, 0x94022050, 0x940c20ca, 0x940e8101, 0},
        {14, 0x94022050, 0x940c20fa, 0x940e8101, 0}
#endif
};
#define	NUM_OF_2523_CHNL	(sizeof(RF2523RegTable) / sizeof(RTMP_RF_REGS))

RTMP_RF_REGS RF2524RegTable[] = {
//      ch   R1          R2          R3(TX0~4=0) R4
        {1,  0x94032020, 0x94080c9e, 0x94000101, 0x94000a1b},
        {2,  0x94032020, 0x94080ca2, 0x94000101, 0x94000a1b},
        {3,  0x94032020, 0x94080ca6, 0x94000101, 0x94000a1b},
        {4,  0x94032020, 0x94080caa, 0x94000101, 0x94000a1b},
        {5,  0x94032020, 0x94080cae, 0x94000101, 0x94000a1b},
        {6,  0x94032020, 0x94080cb2, 0x94000101, 0x94000a1b},
        {7,  0x94032020, 0x94080cb6, 0x94000101, 0x94000a1b},
        {8,  0x94032020, 0x94080cba, 0x94000101, 0x94000a1b},
        {9,  0x94032020, 0x94080cbe, 0x94000101, 0x94000a1b},
        {10, 0x94032020, 0x94080d02, 0x94000101, 0x94000a1b},
        {11, 0x94032020, 0x94080d06, 0x94000101, 0x94000a1b},
        {12, 0x94032020, 0x94080d0a, 0x94000101, 0x94000a1b},
        {13, 0x94032020, 0x94080d0e, 0x94000101, 0x94000a1b},
        {14, 0x94032020, 0x94080d1a, 0x94000101, 0x94000a03}
};
#define	NUM_OF_2524_CHNL	(sizeof(RF2524RegTable) / sizeof(RTMP_RF_REGS))
            
RTMP_RF_REGS RF2525RegTable[] = {
//      ch   R1          R2          R3(TX0~4=0) R4
        {1,  0x94022020, 0x94080c9e, 0x94060111, 0x94000a1b},
        {2,  0x94022020, 0x94080ca2, 0x94060111, 0x94000a1b},
        {3,  0x94022020, 0x94080ca6, 0x94060111, 0x94000a1b},
        {4,  0x94022020, 0x94080caa, 0x94060111, 0x94000a1b},
        {5,  0x94022020, 0x94080cae, 0x94060111, 0x94000a1b},
        {6,  0x94022020, 0x94080cb2, 0x94060111, 0x94000a1b},
        {7,  0x94022020, 0x94080cb6, 0x94060111, 0x94000a1b},
        {8,  0x94022020, 0x94080cba, 0x94060111, 0x94000a1b},
        {9,  0x94022020, 0x94080cbe, 0x94060111, 0x94000a1b},
        {10, 0x94022020, 0x94080d02, 0x94060111, 0x94000a1b},
        {11, 0x94022020, 0x94080d06, 0x94060111, 0x94000a1b},
        {12, 0x94022020, 0x94080d0a, 0x94060111, 0x94000a1b},
        {13, 0x94022020, 0x94080d0e, 0x94060111, 0x94000a1b},
        {14, 0x94022020, 0x94080d1a, 0x94060111, 0x94000a03}
};
#define	NUM_OF_2525_CHNL	(sizeof(RF2525RegTable) / sizeof(RTMP_RF_REGS))

RTMP_RF_REGS RF5222RegTable[] = {
//      ch   R1          R2          R3(TX0~4=0) R4
        {1,  0x94022020, 0x94001136, 0x94000101, 0x94000a0b},
        {2,  0x94022020, 0x9400113a, 0x94000101, 0x94000a0b},
        {3,  0x94022020, 0x9400113e, 0x94000101, 0x94000a0b},
        {4,  0x94022020, 0x94001182, 0x94000101, 0x94000a0b},
        {5,  0x94022020, 0x94001186, 0x94000101, 0x94000a0b},
        {6,  0x94022020, 0x9400118a, 0x94000101, 0x94000a0b},
        {7,  0x94022020, 0x9400118e, 0x94000101, 0x94000a0b},
        {8,  0x94022020, 0x94001192, 0x94000101, 0x94000a0b},
        {9,  0x94022020, 0x94001196, 0x94000101, 0x94000a0b},
        {10, 0x94022020, 0x9400119a, 0x94000101, 0x94000a0b},
        {11, 0x94022020, 0x9400119e, 0x94000101, 0x94000a0b},
        {12, 0x94022020, 0x940011a2, 0x94000101, 0x94000a0b},
        {13, 0x94022020, 0x940011a6, 0x94000101, 0x94000a0b},
        {14, 0x94022020, 0x940011ae, 0x94000101, 0x94000a1b},
#if 0
        // still lack of MMAC(Japan) ch 34,38,42,46
        
        {36, 0x94022010, 0x94018896, 0x94000101, 0x94000a1f},
        {40, 0x94022010, 0x9401889a, 0x94000101, 0x94000a1f},
        {44, 0x94022010, 0x9401889e, 0x94000101, 0x94000a1f},
        {48, 0x94022010, 0x940188a2, 0x94000101, 0x94000a1f},
        {52, 0x94022010, 0x940188a6, 0x94000101, 0x94000a1f},
        {66, 0x94022010, 0x940188aa, 0x94000101, 0x94000a1f},
        {60, 0x94022010, 0x940188ae, 0x94000101, 0x94000a1f},
        {64, 0x94022010, 0x940188b2, 0x94000101, 0x94000a1f},
        
        {100, 0x94022010, 0x94008802, 0x94000101, 0x94000a0f},
        {104, 0x94022010, 0x94008806, 0x94000101, 0x94000a0f},
        {108, 0x94022010, 0x9400880a, 0x94000101, 0x94000a0f},
        {112, 0x94022010, 0x9400880e, 0x94000101, 0x94000a0f},
        {116, 0x94022010, 0x94008812, 0x94000101, 0x94000a0f},
        {120, 0x94022010, 0x94008816, 0x94000101, 0x94000a0f},
        {124, 0x94022010, 0x9400881a, 0x94000101, 0x94000a0f},
        {128, 0x94022010, 0x9400881e, 0x94000101, 0x94000a0f},
        {132, 0x94022010, 0x94008822, 0x94000101, 0x94000a0f},
        {136, 0x94022010, 0x94008826, 0x94000101, 0x94000a0f},
        {140, 0x94022010, 0x9400882a, 0x94000101, 0x94000a0f},
        
        {149, 0x94022020, 0x940090a6, 0x94000101, 0x94000a07},
        {153, 0x94022020, 0x940090ae, 0x94000101, 0x94000a07},
        {157, 0x94022020, 0x940090b6, 0x94000101, 0x94000a07},
        {161, 0x94022020, 0x940090be, 0x94000101, 0x94000a07}
#endif
};
#define	NUM_OF_5222_CHNL	(sizeof(RF5222RegTable) / sizeof(RTMP_RF_REGS))

/*
    ==========================================================================
    Description:
        initialize the MLME state machine and its data structure (queue, spinlock, 
        timer, state machines).
    Return:
        always return NDIS_STATUS_SUCCESS
    ==========================================================================
*/
NDIS_STATUS MlmeInit(
    IN PRTMP_ADAPTER pAd) 
{
    NDIS_STATUS Status = NDIS_STATUS_SUCCESS;

    DBGPRINT(RT_DEBUG_TRACE, "--> MLME Initialize\n");
    
    do 
    {
        Status = MlmeQueueInit(&pAd->Mlme.Queue);
        if(Status != NDIS_STATUS_SUCCESS) 
        {
            break;
        }

        NdisAllocateSpinLock(&pAd->Mlme.TaskLock);
        pAd->Mlme.Running = FALSE;
        pAd->Mlme.TimBitmap = 0;
        pAd->Mlme.DtimCount = 0;
        pAd->Mlme.DtimPeriod = DEFAULT_DTIM_PERIOD;

        // init state machines
        AssocStateMachineInit(pAd, &pAd->Mlme.AssocMachine, pAd->Mlme.AssocFunc);
        AuthStateMachineInit(pAd, &pAd->Mlme.AuthMachine, pAd->Mlme.AuthFunc);
        AuthRspStateMachineInit(pAd, &pAd->Mlme.AuthRspMachine, pAd->Mlme.AuthRspFunc);
        SyncStateMachineInit(pAd, &pAd->Mlme.SyncMachine, pAd->Mlme.SyncFunc);
        WpaStateMachineInit(pAd,&pAd->Mlme.WpaMachine,pAd->Mlme.WpaFunc);

        // Since we are using switch/case to implement it, the init is different from the above
        // state machine init
        // MlmeCntlInit(pAd, &pAd->Mlme.CntlMachine, NULL);

        // Regularly check the timer
        init_timer(&pAd->Mlme.PeriodicTimer);
        pAd->Mlme.PeriodicTimer.expires = jiffies + MLME_TASK_EXEC_INTV;
        pAd->Mlme.PeriodicTimer.data = (unsigned long)pAd;
        pAd->Mlme.PeriodicTimer.function = &MlmePeriodicExec;
        add_timer(&pAd->Mlme.PeriodicTimer);

        if (pAd->PortCfg.LedMode == LED_MODE_TXRX_ACTIVITY)
        {
            init_timer(&pAd->PortCfg.LedCntl.BlinkTimer);
            pAd->PortCfg.LedCntl.BlinkTimer.expires = jiffies + (255 * HZ)/1000;
            pAd->PortCfg.LedCntl.BlinkTimer.data = (unsigned long)pAd;
            pAd->PortCfg.LedCntl.BlinkTimer.function = &AsicLedPeriodicExec;
            add_timer(&pAd->PortCfg.LedCntl.BlinkTimer);
        }

		// software-based RX Antenna diversity
		init_timer(&pAd->PortCfg.RxAnt.RxAntDiversityTimer);
        pAd->PortCfg.RxAnt.RxAntDiversityTimer.data = (unsigned long)pAd;
        pAd->PortCfg.RxAnt.RxAntDiversityTimer.function = &AsicRxAntEvalTimeout;

    } while (FALSE);

    DBGPRINT(RT_DEBUG_TRACE, "<-- MLME Initialize\n");

    return Status;
}

/*
    ==========================================================================
    Description:
        main loop of the MLME
    Pre:
        Mlme has to be initialized, and there are something inside the queue
    Note:
        This function is invoked from MPSetInformation and MPReceive;
        This task guarantee only one MlmeHandler will run. 
    ==========================================================================
 */
VOID MlmeHandler(
    IN PRTMP_ADAPTER pAd) 
{
    MLME_QUEUE_ELEM        *Elem = NULL;

    // Only accept MLME and Frame from peer side, no other (control/data) frame should
    // get into this state machine

    NdisAcquireSpinLock(&pAd->Mlme.TaskLock);
    if(pAd->Mlme.Running) 
    {
        NdisReleaseSpinLock(&pAd->Mlme.TaskLock);
        return;
    } 
    else 
    {
        pAd->Mlme.Running = TRUE;
    }
    NdisReleaseSpinLock(&pAd->Mlme.TaskLock);

    while (!MlmeQueueEmpty(&pAd->Mlme.Queue)) 
    {
        //From message type, determine which state machine I should drive
        if (MlmeDequeue(&pAd->Mlme.Queue, &Elem)) 
        {
            // if dequeue success
            switch (Elem->Machine) 
            {
                case ASSOC_STATE_MACHINE:
                    StateMachinePerformAction(pAd, &pAd->Mlme.AssocMachine, Elem);
                    break;
                case AUTH_STATE_MACHINE:
                    StateMachinePerformAction(pAd, &pAd->Mlme.AuthMachine, Elem);
                    break;
                case AUTH_RSP_STATE_MACHINE:
                    StateMachinePerformAction(pAd, &pAd->Mlme.AuthRspMachine, Elem);
                    break;
                case SYNC_STATE_MACHINE:
                    StateMachinePerformAction(pAd, &pAd->Mlme.SyncMachine, Elem);
                    break;
//              case MLME_CNTL_STATE_MACHINE:
//                  MlmeCntlMachinePerformAction(pAd, &pAd->Mlme.CntlMachine, Elem);
//                  break;
                case WPA_STATE_MACHINE:
                    StateMachinePerformAction(pAd, &pAd->Mlme.WpaMachine, Elem);
                    break;
                default:
                    DBGPRINT(RT_DEBUG_TRACE, "ERROR: Illegal machine in MlmeHandler()\n");
                    break;
            } // end of switch

            // free MLME element
            Elem->Occupied = FALSE;
            Elem->MsgLen = 0;
            
        }
        else {
            DBGPRINT(RT_DEBUG_ERROR, "ERROR: empty Elem in MlmeQueue\n");
        }
    }

    NdisAcquireSpinLock(&pAd->Mlme.TaskLock);
    pAd->Mlme.Running = FALSE;
    NdisReleaseSpinLock(&pAd->Mlme.TaskLock);
}

/*
    ==========================================================================
    Description:
        Destructor of MLME (Destroy queue, state machine, spin lock and timer)
    Parameters:
        Adapter - NIC Adapter pointer
    Post:
        The MLME task will no longer work properly
    ==========================================================================
 */
VOID MlmeHalt(
    IN PRTMP_ADAPTER pAd) 
{
    DBGPRINT(RT_DEBUG_TRACE, "==> MlmeHalt\n");

    del_timer_sync(&pAd->Mlme.PeriodicTimer);
	del_timer_sync(&pAd->PortCfg.RxAnt.RxAntDiversityTimer);
    if (pAd->PortCfg.LedMode == LED_MODE_TXRX_ACTIVITY)
        del_timer_sync(&pAd->PortCfg.LedCntl.BlinkTimer);

    MlmeQueueDestroy(&pAd->Mlme.Queue);

	DBGPRINT(RT_DEBUG_TRACE, "<== MlmeHalt\n");
}

/*
    ==========================================================================
    Description:
        Confirmation function for outgoing MLME packets. The buffer is released
    Parameters:
        Adapter - NIC Adapter pointer
        Msg     - Message returned from ASIC
        MsgLen  - message length, not used in this implementation
    Pre:
        The Msg has to be allocated by NdisAllocateMemoryWithTag
    ==========================================================================
 */
VOID MlmeConfirm(
    IN PRTMP_ADAPTER pAd, 
    IN VOID *Msg, 
    IN INT MsgLen) 
{
    kfree(Msg);
}


/*
    ==========================================================================
    Description:
        This routine is executed every second -
        1. Decide the overall channel quality
        2. Check if need to upgrade the TX rate to any client
        3. perform MAC table maintenance, including ageout no-traffic clients, 
           and release packet buffer in PSQ is fail to TX in time.
    ==========================================================================
 */
VOID MlmePeriodicExec(
    IN	unsigned long data) 
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    pAd->Mlme.Now32 = jiffies;

    // check every 12 second. If no U2M in the past 12 second, then AvgRSSI is no longer a 
    // valid indication of the distance between this AP and its clients. In this case, we presume a
    // mid AvgRSSI (say -60 dbm), so that no extreme TX power calibration and BBP R17 tuning
    // rules will be applied.
    if (pAd->Mlme.PeriodicRound % 12 == 2)
    {
        if (pAd->PortCfg.NumOfAvgRssiSample == 0)
        {
            pAd->PortCfg.AvgRssi = RSSI_TO_DBM_OFFSET - 60;
            DBGPRINT(RT_DEBUG_TRACE,"MlmePeriodicExec: no traffic, reset Avg RSSI= %d dbm\n",pAd->PortCfg.AvgRssi - RSSI_TO_DBM_OFFSET);
        }
        else
            pAd->PortCfg.NumOfAvgRssiSample = 0;
    }

    AsicAdjustTxPower(pAd);

	// if Rx Antenna is DIVERSITY ON, then perform Software-based diversity evaluation
    if ((pAd->PortCfg.CurrentRxAntenna == 0xff) && (pAd->Mlme.PeriodicRound % 4 ==1))
        AsicEvaluateSecondaryRxAnt(pAd);
    
	// danamic tune BBP R17 to find a balance between sensibility and noise isolation
    // 2003-12-05 to avoid collision with MAC ASIC, perform R17 tuning only within 20
    //            second after AP startup
    if ((pAd->PortCfg.Rt2560Version >= RT2560_VER_D) || (pAd->Mlme.PeriodicRound < 20))
        AsicBbpTuning(pAd);

    // walk through MAC table, see if switching TX rate is required
    if (pAd->PortCfg.EnableAutoRateSwitching)
        MlmeCheckDynamicTxRateSwitching(pAd);

    // MAC table maintenance
    MacTableMaintenance(pAd);

	pAd->Mlme.PeriodicRound++;
    pAd->Mlme.PeriodicTimer.expires = jiffies + MLME_TASK_EXEC_INTV;
    add_timer(&pAd->Mlme.PeriodicTimer);
}

/*
    ==========================================================================
    Description:
        This routine calculates TxPER, RxPER of the past N-sec period. 
        ChannelQuality is also calculated here.
    Output:
        Mlme.ChannelQuality - 0..100
    ==========================================================================
 */
VOID MlmeCheckChannelQuality(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Now32)
{
    ULONG RtsFailCnt, TxFailCnt, TxOkCnt, TxRetryCnt, TxCnt, TxPER;
    ULONG RxFailCnt, RxOkCnt, RxCnt, RxPER, TxRetryRatio, Cnt0;

    //
    // calculate TX PER
    //
    TxFailCnt     = pAd->WlanCounters.FailedCount - 
                    pAd->Mlme.PrevWlanCounters.FailedCount;
    RtsFailCnt    = pAd->WlanCounters.RTSFailureCount - 
                    pAd->Mlme.PrevWlanCounters.RTSFailureCount;
    TxRetryCnt	  = pAd->WlanCounters.RetryCount - 
                    pAd->Mlme.PrevWlanCounters.RetryCount;
    TxOkCnt       = pAd->WlanCounters.TransmittedFragmentCount - 
                    pAd->Mlme.PrevWlanCounters.TransmittedFragmentCount;
    TxCnt = TxOkCnt + TxFailCnt;

    if (TxCnt)    
        pAd->Mlme.TxPER = (TxFailCnt * 100) / TxCnt;
    else
        pAd->Mlme.TxPER = 0;
    
    if (TxCnt < 10) // if less than 10 TX sample, skip TX related statistics
    {
        TxPER = 0;  // don't take TxPER into CQI consideration if too few sample
        TxRetryRatio = 0;
    }
    else 
    {
        TxPER = pAd->Mlme.TxPER;
        TxRetryRatio = ((TxRetryCnt + TxFailCnt) * 100) / TxCnt;
    }

    //
    // calculate RX PER
    //

	// Update FCS counters
    RTMP_IO_READ32(pAd, CNT0, &Cnt0);
    pAd->WlanCounters.FCSErrorCount += Cnt0;

    RxOkCnt   = pAd->WlanCounters.ReceivedFragmentCount - 
                pAd->Mlme.PrevWlanCounters.ReceivedFragmentCount;
    RxFailCnt = pAd->WlanCounters.FCSErrorCount - 
                pAd->Mlme.PrevWlanCounters.FCSErrorCount;
    RxCnt = RxOkCnt + RxFailCnt;

    if (RxCnt)
        pAd->Mlme.RxPER = (RxFailCnt * 100) / RxCnt;
    else
        pAd->Mlme.RxPER = 0;
    
    if (RxCnt < 10)
        RxPER = 0;  // don't take RxPER into ChannelQuality consideration if too few sample
    else
        RxPER = pAd->Mlme.RxPER;

    //
    // ChannelQuality = W1*RSSI + W2*TxRatryRatio + W3*RxPER    (RSSI 0..100), (TxPER 100..0), (RxPER 100..0)
    //
    {
        pAd->Mlme.ChannelQuality = (RSSI_WEIGHTING * pAd->PortCfg.LastRssi + 
                             TX_WEIGHTING * (100 - TxRetryRatio) + 
                             RX_WEIGHTING* (100 - RxPER)) / 100;
        if (pAd->Mlme.ChannelQuality >= 100)
            pAd->Mlme.ChannelQuality = 100;
    }
    
    DBGPRINT(RT_DEBUG_INFO, "MMCHK - CQI= %d, (Tx=%d/%d/%d, Rx=%d/%d, RSSI=%d)\n", 
        pAd->Mlme.ChannelQuality, TxFailCnt, TxRetryCnt, TxCnt, RxFailCnt, RxCnt, pAd->PortCfg.LastRssi);

    // latch current WLAN counters for next check-for-roaming usage
    NdisMoveMemory(&pAd->Mlme.PrevWlanCounters, &pAd->WlanCounters, sizeof(COUNTER_802_11));

}

/*
    ==========================================================================
    Description:
        This routine walks through the MAC table, see if TX rate change is 
        required for each associated client. 
    Output:
        pEntry->CurrTxRate - 
    NOTE:
        call this routine every second
    ==========================================================================
 */
VOID MlmeCheckDynamicTxRateSwitching(
    IN PRTMP_ADAPTER pAd)
{
    int i;
    MAC_TABLE_ENTRY *pEntry;
    UCHAR UpRate, DownRate, CurrRate;
    USHORT  *pRateUpPER, *pRateDownPER;
    
    // 2560D and after has implemented ASIC-based OFDM rate switching,
    // but not 2560C abd before. thus software use different PER for rate switching
    if (pAd->PortCfg.Rt2560Version >= RT2560_VER_D)
    {
        pRateUpPER = &NewRateUpPER[0];
        pRateDownPER = &NewRateDownPER[0];
    }
    else
    {
        pRateUpPER = &OldRateUpPER[0];
        pRateDownPER = &OldRateDownPER[0];
    }

	//
    // walk through MAC table, see if need to change AP's TX rate toward each entry
    //
    for (i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
    {
    	USHORT TxTotalCnt, TxErrorRatio;
        BOOLEAN fUpgradeQuality = FALSE;
    	
        pEntry = &pAd->MacTab.Content[i];

        // only associated STA counts
        if ((pEntry->Valid == FALSE) || (pEntry->Sst != SST_ASSOC))
            continue;

        TxTotalCnt = pEntry->OneSecTxOkCount + pEntry->OneSecTxRetryOkCount + pEntry->OneSecTxFailCount;
        
        // skip those STA that has no traffic in the past period
        if (TxTotalCnt == 0)
            continue;
            
        // decide the next upgrade rate and downgrade rate, if any
        CurrRate = pEntry->CurrTxRate;
        if (pEntry->MaxSupportedRate < RATE_FIRST_OFDM_RATE)
        {
            UpRate = Phy11BNextRateUpward[CurrRate];
            DownRate = Phy11BNextRateDownward[CurrRate];
        }
        else 
        {
            UpRate = Phy11BGNextRateUpward[CurrRate];
            DownRate = Phy11BGNextRateDownward[CurrRate];
        }

        // calculate Tx with retry ratio when enough samples are available
        if (TxTotalCnt > 15)
        {
            TxErrorRatio = ((pEntry->OneSecTxRetryOkCount + pEntry->OneSecTxFailCount) *100) / TxTotalCnt;

            // downgrade TX quality if retry+error ratio reached
            if (((CurrRate >= RATE_24) && (pEntry->OneSecTxFailCount >=2)) ||
                (TxErrorRatio >= pRateDownPER[CurrRate]))
            {
                pEntry->TxQuality[CurrRate] += DRS_TX_QUALITY_WORST_BOUND;   // degrade quality
                if (pEntry->TxQuality[CurrRate] > DRS_TX_QUALITY_WORST_BOUND)
                    pEntry->TxQuality[CurrRate] = DRS_TX_QUALITY_WORST_BOUND;
            }
            // upgrade TX quality if retry+error ratio reached
            else if (TxErrorRatio <= pRateUpPER[CurrRate])
            {
                fUpgradeQuality = TRUE;
                pEntry->TxQuality[DownRate] = 0;     // next DOWN rate's quality should also very good
                if (pEntry->TxQuality[CurrRate])
                    pEntry->TxQuality[CurrRate] --;  // quality very good in CurrRate
                
                if (pEntry->TxRateUpPenalty)
                    pEntry->TxRateUpPenalty --;
                else if (pEntry->TxQuality[UpRate])
                    pEntry->TxQuality[UpRate] --;    // may improve next UP rate's quality
            }
        }

        // if not enough TX samples, decide by heuristic rules
        else
        {
            TxErrorRatio = 0;  // too few samples
            
            // Downgrade TX quality upon any TX failure in the past second
            if (pEntry->OneSecTxFailCount)
            {
                if ((pEntry->OneSecTxFailCount <= 1) && (pEntry->OneSecTxOkCount + pEntry->OneSecTxRetryOkCount))
                {
                    pEntry->TxQuality[CurrRate] += 2;  // degrade quality
                    if (pEntry->TxQuality[CurrRate] > DRS_TX_QUALITY_WORST_BOUND)
                        pEntry->TxQuality[CurrRate] = DRS_TX_QUALITY_WORST_BOUND;
                }
                else // more than 2 failure, or no TX ok cases
                {
                    pEntry->TxQuality[CurrRate] += 4;   // quality bad
                    if (pEntry->TxQuality[CurrRate] > DRS_TX_QUALITY_WORST_BOUND)
                        pEntry->TxQuality[CurrRate] = DRS_TX_QUALITY_WORST_BOUND;
                    
                    pEntry->TxQuality[UpRate] ++;     // quality must not good for UP rate
                    if (pEntry->TxQuality[UpRate] > DRS_TX_QUALITY_WORST_BOUND)
                        pEntry->TxQuality[UpRate] = DRS_TX_QUALITY_WORST_BOUND;
                }
            }
        
            // upgrade TX quality if -
            // 1. no TX failure but do have TX ok case, and
            // 2. there's more one-time-ok cases than retry-ok cases in the past second
            else if (pEntry->OneSecTxOkCount > pEntry->OneSecTxRetryOkCount)
            {
                fUpgradeQuality = TRUE;
                pEntry->TxQuality[DownRate] = 0;     // next DOWN rate's quality should also very good
                if (pEntry->TxQuality[CurrRate])
                    pEntry->TxQuality[CurrRate] --;  // quality very good in CurrRate
                
                if (pEntry->TxRateUpPenalty)
                    pEntry->TxRateUpPenalty --;
                else if (pEntry->TxQuality[UpRate])
                    pEntry->TxQuality[UpRate] --;    // may improve next UP rate's quality

            }
        }
        
        DBGPRINT(RT_DEBUG_TRACE,"DRS - [AID# %d] Qty[%d]=%d PER=%d%% %d-sec, Qty[%d]=%d Pty=%d\n", 
            pEntry->Aid, RateIdToMbps[CurrRate], pEntry->TxQuality[CurrRate],
            TxErrorRatio, pEntry->CurrTxRateStableTime,
            RateIdToMbps[UpRate], pEntry->TxQuality[UpRate],
            pEntry->TxRateUpPenalty);
        
        // we're going to ugrade CurrRate to UpRate at next few seconds, 
        // but before that, we'd better try a NULL frame @ UpRate and 
        // see if UpRate is stable or not. If this NULL frame fails, it will
        // downgrade TxQuality[CurrRate], so that STA won't switch to
        // to UpRate in the next second
        if ((fUpgradeQuality == TRUE)          &&
            (UpRate != CurrRate)               && 
            (pEntry->TxQuality[CurrRate] <= 1) &&
            (pEntry->TxQuality[UpRate] <= 1))
        {
            DBGPRINT(RT_DEBUG_TRACE,"2 test frames at UpRate = %d Mbps\n",RateIdToMbps[UpRate]);
            ApEnqueueNullFrame(pAd, &pEntry->Addr, UpRate, -1);
            ApEnqueueNullFrame(pAd, &pEntry->Addr, UpRate, -1);
        }

        pEntry->CurrTxRateStableTime ++;
        
        // perform DRS - consider TxRate Down frist, then rate up
        //     rate down, if current TX rate's quality is not good
        //     rate up, if UPRate's quality is very good
        if (pEntry->TxQuality[CurrRate] >= 3) 
        {
            if (CurrRate != DownRate)
            {
                // shorter stable time require more penalty in next rate UP
                // criteria
                if (pEntry->CurrTxRateStableTime < 4)      // less then 4 sec
                    pEntry->TxRateUpPenalty = DRS_PENALTY; // add 8 sec penalty
                else if (pEntry->CurrTxRateStableTime < 8) // less then 8 sec
                    pEntry->TxRateUpPenalty = 2;           // add 2 sec penalty
                else
                    pEntry->TxRateUpPenalty = 0;           // no penalty

                pEntry->CurrTxRate = DownRate;
                pEntry->CurrTxRateStableTime = 0;
                DBGPRINT(RT_DEBUG_TRACE,"DRS - AID#%d: change TX rate to %d Mbps\n", pEntry->Aid, RateIdToMbps[DownRate]);
            }
        }
        else if ((pEntry->TxQuality[CurrRate] <=0) && (pEntry->TxQuality[UpRate] <=0))
        {
            if (CurrRate != UpRate)
            {
                pEntry->CurrTxRate = UpRate;
                pEntry->CurrTxRateStableTime = 0;
                DBGPRINT(RT_DEBUG_TRACE,"DRS - AID#%d: change TX rate to %d Mbps\n", pEntry->Aid, RateIdToMbps[UpRate]);
            }
        }

        // reset all OneSecxxx counters
        pEntry->OneSecTxFailCount = 0;
        pEntry->OneSecTxOkCount = 0;
        pEntry->OneSecTxRetryOkCount = 0;
    }

#ifdef	WDS
    for (i=0; i<pAd->WdsTab.Num; i++)
    {
    	USHORT TxTotalCnt, TxErrorRatio;
        BOOLEAN fUpgradeQuality = FALSE;
        RT_802_11_WDS_ENTRY *pEntry = &pAd->WdsTab.WdsEntry[i];

		TxTotalCnt = pEntry->OneSecTxOkCount + pEntry->OneSecTxRetryOkCount + pEntry->OneSecTxFailCount;
        
        // skip those STA that has no traffic in the past period
        if (TxTotalCnt == 0)
            continue;

        // decide the next upgrade rate and downgrade rate, if any
        CurrRate = pEntry->CurrTxRate;
        if (pEntry->MaxSupportedRate < RATE_FIRST_OFDM_RATE)
        {
            UpRate = Phy11BNextRateUpward[CurrRate];
            DownRate = Phy11BNextRateDownward[CurrRate];
        }
        else 
        {
            UpRate = Phy11BGNextRateUpward[CurrRate];
            DownRate = Phy11BGNextRateDownward[CurrRate];
        }
        
        // calculate Tx with retry ratio when enough samples are available
        if (TxTotalCnt > 15)
        {
            TxErrorRatio = ((pEntry->OneSecTxRetryOkCount + pEntry->OneSecTxFailCount) *100) / TxTotalCnt;

            // downgrade TX quality if retry+error ratio reached
            if (((CurrRate >= RATE_24) && (pEntry->OneSecTxFailCount >=2)) ||
                (TxErrorRatio >= pRateDownPER[CurrRate]))
            {
                pEntry->TxQuality[CurrRate] += DRS_TX_QUALITY_WORST_BOUND;   // degrade quality
                if (pEntry->TxQuality[CurrRate] > DRS_TX_QUALITY_WORST_BOUND)
                    pEntry->TxQuality[CurrRate] = DRS_TX_QUALITY_WORST_BOUND;
            }
            // upgrade TX quality if retry+error ratio reached
            else if (TxErrorRatio <= pRateUpPER[CurrRate])
            {
                fUpgradeQuality = TRUE;
                pEntry->TxQuality[DownRate] = 0;     // next DOWN rate's quality should also very good
                if (pEntry->TxQuality[CurrRate])
                    pEntry->TxQuality[CurrRate] --;  // quality very good in CurrRate
                
                if (pEntry->TxRateUpPenalty)
                    pEntry->TxRateUpPenalty --;
                else if (pEntry->TxQuality[UpRate])
                    pEntry->TxQuality[UpRate] --;    // may improve next UP rate's quality
            }
        }
        else
        {
            TxErrorRatio = 0;  // too few samples
        
            // Downgrade TX quality upon any TX failure in the past second
            if (pEntry->OneSecTxFailCount)
            {
                if ((pEntry->OneSecTxFailCount <= 1) && (pEntry->OneSecTxOkCount + pEntry->OneSecTxRetryOkCount))
                {
                    pEntry->TxQuality[CurrRate] += 2;  // degrade quality
                    if (pEntry->TxQuality[CurrRate] > DRS_TX_QUALITY_WORST_BOUND)
                        pEntry->TxQuality[CurrRate] = DRS_TX_QUALITY_WORST_BOUND;
                }
                else // more than 2 failure, or no TX ok cases
                {
                    pEntry->TxQuality[CurrRate] += 4;   // quality bad
                    if (pEntry->TxQuality[CurrRate] > DRS_TX_QUALITY_WORST_BOUND)
                        pEntry->TxQuality[CurrRate] = DRS_TX_QUALITY_WORST_BOUND;
                    
                    pEntry->TxQuality[UpRate] ++;     // quality must not good for UP rate
                    if (pEntry->TxQuality[UpRate] > DRS_TX_QUALITY_WORST_BOUND)
                        pEntry->TxQuality[UpRate] = DRS_TX_QUALITY_WORST_BOUND;
                }
            }
            // upgrade TX quality if -
            // 1. no TX failure but do have TX ok case, and
            // 2. there's more one-time-ok cases than retry-ok cases in the past second
            else if (pEntry->OneSecTxOkCount > pEntry->OneSecTxRetryOkCount)
            {
                fUpgradeQuality = TRUE;
                pEntry->TxQuality[DownRate] = 0;     // next DOWN rate's quality should also very good
                if (pEntry->TxQuality[CurrRate])
                    pEntry->TxQuality[CurrRate] --;  // quality very good in CurrRate
                

                if (pEntry->TxRateUpPenalty)
                    pEntry->TxRateUpPenalty --;
                else if (pEntry->TxQuality[UpRate])
                    pEntry->TxQuality[UpRate] --;    // may improve next UP rate's quality

            }
        }    

        DBGPRINT(RT_DEBUG_TRACE,"WDS DRS - Qty[%d]=%d PER=%d%% %d-sec, Qty[%d]=%d Pty=%d\n",
            RateIdToMbps[CurrRate], pEntry->TxQuality[CurrRate],
            TxErrorRatio, pEntry->CurrTxRateStableTime,
            RateIdToMbps[UpRate], pEntry->TxQuality[UpRate],
            pEntry->TxRateUpPenalty);
        
        // we're going to ugrade CurrRate to UpRate at next few seconds, 
        // but before that, we'd better try a NULL frame @ UpRate and 
        // see if UpRate is stable or not. If this NULL frame fails, it will
        // downgrade TxQuality[CurrRate], so that STA won't switch to
        // to UpRate in the next second
	    if ((fUpgradeQuality == TRUE)          &&
	        (UpRate != CurrRate)               && 
            (pEntry->TxQuality[CurrRate] <= 1) &&
            (pEntry->TxQuality[UpRate] <= 1))
        {
            DBGPRINT(RT_DEBUG_TRACE,"2 test frames at UpRate = %d Mbps\n",RateIdToMbps[UpRate]);
            ApEnqueueNullFrame(pAd, (PMACADDR)&pEntry->WdsAddr, UpRate, i);
            ApEnqueueNullFrame(pAd, (PMACADDR)&pEntry->WdsAddr, UpRate, i);
        }

        pEntry->CurrTxRateStableTime ++;
        
        // perform DRS - consider TxRate Down frist, then rate up
        //     rate down, if current TX rate's quality is not good
        //     rate up, if UPRate's quality is very good
        if (pEntry->TxQuality[CurrRate] >= 3) 
        {
            if (CurrRate != DownRate)
            {
                // shorter stable time require more penalty in next rate UP
                // criteria
                if (pEntry->CurrTxRateStableTime < 4)      // less then 4 sec
                    pEntry->TxRateUpPenalty = DRS_PENALTY; // add 8 sec penalty
                else if (pEntry->CurrTxRateStableTime < 8) // less then 8 sec
                    pEntry->TxRateUpPenalty = 2;           // add 2 sec penalty
                else
                    pEntry->TxRateUpPenalty = 0;           // no penalty

                pEntry->CurrTxRate = DownRate;
                pEntry->CurrTxRateStableTime = 0;
                DBGPRINT(RT_DEBUG_TRACE,"WDS DRS - change TX rate to %d Mbps\n", RateIdToMbps[DownRate]);
            }
        }
        else if ((pEntry->TxQuality[CurrRate] <=0) && (pEntry->TxQuality[UpRate] <=0))
        {
            if (CurrRate != UpRate)
            {
                pEntry->CurrTxRate = UpRate;
                pEntry->CurrTxRateStableTime = 0;
                DBGPRINT(RT_DEBUG_TRACE,"WDS DRS - change TX rate to %d Mbps\n", RateIdToMbps[UpRate]);
            }
        }

        // reset all OneSecxxx counters
        pEntry->OneSecTxFailCount = 0;
        pEntry->OneSecTxOkCount = 0;
        pEntry->OneSecTxRetryOkCount = 0;
    }
#endif
}

/*
    ==========================================================================
    Description:
        modify TX preamble mode
    ==========================================================================
 */
VOID MlmeSetTxPreamble(
    IN PRTMP_ADAPTER pAd, 
    IN USHORT TxPreamble)
{
    ULONG Plcp1MCsr = 0x00700400;     // 0x13c, ACK/CTS PLCP at 1 Mbps
    ULONG Plcp2MCsr = 0x00380401;     // 0x140, ACK/CTS PLCP at 2 Mbps
    ULONG Plcp5MCsr = 0x00150402;     // 0x144, ACK/CTS PLCP at 5.5 Mbps
    ULONG Plcp11MCsr = 0x000b8403;     // 0x148, ACK/CTS PLCP at 11 Mbps
    
    if (TxPreamble == Rt802_11PreambleShort)
    {
        DBGPRINT(RT_DEBUG_TRACE, "MlmeSetTxPreamble - CCK use SHORT PREAMBLE\n");
//      Plcp1MCsr |= 0x00000008; // 1Mbps should always use long preamble
        Plcp2MCsr |= 0x00000008;
        Plcp5MCsr |= 0x00000008;
        Plcp11MCsr |= 0x00000008;
        pAd->PortCfg.TxPreamble = Rt802_11PreambleShort;
        pAd->PortCfg.CapabilityInfo |= 0x0020;  // set SHORT PREAMBLE CAPABLE bit = 1
    }
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, "MlmeSetTxPreamble - CCK use LONG PREAMBLE\n");
        pAd->PortCfg.TxPreamble = Rt802_11PreambleLong;
        pAd->PortCfg.CapabilityInfo &= 0xffdf;  // set SHORT PREAMBLE CAPABLE bit = 0
    }

    RTMP_IO_WRITE32(pAd, PLCP1MCSR, Plcp1MCsr);
    RTMP_IO_WRITE32(pAd, PLCP2MCSR, Plcp2MCsr);
    RTMP_IO_WRITE32(pAd, PLCP5MCSR, Plcp5MCsr);
    RTMP_IO_WRITE32(pAd, PLCP11MCSR, Plcp11MCsr);
}
    
/*
    ==========================================================================
    Description:
        update BasicRateBitmap, MaxBasicRate, MaxTxRate ,.. based on
        current supported rate set and desired rate set.
    ==========================================================================
 */
VOID MlmeUpdateTxRates(
    IN PRTMP_ADAPTER pAd)
{
    int i, num;
    UCHAR Rate, MaxDesire = RATE_1, MaxSupport = RATE_1; 
    ULONG BasicRateBitmap = 0;
    UCHAR CurrBasicRate = RATE_1;

    // find max desired rate
    num = 0;
    for (i=0; i<MAX_LEN_OF_SUPPORTED_RATES; i++)
    {
        switch (pAd->PortCfg.DesiredRates[i] & 0x7f)
        {
            case 2:  Rate = RATE_1;   num++;   break;
            case 4:  Rate = RATE_2;   num++;   break;
            case 11: Rate = RATE_5_5; num++;   break;
            case 22: Rate = RATE_11;  num++;   break;
            case 12: Rate = RATE_6;   num++;   break;
            case 18: Rate = RATE_9;   num++;   break;
            case 24: Rate = RATE_12;  num++;   break;
            case 36: Rate = RATE_18;  num++;   break;
            case 48: Rate = RATE_24;  num++;   break;
            case 72: Rate = RATE_36;  num++;   break;
            case 96: Rate = RATE_48;  num++;   break;
            case 108: Rate = RATE_54; num++;   break;
            default: Rate = RATE_1;   break;
        }
        if (MaxDesire < Rate)  MaxDesire = Rate;
    }

    // Auto rate switching is enabled only if more than one DESIRED RATES are 
    // specified; otherwise disabled
    if (num <= 1)
        pAd->PortCfg.EnableAutoRateSwitching = FALSE;
    else
        pAd->PortCfg.EnableAutoRateSwitching = TRUE;

    // find max supported rate
    for (i=0; i<pAd->PortCfg.SupportedRatesLen; i++)
    {
        switch (pAd->PortCfg.SupportedRates[i] & 0x7f)
        {
            case 2: Rate = RATE_1;   
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0001;  
                    break;
            case 4: Rate = RATE_2;   
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0002;  
                    break;
            case 11: 
                    Rate = RATE_5_5; 
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0004;  
                    break;
            case 22: 
                    Rate = RATE_11;  
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0008;  
                    break;
            case 12: 
                    Rate = RATE_6;   
//                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0010;  
                    break;
            case 18: 
                    Rate = RATE_9;   
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0020;  
                    break;
            case 24: 
                    Rate = RATE_12;  
//                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0040;  
                    break;
            case 36: 
                    Rate = RATE_18;  
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0080;  
                    break;
            case 48: 
                    Rate = RATE_24;
//                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0100;
                    break;
            case 72: 
                    Rate = RATE_36;  
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0200;  
                    break;
            case 96: 
                    Rate = RATE_48;  
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0400;  
                    break;
            case 108: 
                    Rate = RATE_54; 
                    if (pAd->PortCfg.SupportedRates[i] & 0x80) 
                        BasicRateBitmap |= 0x0800;  
                    break;
            default:  
                    Rate = RATE_1;   
                    break;
        }
        if (MaxSupport < Rate)  MaxSupport = Rate;

        // decide max basic rate, which is used to send MLME frames
//      if ((pAd->PortCfg.SupportedRates[i] & 0x80) && (MaxBasicRate < Rate))
//      {
            // use only 11b CCK as max basic rate when operate in B/G mixed mode, B-only mode
//          if (Rate < RATE_FIRST_OFDM_RATE)  MaxBasicRate = Rate;
//      }
    }

    // max tx rate = min {max desire rate, max supported rate}
    if (MaxSupport < MaxDesire)
        pAd->PortCfg.MaxTxRate = pAd->PortCfg.TxRate = MaxSupport;
    else
        pAd->PortCfg.MaxTxRate = pAd->PortCfg.TxRate = MaxDesire;

//  pAd->PortCfg.MaxBasicRate = MaxBasicRate;
	RTMP_IO_WRITE32(pAd, ARCSR1, BasicRateBitmap);

    // calculate the exptected ACK rate for each TX rate. This info is used to caculate
    // the DURATION field of outgoing uniicast DATA/MGMT frame
    for (i=0; i<MAX_LEN_OF_SUPPORTED_RATES; i++)
    {
        if (BasicRateBitmap & (0x01 << i))
            CurrBasicRate = (UCHAR)i;
        pAd->PortCfg.ExpectedACKRate[i] = CurrBasicRate;
        DBGPRINT(RT_DEBUG_INFO,"Expected ACK rate for %d Mbps = %d Mbps\n", RateIdToMbps[i], RateIdToMbps[CurrBasicRate]);
    }

    pAd->PortCfg.MlmeRate = RATE_2;
    pAd->PortCfg.RtsRate = RATE_2;
    
    DBGPRINT(RT_DEBUG_TRACE, " MlmeUpdateTxRates (MaxDesire=%d Mbps, MaxSupport=%d Mbps, MaxTxRate=%d Mbps, Rate Switching =%d)\n", 
             RateIdToMbps[MaxDesire], RateIdToMbps[MaxSupport], RateIdToMbps[pAd->PortCfg.MaxTxRate], pAd->PortCfg.EnableAutoRateSwitching);
    DBGPRINT(RT_DEBUG_TRACE, " MlmeUpdateTxRates (RtsRate=%d Mbps, MlmeRate=%d Mbps, BasicRateBitmap=0x%04x)\n", 
             RateIdToMbps[pAd->PortCfg.RtsRate], RateIdToMbps[pAd->PortCfg.MlmeRate], BasicRateBitmap);
}

/*
    ==========================================================================
    Description:
        init the management mac frame header with input parameters
    ==========================================================================
 */
VOID MgtMacHeaderInit(
    IN	PRTMP_ADAPTER	pAd, 
    IN OUT PMACHDR Hdr, 
    IN UCHAR Subtype, 
    IN UCHAR ToDs, 
    IN PMACADDR Ds, 
    IN PMACADDR Bssid) 
{
    NdisZeroMemory(Hdr, sizeof(MACHDR));
    Hdr->Type = BTYPE_MGMT;
    Hdr->SubType = Subtype;
    Hdr->Tods = ToDs;
    COPY_MAC_ADDR(&Hdr->Addr1, Ds);
    COPY_MAC_ADDR(&Hdr->Addr2, &pAd->CurrentAddress);
    COPY_MAC_ADDR(&Hdr->Addr3, Bssid);
}


/*
    ==========================================================================
    Description:
        This routine build an outgoing frame, and fill all information specified 
        in argument list to the frame body. The actual frame size is the summation 
        of all arguments.
    input params:
        Buffer - pointer to a pre-allocated memory segment
        args - a list of <int arg_size, arg> pairs.
        NOTE NOTE NOTE!!!! the last argument must be NULL, otherwise this
                           function will FAIL!!!
    return:
        Size of the buffer
    usage:  
        MakeOutgoingFrame(Buffer, output length, 2, &fc, 2, &dur, 6, p_addr1, 6,p_addr2, END_OF_ARGS);
    ==========================================================================
 */
ULONG MakeOutgoingFrame(
    OUT CHAR *Buffer, 
    OUT ULONG *FrameLen, ...) 
{
    CHAR   *p;
    int     leng;
    ULONG   TotLeng;
    va_list Args;

    // calculates the total length
    TotLeng = 0;
    va_start(Args, FrameLen);
    do 
    {
        leng = va_arg(Args, int);
        if (leng == END_OF_ARGS) 
        {
            break;
        }
        p = va_arg(Args, PVOID);
        NdisMoveMemory(&Buffer[TotLeng], p, leng);
        TotLeng = TotLeng + leng;
    } while(TRUE);

    va_end(Args); /* clean up */
    *FrameLen = TotLeng;
    return TotLeng;
}

/*
    ==========================================================================
    Description:
        Initialize The MLME Queue, used by MLME Functions
    NOTE:
        Because this is done only once (at the init stage), no need to be locked
    ==========================================================================
 */
NDIS_STATUS MlmeQueueInit(
    IN MLME_QUEUE *Queue) 
{
    INT i;

    NdisAllocateSpinLock(&Queue->Lock);

    Queue->Num  = 0;
    Queue->Head = 0;
    Queue->Tail = 0;

    for (i = 0; i < MAX_LEN_OF_MLME_QUEUE; i++) 
    {
        Queue->Entry[i].Occupied = FALSE;
        Queue->Entry[i].MsgLen = 0;
        //NdisZeroMemory(Queue->Entry[i].Msg, MAX_LEN_OF_MLME_QUEUE_MSG);//(tt_lin)
    }

    return NDIS_STATUS_SUCCESS;
}


/*! \brief   Enqueue a message for other threads, if they want to send messages to MLME thread
 *  \param  *Queue    The MLME Queue
 *  \param   Machine  The State Machine Id
 *  \param   MsgType  The Message Type
 *  \param   MsgLen   The Message length
 *  \param  *Msg      The message pointer
 *  \return  TRUE if enqueue is successful, FALSE if the queue is full
 *  \pre
 *  \post
 *  \note    The message has to be initialized
 */
BOOLEAN MlmeEnqueue(
    OUT MLME_QUEUE *Queue, 
    IN ULONG Machine, 
    IN ULONG MsgType, 
    IN ULONG MsgLen, 
    IN VOID *Msg) 
{
    INT Tail;

    if (MlmeQueueFull(Queue)) 
    {
        DBGPRINT(RT_DEBUG_ERROR, "MlmeEnqueue full, msg dropped and may corrupt MLME\n");
        return FALSE;
    }

    NdisAcquireSpinLock(&(Queue->Lock));
    Tail = Queue->Tail;
    Queue->Tail++;
    Queue->Num++;
    if (Queue->Tail == MAX_LEN_OF_MLME_QUEUE) 
    {
        Queue->Tail = 0;
    }
    NdisReleaseSpinLock(&(Queue->Lock));
    DBGPRINT(RT_DEBUG_INFO, "MlmeEnqueue, num=%d\n",Queue->Num);
    
    Queue->Entry[Tail].Occupied = TRUE;
    Queue->Entry[Tail].Machine = Machine;
    Queue->Entry[Tail].MsgType = MsgType;
    Queue->Entry[Tail].MsgLen  = MsgLen;
    NdisMoveMemory(Queue->Entry[Tail].Msg, Msg, MsgLen);
    return TRUE;
}

/*! \brief   This function is used when Recv gets a MLME message
 *  \param  *Queue           The MLME Queue
 *  \param   TimeStampHigh   The upper 32 bit of timestamp
 *  \param   TimeStampLow    The lower 32 bit of timestamp
 *  \param   Rssi            The receiving RSSI strength
 *  \param   MsgLen          The length of the message
 *  \param  *Msg             The message pointer
 *  \return  TRUE if everything ok, FALSE otherwise (like Queue Full)
 *  \pre
 *  \post
 */
BOOLEAN MlmeEnqueueForRecv(
    IN	PRTMP_ADAPTER	pAd, 
    OUT MLME_QUEUE *Queue, 
//  IN ULONG TimeStampHigh, 
//  IN ULONG TimeStampLow,
    IN UCHAR Rssi, 
    IN ULONG MsgLen, 
    IN VOID *Msg) 
{
    INT          Tail, Machine;
#ifdef RT2500_DBG
	MACFRAME    *Fr = (MACFRAME *)Msg;
#endif
    ULONG        MsgType;

    if (MlmeQueueFull(Queue) ) 
    {
        DBGPRINT(RT_DEBUG_ERROR, "MlmeEnqueueForRecv (queue full error) \n");
        return FALSE;
    }

    if (!MsgTypeSubst(Msg, &Machine, &MsgType)) 
    {
        DBGPRINT(RT_DEBUG_ERROR, "MlmeEnqueueForRecv (drop mgmt->subtype=%d)\n",Fr->Hdr.SubType);
        return FALSE;
    }
    
    // OK, we got all the informations, it is time to put things into queue
    NdisAcquireSpinLock(&(Queue->Lock));
    Tail = Queue->Tail;
    Queue->Tail++;
    Queue->Num++;
    if (Queue->Tail == MAX_LEN_OF_MLME_QUEUE) 
    {
        Queue->Tail = 0;
    }
    NdisReleaseSpinLock(&(Queue->Lock));
    DBGPRINT(RT_DEBUG_INFO, "MlmeEnqueueForRecv, num=%d\n",Queue->Num);
    
    Queue->Entry[Tail].Occupied = TRUE;
    Queue->Entry[Tail].Machine = Machine;
    Queue->Entry[Tail].MsgType = MsgType;
    Queue->Entry[Tail].MsgLen  = MsgLen;
//  Queue->Entry[Tail].TimeStamp = TimeStampLow;
//  Queue->Entry[Tail].TimeStamp = TimeStampHigh;
    Queue->Entry[Tail].Rssi = Rssi;
    //NdisMoveMemory(Queue->Entry[Tail].Msg, Msg, MsgLen);(tt_lin)
    
	//Change to Rx-buffer pointer exchange(tt_lin)
	{
	void* va_data_addr;
	dma_addr_t pa_data_addr;
	PRXD_STRUC pRxD;

	NdisAcquireSpinLock(&pAd->RxRingLock);
	// Write RxD buffer address & allocated buffer length
	//1. temp storge values
	pa_data_addr = Queue->Entry[Tail].pa_data_addr;
	va_data_addr = Queue->Entry[Tail].Msg;
	//2. save the Rx-buffer-address to Mlme Query
	Queue->Entry[Tail].Msg = pAd->RxRing[pAd->CurDecryptIndex].va_data_addr;
	Queue->Entry[Tail].pa_data_addr = pAd->RxRing[pAd->CurDecryptIndex].pa_data_addr;
	//3. save the temp data to Rx-buffer-address
	pAd->RxRing[pAd->CurDecryptIndex].va_data_addr = va_data_addr;
	pAd->RxRing[pAd->CurDecryptIndex].pa_data_addr = pa_data_addr;
	//4. change the BufferAddressPa on Rx-Descriptor
	pRxD = (PRXD_STRUC) pAd->RxRing[pAd->CurDecryptIndex].va_addr;
#ifndef BIG_ENDIAN
	pRxD->BufferAddressPa = pa_data_addr;//pointer to Rx descriptor
#else
	pRxD->BufferAddressPa = SWAP32(pa_data_addr);//pointer to Rx descriptor
#endif
	NdisReleaseSpinLock(&pAd->RxRingLock);
	}


    MlmeHandler(pAd);

    return TRUE;
}

/*! \brief   Dequeue a message from the MLME Queue
 *  \param  *Queue    The MLME Queue
 *  \param  *Elem     The message dequeued from MLME Queue
 *  \return  TRUE if the Elem contains something, FALSE otherwise
 *  \pre
 *  \post
 */
BOOLEAN MlmeDequeue(
    IN MLME_QUEUE *Queue, 
    OUT MLME_QUEUE_ELEM **Elem) 
{
    NdisAcquireSpinLock(&(Queue->Lock));
    *Elem = &(Queue->Entry[Queue->Head]);
    Queue->Num--;
    Queue->Head++;
    if (Queue->Head == MAX_LEN_OF_MLME_QUEUE) 
    {
        Queue->Head = 0;
    }
    NdisReleaseSpinLock(&(Queue->Lock));
    DBGPRINT(RT_DEBUG_INFO, "MlmeDequeue, num=%d\n",Queue->Num);

    return TRUE;
}

/*! \brief  test if the MLME Queue is empty
 *  \param  *Queue    The MLME Queue
 *  \return TRUE if the Queue is empty, FALSE otherwise
 *  \pre
 *  \post
 */
BOOLEAN MlmeQueueEmpty(
    IN MLME_QUEUE *Queue) 
{
    BOOLEAN Ans;

    NdisAcquireSpinLock(&(Queue->Lock));
    Ans = (Queue->Num == 0);
    NdisReleaseSpinLock(&(Queue->Lock));

    return Ans;
}

/*! \brief   test if the MLME Queue is full
 *  \param   *Queue      The MLME Queue
 *  \return  TRUE if the Queue is empty, FALSE otherwise
 *  \pre
 *  \post
 */
BOOLEAN MlmeQueueFull(
    IN MLME_QUEUE *Queue) 
{
    BOOLEAN Ans;

    NdisAcquireSpinLock(&(Queue->Lock));
    Ans = (Queue->Num == MAX_LEN_OF_MLME_QUEUE);
    NdisReleaseSpinLock(&(Queue->Lock));

    return Ans;
}

/*! \brief   The destructor of MLME Queue
 *  \param 
 *  \return
 *  \pre
 *  \post
 *  \note   Free spin lock only
 */
VOID MlmeQueueDestroy(
    IN MLME_QUEUE *Queue) 
{
}

/*! \brief   To substitute the message type if the message is coming from external
 *  \param  *Fr            The frame received
 *  \param  *Machine       The state machine
 *  \param  *MsgType       the message type for the state machine
 *  \return TRUE if the substitution is successful, FALSE otherwise
 *  \pre
 *  \post
 */
BOOLEAN MsgTypeSubst(
    IN VOID *Msg, 
    OUT INT *Machine, 
    OUT INT *MsgType) 
{
    USHORT Seq;
    UCHAR 	EAPType; 
    MACFRAME    *Fr = (MACFRAME *)Msg;
    BOOLEAN     Return;
// TODO:
// only PROBE_REQ can be broadcast, all others must be unicast-to-me && is_mybssid; otherwise, 
// ignore this frame

    // wpa EAPOL PACKET
    if (Fr->Hdr.Type == BTYPE_DATA) 
    {    
        *Machine = WPA_STATE_MACHINE;
        EAPType = *((UCHAR*)Msg+LENGTH_802_11+LENGTH_802_1_H+1);
        Return = WPAMsgTypeSubst(EAPType, MsgType);
        return Return;

    }
    else if (Fr->Hdr.Type == BTYPE_MGMT) 		
    {
    	switch (Fr->Hdr.SubType) 
    	{
		case SUBTYPE_ASSOC_REQ:
			*Machine = ASSOC_STATE_MACHINE;
			*MsgType = MT2_PEER_ASSOC_REQ;
			break;
//      case SUBTYPE_ASSOC_RSP:
//          *Machine = ASSOC_STATE_MACHINE;
//          *MsgType = MT2_PEER_ASSOC_RSP;
//          break;
        	case SUBTYPE_REASSOC_REQ:
			*Machine = ASSOC_STATE_MACHINE;
			*MsgType = MT2_PEER_REASSOC_REQ;
			break;
//      case SUBTYPE_REASSOC_RSP:
//          *Machine = ASSOC_STATE_MACHINE;
//          *MsgType = MT2_PEER_REASSOC_RSP;
//          break;
        	case SUBTYPE_PROBE_REQ:
			*Machine = SYNC_STATE_MACHINE;				
			*MsgType = MT2_PEER_PROBE_REQ;
            	break;
//      case SUBTYPE_PROBE_RSP:
//          *Machine = SYNC_STATE_MACHINE;
//          *MsgType = MT2_PEER_PROBE_RSP;
//          break;
        	case SUBTYPE_BEACON:
			*Machine = SYNC_STATE_MACHINE;
			*MsgType = MT2_PEER_BEACON;
            	break;
//      case SUBTYPE_ATIM:
//          *Machine = SYNC_STATE_MACHINE;
//          *MsgType = MT2_PEER_ATIM;
//          break;
        	case SUBTYPE_DISASSOC:
			*Machine = ASSOC_STATE_MACHINE;
			*MsgType = MT2_PEER_DISASSOC_REQ;
			break;
        	case SUBTYPE_AUTH:
            // get the sequence number from payload 24 Mac Header + 2 bytes algorithm
			NdisMoveMemory(&Seq, &Fr->Octet[2], sizeof(USHORT));
			if (Seq == 1 || Seq == 3) 
			{
				*Machine = AUTH_RSP_STATE_MACHINE;
				*MsgType = MT2_PEER_AUTH_ODD;
			} 
//          else if (Seq == 2 || Seq == 4) 
//          {
//              *Machine = AUTH_STATE_MACHINE;
//              *MsgType = MT2_PEER_AUTH_EVEN;
//          } 
			else 
			{
				DBGPRINT(RT_DEBUG_TRACE,"wrong AUTH seq=%d\n",Seq);
				return FALSE;
			}
			break;
        	case SUBTYPE_DEAUTH:
			*Machine = AUTH_RSP_STATE_MACHINE;
			*MsgType = MT2_PEER_DEAUTH;
			break;
        	default:
			return FALSE;
			break;
    	}

    		return TRUE;
    }

    return FALSE;
}


/*! \brief Initialize the state machine.
 *  \param *S           pointer to the state machine 
 *  \param  Trans       State machine transition function
 *  \param  StNr        number of states 
 *  \param  MsgNr       number of messages 
 *  \param  DefFunc     default function, when there is invalid state/message combination 
 *  \param  InitState   initial state of the state machine 
 *  \param  Base        StateMachine base, internal use only
 *  \pre p_sm should be a legal pointer
 *  \post
 */

VOID StateMachineInit(
    IN STATE_MACHINE *S, 
    IN STATE_MACHINE_FUNC Trans[], 
    IN ULONG StNr, 
    IN ULONG MsgNr, 
    IN STATE_MACHINE_FUNC DefFunc, 
    IN ULONG InitState, 
    IN ULONG Base) 
{
    ULONG i, j;

    // set number of states and messages
    S->NrState = StNr;
    S->NrMsg   = MsgNr;
    S->Base    = Base;

    S->TransFunc  = Trans;
    
    // init all state transition to default function
    for (i = 0; i < StNr; i++) 
    {
        for (j = 0; j < MsgNr; j++) 
        {
            S->TransFunc[i * MsgNr + j] = DefFunc;
        }
    }
    
    // set the starting state
    S->CurrState = InitState;

}

/*! \brief This function fills in the function pointer into the cell in the state machine 
 *  \param *S   pointer to the state machine
 *  \param St   state
 *  \param Msg  incoming message
 *  \param f    the function to be executed when (state, message) combination occurs at the state machine
 *  \pre *S should be a legal pointer to the state machine, st, msg, should be all within the range, Base should be set in the initial state
 *  \post
 */
VOID StateMachineSetAction(
    IN STATE_MACHINE *S, 
    IN ULONG St, 
    IN ULONG Msg, 
    IN STATE_MACHINE_FUNC Func) 
{
    ULONG MsgIdx;
    
    MsgIdx = Msg - S->Base;

    if (St < S->NrState && MsgIdx < S->NrMsg) 
    {
        // boundary checking before setting the action
        S->TransFunc[St * S->NrMsg + MsgIdx] = Func;
    } 
}

/*! \brief   This function does the state transition
 *  \param   *Adapter the NIC adapter pointer
 *  \param   *S       the state machine
 *  \param   *Elem    the message to be executed
 *  \return   None
 */
VOID StateMachinePerformAction(
    IN	PRTMP_ADAPTER	pAd, 
    IN STATE_MACHINE *S, 
    IN MLME_QUEUE_ELEM *Elem) 
{
    (*(S->TransFunc[S->CurrState * S->NrMsg + Elem->MsgType - S->Base]))(pAd, Elem);
}

/*
    ==========================================================================
    Description:
        The drop function, when machine executes this, the message is simply 
        ignored. This function does nothing, the message is freed in 
        StateMachinePerformAction()
    ==========================================================================
 */
VOID Drop(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
#if 0
    if ((Elem->MsgType == MT2_PEER_BEACON) ||
        (Elem->MsgType == MT2_PEER_PROBE_REQ) ||
        (Elem->MsgType == MT2_PEER_PROBE_RSP))
        ;
    else
    {
        DBGPRINT(RT_DEBUG_TRACE, ("Warn:>>Drop Msg=%d<<\n",Elem->MsgType));
    }
#endif    
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID LfsrInit(
    IN PRTMP_ADAPTER pAd, 
    IN ULONG Seed) 
{
    if (Seed == 0) 
        pAd->Mlme.ShiftReg = 1;
    else 
        pAd->Mlme.ShiftReg = Seed;
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
UCHAR RandomByte(
    IN PRTMP_ADAPTER pAd) 
{
    ULONG i;
    UCHAR R, Result;

    R = 0;

    for (i = 0; i < 8; i++) 
    {
        if (pAd->Mlme.ShiftReg & 0x00000001) 
        {
            pAd->Mlme.ShiftReg = ((pAd->Mlme.ShiftReg ^ LFSR_MASK) >> 1) | 0x80000000;
            Result = 1;
        } 
        else 
        {
            pAd->Mlme.ShiftReg = pAd->Mlme.ShiftReg >> 1;
            Result = 0;
        }
        R = (R << 1) | Result;
    }

    return R;
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicSwitchChannel(
    IN PRTMP_ADAPTER pAd, 
    IN UCHAR Channel) 
{
    ULONG R3;
    UCHAR index;

    R3 = pAd->PortCfg.ChannelTxPower[Channel - 1];

    // E2PROM value is calibrated for maximum power (i.e. 100%)
    // We lower TX power here according to the percentage specified from UI
    if (pAd->PortCfg.TxPowerPercentage > 90)        // 91~100%, treat as 100%
        ;
    else if (pAd->PortCfg.TxPowerPercentage > 60)   // 61~90%, treat as 75%
        R3 -= 1;    // minus 1 db
    else if (pAd->PortCfg.TxPowerPercentage > 30)   // 31~60%, treat as 50%
        R3 -= 3;    // minus 3 db
    else if (pAd->PortCfg.TxPowerPercentage > 15)   // 16~30%, treat as 25%
        R3 -= 6;    // minus 6 db
    else if (pAd->PortCfg.TxPowerPercentage > 9)    // 10~15%, treat as 12.5%
        R3 -= 9;    // minus 9 db
    else                                            // 0~9%, treat as 6.25%
        R3 -= 12;   // minus 12 db

    R3 = R3 << 9; // shift TX power control to correct RF register bit position

    switch (pAd->PortCfg.RfType)
    {
        case RFIC_2522:
            for (index = 0; index < NUM_OF_2522_CHNL; index++)
            {
                if (Channel == RF2522RegTable[index].Channel)
                {
	                R3 = R3 | RF2522RegTable[index].R3; // set TX power
                    RTMP_RF_IO_WRITE32(pAd, RF2522RegTable[index].R1);
                    RTMP_RF_IO_WRITE32(pAd, RF2522RegTable[index].R2);
                    RTMP_RF_IO_WRITE32(pAd, R3);
                    pAd->PortCfg.LatchRfRegs.Channel = Channel;
                    pAd->PortCfg.LatchRfRegs.R1 = RF2522RegTable[index].R1;
                    pAd->PortCfg.LatchRfRegs.R2 = RF2522RegTable[index].R2;
                    pAd->PortCfg.LatchRfRegs.R3 = R3;
                    pAd->PortCfg.LatchRfRegs.R4 = RF2522RegTable[index].R4;
                    break;
                }
            }
            break;

        case RFIC_2523:
            for (index = 0; index < NUM_OF_2523_CHNL; index++)
            {
                if (Channel == RF2523RegTable[index].Channel)
                {
	                R3 = R3 | RF2523RegTable[index].R3; // set TX power
                    RTMP_RF_IO_WRITE32(pAd, RF2523RegTable[index].R1);
                    RTMP_RF_IO_WRITE32(pAd, RF2523RegTable[index].R2);
                    RTMP_RF_IO_WRITE32(pAd, R3);
                    RTMP_RF_IO_WRITE32(pAd, RF2523RegTable[index].R4);
                    pAd->PortCfg.LatchRfRegs.Channel = Channel;
                    pAd->PortCfg.LatchRfRegs.R1 = RF2523RegTable[index].R1;
                    pAd->PortCfg.LatchRfRegs.R2 = RF2523RegTable[index].R2;
                    pAd->PortCfg.LatchRfRegs.R3 = R3;
                    pAd->PortCfg.LatchRfRegs.R4 = RF2523RegTable[index].R4;
                    break;
                }
            }
            break;

        case RFIC_2524:
            for (index = 0; index < NUM_OF_2524_CHNL; index++)
            {
                if (Channel == RF2524RegTable[index].Channel)
                {
	                R3 = R3 | RF2524RegTable[index].R3; // set TX power
                    RTMP_RF_IO_WRITE32(pAd, RF2524RegTable[index].R1);
                    RTMP_RF_IO_WRITE32(pAd, RF2524RegTable[index].R2);
                    RTMP_RF_IO_WRITE32(pAd, R3);
                    RTMP_RF_IO_WRITE32(pAd, RF2524RegTable[index].R4);
                    pAd->PortCfg.LatchRfRegs.Channel = Channel;
                    pAd->PortCfg.LatchRfRegs.R1 = RF2524RegTable[index].R1;
                    pAd->PortCfg.LatchRfRegs.R2 = RF2524RegTable[index].R2;
                    pAd->PortCfg.LatchRfRegs.R3 = R3;
                    pAd->PortCfg.LatchRfRegs.R4 = RF2524RegTable[index].R4;
                    break;
                }
            }
            break;
            
        case RFIC_2525:
            for (index = 0; index < NUM_OF_2525_CHNL; index++)
            {
                if (Channel == RF2525RegTable[index].Channel)
                {
	                R3 = R3 | RF2525RegTable[index].R3; // set TX power
                    RTMP_RF_IO_WRITE32(pAd, RF2525RegTable[index].R1);
                    RTMP_RF_IO_WRITE32(pAd, RF2525RegTable[index].R2);
                    RTMP_RF_IO_WRITE32(pAd, R3);
                    RTMP_RF_IO_WRITE32(pAd, RF2525RegTable[index].R4);
                    pAd->PortCfg.LatchRfRegs.Channel = Channel;
                    pAd->PortCfg.LatchRfRegs.R1 = RF2525RegTable[index].R1;
                    pAd->PortCfg.LatchRfRegs.R2 = RF2525RegTable[index].R2;
                    pAd->PortCfg.LatchRfRegs.R3 = R3;
                    pAd->PortCfg.LatchRfRegs.R4 = RF2525RegTable[index].R4;
                    break;
                }
            }
            break;
            
        case RFIC_5222:
            for (index = 0; index < NUM_OF_5222_CHNL; index++)
            {
                if (Channel == RF5222RegTable[index].Channel)
                {
	                R3 = R3 | RF5222RegTable[index].R3; // set TX power
                    RTMP_RF_IO_WRITE32(pAd, RF5222RegTable[index].R1);
                    RTMP_RF_IO_WRITE32(pAd, RF5222RegTable[index].R2);
                    RTMP_RF_IO_WRITE32(pAd, R3);
                    RTMP_RF_IO_WRITE32(pAd, RF5222RegTable[index].R4);
                    pAd->PortCfg.LatchRfRegs.Channel = Channel;
                    pAd->PortCfg.LatchRfRegs.R1 = RF5222RegTable[index].R1;
                    pAd->PortCfg.LatchRfRegs.R2 = RF5222RegTable[index].R2;
                    pAd->PortCfg.LatchRfRegs.R3 = R3;
                    pAd->PortCfg.LatchRfRegs.R4 = RF5222RegTable[index].R4;
                    break;
                }
            }
            break;

        default:
            break;
    }

    DBGPRINT(RT_DEBUG_TRACE, "AsicSwitchChannel(RF=%d) to #%d, TxPwr=%d%%, R1=0x%08x, R2=0x%08x, R3=0x%08x, R4=0x%08x\n",
        pAd->PortCfg.RfType, 
        pAd->PortCfg.LatchRfRegs.Channel, 
        pAd->PortCfg.TxPowerPercentage,
        pAd->PortCfg.LatchRfRegs.R1, 
        pAd->PortCfg.LatchRfRegs.R2, 
        pAd->PortCfg.LatchRfRegs.R3, 
        pAd->PortCfg.LatchRfRegs.R4);
}

/*
    ==========================================================================
    Description:
        This function is required for 2421 only, and should not be used during
        site survey. It's only required after NIC decided to stay at a channel
        for a longer period.
        When this function is called, it's always after AsicSwitchChannel().
    ==========================================================================
 */
VOID AsicLockChannel(
    IN PRTMP_ADAPTER pAd, 
    IN UCHAR Channel) 
{
    UCHAR r70;

    del_timer_sync(&pAd->PortCfg.RfTuningTimer);
	pAd->PortCfg.RfTuningTimer.expires = jiffies + 1;	// 1 msec timer to turn OFF RF auto tuning
	add_timer(&pAd->PortCfg.RfTuningTimer);

	RTMP_BBP_IO_READ32_BY_REG_ID(pAd, 70, &r70);
    if (Channel == 14)
        r70 |= 0x08;    // turn on BBP R70 Japan filter bit on ch14
    else
        r70 &= 0xf7;    // turn off BBP R70 Japn filter bit when not ch14
	RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, 70, r70);
}

VOID AsicRfTuningExec(
    IN	unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    switch (pAd->PortCfg.RfType)
    {
        case RFIC_2522:
        case RFIC_2524:
        case RFIC_2525:
        case RFIC_5222:
            pAd->PortCfg.LatchRfRegs.R1 &= 0xfffdffff;  // RF R1.bit17 "tune_en1" OFF
            pAd->PortCfg.LatchRfRegs.R3 &= 0xfffffeff;   // RF R3.bit8 "tune_en2" OFF
            RTMP_RF_IO_WRITE32(pAd, pAd->PortCfg.LatchRfRegs.R1); 
            RTMP_RF_IO_WRITE32(pAd, pAd->PortCfg.LatchRfRegs.R3); 
            DBGPRINT(RT_DEBUG_TRACE, "AsicRfTuningExec(R1=0x%x,R3=0x%x)\n",pAd->PortCfg.LatchRfRegs.R1,pAd->PortCfg.LatchRfRegs.R3);
            break;
            
        case RFIC_2523:
            pAd->PortCfg.LatchRfRegs.R3 &= 0xfffffeff;   // RF R3.bit8 "tune_en2" OFF
            RTMP_RF_IO_WRITE32(pAd, pAd->PortCfg.LatchRfRegs.R3); 
            DBGPRINT(RT_DEBUG_TRACE, "AsicRfTuningExec(R3=0x%x)\n",pAd->PortCfg.LatchRfRegs.R3);
            break;

        default:
            break;
    }
}

/*
    ==========================================================================
    Description:
        Gives CCK TX rate 2 more dB TX power.
        This routine works only in LINK UP in INFRASTRUCTURE mode.

        calculate desired Tx power in RF R3.Tx0~5,  should consider -
        1. TxPowerPercentage
        2. auto calibration based on TSSI feedback
        3. extra 2 db for CCK
        4. -10 db upon very-short distance (AvgRSSI >= -40db) to AP
    ==========================================================================
 */
VOID AsicAdjustTxPower(
    IN PRTMP_ADAPTER pAd) 
{
    ULONG R3, Channel, CurrTxPwr;

    if ((pAd->PortCfg.Channel >= 1) && (pAd->PortCfg.Channel <= 14))
        Channel = pAd->PortCfg.Channel;
    else 
        Channel = 1;  // don't have calibration info for 11A, temporarily use Channel 1
    
    // get TX Power base from E2PROM
    R3 = pAd->PortCfg.ChannelTxPower[Channel - 1];
    if (R3 > 31)  R3 = 31;

    // E2PROM setting is calibrated for maximum TX power (i.e. 100%)
    // We lower TX power here according to the percentage specified from UI
    if (pAd->PortCfg.TxPowerPercentage > 90)       // 91 ~ 100%, treat as 100% in terms of mW
    {
        // low TX power upon very-short distance to STA to solve some STA's RX problem
        // in this case, no TSSI compensation is required.
        if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOWEST_TX_POWER))
            R3 -= LOWEST_TX_POWER_DELTA;
        else if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOW_TX_POWER))
            R3 -= LOW_TX_POWER_DELTA;
//        else if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_MID_TX_POWER))
//            R3 -= MID_TX_POWER_DELTA;
    }
    else if (pAd->PortCfg.TxPowerPercentage > 60)  // 61 ~ 90%, treat as 75% in terms of mW
    {
        // low TX power upon very-short distance to STA to solve some STA's RX problem
        if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOWEST_TX_POWER))
            R3 -= LOWEST_TX_POWER_DELTA;
        else if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOW_TX_POWER))
            R3 -= LOW_TX_POWER_DELTA;
//        else if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_MID_TX_POWER))
//            R3 -= MID_TX_POWER_DELTA;
        else
            R3 -= 1;
    }
    else if (pAd->PortCfg.TxPowerPercentage > 30)  // 31 ~ 60%, treat as 50% in terms of mW
    {
        // low TX power upon very-short distance to STA to solve some STA's RX problem
        if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOWEST_TX_POWER))
            R3 -= LOWEST_TX_POWER_DELTA;
//        else if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOW_TX_POWER))
//            R3 -= LOW_TX_POWER_DELTA;
        else
            R3 -= 3;
    }
    else if (pAd->PortCfg.TxPowerPercentage > 15)  // 16 ~ 30%, treat as 25% in terms of mW
    {
        // low TX power upon very-short distance to STA to solve some STA's RX problem
//        if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOWEST_TX_POWER))
//            R3 -= LOWEST_TX_POWER_DELTA;
//        else if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOW_TX_POWER))
//            R3 -= LOW_TX_POWER_DELTA;
//        else
            R3 -= 6;
    }
    else if (pAd->PortCfg.TxPowerPercentage > 9)   // 10 ~ 15%, treat as 12.5% in terms of mW
    {
        // low TX power upon very-short distance to STA to solve some STA's RX problem
//        if (pAd->PortCfg.AvgRssi > (RSSI_TO_DBM_OFFSET - RSSI_FOR_LOWEST_TX_POWER))
//            R3 -= LOWEST_TX_POWER_DELTA;
//        else
            R3 -= 9;
    }
    else                                           // 0 ~ 9 %, treat as MIN(~3%) in terms of mW
        R3 -= 12;

    if (R3 > 31)  R3 = 0;   // negative value, set as minimum 0
    
    // compare the desired R3.TxPwr value with current R3, if not equal
    // set new R3.TxPwr
    CurrTxPwr = (pAd->PortCfg.LatchRfRegs.R3 >> 9) & 0x0000001f;
    if (CurrTxPwr != R3)
    {
        CurrTxPwr = R3;
        R3 = (pAd->PortCfg.LatchRfRegs.R3 & 0xffffc1ff) | (R3 << 9);
        RTMP_RF_IO_WRITE32(pAd, R3);
        pAd->PortCfg.LatchRfRegs.R3 = R3;
    }
    DBGPRINT(RT_DEBUG_INFO, "AsicAdjustTxPower = %d, AvgRssi = %d\n",
        CurrTxPwr, pAd->PortCfg.AvgRssi - RSSI_TO_DBM_OFFSET);
    
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicSetBssid(
    IN PRTMP_ADAPTER pAd, 
    IN MACADDR *pBssid) 
{
    ULONG         Addr4;

    COPY_MAC_ADDR(&pAd->PortCfg.Bssid, pBssid);
    Addr4 = (ULONG)(pBssid->Octet[0]) | 
            (ULONG)(pBssid->Octet[1] << 8) | 
            (ULONG)(pBssid->Octet[2] << 16) |
            (ULONG)(pBssid->Octet[3] << 24);
    RTMP_IO_WRITE32(pAd, CSR5, Addr4);
    
    Addr4 = (ULONG)(pBssid->Octet[4]) | (ULONG)(pBssid->Octet[5] << 8);
    RTMP_IO_WRITE32(pAd, CSR6, Addr4);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicDisableSync(
    IN PRTMP_ADAPTER pAd) 
{
    DBGPRINT(RT_DEBUG_TRACE, "--->Disable TSF synchronization\n");
    RTMP_IO_WRITE32(pAd, CSR14, 0x00000000);
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicEnableBssSync(
    IN PRTMP_ADAPTER pAd) 
{
    CSR12_STRUC Csr12;
    CSR13_STRUC Csr13;
    CSR14_STRUC Csr14;
    BCNCSR1_STRUC Bcncsr1;
//  BOOLEAN IsApPc;

    DBGPRINT(RT_DEBUG_TRACE, "--->AsicEnableBssSync(INFRA mode)\n");
    
    RTMP_IO_WRITE32(pAd, CSR14, 0x00000000);

    Csr12.word = 0;
    Csr12.field.BeaconInterval = pAd->PortCfg.BeaconPeriod << 4; // ASIC register in units of 1/16 TU
    Csr12.field.CfpMaxDuration = 0; // pAd->PortCfg.CfpMaxDuration << 4; // ASIC register in units of 1/16 TU
    RTMP_IO_WRITE32(pAd, CSR12, Csr12.word);

    Csr13.word = 0;
    Csr13.field.CfpPeriod = 0; // pAd->PortCfg.CfpDurRemain << 4; // ASIC register in units of 1/16 TU
    RTMP_IO_WRITE32(pAd, CSR13, Csr13.word);

    Bcncsr1.word = 0;
//    Bcncsr1.field.Preload = TBTT_PRELOAD_TIME; // we guess TBTT is 2 TU ahead of BEACON-RxEnd time
    Bcncsr1.field.BeaconCwMin = 1; // 5;
    RTMP_IO_WRITE32(pAd, BCNCSR1, Bcncsr1.word);

//  IsApPc = (CAP_IS_CF_POLLABLE_ON(pAd->PortCfg.CapabilityInfo) && 
//            CAP_IS_CF_POLL_REQ_ON(pAd->PortCfg.CapabilityInfo));
//  IsApPc = FALSE; // TODO: not support so far
    
    Csr14.word = 0;
    Csr14.field.TsfCount = 1;
    Csr14.field.TsfSync = 2; // sync TSF in AD HOC mode??
//  if (IsApPc) 
//  {
//      Csr14.field.CfpCntPreload = 0; // pAd->PortCfg.CfpCount;
//      Csr14.field.Tcfp = 1;
//  }
    Csr14.field.BeaconGen = 1;
//  Csr14.field.TbcnPreload = (pAd->PortCfg.BeaconPeriod - 30) << 4; // TODO: ???? 1 TU ???
    Csr14.field.Tbcn = 1;
    RTMP_IO_WRITE32(pAd, CSR14, Csr14.word);
    
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicLedPeriodicExec(
    IN	unsigned long data)
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;
    ULONG LedCsr = 0x0000FFFF; // 0x0000461E;
    
    pAd->PortCfg.LedCntl.fOdd = ! pAd->PortCfg.LedCntl.fOdd;

    LedCsr |= 0x00010000; // enable hardwired TX activity LED
    if (pAd->PortCfg.LedCntl.fOdd && pAd->PortCfg.LedCntl.fRxActivity) 
        LedCsr |= 0x00020000; // turn on software-based RX activity LED
    pAd->PortCfg.LedCntl.fRxActivity = FALSE;

    if (LedCsr != pAd->PortCfg.LedCntl.LastLedCsr)
    {
        pAd->PortCfg.LedCntl.LastLedCsr = LedCsr;
        RTMP_IO_WRITE32(pAd, LEDCSR, LedCsr);
    }

    pAd->PortCfg.LedCntl.BlinkTimer.expires = jiffies + (255 * HZ)/1000;
    add_timer(&pAd->PortCfg.LedCntl.BlinkTimer);
}

// pAd->PortCfg.CurrentRxAntenna
// 0xff: diversity, 0:antenna A, 1:antenna B
VOID AsicSetRxAnt(
    IN PRTMP_ADAPTER pAd) 
{
    UCHAR   RxValue;
    
    del_timer_sync(&pAd->PortCfg.RxAnt.RxAntDiversityTimer);
    pAd->PortCfg.RxAnt.AvgRssi[0] = 0;  // reset Ant-A's RSSI history
    pAd->PortCfg.RxAnt.AvgRssi[1] = 0;  // reset Ant-B's RSSI history
   	pAd->PortCfg.RxAnt.PrimaryInUsed  = TRUE;
    
    if (pAd->PortCfg.CurrentRxAntenna == 0xff)     // Diversity
    {
       	pAd->PortCfg.RxAnt.PrimaryRxAnt   = 1;  // assume ant-B
       	pAd->PortCfg.RxAnt.SecondaryRxAnt = 0;  // assume ant-A
    }
    else if (pAd->PortCfg.CurrentRxAntenna == 0)   // ant-A
    {
       	pAd->PortCfg.RxAnt.PrimaryRxAnt   = 0;  // assume ant-A
       	pAd->PortCfg.RxAnt.SecondaryRxAnt = 1;  // assume ant-B
    }
    else                                           // ant-B
    {
       	pAd->PortCfg.RxAnt.PrimaryRxAnt   = 1;  // assume ant-B
       	pAd->PortCfg.RxAnt.SecondaryRxAnt = 0;  // assume ant-A
    }

    DBGPRINT(RT_DEBUG_TRACE,"AntDiv - set RxAnt=%d, primary=%d, second=%d\n",
        pAd->PortCfg.CurrentRxAntenna, pAd->PortCfg.RxAnt.PrimaryRxAnt, pAd->PortCfg.RxAnt.SecondaryRxAnt);
    
    // use primary antenna
//  RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Rx_Configure, &RxValue);
    RxValue = pAd->PortCfg.BbpWriteLatch[BBP_Rx_Configure];
    if (pAd->PortCfg.RxAnt.PrimaryRxAnt == 0) // ant-A
        RxValue = (RxValue & 0xFC) | 0x00;
    else                                      // ant-B
		RxValue = (RxValue & 0xFC) | 0x02;
   	RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Rx_Configure, RxValue);
        
}

// switch to secondary RxAnt for a while to collect it's average RSSI
// also set a timeout routine to DO the actual evaluation. If evaluation 
// result shows a much better RSSI using secondary RxAnt, then a official
// RX antenna switch is performed.
VOID AsicEvaluateSecondaryRxAnt(
    IN PRTMP_ADAPTER pAd) 
{
    UCHAR  RxValue;

    if (pAd->PortCfg.CurrentRxAntenna != 0xff)
        return;
    
   	pAd->PortCfg.RxAnt.PrimaryInUsed  = FALSE;
    pAd->PortCfg.RxAnt.AvgRssi[pAd->PortCfg.RxAnt.SecondaryRxAnt] = 0;

    DBGPRINT(RT_DEBUG_TRACE,"AntDiv - evaluate Ant #%d\n", pAd->PortCfg.RxAnt.SecondaryRxAnt);
    
    // temporarily switch to secondary antenna
//  RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Rx_Configure, &RxValue);
    RxValue = pAd->PortCfg.BbpWriteLatch[BBP_Rx_Configure];
    if (pAd->PortCfg.RxAnt.SecondaryRxAnt == 0) // ant-A
        RxValue = (RxValue & 0xFC) | 0x00;
    else                                        // ant-B
		RxValue = (RxValue & 0xFC) | 0x02;
   	RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Rx_Configure, RxValue);

    // a one-shot timer to end the evalution
	pAd->PortCfg.RxAnt.RxAntDiversityTimer.expires = jiffies + (10 * HZ)/1000;
    add_timer(&pAd->PortCfg.RxAnt.RxAntDiversityTimer);
}

// this timeout routine collect AvgRssi[SecondaryRxAnt] and decide if
// SecondaryRxAnt is much better than PrimaryRxAnt
VOID AsicRxAntEvalTimeout(
    IN	unsigned long data) 
{
    RTMP_ADAPTER *pAd = (RTMP_ADAPTER *)data;

    if (pAd->PortCfg.RxAnt.PrimaryInUsed == TRUE)
        return;

    // 10-db or more will we consider switch antenna
    if (pAd->PortCfg.RxAnt.AvgRssi[pAd->PortCfg.RxAnt.SecondaryRxAnt] >
        (10 + pAd->PortCfg.RxAnt.AvgRssi[pAd->PortCfg.RxAnt.PrimaryRxAnt]))
    {
        UCHAR temp;
        // secondary antenna is much better than primary, switch RX antenna
        temp = pAd->PortCfg.RxAnt.PrimaryRxAnt;
        pAd->PortCfg.RxAnt.PrimaryRxAnt = pAd->PortCfg.RxAnt.SecondaryRxAnt;
        pAd->PortCfg.RxAnt.SecondaryRxAnt = temp;
        
        DBGPRINT(RT_DEBUG_TRACE,"AntDiv - Switch to Ant #%d, RSSI[0,1]=<%d, %d>\n",
            pAd->PortCfg.RxAnt.PrimaryRxAnt, pAd->PortCfg.RxAnt.AvgRssi[0], pAd->PortCfg.RxAnt.AvgRssi[1]);
    }
    else
    {
        UCHAR RxValue;
        // end of evaluation, swicth back to primary antenna
//      RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Rx_Configure, &RxValue);
        RxValue = pAd->PortCfg.BbpWriteLatch[BBP_Rx_Configure];
        if (pAd->PortCfg.RxAnt.PrimaryRxAnt == 0) // ant-A
            RxValue = (RxValue & 0xFC) | 0x00;
        else                                      // ant-B
		    RxValue = (RxValue & 0xFC) | 0x02;
   	    RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Rx_Configure, RxValue);
        DBGPRINT(RT_DEBUG_TRACE,"AntDiv - remain Ant #%d, RSSI[0,1]=<%d, %d>\n",
            pAd->PortCfg.RxAnt.PrimaryRxAnt, pAd->PortCfg.RxAnt.AvgRssi[0], pAd->PortCfg.RxAnt.AvgRssi[1]);
    }

    pAd->PortCfg.RxAnt.AvgRssi[0] = 0;  // reset Ant-A's RSSI history
    pAd->PortCfg.RxAnt.AvgRssi[1] = 0;  // reset Ant-B's RSSI history
   	pAd->PortCfg.RxAnt.PrimaryInUsed  = TRUE;
}

/*
    ==========================================================================
    Description:
    ==========================================================================
 */
VOID AsicSetSlotTime(
    IN PRTMP_ADAPTER pAd,
    IN BOOLEAN UseShortSlotTime) 
{
    CSR11_STRUC Csr11;
    CSR18_STRUC Csr18;
    CSR19_STRUC Csr19;
    
    RTMP_IO_READ32(pAd, CSR11, &Csr11.word);
    if (UseShortSlotTime)
        Csr11.field.SlotTime = 9;
    else
        Csr11.field.SlotTime = 20;
    RTMP_IO_WRITE32(pAd, CSR11, Csr11.word);

    RTMP_IO_READ32(pAd, CSR18, &Csr18.word);
    Csr18.field.PIFS = Csr18.field.SIFS + Csr11.field.SlotTime;
    RTMP_IO_WRITE32(pAd, CSR18, Csr18.word);

    Csr19.word = 0;
    Csr19.field.DIFS = Csr18.field.PIFS + Csr11.field.SlotTime;
    if (pAd->PortCfg.PhyMode == PHY_11B)
        Csr19.field.EIFS = 364;  // SIFS + ACK @1Mbps
    else
        Csr19.field.EIFS = 60;   // roughly = SIFS + ACK @6Mbps
    RTMP_IO_WRITE32(pAd, CSR19, Csr19.word);
    
#if 1
    // force using short SLOT time for FAE to demo performance only
    if (pAd->PortCfg.EnableTxBurst == 1)
        Csr11.field.SlotTime = 9;
    RTMP_IO_WRITE32(pAd, CSR11, Csr11.word);
#endif

    DBGPRINT(RT_DEBUG_TRACE, "AsicSetSlotTime(=%d us, CSR18=0x%08x, CSR19=0x%08x)\n",
        Csr11.field.SlotTime, Csr18.word, Csr19.word);
}

/*
    ==========================================================================
    Description:
        danamic tune BBP R17 to find a balance between sensibility and 
        noise isolation
    ==========================================================================
 */
VOID AsicBbpTuning(
    IN PRTMP_ADAPTER pAd)
{
    ULONG Value;
    UCHAR R17;
    ULONG FalseCcaUpperThreshold = pAd->PortCfg.BbpTuning.FalseCcaUpperThreshold << 7;
    INT dbm = pAd->PortCfg.AvgRssi - RSSI_TO_DBM_OFFSET;
    
    if ((pAd->PortCfg.BbpTuningEnable == FALSE) || (pAd->PortCfg.BbpTuning.VgcDelta == 0))
        return;

    //  RTMP_BBP_IO_READ32_BY_REG_ID(pAd, 17, &R17);
    R17 = pAd->PortCfg.BbpWriteLatch[17];

    // Rule 0.
    // when RSSI is too weak, many signals will become false CCA thus affect R17 tuning.
    // so in this case, just stop R17 tuning
    if ((pAd->Mlme.PeriodicRound > 20) && (dbm < -80))
    {
        DBGPRINT(RT_DEBUG_INFO, "Avg RSSI = %d dbm, stop R17 at 0x%x\n", dbm, R17);
        return;
    }
        
    // Rule 1. "special big-R17 for short-distance" applies on 
    //   - RT2560D and later (ver>=4), and 
    //   - only in INFRASTRUCTURE mode, and
    //   - currently not in SCANNING
    
	if (pAd->PortCfg.Rt2560Version >= RT2560_VER_D)
	{
	    if ((dbm >= RSSI_HIGH_WATERMARK) && (R17 != BBP_R17_LOW_SENSIBILITY))
	    {
	        R17 = BBP_R17_LOW_SENSIBILITY;
    	    RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, 17, R17);
            DBGPRINT(RT_DEBUG_INFO, "Avg RSSI = %d dbm, fixed R17 at 0x%x\n", dbm, R17);
	    }
	    else if ((dbm < RSSI_LOW_WATERMARK) && (R17 == BBP_R17_LOW_SENSIBILITY))
	    {
	        R17 = pAd->PortCfg.LastR17Value;
    	    RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, 17, R17);
            DBGPRINT(RT_DEBUG_INFO, "Avg RSSI = %d dbm, re-tune R17 from 0x%x\n", dbm, R17);
	    }

        // fixed R17 for short distance; no dyanamic tuning is required
        if (R17 == BBP_R17_LOW_SENSIBILITY)
            return;
	}

	// Read CCA error count
	RTMP_IO_READ32(pAd, CNT3, &Value);
	pAd->PrivateInfo.CCAErrCnt = (Value & 0x0000ffff);

	if ((pAd->PrivateInfo.CCAErrCnt > FalseCcaUpperThreshold) &&
	    (R17 < pAd->PortCfg.BbpTuning.VgcUpperBound))
	{
	    R17 += pAd->PortCfg.BbpTuning.VgcDelta;
        pAd->PortCfg.LastR17Value = R17;
	    RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, 17, R17);
        DBGPRINT(RT_DEBUG_INFO, "++R17= 0x%x\n", R17);
	}
	else if ((pAd->PrivateInfo.CCAErrCnt < pAd->PortCfg.BbpTuning.FalseCcaLowerThreshold) &&
	    (R17 > pAd->PortCfg.VgcLowerBound))
	{
	    R17 -= pAd->PortCfg.BbpTuning.VgcDelta;
        pAd->PortCfg.LastR17Value = R17;
	    RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, 17, R17);
        DBGPRINT(RT_DEBUG_INFO, "--R17= 0x%x\n", R17);
	}

    DBGPRINT(RT_DEBUG_INFO, "flase CCA= %d, Avg RSSI= %d dbm, R17= 0x%02x\n", 
	    pAd->PrivateInfo.CCAErrCnt, dbm, pAd->PortCfg.BbpWriteLatch[17]);
}
