/****************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ****************************************************************************

    Module Name:
    assoc.c
 
    Abstract:
    Handle association related requests either from WSTA or from local MLME
 
    Revision History:
    Who         When          What
    --------    ----------    ----------------------------------------------
    John Chang  08-04-2003    created for 11g soft-AP
 */

#include "rt_config.h"

UCHAR	CipherSuiteWPA[] = {
		0x00, 0x50, 0xf2, 0x01,	// oui
		0x01, 0x00,				// Version
		0x00, 0x50, 0xf2, 0x02,	// Multicast
		0x01, 0x00,				// Number of unicast
		0x00, 0x50, 0xf2, 0x02,	// unicast
		0x01, 0x00,				// number of authentication method
		0x00, 0x50, 0xf2, 0x01	// authentication
		};
UCHAR	CipherSuiteWPALen = (sizeof(CipherSuiteWPA) / sizeof(UCHAR));

UCHAR	CipherSuiteWPAPSK[] = {
		0x00, 0x50, 0xf2, 0x01,	// oui
		0x01, 0x00,				// Version
		0x00, 0x50, 0xf2, 0x02,	// Multicast
		0x01, 0x00,				// Number of unicast
		0x00, 0x50, 0xf2, 0x02,	// unicast
		0x01, 0x00,				// number of authentication method
		0x00, 0x50, 0xf2, 0x02	// authentication
		};
UCHAR	CipherSuiteWPAPSKLen = (sizeof(CipherSuiteWPAPSK) / sizeof(UCHAR));

/*  
    ==========================================================================
    Description: 
        association state machine init, including state transition and timer init
    Parameters: 
        S - pointer to the association state machine
    Note:
        The state machine looks like the following 
        
                                    ASSOC_IDLE             
        MT2_MLME_DISASSOC_REQ    mlme_disassoc_req_action 
        MT2_PEER_DISASSOC_REQ    peer_disassoc_action     
        MT2_PEER_ASSOC_REQ       drop                     
        MT2_PEER_REASSOC_REQ     drop                     
        MT2_CLS3ERR              cls3err_action           
    ==========================================================================
 */
VOID AssocStateMachineInit(
    IN	PRTMP_ADAPTER	pAd, 
    IN  STATE_MACHINE *S, 
    OUT STATE_MACHINE_FUNC Trans[]) 
{
    StateMachineInit(S, (STATE_MACHINE_FUNC*)Trans, MAX_ASSOC_STATE, MAX_ASSOC_MSG, (STATE_MACHINE_FUNC)Drop, ASSOC_IDLE, ASSOC_MACHINE_BASE);

    StateMachineSetAction(S, ASSOC_IDLE, MT2_MLME_DISASSOC_REQ, (STATE_MACHINE_FUNC)MlmeDisassocReqAction);
    StateMachineSetAction(S, ASSOC_IDLE, MT2_PEER_DISASSOC_REQ, (STATE_MACHINE_FUNC)PeerDisassocReqAction);
    StateMachineSetAction(S, ASSOC_IDLE, MT2_PEER_ASSOC_REQ, (STATE_MACHINE_FUNC)PeerAssocReqAction);
    StateMachineSetAction(S, ASSOC_IDLE, MT2_PEER_REASSOC_REQ, (STATE_MACHINE_FUNC)PeerReassocReqAction);
//    StateMachineSetAction(S, ASSOC_IDLE, MT2_CLS3ERR, (STATE_MACHINE_FUNC)Cls3errAction);
}

/*
    ==========================================================================
    Description:
        peer assoc req handling procedure
    Parameters:
        Adapter - Adapter pointer
        Elem - MLME Queue Element
    Pre:
        the station has been authenticated and the following information is stored
    Post  :
        -# An association response frame is generated and sent to the air
    ==========================================================================
 */
VOID PeerAssocReqAction(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
    MACADDR          Addr2;
    MACHDR           AssocRspHdr;
    UCHAR            SsidIe = IE_SSID, RateIe = IE_SUPP_RATES;
    USHORT           ListenInterval;
    USHORT           CapabilityInfo;
    USHORT           StatusCode = 0;
    USHORT           Aid;
    UCHAR           *OutBuffer = NULL;
    NDIS_STATUS      NStatus;
    ULONG            FrameLen = 0;
    char             Ssid[32];
    UCHAR            SsidLen;
    UCHAR            SupportedRatesLen;
    UCHAR            SupportedRates[MAX_LEN_OF_SUPPORTED_RATES];
    UCHAR            MaxSupportedRate = 0;
    int              i;
    MAC_TABLE_ENTRY  *pEntry;
    UCHAR            RSNIE_Len;
    UCHAR            RSN_IE[MAX_LEN_OF_RSNIE];

    // 1. frame sanity check
    if (! PeerAssocReqSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, &CapabilityInfo, &ListenInterval, &SsidLen, &Ssid[0], &SupportedRatesLen, &SupportedRates[0],RSN_IE, &RSNIE_Len)) 
        return;

	// for hidden SSID sake, SSID in AssociateRequest should be fully verified
    if ((SsidLen != pAd->PortCfg.SsidLen) || (NdisEqualMemory(Ssid, pAd->PortCfg.Ssid, SsidLen)==0))
        return;

    // ignore request from unwanted STA
    if (! ApCheckAccessControlList(pAd, &Addr2))
        return;
    
    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - receive ASSOC request from %02x:%02x:%02x:%02x:%02x:%02x\n",
        Addr2.Octet[0],Addr2.Octet[1],Addr2.Octet[2],Addr2.Octet[3],Addr2.Octet[4],Addr2.Octet[5]);
    
    // supported rates array may not be sorted. sort it and find the maximum rate
    for (i=0; i<SupportedRatesLen; i++)
    {
        if (MaxSupportedRate < (SupportedRates[i] & 0x7f)) 
            MaxSupportedRate = SupportedRates[i] & 0x7f;
    }            
    
    // 2. qualify this STA's auth_asoc status in the MAC table, decide StatusCode
    StatusCode = BuildAssociation(pAd, &Addr2, CapabilityInfo, MaxSupportedRate, RSN_IE, &RSNIE_Len,&Aid);

    // 3. send Association Response
    OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_KERNEL);
    if(OutBuffer == NULL)
        return;
        
    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - Send ASSOC response (Status=%d)...\n", StatusCode);
    Aid |= 0xc000; // 2 most significant bits should be ON
    MgtMacHeaderInit(pAd, &AssocRspHdr, SUBTYPE_ASSOC_RSP, 0, &Addr2, &pAd->PortCfg.Bssid);

    if ((pAd->PortCfg.PhyMode == PHY_11BG_MIXED) || (pAd->PortCfg.PhyMode == PHY_11G))
    {
        UCHAR SupportedRatesLen = 8;
        UCHAR ExtendedRatesIe = IE_EXT_SUPP_RATES;
        UCHAR ExtendedRatesLen = pAd->PortCfg.SupportedRatesLen - SupportedRatesLen;
        
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                          sizeof(MACHDR),       &AssocRspHdr,
                          2,                    &CapabilityInfo,
                          2,                    &StatusCode,
                          2,                    &Aid,
                          1,                    &RateIe,
                          1,                    &SupportedRatesLen,
                          SupportedRatesLen,    pAd->PortCfg.SupportedRates,
                          1,                    &ExtendedRatesIe,
                          1,                    &ExtendedRatesLen,
                          ExtendedRatesLen,     &pAd->PortCfg.SupportedRates[SupportedRatesLen],
                          END_OF_ARGS);
    }
    else
    {
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                          sizeof(MACHDR),       &AssocRspHdr,
                          2,                    &CapabilityInfo,
                          2,                    &StatusCode,
                          2,                    &Aid,
                          1,                    &RateIe,
                          1,                    &pAd->PortCfg.SupportedRatesLen,
                          pAd->PortCfg.SupportedRatesLen, pAd->PortCfg.SupportedRates,
                          END_OF_ARGS);
    }
    MiniportMMRequest(pAd, OutBuffer, FrameLen);
}

/*
    ==========================================================================
    Description:
        mlme reassoc req handling procedure
    Parameters:
        Elem - 
    Pre:
        -# SSID  (Adapter->PortCfg.ssid[])
        -# BSSID (AP address, Adapter->PortCfg.bssid)
        -# Supported rates (Adapter->PortCfg.supported_rates[])
        -# Supported rates length (Adapter->PortCfg.supported_rates_len)
        -# Tx power (Adapter->PortCfg.tx_power)
    ==========================================================================
 */
VOID PeerReassocReqAction(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
    MACADDR          ApAddr, Addr2;
    MACHDR           ReassocRspHdr;
    UCHAR            SsidIe = IE_SSID, RateIe = IE_SUPP_RATES;
    USHORT           CapabilityInfo, ListenInterval;
    USHORT           StatusCode = MLME_SUCCESS;
    USHORT           Aid = 1;
    ULONG            FrameLen = 0;
    NDIS_STATUS      NStatus;
    UCHAR            *OutBuffer = NULL;
    char             Ssid[32];
    UCHAR            SsidLen;
    UCHAR            SupportedRatesLen;
    UCHAR            SupportedRates[MAX_LEN_OF_SUPPORTED_RATES];
    UCHAR            MaxSupportedRate = 0;
    int              i;
    MAC_TABLE_ENTRY  *pEntry;
    UCHAR            RSNIE_Len;
    UCHAR            RSN_IE[MAX_LEN_OF_RSNIE];
   
    // 1. frame sanity check
    if (! PeerReassocReqSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, &CapabilityInfo, &ListenInterval, &ApAddr, &SsidLen, &Ssid[0], &SupportedRatesLen, &SupportedRates[0],RSN_IE, &RSNIE_Len)) 
        return;

	// for hidden SSID sake, SSID in AssociateRequest should be fully verified
    if ((SsidLen != pAd->PortCfg.SsidLen) || (NdisEqualMemory(Ssid, pAd->PortCfg.Ssid, SsidLen)==0))
        return;
    
    // ignore request from unwanted STA
    if (! ApCheckAccessControlList(pAd, &Addr2))
        return;
    
    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - receive RE-ASSOC request from %02x:%02x:%02x:%02x:%02x:%02x\n",
        Addr2.Octet[0],Addr2.Octet[1],Addr2.Octet[2],Addr2.Octet[3],Addr2.Octet[4],Addr2.Octet[5]);
    
    // supported rates array may not be sorted. sort it and find the maximum rate
    for (i=0; i<SupportedRatesLen; i++)
    {
        if (MaxSupportedRate < (SupportedRates[i] & 0x7f))
            MaxSupportedRate = SupportedRates[i] & 0x7f;
    }            
    
    // 2. qualify this STA's auth_asoc status in the MAC table, decide StatusCode
    StatusCode = BuildAssociation(pAd, &Addr2, CapabilityInfo, MaxSupportedRate, RSN_IE, &RSNIE_Len, &Aid);

    // 3. reply Re-association Response
    OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_KERNEL);
    if(OutBuffer == NULL)
        return;

    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - Send RE-ASSOC response (Status = %d)...\n", StatusCode);
    Aid |= 0xc000; // 2 most significant bits should be ON
    MgtMacHeaderInit(pAd, &ReassocRspHdr, SUBTYPE_REASSOC_RSP, 0, &Addr2, &pAd->PortCfg.Bssid);

    if ((pAd->PortCfg.PhyMode == PHY_11BG_MIXED) || (pAd->PortCfg.PhyMode == PHY_11G))
    {
        UCHAR SupportedRatesLen = 8;
        UCHAR ExtendedRatesIe = IE_EXT_SUPP_RATES;
        UCHAR ExtendedRatesLen = pAd->PortCfg.SupportedRatesLen - SupportedRatesLen;
        
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                          sizeof(MACHDR),       &ReassocRspHdr,
                          2,                    &CapabilityInfo,
                          2,                    &StatusCode,
                          2,                    &Aid,
                          1,                    &RateIe,
                          1,                    &SupportedRatesLen,
                          SupportedRatesLen,    pAd->PortCfg.SupportedRates,
                          1,                    &ExtendedRatesIe,
                          1,                    &ExtendedRatesLen,
                          ExtendedRatesLen,     &pAd->PortCfg.SupportedRates[SupportedRatesLen],
                          END_OF_ARGS);
    }
    else
    {
        MakeOutgoingFrame(OutBuffer,            &FrameLen,
                          sizeof(MACHDR),       &ReassocRspHdr,
                          2,                    &CapabilityInfo,
                          2,                    &StatusCode,
                          2,                    &Aid,
                          1,                    &RateIe,
                          1,                    &pAd->PortCfg.SupportedRatesLen, 
                          pAd->PortCfg.SupportedRatesLen,   pAd->PortCfg.SupportedRates,
                          END_OF_ARGS);
    }
    MiniportMMRequest(pAd, OutBuffer, FrameLen);
}

/*
    ==========================================================================
    Description:
        left part of IEEE 802.11/1999 p.374 
    Parameters:
        Elem - MLME message containing the received frame
    ==========================================================================
 */
VOID PeerDisassocReqAction(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
    MACADDR       Addr2;
    USHORT        Reason;
    MAC_TABLE_ENTRY       *pEntry;

    if (! PeerDisassocReqSanity(pAd, Elem->Msg, Elem->MsgLen, &Addr2, &Reason)) 
        return;

    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - receive DIS-ASSOC request from %02x:%02x:%02x:%02x:%02x:%02x\n",
        Addr2.Octet[0],Addr2.Octet[1],Addr2.Octet[2],Addr2.Octet[3],Addr2.Octet[4],Addr2.Octet[5]);
    
    pEntry = MacTableLookup(pAd, &Addr2);
    if (pEntry)
    {
        ApLogEvent(pAd, &Addr2, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, &Addr2);
    }
}


/*
    ==========================================================================
    Description:
        Upper layer orders to disassoc s STA
    Parameters:
        Elem -
    ==========================================================================
 */
VOID MlmeDisassocReqAction(
    IN PRTMP_ADAPTER pAd, 
    IN MLME_QUEUE_ELEM *Elem) 
{
    MLME_DISASSOC_REQ_STRUCT *DisassocReq;
    MACHDR                DisassocHdr;
    CHAR                 *OutBuffer = NULL;
    ULONG                 FrameLen = 0;
    NDIS_STATUS           NStatus;
    MAC_TABLE_ENTRY       *pEntry;

    DisassocReq = (MLME_DISASSOC_REQ_STRUCT *)(Elem->Msg);

    pEntry = MacTableLookup(pAd, &DisassocReq->Addr);
    if (pEntry)
    {
        ApLogEvent(pAd, &DisassocReq->Addr, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, &DisassocReq->Addr);
    }

    // 2. send out a DISASSOC request frame
    OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_KERNEL);
    if(OutBuffer == NULL)
        return;
    
    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - MLME disassociates %02x:%02x:%02x:%02x:%02x:%02x; Send DISASSOC request\n",
        DisassocReq->Addr.Octet[0],DisassocReq->Addr.Octet[1],DisassocReq->Addr.Octet[2],
        DisassocReq->Addr.Octet[3],DisassocReq->Addr.Octet[4],DisassocReq->Addr.Octet[5]);
    MgtMacHeaderInit(pAd, &DisassocHdr, SUBTYPE_DISASSOC, 0, &DisassocReq->Addr, &pAd->PortCfg.Bssid);
    MakeOutgoingFrame(OutBuffer,            &FrameLen, 
                        sizeof(MACHDR),       &DisassocHdr, 
                        2,                    &DisassocReq->Reason, 
                        END_OF_ARGS);
    MiniportMMRequest(pAd, OutBuffer, FrameLen);
}


/*
    ==========================================================================
    Description:
        right part of IEEE 802.11/1999 page 374
    Note: 
        This event should never cause ASSOC state machine perform state
        transition, and has no relationship with CNTL machine. So we separate
        this routine as a service outside of ASSOC state transition table.
    ==========================================================================
 */
VOID Cls3errAction(
    IN PRTMP_ADAPTER pAd, 
    IN PMACADDR      pAddr) 
{
    MACHDR                DisassocHdr;
    CHAR                 *OutBuffer = NULL;
    ULONG                 FrameLen = 0;
    USHORT                Reason = REASON_CLS3ERR;
    MAC_TABLE_ENTRY       *pEntry;

    pEntry = MacTableLookup(pAd, pAddr);
    if (pEntry)
    {
        //ApLogEvent(pAd, &pAddr, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, pAddr);
    }
    
    // 2. send out a DISASSOC request frame
    OutBuffer = kmalloc(MAX_LEN_OF_MLME_BUFFER, GFP_KERNEL);
    if(OutBuffer == NULL)
        return;
    
    DBGPRINT(RT_DEBUG_TRACE, "ASSOC - Class 3 Error, Send DISASSOC frame to %02x:%02x:%02x:%02x:%02x:%02x\n",
        pAddr->Octet[0],pAddr->Octet[1],pAddr->Octet[2],pAddr->Octet[3],pAddr->Octet[4],pAddr->Octet[5]);
    MgtMacHeaderInit(pAd, &DisassocHdr, SUBTYPE_DISASSOC, 0, pAddr, &pAd->PortCfg.Bssid);
    MakeOutgoingFrame(OutBuffer,            &FrameLen, 
                      sizeof(MACHDR),       &DisassocHdr, 
                      2,                    &Reason, 
                      END_OF_ARGS);
    MiniportMMRequest(pAd, OutBuffer, FrameLen);
}
 
 /*
     ==========================================================================
     Description:
        search for entire MAC table to find next available AID to be used
        if none is available, return 0
     ==========================================================================
  */
USHORT AssignAid(
    IN PRTMP_ADAPTER pAd)
{
    USHORT Aid=0, i;
    UCHAR AidUsed[MAX_LEN_OF_MAC_TABLE];

    // find all used AID    
    NdisZeroMemory(AidUsed, MAX_LEN_OF_MAC_TABLE);
    for (i=0;i<MAX_LEN_OF_MAC_TABLE;i++)
        if (pAd->MacTab.Content[i].Valid)
            AidUsed[pAd->MacTab.Content[i].Aid] = 1;

    // find smallest unused AID
    for (i=1; i<MAX_LEN_OF_MAC_TABLE; i++)
        if (AidUsed[i]==0)
        {
            Aid = i;
            break;
        }

    return Aid;
}

/*
    ==========================================================================
    Description:
       assign a new AID to the newly associated/re-associated STA and
       decide its MaxSupportedRate and CurrTxRate. Both rates should not
       exceed AP's capapbility
    Return:
       MLME_SUCCESS - association successfully built
       others - association failed due to resource issue
    ==========================================================================
 */
USHORT BuildAssociation(
    IN PRTMP_ADAPTER pAd,
    IN PMACADDR      pAddr,
    IN USHORT        CapabilityInfo,
    IN UCHAR         MaxSupportedRateIn500Kbps,
    IN UCHAR         *RSN,
    IN UCHAR         *pRSNLen,
    OUT USHORT       *pAid)
{
    USHORT           StatusCode = MLME_SUCCESS;
    PMAC_TABLE_ENTRY pEntry;
    UCHAR            MaxSupportedRate = RATE_11;

    switch (MaxSupportedRateIn500Kbps)
    {
        case 200: MaxSupportedRate = RATE_100;  break;
        case 144: MaxSupportedRate = RATE_72;   break;
        case 108: MaxSupportedRate = RATE_54;   break;
        case 96:  MaxSupportedRate = RATE_48;   break;
        case 72:  MaxSupportedRate = RATE_36;   break;
        case 48:  MaxSupportedRate = RATE_24;   break;
        case 36:  MaxSupportedRate = RATE_18;   break;
        case 24:  MaxSupportedRate = RATE_12;   break;
        case 18:  MaxSupportedRate = RATE_9;    break;
        case 12:  MaxSupportedRate = RATE_6;    break;
        case 22:  MaxSupportedRate = RATE_11;   break;
        case 11:  MaxSupportedRate = RATE_5_5;  break;
        case 4:   MaxSupportedRate = RATE_2;    break;
        case 2:   MaxSupportedRate = RATE_1;    break;
        default:  MaxSupportedRate = RATE_11;   break;
    }

    if ((pAd->PortCfg.PhyMode == PHY_11G) && (MaxSupportedRate < RATE_FIRST_OFDM_RATE))
        return MLME_ASSOC_REJ_DATA_RATE;

    pEntry = MacTableLookup(pAd, pAddr);
    if (pEntry && ((pEntry->Sst == SST_AUTH) || (pEntry->Sst == SST_ASSOC)))
    {
        // TODO:
        // should qualify other parameters, for example - capablity, supported rates, listen interval, ... etc
        // to decide the Status Code
        *pAid = AssignAid(pAd);
        pEntry->Aid = *pAid;
        
        NdisMoveMemory(pEntry->RSN_IE, RSN, *pRSNLen);
        pEntry->RSNIE_Len = *pRSNLen;
        pEntry->Flag = 0;

        if (*pAid == 0)
            StatusCode = MLME_ASSOC_REJ_UNABLE_HANDLE_STA;
        else
        {
            pEntry->Sst = SST_ASSOC;
            pEntry->MaxSupportedRate = min(pAd->PortCfg.MaxTxRate, MaxSupportedRate);
            pEntry->CapabilityInfo = CapabilityInfo;
            if (pAd->PortCfg.AuthMode == Ndis802_11AuthModeWPAPSK)
            {
                pEntry->PrivacyFilter = Ndis802_11PrivFilter8021xWEP;
                pEntry->WpaState = AS_INITPSK;
            }
            else if (pAd->PortCfg.AuthMode >= Ndis802_11AuthModeWPA)
            {
                pEntry->PrivacyFilter = Ndis802_11PrivFilter8021xWEP;
                pEntry->WpaState = AS_AUTHENTICATION;
            }
            pEntry->Flag = 0;
            pEntry->PortSecured = WPA_802_1X_PORT_NOT_SECURED;
            DBGPRINT(RT_DEBUG_TRACE, "  WepStatus= %d, pEntry->WpaState = %d  \n", pAd->PortCfg.WepStatus, pEntry->WpaState);
#if 0
            // If dynamic rate switching is enabled, starts from s more reliable rate to 
            // increase STA's DHCP succeed rate
            if (pAd->PortCfg.EnableAutoRateSwitching)
            {
                pEntry->CurrTxRate = min(pEntry->MaxSupportedRate, (UCHAR)RATE_11); 
            }
            else
#endif
                pEntry->CurrTxRate = pEntry->MaxSupportedRate;

            ApLogEvent(pAd, &pEntry->Addr, EVENT_ASSOCIATED);
            ApUpdateCapabilityAndErpIe(pAd);
            
            if ((pEntry->RetryTimerRunning == FALSE) && (pAd->PortCfg.AuthMode == Ndis802_11AuthModeWPAPSK))
            {               // retry up to 3 times at interval 1 second
                DBGPRINT(RT_DEBUG_TRACE, "   add_timer  Timer   ====>>>   \n");
                pEntry->RetryTimerRunning = TRUE;
                init_timer(&pEntry->RetryTimer);
                pEntry->RetryTimer.expires = jiffies + WPA_RETRY_EXEC_INTV;
                pEntry->RetryTimer.data = (unsigned long)pAd;
                pEntry->RetryTimer.function = &WPARetryExec;
                add_timer(&pEntry->RetryTimer);
                pEntry->ReTryCounter = PAIR_HS_RETRY_TIMER_CTR;
            }
            
			DBGPRINT(RT_DEBUG_TRACE,"assign AID=%d to STA %02x:%02x:%02x:%02x:%02x:%02x, MaxSupportedRate=%d Mbps, CurrTxRate=%d Mbps\n", 
                *pAid, pAddr->Octet[0], pAddr->Octet[1], pAddr->Octet[2], pAddr->Octet[3],
                pAddr->Octet[4], pAddr->Octet[5], RateIdToMbps[pEntry->MaxSupportedRate], RateIdToMbps[pEntry->CurrTxRate]);
			DBGPRINT(RT_DEBUG_TRACE, "RSNIE_Len = 0x%x, pEntry->RSNIE_Len = %d\n",*pRSNLen,pEntry->RSNIE_Len);
			DBGPRINT(RT_DEBUG_TRACE, "pEntry->PrivacyFilter = %x\n",pEntry->PrivacyFilter );
            StatusCode = MLME_SUCCESS;
        }
    }
    else // CLASS 3 error should have been handled beforehand; here should be MAC table full
        StatusCode = MLME_ASSOC_REJ_UNABLE_HANDLE_STA;

    return StatusCode;
}

