/******************************************************************************

    File:               isl38xx_mgt.c
    Author:             W.Termorshuizen
    Version:            0.3.0.4
    Created:            June 19th, 2002
    Updated:            November 12th, 2002
    Company:            Intersil Americas Inc.
    Description:        This file contains the linux driver for the Intersil
                        Wireless LAN adapters.
    History:	    	Maintained in file isl38xx.h
		
******************************************************************************/
#define __KERNEL_SYSCALLS__


#include <linux/version.h>
#ifdef MODULE
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif


#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/wait.h>

#include <asm/processor.h>      // Processor type for cache alignment.
#include <asm/bitops.h>
#include <asm/io.h>

#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include <pcmcia/driver_ops.h>
#include <asm/uaccess.h>
#include <linux/unistd.h>
#include <linux/wireless.h>

#include "bloboidv2.h"      // additional types and defs for isl38xx fw
#include "isl38xx.h"

#if WIRELESS_EXT > 12
#if EVENT_SUPPORT == WE_EVENTS
#include <net/iw_handler.h>
#endif	// EVENT_SUPPORT
#endif	// WIRELESS_EXT > 12 




#define DEBUG( f, args... ) 	K_DEBUG( f, pc_debug, args )

/******************************************************************************
        Global variable definition section
******************************************************************************/
extern int pc_debug;
extern int init_mode;
extern int init_channel;
extern int init_bsstype;
extern int init_wep;
extern int init_filter;
extern int init_wds;
extern int init_authen;
extern int init_dot1x;
extern char *init_ssid;

DECLARE_WAIT_QUEUE_HEAD(mgmt_queue_wait);
int last_key_id = 0;
struct iw_priv_args privtab[] =
{
        { SIOCIWFIRSTPRIV + 0x00, IW_PRIV_TYPE_INT | 1, 0,
                "setauthenten" },
        { SIOCIWFIRSTPRIV + 0x01, 0, IW_PRIV_TYPE_INT | 1,
                "getauthenten" },
        { SIOCIWFIRSTPRIV + 0x02, IW_PRIV_TYPE_INT | 1, 0,
                "setunencrypt" },
        { SIOCIWFIRSTPRIV + 0x03, 0, IW_PRIV_TYPE_INT | 1,
                "getunencrypt" },
        { SIOCIWFIRSTPRIV + 0x04, IW_PRIV_TYPE_INT | 1, 0,
                "setprivinvok" },
        { SIOCIWFIRSTPRIV + 0x05, 0, IW_PRIV_TYPE_INT | 1,
                "getprivinvok" },
        { SIOCIWFIRSTPRIV + 0x06, IW_PRIV_TYPE_INT | 1, 0,
                "setdefkeyid" },
        { SIOCIWFIRSTPRIV + 0x07, 0, IW_PRIV_TYPE_INT | 1,
                "getdefkeyid" },
        { SIOCIWFIRSTPRIV + 0x08, IW_PRIV_TYPE_CHAR | 16, 0,
                "setdefkeyx" },
        { SIOCIWFIRSTPRIV + 0x09, IW_PRIV_TYPE_INT | 1, IW_PRIV_TYPE_CHAR | 16,
                "getdefkeyx" },
        { SIOCIWFIRSTPRIV + 0x0A, 0, IW_PRIV_TYPE_INT | 4,
                "getprivstat" },
        { SIOCIWFIRSTPRIV + 0x0B, 0, IW_PRIV_TYPE_INT | IW_MAX_BITRATES,
                "getsuprates" },
        { SIOCIWFIRSTPRIV + 0x0C, IW_PRIV_TYPE_INT | 1, 0,
                "setfixedrate" },
        { SIOCIWFIRSTPRIV + 0x0D, 0, IW_PRIV_TYPE_INT | 1,
                "getfixedrate" },
        { SIOCIWFIRSTPRIV + 0x0E, IW_PRIV_TYPE_INT | 1, 0,
                "setbeaconper" },
        { SIOCIWFIRSTPRIV + 0x0F, 0, IW_PRIV_TYPE_INT | 1,
                "getbeaconper" },
        { SIOCIWFIRSTPRIV + 0x10, IW_PRIV_TYPE_CHAR | 12, 0,
                "wdslinkadd" },
        { SIOCIWFIRSTPRIV + 0x11, IW_PRIV_TYPE_CHAR | 12, 0,
                "wdslinkdel" },
        { SIOCIWFIRSTPRIV + 0x12, IW_PRIV_TYPE_CHAR | 12, 0,
                "eapauthen" },
        { SIOCIWFIRSTPRIV + 0x13, IW_PRIV_TYPE_CHAR | 12, 0,
                "eapunauth" },
        { SIOCIWFIRSTPRIV + 0x14, IW_PRIV_TYPE_INT | 1, 0,
                "setdot1xen" },
        { SIOCIWFIRSTPRIV + 0x15, 0, IW_PRIV_TYPE_INT | 1,
                "getdot1xen" },
        { SIOCIWFIRSTPRIV + 0x16, IW_PRIV_TYPE_CHAR | 32, 0,
                "setstakeyx" },	// <address>:<key_nr>:<key>
        { SIOCIWFIRSTPRIV + 0x17, IW_PRIV_TYPE_CHAR | 32, IW_PRIV_TYPE_CHAR | 32,
                "getstakeyx" },	// <address>:<key_nr> -> <key>
        { SIOCIWFIRSTPRIV + 0x18, IW_PRIV_TYPE_INT | 1, 0,
                "setrtsthresh" },
        { SIOCIWFIRSTPRIV + 0x19, 0, IW_PRIV_TYPE_INT | 1,
                "getrtsthresh" },

        { SIOCIWFIRSTPRIV + 0x1F, 0, IW_PRIV_TYPE_BYTE | 16,
                "checktraps" }
};


/******************************************************************************
    System scheduling functions
******************************************************************************/
void schedule_wait(int period)
{
    // wait the period indicated
    set_current_state(TASK_INTERRUPTIBLE);
    schedule_timeout((period * HZ) / 1000);
}

/******************************************************************************
    Frame (re)format functions
******************************************************************************/
void pimfor_encode_header(int operation, unsigned long oid,
                                 int device_id, int flags, int length,
                                 pimfor_header * header)
{
    // byte oriented members
    header->version = PIMFOR_VERSION;
    header->operation = operation;
    header->device_id = device_id;
    header->flags = flags;

    // word oriented members with byte order depending on the flags
    if (flags & PIMFOR_FLAG_LITTLE_ENDIAN)
    {
        // use little endian coding
        header->oid[3] = oid >> 24;
        header->oid[2] = oid >> 16;
        header->oid[1] = oid >> 8;
        header->oid[0] = oid;
        header->length[3] = 0;
        header->length[2] = 0;
        header->length[1] = length >> 8;
        header->length[0] = length;
    }
    else
    {
        // use big endian coding
        header->oid[0] = oid >> 24;
        header->oid[1] = oid >> 16;
        header->oid[2] = oid >> 8;
        header->oid[3] = oid;
        header->length[0] = 0;
        header->length[1] = 0;
        header->length[2] = length >> 8;
        header->length[3] = length;
    }
}

void pimfor_decode_header(pimfor_header * header, int *operation,
                                 unsigned long *oid, int *device_id,
                                 int *flags, int *length)
{
    // byte oriented members
    *operation = header->operation;
    *device_id = header->device_id;
    *flags = header->flags;

    // word oriented members with byte order depending on the flags
    if (*flags & PIMFOR_FLAG_LITTLE_ENDIAN)
    {
        // use little endian coding
        *oid = ((int) header->oid[3] << 24) |
            ((int) header->oid[2] << 16) |
            ((int) header->oid[1] << 8) | ((int) header->oid[0]);
        *length = ((int) header->length[1] << 8) |
            ((int) header->length[0]);
    }
    else
    {
        // use big endian coding
        *oid = ((int) header->oid[0] << 24) |
            ((int) header->oid[1] << 16) |
            ((int) header->oid[2] << 8) | ((int) header->oid[3]);
        *length = ((int) header->length[2] << 8) |
            ((int) header->length[3]);
    }
}

/******************************************************************************
    Device Interface functions
******************************************************************************/
int isl38xx_transmit_pimfor(isl38xx_private * private_config)
{
    int fragments, counter;
    isl38xx_control_block *control_block = private_config->control_block;
    isl38xx_fragment *fragment;
    u32 index, reg, in_queue, in_shadowq;
    queue_entry *entry;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_transmit_pimfor \n");
#endif

    // check whether there is a pending response in the receive queue which
    // needs to be handled first before applying a new frame
    if( in_queue = isl38xx_in_queue(control_block, ISL38XX_CB_RX_MGMTQ),
        in_queue != 0 )
    {
        // receive queue not empty
#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "receive queue not empty\n");
#endif
        return 0;
    }

        // read the number of entries in the transmit queue and check whether the
        // transmit queue is empty for receiving the next pimfor frame
    if( in_queue = isl38xx_in_queue(control_block, ISL38XX_CB_TX_MGMTQ),
        in_queue != 0 )
    {
        // transmit queue not empty
#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "transmit queue not empty\n");
#endif
        return 0;
    }

        // first check whether a clean up should take place on the shadow queue
    while( private_config->index_mgmt_tx > in_queue )
    {
        // free the first entry in the shadow queue which contains the
        // PIMFOR frame transmitted to the device
        get_queue(private_config->remapped_device_base,
                &private_config->mgmt_tx_shadowq, &entry);
        put_queue(private_config->remapped_device_base,
                &private_config->mgmt_tx_freeq, entry);

        // decrement the real management index
        private_config->index_mgmt_tx--;
    }

    // check whether there is at least one frame in the shadowq
    if( in_shadowq = queue_size( private_config->remapped_device_base,
        &private_config->mgmt_tx_shadowq ), !in_shadowq )
    {
        // no entries in the shadowq, quit
#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "shadowq empty, no more frames\n");
#endif
        return 0;
    }

    // there is a frame in the shadowq, get its fragments count
    entry = (queue_entry *) &private_config->mgmt_tx_shadowq;
    entry = (queue_entry *) entry->next;
    fragments = entry->fragments;

    // transmit the fragment(s) by placing them in the transmit queue
    for (counter = 0; counter < fragments; counter++)
    {
        u16 fragment_flags;
        u32	*drvcur = &control_block->driver_curr_frag[ISL38XX_CB_TX_MGMTQ];

        // determine the fragment flags depending on the nr of fragments
        fragment_flags = fragments - counter > 1 ? FRAGMENT_FLAG_MF : 0;

        // get a pointer to the destination fragment using the frame counter
        // and link it with the free queue entry data block
//        dcache_inval_range(drvcur, sizeof(u32));
        index = le32_to_cpu(*drvcur) % ISL38XX_CB_MGMT_QSIZE;
        fragment = &control_block->tx_data_mgmt[index];

#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "Entry address %p\n", entry);
        DEBUG(SHOW_TRACING, "Index in transmit queue %i\n", index);
        DEBUG(SHOW_TRACING, "Fragment address %p\n", fragment);
#endif

        fragment->address = cpu_to_le32(entry->dev_address);
        fragment->size = cpu_to_le16(entry->size);
        fragment->flags = cpu_to_le16(fragment_flags);

        // increment the index pointer, wrapping is not done on queue but
        // on variable type range
        add_le32p(drvcur, 1);

        // increment the queue entry pointer
        entry = (queue_entry *) entry->next;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Frame with [%i] fragments added to transmit queue \n",
      fragments);
#endif

    // increment the read management queue index
    private_config->index_mgmt_tx += fragments;

    // trigger the device by setting the Update bit in the Device Int register
    reg = readl(private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    reg |= ISL38XX_DEV_INT_UPDATE;
    writel(reg, private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    return 0;
}

int isl38xx_queue_pimfor(isl38xx_private * private_config,
                                  int operation, unsigned long oid,
                                  int flags, void *data, int length,
                                  int use_tunnel_oid)
{
    int fragments, counter;
    u32 source, destination;
    queue_entry *entry;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_queue_pimfor \n");
#endif

    // determine the amount of fragments needed for the frame in case of a SET
    if (use_tunnel_oid)
    {
        fragments = (length + 2 * PIMFOR_HEADER_SIZE) / MGMT_FRAME_SIZE;
        if ((length + 2 * PIMFOR_HEADER_SIZE) % MGMT_FRAME_SIZE != 0)
            fragments++;
    }
    else
    {
        fragments = (length + PIMFOR_HEADER_SIZE) / MGMT_FRAME_SIZE;
        if ((length + PIMFOR_HEADER_SIZE) % MGMT_FRAME_SIZE != 0)
        fragments++;
    }

    // check whether data is actually written to the device
    if (operation == PIMFOR_OP_GET)
    {
        // operation is get, no data is actually written, reset the fragments
        // counter to send a single fragment
        fragments = 1;
    }

    // check whether the number of fragments doesn't exceed the maximal possible
    // amount of frames in the transmit queue
    if( fragments > ISL38XX_CB_MGMT_QSIZE )
    {
        // Oops, the fragment count exceeds the maximal value
        DEBUG(SHOW_ERROR_MESSAGES, "Error: Frame with [%i] fragments is too "
            "large \n", fragments );
        return -1;
    }

    // check whether there is enough room for the fragment(s) in the freeq
    if( queue_size( private_config->remapped_device_base,
        &private_config->mgmt_tx_freeq ) < fragments )
    {
        // Oops, the management freeq does not have enough entries
        DEBUG(SHOW_ERROR_MESSAGES, "Error: Management FreeQ not enough entries "
            "for [%i] fragments \n", fragments );
        return -1;
    }

    // queue the frame(s) in the transmit shadow queue
    for (counter = 0, source = (u32) data; counter < fragments; counter++)
    {
        u16 fragment_size = 0;

        // get an entry from the free queue for hosting the frame
        get_queue(private_config->remapped_device_base,
            &private_config->mgmt_tx_freeq, &entry);

        // set the destination address to the entry data block for building
        // the PIMFOR frame there
        destination = entry->host_address;

#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "Entry address %p\n", entry);
        DEBUG(SHOW_TRACING, "Destination address %x\n", destination);
        DEBUG(SHOW_TRACING, "Destination for device %x\n", entry->dev_address);
#endif

        // place the PIMFOR header in the first fragment
        if (counter == 0)
        {
            // check whether the frame should have a tunnel oid
            if (use_tunnel_oid)
            {
                // start with a tunnel oid header
                pimfor_encode_header(PIMFOR_OP_SET, OID_INL_TUNNEL,
                     PIMFOR_DEV_ID_MHLI_MIB, flags,
                     length + PIMFOR_HEADER_SIZE,
                     (pimfor_header *) destination);

                // set the fragment size and increment the destin pointer
                fragment_size += PIMFOR_HEADER_SIZE;
                destination += PIMFOR_HEADER_SIZE;
            }

            // create the header directly in the fragment data area
            pimfor_encode_header(operation, oid, PIMFOR_DEV_ID_MHLI_MIB,
                 flags, length, (pimfor_header *) destination);

#if VERBOSE > SHOW_ERROR_MESSAGES 
            // display the buffer contents for debugging
            display_buffer((char *) destination, 12);
#endif

            // set the fragment size and increment the destin pointer
            fragment_size += PIMFOR_HEADER_SIZE;
            destination += PIMFOR_HEADER_SIZE;
        }

        // add the data to the fragment in case of a set or for some
		// specific objects
        if( (operation == PIMFOR_OP_SET) || (oid == DOT11_OID_STAKEY) )
        {
            int size = (length > (MGMT_FRAME_SIZE - fragment_size)) ?
                MGMT_FRAME_SIZE - fragment_size : length;
            memcpy((void *) destination, (void *) source, size);

            // set the fragment size and flag members
            fragment_size += size;

#if VERBOSE > SHOW_ERROR_MESSAGES 
            // display the buffer contents for debugging
            display_buffer((char *) destination, 12);
#endif

            // update the source pointer and the data length
            source += size;
            length -= size;
        }

        // save the number of fragments still to go in the frame together with
        // the fragment size
        entry->fragments = fragments - counter;
        entry->size = fragment_size;

        // store the entry in the tx shadow queue
        put_queue(private_config->remapped_device_base,
            &private_config->mgmt_tx_shadowq, entry);
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Frame with [%i] fragments added to shadow tx queue \n",
        fragments);
#endif

    return 0;
}

int isl38xx_receive_pimfor(isl38xx_private * private_config)
{
    struct net_device *nw_device = private_config->my_module;
    isl38xx_control_block *control_block = private_config->control_block;
    pimfor_header *received_header;
    char *received_data;
    int device_id, operation, flags, length;
    unsigned long oid;
    queue_entry *entry, *new_entry;
    u32 reg;

#if EVENT_SUPPORT == WE_EVENTS
    union iwreq_data wrq;
#endif

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_receive_pimfor \n");
#endif

    // get the received response from the shadow queue
    get_queue(private_config->remapped_device_base,
                &private_config->mgmt_rx_shadowq, &entry);

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Entry address 0x%p\n", entry);
    DEBUG(SHOW_TRACING, "Entry device address 0x%x\n", entry->dev_address);
#endif

    // determine how the process the fragment depending on the flags in the
    // PIMFOR header, check for tunnel oid PIMFOR frames
    received_header = (pimfor_header *) entry->host_address;
    if (pimfor_decode_header(received_header, &operation, &oid, &device_id,
        &flags, &length), oid == OID_INL_TUNNEL)
    {
#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_PIMFOR_FRAMES, "PIMFOR: Tunnel OID frame received \n");
#endif

        // PMFOR frame in fragment has the tunnel oid, redo the header decode
        received_header = (pimfor_header *) (entry->host_address +
            PIMFOR_HEADER_SIZE);
        pimfor_decode_header(received_header, &operation, &oid, &device_id,
            &flags, &length);
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_PIMFOR_FRAMES,
        "PIMFOR: op %i, oid 0x%lx, device %i, flags 0x%x length 0x%x \n",
        operation, oid, device_id, flags, length);

    // display the buffer contents for debugging
    display_buffer((char *) received_header, sizeof(pimfor_header));
    display_buffer(((char *) received_header) + PIMFOR_HEADER_SIZE, length );
#endif

    // PIMFOR frame created by this driver, handle its contents
    // check whether the frame contains driver initialization data
    if (oid == GEN_OID_MACADDRESS)
    {
        // a MAC address response message, store its contents
        if ((operation == PIMFOR_OP_RESPONSE) && (length == 6))
        {
            // copy the MAC address received
            received_data = (char *) received_header + PIMFOR_HEADER_SIZE;
            memcpy(nw_device->dev_addr, received_data, 6);
        }
    }
    else if (oid == OID_INL_MODE)
    {
        // the mode response message, store its contents
        if ((operation == PIMFOR_OP_RESPONSE) && (length == 4))
        {
            // copy the low byte which is at the first byte location
            received_data = (char *) received_header + PIMFOR_HEADER_SIZE;
            private_config->mode = (int) *received_data;
        }
    }
    else if (oid == GEN_OID_LINKSTATE)
    {
        // the mode response message, store its contents
        if ((operation == PIMFOR_OP_RESPONSE) && (length == 4))
        {
            // copy the low byte which is at the first byte location
            received_data = (char *) received_header + PIMFOR_HEADER_SIZE;
            private_config->linkstate = (int) *received_data;
        }
    }

    // perform a check on the freeq whether the entry can be queued to other
    // queues or must be hold to make sure there are enough
    if( get_queue(private_config->remapped_device_base,
        &private_config->mgmt_rx_freeq, &new_entry) == 0)
    {
        // succeeded in getting a new entry from the freeq
        // refresh the device mgmt queue entry
        control_block->rx_data_mgmt[private_config->index_mgmt_rx].address =
            cpu_to_le32(new_entry->dev_address);

        // put the new entry in the mgmt rx shadow queue
        put_queue(private_config->remapped_device_base,
                &private_config->mgmt_rx_shadowq, new_entry);

        // deliver the held entry to the appropriate queue
        if (flags & PIMFOR_FLAG_APPLIC_ORIGIN)
        {
            // PIMFOR frame originated by higher application, send the entire
            // fragment up without processing

#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "PIMFOR Application Originated \n");
#endif

            // send the entry to the PIMFOR receive queue for further processing
            put_queue(private_config->remapped_device_base,
                &private_config->pimfor_rx_queue, entry);
        }
        else
        {
            // check whether the frame concerns a response, error or trap
            if (operation == PIMFOR_OP_TRAP)
            {
                // received a trap frame
                received_data = (char *) received_header + PIMFOR_HEADER_SIZE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
                DEBUG(SHOW_TRAPS,
                    "TRAP: oid 0x%lx, device %i, flags 0x%x length %i\n",
                    oid, device_id, flags, length);
#endif

                // check whether the TRAP should be deleted or queued depending
                // on the queue size and whether the queue is read
                if( queue_size( private_config->remapped_device_base,
                    &private_config->trap_rx_queue ) > MAX_TRAP_RX_QUEUE )
                {
                    // free the entry to the freeq
                    put_queue(private_config->remapped_device_base,
                            &private_config->mgmt_rx_freeq, entry);
                }
                else
                {
                    // send the entry to the TRAP receive q
                    put_queue(private_config->remapped_device_base,
                            &private_config->trap_rx_queue, entry);

#if EVENT_SUPPORT == WE_EVENTS
    	    	    // signal user space application(s) that a TRAP is queued
		    // use the AP MAC address get ioctl for signalling
		    wrq.ap_addr.sa_family = ARPHRD_ETHER;
		    memcpy( wrq.ap_addr.sa_data, nw_device->dev_addr, 6 );
		    wireless_send_event( nw_device, SIOCGIWAP, &wrq, "ISIL" );
#elif EVENT_SUPPORT == ISL_EVENTS


#endif // EVENT_SUPPORT				
                }
            }
            else
            {
                // received either a repsonse or an error
                // send the entry to the IOCTL receive q for further processing
                put_queue(private_config->remapped_device_base,
                        &private_config->ioctl_rx_queue, entry);

                // signal the waiting procs that a response has been received
                wake_up_interruptible(&mgmt_queue_wait);
            }
        }
    }
    else
    {
        // not succeeded, use the held one and drop its contents
        // refresh the device mgmt queue entry
        control_block->rx_data_mgmt[private_config->index_mgmt_rx].address =
            cpu_to_le32(entry->dev_address);

        // put the old entry back in the mgmt rx shadow queue
        put_queue(private_config->remapped_device_base,
            &private_config->mgmt_rx_shadowq, entry);
    }

    // increment the driver read pointer
    add_le32p(&control_block->driver_curr_frag[ISL38XX_CB_RX_MGMTQ], 1);

    // trigger the device by setting the Update bit in the Device Int register
    reg = readl(private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    reg |= ISL38XX_DEV_INT_UPDATE;
    writel(reg, private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // increment and wrap the read index for the rx management queue
    private_config->index_mgmt_rx = (private_config->index_mgmt_rx + 1) %
        ISL38XX_CB_MGMT_QSIZE;

    return 0;
}

int wait_for_oid_response(isl38xx_private * private_config, long oid,
                                 void **data)
{
    queue_entry *entry;
    pimfor_header *received_header;
    int q_size, counter, device_id, operation, flags, rlength, rvalue = 0;
    unsigned long received_oid;

    DECLARE_WAITQUEUE(wait, current);

    // check for other processes already waiting for response
    if (private_config->processes_in_wait == 0)
    {
        // no process in the wait queue, flush the queue
        while (queue_size(private_config->remapped_device_base,
            &private_config->ioctl_rx_queue))
        {
            // flush each entry remaining in the queue
            get_queue(private_config->remapped_device_base,
                &private_config->ioctl_rx_queue, &entry);
            put_queue(private_config->remapped_device_base,
                &private_config->mgmt_rx_freeq, entry);
        }
    }

    // trigger the transmission of the management shadowq before going to sleep
    isl38xx_transmit_pimfor( private_config );

    // add the process to the wait queue and increment the process counter
    add_wait_queue(&mgmt_queue_wait, &wait);
    current->state = TASK_INTERRUPTIBLE;
    private_config->processes_in_wait++;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Process enter sleep mode\n");
#endif

    while (1)
    {
        // go to sleep with the alarm bells on sharp
        interruptible_sleep_on_timeout(&mgmt_queue_wait, ISL38XX_WAIT_CYCLE);

        // hello there, I got woken up because there is either a response out
        // there or the bells went off, check whether the lock is up
        while (private_config->ioctl_queue_lock)
        {
            // the lock is up, another process is reading the ioctl rx queue
            schedule_timeout(1 * HZ / 10);
        }

        // It's my turn, set the lock rightaway
        private_config->ioctl_queue_lock = 1;

        // check the IOCTL receive queue for responses
        if( q_size = queue_size(private_config->remapped_device_base,
            &private_config->ioctl_rx_queue), q_size != 0 )
        {
            // something in the queue, check every entry in the queue
            for (counter = 0; counter < q_size; counter++)
            {
                // get the entry from the queue and read the header
                get_queue(private_config->remapped_device_base,
                    &private_config->ioctl_rx_queue, &entry);
                received_header = (pimfor_header *) entry->host_address;

                // check whether the oid response is for me
                if( pimfor_decode_header(received_header, &operation,
                    &received_oid, &device_id, &flags,&rlength ),
                    oid == received_oid )
                {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                    DEBUG(SHOW_TRACING, "IOCTL response found in queue.\n");
#endif

                    // found my response, check the response type
                    if( operation == PIMFOR_OP_RESPONSE )
                    {
                        // return the address of the data field
                        *data = (void *) received_header + PIMFOR_HEADER_SIZE;

#if VERBOSE > SHOW_ERROR_MESSAGES 
                        DEBUG(SHOW_TRACING, "Response source 0x%p\n", *data);
#endif
                    }
                    else
                    {
                        // error or not supported operation
                        rvalue = -ENOTSUPP;
                    }

                    // free the entry
                    put_queue(private_config->remapped_device_base,
                        &private_config->mgmt_rx_freeq, entry);

                    // continue and leave the sleepingroom
                    goto leave;
                }

                // put the entry back in the queue
                put_queue(private_config->remapped_device_base,
                    &private_config->ioctl_rx_queue, entry);
            }

            // checked all entries in the queue without success
            rvalue = -ETIMEDOUT;
            goto leave;
        }
        else
        {
            // nothing in the queue, that's got to be a timeout
            rvalue = -EBUSY;
            goto leave;
        }
    }

leave:
    private_config->processes_in_wait--;
    private_config->ioctl_queue_lock = 0;
    current->state = TASK_RUNNING;
    remove_wait_queue(&mgmt_queue_wait, &wait);
    return rvalue;
}

/******************************************************************************
    Driver Management functions
******************************************************************************/
void isl38xx_initialize( isl38xx_private *private_config )
{
    char configmode[4] = { INL_CONFIG_MANUALRUN, 0, 0, 0 };
    char cardmode[4] = { 0, 0, 0, 0 };
    char bsstype[4] =  { 0, 0, 0, 0 };
    char channel[4] =  { 0, 0, 0, 0 };
    char authen[4] = { 0, 0, 0, 0 };
    char invoke[4] = { 0, 0, 0, 0 };
    char exunencrypt[4] = { 0, 0, 0, 0 };
    char dot1xen[4] = { 0, 0, 0, 0 };
    struct obj_key key = { DOT11_PRIV_WEP, 0, "" };
    struct obj_ssid essid_obj = { sizeof( CARD_DEFAULT_SSID ) - 1,
                                CARD_DEFAULT_SSID };
    struct obj_buffer psm_buffer = { cpu_to_le32(PSM_BUFFER_SIZE), 0 };

    // store all device initialization management frames in the
    // management transmit shadow queue
    cardmode[0] = (char) init_mode;
    channel[0] = (char) init_channel;
    channel[1] = (char) (init_channel >> 8);
    bsstype[0] = (char) init_bsstype;
    authen[0] = (char) init_authen;
    invoke[0] = (char) init_wep;
    exunencrypt[0] = (char) init_filter;
    dot1xen[0] = (char) init_dot1x;
    if( init_ssid != NULL )
    {
        // parameter specified when loading the module
        essid_obj.length = strlen( init_ssid );
        strcpy( essid_obj.octets, init_ssid );
    }
    psm_buffer.base = (void *) cpu_to_le32(private_config->device_psm_buffer);
    if( init_wds )
    {
        // enable wds mode in the host interface
        configmode[0] |= INL_CONFIG_WDS;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "cardmode: %i \n", init_mode );
    DEBUG(SHOW_TRACING, "channel: %i \n", init_channel );
    DEBUG(SHOW_TRACING, "bsstype: %i \n", init_bsstype );
    DEBUG(SHOW_TRACING, "wep: %i \n", init_wep );
    DEBUG(SHOW_TRACING, "authen: %i \n", init_authen );
    DEBUG(SHOW_TRACING, "filter: %i \n", init_filter );
    DEBUG(SHOW_TRACING, "wds: %i \n", init_wds );
    DEBUG(SHOW_TRACING, "ssid: %i %s \n", essid_obj.length, essid_obj.octets );
    DEBUG(SHOW_TRACING, "psm: %p %li\n", psm_buffer.base, psm_buffer.size );
#endif

    // store all device initialization management frames in the
    // management transmit shadow queue

    //  get firmware configuration data
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        GEN_OID_MACADDRESS, 0, NULL, 6, 0);

    // set the configuration mode to manual run & wds option
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        OID_INL_CONFIG, 0, &configmode[0], 4, 0);

    // set the mode for stopping the device
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        OID_INL_MODE, 0, &cardmode[0], 4, 0);

    // set the bsstype
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_BSSTYPE, 0, &bsstype[0], 4, 0);

    // set the channel
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_CHANNEL, 0, &channel[0], 4, 0);

    // set the SSID
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_SSID, 0, (void *) &essid_obj, sizeof(struct obj_ssid), 0);

    // set the PSM buffer object
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_PSMBUFFER, 0, (void *) &psm_buffer,
        sizeof(struct obj_buffer), 0);

	// set the authentication enable option
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, 
		DOT11_OID_AUTHENABLE, 0, &authen[0], 4, 0);

    // set the privacy invoked mode
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_PRIVACYINVOKED, 0, &invoke[0], 4, 0);

    // set the exunencrypted object
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_EXUNENCRYPTED, 0, &exunencrypt[0], 4, 0);

    // set default key 1
    strcpy( key.key, CARD_DEFAULT_KEY1 );
    key.length = strlen(CARD_DEFAULT_KEY1) > sizeof(key.key) ?
        sizeof(key.key) : strlen(CARD_DEFAULT_KEY1);
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_DEFKEY1, 0,
        &key, sizeof(struct obj_key), 0);

    // set default key 2
    strcpy( key.key, CARD_DEFAULT_KEY2 );
    key.length = strlen(CARD_DEFAULT_KEY2) > sizeof(key.key) ?
        sizeof(key.key) : strlen(CARD_DEFAULT_KEY2);
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_DEFKEY2, 0,
        &key, sizeof(struct obj_key), 0);

    // set default key 3
    strcpy( key.key, CARD_DEFAULT_KEY3 );
    key.length = strlen(CARD_DEFAULT_KEY3) > sizeof(key.key) ?
        sizeof(key.key) : strlen(CARD_DEFAULT_KEY3);
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_DEFKEY3, 0,
        &key, sizeof(struct obj_key), 0);

    // set default key 4
    strcpy( key.key, CARD_DEFAULT_KEY4 );
    key.length = strlen(CARD_DEFAULT_KEY4) > sizeof(key.key) ?
        sizeof(key.key) : strlen(CARD_DEFAULT_KEY4);
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_DEFKEY4, 0,
        &key, sizeof(struct obj_key), 0);

    // set the dot1x enable object
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_DOT1XENABLE, 0, &dot1xen[0], 4, 0);

    // set the mode again for starting the device
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        OID_INL_MODE, 0, &cardmode[0], 4, 0);

    // call the transmit function for triggering the transmitting of the
    // first frame in the shadow queue
    isl38xx_transmit_pimfor( private_config );
}

int isl38xx_ioctl_getrange(struct net_device *nwdev,
    struct iw_range *range )
{
    isl38xx_private *private_config = nwdev->priv;
    struct iw_range range_obj;
    char *data;
    int rvalue, counter;

    // request the device for the rates
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
       DOT11_OID_RATES, 0, NULL, IW_MAX_BITRATES+1, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_RATES,
        (void **) &data), rvalue != 0)
    {
        // error in getting the response, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "getrates: bufaddr %p \n", range );
#endif

    // set the wireless extension version number
    range_obj.we_version_compiled = WIRELESS_EXT;

    for( counter=0; counter < IW_MAX_BITRATES; counter++, data++ )
    {
        // check whether the rate value is set
        if( *data )
        {
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG( SHOW_TRACING, "rate: basic %i value %x \n",
                (*data & 0x80) >> 7, *data & 0x7F );
#endif

            // convert the bitrates to Mbps
            range_obj.bitrate[counter] = /*(double)*/(*data & 0x7F);
            range_obj.bitrate[counter] *= 1000000;

            // increment the number of rates counter
            range_obj.num_bitrates++;
        }
    }

    // return the data if the destination buffer is defined
    if (copy_to_user( (void *) range, (void *) &range_obj,
        sizeof(struct iw_range)))
        return -EFAULT;

    return 0;
}

int isl38xx_ioctl_setrange(struct net_device *nwdev,
    struct iw_range *range )
{
    return 0;
}

#if WIRELESS_EXT > 11

int isl38xx_ioctl_getstats(struct net_device *nwdev,
    struct iw_statistics *stats )
{
    isl38xx_private *private_config = nwdev->priv;
    struct iw_statistics stats_obj = { 0 };
    long *data;
    int rvalue;

    // request the device for the rates
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
       DOT11_OID_MPDUTXFAILED, 0, NULL, 4, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_MPDUTXFAILED,
        (void **) &data), rvalue != 0)
    {
        // error in getting the response, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "MPDU Tx Failed: %lu \n", *data );
        stats_obj.discard.retries = *data;
#endif

    // return the data if the destination buffer is defined
    if (copy_to_user( (void *) stats, (void *) &stats_obj,
        sizeof(struct iw_statistics)))
        return -EFAULT;

    return 0;
}


int isl38xx_ioctl_setstats(struct net_device *nwdev,
    struct iw_statistics *stats )
{
    return 0;
}

#endif

int isl38xx_ioctl_getauthenten(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_AUTHENABLE, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_AUTHENABLE,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    *flag = (long) *data;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Authenticate enable: %i\n", (int) *flag );
#endif

    return 0;
}

int isl38xx_ioctl_setauthenten(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    char filter[4] = { 0, 0, 0, 0 };
    void *data;

    filter[0] = (char) *flag;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Authenticate enable: %i\n", (int) filter[0] );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_AUTHENABLE,
        0, &filter[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_AUTHENABLE,
        &data);
}

int isl38xx_ioctl_getprivfilter(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_EXUNENCRYPTED, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_EXUNENCRYPTED,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    *flag = (long) *data;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Privacy filter: %i\n", (int) *flag );
#endif

    return 0;
}

int isl38xx_ioctl_setprivfilter(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    char filter[4] = { 0, 0, 0, 0 };
    void *data;

    filter[0] = (char) *flag;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Privacy filter: %i\n", (int) filter[0] );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_EXUNENCRYPTED,
        0, &filter[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_EXUNENCRYPTED,
        &data);
}

int isl38xx_ioctl_getprivinvoke(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_PRIVACYINVOKED, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_PRIVACYINVOKED,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    *flag = (long) *data;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Privacy invoked: %i\n", (int) *flag );
#endif

    return 0;
}

int isl38xx_ioctl_setprivinvoke(struct net_device *nwdev,
        struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    char invoke[4] = { 0, 0, 0, 0 };
    void *data;

    invoke[0] = (char) *flag;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Privacy invoked: %i\n", (int) invoke[0] );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_PRIVACYINVOKED, 0, &invoke[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_PRIVACYINVOKED,
        &data);
}

int isl38xx_ioctl_getdefkeyid(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *key = (long *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_DEFKEYID, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_DEFKEYID,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Default Key ID %i\n", (int) *key );
#endif
    *key = (long) *data;

    return 0;
}

int isl38xx_ioctl_setdefkeyid(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *key = (long *) wrq->u.data.pointer;
    char keyid[4] = { 0, 0, 0, 0 };
    void *data;

    keyid[0] = (char) *key;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Default Key ID: %i\n", (int) keyid[0] );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_DEFKEYID, 0, &keyid[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_DEFKEYID,
        &data);
}

int isl38xx_ioctl_getdefkeyx(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    int *key = (int *) wrq->u.data.pointer;
    int rvalue;
    struct obj_key *pkey;
    unsigned long oid;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Default Key %i\n", (int) *key );
#endif

    switch (*key)
    {
        case 0:
            oid = DOT11_OID_DEFKEY1;
            break;

        case 1:
            oid = DOT11_OID_DEFKEY2;
            break;

        case 2:
            oid = DOT11_OID_DEFKEY3;
            break;

        case 3:
            oid = DOT11_OID_DEFKEY4;
            break;

        default:
            return -EOPNOTSUPP;
    }

    // store the key value for future write operations
    last_key_id = *key;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET, oid, 0, NULL,
        sizeof(struct obj_key), 0);
    if (rvalue = wait_for_oid_response(private_config, oid, (void **) &pkey),
        rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    if (copy_to_user( (char *) key, (void *) pkey->key, pkey->length))
        return -EFAULT;
    wrq->u.data.length = sizeof(pkey->key);

    return 0;
}

int isl38xx_ioctl_setdefkeyx(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    char *key = (char *) wrq->u.data.pointer;
    void *data;
    struct obj_key defkey;
    unsigned long oid;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Default Key %i \n", last_key_id );
#endif

    switch (last_key_id)
    {
        case 0:
            oid = DOT11_OID_DEFKEY1;
            break;

        case 1:
            oid = DOT11_OID_DEFKEY2;
            break;

        case 2:
            oid = DOT11_OID_DEFKEY3;
            break;

        case 3:
            oid = DOT11_OID_DEFKEY4;
            break;

        default:
            return -EOPNOTSUPP;
    }

    defkey.type = DOT11_PRIV_WEP;
    defkey.length = strlen(key) > sizeof(defkey.key) ? sizeof(defkey.key) :
        strlen(key);
    if (copy_from_user(defkey.key, key, defkey.length))
        return -EFAULT;

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, oid, 0, &defkey,
        sizeof(struct obj_key), 0);

    return wait_for_oid_response(private_config, oid, &data);
}

int isl38xx_ioctl_getprivstat(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    int *stat = (int *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get transmit rejected frame count
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_PRIVTXREJECTED, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_PRIVTXREJECTED,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "MPDU Tx Rejected %i\n", (int) *data );
#endif
    *stat = (int) *data;
    stat++;

    // send a get object to get rejected unencrypted frame count
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_PRIVRXPLAIN, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_PRIVRXPLAIN,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "MPDU Rx Plain %i\n", (int) *data );
#endif
    *stat = (int) *data;
    stat++;

    // send a get object to get rejected crypted frame count
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_PRIVRXFAILED, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_PRIVRXFAILED,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "MPDU Rx Failed %i\n", (int) *data );
#endif
    *stat = (int) *data;
    stat++;

    // send a get object to get received no key frames
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_PRIVRXNOKEY, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_PRIVRXNOKEY,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "MPDU Rx No Key %i\n", (int) *data );
#endif
    *stat = (int) *data;

    return 0;
}

int isl38xx_ioctl_getsuprates(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    int *rates = (int *) wrq->u.data.pointer;
    double rate;
    char *data;
    int rvalue, counter;

    // request the device for the rates
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
       DOT11_OID_SUPPORTEDRATES, 0, NULL, IW_MAX_BITRATES+1, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_SUPPORTEDRATES,
        (void **) &data), rvalue != 0)
    {
        // error in getting the response, return the value
        return rvalue;
    }

    for( counter=0; counter < IW_MAX_BITRATES; counter++, data++ )
    {
        // check whether the rate value is set
        if( *data )
        {
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG( SHOW_TRACING, "rate: basic %i value %x \n",
                (*data & 0x80) >> 7, *data & 0x7F );
#endif

            // convert the bitrates to Mbps
            rate = (double)(*data & 0x7F) * 500000;
            *rates = (int) rate;
            rates++;
        }
    }

    return 0;
}

int isl38xx_ioctl_getfixedrate(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *rate = (long *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_ALOFT_FIXEDRATE, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_ALOFT_FIXEDRATE,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    *rate = (long) *data;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Fixed Rate index: %i\n", (int) *rate );
#endif

    return 0;
}

int isl38xx_ioctl_setfixedrate(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *index = (long *) wrq->u.data.pointer;
    char rate[4] = { 0, 0, 0, 0 };
    void *data;

    rate[0] = (char) *index;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Fixed Rate index: %i\n", (int) rate[0] );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_ALOFT_FIXEDRATE,
        0, &rate[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_ALOFT_FIXEDRATE,
        &data);
}

int isl38xx_ioctl_getbeaconper(struct net_device *nwdev,
        struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_BEACONPERIOD, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_BEACONPERIOD,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    *flag = (long) *data;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get Beacon Period: %i\n", (int) *flag );
#endif

    return 0;
}

int isl38xx_ioctl_setbeaconper(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    char period[4] = { 0, 0, 0, 0 };
    void *data;

    period[0] = (char) *flag;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set Beacon Period: %i\n", (int) period[0] );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_BEACONPERIOD,
        0, &period[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_BEACONPERIOD,
        &data);
}

int isl38xx_ioctl_getrate(struct net_device *nwdev,
    struct iw_param *bitrate )
{
    isl38xx_private *private_config = nwdev->priv;
    double rate;
    int rvalue;
    char *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        GEN_OID_LINKSTATE, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, GEN_OID_LINKSTATE,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Current rate value %i\n", (int) *data );
#endif

    rate = (double) *data * 500000;
    bitrate->value = (int) rate;

    return 0;
}

int isl38xx_ioctl_wdslinkadd(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    unsigned char *address = (unsigned char *) wrq->u.data.pointer;
    unsigned char macaddr[10];
    void *data;
    int rvalue;
    u32 *address_hi, *entry_hi;
    u16 *address_lo, *entry_lo;

    // convert the address string to a MAC address
    string_to_macaddress( address, macaddr );

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "WDS Link Add addr %02X:%02X:%02X:%02X:%02X:%02X\n",
        macaddr[0], macaddr[1], macaddr[2],
        macaddr[3], macaddr[4], macaddr[5] );
#endif

    // transmit the object to the card
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_WDSLINKADD, 0, &macaddr[0], 6, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if( rvalue = wait_for_oid_response(private_config, DOT11_OID_WDSLINKADD,
        &data ), !rvalue )
    {
        // a successfull addition of a WDS Link address, update the table
        address_hi = (u32 *) &macaddr[0];
        address_lo = (u16 *) &macaddr[4];

        entry_hi = (u32 *) &private_config->wds_link_table[0].address[0];
        entry_lo = (u16 *) &private_config->wds_link_table[0].address[4];

        *entry_hi = *address_hi;
        *entry_lo = *address_lo;

        private_config->wds_link_count = 1;
    }

    return rvalue;
}

int isl38xx_ioctl_wdslinkdel(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    unsigned char *address = (unsigned char *) wrq->u.data.pointer;
    unsigned char macaddr[10];
    void *data;
    int rvalue;
    u32 *address_hi, *entry_hi;
    u16 *address_lo, *entry_lo;

    // convert the address string to a MAC address
    string_to_macaddress( address, macaddr );

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "WDS Link Remove addr %02X:%02X:%02X:%02X:%02X:%02X\n",
        macaddr[0], macaddr[1], macaddr[2],
        macaddr[3], macaddr[4], macaddr[5] );
#endif

    // transmit the object to the card
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_WDSLINKREMOVE, 0, &macaddr[0], 6, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if( rvalue = wait_for_oid_response(private_config, DOT11_OID_WDSLINKREMOVE,
        &data ), !rvalue )
    {
        // a successfull addition of a WDS Link address, update the table
        address_hi = (u32 *) &macaddr[0];
        address_lo = (u16 *) &macaddr[4];

        entry_hi = (u32 *) &private_config->wds_link_table[0].address[0];
        entry_lo = (u16 *) &private_config->wds_link_table[0].address[4];

        *entry_hi = 0x00000000;
        *entry_lo = 0x0000;

        private_config->wds_link_count = 0;
    }

    return rvalue;
}

int isl38xx_ioctl_eapauthen(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    unsigned char *address = (unsigned char *) wrq->u.data.pointer;
    unsigned char macaddr[10];
    void *data;

    // convert the address string to a MAC address
    string_to_macaddress( address, macaddr );

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "EAP Auth STA addr %02X:%02X:%02X:%02X:%02X:%02X\n",
        macaddr[0], macaddr[1], macaddr[2],
        macaddr[3], macaddr[4], macaddr[5] );
#endif

    // transmit the object to the card
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_EAPAUTHSTA, 0, &macaddr[0], 6, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    return wait_for_oid_response(private_config, DOT11_OID_EAPAUTHSTA, &data);
}

int isl38xx_ioctl_eapunauth(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    unsigned char *address = (unsigned char *) wrq->u.data.pointer;
    unsigned char macaddr[10];
    void *data;

    // convert the address string to a MAC address
    string_to_macaddress( address, macaddr );

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "EAP Unauth STA addr %02X:%02X:%02X:%02X:%02X:%02X\n",
        macaddr[0], macaddr[1], macaddr[2],
        macaddr[3], macaddr[4], macaddr[5] );
#endif

    // transmit the object to the card
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_EAPUNAUTHSTA, 0, &macaddr[0], 6, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    return wait_for_oid_response(private_config, DOT11_OID_EAPUNAUTHSTA, &data);
}


int isl38xx_ioctl_getdot1xen(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    long *flag = (long *) wrq->u.data.pointer;
#endif
    int rvalue;
    char *data;

    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_DOT1XENABLE, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_DOT1XENABLE,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get .1x flag: %i\n", (int) *data );
        *flag = (long) *data;
#endif

    return 0;
}

int isl38xx_ioctl_setdot1xen(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *flag = (long *) wrq->u.data.pointer;
    char enabled[4] = { 0, 0, 0, 0 };
    void *data;

    enabled[0] = (char) *flag;
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set .1x flag: %i\n", (int) enabled[0] );
#endif

    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_DOT1XENABLE, 0, &enabled[0], 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_DOT1XENABLE,
        &data);
}

int isl38xx_ioctl_getstakeyx(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    unsigned char *parameter = (unsigned char *) wrq->u.data.pointer;
	struct obj_stakey stakey, *pstakey;
    int rvalue;
	
    // convert the address string to a MAC address
    string_to_macaddress( parameter, stakey.address );

	// increase the read parameter pointer and read the key number
	parameter += 13;
	stakey.keyid = *parameter - '0';;
	
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "sta addr %02X:%02X:%02X:%02X:%02X:%02X key %i\n",
        (unsigned char) stakey.address[0], (unsigned char) stakey.address[1], 
		(unsigned char) stakey.address[2], (unsigned char) stakey.address[3],
		(unsigned char) stakey.address[4], (unsigned char) stakey.address[5],
		(int) stakey.keyid );
#endif

    // send a get object to get the stations entry
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET, DOT11_OID_STAKEY, 0,
		&stakey, sizeof(struct obj_stakey), 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_STAKEY, 
		(void **) &pstakey), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "key %s length %i\n", pstakey->key, pstakey->length );
#endif

    parameter = (unsigned char *) wrq->u.data.pointer;
	if (copy_to_user( (char *) parameter, (void *) &pstakey->key, 
		sizeof(pstakey->key) ))
        return -EFAULT;
		
    wrq->u.data.length = sizeof(pstakey->key)+1;
	parameter += sizeof(pstakey->key);
	*parameter = '\0';	

    return 0;
}

int isl38xx_ioctl_setstakeyx(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    unsigned char *parameter = (unsigned char *) wrq->u.data.pointer;
	struct obj_stakey stakey;
	void *data;
		
    // convert the address string to a MAC address
    string_to_macaddress( parameter, stakey.address );

	// increase the read parameter pointer and read the key number
	parameter += 13;
	stakey.keyid = *parameter - '0';
	
	// increase the read parameter pointer to read the key
	parameter += 2;
    stakey.type = DOT11_PRIV_WEP;
    stakey.length = strlen(parameter) > sizeof(stakey.key) ? sizeof(stakey.key) :
        strlen(parameter);
    if (copy_from_user( stakey.key, parameter, stakey.length))
        return -EFAULT;
	
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "sta addr %02X:%02X:%02X:%02X:%02X:%02X key %i\n",
        (unsigned char) stakey.address[0], (unsigned char) stakey.address[1], 
		(unsigned char) stakey.address[2], (unsigned char) stakey.address[3],
		(unsigned char) stakey.address[4], (unsigned char) stakey.address[5],
		(int) stakey.keyid );
#endif

    // send a get object to get the stations entry
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET, DOT11_OID_STAKEY, 0,
		&stakey, sizeof(struct obj_stakey), 0);
    return wait_for_oid_response(private_config, DOT11_OID_STAKEY, &data );
}

int isl38xx_ioctl_getessid(struct net_device *nwdev,
     struct iw_point *erq)
{
    isl38xx_private *private_config = nwdev->priv;
    struct obj_ssid *essid_obj;
    int rvalue;

    // request the device for the essid
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
       DOT11_OID_SSID, 0, NULL, sizeof(struct obj_ssid), 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_SSID,
        (void **) &essid_obj), rvalue != 0)
    {
        // error in getting the response, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "getessid: parameter at %p\n", (void *) essid_obj);
#endif

    // copy the data to the return structure
    erq->flags = 1;             // set ESSID to ON for Wireless Extensions
    erq->length = essid_obj->length + 1;
    if (erq->pointer)
        if (copy_to_user(erq->pointer, essid_obj->octets, erq->length))
            return -EFAULT;

    return 0;
}

int isl38xx_ioctl_setessid(struct net_device *nwdev,
                                 struct iw_point *erq)
{
    isl38xx_private *private_config = nwdev->priv;
    struct obj_ssid essid_obj;
    void *data;

    // prepare the structure for the set object
    essid_obj.length = erq->length - 1;

    if (copy_from_user(essid_obj.octets, erq->pointer, essid_obj.length))
        return -EFAULT;

    // request the device for the essid
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_SSID, 0, (void *) &essid_obj, sizeof(struct obj_ssid), 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    return wait_for_oid_response(private_config, DOT11_OID_SSID, &data);
}

int isl38xx_ioctl_getbssid(struct net_device *nwdev, char *id)
{
    isl38xx_private *private_config = nwdev->priv;
    void *data;
    int rvalue;

    // request the device for the essid
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_BSSID, 0, NULL, 6, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_BSSID, &data),
        rvalue != 0)
    {
        // error in getting the response, return the value
        return rvalue;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "getbssid: parameter at %p\n", data);
#endif

    // return the data if the destination buffer is defined
    if (id)
        if (copy_to_user(id, data, 6))
            return -EFAULT;

    return 0;
}

int isl38xx_ioctl_setbssid(struct net_device *nwdev, char *id)
{
    isl38xx_private *private_config = nwdev->priv;
    char bssid[6];
    void *data;

    // prepare the structure for the set object
    if (copy_from_user(&bssid[0], id, 6))
        return -EFAULT;

    // send a set object to set the bssid
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_BSSID, 0, &bssid[0], 6, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    return wait_for_oid_response(private_config, DOT11_OID_BSSID, &data);
}

int isl38xx_ioctl_getmode(struct net_device *nwdev, int *mode)
{
    isl38xx_private *private_config = nwdev->priv;
    int bsstype;
    int rvalue;
    char *data;

    // send a get object to get the bsstype
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_BSSTYPE, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_BSSTYPE,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }
    else
    {
        // returned the bsstype, store it
        bsstype = (int) *data;
    }

    // send a get object to get the mode
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        OID_INL_MODE, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, OID_INL_MODE,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }


#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "getmode: parameter at %p\n", data);
#endif

    // translate the cards mode to the wireless mode values as good as possible
    switch ((int) *data)
    {
        case INL_MODE_AP:
            *mode = IW_MODE_MASTER;
            break;

        case INL_MODE_CLIENT:
            if (bsstype == DOT11_BSSTYPE_IBSS)
                *mode = IW_MODE_ADHOC;
            else if (bsstype == DOT11_BSSTYPE_INFRA)
                *mode = IW_MODE_INFRA;
            else
                *mode = IW_MODE_AUTO;
            break;

        default:                    // DOT11_BSSTYPE_ANY
            *mode = IW_MODE_AUTO;
            break;
    }

    return 0;
}


int isl38xx_ioctl_setmode(struct net_device *nwdev, int *mode)
{
    isl38xx_private *private_config = nwdev->priv;
    char cardmode[4] = { 0, 0, 0, 0 };
    char bsstype[4] = { 0, 0, 0, 0 };
    void *data;

    switch (*mode)
    {
        case IW_MODE_ADHOC:
            bsstype[0] = DOT11_BSSTYPE_IBSS;
            cardmode[0] = INL_MODE_CLIENT;
            break;

        case IW_MODE_INFRA:
            bsstype[0] = DOT11_BSSTYPE_INFRA;
            cardmode[0] = INL_MODE_CLIENT;
            break;

        case IW_MODE_MASTER:
            bsstype[0] = DOT11_BSSTYPE_INFRA;
            cardmode[0] = INL_MODE_AP;
            break;

        case IW_MODE_AUTO:
            bsstype[0] = DOT11_BSSTYPE_ANY;
            cardmode[0] = INL_MODE_CLIENT;
            break;

        default:
            return -EOPNOTSUPP;
    }

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_BSSTYPE, 0, &bsstype[0], 4, 0);
    wait_for_oid_response(private_config, DOT11_OID_BSSTYPE, &data);

    // send a set object to set the mode by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        OID_INL_MODE, 0, &cardmode[0], 4, 0);
    return wait_for_oid_response(private_config, OID_INL_MODE, &data);
}

int isl38xx_ioctl_getfreq(struct net_device *nwdev,
                                struct iw_freq *frq)
{
    isl38xx_private *private_config = nwdev->priv;
    char *data;
    long channel;
    int rvalue;

    // request the device for the channel
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_CHANNEL, 0, NULL, 4, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_CHANNEL,
        (void **) &data), rvalue != 0)
    {
        // error in getting the response, return the value
        return rvalue;
    }

        // convert the data to the channel parameter
    channel = (int) *data;
    data++;
    channel += ((int) *data) << 8;
    data++;
    channel += ((int) *data) << 16;
    data++;
    channel += ((int) *data) << 24;

    // check whether it concern a channel number or a frequency
    // this becomes difficult because the frequency is in kHz
    if (channel < 1000)
    {
        // interpret the data as a channel
        frq->m = channel;
        frq->e = 0;
    }
    else
    {
        // interpret the data as frequency
        frq->m = channel;
        frq->e = 3;
    }

    return 0;
}

int isl38xx_ioctl_setfreq(struct net_device *nwdev,
                                struct iw_freq *frq)
{
    isl38xx_private *private_config = nwdev->priv;
    char channel[4] = { 0, 0, 0, 0 }, *data;
    long mantisse;
    int exponent;

    // prepare the structure for the set object
    if (frq->m < 1000)
    {
        // structure value contains a channel indication
        channel[0] = (char) frq->m;
        channel[1] = (char) (frq->m >> 8);
    }
    else
    {
        // structure contains a frequency indication
        // mantisse is larger than 1000, so make it kHz
        mantisse = frq->m / 1000;
        exponent = frq->e;

        // process the exponent
        while (exponent > 0)
        {
            mantisse *= 10;
            exponent--;
        }

        // convert the frequency to the right byte order
        channel[0] = (char) mantisse;
        channel[1] = (char) (mantisse >> 8);
        channel[2] = (char) (mantisse >> 16);
        channel[3] = (char) (mantisse >> 24);
    }

    // send a set object to set the channel by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
            DOT11_OID_CHANNEL, 0, &channel[0], 4, 0);

    // enter sleep mode after supplying oid and local buffers for reception
    // to the management queue receive mechanism
    return wait_for_oid_response(private_config, DOT11_OID_CHANNEL,
        (void **) &data);
}

int isl38xx_ioctl_getrtsthresh(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *thresh = (long *) wrq->u.data.pointer;
    int rvalue;
    long *data;

    // send a get object to get the filter setting
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_GET,
        DOT11_OID_RTSTHRESH, 0, NULL, 4, 0);
    if (rvalue = wait_for_oid_response(private_config, DOT11_OID_RTSTHRESH,
        (void **) &data), rvalue != 0)
    {
        // an error occurred, return the value
        return rvalue;
    }

    *thresh = *data;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Get RTS Thresh 0x%08lx (%li)\n", *thresh, *thresh );
#endif

    return 0;
}

int isl38xx_ioctl_setrtsthresh(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    long *thresh = (long *) wrq->u.data.pointer;
    void *data;
	
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Set RTS Thresh: 0x%08lx\n", (long) *thresh );
#endif

    // send a set object to set the bsstype by placing it in the transmit queue
    isl38xx_queue_pimfor(private_config, PIMFOR_OP_SET,
        DOT11_OID_RTSTHRESH, 0, thresh, 4, 0);
    return wait_for_oid_response(private_config, DOT11_OID_RTSTHRESH, &data);
}

int isl38xx_ioctl_checktraps(struct net_device *nwdev,
    struct iwreq *wrq )
{
    isl38xx_private *private_config = nwdev->priv;
    char *frame = (char *) wrq->u.data.pointer;
    queue_entry *entry;
    int device_id, operation, flags, length;
    unsigned long oid;

    // check whether the trap queue is empty
    if( queue_size( private_config->remapped_device_base,
        &private_config->trap_rx_queue ) == 0 )
    {
        // no entries in the trap queue
#if VERBOSE > SHOW_ERROR_MESSAGES 
        DEBUG(SHOW_TRACING, "TRAP receive queue empty\n");
#endif
        return -ENOMSG;
    }

    // read a frame from the queue and return it
    get_queue( private_config->remapped_device_base,
        &private_config->trap_rx_queue, &entry );

    // decode the PIMFOR frame for data length
    pimfor_decode_header( (pimfor_header *) entry->host_address, &operation,
        &oid, &device_id, &flags, &length );

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG( SHOW_TRACING, "TRAP: oid 0x%lx, device %i, flags 0x%x length %i\n",
	    oid, device_id, flags, length );
#endif

    // copy the frame to the data buffer
    if (copy_to_user( frame, (void *) entry->host_address, PIMFOR_HEADER_SIZE+length ))
    {
        // free the entry to the freeq
        put_queue(private_config->remapped_device_base,
            &private_config->mgmt_rx_freeq, entry);

        return -EFAULT;
    }

    // free the entry to the freeq
    put_queue(private_config->remapped_device_base,
        &private_config->mgmt_rx_freeq, entry);

    return 0;
}

int isl38xx_ioctl(struct net_device *nwdev, struct ifreq *rq, int cmd )
{
    struct iwreq *wrq = (struct iwreq *) rq;
    isl38xx_private *private_config = nwdev->priv;
    int error = 0;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    void *device = private_config->remapped_device_base;

    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_ioctl cmd %x \n", cmd);

    DEBUG(SHOW_QUEUE_INDEXES, "Mgmt Tx Qs: free[%i] shdw[%i]\n",
        queue_size(device, &private_config->mgmt_tx_freeq),
        queue_size(device, &private_config->mgmt_tx_shadowq));
#endif

    switch (cmd)
    {
        case SIOCGIWNAME:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWNAME\n", nwdev->name);
#endif
    	    if( private_config->pci_dev_id == PCIDEVICE_ISL3877 )
                strncpy( wrq->u.name, "PRISM Indigo", IFNAMSIZ );
	    else if( private_config->pci_dev_id == PCIDEVICE_ISL3890 )
                strncpy( wrq->u.name, "PRISM Duette", IFNAMSIZ );
	    else 
                strncpy( wrq->u.name, "Unsupported card type", IFNAMSIZ );
            break;

        case SIOCGIWRATE:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWRATE\n", nwdev->name);
#endif
            error = isl38xx_ioctl_getrate(nwdev, &wrq->u.bitrate );
            break;

        case SIOCGIWRANGE:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWRANGE\n", nwdev->name);
#endif
            error = isl38xx_ioctl_getrange(nwdev,
                (struct iw_range *) wrq->u.data.pointer );
            wrq->u.data.length = sizeof(struct iw_range);
            break;

#if WIRELESS_EXT > 11
//      case SIOCGIWSTATS:
//          DEBUG(SHOW_TRACING, "%s: SIOCGIWSTATS\n", nwdev->name);
//          error = isl38xx_ioctl_getstats(nwdev,
//              (struct iw_statistics *) wrq->u.data.pointer );
//          break;
#endif

        case SIOCGIWESSID:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWESSID\n", nwdev->name);
#endif
            error = isl38xx_ioctl_getessid(nwdev, &wrq->u.essid);
            break;

        case SIOCSIWESSID:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCSIWESSID\n", nwdev->name);
#endif
            // check which operation is requied depending on the flag
            if (wrq->u.essid.flags == 0)
            {
            // disable essid checking (ESSID promiscuous)
                // is it supported by the card ?
            return -EOPNOTSUPP;
            }
            else
            {
                // set the essid to the value supplied
                error = isl38xx_ioctl_setessid(nwdev, &wrq->u.essid);
            }

            break;

        case SIOCGIWAP:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWAP\n", nwdev->name);
#endif
            wrq->u.ap_addr.sa_family = ARPHRD_ETHER;
            error = isl38xx_ioctl_getbssid(nwdev, &wrq->u.ap_addr.sa_data[0]);
            break;

        case SIOCSIWAP:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCSIWAP\n", nwdev->name);
#endif
            error = isl38xx_ioctl_setbssid(nwdev, &wrq->u.ap_addr.sa_data[0]);
            break;

        case SIOCSIWMODE:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCSIWMODE\n", nwdev->name);
#endif
            error = isl38xx_ioctl_setmode(nwdev, &wrq->u.mode);
            break;

        case SIOCGIWMODE:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWMODE\n", nwdev->name);
#endif
            error = isl38xx_ioctl_getmode(nwdev, &wrq->u.mode);
            break;

        case SIOCGIWFREQ:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWFREQ\n", nwdev->name);
#endif
            error = isl38xx_ioctl_getfreq(nwdev, &wrq->u.freq);
            break;

        case SIOCSIWFREQ:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCSIWFREQ\n", nwdev->name);
#endif
            error = isl38xx_ioctl_setfreq(nwdev, &wrq->u.freq);
            break;

        case SIOCGIWPRIV:
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCGIWPRIV\n", nwdev->name);
#endif
            if (wrq->u.data.pointer)
            {
                if( error = verify_area(VERIFY_WRITE, wrq->u.data.pointer,
                    sizeof(privtab)), error )
                {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                    DEBUG(SHOW_TRACING, "verify_area error %i\n", error );
#endif
                    break;
                }

                wrq->u.data.length = sizeof(privtab) / sizeof(privtab[0]);
                if (copy_to_user(wrq->u.data.pointer, privtab, sizeof(privtab)))
                {
#if VERBOSE > SHOW_ERROR_MESSAGES 
                    DEBUG(SHOW_TRACING, "copy_to_user error %i\n", error );
#endif
                    error = -EFAULT;
                }
            }
            break;

        case SIOCIWFIRSTPRIV + 0x00:                    // setauthenten
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+0 (setauthenten)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setauthenten(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x01:                    // getauthenten
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+1 (getauthenten)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getauthenten(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x02:                    // setunencrypted
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+2 (setunencrypt)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setprivfilter(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x03:                    // getunencrypted
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+3 (getunencrypt)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getprivfilter(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x04:                    // setprivinvoke
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+4 (setprivinvoke)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setprivinvoke(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x05:                    // getprivinvoke
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+5 (getprivinvoke)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getprivinvoke(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x06:                    // setdefkeyid
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+6 (setdefkeyid)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setdefkeyid(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x07:                    // getdefkeyid
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+7 (getdefkeyid)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getdefkeyid(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x08:                    // setdefkeyx
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+8 (setdefkeyx)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setdefkeyx(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x09:                    // getdefkeyx
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+9 (getdefkeyx)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getdefkeyx(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x0A:                    // getprivstat
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+10 (getprivstat)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getprivstat(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x0B:                    // getsuprates
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+11 (getsuprates)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getsuprates(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x0C:                    // setfixedrate
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+12 (setfixedrate)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setfixedrate(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x0D:                    // getfixedrate
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+13 (getfixedrate)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getfixedrate(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x0E:                    // setbeaconper
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+14 (setbeaconper)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setbeaconper(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x0F:                    // getauthenten
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+15 (getbeaconper)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getbeaconper(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x10:                    // wdslinkadd
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+16 (wdslinkadd)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_wdslinkadd(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x11:                    // wdslinkdel
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+17 (wdslinkdel)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_wdslinkdel(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x12:                    // eapauthen
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+18 (eapauthen)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_eapauthen(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x13:                    // eapunauth
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+19 (eapunauth)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_eapunauth(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x14:                    // setdot1xen
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+20 (setdot1xen)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setdot1xen(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x15:                    // getdot1xen
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+21 (getdot1xen)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getdot1xen(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x16:                    // setstakeyx
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+22 (setstakeyx)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setstakeyx(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x17:                    // getstakeyx
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+23 (getstakeyx)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getstakeyx(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x18:                    // setrtsthresh
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+24 (setrtsthresh)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_setrtsthresh(nwdev, wrq);
            break;

        case SIOCIWFIRSTPRIV + 0x19:                    // getrtsthresh
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+25 (getrtsthresh)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_getrtsthresh(nwdev, wrq);
            break;


        case SIOCIWFIRSTPRIV + 0x1F:                    // checktraps
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "%s: SIOCIWFIRSTPRIV+31 (checktraps)\n",
                nwdev->name);
#endif
            error = isl38xx_ioctl_checktraps(nwdev, wrq);
            break;

        default:
            error = -EOPNOTSUPP;
    }

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_QUEUE_INDEXES,
        "Mgmt Rx Qs: f[%i] s[%i] i[%i] t[%i] p[%i]\n",
        queue_size(device, &private_config->mgmt_rx_freeq),
        queue_size(device, &private_config->mgmt_rx_shadowq),
        queue_size(device, &private_config->ioctl_rx_queue),
        queue_size(device, &private_config->trap_rx_queue),
        queue_size(device, &private_config->pimfor_rx_queue));
#endif

    return error;
}







