/******************************************************************************

    File:               isl38xx_eth.c
    Author:             W.Termorshuizen
    Version:            0.3.0.3
    Created:            June 19th, 2002
    Updated:            November 11th, 2002
    Company:            Intersil Americas Inc.
    Description:        This file contains the linux driver for the Intersil
                        Wireless LAN adapters.
    History:	    	Maintained in file isl38xx.h
		
******************************************************************************/
#define __KERNEL_SYSCALLS__


#include <linux/version.h>
#ifdef MODULE
#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif


#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/wait.h>

#include <asm/processor.h>      // Processor type for cache alignment.
#include <asm/bitops.h>
#include <asm/io.h>

#include <linux/netdevice.h>
#include <linux/if_arp.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include <pcmcia/driver_ops.h>
#include <asm/uaccess.h>
#include <linux/unistd.h>
#include <linux/wireless.h>

#include "bloboidv2.h"      // additional types and defs for isl38xx fw
#include "isl38xx.h"


#define DEBUG( f, args... ) 	K_DEBUG( f, pc_debug, args )

/******************************************************************************
        Global variable definition section
******************************************************************************/
extern int pc_debug;
extern int init_wds;

/******************************************************************************
    Network Interface functions
******************************************************************************/
void isl38xx_cleanup_transmit( isl38xx_private *private_config,
    isl38xx_control_block *control_block )
{
	struct sk_buff *skb;
	u32	*devcur =
			&control_block->device_curr_frag[ISL38XX_CB_TX_DATA_LQ];
	u32 index;
	
	// compare the control block read pointer with the free pointer
	dcache_inval_range(devcur, sizeof (u32));

	while( private_config->free_data_tx != le32_to_cpu(*devcur) )
	{
		// read the index of the first fragment to be freed
		index = private_config->free_data_tx % ISL38XX_CB_TX_QSIZE;

		// check for holes in the arrays caused by multi fragment frames 
		// searching for the last fragment of a frame
    	if( private_config->pci_map_tx_address[index] != (u32) NULL )
		{
			// entry is the last fragment of a frame
			// free the skb structure and unmap pci memory
			skb = private_config->data_low_tx[index];
			pci_unmap_single( private_config->pci_device, (u32)	skb->data, 
				skb->len, PCI_DMA_TODEVICE );
       		dev_kfree_skb( skb );
		}
		
		// increment the free data low queue pointer
		private_config->free_data_tx++;
		dcache_inval_range(devcur, sizeof (u32));
	}
}

int isl38xx_transmit(struct sk_buff *skb, struct net_device *nwdev)
{
    isl38xx_private *private_config = nwdev->priv;
    isl38xx_control_block *control_block = private_config->control_block;
    isl38xx_fragment *fragment;
    u32 index, reg, counter, pci_map_address;
    int fragments;
    int frame_size;
    int offset;
    struct sk_buff *newskb;
    int newskb_offset;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_transmit \n");
#endif

    // check whether the device is running
//      if( !netif_running( nwdev ) )
//      {
//              DEBUG( SHOW_ERROR_MESSAGES, "%s: Tx on stopped device!\n",
//                     nwdev->name);
//              return 1;
//      }

//      tx_timeout_check( nwdev, isl38xx_tx_timeout );
//      skb_tx_check( nwdev, skb );

    // determine the amount of fragments needed to store the frame
    frame_size = skb->len < ETH_ZLEN ? ETH_ZLEN : skb->len;
    if( init_wds ) frame_size += 6;
    fragments = frame_size / MAX_FRAGMENT_SIZE;
    fragments += frame_size % MAX_FRAGMENT_SIZE == 0 ? 0 : 1;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_TRACING, "Fragments needed for frame %i\n", fragments);
#endif

    // check whether the destination queue has enough fragments for the frame
    if (ISL38XX_CB_TX_QSIZE - isl38xx_in_queue( control_block, 
		ISL38XX_CB_TX_DATA_LQ ) < fragments )
    {
        // Error, cannot add the frame because the queue is full
        DEBUG(SHOW_ERROR_MESSAGES, "Error: Queue [%i] not enough room\n",
	        ISL38XX_CB_TX_DATA_LQ);
        return -EBUSY;
    }

    // Check alignment and WDS frame formatting. The start of the packet should
    // be aligned on a 4-byte boundary. If WDS is enabled add another 6 bytes
    // and add WDS address information
    if( ((int) skb->data & 0x03) | init_wds )
    {
        // get the number of bytes to add and re-align
        offset = (int) skb->data & 0x03;
		offset += init_wds ? 6 : 0;		

		// check whether the current skb can be used 
        if (!skb_cloned(skb) && (skb_tailroom(skb) >= offset))
        {
            unsigned char *src = skb->data;

#if VERBOSE > SHOW_ERROR_MESSAGES 
	        DEBUG(SHOW_TRACING, "skb offset %i wds %i\n", offset, init_wds );
#endif

			// align the buffer on 4-byte boundary
            skb_reserve(skb, (int) skb->data & 0x03 );
            if( init_wds )
            {
            	// wds requires an additional address field of 6 bytes
				skb_put( skb, 6 );			
			    memmove( skb->data+6, src, skb->len );
                memcpy( skb->data,
                    private_config->wds_link_table[0].address, 6 );
            }
            else
			{
                memmove( skb->data, src, skb->len );
			}
			
#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "memmove %p %p %i \n", skb->data, src,
                skb->len );
#endif
        }
        else
        {
            newskb = dev_alloc_skb( init_wds ? skb->len+6 : skb->len );
            newskb_offset = (int) newskb->data & 0x03;

            // Check if newskb->data is aligned
            if( newskb_offset )
                skb_reserve(newskb, newskb_offset);

            skb_put(newskb, init_wds ? skb->len+6 : skb->len );
            if( init_wds )
            {
                memcpy( newskb->data+6, skb->data, skb->len );
                memcpy( newskb->data,
                    private_config->wds_link_table[0].address, 6 );
            }
            else
                memcpy( newskb->data, skb->data, skb->len );

#if VERBOSE > SHOW_ERROR_MESSAGES 
            DEBUG(SHOW_TRACING, "memcpy %p %p %i wds %i\n", newskb->data,
                skb->data, skb->len, init_wds );
#endif

            newskb->dev = skb->dev;
            dev_kfree_skb(skb);
            skb = newskb;
        }
    }

    // display the buffer contents for debugging
#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG( SHOW_BUFFER_CONTENTS, "\ntx %p ", skb->data );
    display_buffer((char *) skb->data, skb->len );
#endif

	// map the skb buffer to pci memory for DMA operation
	pci_map_address = (u32) pci_map_single( private_config->pci_device, 
		(void *) skb->data, skb->len, PCI_DMA_TODEVICE );
    if( pci_map_address == (u32) NULL )
	{
    	// error mapping the buffer to device accessable memory address
	    DEBUG(SHOW_ERROR_MESSAGES, "Error map DMA addr, data lost\n" );

	    // free the skbuf structure before aborting
    	dev_kfree_skb(skb);
	    return -EIO;
    }

	// place each fragment in the control block structure and store in the last
	// needed fragment entry of the pci_map_tx_address and data_low_tx arrays 
	// the skb frame information
    for (counter = 0; counter < fragments; counter++ )
	{
		// get a pointer to the target control block fragment
	    index = (counter + le32_to_cpu(
	        control_block->driver_curr_frag[ISL38XX_CB_TX_DATA_LQ])) %
    	    ISL38XX_CB_TX_QSIZE;
	    fragment = &control_block->tx_data_low[index];
	
		// check whether this frame fragment is the last one
		if( counter == fragments-1 )
		{
			// the fragment is the last one, add the streaming DMA mapping for 
			// proper PCI bus operation
    		private_config->pci_map_tx_address[index] = pci_map_address;
			
			// store the skb address for future freeing 
			private_config->data_low_tx[index] = skb;
		}
		else
		{
			// the fragment will be followed by more fragments
			// clear the pci_map_tx_address and data_low_tx entries
		    private_config->pci_map_tx_address[index] = (u32) NULL;
			private_config->data_low_tx[index] = NULL;
		}
			
		// set the proper fragment start address and size information
        fragment->address = cpu_to_le32( pci_map_address + 
			counter*MAX_FRAGMENT_SIZE );
        fragment->size = cpu_to_le16( frame_size > MAX_FRAGMENT_SIZE ?
            MAX_FRAGMENT_SIZE : frame_size );
        frame_size -= MAX_FRAGMENT_SIZE;
	}

	// ready loading all fragments to the control block and setup pci mapping
	// update the control block interface information
    add_le32p(&control_block->driver_curr_frag[ISL38XX_CB_TX_DATA_LQ],
        fragments);
    private_config->data_fragments_in_wait += fragments;

    // check whether the data tx queue is full now
    if( ISL38XX_CB_TX_QSIZE - isl38xx_in_queue( control_block, 
		ISL38XX_CB_TX_DATA_LQ ) < ISL38XX_MIN_QTHRESHOLD )
    {
        // stop the transmission of network frames to the driver
        netif_stop_queue(nwdev);

        // set the full flag for the data low transmission queue
        private_config->data_low_tx_full = 1;
    }

    // trigger the device by setting the Update bit in the Device Int register
    reg = readl(private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    reg |= ISL38XX_DEV_INT_UPDATE;
    writel(reg, private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    // set the transmission time
    nwdev->trans_start = jiffies;
    private_config->statistics.tx_packets++;
    private_config->statistics.tx_bytes += skb->len;

    // cleanup the data low transmit queue
	isl38xx_cleanup_transmit(private_config, private_config->control_block );
    return 0;
}

int isl38xx_receive(isl38xx_private * private_config)
{
    struct net_device *nw_device = private_config->my_module;
    isl38xx_control_block *control_block = private_config->control_block;
    struct sk_buff *skb;
    u16 size;
    u32 index, reg, offset;
    u32	*drvcur;
    unsigned char *src;

#if VERBOSE > SHOW_ERROR_MESSAGES
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_receive\n");
#endif

    // the device has written an Ethernet frame in the data area
    // of the sk_buff without updating the structure, do it now
	index = private_config->free_data_rx % ISL38XX_CB_RX_QSIZE;
    dcache_inval_range(&control_block->rx_data_low[index],
		sizeof(isl38xx_fragment));
    size = le16_to_cpu(control_block->rx_data_low[index].size);
	skb = private_config->data_low_rx[index];
    offset = ((u32) le32_to_cpu(control_block->rx_data_low[index].address) -
		(u32)skb->data) & 3;

#if VERBOSE > SHOW_ERROR_MESSAGES
    DEBUG( SHOW_TRACING, "frq->addr %x skb->data %p offset %u truesize %u\n",
        control_block->rx_data_low[index].address,
		skb->data, offset, skb->truesize);
#endif

    // delete the streaming DMA mapping before processing the skb
    pci_unmap_single( private_config->pci_device, (dma_addr_t) skb->data,
        skb->len, PCI_DMA_FROMDEVICE );

	// update the skb structure and align the buffer
    skb_put( skb, size );
    if( offset )
    {
		// shift the buffer allocation offset bytes to get the right frame
        skb_pull( skb, 2 );
		skb_put( skb, 2 );
    }
    
#if VERBOSE > SHOW_ERROR_MESSAGES
    // display the buffer contents for debugging
    DEBUG(SHOW_BUFFER_CONTENTS, "\nrx %p ", skb->data);
    display_buffer((char *) skb->data, skb->len);
#endif

    // check whether WDS is enabled and whether the data frame is a WDS frame
    if( init_wds )
    {
        // WDS enabled, check for the wds address on the first 6 bytes of the buffer
        if( address_compare( (unsigned char *) skb->data,
            (unsigned char *) private_config->wds_link_table[0].address))
        {
            // addresses match, remove the WDS address from the frame
            src = skb->data+6;
            memmove( skb->data, src, skb->len-6);
            skb_trim( skb, skb->len-6);
        }
    }

#if VERBOSE > SHOW_ERROR_MESSAGES
    DEBUG(SHOW_TRACING, "Fragment size %i in skb at %p\n", size, skb);
    DEBUG(SHOW_TRACING, "Skb data at %p, length %i\n", skb->data, skb->len);

    // display the buffer contents for debugging
    DEBUG(SHOW_BUFFER_CONTENTS, "\nrx %p ", skb->data);
    display_buffer((char *) skb->data, skb->len);
#endif

    // do some additional sk_buff and network layer parameters
    skb->dev = nw_device;
    skb->protocol = eth_type_trans(skb, nw_device);
    skb->ip_summed = CHECKSUM_NONE;
    private_config->statistics.rx_packets++;
    private_config->statistics.rx_bytes += size;

    // deliver the skb to the network layer
    netif_rx(skb);

    // increment the read index for the rx data low queue
    private_config->free_data_rx++;

    // add one or more sk_buff structures
    drvcur = &control_block->driver_curr_frag[ISL38XX_CB_RX_DATA_LQ];
//    dcache_inval_range(drvcur, sizeof(u32));
    while( index = le32_to_cpu( *drvcur ),
		index - private_config->free_data_rx < ISL38XX_CB_RX_QSIZE)
    {
        // allocate an sk_buff for received data frames storage
	    // include any required alignment operations
	    skb = dev_alloc_skb(MAX_FRAGMENT_SIZE+2);
        if (skb == NULL)
        {
            // error allocating an sk_buff structure elements
            DEBUG(SHOW_ERROR_MESSAGES, "Error allocating skb\n");
            break;
        }

		// store the new skb structure pointer
		index = index % ISL38XX_CB_RX_QSIZE;
		private_config->data_low_rx[index] = skb;

        // set the streaming DMA mapping for proper PCI bus operation
		private_config->pci_map_rx_address[index] = (u32) pci_map_single(
			private_config->pci_device, (void *) skb->data, skb->len, 
			PCI_DMA_FROMDEVICE );
        if( private_config->pci_map_rx_address[index] == (u32)NULL)
        {
            // error mapping the buffer to device accessable memory address
            DEBUG(SHOW_ERROR_MESSAGES, "Error mapping DMA address\n");

            // free the skbuf structure before aborting
            dev_kfree_skb((struct sk_buff *) skb);
            break;
        }

        // update the fragment address
        control_block->rx_data_low[index].address = cpu_to_le32(
		 	private_config->pci_map_rx_address[index]);

        // increment the driver read pointer
        add_le32p(&control_block->driver_curr_frag[ISL38XX_CB_RX_DATA_LQ], 1);
    }

    // trigger the device by setting the Update bit in the Device Int register
    reg = readl(private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    reg |= ISL38XX_DEV_INT_UPDATE;
    writel(reg, private_config->remapped_device_base + ISL38XX_DEV_INT_REG);
    udelay(ISL38XX_WRITEIO_DELAY);

    return 0;
}

void isl38xx_tx_timeout(struct net_device *nwdev)
{
    isl38xx_private *private_config = nwdev->priv;
    struct net_device_stats *statistics = &private_config->statistics;
    int error = 0;

#if VERBOSE > SHOW_ERROR_MESSAGES 
    DEBUG(SHOW_FUNCTION_CALLS, "isl38xx_tx_timeout\n");
#endif

    // increment the transmit error counter
    statistics->tx_errors++;

    // reset the device and restore its configuration
    if (error = isl38xx_reset(private_config), error != 0) {
        DEBUG(SHOW_ERROR_MESSAGES, "Error %d on TX timeout card reset!\n",
            error);
    } else {
        DEBUG(SHOW_ERROR_MESSAGES, "Driver stopped TX/RX ......\n");

//        nwdev->trans_start = jiffies;
//        netif_wake_queue( nwdev );
    }

    return;
}
