/******************************************************************************

    File:               isl38xx.h
    Author:             W.Termorshuizen
    Version:            0.3.0.5
    Created:            June 19th, 2002
    Updated:            November 20th, 2002
    Company:            Intersil Americas Inc.
    Description:        This file contains the linux driver for the Intersil
                        Wireless LAN adapter based on the 38xx chips.
    History:
    version 0.0.1.0:    June 19th, 2002
    -   Simple PCI driver for firmware upload and polling for device
        initialization

    version 0.0.2.0:    June 23rd, 2002
    -   Added interrupt handler for device initialization detection and other
        interrupt source detection

    version 0.0.3.0:    July 8th, 2002
    -   Added device initialization for kernel
    -   Added PIMFOR frame encoding/decoding and transmission/reception
    -   Added IOCTL interface & support for Wireless Extensions:
        * SIOCGIWNAME   get name
        * SIOCGIWESSID  get ESSID               ->      DOT11_OID_SSID
        * SIOCSIWESSID  set ESSID               ->      DOT11_OID_SSID
        * SIOCGIWAP     get AP MAC address      ->      DOT11_OID_BSSID
        * SIOCSIWAP     set AP MAC addres       ->      DOT11_OID_BSSID
        * SIOCSIWMODE   set Mode                ->      DOT11_OID_BSSTYPE
        * SIOCGIWMODE   get Mode                ->      DOT11_OID_BSSTYPE
        mapped modes:   Ad-hoc on IBSS, client mode
                        Managed on Infra, client mode
                        Master on Infra, AP mode
    -   Added more compile time configuration for debug messages
    -   Added OID response processing of Linkstate, MAC address & Mode

    version 0.0.4.0:    July 11th, 2002
    -   Changed IOCTL wait delay to linux wait queue mechanism
    -   Changed the driver internal frame handling to queueing mechanism

    version 0.0.5.0:    July 18th, 2002
    -   Removed queue entry memory allocation for frame data/skbuf storage
    -   Added support for skbs from network interface in queueing mechanism
    -   Added IOCTL interface & support for Wireless Extensions:
        * SIOCGIWFREQ   get frequency/channel -> DOT11_OID_CHANNEL
        * SIOCSIWFREQ   set frequency/channel -> DOT11_OID_CHANNEL

    version 0.0.6.0:    July 24th, 2002
    -   (mthuys) Fixed alignment of TX and RX frames. Now basic
        frame TX and RX works.
    -   added endianness conversion between host and card.

    version 0.0.7.0:    August 6th, 2002
    -   Added additional debug string commands

    version 0.0.7.1:    August 7th, 2002
    -   Added module parameter pc_debug for commandline control of debug
        messages

    version 0.0.7.2:    August 12th, 2002
    -   Added PCI DMA support check before memory allocation

    version 0.0.7.3:    August 13th, 2002
    -   Changed condition for freeing skb frame in isl38XX_transmit function
    -   Added additional debug statements for testing skb frame alignment code

    version 0.0.7.4:    August 13th, 2002
    -   Changed the virtual to bus conversion for skb buffers to streaming DMA
        buffer mapping

    version 0.0.8.0:    August 14th, 2002
    -   Added card initialization enabling AP mode, channel setting & SSID
    -   Updated display buffer function controlling buffer display properly

    version 0.0.8.1:    August 15th, 2002
    -   Changed the pimfor transmit mechanism to buffer more pimfor frames than
        the transmit queueu size (=4) and process them one by one through the
        transmit queue host interface

    version 0.0.8.2:    August 16th, 2002
    -   Changed the places where pimfor transmit function is called due to
        deadlock situation with lower speed hosts
    -   Added a check on the management receive queue in the pimfor transmit
        function blocking the transmission of a management frame when a
        response is still pending

    version 0.0.8.3:    August 16th, 2002
    -   Added Configuration Object for Manual Run configuration procedure
    -   Fixed little timing gab in management frame response handling

    version 0.0.8.4:    August 20th, 2002
    -   Fixed the essid object length bug by reducing the length

    version 0.0.8.5:    August 29th, 2002
    -   Added parameters for ssid, channel, mode & bsstype

    version 0.0.9.0:    September 4th, 2002
    -   Added IOCTL interface & support for Wireless Extensions:
        * SIOCGIWRANGE  get range ->    DOT11_OID_RATES
        * SIOCSIWSTATS  get stats ->    DOT11_OID_MPDUTXFAILED

    version 0.0.9.1:    September 9th, 2002
    -   Added Private ioctl interface support

    version 0.0.9.2:    September 11th, 2002
    -   Added prvate ioctls for objects:
        * aExUnencrypted, set & get,    DOT11_OID_EXUNENCRYPTED
        * aPrivacyInvoked, set & get,   DOT11_OID_PRIVACYINVOKED
        * aDefKeyId, set & get,         DOT11_OID_DEFKEYID
        * aDefKey1-4, set & get,        DOT11_OID_DEFKEY1 - DOT11_OID_DEFKEY4

    version 0.0.9.3:    September 12th, 2002
    -   Added default keys 1-4 to the default setup
    -   Increesed the management transmit freeq with 8 entries
    -   Added WEP enable/disable option in default setup
    -   Added module parameter for initial WEP mode setting

    version 0.0.9.4:    September 18th, 2002
    -   Added ExUnencrypted to the default setup
    -   Added module parameter for initial ExUnencrypted filter setting
    -   Added private ioctls for objects:
        * aPrivTxRejected, get,         DOT11_OID_PRIVTXREJECTED
        * aPrivRxPlain, get,            DOT11_OID_PRIVRXPLAIN
        * aPrivRxFailed, get,           DOT11_OID_PRIVRXFAILED
        * aPrivNoKey, get,              DOT11_OID_PRIVRXNOKEY
        * aSupportedRates, get          DOT11_OID_SUPPORTEDRATES
    -   Added SIOCGIWRATE get rate ->   GEN_OID_LINKSTATE

    version 0.0.9.5:    September 20th, 2002
    -   Added default setting the PSM buffer through its object:
        * aPSMBuffer, set               DOT11_OID_PSMBUFFER
    -   Added private ioctls for objects:
        aAuthEnable, set & get,         DOT11_OID_AUTHENABLE
        aBeaconPeriod, set & get,       DOT11_OID_BEACONPERIOD

    version 0.0.9.6:    September 26th, 2002
    -   Added private ioctls for objects:
        * aWDSLinkAdd, set,              DOT11_OID_WDSLINKADD
        * aWDSLinkRemove, set,           DOT11_OID_WDSLINKREMOVE
        * aEAPAuthSTA, set,              DOT11_OID_EAPAUTHSTA
        * aEAPUnauthSTA, set,            DOT11_OID_EAPUNAUTHSTA
        * aDot1xEnable, set & get,       DOT11_OID_DOT1XENABLE
    -   Added private ioctl for reading TRAP frames from the TRAP rx queue.

    version 0.1.0.0:    October 8th, 2002
    -   Added WDS host module parameter & device initialization
    -   Added WDS Link array for storing enabled WDS Links
    -   Added Update bit set operations in frame receive functions

    version 0.1.0.1:	October 16th,2002
    -	Added authenticate module parameter 
    -	Added authenticate object in initial configuration sequence
    -	Added Dot1xEnable in configuration sequence and as module parameter
    -	Added private ioctl for object:
		* aSTAKey, set & get,			DOT11_OID_STAKEY

    version 0.2.0.0:	October 17th, 2002
    -	Changed the driver to support the new Wirelesss Extensions API which
	requires the 2.4.10 kernel to be patched. The new API supplies the  
	driver with event facilities to signal user space applications.
    - 	Added SUPPORT_WE_EVENTS constant definitions for Wireless Extensions
	Event signalling through a netlink interface.

    version 0.3.0.0:	October 31st, 2002
    -	Split the single source file isl38XX_cb.c, version 0.2.0.0, in separate
    	header and source files for multiple environment support.
    -	Changed filename template to isl38xx for multiple 38xx intersil chip 
    	support with one driver.
    -	Added a driver configuration section in the header file

    version 0.3.0.1:	November 5th, 2002
    -	Fixed skb buffer allignment bug when using WDS

	version 0.3.0.2:	November 11th, 2002
	-	Added compile out definition for all data path debugging statements
	-	Changed the debugging constant definitions for this purpose
	-	Optimization of the data path transmit and receive functions

	version 0.3.0.3:	November 11th, 2002
	- 	Moved the transmit cleanup functional call from interrupt handler to
		transmit function

	version 0.3.0.4:	November 12th, 2002
    -	Added private ioctl for object:
		* aRTSThresh, set & get,		DOT11_OID_RTSTHRESH
	
	version 0.3.0.5:	November 20th, 2002
	-	Added bit reset instruction ClockRun bit in Control Status Register to
		solve PCI communication problem for Duette cards.
			
******************************************************************************/

#ifndef _ISL38XX_H
#define _ISL38XX_H

/******************************************************************************

    Driver configuration section for driver compilation settings.
  
    The driver includes support for the following environments:
    -	Wireless Extensions (including Wireless Events)
    -	Hotplug kernel support
    -	PCMCIA kernel support
    -	Embedded (raw PCI) kernel support
  
    Another selection involves the management signalling method used. When
    signalling is used the kernel needs to be patched accordingly. There are
    two signalling mechanisms:
    -	Wireless Events support
    -	Intersil Events support

******************************************************************************/
 

/******************************************************************************

    Select the target system configuration where the driver will be used by
    setting the following target definition. The definition is used in this 
    header and the other source files to outcomment/select the relevant code.

******************************************************************************/

#undef	CONFIG_PCMCIA	// Stupidity for now. What moron wrote this anyway?
#define CONFIG_PCMCIA		0
#define CONFIG_HOTPLUG		1	// not tested, for the near future !
#define CONFIG_RAWPCI		2

#define TARGET_SYSTEM		CONFIG_RAWPCI

/******************************************************************************

    Select the management signalling mechanism to be used.

******************************************************************************/

#define NO_EVENTS		0
#define WE_EVENTS		1
#define ISL_EVENTS		2

#define EVENT_SUPPORT		ISL_EVENTS


// General driver definitions
#define PCIVENDOR_INTERSIL                      0x1260UL
#define PCIDEVICE_ISL3877                       0x3877UL
#define PCIDEVICE_ISL3890                       0x3890UL
#define DRIVER_NAME				"isl38xx"

//#if TARGET_SYSTEM == CONFIG_RAWPCI
//#define ISL3877_IMAGE_FILE                      "/tmp/uAP/isl3877.arm"
//#define ISL3890_IMAGE_FILE                      "/tmp/uAP/isl3890.arm"
//#else
#define ISL3877_IMAGE_FILE                      "/etc/pcmcia/isl3877.arm"
#define ISL3890_IMAGE_FILE  	    	    	"/etc/pcmcia/isl3890.arm"
//#endif

// Debugging verbose definitions
#define SHOW_NOTHING                            0x00    // overrules everything
#define SHOW_ANYTHING                           0xFF
#define SHOW_ERROR_MESSAGES                     0x01
#define SHOW_TRAPS                              0x02
#define SHOW_PIMFOR_FRAMES                      0x04
#define SHOW_QUEUE_INDEXES                      0x08
#define SHOW_FUNCTION_CALLS                     0x10
#define SHOW_BUFFER_CONTENTS                    0x20
#define SHOW_TRACING                            0x40
#define VERBOSE					SHOW_ERROR_MESSAGES

// Default card definitions
#define CARD_DEFAULT_SSID                       "uAPtest"
#define CARD_DEFAULT_CHANNEL                    1
#define CARD_DEFAULT_MODE                       INL_MODE_AP
#define CARD_DEFAULT_BSSTYPE                    DOT11_BSSTYPE_INFRA
#define CARD_DEFAULT_KEY1                       "default_key_1"
#define CARD_DEFAULT_KEY2                       "default_key_2"
#define CARD_DEFAULT_KEY3                       "default_key_3"
#define CARD_DEFAULT_KEY4                       "default_key_4"
#define CARD_DEFAULT_WEP                        0
#define CARD_DEFAULT_FILTER                     0
#define CARD_DEFAULT_WDS                        0
#define	CARD_DEFAULT_AUTHEN			DOT11_AUTH_OS
#define	CARD_DEFAULT_DOT1X			0

// ISL38XX Access Point Specific definitions
#define ISL38XX_MAX_WDS_LINKS                   8

// ISL38XX Host Interface Definitions
#define ISL38XX_PCI_MEM_SIZE                    0x02000
#define ISL38XX_MEMORY_WINDOW_SIZE              0x01000
#define ISL38XX_DEV_FIRMWARE_ADDRES             0x20000
#define ISL38XX_WRITEIO_DELAY                   10              // in us
#define ISL38XX_RESET_DELAY                     50              // in ms
#define ISL38XX_WAIT_CYCLE                      100             // in ms
#define ISL38XX_TX_TIMEOUT                      (2*HZ)

// PCI Memory Area
#define ISL38XX_HARDWARE_REG                    0x0000
#define ISL38XX_CARDBUS_CIS                     0x0800
#define ISL38XX_DIRECT_MEM_WIN                  0x1000

// Hardware registers
#define ISL38XX_DEV_INT_REG                     0x0000
#define ISL38XX_INT_IDENT_REG                   0x0010
#define ISL38XX_INT_ACK_REG                     0x0014
#define ISL38XX_INT_EN_REG                      0x0018
#define ISL38XX_CTRL_BLK_BASE_REG               0x0020
#define ISL38XX_DIR_MEM_BASE_REG                0x0030
#define ISL38XX_CTRL_STAT_REG                   0x0078

// Device Interrupt register bits
#define ISL38XX_DEV_INT_RESET                   0x0001
#define ISL38XX_DEV_INT_UPDATE                  0x0002
#define ISL38XX_DEV_INT_WAKEUP                  0x0008
#define ISL38XX_DEV_INT_SLEEP                   0x0010

// Interrupt Identification/Acknowledge/Enable register bits
#define ISL38XX_INT_IDENT_UPDATE                0x0002
#define ISL38XX_INT_IDENT_INIT                  0x0004
#define ISL38XX_INT_IDENT_WAKEUP                0x0008
#define ISL38XX_INT_IDENT_SLEEP                 0x0010
#define ISL38XX_INT_SOURCES                     0x001E

// Control/Status register bits
#define	ISL38XX_CTRL_STAT_CLKRUN		0x00800000
#define ISL38XX_CTRL_STAT_RESET                 0x10000000
#define ISL38XX_CTRL_STAT_RAMBOOT               0x20000000
#define ISL38XX_CTRL_STAT_STARTHALTED           0x40000000
#define ISL38XX_CTRL_STAT_HOST_OVERRIDE         0x80000000

// Control Block definitions
#define ISL38XX_CB_RX_DATA_LQ                   0
#define ISL38XX_CB_TX_DATA_LQ                   1
#define ISL38XX_CB_RX_DATA_HQ                   2
#define ISL38XX_CB_TX_DATA_HQ                   3
#define ISL38XX_CB_RX_MGMTQ                     4
#define ISL38XX_CB_TX_MGMTQ                     5
#define ISL38XX_CB_QCOUNT                       6
#define ISL38XX_CB_RX_QSIZE                     8
#define ISL38XX_CB_TX_QSIZE                     32
#define ISL38XX_CB_MGMT_QSIZE                   4
#define ISL38XX_MIN_QTHRESHOLD                  4       // fragments

// Memory Manager definitions
#define MGMT_FRAME_SIZE                         256     // bytes
#define DATA_FRAME_SIZE                         32      // > queue_entry size
#define MGMT_TX_FRAME_COUNT                     24      // max 4 + spare 4 + 8 init
#define MGMT_RX_FRAME_COUNT                     24      // 4*4 + spare 8
#define DATA_TX_FRAME_COUNT                     96      // 2*32 + 2*16 spare
#define DATA_RX_FRAME_COUNT                     24      // 4*4 + 2*4 spare
#define MGMT_FRAME_COUNT                        (MGMT_TX_FRAME_COUNT + MGMT_RX_FRAME_COUNT)
#define DATA_FRAME_COUNT                        (DATA_TX_FRAME_COUNT + DATA_RX_FRAME_COUNT)
#define MGMT_QBLOCK                             MGMT_FRAME_COUNT * MGMT_FRAME_SIZE
#define DATA_QBLOCK                             DATA_FRAME_COUNT * DATA_FRAME_SIZE
#define CONTROL_BLOCK_SIZE                      1024    // should be enough
#define PSM_FRAME_SIZE                          1536
#define PSM_FRAME_COUNT                         100
#define PSM_BUFFER_SIZE                         PSM_FRAME_SIZE * PSM_FRAME_COUNT
#define MAX_TRAP_RX_QUEUE                       4

// calculate the host memory block based on the number of entries required for
// the queueing mechanism including enough room for the isl38XX_control_block
// structure
#define ISL38XX_HOST_MEM_BLOCK  MGMT_QBLOCK + DATA_QBLOCK + CONTROL_BLOCK_SIZE \
                                + PSM_BUFFER_SIZE

// Fragment package definitions
#define FRAGMENT_FLAG_MF                        0x0001
#define MAX_FRAGMENT_SIZE                       1536

// PIMFOR package definitions
#define PIMFOR_ETHERTYPE                        0x8828
#define PIMFOR_HEADER_SIZE                      12
#define PIMFOR_VERSION                          1
#define PIMFOR_OP_GET                           0
#define PIMFOR_OP_SET                           1
#define PIMFOR_OP_RESPONSE                      2
#define PIMFOR_OP_ERROR                         3
#define PIMFOR_OP_TRAP                          4
#define PIMFOR_OP_RESERVED                      5       // till 255
#define PIMFOR_DEV_ID_MHLI_MIB                  0
#define PIMFOR_FLAG_APPLIC_ORIGIN               0x01
#define PIMFOR_FLAG_LITTLE_ENDIAN               0x02

// Driver specific oid definitions (NDIS driver)
#define OID_INL_TUNNEL                          0xFF020000
#define OID_INL_MEMADDR                         0xFF020001
#define OID_INL_MEMORY                          0xFF020002
#define OID_INL_MODE                            0xFF020003
#define OID_INL_COMPONENT_NR                    0xFF020004
#define OID_INL_VERSION                         0xFF020005
#define OID_INL_INTERFACE_ID                    0xFF020006
#define OID_INL_COMPONENT_ID                    0xFF020007
#define OID_INL_CONFIG                          0xFF020008

// Mode object definitions (OID_INL_MODE)
#define INL_MODE_NONE                           -1
#define INL_MODE_PROMISCUOUS                    0
#define INL_MODE_CLIENT                         1
#define INL_MODE_AP                             2
#define INL_MODE_SNIFFER                        3

// Config object definitions (OID_INL_CONFIG)
#define INL_CONFIG_MANUALRUN                    0x01
#define INL_CONFIG_FRAMETRAP                    0x02
#define INL_CONFIG_RXANNEX                      0x04
#define INL_CONFIG_TXANNEX                      0x08
#define INL_CONFIG_WDS                          0x10

/*
 *  Type definition section
 *
 *  The PIMFOR package is defined using all unsigned bytes, because this makes
 *  it easier to handle 32 bits words on the two different systems the structure
 *  defines only the header allowing copyless frame handling
 */
typedef struct
{
    u8 version;
    u8 operation;
    u8 oid[4];                          // 32 bits word
    u8 device_id;
    u8 flags;
    u8 length[4];                       // 32 bits word
} pimfor_header;

typedef struct
{
    u32 address;                        // physical address on host
    u16 size;                           // packet size
    u16 flags;                          // set of bit-wise flags
} isl38xx_fragment;

typedef struct
{
    u32 driver_curr_frag[ISL38XX_CB_QCOUNT];
    u32 device_curr_frag[ISL38XX_CB_QCOUNT];
    isl38xx_fragment rx_data_low[ISL38XX_CB_RX_QSIZE];
    isl38xx_fragment tx_data_low[ISL38XX_CB_TX_QSIZE];
    isl38xx_fragment rx_data_high[ISL38XX_CB_RX_QSIZE];
    isl38xx_fragment tx_data_high[ISL38XX_CB_TX_QSIZE];
    isl38xx_fragment rx_data_mgmt[ISL38XX_CB_MGMT_QSIZE];
    isl38xx_fragment tx_data_mgmt[ISL38XX_CB_MGMT_QSIZE];
} isl38xx_control_block;

typedef struct
{
    u32 host_address;                   // mem address for the host
    u32 dev_address;                    // mem address for the device
    u16 size;                           // size of memory block or queue
    u16 fragments;                      // nr of fragments in data part
    void *next;                         // entry pointer to next in list
    void *last;                         // for quick queue handling
} queue_entry, queue_root;

typedef struct
{
        u16 valid;                      // valid address or entry in use
        char address[6];                // MAC address
} address_entry;

typedef struct
{
    // PCI bus allocation & configuration members
    u8 pci_bus;                         // PCI bus id location
    u8 pci_dev_fn;                      // PCI bus func location
    u8 pci_irq;                         // PCI irq line
    u16 pci_dev_id;	    	    	// PCI device id

    struct pci_dev *pci_device;         // PCI structure information
    void *remapped_device_base;         // remapped device base address
    dma_addr_t device_host_address;     // host memory address for dev
    dma_addr_t device_psm_buffer;   	// host memory for PSM buffering
    void *driver_mem_address;           // memory address for the driver
    void *device_id;                    // unique ID for shared IRQ

    // kernel configuration members
//    struct net_device *next_module;
    struct net_device *my_module;       // makes life easier....

    // device queue interface members
    isl38xx_control_block *control_block;   // device control block
    u32 index_mgmt_rx;                  // real index mgmt rx queue
    u32 index_mgmt_tx;              	// read index mgmt tx queue
    u32 free_data_rx;                 	// free pointer data rx queue
    u32 free_data_tx;                	// free pointer data tx queue
    u32 data_low_tx_full;               // full detected flag

    // root free queue definitions
    queue_root mgmt_tx_freeq;           // queue entry container mgmt tx
    queue_root mgmt_rx_freeq;           // queue entry container mgmt rx

    // driver queue definitions
    queue_root mgmt_tx_shadowq;         // shadow queue mgmt tx
    queue_root mgmt_rx_shadowq;         // shadow queue mgmt rx
    queue_root ioctl_rx_queue;          // ioctl rx queue
    queue_root trap_rx_queue;           // TRAP pimfor rx queue
    queue_root pimfor_rx_queue;         // PIMFOR generated rx queue
	struct sk_buff *data_low_tx[ISL38XX_CB_TX_QSIZE];
	struct sk_buff *data_low_rx[ISL38XX_CB_RX_QSIZE];
	u32 pci_map_tx_address[ISL38XX_CB_TX_QSIZE];
	u32 pci_map_rx_address[ISL38XX_CB_RX_QSIZE];
	
    // driver network interface members
    struct net_device_stats statistics;

    // driver operational members
    int mode;                           // device operation mode
    int linkstate;                      // device link state
    int ioctl_queue_lock;               // flag for locking queue read
    int processes_in_wait;              // process wait counter
    int data_fragments_in_wait;         // fragments ready for device
    int wds_link_count;             	// number of wds link entries in table
    address_entry wds_link_table[ISL38XX_MAX_WDS_LINKS]; // WDS Link table
} isl38xx_private;


/*
 *  Function definitions
 */

//#if TARGET_SYSTEM == RAWPCI
//#define K_DEBUG( f, m, args... ) if(( f & m ) != 0 ) printk( KERN_DEBUG args )
//#else
#define K_DEBUG( f, m, args... ) if(( f & m ) != 0 ) printk( KERN_INFO args )
//#endif

static inline void add_le32p(u32 *le_number, u32 add)
{
    *le_number = cpu_to_le32(le32_to_cpup(le_number) + add);
}

static inline u32 isl38xx_in_queue(isl38xx_control_block * control_block,
                             int queue)
{
	u32	*drvcur = &control_block->driver_curr_frag[queue];
	u32	*devcur = &control_block->device_curr_frag[queue];

    // determine the amount of fragments in the queue depending on the type
    // of the queue, either transmit or receive
//    dcache_inval_range(drvcur, sizeof (u32));
    dcache_inval_range(devcur, sizeof(u32));
    if ((queue == ISL38XX_CB_TX_DATA_LQ)
        || (queue == ISL38XX_CB_TX_DATA_HQ)
        || (queue == ISL38XX_CB_TX_MGMTQ))
    {
        return (u32)((int) le32_to_cpu(*drvcur)-
                (int) le32_to_cpu(*devcur));
    }
    else if (queue == ISL38XX_CB_RX_MGMTQ)
    {
        return (u32) (ISL38XX_CB_MGMT_QSIZE -
        ((int) le32_to_cpu(*drvcur) - (int) le32_to_cpu(*devcur)));
    }
    else
    {
        return (u32) (ISL38XX_CB_RX_QSIZE -
      ((int) le32_to_cpu(*drvcur) - (int) le32_to_cpu(*devcur)));
    }
}


/*
 *  Function prototypes
 */
 
void display_buffer( char *, int );
void schedule_wait( int );
static inline volatile void	enable_update_interrupt(void *device)
{
	u32	reg;

	/* set the update queue status bit in the Interrupt Enable Register */
	reg = readl(device + ISL38XX_INT_EN_REG);
	reg |= ISL38XX_INT_IDENT_UPDATE;
	writel(reg, device + ISL38XX_INT_EN_REG);
}

static inline volatile void	disable_update_interrupt(void *device)
{
	u32	reg;

	/* clear the update queue status bit in the Interrupt Enable Register */
	reg = readl(device + ISL38XX_INT_EN_REG);
	reg &= ~ISL38XX_INT_IDENT_UPDATE;
	writel(reg, device + ISL38XX_INT_EN_REG);
}

void print_frame( struct sk_buff * );
void string_to_macaddress( unsigned char *, unsigned char * );
int address_compare( unsigned char *, unsigned char * );

void pimfor_encode_header( int, unsigned long, int, int, int, pimfor_header * );
void pimfor_decode_header( pimfor_header *, int *, unsigned long *, int *,
    	    	    	    	int *, int * );

void init_queue( queue_root * );
int put_queue( void *, queue_root *, queue_entry * );
int get_queue( void *, queue_root *, queue_entry ** );
//int queue_size( void *, queue_root * );

static inline int	queue_size(void *dev, queue_root *queue)
{
	return (int)(queue)->size;
}

int isl38xx_transmit_pimfor( isl38xx_private * );
int isl38xx_queue_pimfor( isl38xx_private *, int, unsigned long, int,
    	    	    	    	void *, int, int );
int isl38xx_receive_pimfor( isl38xx_private * );
int wait_for_oid_response( isl38xx_private *, long, void ** );

void isl38xx_interrupt( int, void *, struct pt_regs * );

void isl38xx_cleanup_transmit( isl38xx_private *, isl38xx_control_block * );
int isl38xx_transmit( struct sk_buff *, struct net_device * );
int isl38xx_receive( isl38xx_private * );
void isl38xx_tx_timeout( struct net_device * );

void isl38xx_initialize( isl38xx_private * );

int isl38xx_ioctl_getrange( struct net_device *, struct iw_range * );
int isl38xx_ioctl_setrange( struct net_device *, struct iw_range * );

#if WIRELESS_EXT > 11
int isl38xx_ioctl_getstats( struct net_device *, struct iw_statistics *);
int isl38xx_ioctl_setstats( struct net_device *, struct iw_statistics *);

#endif

int isl38xx_ioctl_getauthenten( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setauthenten( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getprivfilter( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setprivfilter( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getprivinvoke( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setprivinvoke( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getdefkeyid( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setdefkeyid( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getdefkeyx( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setdefkeyx( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getprivstat( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getsuprates( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getfixedrate( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setfixedrate( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getbeaconper( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setbeaconper( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getrate( struct net_device *, struct iw_param * );
int isl38xx_ioctl_wdslinkadd( struct net_device *, struct iwreq * );
int isl38xx_ioctl_wdslinkdel( struct net_device *, struct iwreq * );
int isl38xx_ioctl_eapauthen( struct net_device *, struct iwreq * );
int isl38xx_ioctl_eapunauth( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getdot1xen( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setdot1xen( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getstakeyx( struct net_device *, struct iwreq * );
int isl38xx_ioctl_setstakeyx( struct net_device *, struct iwreq * );
int isl38xx_ioctl_getessid( struct net_device *, struct iw_point * );
int isl38xx_ioctl_setessid( struct net_device *, struct iw_point * );
int isl38xx_ioctl_getbssid( struct net_device *, char * );
int isl38xx_ioctl_setbssid( struct net_device *, char * );
int isl38xx_ioctl_getmode( struct net_device *, int * );
int isl38xx_ioctl_setmode( struct net_device *, int * );
int isl38xx_ioctl_getfreq( struct net_device *, struct iw_freq * );
int isl38xx_ioctl_setfreq( struct net_device *, struct iw_freq * );
int isl38xx_ioctl_checktraps( struct net_device *, struct iwreq * );
int isl38xx_ioctl( struct net_device *, struct ifreq *, int );

int isl38xx_open( struct net_device * );
int isl38xx_close( struct net_device * );
void isl38xx_set_multicast_list( struct net_device * );
struct net_device_stats *isl38xx_statistics( struct net_device * );

int isl38xx_reset( isl38xx_private * );
void isl38xx_interface_reset( isl38xx_private * );


#if TARGET_SYSTEM == CONFIG_PCMCIA

struct net_device *isl38xx_probe( struct net_device *, long, int );
int isl38xx_config( struct pci_dev *, struct net_device * );

dev_node_t *isl38xx_attach( dev_locator_t * );
void isl38xx_suspend( dev_node_t * );
void isl38xx_resume( dev_node_t * );
void isl38xx_detach( dev_node_t * );
#endif

//static int init_module( void );
//static void cleanup_module( void );

#endif	// _ISL38XX_H
