/*  $Header: /proj/software/pub/CVSROOT/uClinux/linux/drivers/net/wireless/intersil/islmvc_mgt.c,v 1.2 2003/03/21 20:12:29 mrustad Exp $
 *  
 *  Copyright (C) 2002 Intersil Americas Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/* Begin of include files */
#include <linux/skbuff.h>
#include <linux/rtnetlink.h>
#include <linux/net.h>
#include <linux/netdevice.h>
#include <net/sock.h>
#include <linux/netlink.h>

#include <asm/arch/blobcalls.h>
#include <linux_mvc_oids.h>
#include "islmvc_mgt.h"
#include "islmvc.h"
#include "isl_wds.h"
#include "isl_mgt.h"
/* End of include files */

struct wds_link_msg {
    unsigned char mac[6];
    char name[32];
} __attribute__ ((packed));

/* pointer to the netlink socket */
struct sock *nl_sock = NULL;

/* Static variable for the trap sequence counter */
__u32 trap_seq = 0; 

extern unsigned int src_seq = 0;

struct frame_attachment *frame_attach[256][MAX_ASSOC_IDS];


/**************************************************************** 
 *								
 *      		Handlers
 *								
 ***************************************************************/

static int islmvc_send_mvc_msg_hndl( unsigned int dev_type, unsigned int dev_seq, 
                                    unsigned int src_id,   unsigned int src_seq,
                                    unsigned int prismop,  unsigned int oid, void *data, unsigned long len);
static int islmvc_wdslink_add_hndl( unsigned int, unsigned int, unsigned int, unsigned,
                             unsigned int, unsigned int, void *, unsigned long );
static int islmvc_wdslink_del_hndl( unsigned int, unsigned int, unsigned int, unsigned,
                             unsigned int, unsigned int, void *, unsigned long );
static int islmvc_dot11_oid_attachment_hndl( unsigned int src_id,   unsigned int src_seq, 
                             unsigned int dev_id,   unsigned int dev_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len);
static int islmvc_dot11_oid_authenticate_hndl( unsigned int src_id,   unsigned int src_seq, 
                             unsigned int dev_id,   unsigned int dev_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len);
static int islmvc_dot11_oid_associate_hndl( unsigned int src_id,   unsigned int src_seq, 
                             unsigned int dev_id,   unsigned int dev_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len);

#ifdef HANDLE_ASSOC_IN_DRIVER
static int islmvc_dot11_oid_authenticate_hndl( unsigned int src_id, unsigned int src_seq, 
                                     unsigned int dev_id,   unsigned int dev_seq,
                                     unsigned int op,       unsigned int oid, void *data, unsigned long len);
static int islmvc_dot11_oid_associate_hndl( unsigned int src_id, unsigned int src_seq, 
                                     unsigned int dev_id,   unsigned int dev_seq,
                                     unsigned int op,       unsigned int oid, void *data, unsigned long len);
#endif /* HANDLE_ASSOC_IN_DRIVER */

#ifdef BLOB_V2
static int islmvc_gen_oid_led_set_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len);
static int islmvc_gen_oid_led_clear_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len);
#endif /* BLOB_V2 */


static int mgmt_set_oid_attachment(char *data, struct net_device *dev, unsigned long length);

unsigned char * pda_get_mac_address(void);




/*
 * Initializes all handlers for the specific OIDs and a generic handler for the rest.
 *
 */
void islmvc_mgt_init(struct net_device *dev)
{
    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "islmvc_mgt_init for dev %s\n", dev->name); 
    
    mgt_init();
    mgt_indication_handler ( DEV_NETWORK, dev->ifindex, 0, islmvc_send_mvc_msg_hndl );
    printk(KERN_ERR "mgt_indication_handler: %d\n", dev->ifindex );
    
    if (GET_DEV_TYPE(dev->base_addr) == BIS_DT_802DOT11) /* Only needed for 802.11 devices */
    {
        mgt_indication_handler ( DEV_NETWORK, dev->ifindex, DOT11_OID_WDSLINKADD,   islmvc_wdslink_add_hndl );
        mgt_indication_handler ( DEV_NETWORK, dev->ifindex, DOT11_OID_WDSLINKREMOVE,islmvc_wdslink_del_hndl );
        mgt_indication_handler ( DEV_NETWORK, dev->ifindex, DOT11_OID_ATTACHMENT,   islmvc_dot11_oid_attachment_hndl );

#ifdef HANDLE_ASSOC_IN_DRIVER
        mgt_response_handler ( 0, 0, DOT11_OID_AUTHENTICATE, islmvc_dot11_oid_authenticate_hndl );    
        mgt_response_handler ( 0, 0, DOT11_OID_ASSOCIATE,    islmvc_dot11_oid_associate_hndl );    
#endif
    }

    /* Mgmt handlers for setting/clearing the LEDs. */
#ifdef BLOB_V2
    mgt_indication_handler ( DEV_NETWORK, dev->ifindex, GEN_OID_LED_SET,   islmvc_gen_oid_led_set_hndl );
    mgt_indication_handler ( DEV_NETWORK, dev->ifindex, GEN_OID_LED_CLEAR, islmvc_gen_oid_led_clear_hndl );
#endif BLOB_V2

}


/**************************************************************** 
 *								*
 *    		Begin of management handling functions		*
 *								*
 ***************************************************************/




/**********************************************************************
 *  islmvc_set_mvc_msg
 *
 *  DESCRIPTION: XXX
 *
 *  PARAMETERS:	 XXX
 *
 *  RETURN:	 XXX
 *
 **********************************************************************/
int islmvc_set_mvc_msg(struct net_device *dev, int oid, long *data, long data_len)
{
    struct msg_conf setmsg;

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "islmvc_set_mvc_msg, dev %s, oid %x, data %p, length %ld\n",
          dev->name, oid, data, data_len); 

    setmsg.operation = OPSET;
    setmsg.oid       = oid;
    setmsg.data      = data;
#ifdef BLOB_V2
    setmsg.size      = data_len; /* MVC v2 only */
#endif
    
    if (dev_conf(GET_DEV_NR(dev->base_addr), &setmsg) < BER_NONE)
        return -EINVAL;
    
    return 0;
}


/**********************************************************************
 *  islmvc_get_mvc_msg
 *
 *  DESCRIPTION: XXX
 *
 *  PARAMETERS:	 XXX
 *
 *  RETURN:	 XXX
 *
 **********************************************************************/
int islmvc_get_mvc_msg(struct net_device *dev, int oid, long *data, long data_len)
{
    struct msg_conf getmsg;
    
    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "islmvc_get_mvc_msg, dev %s, oid %x, data %p, size %ld\n",
          dev->name, oid, data, data_len); 

    getmsg.operation = OPGET;
    getmsg.oid       = oid;
#ifdef BLOB_V2
    getmsg.size      = data_len; /* MVC v2 only */
#endif 

    if (dev_conf (GET_DEV_NR(dev->base_addr), &getmsg) == EOVERFLOW)
    {
#ifdef BLOB_V2        
        printk("Size passed to MVC (%ld) not correct. MVC needs %ld\n", data_len, getmsg.size);
#else
        printk("Size passed to MVC (%ld) not correct.\n", data_len);
#endif
        return -1;
    }

    memcpy(data, getmsg.data, data_len);

    return 0;
}


/* All OIDs */

#define PRISMOP2DEVOP(op) (op == PIMFOR_OP_GET ) ? OPGET : ( (op == PIMFOR_OP_SET ) ? OPSET : -1 )

static int islmvc_send_mvc_msg_hndl( unsigned int dev_type, unsigned int dev_seq, 
                                    unsigned int src_id,   unsigned int src_seq,
                                    unsigned int prismop,  unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
    unsigned int devop = PRISMOP2DEVOP(prismop);

    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_send_mvc_msg_hndl, dev_type %d, dev_seq %d, src_id %d, src_seq %d, prismop %d, oid %x, data %p, len %lu \n",
          dev_type, dev_seq, src_id, src_seq, prismop, oid, data, len);

    if ( devop < 0 )
        return -1;

    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );

    /* src_id and src_seq sunk */ 

    if ( islmvc_send_mvc_msg( dev, devop, oid, data, len ) < 0 ) {
        return -1;
    }

    if ( (mgt_response( src_id, src_seq, dev_type, dev_seq, PIMFOR_OP_RESPONSE, oid, data, len)) < 0 ) {
        return -1;
    }
    
    return 0;
}


/**********************************************************************
 *  islmvc_send_mvc_msg
 *
 *  DESCRIPTION: XXX
 *
 *  PARAMETERS:	 XXX
 *
 *  RETURN:	 XXX
 *
 **********************************************************************/
int islmvc_send_mvc_msg(struct net_device *dev, int operation, int oid, long* data, unsigned long data_len)
{
    int error = 0;
    struct msg_conf msg;
    
    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "islmvc_send_mvc_msg, dev %s, oper %d, oid %x, data %p, len %lu. \n",
          dev->name, operation, oid, data, data_len);

    msg.operation = operation;
    msg.oid       = oid;
    msg.data      = data;
#ifdef BLOB_V2
    msg.size      = data_len; /* MVC v2 only */
#endif


#ifdef TEST_ON_POLDHU 
    /* 
       OIDs for the blob device need some special handling. Don't know if MVC v2 still needs a 
       construction like this? 
    */
    
    if ( (oid & 0xFF000000) == 0x80000000) /* This is an OID for the blob device */
    {
        DEBUG(DBG_MGMT, "Msg for the BLOB device...\n"); 
        /* This is an OID for the BLOB device. We only allow access to the blob device via eth0  */
        if (GET_DEV_TYPE(dev->base_addr) == BIS_DT_802DOT3)
        {
            /* Check if we're manipulating the WLAN LED.
             * If so, we should halt the WLAN device.
             */
            error = BER_NONE;
            switch (oid)
            {
                case BLOB_OID_LEDSET:
                case BLOB_OID_LEDCLEAR:
                if ( (*data & 2) &&
                         (dev_state(1 /* DEVICE_WLAN*/ ) == STRUNNING) )
                {
                    error = dev_halt(1/* DEVICE_WLAN*/);
                }
                break;
                case BLOB_OID_LED_RESET:
                if ( (*data & 2) &&
                         (dev_state(1 /* DEVICE_WLAN*/) == STREADY) )
                {
                    error = dev_run(1 /* DEVICE_WLAN*/);
                }
                return (error < BER_NONE) ? -EINVAL : 0;
                default:
                /* Not one of these */
                break;
            }
            if (error < BER_NONE)
                return -EINVAL;

            if ((error = blob_conf(&msg)) < BER_NONE)
                return -EINVAL;
        }
        else
        {
            return -EOPNOTSUPP;
        }
    }
    else
#endif /* TEST_ON_POLDHU */
    
    {
        /* Send message to MVC */
        error = dev_conf(GET_DEV_NR(dev->base_addr), &msg);
    }

    /* Copy the message data for retrieval by the calling function. */
    memcpy(data, msg.data, data_len); 

    return(error);
}


/**************************************************************** 
 *								*
 *    		End of management handling functions		*
 *								*
 ***************************************************************/


/**************************************************************** 
 *								*
 *    	Functions to retrieve the MAC address from the PDA      *
 *      (Production Data Area)                  	        *
 *								*
 ***************************************************************/
#define PDR_END                                 0x0000
#define PDR_MAC_ADDRESS                         0x0101

#define PTR_PDA_AREA_START 0x00000208 /* Contents of address defines the start of the PDA area. */
#ifdef CONFIG_MTD_PHYSMAP_START
static unsigned long *Pda_area_start = (unsigned long*) (CONFIG_MTD_PHYSMAP_START + PTR_PDA_AREA_START);
#else
// configuration not set, assume it is 0.
static unsigned long *Pda_area_start = (unsigned long*) PTR_PDA_AREA_START;
#endif

struct Pdr_Struct {
   unsigned short   PdrLength;
   unsigned short   PdrCode;
   unsigned char    PdrData[2];
};

/**********************************************************************
 *  pda_find_pdr
 *
 *  DESCRIPTION: Finds a Production Data Record in the PDA. (Production Data Area)
 *
 *  RETURN:	 Pointer to found PDR data, or NULL when not found.
 *
 **********************************************************************/
static struct Pdr_Struct *pda_find_pdr (unsigned short pdrCode)
{
   unsigned short   pdaOffset = 0;
   unsigned short    *pdaPtr;    /* Used as index through the PDA pointing to start of records. */
   struct Pdr_Struct *pdrPtr;    /* Used as pointer to the contents of the record. */
   
   pdaPtr = (unsigned short *) *Pda_area_start;
   pdrPtr = (struct Pdr_Struct *) pdaPtr;
   
   DEBUG(DBG_FUNC_CALL | DBG_MGMT, "pda_find_pdr, pdrCode = %d\n", pdrCode);

  /* Walk through the cached PDA and check each PDR for it's PDRcode to match the one we are looking for. */
   while (pdrPtr->PdrCode != PDR_END) {
      if (pdrPtr->PdrCode == pdrCode) {
         return pdrPtr;
      } else {
         pdaOffset = pdrPtr->PdrLength + 1;
         /* If we get past 8k or we have an invalid PdrLength, return NULL 
            since this indicates an invalid PDA. */
         if ((pdaOffset > 8192) || (pdrPtr->PdrLength == 65535) || (pdrPtr->PdrLength == 0)) {
             return (struct Pdr_Struct *)0;
         }
        /* point to next record */
         pdaPtr = (unsigned short *)pdaPtr + pdaOffset;
         pdrPtr = (struct Pdr_Struct *)pdaPtr;
      }
   };

   if (pdrCode == PDR_END) {
      /* If we were looking for PDR_END, we found it */
       return pdrPtr;
   } else {
      /* We were looking for another PDR and did not find it */
     return (struct Pdr_Struct *)0;
   }
}


/**********************************************************************
 *  pda_get_mac_address
 *
 *  DESCRIPTION: Gets the MAC address from the PDA. (Production Data Area)
 *
 *  RETURN:	 Pointer to MAC address, or NULL when not found.
 *
 **********************************************************************/
unsigned char * pda_get_mac_address(void)
{
   struct Pdr_Struct *pdrStructPtr;

   DEBUG(DBG_FUNC_CALL | DBG_MGMT, "pda_get_mac_address\n");
   
   pdrStructPtr = pda_find_pdr (PDR_MAC_ADDRESS);
   if (pdrStructPtr) {
       return (unsigned char *)(pdrStructPtr->PdrData);
   } else {
       return (unsigned char *)0;
   }
}


/**************************************************************** 
 *								*
 *      	End of PDA MAC address retrieve functions       *
 *								*
 ***************************************************************/




/****************************************************************
 *								
 *    			WDS Link handlers  	
 *								
 ***************************************************************/

static int islmvc_wdslink_add_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
    struct wds_link_msg *wlmp;
    struct wds_priv *wdsp;

    _DEBUG(DBG_FUNC_CALL | DBG_MGMT, "islmvc_wdslink_add_hndl, dev_id %d, dev_seq %d, src_id %d, src_seq %d, op %d, oid %x, data %p, len %lu \n",
          dev_id, dev_seq, src_id, src_seq, op, oid, data, len);
    
    if ( len != sizeof(struct wds_link_msg)) 
        return -1;
        
    wlmp = (struct wds_link_msg *)data;

    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );
    
    wdsp = ((struct net_local *)dev->priv)->wdsp;

    if ( add_wds_link(dev, wdsp, wlmp->mac, wlmp->name) < 0 )
        return -1;

    if ( islmvc_send_mvc_msg( dev, op, oid, (long *)wlmp->mac, 6 ) < 0 )
        return -1;
    /*
    printk ( KERN_ERR "islmvc_wdslink_add_hndl(%x.%x.%x.%x.%x.%x, %s)\n", 
             wlmp->mac[0], wlmp->mac[1], wlmp->mac[2], wlmp->mac[3], wlmp->mac[4], wlmp->mac[5], wlmp->name );
    */
    if ( (mgt_response( src_id, src_seq, dev_id, dev_seq, PIMFOR_OP_RESPONSE, oid, data, len)) < 0 ) 
        return -1;
    
    return 0;
}

static int islmvc_wdslink_del_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
    struct wds_link_msg *wlmp;
    struct wds_priv *wdsp;
        
    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_wdslink_del_hndl, dev_id %d, dev_seq %d, src_id %d, src_seq %d, op %d, oid %x, data %p, len %lu \n",
          dev_id, dev_seq, src_id, src_seq, op, oid, data, len);

    if ( len != sizeof(struct wds_link_msg)) 
        return -1;
        
    wlmp = (struct wds_link_msg *)data;

    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
   	dev_put( dev );
	
    wdsp = ((struct net_local *)dev->priv)->wdsp;
    
    if ( del_wds_link(dev, wdsp, wlmp->mac, wlmp->name ) < 0 )
        return -1;
    
    if ( islmvc_send_mvc_msg( dev, op, oid, (long *)wlmp->mac, 6 ) < 0 ) 
        return -1;
    /*
    printk ( KERN_ERR "islmvc_wdslink_del_hndl(%x.%x.%x.%x.%x.%x, %s)\n", 
             wlmp->mac[0], wlmp->mac[1], wlmp->mac[2], wlmp->mac[3], wlmp->mac[4], wlmp->mac[5], wlmp->name );
    */
    if ( (mgt_response( src_id, src_seq, dev_id, dev_seq, PIMFOR_OP_RESPONSE, oid, data, len)) < 0 ) 
        return -1;

    return 0;
}

/****************************************************************
 *								
 *    			DOT11 OID Attachment handler
 *								
 ***************************************************************/
static int islmvc_dot11_oid_attachment_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
        
    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_dot11_oid_attachment_hndl, dev_id %d, dev_seq %d, src_id %d, src_seq %d, op %d, oid %x, data %p, len %lu \n",
          dev_id, dev_seq, src_id, src_seq, op, oid, data, len);
    
    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );
    
    if ( mgmt_set_oid_attachment(data, dev, len) < 0 )
        return -1;
    
    if ( (mgt_response( src_id, src_seq, dev_id, dev_seq, PIMFOR_OP_RESPONSE, oid, data, len)) < 0 ) 
        return -1;

    return 0;
}


/****************************************************************
 *								
 *   		DOT11 OID Authentication/Association handlers
 *								
 ***************************************************************/

#ifdef HANDLE_ASSOC_IN_DRIVER
/**********************************************************************
 *
 *  mgmt_authenticate
 *
 **********************************************************************/
static void mgmt_authenticate (struct net_device *dev, char *data)
{   
   struct obj_mlme *obj = (struct obj_mlme*) data;
   
   DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_authenticate, dev %s, data %p, (macaddress %s) \n", 
         dev->name, data, hwaddr2string(obj->address));
   
   /* When there's no authentication application, we succeed always via this way... */
   obj->code  = 0; // = DOT11_SC_SUCCESFUL;
   islmvc_send_mvc_msg(dev, OPSET, DOT11_OID_AUTHENTICATE, (long *) obj, sizeof(struct obj_mlme));
   
}


/* DOT11_OID_AUTHENTICATE */
static int islmvc_dot11_oid_authenticate_hndl( unsigned int src_id, unsigned int src_seq, 
                                     unsigned int dev_id,   unsigned int dev_seq,
                                     unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
    
    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_dot11_oid_authenticate_hndl, src_id %d, src_seq %d, dev_id %d, dev_seq %d, op %d, oid %x, data %p, len %lu \n",
          src_id, src_seq, dev_id, dev_seq, op, oid, data, len);

    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );
    
    mgmt_authenticate (dev, (char*) data);
    
    return 0;
}


/**********************************************************************
 *
 *  mgmt_associate
 *
 **********************************************************************/
static void mgmt_associate (struct net_device *dev, char *data)
{   
   struct obj_mlme *obj = (struct obj_mlme*) data;
   
   DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_associate, dev %s, macaddress = %s\n", 
         dev->name, hwaddr2string(obj->address));
   
   if (obj->state == DOT11_STATE_ASSOCING) 
   {
       /* When there's no authentication application, we succeed always via this way... */
       obj->code  = 0; // = DOT11_SC_SUCCESFUL;
       islmvc_send_mvc_msg(dev, OPSET, DOT11_OID_ASSOCIATE, (long *) obj, sizeof(struct obj_mlme));
   }
}


/* DOT11_OID_ASSOCIATE */
static int islmvc_dot11_oid_associate_hndl( unsigned int src_id, unsigned int src_seq, 
                                     unsigned int dev_id,   unsigned int dev_seq,
                                     unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;

    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_dot11_oid_associate_hndl, src_id %d, src_seq %d, dev_id %d, dev_seq %d, op %d, oid %x, data %p, len %lu \n",
          src_id, src_seq, dev_id, dev_seq, op, oid, data, len);

    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );
    
    mgmt_associate (dev, (char*) data);
    
    return 0;
}

#endif /* HANDLE_ASSOC_IN_DRIVER */


/****************************************************************
 *								
 *   		GENERIC OID SET/CLEAR LED handlers
 *								
 ***************************************************************/

#ifdef BLOB_V2
static int islmvc_gen_oid_led_set_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
        
    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_gen_oid_les_set_hndl, src_id %d, src_seq %d, dev_id %d, dev_seq %d, op %d, oid %x, data %p, len %lu \n",
          src_id, src_seq, dev_id, dev_seq, op, oid, data, len);
    
    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );
    
    if ( mgmt_set_led(dev) < 0 )
        return -1;
    
    if ( (mgt_response( src_id, src_seq, dev_id, dev_seq, PIMFOR_OP_RESPONSE, oid, data, len)) < 0 ) 
        return -1;

    return 0;
}


static int islmvc_gen_oid_led_clear_hndl( unsigned int dev_id,   unsigned int dev_seq, 
                             unsigned int src_id,   unsigned int src_seq,
                             unsigned int op,       unsigned int oid, void *data, unsigned long len)
{
    struct net_device *dev;
        
    _DEBUG(DBG_FUNC_CALL | DBG_MGMT,
          "islmvc_gen_oid_led_clear_hndl, src_id %d, src_seq %d, dev_id %d, dev_seq %d, op %d, oid %x, data %p, len %lu \n",
          src_id, src_seq, dev_id, dev_seq, op, oid, data, len);

    if ( ! (dev = dev_get_by_index(dev_seq)) ) 
        return -1;
	dev_put( dev );
    
    if ( mgmt_clear_led(dev) < 0 )
        return -1;
    
    if ( (mgt_response( src_id, src_seq, dev_id, dev_seq, PIMFOR_OP_RESPONSE, oid, data, len)) < 0 ) 
        return -1;

    return 0;
}
#endif /* BLOB_V2 */


/****************************************************************
 *								
 *    			Client table functions
 *								
 ***************************************************************/

#ifdef ENABLE_CLIENT_TABLE_CODE

/**********************************************************************
 *  mac_hash
 *
 *  DESCRIPTION: Computes hash value for a given mac-address. 
 *               Taken from bridging code by Lennert Buytenhek.
 *
 *  PARAMETERS:	 Pointer to mac address.
 *
 *  RETURN:      Hash value.
 *
 *  NOTE:        Inline function.	 
 *
 **********************************************************************/
static inline int mac_hash(unsigned char *mac)
{
    unsigned long x;

    //FIXME, use faster hash func.
    
    x = mac[0];
    x = (x << 2) ^ mac[1];
    x = (x << 2) ^ mac[2];
    x = (x << 2) ^ mac[3];
    x = (x << 2) ^ mac[4];
    x = (x << 2) ^ mac[5];

    x ^= x >> 8;

    return x & (MAC_HASH_SIZE - 1);
}


/*
 * Inline function that links client into the association table.
 */
static inline void client_link(struct net_local *lp, struct client_information *cip, int hash)
{
	cip->next_client = lp->assoc_table[hash];

        if(cip->next_client != NULL)
            cip->next_client->pprev_client = &cip->next_client; //FIXME, check neccesity...
        lp->assoc_table[hash] = cip;
        cip->pprev_client = &lp->assoc_table[hash];
}


/*
 * Inline function that removes client from the association table.
 */
static inline void client_unlink(struct client_information *cip)
{
	*(cip->pprev_client) = cip->next_client;
	if (cip->next_client != NULL)
		cip->next_client->pprev_client = cip->pprev_client;
	cip->next_client = NULL;
	cip->pprev_client= NULL;
}


/**********************************************************************
 *  mgmt_find_client
 *
 *  DESCRIPTION: Searches the association table of the device for the
 *               given client mac-address.
 *
 *  PARAMETERS:	 - pointer to device that holds the association table.
 *               - pointer to mac address we're searching for.
 *
 *  RETURN:      Pointer to client_information struct or
 *               NULL when client is not found.
 *
 *  NOTE:        Inline function.	 
 *
 **********************************************************************/
static inline struct client_information *mgmt_find_client(struct net_device *dev, char *address)
{
    struct net_local *lp = dev->priv;
    int hash = mac_hash(address);
    struct client_information *cip = lp->assoc_table[hash];

    while(cip != NULL) 
    {
        if (!memcmp(cip->mac_addr, address, ETH_ALEN))
            return (cip);

        cip = cip->next_client;
    }
    
    _DEBUG(DBG_MGMT, "Address %s not found in assoc table... \n", hwaddr2string(address));

    return NULL;
}


static int mgmt_add_client(struct net_device *dev, char *address)
{
    struct client_information *cip;
    struct net_local *lp = dev->priv;
    int hash;

    hash = mac_hash(address);

    cip = lp->assoc_table[hash];
    
    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_add_client, dev = %s, hash = %i, addr = %s \n",
                    dev->name, hash, hwaddr2string(address));
    
    while(cip != NULL) 
    {
        if (!memcmp(cip->mac_addr, address, ETH_ALEN))
        {
            DEBUG(DBG_MGMT, "address %s already in assoc_table...\n", hwaddr2string(address));
            return -EINVAL;
        }

        cip = cip->next_client;
    }

    cip = kmalloc(sizeof(*cip), GFP_ATOMIC);
    if (cip == NULL)
    {
        printk("Unable to allocate memory for client \n");
        return -ENOMEM;
    }

    DEBUG(DBG_MGMT, "Address %s not yet in table, going to add it...", hwaddr2string(address));
    
    /* Fill in the client information fields... */
    memcpy(cip->mac_addr, address, ETH_ALEN);
    
    /* Link the client into the assoc table... */
    client_link(lp, cip, hash);

    return 0;

}


/* 
 * mgmt_del_client
 *
 * Removes a client from the assoc table of the device.
 *
 * RETURN: 0 when client is removed from assocation table.
 *        -1 when client is not removed from assocation table.
 */
static int mgmt_del_client(struct net_device *dev, char *address)
{
    struct client_information *cip;
    struct net_local *lp = dev->priv;
    int hash;
    
    hash = mac_hash(address);

    cip = lp->assoc_table[hash];

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_del_client, dev = %s, hash = %i, addr = %s \n",
                    dev->name, hash, hwaddr2string(address));
    
    while(cip != NULL) 
    {
        if (!memcmp(cip->mac_addr, address, ETH_ALEN))
        {
            DEBUG(DBG_MGMT, "found address %2.2x.%2.2x.%2.2x.%2.2x.%2.2x.%2.2x to remove from assoc_table...\n",
                   address[0], address[1], address[2], address[3], address[4], address[5]);
            
            client_unlink(cip);
            kfree(cip); 
            return 0;
        }

        cip = cip->next_client;
    }

    DEBUG(DBG_MGMT, "Address %2.2x.%2.2x.%2.2x.%2.2x.%2.2x.%2.2x not found in assoc table... \n",
                    address[0], address[1], address[2], address[3], address[4], address[5]);

    return -1;
}

#endif /* ENABLE_CLIENT_TABLE_CODE */


/**********************************************************************
 *  mgmt_make_obj_attachment
 *
 *  DESCRIPTION: Calculates the size of all cached information elements.
 *               Mallocs a new obj attachment with this size.
 *               Copies all information elements into the data of this 
 *               obj attachment.
 *
 *  PARAMETERS:	 Pointer to frame_attachment struct. 
 *               Pointer to old object attachment for oa type and id.
 *
 *  RETURN:	 Pointer to new object attachment when succesfully malloced
 *               and data succesfully copied.
 *               NULL when information element == NULL or kmalloc failed.
 *
 **********************************************************************/
static struct obj_attachment *mgmt_make_obj_attachment(struct frame_attachment *fa, struct obj_attachment *old_oa)
{
    int i;
    
    int totalsize = 0;
    int offset = 0;
    struct information_element *ie = fa->ie_root;
    struct obj_attachment *oa = NULL;

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_make_obj_attachment, fa %p, old_oa %p \n", fa, old_oa);
    
    if (ie == NULL)
    {
        printk(KERN_WARNING "No information elements for this attachment found. \n");
        return NULL;
    }

    /* We need to determine the total size of the information elements first. */
    for (i=0; (i <= MAX_IE_ELEMENTS && ie != NULL); i++, ie=ie->next)
        totalsize += (short)ie->size; 
    
    /* Set the pointer back to it's roots... */
    ie = fa->ie_root;
    
    /* Malloc a buffer for the obj_attachment + the data. */ 
    if ( (oa = kmalloc( (sizeof(struct obj_attachment) + totalsize), GFP_KERNEL)) == NULL)
        return NULL;
    
    oa->type = old_oa->type;
    oa->id   = old_oa->id;
    oa->size = totalsize;

    /* Copy the information elements into the buffer. */
    for (i=0; (i <= MAX_IE_ELEMENTS && ie != NULL); i++, ie=ie->next)
     {
        if (ie->data == NULL || ie->size == 0)
            continue;
        memcpy((oa->data+offset), ie->data, ie->size);
        offset += ie->size;
    }
    
    return(oa);
}

/**********************************************************************
 *  mgmt_add_information_element
 *
 *  DESCRIPTION: Checks if we have an information element for this element id.
 *               If so, it replaces the current element data for the new one,
 *               or removes the element when size == 0.
 *               If no previous information element for this id was found,
 *               a new information element is malloced, and put into the list.
 *
 *  PARAMETERS:	 Pointer to frame_attachment for this frame type.
 *               Pointer to the information element data. Contains element id.
 *               ie_size, which is the size of ie_data.   
 *
 *  RETURN:	 0 when succesfully added or replaced.
 *               1 when succesfully removed.
 *               -1 when element was not found, and MAX_IE_ELEMENTS was reached.
 *               -ENOMEM when we couldn't allocate memory for the information
 *               element or elements data.
 *
 **********************************************************************/
static int mgmt_add_information_element(struct frame_attachment *fa, char *ie_data, short ie_size)
{
    int i;
    unsigned char elem_id;
    
    int found_element                = FALSE;
    struct information_element *ie   = fa->ie_root;
    struct information_element *prev = ie;

    /* The element id should be set as first parameter in the information element data. */
    elem_id  = *ie_data;
    
    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_add_information_element, fa %p, ie_data %p, ie_size %d \n", fa, ie_data, ie_size);
    
    /* Check if we have an id for this element */
    for (i=0; (ie != NULL && i <= MAX_IE_ELEMENTS); i++)
    {   
        if (ie->elem_id == elem_id)
        {
            found_element = TRUE;

            if (ie->data == NULL)
                DEBUG(DBG_MGMT, "Found information element, but it contains no data. \n");
            
            DEBUG(DBG_MGMT, "Found information element %d with size %d\n", ie->elem_id, ie->size);
            
            /* Valid information elements must have a size greater than 1. */
            if (ie_size <= 1) {
                DEBUG(DBG_MGMT, "Going to remove information element %d \n", elem_id);
                
                if (ie == fa->ie_root)
                    fa->ie_root = ie->next;
                else 
                    prev->next = ie->next;
                kfree(ie); 
                return 1; 
            }
            else {
                /* If size is the same or less as the previous data size and we have a buffer, copy the data */
                if (ie->size <= ie_size && ie->data != NULL) {
                    ie->size = ie_size;
                    memcpy(ie->data, ie_data, ie->size);
                    DEBUG(DBG_MGMT, "IE size (%d) is same or less as the previous data size (%d). \n", ie_size, ie->size);
                    return 0; 
                }
                else {
                    /* We need to malloc a new area for the data. */
                    kfree(ie->data); /* Free the old area first! */
                    
                    if ( (ie->data = kmalloc(ie_size, GFP_KERNEL)) == NULL) {
                        printk(KERN_ERR "kmalloc of data for information element %d failed. \n", ie->elem_id);
                        ie->size = 0;
                        return -ENOMEM;
                    }
                    ie->size = ie_size;
                    memcpy(ie->data, ie_data, ie->size);
                    DEBUG(DBG_MGMT, "IE size (%d) is greater then the previous data size (%d). \n", ie_size, ie->size);
                    return 0;
                }
            }
        }
        else {
            prev = ie;
            ie   = ie->next;
        }
    }
    
    if (i == MAX_IE_ELEMENTS && !found_element) {
        printk(KERN_WARNING "Element not found. Maximum information elements (%d) reached. \n", MAX_IE_ELEMENTS);
        return -1;
    }

    if (!found_element) {
        DEBUG(DBG_MGMT, "We didn't find an element... add it.\n");
        
        if (ie_size == 0) {
            /* If ie_size == 0, we're asked to remove this element.
               But since we didn't find an element in our list we can't do that. */
            DEBUG(DBG_MGMT, "Asked to remove information element (%d), but no previous element found. \n", ie->elem_id);
            return -1;
        }
        
        /* Malloc a new information element. */
        if ( (ie = kmalloc(sizeof(struct information_element), GFP_KERNEL)) == NULL) {
            printk("Couldn't malloc memory for information element. \n");
            return -ENOMEM;
        }
        
        ie->elem_id = elem_id;
        ie->size    = 0;
        ie->data    = NULL;
        ie->next    = NULL;
        
        if ( (ie->data = kmalloc(ie_size, GFP_KERNEL)) == NULL) {
            printk(KERN_ERR "kmalloc of data for information element failed. \n");
            kfree(ie);
            ie->size = 0;
            return -ENOMEM;
        }
        ie->size = ie_size;
        memcpy(ie->data, ie_data, ie_size);
        
        /* Add the information element to the list. */
        DEBUG(DBG_MGMT, "Add the information element to the list.\n");
        ie->next    = fa->ie_root;
        fa->ie_root = ie;
        
        return 0; 
    }
    
    /* We shouldn't come here... */
    DEBUG(DBG_MGMT, "mgmt_add_information_element, shouldn't come here... \n");
    return -1;
}


/*
 * Finds frame attachment for frame type and association id. 
 *
 * - Returns pointer to frame attachment when found.
 * - Returns new frame_attachment for this type when no frame attachment
 *   was found and malloc of new frame attachment succeeded. 
 * - Returns NULL when malloc failed.
 */
static struct frame_attachment *find_frame_attach(short type, short id)
{
    struct frame_attachment *fa   = frame_attach[type][id]; 
    struct frame_attachment *prev = NULL;
    prev->next = NULL;

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "find_frame_attach, type %x, id(+1) %x, (fa = %p).\n", type, id, fa);
    
    if (fa == NULL) /* Not found */
    {
        /* No previous frame_attachment for this type and id. Malloc one. */
        if ( (fa = kmalloc(sizeof(struct frame_attachment), GFP_KERNEL)) == NULL)
        {
            printk(KERN_ERR "Couldn't malloc space for frame_attachment. \n");
            return NULL;
        }
        fa->next        = prev->next;
        fa->frame_type  = type;
        fa->ie_root     = NULL;
        fa->all_ie_data = NULL;
        
        frame_attach[type][id] = fa;
        DEBUG(DBG_MGMT, "find_frame_attach, no frame attachment found. Added it to the list. fa = %p\n", fa);
        return(fa);
    }
    else {/* Found it */
        DEBUG(DBG_MGMT, "find_frame_attach, frame attachment found. fa = %p\n", fa);
        return(fa);

    }
}

/**********************************************************************
 *  mgmt_cache_frame_attachment
 *
 *  DESCRIPTION: Information elements in frame attachments must be cached
 *               in the driver, as the MVC won't do that for us.
 *
 *  PARAMETERS:	 data       - pointer to the struct obj_attachment.
 *               data_len   - length of the
 *               dev        - pointer to device we need to send information
 *                            elements to.
 *
 *  RETURN:	 0  when caching and sending the information elements to
 *                  the MVC succeeded.
 *               -1 when caching and sending information elements to the
 *                  MVC failed.
 *
 **********************************************************************/
static int mgmt_cache_frame_attachment(struct net_device *dev, char *data)
{
    int error;
    
    struct frame_attachment *fa     = NULL;
    struct obj_attachment   *oa     = NULL;
    struct obj_attachment   *old_oa = (struct obj_attachment *)data; 
    
    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_cache_frame_attachment, dev %s, data %p \n", dev->name, data);
    
    /* Find the corresponding frame_attachment for this type and id. If none exists, one is malloced for us.*/
    if ( (fa = find_frame_attach(old_oa->type, old_oa->id+1)) == NULL) { /* id+1, as id can be -1. We map -1 to 0 this way. */
        printk(KERN_ERR "Couldn't malloc space for frame attachment. \n");
        return -ENOMEM;
    }
    
    /* Add information element to this frame attachment. */
    if ( (error = mgmt_add_information_element(fa, old_oa->data, old_oa->size)) < 0) {
        printk(KERN_WARNING "Error %d. Couldn't add information element to frame attachment (type %d, id %d). \n", error, old_oa->type, old_oa->id);
        return -1;
    }
    
    /* Okay, we have a valid list of information elements now. Place them in one obj attachment. */
    oa = mgmt_make_obj_attachment(fa, old_oa);
    
    if (oa == NULL) {
        printk(KERN_ERR "Couldn't make new object attachment for information elements. \n");
        return -ENOMEM;
    }
    
    error = islmvc_send_mvc_msg(dev, OPSET, DOT11_OID_ATTACHMENT, (long*)oa, oa->size);
    
    kfree(oa); /* No need to save it, as all data is available in the information elements list. */
    
    if (error < 0) {
        DEBUG(DBG_MGMT, "Error %d setting DOT11_OID_ATTACHMENT. \n", error);
        return -1;
    }

    return 0;
}


/**********************************************************************
 *  mgmt_set_oid_attachment
 *
 *  DESCRIPTION: Information elements in frame attachments for management
 *               frame types must be cached in the driver,
 *               as the MVC won't do that for us.
 *
 *  PARAMETERS:	 data       - pointer to the struct obj_attachment.
 *               dev        - pointer to device we need to send information
 *                            elements to.
 *               length     - Length of obj_attachment.
 *
 *  RETURN:	 0  when caching and sending the information elements to
 *                  the MVC succeeded.
 *               -1 when caching and sending information elements to the
 *                  MVC failed.
 *
 **********************************************************************/
static int mgmt_set_oid_attachment(char *data, struct net_device *dev, unsigned long length)
{
    int err;
    
    struct obj_attachment *oa = (struct obj_attachment *)data; 

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_set_oid_attachment, data %p, dev %s, length %lu, (type = %x) \n",
          data, dev->name, length, oa->type);
    
    /* Only cache information elements for management frame types. 
       Send all other frame attachments through un-cached.*/
    if ((oa->type & 0x0C /* D11FT_RESERVED */) != 0) {
        DEBUG(DBG_MGMT, "mgmt_set_oid_attachment, No management frame type. Send through uncached. \n");
        return (islmvc_send_mvc_msg(dev, OPSET, DOT11_OID_ATTACHMENT, (long*)data, length));
    }
    
    if ( (err = mgmt_cache_frame_attachment(dev, data)) != 0)
        return -1;
    
    return 0;
}


/**********************************************************************
 *  mgmt_set_led
 *
 *  DESCRIPTION: Sets the LED for device dev. 
 *
 *  PARAMETERS:  dev - pointer to device we want to set the LED for.
 *
 *  RETURN:	 0  when setting the LED succeeded
 *               -1 when setting the LED failed.
 *
 *  NOTE:       This code is ISL3893 specific.
 *
 **********************************************************************/
#ifdef BLOB_V2
int mgmt_set_led(struct net_device *dev)
{
    long data;  
    long mask;
    long led; 

    long bank = 2; /* GP bank 3 */

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_set_led for %s \n", dev->name);

    /* Determine which LED to set. */
    switch (GET_DEV_NR(dev->base_addr))
    {
        case ETH_LAN: 
        {
            led = 0x08; /* GP3[3] */
            break;
        }
        case ETH_WLAN:
        {
            led = 0x02; /* GP3[1] */
            break;
        }
        case ETH_WAN: 
        {
            led = 0x04; /* GP3[2] */
            break;
        }
        default:
        {
            printk(KERN_INFO "Couldn't determine LED to clear. \n");
            return -1;
        }
    }

    /* Select the GPIO bank for this LED */
    if (islmvc_set_mvc_msg(dev, BLOB_OID_GPIOBANK, &bank, sizeof(long)) < 0 )
        return -1;
    
    /* Get the current status of the LEDs */
    if (islmvc_get_mvc_msg(dev, BLOB_OID_GPIODATA, &data, sizeof(long)) < 0 )
        return -1;
    
    /* Configure the GPIO write mask */
    mask = ((led ^ data) & led);
    
    if (islmvc_set_mvc_msg(dev, BLOB_OID_GPIOWM, &mask, sizeof(long)) < 0 )
        return -1;
    
    /* Write the data to the GPIO bank */
    if (islmvc_set_mvc_msg(dev, BLOB_OID_GPIODATA, &data, sizeof(long)) < 0 )
        return -1;            
    return 0;
}


/**********************************************************************
 *  mgmt_clear_led
 *
 *  DESCRIPTION: Clears the LED for device dev. 
 *
 *  PARAMETERS:  dev - pointer to device we want to clear the LED for.
 *
 *  RETURN:	 0  when clearing the LED succeeded
 *               -1 when clearing the LED failed.
 *
 *  NOTE:       This code is ISL3893 specific.
 *
 **********************************************************************/
int mgmt_clear_led(struct net_device *dev)
{
    long data;  
    long mask;
    long led; 

    long bank = 2; /* GP Bank 3 */

    DEBUG(DBG_FUNC_CALL | DBG_MGMT, "mgmt_clear_led for %s \n", dev->name);
    
    /* Determine which LED to clear. */
    switch (GET_DEV_NR(dev->base_addr))
    {
        case ETH_LAN:  
        {
            led = 0x08; /* GP3[3] */
            break;
        }
        case ETH_WLAN: 
        {
            led = 0x02; /* GP3[1] */
            break;
        }
        case ETH_WAN: 
        {
            led = 0x04; /* GP3[2] */
            break;
        }
        default:
        {
            printk(KERN_INFO "Couldn't determine LED to clear. \n");
            return -1;
        }
    }
    
    /* Select the GPIO bank for LEDs */
    if (islmvc_set_mvc_msg(dev, BLOB_OID_GPIOBANK, &bank, sizeof(long)) < 0 )
        return -1;
    
    /* Get the current status of the LEDs */
    if (islmvc_get_mvc_msg(dev, BLOB_OID_GPIODATA, &data, sizeof(long)) < 0 )
        return -1;
    
    /* Configure the GPIO write mask */
    mask = (led & data);
    
    if (islmvc_set_mvc_msg(dev, BLOB_OID_GPIOWM, &mask, sizeof(long)) < 0 )
        return -1;
    
    /* Write the data to the GPIO bank */
    if (islmvc_set_mvc_msg(dev, BLOB_OID_GPIODATA, &data, sizeof(long)) < 0 )
        return -1;
    
    return 0;
}
#endif /* BLOB_V2 */


/****************************************************************
 *								*
 *    			End of management functions		*
 *								*
 ***************************************************************/


/***************************************************************
*								*
*    		Start of generic functions		        *
*								*
***************************************************************/


void do_hard_print(char *string)
{   

#ifdef CONFIG_POLDEBUG_PRINTK

       struct msg_data debug_msg; 

       debug_msg.data = string;
       debug_msg.length = strlen(string) + 1;
       debug_msg.max = 0;

       dev_write(DEVICE_DEBUG, &debug_msg);
      
#endif /* CONFIG_POLDEBUG_PRINTK */

}


/*
 * Returns a pointer to the human-readable hardware address of
 * the client. Buffer is static from function and overwritten
 * with each call
 */
char *hwaddr2string(char *a)
{
    static char buf[18];

    snprintf(buf, 18, "%02x:%02x:%02x:%02x:%02x:%02x",
	    a[0], a[1], a[2], a[3], a[4], a[5]);

    return(buf);
}


/****************************************************************
 *								*
 *    		    End of generic functions                    *
 *								*
 ***************************************************************/



