/*
 $Header: /projects/proj/software/pub/CVSROOT/uClinux/linux/drivers/net/phyethgeneric.c,v 1.18 2003/01/13 18:20:17 rsewill Exp $
 */

/******************************************************************/
/* Copyright (c) 2002 BRECIS Communications                       */
/*                                                                */
/*     This program is free software; you can redistribute it     */
/*     and/or modify it under the terms of the GNU General        */
/*     Public License as published by the Free Software           */
/*     Foundation; either version 2 of the License, or (at your   */
/*     option) any later version.                                 */
/*                                                                */
/*     This program is distributed in the hope that it will be    */
/*     useful, but WITHOUT ANY WARRANTY; without even the implied */
/*     warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR    */
/*     PURPOSE.  See the GNU General Public License for more      */
/*     details.                                                   */
/*                                                                */
/*     You should have received a copy of the GNU General Public  */
/*     License along with this program; if not, write to the Free */
/*     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA     */
/*     02139, USA.                                                */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

#include <asm/brecis/prom.h>
#include "brecismspeth.h"
#include "phyethgeneric.h"

static U32 selectPhyRegister(MSP_DEVICE *pDrvCtrl, U32 reg)
{
    int i;

    WrPhy(pDrvCtrl, MD_CA, (pDrvCtrl->macPhyId | MD_CA_BUSY_BIT | reg));

    i = 0;
    do
    {
        udelay(MD_REGISTER_UDELAY);
        if(((RdPhy(pDrvCtrl, MD_CA)) & MD_CA_BUSY_BIT) == 0)
            return 0;

        i += MD_REGISTER_UDELAY;
    } while (i < PHYG_UDELAY_FOR_RESPONSE);

    return 1;
}

int readPhyRegister(MSP_DEVICE *pDrvCtrl, U32 reg, U32 *variable)
{
	unsigned long flags;

	spin_lock_irqsave(&pDrvCtrl->phy_base->lock, flags);

	if (selectPhyRegister(pDrvCtrl, reg))
	{
		spin_unlock_irqrestore( &pDrvCtrl->phy_base->lock, flags);
		return 1;
	}

	*variable = RdPhy(pDrvCtrl,MD_DATA);

	spin_unlock_irqrestore( &pDrvCtrl->phy_base->lock, flags);

	return 0;
}

int writePhyRegister(MSP_DEVICE *pDrvCtrl, U32 reg, U32 value)
{
	unsigned long flags;

	spin_lock_irqsave(&pDrvCtrl->phy_base->lock, flags);

	WrPhy(pDrvCtrl, MD_DATA, value);

	if (selectPhyRegister(pDrvCtrl, reg | MD_CA_Wr))
	{
		spin_unlock_irqrestore( &pDrvCtrl->phy_base->lock, flags);
		return 1;
	}

	spin_unlock_irqrestore(&pDrvCtrl->phy_base->lock, flags);

	return 0;
}

/*************************************************************************
*
* MSPPHYGCheckCable - Check for Cable connection to the PHY 
*
* RETURNS: non-zero if cable is okay or no PHY or we can't read PHY register
**************************************************************************/
U32 MSPPHYGCheckCable(MSP_DEVICE *pDrvCtrl)
{
	U32	data;

	switch(pDrvCtrl->phyType)
	{
		case PHYTYPE_NONE:
		case PHYTYPE_KENDINSWITCH:
			return 1;	/* no PHY, cable is always okay */

		default:
			break;
	}

	if (readPhyRegister(pDrvCtrl, PHYG_STATUS_REG_ONE, &data)) 
	{
		printk(KERN_INFO "MSPPHYGCheckCable: BUSY Cannot Read PHYG_CONTROL\n");
		return 1;  /* don't know what to do, pretend cable is okay */
	}
	
	return data & PHYGb_STATUS_REG_ONE_LINK_STATUS;
}


/*************************************************************************
*
* MSPPHYGGetSpeedAndDuplex - Gets the Working speed from the PHY 
*
* RETURNS: MSP_SUCCESS or ERROR 
*  
**************************************************************************/
U32 MSPPHYGGetSpeedAndDuplex(MSP_DEVICE *pDrvCtrl)
{
	U32 reg4;
	U32 reg5;
	U32 data;
	
	printk(KERN_INFO "MSPPHYGGetSpeedAndDuplex: for MAC# %d\n", 
	       pDrvCtrl->unit);

	/* Read Register 0 (PHYG_CONTROL) */
	if (readPhyRegister(pDrvCtrl, PHYG_CONTROL, &data)) 
	{
		printk(KERN_INFO "MSPPHYGGetSpeedAndDuplex: BUSY Cannot Read PHYG_CONTROL\n");
		return MSP_MAC_PHY_ERROR ;
	}
	
#ifdef _MSP2100	
	// add by hook 20030610
	// bypass readPhyRegister for eth2, because hardware fixed it to 100_FULL
	if (pDrvCtrl->unit == LANPORT)
		data = PHYGb_CONTROL_SPEED_100_FULL;
	// end of add by hook
#endif

	if ((data & PHYGb_CONTROL_AUTONEG) == 0)
	{
		printk(KERN_INFO "PHY is in MANUAL CONFIGURATION MODE\n");
		if (data & PHYGb_CONTROL_SELECT_SPEED)
		{
			printk(KERN_INFO "PHY is set for 100 Mbps\n");
			pDrvCtrl->speed = MSPEth_100MBS ;
		}
		else
		{
			printk(KERN_INFO "PHY is set for 10 Mbps\n");
			pDrvCtrl->speed = MSPEth_10MBS ;
		}

		if (data & PHYGb_CONTROL_DUPLEXMODE)
		{
			printk(KERN_INFO "PHY is set for Full Duplex\n");
			pDrvCtrl->duplexMode = MAC_DPX_MODE_FULL;
		}
		else
		{
			printk(KERN_INFO "PHY is set for Half Duplex\n");
			pDrvCtrl->duplexMode = MAC_DPX_MODE_HALF;
		}
		return MSP_SUCCESS;
	}

	printk(KERN_INFO "PHY is in AUTO NEGOTIATION MODE\n");
	/* doing auto negotiation */
	/* Read Register 4 (PHYG_AUTO_NEG_ADVERTISING) */
	if (readPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_ADVERTISING, &reg4)) 
	{
		printk(KERN_INFO "MSPPHYGGetSpeedAndDuplex: PHY BUSY Cannot Read PHYG_AUTO_NEG_ADVERTISING\n");
		return MSP_MAC_PHY_ERROR ;
	}

	/* Read Register 5 (PHYG_AUTO_NEG_LINK_PARTNER_ABILITY) */
	if (readPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_LINK_PARTNER_ABILITY, &reg5)) 
	{
		printk(KERN_INFO "MSPPHYGGetSpeedAndDuplex PHY BUSY Cannot Read PHYG_AUTO_NEG_LINK_PARTNER_ABILITY\n");
		return(MSP_MAC_PHY_ERROR);
	}

	data = reg4 & reg5;

	if (data & PHYGb_AUTO_NEG_ADVERT_100BASE_T4)
	{
		pDrvCtrl->speed = MSPEth_100MBS ;
		pDrvCtrl->duplexMode = MAC_DPX_MODE_HALF;
		printk(KERN_INFO "PHY is set for 100Mbps Half Duplex\n");
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_100BASE_TX_FULL)
	{
		pDrvCtrl->speed = MSPEth_100MBS ;
		pDrvCtrl->duplexMode = MAC_DPX_MODE_FULL;
		printk(KERN_INFO "PHY is set for 100Mbps Full Duplex\n");
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_100BASE_TX)
	{
		pDrvCtrl->speed = MSPEth_100MBS ;
		pDrvCtrl->duplexMode = MAC_DPX_MODE_HALF;
		printk(KERN_INFO "PHY is set for 100Mbps Half Duplex\n");
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_10BASE_T_FULL)
	{
		pDrvCtrl->speed = MSPEth_10MBS ;
		pDrvCtrl->duplexMode = MAC_DPX_MODE_FULL;
		printk(KERN_INFO "PHY is set for 10Mbps Full Duplex\n");
	}
	else if (data & PHYGb_AUTO_NEG_ADVERT_10BASE_T)
	{
		pDrvCtrl->speed = MSPEth_10MBS ;
		pDrvCtrl->duplexMode = MAC_DPX_MODE_HALF;
		printk(KERN_INFO "PHY is set for 10Mbps Half Duplex\n");
	}
	return MSP_SUCCESS;
}


U32 MSPPHYGWaitForLink(MSP_DEVICE *pDrvCtrl) 
{
	U32	data;

	/* wait for link or auto negotiation to complete */

	/* Read Register 1 (PHYG_STATUS_REG_ONE) */
	if (readPhyRegister(pDrvCtrl, PHYG_STATUS_REG_ONE, &data)) 
	{
		printk(KERN_INFO "MSPPHYGWaitForLink: PHY BUSY PHYG_STATUS_REG_ONE\n");
		pDrvCtrl->phyWaiting = 0;
		return MSP_MAC_PHY_ERROR ;
	}
	
	if(pDrvCtrl->phyAutoNegot)
	{
		if(data & PHYGb_STATUS_REG_ONE_AUTO_NEG_DONE) 
		{
			printk(KERN_INFO "AUTO NEGOTIATION COMPLETE\n");
			pDrvCtrl->phyWaiting = 0;
			return MSPPHYGGetSpeedAndDuplex(pDrvCtrl);
		}
	}
	else if(data & PHYGb_STATUS_REG_ONE_LINK_STATUS)
	{
		printk(KERN_INFO "LINK DETECTED\n");
		pDrvCtrl->phyWaiting = 0;
		return MSPPHYGGetSpeedAndDuplex(pDrvCtrl);
	}

	if (--pDrvCtrl->phyWaiting == 0)
	{
		printk(KERN_INFO "**** ERROR *** MAC#: %d, NO LINK DETECTED\n",
		       pDrvCtrl->unit);
		return MSP_FAIL;
	}

	return MSP_MAC_PHY_NO_LINK ;
}

/*************************************************************************
*
* MSPPHYGSetup - initialize and configure the PHY device.
* This routine initialize and configure the PHY device 
*
* RETURNS: OK, or ERROR
**************************************************************************/
void MSPPHYGSetup (MSP_DEVICE *pDrvCtrl) 
{
	U32	data;

	printk(KERN_INFO "Entering MSPPHYGSetup() for MAC#: %d\n", 
	       pDrvCtrl->unit);

	printk(KERN_INFO
	       "Using PHY address %x with base address %x for MAC#: %d\n",
	       (unsigned int) pDrvCtrl->macPhyId, 
	       (unsigned int) pDrvCtrl->phy_base->mac_base, 
	       pDrvCtrl->unit);

	/*
	 * Loopback=disabled
	 * Speed = 100/10
	 * Auto-Negotiation = Disabled
	 * Power Down = 0 normal
	 * Isolate = 0 normal
	 * Restart Auto-Negotiation = 0
	 * Duplex = 1 Full / 0 Half
	 * Collision Test = 0 Disabled
	 *
	 * For Full Dup 10Mbps:  0x0100
	 * For Half Dup 10Mbps:  0x0000
	 * For Full Dup 100Mbps: 0x2100
	 * For Half Dup 100Mbps: 0x2000
	 *
	 * Auto Negotiation Enabled: 0x1000
	 */
	switch(pDrvCtrl->ethMode)
	{
		case ETH_AUTO:
			printk(KERN_INFO "PHY is set for Auto-Negotiation \n");
			data = PHYGb_CONTROL_AUTONEG;
			break;

		case ETH_10Mbps_HALF:
			printk(KERN_INFO "PHY is set for 10 Mbps Half Duplex \n");
			data = PHYGb_CONTROL_SPEED_10_HALF;
			break;

		case ETH_10Mbps_FULL:
			printk(KERN_INFO "PHY is set for 10 Mbps Full Duplex \n");
			data = PHYGb_CONTROL_SPEED_10_FULL;
			break;
		
		case ETH_100Mbps_HALF:
			printk(KERN_INFO "PHY is set for 100 Mbps Half Duplex \n");
			data = PHYGb_CONTROL_SPEED_100_HALF;
			break;
		
		case ETH_100Mbps_FULL:
			printk(KERN_INFO "PHY is set for 100 Mbps Full Duplex \n");
			data = PHYGb_CONTROL_SPEED_100_FULL;
			break;	
	}


	/* send the data to the PHY */
   	if (writePhyRegister(pDrvCtrl, PHYG_CONTROL, data) )
	{
		printk(KERN_INFO "MSPPHYGSetup():  PHY BUSY Setting Speed/Mode\n");
		return;
	}
	
	/* Read Register 0 (PHYG_CONTROL) */
	if (readPhyRegister(pDrvCtrl, PHYG_CONTROL, &data)) 
	{
		printk(KERN_INFO "MSPPHYGWaitForLink: PHY BUSY PHYG_STATUS_REG_ONE\n");
		return;
	}

#ifdef _MSP2100	
// add by hook 20030610
// add the if condition, MSP2100's PHY don't set AutoNegot mode when pDrvCtrl->unit is LAN port 	
	if (pDrvCtrl->unit != LANPORT)
		pDrvCtrl->phyAutoNegot = data & PHYGb_CONTROL_AUTONEG;
// end of add by hook	
#endif

	pDrvCtrl->phyWaiting = MAX_TIME_TO_WAIT_FOR_LINK;
}


/*************************************************************************
*
* MSPPHYGDisplayGenericRegisters - displays the registers 
*  
**************************************************************************/
void MSPPHYGDisplayGenericRegisters(U32 interfaceNum)
{
	MSP_DEVICE	*pDrvCtrl; 

	if (interfaceNum >= MSP_ETH_MAX_UNITS)
	{
		printk (KERN_WARNING "interface %d out of range\n", 
			(int) interfaceNum);
		return;
	}

	pDrvCtrl = gEthDrvCtrl[interfaceNum];

	printk(KERN_INFO "\n---------------------------------------------------\n");
	printk(KERN_INFO "Displaying PHY Registers.\n");
	printPhyRegister(pDrvCtrl, PHYG_CONTROL );
	printPhyRegister(pDrvCtrl, PHYG_STATUS_REG_ONE );
	printPhyRegister(pDrvCtrl, PHYG_PHY_ID_REG_ONE );
	printPhyRegister(pDrvCtrl, PHYG_PHY_ID_REG_TWO );
	printPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_ADVERTISING );
	printPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_LINK_PARTNER_ABILITY );
	printPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_EXPANSION );
	printPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_NEXT_PAGE_TX );
	printPhyRegister(pDrvCtrl, PHYG_AUTO_NEG_LINK_PARTNER_RX_NEXT );
}



void printPhyRegister(MSP_DEVICE *pDrvCtrl, U32 reg)
{
	U32	data;

	if (readPhyRegister(pDrvCtrl, reg, &data) )
	{
		printk(KERN_INFO "printPhyRegister: PHY BUSY\n");
		return;
	}

	printk(KERN_INFO "PHY Register at Offset %d = 0x%04X\n", 
	       (int) reg, (int) data);
}


void MSPPHYGIsolateON(MSP_DEVICE *pDrvCtrl)
{
	U32 data;

	if (readPhyRegister(pDrvCtrl, PHYG_CONTROL, &data)) 
	{
		printk(KERN_INFO "MSPPHYGIsolateON: PHY BUSY PHYG_CONTROL\n");
		return;
	}	

	data = data | PHYGb_CONTROL_ISOLATE;

	/* send the data to the PHY */
   	if (writePhyRegister(pDrvCtrl, PHYG_CONTROL, data) )
	{
		printk(KERN_INFO "MSPEthIsolatePHYON():  PHY BUSY\n");
		return;
	}
}

void MSPPHYGIsolateOFF(MSP_DEVICE *pDrvCtrl)
{
	U32 data;

	if (readPhyRegister(pDrvCtrl, PHYG_CONTROL, &data)) 
	{
		printk(KERN_INFO "MSPPHYGIsolate:  Cannot Read PHYG_CONTROL\n");
		return;
	}	

	data = data & ~PHYGb_CONTROL_ISOLATE;

	/* send the data to the PHY */
   	if (writePhyRegister(pDrvCtrl, PHYG_CONTROL, data) )
	{
		printk(KERN_INFO "MSPPHYGIsolate:  PHY BUSY\n");
		return;
	}
}


