/*
 $Header: /proj/software/pub/CVSROOT/uClinux/linux/drivers/brecis/dma_bcopy.c,v 1.1 2002/05/21 22:56:42 swahl Exp $
 */


/******************************************************************/
/* Copyright (c) 2001, 2002 BRECIS Communications                 */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/system.h>

#define	_ASMLANGUAGE	1

#include <asm/brecis/BrecisTriadMap.h>

#include <brecis/NanoDelay.h>
#include <brecis/dma_bcopy.h>

/* #define DEBUG y */
#define BCOPY_DEBUG	KERN_ALERT	/* make sure it shows up */
#ifdef DEBUG
#define DBG_CPY(a1, a2...)	printk(BCOPY_DEBUG "BCOPY " a1, ##a2)
#else
#define DBG_CPY(a...)
#endif

#define MSP_BCOPY_EFFICIENCY 32
#define	REG(x)	(volatile unsigned long *)(x)

#define SYNC() __asm__ __volatile__("sync");

/***
 * CACHE stuff.
 */

/* define for a Cache Invalidate */
#define msp_hit_inval_dcache(addr)    __asm__ __volatile__ \
    ( \
	".set push\n\t.set mips3\n\t.set noreorder\n\t" \
	"cache 0x11, (%0)\n\tnop\n\t.set pop" : : "r" (addr) \
    )


/***
 * invalidate_buffer - Invalidate data cache
 * Stolen From:         triad.c (ethernet driver
 */
static void invalidate_buffer(void *bufaddr, unsigned int length)
{
	unsigned long	addr = (unsigned long) bufaddr;
	unsigned long	end = addr + length;

	while (addr <= end)
	{
		msp_hit_inval_dcache(addr);
		addr += L1_CACHE_BYTES;
	}
}


/***
 * Name  :	dma_bcopy
 * Input :	SrcAdd	- address of source to copy					
 *		DestAdd	- address of destination to copy
 *		Mode	- Mode to transfer
 *		len	- length of the buffer to transfer
 * Output:	return the error if transfer not completed
 *
 * this routine will transfer the 2 block of address if Fill mode
 * on the SrcAdd will be the value of fill register.
 * Otherwise everything work at normal
 */
int	dma_bcopy(void *SrcAdd, void *DestAdd,
		unsigned int mode, unsigned int len)
{
	unsigned long	status;
	unsigned long	flags;

	if (mode & BCOPY_INTERRUPT_ENABLE)
	{
		printk(KERN_WARNING "copy driver called for async copy");
		return -EINVAL;
	}

	if ((mode == BC_CP_DINC_SINC_NINT_MODE) && (len < MSP_BCOPY_EFFICIENCY))
	{
		unsigned char *in = SrcAdd;
		unsigned char *out = DestAdd;
		
		for (status = 0; status < len; status++)
		{
			*out++ = *in++;
		}
		return 0;
	}

	/* Ensure any data in the write buffers is now in SDRAM */

	save_and_cli(flags);
	SYNC();

	if (*REG(TRIAD_BCOPY_STATUS_REG) & BCOPY_STATUS_BUSY)
	{
		restore_flags(flags);
		return -EINVAL;
	}

	if (mode & BCOPY_MODE_CNTL_FILL_MODE)
		*REG(TRIAD_BCOPY_FILL_REG) = (unsigned long)SrcAdd;	/* program Fill register - fill mode */
	else
		*REG(TRIAD_BCOPY_SOURCE_ADD_REG) = (unsigned long)SrcAdd;

	*REG(TRIAD_BCOPY_DEST_ADD_REG) = (unsigned long)DestAdd;	/* set destination address */
	*REG(TRIAD_BCOPY_BYTE_COUNT_REG) = len;	/* Length of buffer */
	*REG(TRIAD_BCOPY_MODE_CNTL_REG) = mode;	/* Start and Mode */

	/* Synchronous copy */
	do
	{
		NanoDelay(500);
		status = *REG(TRIAD_BCOPY_STATUS_REG);
	} while (!(status & BCOPY_STATUS_TRANSFER_COMPLETE));

	invalidate_buffer((void *) DestAdd, len);

	*REG(TRIAD_BCOPY_MODE_CNTL_REG) = 0;		/* Stop Bcopy */
	
	if (status & BCOPY_STATUS_TRANSFER_ERROR)
	{
		/* Clear the error condition */
		*REG(TRIAD_BCOPY_STATUS_REG) = BCOPY_STATUS_CLEAR_ERROR;
		restore_flags(flags);
		return -EFAULT;
	}
	restore_flags(flags);	
	return 0;
}

		

/***
 * msp_bcopy_open
 *
 * prepare to do operations with the security engine
*/

static int msp_bcopy_open(struct inode *inode, struct file *file)
{
	DBG_CPY("driver open\n") ;
	return 0 ;
}

/***
 * msp_bcopy_close
 *
 * finish operations with the security engine
 */

static int msp_bcopy_close(struct inode *inode, struct file *file)
{
	DBG_CPY("driver close\n") ;
	return 0 ;
}


/***
 * msp_bcopy_ioctl
 *
 * do something with the copy engine
 */

static int msp_bcopy_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	cpy_ioctl	cmdbuf;
	int result;

	DBG_CPY("driver ioctl\n");

	if (cmd != BRECIS_COPY_CTL)
	{
		DBG_CPY("bad ioctl\n");
		return -EINVAL;
	}

	if ( (copy_from_user(&cmdbuf, (void *)arg, sizeof(cmdbuf)))
		|| !access_ok(VERIFY_READ, cmdbuf.src,
			(cmdbuf.mode & BCOPY_MODE_SOURCE_INC_MODE) ? cmdbuf.length : 4)
		|| !access_ok(VERIFY_WRITE, cmdbuf.dest,
			(cmdbuf.mode & BCOPY_MODE_DEST_INC_MODE) ? cmdbuf.length : 4) )
	{
		DBG_CPY("user fault\n");
		return -EFAULT;
	}

	DBG_CPY("Calling copy operation\n");

	result = dma_bcopy(cmdbuf.src, cmdbuf.dest, cmdbuf.mode, cmdbuf.length);

	/* XXX get any failure info */
	
	DBG_CPY("returned %d from copy operation\n", result) ;

	return result;
}


/***
 * various structures dealing with registering the driver with
 * the kernel.
 */

static struct file_operations msp_bcopy_fops = {
	owner:    THIS_MODULE,
	open:     msp_bcopy_open,
	release:  msp_bcopy_close,
	ioctl:    msp_bcopy_ioctl,
} ;

static struct miscdevice msp_bcopy_miscdev = {
	minor:        141,
	name:         "msp_bcopy",
	fops:         &msp_bcopy_fops,
};
	

/***
 * msp_bcopy_init
 *
 * Initialize the hardware, and install the driver.
 */

static int __init msp_bcopy_init(void)
{
	misc_register(&msp_bcopy_miscdev);

//	msp_bcopy_proc_register();

	printk(KERN_INFO "Copy engine driver installed\n");

	return 0;
}

static void __exit msp_bcopy_exit(void)
{
	printk(KERN_INFO "Copy engine driver removed\n");

	misc_deregister(&msp_bcopy_miscdev);

//	msp_bcopy_proc_unregister();
}

#if 0
static int
proc_calc_metrics(char *page, char **start, off_t off,
		  int count, int *eof, int len)
{
	if (len <= off + count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len > count) len = count;
	if (len < 0) len = 0;
	return len;
}

static int msp_bcopy_read_proc(char *page, char **start, off_t off,
			     int count, int *eof, void *data)
{
	int len;

	len = sprintf(page, "mknod /dev/secdev c %d %d\n",
		      MISC_MAJOR,
		      msp_bcopy_miscdev.minor);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int msp_bcopy_proc_register(void)
{
	proc_entry = create_proc_read_entry("brecis_sec", 0, NULL,
					    msp_bcopy_read_proc, NULL);
	return proc_entry != 0;
}

static void msp_bcopy_proc_unregister(void)
{
	if (!proc_entry) return;

	remove_proc_entry(proc_entry->name, proc_entry->parent);
	proc_entry = 0;
}
#endif

module_init(msp_bcopy_init);
module_exit(msp_bcopy_exit);

EXPORT_SYMBOL(dma_bcopy);

/*	End of dma_bcopy.c	*/
