/*
 * Kevin D. Kissell, kevink@mips.com and Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * This program is free software; you can distribute it and/or modify it
 * under the terms of the GNU General Public License (Version 2) as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * MIPS32 CPU variant specific MMU/Cache routines.
 */
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>

#include <asm/bootinfo.h>
#include <asm/cpu.h>
#include <asm/bcache.h>
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/system.h>
#include <asm/mmu_context.h>
#include <asm/brecis/BrecisMemCtl.h>

/* #define REDUCECACHE */
#ifdef REDUCECACHE
/* 
 * To reduce the cache used, define the symbol REDUCECACHE
 * and set I_CACHE_SIZE and D_CACHE_SIZE below to the desired 
 * amount of caches to use for the instruction and data cache respectively.
 */
#define I_CACHE_SIZE		8192
#define D_CACHE_SIZE		8192
#define NUM_CACHE_LINES		1024
#define CACHE_LINE_SIZE		16
#define CACHE_ADDRESS_MASK	( ~(CACHE_LINE_SIZE - 1) )
#define REAL_CACHE_SIZE		(NUM_CACHE_LINES * CACHE_LINE_SIZE)
/* 
 * insure tmpmemory is large enough so we can start locking on a cache
 * line boundary and completely fill the cache
 */
unsigned char tmpmemory[REAL_CACHE_SIZE+CACHE_LINE_SIZE];
#endif

/* CP0 hazard avoidance. */
#define BARRIER __asm__ __volatile__(".set noreorder\n\t" \
				     "nop; nop; nop; nop; nop; nop;\n\t" \
				     ".set reorder\n\t")

/* Primary cache parameters. */
unsigned int icache_size, dcache_size; /* Size in bytes */
unsigned int dcache_threshold;
#if	!(defined(CONFIG_CPU_MIPS32))
unsigned int ic_lsize, dc_lsize;       /* LineSize in bytes */

/* Secondary cache (if present) parameters. */
unsigned int scache_size, sc_lsize;	/* Again, in bytes */
#endif	/* !(CONFIG_CPU_MIPS32) */

#include <asm/cacheops.h>
#include <asm/mips32_cache.h>

#undef DEBUG_CACHE

/*
 * Dummy cache handling routines for machines without boardcaches
 */
//static void no_sc_noop(void) {}

//static struct bcache_ops no_sc_ops = {
//	(void *)no_sc_noop, (void *)no_sc_noop,
//	(void *)no_sc_noop, (void *)no_sc_noop
//};

//struct bcache_ops *bcops = &no_sc_ops;


/*
 * Zero an entire page.
 */

void mips32_clear_page_dc(unsigned long page)
{
	unsigned long i;

	for (i=page; i<page+PAGE_SIZE; i+=4)
	        *(unsigned long *)(i) = 0;
}

void mips32_copy_page_dc(unsigned long to, unsigned long from)
{
	unsigned long i;

	for (i=0; i<PAGE_SIZE; i+=4)
	        *(unsigned long *)(to+i) = *(unsigned long *)(from+i);
}


void mips32_flush_cache_all_pc(void)
{
	wmb();
	blast_dcache(); blast_icache();
}


void mips32_flush_cache_range_pc(struct mm_struct *mm,
				     unsigned long start,
				     unsigned long end)
{
	wmb();
	blast_hit_idcache(start, end - start);
}

void mips32_dcache_inval_range(void *addr, size_t size)
{
	blast_hit_dcache((unsigned long)addr, size);
}

/*
 * On architectures like the Sparc, we could get rid of lines in
 * the cache created only by a certain context, but on the MIPS
 * (and actually certain Sparc's) we cannot.
 */

void mips32_flush_cache_mm_pc(struct mm_struct *mm)
{
	mips32_flush_cache_all_pc();
}


void mips32_flush_cache_page_pc(struct vm_area_struct *vma,
				    unsigned long page)
{
#if 0 // Fixme if you can ...MaTed---
	//...MaTed--- Is page valid for call
	page = (KSEG0 + (page & (scache_size - 1)));
	blast_dcache_page_indexed(page);
#else  // 0
	wmb();
	blast_dcache_page(page);
#endif // 0
}

/* If the addresses passed to these routines are valid, they are
 * either:
 *
 * 1) In KSEG0, so we can do a direct flush of the page.
 * 2) In KSEG2, and since every process can translate those
 *    addresses all the time in kernel mode we can do a direct
 *    flush.
 * 3) In KSEG1, no flush necessary.
 */

void mips32_flush_page_to_ram_pc(struct page *page)
{
	wmb();
//	blast_dcache_page((unsigned long)page_address(page));
}

#if 1
void
mips32_flush_icache_range(unsigned long start, unsigned long end)
{
	flush_cache_all();
}
#endif	/* 0 */

void
mips32_flush_icache_page(struct vm_area_struct *vma, struct page *page)
{
	int address;

	if (!(vma->vm_flags & VM_EXEC))
		return;

	address = KSEG0 + ((unsigned long)page_address(page) & PAGE_MASK & (dcache_size - 1));
	blast_icache_page_indexed(address);
}

#ifdef CONFIG_NONCOHERENT_IO
/*
 * Writeback and invalidate the primary cache dcache before DMA.
 */
void
mips32_dma_cache_wback_inv_pc(unsigned long addr, unsigned long size)
{
	wmb();
	if (size >= dcache_threshold) {
		blast_dcache();
	} else {
		blast_hit_dcache(addr, size);
	}
	bc_wback_inv(addr, size);
}


void
mips32_dma_cache_inv_pc(unsigned long addr, unsigned long size)
{
	wmb();
	if (size >= dcache_threshold) {
		blast_dcache();
	} else {
		blast_hit_dcache(addr, size);
	}

	bc_inv(addr, size);
}


void
mips32_dma_cache_wback(unsigned long addr, unsigned long size)
{
	panic("mips32_dma_cache called - should not happen.\n");
}
#endif /* CONFIG_NONCOHERENT_IO */

/*
 * While we're protected against bad userland addresses we don't care
 * very much about what happens in that case.  Usually a segmentation
 * fault will dump the process later on anyway ...
 */
void mips32_flush_cache_sigtramp(unsigned long addr)
{
	unsigned long iaddr;

	__sync();	/* Write write buffers to memory */
	iaddr = addr & ~(ICACHE_LINESIZE - 1);
	protected_flush_icache_line(iaddr);
	protected_flush_icache_line(iaddr + ICACHE_LINESIZE);
}

#undef DEBUG_TLB
#undef DEBUG_TLBUPDATE

void local_flush_tlb_all(void)
{
	unsigned long flags;
	unsigned long old_ctx;
	int entry;

#ifdef DEBUG_TLB
	printk("[tlball]");
#endif

	__save_and_cli(flags);
	/* Save old context and create impossible VPN2 value */
	old_ctx = (get_entryhi() & 0xff);
	set_entryhi(KSEG0);
	set_entrylo0(0);
	set_entrylo1(0);
	BARRIER;

	entry = get_wired();

	/* Blast 'em all away. */
	while(entry < mips_cpu.tlbsize) {
	        /* Make sure all entries differ. */  
	        set_entryhi(KSEG0+entry*0x2000);
		set_index(entry);
		BARRIER;
		tlb_write_indexed();
		BARRIER;
		entry++;
	}
	BARRIER;
	set_entryhi(old_ctx);
	__restore_flags(flags);
}

void local_flush_tlb_mm(struct mm_struct *mm)
{
}

void local_flush_tlb_page(struct vm_area_struct *vma, unsigned long page)
{
}

#if 0
void show_regs(struct pt_regs * regs)
{
	/* Saved main processor registers. */
	printk("$0 : %08lx %08lx %08lx %08lx\n",
	       0UL, regs->regs[1], regs->regs[2], regs->regs[3]);
	printk("$4 : %08lx %08lx %08lx %08lx\n",
               regs->regs[4], regs->regs[5], regs->regs[6], regs->regs[7]);
	printk("$8 : %08lx %08lx %08lx %08lx\n",
	       regs->regs[8], regs->regs[9], regs->regs[10], regs->regs[11]);
	printk("$12: %08lx %08lx %08lx %08lx\n",
               regs->regs[12], regs->regs[13], regs->regs[14], regs->regs[15]);
	printk("$16: %08lx %08lx %08lx %08lx\n",
	       regs->regs[16], regs->regs[17], regs->regs[18], regs->regs[19]);
	printk("$20: %08lx %08lx %08lx %08lx\n",
               regs->regs[20], regs->regs[21], regs->regs[22], regs->regs[23]);
	printk("$24: %08lx %08lx\n",
	       regs->regs[24], regs->regs[25]);
	printk("$28: %08lx %08lx %08lx %08lx\n",
	       regs->regs[28], regs->regs[29], regs->regs[30], regs->regs[31]);

	/* Saved cp0 registers. */
	printk("epc   : %08lx\nStatus: %08lx\nCause : %08lx\n",
	       regs->cp0_epc, regs->cp0_status, regs->cp0_cause);
}
#endif
			
void add_wired_entry(unsigned long entrylo0, unsigned long entrylo1,
				      unsigned long entryhi, unsigned long pagemask)
{
        unsigned long flags;
        unsigned long wired;
        unsigned long old_pagemask;
        unsigned long old_ctx;

        __save_and_cli(flags);
        /* Save old context and create impossible VPN2 value */
        old_ctx = (get_entryhi() & 0xff);
        old_pagemask = get_pagemask();
        wired = get_wired();
        set_wired (wired + 1);
        set_index (wired);
        BARRIER;    
        set_pagemask (pagemask);
        set_entryhi(entryhi);
        set_entrylo0(entrylo0);
        set_entrylo1(entrylo1);
        BARRIER;    
        tlb_write_indexed();
        BARRIER;    
    
        set_entryhi(old_ctx);
        BARRIER;    
        set_pagemask (old_pagemask);
        flush_tlb_all();    
        __restore_flags(flags);
}

/* Detect and size the various caches. */
static void __init probe_icache(unsigned long config)
{
        unsigned long config1;
	unsigned int lsize;

        if (!(config & (1 << 31))) {
#if	!(defined(CONFIG_CPU_MIPS32))
	        /* 
		 * Not a MIPS32 compliant CPU. 
		 * Config 1 register not supported, we assume R4k style.
		 */
	        icache_size = 1 << (12 + ((config >> 9) & 7));
		ic_lsize = 16 << ((config >> 5) & 1);
		mips_cpu.icache.linesz = ic_lsize;
		
		/* 
		 * We cannot infer associativity - assume direct map
		 * unless probe template indicates otherwise
		 */
		if(!mips_cpu.icache.ways) mips_cpu.icache.ways = 1;
		mips_cpu.icache.sets = 
			(icache_size / ic_lsize) / mips_cpu.icache.ways;
#else
		/* Not a mips32 cpu, panic */
		panic("Not a MIPS32 cpu");
#endif	/* ! CONFIG_CPU_MIPS32 */
	} else {
	       config1 = read_mips32_cp0_config1(); 

	       if ((lsize = ((config1 >> 19) & 7)))
		       mips_cpu.icache.linesz = 2 << lsize;
	       else 
		       mips_cpu.icache.linesz = lsize;
	       mips_cpu.icache.sets = 64 << ((config1 >> 22) & 7);
	       mips_cpu.icache.ways = 1 + ((config1 >> 16) & 7);

	       icache_size = mips_cpu.icache.sets * mips_cpu.icache.ways * 
		             mips_cpu.icache.linesz;
		if (ICACHE_LINESIZE != mips_cpu.icache.linesz)
			panic("I Cache line size error");
	}
	pkinit("Primary instruction cache %dkb, linesize %d bytes (%d ways)\n",
	       icache_size >> 10, mips_cpu.icache.linesz, mips_cpu.icache.ways);
}

static void __init probe_dcache(unsigned long config)
{
        unsigned long config1;
	unsigned int lsize;

        if (!(config & (1 << 31))) {
#if	!(defined(CONFIG_CPU_MIPS32))
	        /* 
		 * Not a MIPS32 compliant CPU. 
		 * Config 1 register not supported, we assume R4k style.
		 */  
		dcache_size = 1 << (12 + ((config >> 6) & 7));
		dc_lsize = 16 << ((config >> 4) & 1);
		mips_cpu.dcache.linesz = dc_lsize;
		/* 
		 * We cannot infer associativity - assume direct map
		 * unless probe template indicates otherwise
		 */
		if(!mips_cpu.dcache.ways) mips_cpu.dcache.ways = 1;
		mips_cpu.dcache.sets = 
			(dcache_size / dc_lsize) / mips_cpu.dcache.ways;
#else
		panic("Not a MIPS32 cpu");
#endif	/* ! (CONFIG_CPU_MIPS32) */
	} else {
	        config1 = read_mips32_cp0_config1();

		if ((lsize = ((config1 >> 10) & 7)))
		        mips_cpu.dcache.linesz = 2 << lsize;
		else 
		        mips_cpu.dcache.linesz = lsize;
		mips_cpu.dcache.sets = 64 << ((config1 >> 13) & 7);
		mips_cpu.dcache.ways = 1 + ((config1 >> 7) & 7);

		dcache_size = 
			mips_cpu.dcache.sets * mips_cpu.dcache.ways
			* mips_cpu.dcache.linesz;
		if (DCACHE_LINESIZE != mips_cpu.dcache.linesz)
			panic("D Cache line size error");
	}
	dcache_threshold = 4 * dcache_size;
	pkinit("Primary data cache %dkb, linesize %d bytes (%d ways)\n",
	       dcache_size >> 10, mips_cpu.dcache.linesz, mips_cpu.dcache.ways);
}


static void __init setup_noscache_funcs(void)
{
	_clear_page = (void *)mips32_clear_page_dc;
	_copy_page = (void *)mips32_copy_page_dc;
#ifndef	CONFIG_BRECIS
	_flush_cache_all = mips32_flush_cache_all_pc;
	___flush_cache_all = mips32_flush_cache_all_pc;
	_flush_cache_mm = mips32_flush_cache_mm_pc;
	_flush_cache_range = mips32_flush_cache_range_pc;
	_flush_cache_page = mips32_flush_cache_page_pc;
	_flush_page_to_ram = mips32_flush_page_to_ram_pc;

	_flush_icache_page = mips32_flush_icache_page;
#endif	/* ! CONFIG_BRECIS */

#if defined(CONFIG_NONCOHERENT_IO) && ! defined(CONFIG_BRECIS)
	_dma_cache_wback_inv = mips32_dma_cache_wback_inv_pc;
	_dma_cache_wback = mips32_dma_cache_wback;
	_dma_cache_inv = mips32_dma_cache_inv_pc;
#endif	/* CONFIG_NONCOHERENT_IO && ! CONFIG_BRECIS */
}


static void __init probe_tlb(unsigned long config)
{
        unsigned long config1;

        if (!(config & (1 << 31))) {
	        /* 
		 * Not a MIPS32 compliant CPU. 
		 * Config 1 register not supported, we assume R4k style.
		 */  
	        mips_cpu.tlbsize = 48;
	} else {
	        config1 = read_mips32_cp0_config1();
		if (!((config >> 7) & 3))
		        panic("No MMU present");
		else
		        mips_cpu.tlbsize = ((config1 >> 25) & 0x3f) + 1;
	}	

	pkinit("Number of TLB entries %d.\n", mips_cpu.tlbsize);
}

void __init ld_mmu_mips32(void)
{
	unsigned long config = read_32bit_cp0_register(CP0_CONFIG);

	pkinit("CPU revision is: %08x\n", read_32bit_cp0_register(CP0_PRID));

#ifdef CONFIG_MIPS_UNCACHED
	change_cp0_config(CONF_CM_CMASK, CONF_CM_UNCACHED);
#else
	change_cp0_config(CONF_CM_CMASK, CONF_CM_CACHABLE_NONCOHERENT);
#endif

	probe_icache(config);
	probe_dcache(config);
	setup_noscache_funcs();
	probe_tlb(config);

#ifndef	CONFIG_BRECIS
	_flush_cache_sigtramp = mips32_flush_cache_sigtramp;
	_flush_icache_range = mips32_flush_icache_range;	/* Ouch */
#endif	/* ! CONFIG_BRECIS */

	__flush_cache_all();
	write_32bit_cp0_register(CP0_WIRED, 0);

	/*
	 * You should never change this register:
	 *   - The entire mm handling assumes the c0_pagemask register to
	 *     be set for 4kb pages.
	 */
	write_32bit_cp0_register(CP0_PAGEMASK, PM_4K);
	flush_tlb_all();
}


/*
 * MIPS32 4km - no mmu.
 */

void __init ld_mmu_mips32_4km(void)
{
	unsigned long config = read_32bit_cp0_register(CP0_CONFIG);

	pkinit("CPU revision is: %08x\n", read_32bit_cp0_register(CP0_PRID));

#ifdef CONFIG_MIPS_UNCACHED
	change_cp0_config(CONF_CM_CMASK, CONF_CM_UNCACHED);
#else
	change_cp0_config(CONF_CM_CMASK, CONF_CM_CACHABLE_NONCOHERENT);
#endif

	probe_icache(config);
	probe_dcache(config);

#ifdef REDUCECACHE
	/* 
	 * Lock the entire instruction and data caches.
	 * A blast_icache() and a blast_dcache() will unlock and
	 * clear the amount of cache we want to use for testing
	 */
	{
		unsigned long start;
		int count;

		start = ((unsigned long) 
			 &tmpmemory[CACHE_LINE_SIZE-1]) & CACHE_ADDRESS_MASK;

		printk(KERN_INFO "Locking caches using start address %lx\n",
		       start);

		for (count = 0; count < NUM_CACHE_LINES; count++)
		{
			__asm__ __volatile__(
				".set noreorder\n\t"
				".set mips3\n\t"
				"cache %1, (%0)\n\t"
				".set mips0\n\t"
				".set reorder"
				:
				: "r" (start),
				"i" (Fetch_Lock_I));
			
			__asm__ __volatile__(
				".set noreorder\n\t"
				".set mips3\n\t"
				"cache %1, (%0)\n\t"
				".set mips0\n\t"
				".set reorder"
				:
				: "r" (start),
				"i" (Fetch_Lock_D));

			start += CACHE_LINE_SIZE;
		}
	}
	
	/* 
	 * adjust instruction and data cache sizes 
	 * blast_icache() and blast_dcache() will clear 
	 */
	printk(KERN_INFO 
	       "***** Setting I-cache size to %d, D-cache size to %d *****\n", 
	       I_CACHE_SIZE, D_CACHE_SIZE);
	icache_size = I_CACHE_SIZE;
	dcache_size = D_CACHE_SIZE;
	dcache_threshold = 4 * dcache_size;
#endif

	setup_noscache_funcs();

#ifndef	CONFIG_BRECIS
	_flush_cache_sigtramp = mips32_flush_cache_sigtramp;
	_flush_icache_range = mips32_flush_icache_range;	/* Ouch */
#endif	/* ! CONFIG_BRECIS */

	__flush_cache_all();
}
