/* #define DEBUG_IRQ */

/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * Routines for generic manipulation of the interrupts found on the Brecis 
 * MSP5000 board.
 *
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/kernel_stat.h>
#include <linux/random.h>

#include <asm/irq.h>
#include <asm/brecis/BrecisSysRegs.h>
#include <asm/brecis/brecisint.h>
#include <asm/mipsregs.h>
#include <asm/gdb-stub.h>

/*
** All MSP interrupts come into the cpu via the SLM interrupt Status /
** Mask Registers, and the SLM mulitplexes these together onto the
** MIPS I4 hardware interrupt line (which is also known as IRQ 6, bit
** 0x4000 in the MIPS CAUSE and STATUS registers).  A select few of
** the interrupts can be "promoted" to go directly through to the
** other IRQs in the CAUSE/STATUS registers.
**
** In reality, such promotion would save very little time in the
** interrupt processing, and considering that linux doesn't really
** support nested interrupts, would do next to nothing for interrupt
** latency.
**
** So, in the current structure, the only interrupt that doesn't go
** through the SLM irq is the timer interrupt, which is separated out
** in mipsIRQ.S.
**
** IRQ numbers within this system represent the bit position within
** the SLM interrupt status & mask registers.
**
** Here's a list of the more interesting interrupts, taken from
** the MSP5000 Hardware User's manual.
**
**   0 through 7         External interrupt pins 0 through 7
**   8                   Voice Engine
**   9                   Packet Engine
**   10                  Ethernet 0
**   11                  Ethernet 1
**   12                  Security Engine
**   13                  Peripheral Engine
**   14                  Timer 0
**   15                  Timer 1
**   16                  Timer 2
**   17                  MIPS internal Timer
**   18                  Block Copy Engine
**   19                  UART
*/

struct brecis_ictrl_regs *brecis_hw0_icregs
	= (struct brecis_ictrl_regs *)INT_STA_REG;

extern asmlinkage void mipsIRQ(void);
extern void do_IRQ(int irq, struct pt_regs *regs);

unsigned long spurious_count = 0;


/*
 * Generic no controller code
 */

static void enable_none(unsigned int irq) { }
static unsigned int startup_none(unsigned int irq) { return 0; }
static void disable_none(unsigned int irq) { }
static void ack_none(unsigned int irq)
{
	/*
	 * 'what should we do if we get a hw irq event on an illegal vector'.
	 * each architecture has to answer this themselves, it doesnt deserve
	 * a generic callback i think.
	 */
	printk("unexpected interrupt %d\n", irq);
}

/* startup is the same as "enable", shutdown is same as "disable" */
#define shutdown_none	disable_none
#define end_none	enable_none

struct hw_interrupt_type no_irq_type = {
	"none",
	startup_none,
	shutdown_none,
	enable_none,
	disable_none,
	ack_none,
	end_none
};

irq_desc_t irq_desc[NR_IRQS] __cacheline_aligned =
	{ [0 ... NR_IRQS-1] = { 0, &no_irq_type, NULL, 0, SPIN_LOCK_UNLOCKED}};

#ifdef DEBUG_IRQ
#define DEBUG_INT(x...) printk(x)
#else
#define DEBUG_INT(x...)
#endif

void disable_brecis_irq(unsigned int irq_nr)
{

	brecis_hw0_icregs->intenable &= ~(1 << irq_nr);
	set_cp0_status(IE_IRQ4) ;
}

void disable_irq_nosync(unsigned int irq_nr)
{
	disable_brecis_irq(irq_nr);
}

void enable_brecis_irq(unsigned int irq_nr)
{
	brecis_hw0_icregs->intenable |= (1 << irq_nr);
}

void enable_irq(unsigned int irq_nr)
{
	enable_brecis_irq(irq_nr);
}

static void inline ack_brecis_irq(unsigned int irq)
{
	/* clear level sensitive interrupt by writing a 1 */
	brecis_hw0_icregs->intstatus = 1 << irq ;
}

int get_irq_list(char *buf)
{
	int i, len = 0;
	int num = 0;
	struct irqaction *action;

	for (i = 0; i < BRECISINT_END; i++, num++) {
		action = irq_desc[i].action;
		if (!action) 
			continue;
		len += sprintf(buf+len, "%2d: %8d %c %s",
			num, kstat.irqs[0][num],
			(action->flags & SA_INTERRUPT) ? '+' : ' ',
			action->name);
		for (action=action->next; action; action = action->next) {
			len += sprintf(buf+len, ",%s %s",
				(action->flags & SA_INTERRUPT) ? " +" : "",
				action->name);
		}
		len += sprintf(buf+len, " [hw0]\n");
	}
	return len;
}

int request_irq(unsigned int irq, 
		void (*handler)(int, void *, struct pt_regs *),
		unsigned long irqflags, 
		const char *devname,
		void *dev_id)
{  
        struct irqaction *action;

	DEBUG_INT("request_irq: irq=%d, devname = %s\n", irq, devname);

	if (irq >= BRECISINT_END || !handler)
		return -EINVAL;

//	if (((irqflags & SA_SHIRQ) == 0 && irq_desc[irq].action != 0) ||
	action = irq_desc[irq].action;
	if (action != 0 && ((irqflags & SA_SHIRQ) == 0 ||
		(irqflags & action->flags & SA_SHIRQ) == 0))
	{
		printk("Unable to share irq %d\n", irq);
		return -EBUSY;
	}

	action = (struct irqaction *)kmalloc(sizeof(struct irqaction), GFP_KERNEL);
	if (!action)
		return -ENOMEM;

	action->handler = handler;
	action->flags = irqflags;
	action->mask = 0;
	action->name = devname;
	action->dev_id = dev_id;
	action->next = irq_desc[irq].action;
	irq_desc[irq].action = action;
	enable_brecis_irq(irq);

	if(irqflags & SA_SAMPLE_RANDOM)
		rand_initialize_irq(irq);

	return 0;
}

void free_irq(unsigned int irq, void *dev_id)
{
	struct irqaction	*action;
	struct irqaction	**last;

	if (irq >= BRECISINT_END) {
		printk("Trying to free IRQ%d\n",irq);
		return;
	}

	last = &irq_desc[irq].action;
	action = *last;
	while (action != 0 && action->dev_id != dev_id)
		last = &action->next, action = action->next;
	if (action == 0) {
		printk("Failed to find IRQ%d for device %p\n",
			irq, dev_id);
		return;
	}

	*last = action->next;		
	disable_brecis_irq(irq);
	kfree(action);
}

static inline int ls1bit32(unsigned int x)
{
	int b;
	__asm__(".set\tpush\n\t"
		".set\tmips32\n\t"
		"clz\t%0,%1\n\t"
		".set\tpop"
		:"=r" (b)
		: "r" (x));
	return 31-b;
}

void brecis_hw0_irqdispatch(struct pt_regs *regs)
{
	struct irqaction *action;
	unsigned long current_ints ;
	int irq, cpu = smp_processor_id();

	current_ints = ( brecis_hw0_icregs->intstatus
			 & brecis_hw0_icregs->intenable ) ; 

	/* if int_status == 0, then the interrupt has already been cleared */

	while (current_ints != 0)
	{
		irq = ls1bit32(current_ints);
		action = irq_desc[irq].action;

		current_ints &= ~(1<<irq);

		DEBUG_INT("brecis_hw0_irqdispatch: irq=%d\n", irq);

		/* if action == NULL, then no have a handler for the irq */
		if ( action == NULL ) {
			printk("No handler for hw0 irq: %i\n", irq);
			spurious_count++;
			ack_brecis_irq(irq);
			continue;
		}

		ack_brecis_irq(irq) ;
	
		irq_enter(cpu, irq);
		kstat.irqs[0][irq]++;
		do {
			action->handler(irq, action->dev_id, regs);
			if (action->flags & SA_SAMPLE_RANDOM)
				add_interrupt_randomness(irq);
			action = action->next;
		} while (action);
		irq_exit(cpu, irq);
	}

	if (softirq_pending(cpu))
		do_softirq();
}

unsigned long probe_irq_on (void)
{
	return 0;
}


int probe_irq_off (unsigned long irqs)
{
	return 0;
}

#ifdef CONFIG_REMOTE_DEBUG
extern void breakpoint(void);
extern int remote_debug;
#endif

void __init init_IRQ(void)
{
	int i;

	/* 
	 * Mask out all interrupt by writing "0" to all bit position in 
	 * the interrupt enable reg. 
	 */
	brecis_hw0_icregs->intenable = 0x0;    

	/* Now safe to set the exception vector. */
	set_except_vector(0, mipsIRQ);

	for (i = 0; i <= BRECISINT_END; i++) {
		irq_desc[i].status	= IRQ_DISABLED;
		irq_desc[i].action	= 0;
		irq_desc[i].depth	= 1;
	}

	DEBUG_INT("init_IRQ: BRECISINTEND=%d\n", BRECISINT_END);

	/* enable SLM interrupt in CP0 status */
	write_32bit_cp0_register( CP0_STATUS,
				  read_32bit_cp0_register(CP0_STATUS)
				  | STATUSF_IP6 ) ;

#ifdef CONFIG_REMOTE_DEBUG
	if (remote_debug) {
		set_debug_traps();
		breakpoint();
	}
#endif
}
