/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * PROM library functions for acquiring/using memory descriptors given to 
 * us from the YAMON.
 * 
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/bootmem.h>

#include <asm/bootinfo.h>
#include <asm/page.h>
#include <linux/slab.h>
#include <asm/brecis/prom.h>

/* #define DEBUG */

enum yamon_memtypes {
	yamon_dontuse,
	yamon_prom,
	yamon_free,
};
struct prom_pmemblock mdesc[PROM_MAX_PMEMBLOCKS];

#ifdef DEBUG
static char *mtypes[3] = {
	"Dont use memory",
	"YAMON PROM memory",
	"Free memmory",
};
#endif

/* References to section boundaries */
extern char _ftext;
extern char _end;

#define PFN_ALIGN(x)    (((unsigned long)(x) + (PAGE_SIZE - 1)) & PAGE_MASK)


struct prom_pmemblock * __init prom_getmdesc(void)
{
	char *memsize_str;
	char *heaptop_str;
	unsigned int memsize;
	unsigned int heaptop;
	int i;

	memsize_str = prom_getenv("memsize");
	if (!memsize_str) {
		prom_printf("memsize not set in boot prom, set to default (32Mb)\n");
		memsize = 0x02000000;
	} else {
#ifdef DEBUG
		prom_printf("prom_memsize = %s\n", memsize_str);
#endif
		memsize = simple_strtol(memsize_str, NULL, 0);

		if (memsize == 0) {
			/* if memsize is a bad size, use reasonable default */
			memsize = 0x02000000;
		}
		/* convert to physical address (removing caching bits, etc) */
		memsize = PHYSADDR(memsize);
	}

	heaptop_str = prom_getenv("heaptop");
	if (!heaptop_str) {
		heaptop = PHYSADDR(&_ftext);
		prom_printf("heaptop not set in boot prom, set to default x()\n",
			    heaptop);
	} else {
#ifdef DEBUG
		prom_printf("prom_heaptop = %s\n", heaptop_str);
#endif
		heaptop = simple_strtol(heaptop_str, NULL, 16);
		if (heaptop == 0) {
			/* heaptop conversion bad, might have 0xValue */
			heaptop = simple_strtol(heaptop_str, NULL, 0);

			if (heaptop == 0) {
				/* heaptop still bad, use reasonable default */
				heaptop = PHYSADDR(&_ftext);
			}
		}

		/* convert to physical address (removing caching bits, etc) */
		heaptop = PHYSADDR(heaptop);
	}

	memset(mdesc, 0, sizeof(mdesc));

	i = 0;
	mdesc[i].type = yamon_dontuse;
	mdesc[i].base = 0x00000000;
	mdesc[i].size = PFN_ALIGN(0x300 + 0x80); /* jtag interrupt vector + sizeof vector */

	if (heaptop > mdesc[i].base + mdesc[i].size) {
		i++;		/* 1 */
		mdesc[i].type = yamon_prom;
		mdesc[i].base = mdesc[i-1].base + mdesc[i-1].size;
		mdesc[i].size = heaptop - mdesc[i].base;
	}

	if (heaptop != PHYSADDR(&_ftext)) {
		i++;	/* 2 */
		mdesc[i].type = yamon_free;
		mdesc[i].base = heaptop;
		mdesc[i].size = PHYSADDR(&_ftext) - mdesc[i].base;
	}

	i++;		/* 3 */
	mdesc[i].type = yamon_dontuse;
	mdesc[i].base = PHYSADDR(&_ftext);
	mdesc[i].size = PHYSADDR(PFN_ALIGN(&_end)) - mdesc[i].base;

	i++;		/* 4 */
	mdesc[i].type = yamon_free;
	mdesc[i].base = PHYSADDR(PFN_ALIGN(&_end));
	mdesc[i].size = memsize - mdesc[i].base;

	return &mdesc[0];
}

static int __init prom_memtype_classify (unsigned int type)
{
	switch (type) {
	case yamon_free:
		return BOOT_MEM_RAM;
	case yamon_prom:
		return BOOT_MEM_ROM_DATA;
	default:
		return BOOT_MEM_RESERVED;
	}
}

void __init prom_meminit(void)
{
	struct prom_pmemblock *p;

#ifdef DEBUG
	prom_printf("YAMON MEMORY DESCRIPTOR dump:\n");
	p = prom_getmdesc();
	while (p->size) {
		int i = 0;
		prom_printf("[%d,%p]: base<%08lx> size<%08lx> type<%s>\n",
			    i, p, p->base, p->size, mtypes[p->type]);
		p++;
		i++;
	}
#endif
	p = prom_getmdesc();

	while (p->size) {
		long type;
		unsigned long base, size;

		type = prom_memtype_classify (p->type);
		base = p->base;
		size = p->size;

		add_memory_region(base, size, type);
                p++; 
	}
}

#ifdef CHECKOVERLAP     /* slab.c -- no overlapping kmalloc stuff */
extern void check_overlap_remove(void *, int, void *);
#endif	/* CHECKOVERLAP */

void __init
prom_free_prom_memory (void)
{
	int i;
	unsigned long freed = 0;
	unsigned long addr;
	int argc;
	char **argv;
	char **envp;
	char *ptr;
	int len = 0;

	/* preserve environment variables and command line from pmon/bbload */
	/* first preserve the command line */
	for (argc = 0; argc < prom_argc; argc++)
	{
		len += sizeof(char *);			/* length of pointer */
		len += strlen(prom_argv[argc]) + 1;	/* length of string */
	}
	len += sizeof(char *);		/* plus length of null pointer */

	argv = kmalloc(len, GFP_KERNEL);
	ptr = (char *) &argv[prom_argc+1];	/* strings follow array */

	for (argc = 0; argc < prom_argc; argc++)
	{
		argv[argc] = ptr;
		strcpy(ptr, prom_argv[argc]);
		ptr += strlen(prom_argv[argc]) + 1;
	}
	argv[prom_argc] = NULL;		/* end array with null pointer */
	prom_argv = argv;

	/* next preserve the environment variables */
	len = 0;
	i = 0;
	for (envp = prom_envp; *envp != NULL; envp++)
	{
		i++;		/* count number of environment variables */
		len += sizeof(char *);			/* length of pointer */
		len += strlen(*prom_envp);		/* length of string */
	}
	len += sizeof(char *);		/* plus length of null pointer */

	envp = kmalloc(len, GFP_KERNEL);
	ptr = (char *) &envp[i+1];

	for (argc = 0; argc < i; argc++)
	{
		envp[argc] = ptr;
		strcpy(ptr, prom_envp[argc]);
		ptr += strlen(prom_envp[argc]) + 1;
	}
	envp[i] = NULL;			/* end array with null pointer */
	prom_envp = envp;

	for (i = 0; i < boot_mem_map.nr_map; i++) {
		if (boot_mem_map.map[i].type != BOOT_MEM_ROM_DATA)
			continue;
#ifdef CHECKOVERLAP
/* Remove section added in mipsnommu/kernel/setup.c */
check_overlap_remove((void *)__va(boot_mem_map.map[i].addr), boot_mem_map.map[i].size, (void *)&prom_free_prom_memory);
#endif	/* CHECKOVERLAP */
		addr = boot_mem_map.map[i].addr;
		while (addr < boot_mem_map.map[i].addr
			      + boot_mem_map.map[i].size) {
			ClearPageReserved(virt_to_page(__va(addr)));
			set_page_count(virt_to_page(__va(addr)), 1);
			free_page((unsigned long) __va(addr));
			addr += PAGE_SIZE;
			freed += PAGE_SIZE;
		}
	}
	if (freed > 0) {
		printk("Freeing prom memory: %ldkb freed\n", freed >> 10);
	}
}
