/* $Id: init.c,v 1.26 2003/04/27 19:49:03 mrustad Exp $ */
/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 1999,2000 MIPS Technologies, Inc.  All rights reserved.
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * PROM library initialisation code.
 */
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/utsname.h>
#include <linux/proc_fs.h>

#include <asm/io.h>
#include <asm/brecis/prom.h>
#include <asm/brecis/BrecisSysRegs.h>
#include <asm/brecis/BrecisTriadMap.h>

int prom_argc;
char **prom_argv, **prom_envp;
int *prom_vec;

int init_debug = 0;

static char MSPDefaultFeatures[] = "EM";
static char MSP2000DefaultFeatures[] = "EMEMEMSDTDZD";
static char MSP2000REVBDefaultFeatures[] = "EMEMEMSD";
static char MSP5000DefaultFeatures[] = "EMEMSTTTZT";

#ifdef	CONFIG_BRECIS_PCI
void	msp_pci_init(void);
#endif	/* CONFIG_BRECIS_PCI */


char *prom_getenv(char *envname)
{
        /*
	 * Return a pointer to the given environment variable.
	 */

	t_env_var *env = (t_env_var *)prom_envp;
	int i;

	i = strlen(envname);

	while(env->var) {
		if('=' == env->var[i] && 
		   strncmp(envname, env->var, i) == 0) {
			return(&env->var[i+1]);
		}
		env++;
	}
	return(NULL);
}

static inline unsigned char str2hexnum(unsigned char c)
{
	if(c >= '0' && c <= '9')
		return c - '0';
	if(c >= 'a' && c <= 'f')
		return c - 'a' + 10;
	return 0; /* foo */
}

static inline int str2eaddr(unsigned char *ea, unsigned char *str)
{
	int index = 0;
	unsigned char num = 0;

	while (*str != '\0')
	{
		if((*str == '.') || (*str == ':'))
		{
			ea[index++] = num;
			num = 0;
			str++;
		}
		else
		{
			num = num << 4;
			num |= str2hexnum(*str++);
		}
	}

	if (index == 5)
	{
		ea[index++] = num;
		return 0;
	}
	else
	{
		return -1;
	}
}

static inline unsigned long str2hex(unsigned char *str)
{
	int value = 0;

	while (*str) {
		value = value << 4;
		value |= str2hexnum(*str++);
	}

	return value;
}
 
int get_ethernet_addr(char * ethaddr_name, char *ethernet_addr)
{
        char *ethaddr_str;

        ethaddr_str = prom_getenv( ethaddr_name);
	if (!ethaddr_str) {
	        printk("%s not set in boot prom\n", ethaddr_name);
		return -1;
	}
	if (str2eaddr(ethernet_addr, ethaddr_str) == -1)
	{
		printk("%s badly formatted-<%s>\n", ethaddr_name, ethaddr_str);
		return -1;
	}

	if (init_debug > 1) {
	        int i;
		printk("get_ethernet_addr: for %s ", ethaddr_name);
	        for (i=0; i<5; i++)
		        printk("%02x:", (unsigned char)*(ethernet_addr+i));
		printk("%02x\n", *(ethernet_addr+i));
	}

	return 0;
}

static char *getFeatures(void)
{
	char *feature = prom_getenv(FEATURES);

	if (feature == NULL)
	{
		if (strncmp(system_utsname.machine, MACHINE_TYPE_MSP2000REVB,
			    strlen(MACHINE_TYPE_MSP2000REVB)) == 0)
			feature = MSP2000REVBDefaultFeatures;
		else if (strncmp(system_utsname.machine, MACHINE_TYPE_MSP5000,
				 strlen(MACHINE_TYPE_MSP5000)) == 0)
			feature = MSP5000DefaultFeatures;
		else if (strncmp(system_utsname.machine, MACHINE_TYPE_DUET,
				 strlen(MACHINE_TYPE_DUET)) == 0)
			feature = MSP2000DefaultFeatures;
		else
			feature = MSPDefaultFeatures;
	}

	return feature;
}

static char testFeature(char c)
{
	char *feature = getFeatures();

	while (*feature)
	{
		if (*feature++ == c)
			return *feature;
		feature++;
	}

	return FEATURE_NOEXIST;
}

static unsigned long getdeviceid(void)
{
	char *deviceid = prom_getenv(DEVICEID);

	if (deviceid == NULL)
		return *DEV_ID_REG;
	else
		return str2hex(deviceid);
}

char identify_pci(void)
{
	return testFeature(PCI_KEY);
}

char identify_pcimux(void)
{
	return testFeature(PCIMUX_KEY);
}

char identify_sec(void)
{
	return testFeature(SEC_KEY);
}

char identify_spad(void)
{
	return testFeature(SPAD_KEY);
}

char identify_tdm(void)
{
	return testFeature(TDM_KEY);
}

char identify_zsp(void)
{
	return testFeature(ZSP_KEY);
}

static char identify_enetfeature(char key, unsigned long interfaceNum)
{
	char *feature = getFeatures();

	while (*feature)
	{
		if (*feature++ == key && interfaceNum-- == 0)
			return *feature;
		feature++;
	}

	return FEATURE_NOEXIST;
}

char identify_enet(unsigned long interfaceNum)
{
	return identify_enetfeature(ENET_KEY, interfaceNum);
}

char identify_enetTxD(unsigned long interfaceNum)
{
	return identify_enetfeature(ENETTXD_KEY, interfaceNum);
}

unsigned long identify_family()
{
	unsigned long deviceid;

	deviceid = getdeviceid();

	return deviceid & CPU_DEVID_FAMILY;
}

unsigned long identify_revision()
{
	unsigned long deviceid;

	deviceid = getdeviceid();

	return deviceid & CPU_DEVID_REVISION;
}

EXPORT_SYMBOL(get_ethernet_addr) ;
EXPORT_SYMBOL(identify_pci) ;
EXPORT_SYMBOL(identify_sec) ;
EXPORT_SYMBOL(identify_spad) ;
EXPORT_SYMBOL(identify_tdm) ;
EXPORT_SYMBOL(identify_zsp) ;
EXPORT_SYMBOL(identify_enet) ;
EXPORT_SYMBOL(identify_enetTxD) ;
EXPORT_SYMBOL(identify_family) ;
EXPORT_SYMBOL(identify_revision) ;

int __init prom_init(int argc, char **argv, char **envp, int *vec)
{
	unsigned long family;
	unsigned long revision;

	prom_argc = argc;
	prom_argv = argv;
	prom_envp = envp;
	prom_vec  = vec;

	family = identify_family();
	revision = identify_revision();

	switch (family)
	{
	case FAMILY_FPGA:
		if (FPGA_IS_ZEUS(revision))
			strncpy(system_utsname.machine,
				MACHINE_TYPE_ZEUS_FPGA,
				sizeof(system_utsname.machine));
		if (FPGA_IS_POLO(revision))
			strncpy(system_utsname.machine,
				MACHINE_TYPE_POLO_FPGA,
				sizeof(system_utsname.machine));
		else if (FPGA_IS_5000(revision))
			strncpy(system_utsname.machine,
				MACHINE_TYPE_MSP5000_FPGA,
				sizeof(system_utsname.machine));
		else if (FPGA_IS_DUET(revision))
			strncpy(system_utsname.machine,
				MACHINE_TYPE_DUET_FPGA,
				sizeof(system_utsname.machine));
		else
			strncpy(system_utsname.machine,
				MACHINE_TYPE_OTHER_FPGA,
				sizeof(system_utsname.machine));
		break;

	case FAMILY_ZEUS:
		strncpy(system_utsname.machine,
			MACHINE_TYPE_ZEUS,
			sizeof(system_utsname.machine));
		break;

	case FAMILY_POLO:
		strncpy(system_utsname.machine, 
			MACHINE_TYPE_POLO,
			sizeof(system_utsname.machine));
		break;

	case FAMILY_DUET:
		if (revision == 1)
			strncpy(system_utsname.machine, 
				MACHINE_TYPE_DUET, 
				sizeof(system_utsname.machine));
		else
			strncpy(system_utsname.machine, 
				MACHINE_TYPE_MSP2000REVB,
				sizeof(system_utsname.machine));
		break;

	case FAMILY_TRIAD:
		strncpy(system_utsname.machine, 
			MACHINE_TYPE_MSP5000,
			sizeof(system_utsname.machine));
		break;

	default:
		/* we don't recognize the machine */
		strncpy(system_utsname.machine, 
			MACHINE_TYPE_OTHER,
			sizeof(system_utsname.machine));
	}

	setup_prom_printf(0);
	prom_printf("\nLINUX started...\n");
	prom_init_cmdline();
	prom_meminit();

#ifdef CONFIG_BRECIS_PCI  
	if (getdeviceid() & DEV_ID_SINGLE_PC) {	/* If single card mode */
		slmRegs	*sreg = (slmRegs *) SREG_BASE;

		sreg->single_pc_enable = SINGLE_PCCARD;
	}
#endif

#ifdef	CONFIG_BRECIS_PCI
	msp_pci_init();
#endif	/* CONFIG_BRECIS_PCI */

	return 0;
}


