/*
 * Carsten Langgaard, carstenl@mips.com
 * Copyright (C) 2000 MIPS Technologies, Inc.  All rights reserved.
 *
 * ########################################################################
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ########################################################################
 *
 * This is the interface to the remote debugger stub.
 *
 */

#include <linux/serialP.h>
#include <linux/serial_reg.h>

#include <asm/serial.h>
#include <asm/io.h>
#include "gdb_hook.h"

#define	MipsSynch()		__asm__ volatile ("sync")

static struct async_struct kdb_port_info = {0};

extern int putPromChar(char);
extern char getPromChar(void);

int (*putDebugChar)(char);
char (*getDebugChar)(void);
static inline void Delay(void);

static inline void Delay()
{
	int t;

	for (t = 0; t < 10000; t++)
		MipsSynch();
}

static __inline__ unsigned char serial_in(struct async_struct *info, 
							int offset)
{
	if (offset != RR0)
		outb(offset, info->port);

	return inb(info->port);
}

static __inline__ void serial_out(struct async_struct *info, int offset,
							unsigned char value)
{
	if (offset != WR0)
		outb(offset, info->port);

	outb(value, info->port);
}

void rs_kgdb_hook(int tty_no) 
{
	int t;

	spin_lock_init(&kgdb_port_info.xmit_lock);

	kdb_port_info.state = (void *) 1;
	kdb_port_info.magic = SERIAL_MAGIC;
	kdb_port_info.port = (unsigned long) 
		(tty_no == SCC_CHANNEL_A ?
		BRECIS_SCC_REGSA_BASE : BRECIS_SCC_REGSB_BASE);

	/* bring peripheral block out of reset */
	*RST_CLR_REG = 0x200;

	/* delay while peripheral block is coming out of reset */
	Delay();

	/* sync SCC state machine */
	serial_out(&kdb_port_info, WR0, SCC_WR0_NULL_CODE);
	serial_out(&kdb_port_info, WR0, SCC_WR0_NULL_CODE);

	/* reset any error or interrupts */
	serial_out(&kdb_port_info, WR0, SCC_WR0_ERR_RST);
	serial_out(&kdb_port_info, WR0, SCC_WR0_RST_INT);

	/* reset specified channel */
	serial_out(&kdb_port_info, WR9, 
#if 0
			(tty_no == SCC_CHANNEL_A ? 
				SCC_WR9_CH_A_RST : SCC_WR9_CH_B_RST) ||
#else
			SCC_WR9_HDWR_RST |
#endif
			SCC_WR9_NV | SCC_WR9_DLC | SCC_WR9_NO_VEC);

	/* delay before further chip accesses */
	Delay();

	/*
	 * Now, initialize the SCC
	 */
	serial_out(&kdb_port_info, WR4,		/* 1 stop bit, x1 clock */
			SCC_WR4_1_STOP | SCC_WR4_16_CLOCK);
	serial_out(&kdb_port_info, WR8,		/* transmit buffer */
			0x5c);
	serial_out(&kdb_port_info, WR3,		/* 8 bit chars */
			SCC_WR3_RX_8_BITS);
	serial_out(&kdb_port_info, WR5,		/* dtr, 8 bit chars, tx en */
			SCC_WR5_TX_8_BITS | SCC_WR5_DTR | SCC_WR5_TX_EN);
	serial_out(&kdb_port_info, WR10, 	/* clear sync, loop, poll */
			SCC_WR10_NRZ);

	serial_out(&kdb_port_info, WR11,	/* clock mode */
			SCC_WR11_RX_BR_GEN | SCC_WR11_TX_BR_GEN);

	serial_out(&kdb_port_info, WR15,	/* disable ext interrupts */
			1);
	serial_out(&kdb_port_info, WR7S, 0x40);
	serial_out(&kdb_port_info, WR15,	/* disable ext interrupts */
			0);
	
#if 0
	/*
	 * and set the speed of the serial port
	 * (currently hardwired to 9600 8N1
	 */
	t = ((ser->baudfreq / 32) / ser->baudrate) - 2;
#else
	/* I think 0xc is 57600 baud given 16x clocks at this moment */
	t = 0xc;
#endif
	serial_out(&kdb_port_info, WR12, (char) (t & 0xff));
	serial_out(&kdb_port_info, WR13, (char) (t >> 8));

	serial_out(&kdb_port_info, WR14,	/* upper baud rate generator */
			SCC_WR14_BR_EN | SCC_WR14_BR_SRC);

	/* reset interrupts and errors */
	serial_out(&kdb_port_info, WR0, SCC_WR0_RST_INT);
	serial_out(&kdb_port_info, WR0, SCC_WR0_ERR_RST);

	serial_out(&kdb_port_info, WR3,		/* receive parameters */
			SCC_WR3_RX_8_BITS | SCC_WR3_RX_EN);
	serial_out(&kdb_port_info, WR5,		/* transmit parameters */
			SCC_WR5_TX_8_BITS | SCC_WR5_TX_EN | SCC_WR5_DTR | SCC_WR5_RTS);
	serial_out(&kdb_port_info, WR2,		/* interrupt vector */
			0);
	serial_out(&kdb_port_info, WR1,		/* disable ints and xfer mode */
			SCC_WR1_PARITY);
	serial_out(&kdb_port_info, WR9,		/* disable master interrupt control */
			SCC_WR9_NO_RST |
			SCC_WR9_NV | SCC_WR9_DLC | SCC_WR9_NO_VEC);

	/* reset Tx CRC generator and any pending ext/status ints */
	serial_out(&kdb_port_info, WR0, SCC_WR0_RST_TX_CRC);
	serial_out(&kdb_port_info, WR0, SCC_WR0_RST_TX_UND);
	serial_out(&kdb_port_info, WR0, SCC_WR0_RST_INT);
	serial_out(&kdb_port_info, WR0, SCC_WR0_RST_INT);

	/* reset SCC register counter */
	serial_out(&kdb_port_info, WR0, SCC_WR0_NULL_CODE);
}


int rs_putDebugChar(unsigned char c)
{

	if (!kdb_port_info.state) {	/* need to init device first */
		return 0;
	}

	while ((serial_in(&kdb_port_info, RR0) & SCC_RR0_TX_EMPTY) == 0)
		;

	serial_out(&kdb_port_info, WR8, c);

	return 1;
}

unsigned char rs_getDebugChar(void)
{
	unsigned char c;
	unsigned char rr0;
	unsigned char rr1;

	if (!kdb_port_info.state) {	/* need to init device first */
		return 0;
	}

	while (1)
	{
		while (!((rr0 = serial_in(&kdb_port_info, RR0)) & SCC_RR0_RX_AVAIL))
			;

		if (rr0 & SCC_RR0_BREAK)
			continue;	/* ignore any break sequence */
		
		rr1 = serial_in(&kdb_port_info, RR1);
		
		if (rr1 & SCC_RR1_RX_ERRORS)
		{
			serial_out(&kdb_port_info, WR0, SCC_WR0_ERR_RST);
			continue;	/* ignore any framing errors */
		}

		c = serial_in(&kdb_port_info, RR8);

		if (c != 0)		/* ignore any null character */
			return c;
	}

	return 0;
}


#ifdef CONFIG_BRECIS

#include <asm/brecis/brecis.h>

static int brecis_kgdb_active = 0;

void brecis_kgdb_hook(void) 
{
	brecis_kgdb_active = 1;
}

int brecis_putDebugChar(char c)
{

	if (!brecis_kgdb_active) {	/* need to init device first */
		return 0;
	}

	putPromChar(c);

	return 1;
}

char brecis_getDebugChar(void)
{
	char c;

	if (!brecis_kgdb_active) {	/* need to init device first */
		return 0;
	}
	c = getPromChar();
	return(c);
}

#endif
