/* $Id: NanoDelay.c,v 1.1 2002/05/21 22:57:38 swahl Exp $	*/

/*
 * nanodelay - Routine to perform short-duration delays using the MIPS
 *      count register.
 *
 * Written by: Mark Rustad, Brecis Communications, 07/31/01.
 */

/******************************************************************/
/* Copyright (c) 2001, 2002 BRECIS Communications                 */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

#include <linux/module.h>
#include "brecis/NanoDelay.h"

/* In-line macro for COP0 count register access */

#define GetCOP0Count()  \
({ unsigned long __value;       \
        __asm__ __volatile__ ("mfc0 %0, $9" : "=r" (__value));    \
        __value; })

#define OVERHEAD    3   /* Partial overhead adjustment */

static struct
{
    unsigned long   numerator;
    unsigned long   denominator;
}   nanoperiod = { 1, 10 };	/* Assumes 200 MHz */


/***
* NanoDelay - Delay for a short time.
*
* Input:
*  nanos - Number of nanoseconds to delay
*
* Notes:
*  NanoInit must have been called to establish the CPU clock rate
*  before calling this function.
*
*  The caller probably should have interrupts disabled before calling
*  this to have any expectation of real accuracy.
*
* Returns: N/A
*/

void NanoDelay(unsigned int nanos)
{
	unsigned long   count;
	long    difference;

	count = GetCOP0Count(); /* Get the count before doing calculations */
	count += ((nanos * nanoperiod.numerator) / nanoperiod.denominator)
		- OVERHEAD;
	do
		difference = count - GetCOP0Count();
	while (difference > 0);
}


/***
*
* NanoInit - Set timing parameters for NanoDelay.
*
*
* Input:
*  freq - CPU clock frequency in Hz.
*
* Notes:
*  This routine takes the 2:1 ratio between the cpu clock and count
*  register into account, so the caller uses only the actual cpu clock
*  period to initialialize this routine.
*
* Returns: N/A
*/

void NanoInit(unsigned int freq)
{
	unsigned long   denominator;

	/* The following provides for the 2:1 ratio between the cpu clock
	 * and the count register rate.
	 */
	denominator = 2000000000;

	/*
	 * Reduce the numbers. Note that because the denominator
	 * is a constant and only has prime factors of two
	 * and 5, we only need to divide them out. Cool, eh?
	 *
	 * Look at the object code to see how GCC gets all the quotients
	 * and remainders without using any divide instructions. Ah,
	 * just like on the Cyber...
	 */

	while ((denominator & 1) == 0 && (freq & 1) == 0)
		denominator >>= 1, freq >>= 1;

	while ((denominator % 5) == 0 && (freq % 5) == 0)
		denominator /= 5, freq /= 5;

	nanoperiod.numerator = freq;
	nanoperiod.denominator = denominator;
}

EXPORT_SYMBOL(NanoDelay);
EXPORT_SYMBOL(NanoInit);


/*	End of NanoDelay.c	*/

