/* $Header: /proj/software/pub/CVSROOT/uClinux/brecis/vpm/vpm-talk.c,v 1.2 2003/05/28 23:03:17 awarner Exp $ */

/******************************************************************/
/* Copyright (c) 2003 BRECIS Communications                       */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

/*
 * vpm-talk [-c <codec>] [-d] [-p <pkt size>] -[t|l] <port> <device>
 *
 * Bidirectional connection over TCP, where client may
 * select different codecs for each connection.
 *
 * Uses vanilla TCP, not RTP.
 */
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <linux/telephony.h>
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>

char	buffer[1024] ;

int	pkt_size = 320 ;
int	daemonise = 0 ;

void
usage(void)
{
	fprintf(stderr,
		"Usage: vpm-talk [-d] [-l|-t [<host>:]<port>] [-c <codec>] [-p <packet size>] <device>\n") ;
	exit(1) ;
}

/*
 * strtosin() -	Convert a string of the form [<host>:]<port> to
 * 		a sockaddr_in structure. If the host component
 * 		is missing, default to localhost.
 * 		Hardwire address family to AF_INET.
 *
 * 		Return 0 if conversion is OK, !0 otherwise
 * 		(including syntax errors and possible hostname
 * 		lookup failure)
 */
int
strtosin(char *arg, struct sockaddr_in *s)
{
	const char *hostp ;
	const char *portp ;
	char *p ;
	int  tmp ;
	struct hostent *hp ;

	/*
	 * This looks backwards, but it isn't.
	 * Is there a "<host>:" component ?
	 */
	hostp = strtok(arg, ":") ;
	if ((portp = strtok(NULL, NULL)) == NULL) {
		/*
		 * Nope.
		 */
		s->sin_addr.s_addr = INADDR_ANY ;
		portp = arg ;
	} else {
		if ((hp = gethostbyname(hostp)) == NULL) {
			return(h_errno) ;
		}

		/*
		 * Note: we ignore multiple addresses, since this
		 * is a test fixture.
		 */
		s->sin_addr = *((struct in_addr *) *hp->h_addr_list) ;
	}

	s->sin_family = AF_INET ;

	tmp = (int)strtol(portp, &p, 10) ;

	/*
	 * Check that strtol was able to do his job.
	 */
	if ((*portp == '\0') || (p == portp) || (*p != '\0')) {
		return(-1) ;
	}

	s->sin_port = htonl(tmp) ;

	return(0) ;
}

/*
 * pick_codec() -	Given an ascii string, return the
 * 			phone_codec enum matching the requested
 * 			codec, or -1 if no match.
 */
phone_codec
pick_codec(char *codec_name)
{
	if (strcasecmp(codec_name, "g711u") == 0) {
		return(ULAW) ;
	}

	if (strcasecmp(codec_name, "g711a") == 0) {
		return(ALAW) ;
	}

	if (strcasecmp(codec_name, "g723_64") == 0) {
		return(G723_63) ;
	}

	if (strcasecmp(codec_name, "g723_53") == 0) {
		return(G723_53) ;
	}

	if (strcasecmp(codec_name, "g723") == 0) {
		return(G723_63) ;
	}

	if (strcasecmp(codec_name, "g729") == 0) {
		return(G729) ;
	}

	if (strcasecmp(codec_name, "g726") == 0) {
		return(G726_32) ;
	}

	return(-1) ;
}

/*
 * xfer_data() -	Called to tranfer data between a socket and
 * 			a codec. Return the number of bytes transferred,
 * 			or the return value from the read/write on error.
 */
int
xfer_data(int from, int to)
{
	int n ;

	if ((n = read(from, buffer, sizeof(buffer))) > 0) {
		return(write(to, buffer, n)) ;
	}

	return(n) ;
}

int
main(int argc,char *argv[])
{
	extern char *optarg ;
	extern int optind ;
	int c ;
	char *p ;
	char *phone_dev ;
	struct sockaddr_in s_in ;
	int vpm, l, s ;
	phone_codec codecx = ULAW ;
	int talk = 0, list = 0 ;

	while ((c = getopt(argc, argv, "c:dl:p:t:")) >= 0) {
		switch(c) {
		    case 'c':
			if ((codecx = pick_codec(optarg)) <0) {
				fprintf(stderr, "Bad codec: %s\n", optarg) ;
				exit(4) ;
			}
			break ;

		    case 'd':
			daemonise++ ;
			break ;

		    case 'p':
			pkt_size = (int)strtol(optarg, &p, 0) ;
			if ((*p != '\0') || (p == optarg)) {
				usage() ;
			}
			break ;

		    case 'l':
			if (strtosin(optarg, &s_in)) {
				usage() ;
			}
			list++ ;
			break ;

		    case 't':
			if (strtosin(optarg, &s_in)) {
				usage() ;
			}
			talk++ ;
			break ;

		    default:
			usage() ;
			break ;
		}
	}

	/*
	 * Ensure that exactly one of -t or -l options is
	 * specified.
	 */
	if ((talk && list) || !(talk || list)) {
		usage() ;
	}

	if ((phone_dev = argv[optind]) == NULL) {
		usage() ;
	}

	/*
	 * open the codec
	 */
	if ((vpm = open(phone_dev, O_RDWR | O_NONBLOCK)) < 0) {
		perror("vpm open") ;
		exit(3) ;
	}

	/*
	 * open the socket, depending on whether we're a talker
	 * or listener, either connet() or bind() & listen()
	 * respectively.
	 */
	if ((l = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket") ;
		close(vpm) ;
		exit(5) ;
	}

	if (list) {
		if (bind(l, (struct sockaddr *)&s_in, sizeof(s_in)) != 0) {
			perror("bind") ;
			close(l) ;
			close(vpm) ;
			exit(7) ;
		}

		if (listen(l, 0) != 0) {
			perror("listen") ;
			close(l) ;
			close(vpm) ;
			exit(8) ;
		}
	}

	if (talk) {
		if (connect(l, (struct sockaddr *)&s_in, sizeof(s_in)) != 0) {
			perror("connect") ;
			close(vpm) ;
			exit(7) ;
		}
	}

	/*
	 * Daemonise if asked to.
	 */
	if (daemonise && (fork() != 0)) {
		exit(0) ;
	}

	/*
	 * This loop is only executed once for a talker, but
	 * multiple times (once for each incoming call) for
	 * a listener.
	 */
	while ((s = (talk ? l : accept(l, NULL, 0))) >= 0) {
		fd_set read_fd ;
		int max_fd ;

		/*
		 * Set up & start codec.
		 */
		fprintf(stderr, "New connection....\n") ;

		/*
		 * Read or write the codec type as the first data
		 * on the socket for the listener and talker respectively.
		 */
		if (list &&
			(read(s, &codecx, sizeof(codecx)) != sizeof(codecx))) {
			close(s) ;
			continue ;
		}

		if (talk &&
			(write(s, &codecx, sizeof(codecx)) != sizeof(codecx))) {
			close(s) ;
			continue ;
		}

		fprintf(stderr, "Codec = %d\n", (int)codecx) ;

		if (ioctl(vpm, PHONE_REC_CODEC, codecx) < 0) {
			goto end ;
		}

		if (ioctl(vpm, PHONE_PLAY_CODEC, codecx) < 0) {
			goto end ;
		}

		if (ioctl(vpm, PHONE_PLAY_START, NULL)) {
			goto end ;
		}

		if (ioctl(vpm, PHONE_REC_START, NULL)) {
			goto end ;
		}


		max_fd = ((s > vpm) ? s : vpm) + 1 ;
		FD_ZERO(&read_fd) ;
		FD_SET(vpm, &read_fd) ;
		FD_SET(s, &read_fd) ;
		/*
		 * Select here on reads from both the vpm &
		 * the network socket.
		 *
		 * Don't buffer data yet.
		 *
		 * TODO: select on exceptions from the telephony
		 * device for hook/tone events.
		 */
		while(select(max_fd, &read_fd, NULL, NULL, NULL) > 0) {
			/*
			 * Check for read data from the codec that
			 * we will write to the network.
			 */
			if (FD_ISSET(vpm, &read_fd)) {
				if (xfer_data(vpm, s) <= 0) {
					break ;
				}
			}

			/*
			 * Check for read data from the network that
			 * we will write to the codec.
			 */
			if (FD_ISSET(s, &read_fd)) {
				if (xfer_data(s, vpm) <= 0) {
					break ;
				}
			}

			FD_SET(vpm, &read_fd) ;
			FD_SET(s, &read_fd) ;
		}
			
		ioctl(vpm, PHONE_PLAY_STOP, NULL) ;
		ioctl(vpm, PHONE_REC_STOP, NULL) ;

end:
	        fprintf(stderr, "End connection....\n") ;

		/*
		 * If we're a listener, close the socket we
		 * accepted() and go around for more.
		 *
		 * Otherwise, break out of the loop, since
		 * we only execute it once for a talker.
		 */
		if (list) {
			close(s) ;
		} else {
			break ;
		}
	}

	close(l) ;

	close(vpm) ;

	exit(0) ;
}
