/*****************************************************************************
* wanconfig.c	WAN Multiprotocol Router Configuration Utility.
*
* Author:	Nenad Corbic	<ncorbic@sangoma.com>
*               Gideon Hack
*
* Copyright:	(c) 1995-2000 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* May 11, 2001  Alex Feldman    Added T1/E1 support (TE1).
* Apr 16, 2001  David Rokhvarg  Added X25_SRC_ADDR and ACCEPT_CALLS_FROM to 'chan_conftab'
* Nov 15, 2000  Nenad Corbic    Added true interface encoding type option
* Arp 13, 2000  Nenad Corbic	Added the command line argument, startup support.
*                               Thus, one can configure the router using
*                               the command line arguments.
* Jan 28, 2000  Nenad Corbic    Added support for ASYCN protocol.
* Sep 23, 1999  Nenad Corbic    Added support for HDLC STREAMING, Primary
*                               and secondary ports. 
* Jun 02, 1999 	Gideon Hack	Added support for the S514 PCI adapter.
* Jan 07, 1998	Jaspreet Singh	Made changes for 2.1.X.
*				Added support for WANPIPE and API integration.
* Jul 20, 1998	David Fong	Added Inverse ARP option to channel config.
* Jun 26, 1998	David Fong	Added IP_MODE to PPP Configuration.
*				Used for Dynamic IP Assignment.
* Jun 18, 1998	David Fong	Added Cisco HDLC definitions and structures
* Dec 16, 1997	Jaspreet Singh 	Moved IPX and NETWORK to 'chan_conftab'
* Dec 08, 1997	Jaspreet Singh	Added USERID, PASSWD and SYSNAME in 
*				'chan_conftab' 
* Dec 05, 1997	Jaspreet Singh  Added PAP and CHAP in 'chan_conftab'
*				Added AUTHENTICATOR in 'ppp_conftab' 
* Oct 12, 1997	Jaspreet Singh	Added IPX and NETWORK to 'common_conftab'
*				Added MULTICAST to 'chan_conftab'
* Oct 02, 1997  Jaspreet Singh	Took out DLCI from 'fr_conftab'. Made changes 
*				so that a list of DLCI is prepared to 
*				configuring them when emulating a NODE 
* Jul 07, 1997	Jaspreet Singh	Added 'ttl' to 'common_conftab'
* Apr 25, 1997  Farhan Thawar   Added 'udp_port' to 'common_conftab'
* Jan 06, 1997	Gene Kozin	Initial version based on WANPIPE configurator.
*****************************************************************************/

/*****************************************************************************
* Usage:
*   wanconfig [-f {conf_file}]	Configure WAN links and interfaces
*   wanconfig -d {device}	Shut down WAN link
*   wanconfig -h|?		Display help screen
*
* Where:
*   {conf_file}		configuration file name. Default is /etc/wanpipe1.conf
*   {device}		name of the WAN device in /proc/net/wanrouter directory
*
* Optional switches:
*   -v			verbose mode
*   -h or -?		display help screen
*****************************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <linux/sdla_te1.h>	/* Sangoma TE1 definitions */
#include <linux/wanrouter.h>	/* WAN router API definitions */

/****** Defines *************************************************************/

#ifndef	min
#define	min(a,b)	(((a)<(b))?(a):(b))
#endif

#define	is_digit(ch) (((ch)>=(unsigned)'0'&&(ch)<=(unsigned)'9')?1:0)

enum ErrCodes			/* Error codes */
{
	ERR_SYSTEM = 1,		/* system error */
	ERR_SYNTAX,		/* command line syntax error */
	ERR_CONFIG,		/* configuration file syntax error */
	ERR_LIMIT
};

/* Configuration stuff */
#define	MAX_CFGLINE	256
#define	MAX_CMDLINE	256
#define	MAX_TOKENS	32

/****** Data Types **********************************************************/

typedef struct look_up		/* Look-up table entry */
{
	uint	val;		/* look-up value */
	void*	ptr;		/* pointer */
} look_up_t;

typedef struct key_word		/* Keyword table entry */
{
	char*	keyword;	/* -> keyword */
	uint	offset;		/* offset of the related parameter */
	int	dtype;		/* data type */
} key_word_t;

typedef struct data_buf		/* General data buffer */
{
	unsigned size;
	void* data;
} data_buf_t;

/*
 * Data types for configuration structure description tables.
 */
#define	DTYPE_INT	1
#define	DTYPE_UINT	2
#define	DTYPE_LONG	3
#define	DTYPE_ULONG	4
#define	DTYPE_SHORT	5
#define	DTYPE_USHORT	6
#define	DTYPE_CHAR	7
#define	DTYPE_UCHAR	8
#define	DTYPE_PTR	9
#define	DTYPE_STR	10
#define	DTYPE_FILENAME	11

typedef struct chan_def			/* Logical Channel definition */
{
	char name[WAN_IFNAME_SZ+1];	/* interface name for this channel */
	char* addr;			/* -> media address */
	char* conf;			/* -> configuration data */
	char* usedby;			/* used by WANPIPE or API */
	char* descr;			/* -> description */
	struct chan_def* next;		/* -> next channel definition */
} chan_def_t;

typedef struct link_def			/* WAN Link definition */
{
	char name[WAN_DRVNAME_SZ+1];	/* link device name */
	int config_id;			/* configuration ID */
	char* conf;			/* -> configuration data */
	char* descr;			/* -> description */
	chan_def_t* chan;		/* list of channel definitions */
	struct link_def* next;		/* -> next link definition */
} link_def_t;

/****** Function Prototypes *************************************************/

int arg_proc (int argc, char* argv[]);
void show_error	(int err);
int startup(void);
int shutdown (void);
int configure (void);
int parse_conf_file (char* fname);
int build_linkdef_list (FILE* file);
int build_chandef_list (FILE* file);
char* read_conf_section (FILE* file, char* section);
int read_conf_record (FILE* file, char* key);
int configure_link (link_def_t* def);
int configure_chan (int dev, chan_def_t* def);
int set_conf_param (char* key, char* val, key_word_t* dtab, void* conf);
int tokenize (char* str, char **tokens);
char* strstrip (char* str, char* s);
char* strupcase	(char* str);
void* lookup (int val, look_up_t* table);
int name2val (char* name, look_up_t* table);
int read_data_file (char* name, data_buf_t* databuf);
unsigned long filesize (FILE* file);
unsigned int dec_to_uint (unsigned char* str, int len);
unsigned int get_config_data (int, char**);
void show_help(void);
void show_usage(void);
void set_action(int new_action);
int gencat (char *filename); 


/* Parsing the command line arguments, in order
 * to starting WANPIPE from the command line. 
 */ 
int get_devices (FILE *fp, int *argc, char ***argv, char**);
int get_interfaces (FILE *fp, int *argnum, char ***argv_ptr, char *device_name);
int get_hardware (FILE *fp, int *argc_ptr, char ***argv_ptr, char *device_name);
int get_intr_setup (FILE *fp, int *argc_ptr, char ***argv_ptr);
int router_down (char *devname, int ignore_error);
int router_ifdel (char *card_name, char *dev_name);
int conf_file_down (void);
ulong parse_active_channel(char* val);
ulong get_active_channels(int channel_flag, int start_channel, int stop_channel);


extern	int close (int);
/****** Global Data *********************************************************/

char progname[] =	"wanconfig";
char progname_sp[] = 	"         ";
char def_conf_file[] =	"/usr/local/wanrouter/wanpipe.conf";	/* default name */
char router_dir[] =	"/proc/net/wanrouter";	/* location of WAN devices */
char conf_dir[] =	"/etc/wanpipe";
char banner[] =		"WAN Router Configurator. v2.1.0 "
			"(c) 1995-1999 Sangoma Technologies Inc."
;

char usagetext[] = 
	"\n"
	"  wanconfig: Wanpipe device driver configuration tool\n\n"
	"  Usage: wanconfig [ -hvw ] [ -f <config-file> ] [ -U {arg options} ]\n"
       	"                   [ -y <verbose-log> ] [ -z <kernel-log> ]\n"
	"                   [ card <wan-device-name> [ dev <dev-name> | nodev ] ]\n"
	"                   [ help | start | stop | up | down | add | del | reload\n"
	"                     | restart | status | show | config ]\n"
       	"\n";	

char helptext[] =
	"\n"
	"  wanconfig: Wanpipe device driver configuration tool\n\n"
	"Usage:\n"
	"------\n"
	"\twanconfig [ -hvw ] [ -f <config-file> ] [ -U {arg-options} ]\n"
       	"\t                   [ -y <verbose-log> ] [ -z <kernel-log> ]\n"
	"\t                   [ card <wan-dev-name> [ dev <dev-name> | nodev ] ]\n"
	"\t                   [ help | start | stop | up | down | add | del\n"
       	"\t                     | reload | restart | status | show | config ]\n"
	"\n"
	"\tWhere:\n"
	"\n"
	"\t          [-f {config-file}] Specify an overriding configuration file.\n"
	"\t          [-U {arg-options}] Configure WAN link using command line\n"
	"\t                             arguments.\n"
	"\t          [-v]               Verbose output to stdout.\n"
	"\t          [-h] | help        Show this help.\n"
	"\t          [-w]               Enable extra error messages about debug\n"
	"\t                             log locations.\n"
	"\t          [-y]               Give location of wanconfig verbose log\n"
       	"\t                             for -w switch.\n"
	"\t          [-z]               Give location of syslogd kernel logging\n"
       	"\t                             for -w switch.\n"
	"\t          card <wan-dev-name>\n"
	"\t                             Specify which WAN interface/card to\n"
	"\t                             operate on. Name given in\n"
       	"\t                             /proc/net/wanrouter directory.\n"
	"\t          dev <dev-name>     Specify which linux network interface\n"
	"\t                             to operate on.\n"
	"\t          nodev              Turns off creation of interface(s)\n"
	"\t                             when (re)starting card(s).\n"
	"\t          start | up | add   Configure WAN interface(s)/card(s)\n"
	"\t                             and/or create linux network interface(s).\n"
	"\t          stop | down | del  Shut down WAN interface(s)/card(s) and/or\n"
	"\t                             destroy linux network interface(s).\n"
	"\t          reload | restart   Restart WAN interface(s)/card(s) and/or\n"
       	"\t                             recreate linux network interface(s).\n"
	"\t          status | stat | show\n"
	"\t                             Display status information on all\n"
	"\t                             interfaces/cards or for a specific\n"
       	"\t                             WAN interface/card.\n"
	"\t          config             Display configuration information for all\n"
	"\t                             WAN interfaces/cards.\n"
	"\n"
	"\t{config-file}\tconfiguration file (default is\n"
       	"\t\t\t/etc/wanrouter/wanpipe#.conf or\n"
	"\t\t\t/etc/wanrouter/wanpipe.conf in that order)\n"
	"\t{arg-options}\tare as given below\n"
	"\nArg Options:\n"
	"	[devices] \\ \n"
	" 	 <devname> <protocol> \\ \n"
	"	[interfaces] \\ \n"
	"	 <if_name> <devname> <{addr/-}> <operation_mode> \\ \n"
	"	 <if_name> <devname> <{addr/-}> <operation_mode> \\ \n"
        "	 ...   \\ \n"
	"	[devname] \\ \n"
	"	 <hw_option> <hw_option_value> \\ \n"
	"	 <hw_option> <hw_option_value> \\ \n"
	"	 ...  \\ \n"
	"	[if_name] \\ \n"
	"	 <protocol_opton> <protocol_option_value> \\ \n"
	"	 <protocol_opton> <protocol_option_value> \\ \n"
	"	 ... \\ \n\n"
	"	devname		device name. ex:wanpipe# (#=1..16) \n"
	"	protocol	wan protocol. ex: WAN_FR,WAN_CHDLC,WAN_PPP,WAN_X25 \n"
	"	if_name		interface name. ex: wp1_fr16, wp1_ppp \n"
	"	addr/-		For Frame Relay: DLCI number ex: 16, 25\n"
	"			    X25 PVC :    LCN number ex: 1, 2, 4\n"
	"			    X25 SVC :    X25 address ex: @123, @4343 \n"
	"			For other protocol set to: '-'\n"
	"	operation_mode  Mode of operation: WANPIPE - for routing \n"
	"					   API     - raw api interface\n"
	"			The API only applies to WAN_CHDLC, WAN_FR and \n" 
	"			WAN_X25 protocols.\n"
	"	hw_option\n"
	"	protocol_option \n"
	"			Please refer to sample wanpipe#.conf files in \n"
	"			/usr/local/wanrotuer/samples directory for the\n"
	"			appropriate HW/Protocol optoins. \n" 
	"			ex: ioprot, irq, s514_cpu, pci_slot ... \n"
	"	hw_option_value \n"
	"	protocol_option_value \n"
	"			Value of the above options. Refer to the above \n"
	"			sample files. ex: ioport = '0x360' multicast=YES  \n\n"
	"	Example 1: Bring up wanpipe1 device from the command line\n"
	"	wanconfig -U 	[devices] \\ \n"
        "			wanpipe1 WAN_CHDLC \\ \n"
        "			[interfaces] \\ \n"
        "			wp1_chdlc wanpipe1 - WANPIPE \\ \n" 
        "			[wanpipe1] \\ \n"
        "			IOPORT 0x360 \\ \n"
        "			IRQ 7 \\ \n"
        "			Firmware  /usr/local/wanrouter/firmware/cdual514.sfm \\ \n"
        "			CommPort PRI \\ \n"
        "			Receive_Only NO \\ \n"
        "			Interface V35 \\ \n"
        "			Clocking External \\ \n"
        "			BaudRate 1600000 \\ \n"
        "			MTU 1500 \\ \n"
        "			UDPPORT 9000 \\ \n" 
        "			TTL 255 \\ \n"
        "			[wp1_chdlc] \\ \n"
        "			MULTICAST  NO \\ \n"
        "			IGNORE_DCD YES \\ \n"
        "			IGNORE_CTS YES \\ \n"
        "			IGNORE_KEEPALIVE YES \\ \n" 
        "			HDLC_STREAMING YES \\ \n"
        "			KEEPALIVE_TX_TIMER 10000 \n\n"   
	"	Example 2: Shutdown wanpipe1 device from the command line \n"
	"	\n\t\t#> wanconfig card wanpipe1 stop\n\n"
	"	Example 3: Create fr17 linux network interface and \n"
	"                  start wanpipe1 device (if not already started)\n"
	"                  from the command line \n"
	"	\n\t\t#> wanconfig card wanpipe1 dev fr17 start\n\n"
	"	Example 4: Shutdown all WAN devices from the command line \n"
	"	\n\t\t#> wanconfig stop\n"
	

;
char* err_messages[] =				/* Error messages */
{
	"Invalid command line syntax",		/* ERR_SYNTAX */
	"Invalid configuration file syntax",	/* ERR_CONFIG */
	"Unknown error code",			/* ERR_LIMIT */
};
enum	/* modes */
{
	DO_UNDEF,
	DO_START,
	DO_STOP,
	DO_RESTART,
	DO_HELP,
	DO_ARG_CONFIG,
	DO_ARG_DOWN,
	DO_SHOW_STATUS,
	DO_SHOW_CONFIG
} action;				/* what to do */

#define	SLEEP_TIME 1			/* Sleep time after executing ioctl */
int verbose = 0;			/* verbosity level */
char* conf_file = NULL;			/* configuration file */
char *dev_name = NULL;
char *card_name = NULL;
char *krnl_log_file = "/var/log/messages";
char *verbose_log = "/var/log/wanrouter";
int  weanie_flag = 0;
int  nodev_flag = 0;
link_def_t* link_defs;			/* list of WAN link definitions */
union
{
	wandev_conf_t linkconf;		/* link configuration structure */
	wanif_conf_t chanconf;		/* channel configuration structure */
} u;

/*
 * Configuration structure description tables.
 * WARNING:	These tables MUST be kept in sync with corresponding data
 *		structures defined in linux/wanrouter.h
 */
key_word_t common_conftab[] =	/* Common configuration parameters */
{
  { "IOPORT",     offsetof(wandev_conf_t, ioport),     DTYPE_UINT },
  { "MEMADDR",    offsetof(wandev_conf_t, maddr),       DTYPE_UINT },
  { "MEMSIZE",    offsetof(wandev_conf_t, msize),       DTYPE_UINT },
  { "IRQ",        offsetof(wandev_conf_t, irq),         DTYPE_UINT },
  { "DMA",        offsetof(wandev_conf_t, dma),         DTYPE_UINT },
  { "S514CPU",    offsetof(wandev_conf_t, S514_CPU_no), DTYPE_STR },
  { "PCISLOT",    offsetof(wandev_conf_t, PCI_slot_no), DTYPE_UINT },
  { "PCIBUS", 	  offsetof(wandev_conf_t, pci_bus_no),	DTYPE_UINT },
  { "AUTO_PCISLOT",offsetof(wandev_conf_t, auto_pci_cfg), DTYPE_UCHAR },
  { "COMMPORT",   offsetof(wandev_conf_t, comm_port),   DTYPE_UCHAR },
/* TE1 New hardware parameters for T1/E1 board */
  { "MEDIA",    offsetof(wandev_conf_t, te_cfg)+offsetof(sdla_te_cfg_t, media), DTYPE_UCHAR },
  { "LCODE",    offsetof(wandev_conf_t, te_cfg)+offsetof(sdla_te_cfg_t, lcode), DTYPE_UCHAR },
  { "FRAME",    offsetof(wandev_conf_t, te_cfg)+offsetof(sdla_te_cfg_t, frame), DTYPE_UCHAR },
  { "LBO",      offsetof(wandev_conf_t, te_cfg)+offsetof(sdla_te_cfg_t, lbo), DTYPE_UCHAR },
  { "TE_CLOCK", offsetof(wandev_conf_t, te_cfg)+offsetof(sdla_te_cfg_t, te_clock), DTYPE_UCHAR },
  { "ACTIVE_CH",offsetof(wandev_conf_t, te_cfg)+offsetof(sdla_te_cfg_t, active_ch), DTYPE_ULONG },
  { "BAUDRATE",   offsetof(wandev_conf_t, bps),         DTYPE_UINT },
  { "MTU",        offsetof(wandev_conf_t, mtu),         DTYPE_UINT },
  { "UDPPORT",    offsetof(wandev_conf_t, udp_port),    DTYPE_UINT },
  { "TTL",	  offsetof(wandev_conf_t, ttl),		DTYPE_UCHAR },
  { "INTERFACE",  offsetof(wandev_conf_t, interface),   DTYPE_UCHAR },
  { "CLOCKING",   offsetof(wandev_conf_t, clocking),    DTYPE_UCHAR },
  { "LINECODING", offsetof(wandev_conf_t, line_coding), DTYPE_UCHAR },
  { "STATION",    offsetof(wandev_conf_t, station),     DTYPE_UCHAR },
  { "CONNECTION", offsetof(wandev_conf_t, connection),  DTYPE_UCHAR },
  { "LINEIDLE",   offsetof(wandev_conf_t, line_idle),   DTYPE_UCHAR },
  { "OPTION1",    offsetof(wandev_conf_t, hw_opt[0]),   DTYPE_UINT },
  { "OPTION2",    offsetof(wandev_conf_t, hw_opt[1]),   DTYPE_UINT },
  { "OPTION3",    offsetof(wandev_conf_t, hw_opt[2]),   DTYPE_UINT },
  { "OPTION4",    offsetof(wandev_conf_t, hw_opt[3]),   DTYPE_UINT },
  { "FIRMWARE",   offsetof(wandev_conf_t, data_size),   DTYPE_FILENAME },
  { "RXMODE",     offsetof(wandev_conf_t, read_mode),   DTYPE_CHAR },
  { "RECEIVE_ONLY", offsetof(wandev_conf_t, receive_only), DTYPE_UCHAR},	  
  { "BACKUP",     offsetof(wandev_conf_t, backup), DTYPE_UCHAR},
  { "TTY",        offsetof(wandev_conf_t, tty), DTYPE_UCHAR},
  { "TTY_MAJOR",  offsetof(wandev_conf_t, tty_major), DTYPE_UINT},
  { "TTY_MINOR",  offsetof(wandev_conf_t, tty_minor), DTYPE_UINT},
  { "TTY_MODE",   offsetof(wandev_conf_t, tty_mode), DTYPE_UINT},
  { "IGNORE_FRONT_END",  offsetof(wandev_conf_t, ignore_front_end_status), DTYPE_UCHAR},

 /* Bit strm options */
  { "SYNC_OPTIONS",	       offsetof(wandev_conf_t, sync_options), DTYPE_UINT },
  { "RX_SYNC_CHAR",	       offsetof(wandev_conf_t, rx_sync_char), DTYPE_UCHAR},
  { "MONOSYNC_TX_TIME_FILL_CHAR", offsetof(wandev_conf_t, monosync_tx_time_fill_char), DTYPE_UCHAR},
  { "MAX_LENGTH_TX_DATA_BLOCK", offsetof(wandev_conf_t,max_length_tx_data_block), DTYPE_UINT},
  { "RX_COMPLETE_LENGTH",       offsetof(wandev_conf_t,rx_complete_length), DTYPE_UINT},
  { "RX_COMPLETE_TIMER",        offsetof(wandev_conf_t,rx_complete_timer), DTYPE_UINT},	
  
  { NULL, 0, 0 }
 };
 
key_word_t ppp_conftab[] =	/* PPP-specific configuration */
 {
  { "RESTARTTIMER",   offsetof(wan_ppp_conf_t, restart_tmr),   DTYPE_UINT },
  { "AUTHRESTART",    offsetof(wan_ppp_conf_t, auth_rsrt_tmr), DTYPE_UINT },
  { "AUTHWAIT",       offsetof(wan_ppp_conf_t, auth_wait_tmr), DTYPE_UINT },
  
  { "DTRDROPTIMER",   offsetof(wan_ppp_conf_t, dtr_drop_tmr),  DTYPE_UINT },
  { "CONNECTTIMEOUT", offsetof(wan_ppp_conf_t, connect_tmout), DTYPE_UINT },
  { "CONFIGURERETRY", offsetof(wan_ppp_conf_t, conf_retry),    DTYPE_UINT },
  { "TERMINATERETRY", offsetof(wan_ppp_conf_t, term_retry),    DTYPE_UINT },
  { "MAXCONFREJECT",  offsetof(wan_ppp_conf_t, fail_retry),    DTYPE_UINT },
  { "AUTHRETRY",      offsetof(wan_ppp_conf_t, auth_retry),    DTYPE_UINT },
  { "AUTHENTICATOR",  offsetof(wan_ppp_conf_t, authenticator), DTYPE_UCHAR},
  { "IP_MODE",        offsetof(wan_ppp_conf_t, ip_mode),       DTYPE_UCHAR},
  { NULL, 0, 0 }
};

key_word_t chdlc_conftab[] =	/* Cisco HDLC-specific configuration */
 {
  { "IGNORE_DCD", offsetof(wan_chdlc_conf_t, ignore_dcd), DTYPE_UCHAR},
  { "IGNORE_CTS", offsetof(wan_chdlc_conf_t, ignore_cts), DTYPE_UCHAR},
  { "IGNORE_KEEPALIVE", offsetof(wan_chdlc_conf_t, ignore_keepalive), DTYPE_UCHAR},
  { "HDLC_STREAMING", offsetof(wan_chdlc_conf_t, hdlc_streaming), DTYPE_UCHAR},
  { "KEEPALIVE_TX_TIMER", offsetof(wan_chdlc_conf_t, keepalive_tx_tmr), DTYPE_UINT },
  { "KEEPALIVE_RX_TIMER", offsetof(wan_chdlc_conf_t, keepalive_rx_tmr), DTYPE_UINT },
  { "KEEPALIVE_ERR_MARGIN", offsetof(wan_chdlc_conf_t, keepalive_err_margin),    DTYPE_UINT },
  { "SLARP_TIMER", offsetof(wan_chdlc_conf_t, slarp_timer),    DTYPE_UINT },
  { NULL, 0, 0 }
};

key_word_t fr_conftab[] =	/* Frame relay-specific configuration */
{
  { "SIGNALLING",    offsetof(wan_fr_conf_t, signalling),     DTYPE_UINT },
  { "T391", 	     offsetof(wan_fr_conf_t, t391),           DTYPE_UINT },
  { "T392",          offsetof(wan_fr_conf_t, t392),           DTYPE_UINT },
  { "N391",	     offsetof(wan_fr_conf_t, n391),           DTYPE_UINT },
  { "N392",          offsetof(wan_fr_conf_t, n392),           DTYPE_UINT },
  { "N393",          offsetof(wan_fr_conf_t, n393),           DTYPE_UINT },
  { "FR_ISSUE_FS",   offsetof(wan_fr_conf_t, issue_fs_on_startup),DTYPE_UCHAR },
  { "NUMBER_OF_DLCI",    offsetof(wan_fr_conf_t, dlci_num),       DTYPE_UINT },
  { NULL, 0, 0 }
};

key_word_t x25_conftab[] =	/* X.25-specific configuration */
{
  { "LOWESTPVC",    offsetof(wan_x25_conf_t, lo_pvc),       DTYPE_UINT },
  { "HIGHESTPVC",   offsetof(wan_x25_conf_t, hi_pvc),       DTYPE_UINT },
  { "LOWESTSVC",    offsetof(wan_x25_conf_t, lo_svc),       DTYPE_UINT },
  { "HIGHESTSVC",   offsetof(wan_x25_conf_t, hi_svc),       DTYPE_UINT },
  { "HDLCWINDOW",   offsetof(wan_x25_conf_t, hdlc_window),  DTYPE_UINT },
  { "PACKETWINDOW", offsetof(wan_x25_conf_t, pkt_window),   DTYPE_UINT },
  { "CCITTCOMPAT",  offsetof(wan_x25_conf_t, ccitt_compat), DTYPE_UINT },
  { "X25CONFIG",    offsetof(wan_x25_conf_t, x25_conf_opt), DTYPE_UINT }, 	
  { "LAPB_HDLC_ONLY", offsetof(wan_x25_conf_t, LAPB_hdlc_only), DTYPE_UCHAR },
  { "CALL_SETUP_LOG", offsetof(wan_x25_conf_t, logging), DTYPE_UCHAR },
  { "OOB_ON_MODEM",   offsetof(wan_x25_conf_t, oob_on_modem), DTYPE_UCHAR},
  { "T1", 	    offsetof(wan_x25_conf_t, t1), DTYPE_UINT },
  { "T2", 	    offsetof(wan_x25_conf_t, t2), DTYPE_UINT },	
  { "T4", 	    offsetof(wan_x25_conf_t, t4), DTYPE_UINT },
  { "N2", 	    offsetof(wan_x25_conf_t, n2), DTYPE_UINT },
  { "T10_T20", 	    offsetof(wan_x25_conf_t, t10_t20), DTYPE_UINT },
  { "T11_T21", 	    offsetof(wan_x25_conf_t, t11_t21), DTYPE_UINT },	
  { "T12_T22", 	    offsetof(wan_x25_conf_t, t12_t22), DTYPE_UINT },
  { "T13_T23", 	    offsetof(wan_x25_conf_t, t13_t23), DTYPE_UINT },
  { "T16_T26", 	    offsetof(wan_x25_conf_t, t16_t26), DTYPE_UINT },
  { "T28", 	    offsetof(wan_x25_conf_t, t28), DTYPE_UINT },
  { "R10_R20", 	    offsetof(wan_x25_conf_t, r10_r20), DTYPE_UINT },
  { "R12_R22", 	    offsetof(wan_x25_conf_t, r12_r22), DTYPE_UINT },
  { "R13_R23", 	    offsetof(wan_x25_conf_t, r13_r23), DTYPE_UINT },
  { NULL, 0, 0 }
};

key_word_t chan_conftab[] =	/* Channel configuration parameters */
{
  { "IDLETIMEOUT",   	offsetof(wanif_conf_t, idle_timeout), 	DTYPE_UINT },
  { "HOLDTIMEOUT",   	offsetof(wanif_conf_t, hold_timeout), 	DTYPE_UINT },
  { "X25_SRC_ADDR",   	offsetof(wanif_conf_t, x25_src_addr), 	DTYPE_STR},
  { "X25_ACCEPT_DST_ADDR",  offsetof(wanif_conf_t, accept_dest_addr), DTYPE_STR},
  { "X25_ACCEPT_SRC_ADDR",  offsetof(wanif_conf_t, accept_src_addr),  DTYPE_STR},
  { "X25_ACCEPT_USR_DATA",  offsetof(wanif_conf_t, accept_usr_data),  DTYPE_STR},
  { "CIR",           	offsetof(wanif_conf_t, cir), 	   	DTYPE_UINT },
  { "BC",            	offsetof(wanif_conf_t, bc),		DTYPE_UINT },
  { "BE", 	     	offsetof(wanif_conf_t, be),		DTYPE_UINT },
  { "MULTICAST",     	offsetof(wanif_conf_t, mc),		DTYPE_UCHAR},
  { "IPX",	     	offsetof(wanif_conf_t, enable_IPX),	DTYPE_UCHAR},
  { "NETWORK",       	offsetof(wanif_conf_t, network_number),	DTYPE_ULONG}, 
  { "PAP",     	     	offsetof(wanif_conf_t, pap),		DTYPE_UCHAR},
  { "CHAP",          	offsetof(wanif_conf_t, chap),		DTYPE_UCHAR},
  { "USERID",        	offsetof(wanif_conf_t, userid),	 	DTYPE_STR},
  { "PASSWD",        	offsetof(wanif_conf_t, passwd),		DTYPE_STR},
  { "SYSNAME",       	offsetof(wanif_conf_t, sysname),		DTYPE_STR},
  { "INARP", 	     	offsetof(wanif_conf_t, inarp),          	DTYPE_UCHAR},
  { "INARPINTERVAL", 	offsetof(wanif_conf_t, inarp_interval), 	DTYPE_UINT },
  { "INARP_RX",      	offsetof(wanif_conf_t, inarp_rx),          	DTYPE_UCHAR},
  { "IGNORE_DCD",  	offsetof(wanif_conf_t, ignore_dcd),        	DTYPE_UCHAR},
  { "IGNORE_CTS",    	offsetof(wanif_conf_t, ignore_cts),        	DTYPE_UCHAR},
  { "IGNORE_KEEPALIVE", offsetof(wanif_conf_t, ignore_keepalive), 	DTYPE_UCHAR},
  { "HDLC_STREAMING", 	offsetof(wanif_conf_t, hdlc_streaming), 	DTYPE_UCHAR},
  { "KEEPALIVE_TX_TIMER",	offsetof(wanif_conf_t, keepalive_tx_tmr), 	DTYPE_UINT },
  { "KEEPALIVE_RX_TIMER",	offsetof(wanif_conf_t, keepalive_rx_tmr), 	DTYPE_UINT },
  { "KEEPALIVE_ERR_MARGIN",	offsetof(wanif_conf_t, keepalive_err_margin),	DTYPE_UINT },
  { "SLARP_TIMER", 	offsetof(wanif_conf_t, slarp_timer),    DTYPE_UINT },
  { "TTL",        	offsetof(wanif_conf_t, ttl),         DTYPE_UCHAR },
  { "INTERFACE",  	offsetof(wanif_conf_t, interface),   DTYPE_UCHAR },
  { "CLOCKING",   	offsetof(wanif_conf_t, clocking),    DTYPE_UCHAR },
  { "BAUDRATE",   	offsetof(wanif_conf_t, bps),         DTYPE_UINT },
  { "MTU",        	offsetof(wanif_conf_t, mtu),         DTYPE_UINT },
  { "DYN_INTR_CFG",  	offsetof(wanif_conf_t, if_down),     DTYPE_UCHAR },
  { "GATEWAY",  	offsetof(wanif_conf_t, gateway),     DTYPE_UCHAR },
  { "TRUE_ENCODING_TYPE", offsetof(wanif_conf_t,true_if_encoding), DTYPE_UCHAR },

  /* ASYNC Options */
  { "ASYNC_MODE",    	       offsetof(wanif_conf_t, async_mode), DTYPE_UCHAR},	
  { "ASY_DATA_TRANSPARENT",    offsetof(wanif_conf_t, asy_data_trans), DTYPE_UCHAR}, 
  { "RTS_HS_FOR_RECEIVE",      offsetof(wanif_conf_t, rts_hs_for_receive), DTYPE_UCHAR},
  { "XON_XOFF_HS_FOR_RECEIVE", offsetof(wanif_conf_t, xon_xoff_hs_for_receive), DTYPE_UCHAR},
  { "XON_XOFF_HS_FOR_TRANSMIT",offsetof(wanif_conf_t, xon_xoff_hs_for_transmit), DTYPE_UCHAR},
  { "DCD_HS_FOR_TRANSMIT",     offsetof(wanif_conf_t, dcd_hs_for_transmit), DTYPE_UCHAR},	
  { "CTS_HS_FOR_TRANSMIT",     offsetof(wanif_conf_t, cts_hs_for_transmit), DTYPE_UCHAR},
  { "TX_BITS_PER_CHAR",        offsetof(wanif_conf_t, tx_bits_per_char),    DTYPE_UINT },
  { "RX_BITS_PER_CHAR",        offsetof(wanif_conf_t, rx_bits_per_char),    DTYPE_UINT },
  { "STOP_BITS",               offsetof(wanif_conf_t, stop_bits),    DTYPE_UINT },
  { "PARITY",                  offsetof(wanif_conf_t, parity),    DTYPE_UCHAR },
  { "BREAK_TIMER",             offsetof(wanif_conf_t, break_timer),    DTYPE_UINT },	
  { "INTER_CHAR_TIMER",        offsetof(wanif_conf_t, inter_char_timer),    DTYPE_UINT },
  { "RX_COMPLETE_LENGTH",      offsetof(wanif_conf_t, rx_complete_length),    DTYPE_UINT },
  { "XON_CHAR",                offsetof(wanif_conf_t, xon_char),    DTYPE_UINT },	
  { "XOFF_CHAR",               offsetof(wanif_conf_t, xoff_char),    DTYPE_UINT },	   
  { "MPPP_PROT",	       offsetof(wanif_conf_t, protocol),  DTYPE_UCHAR},
  { NULL, 0, 0 }
};

look_up_t conf_def_tables[] =
{
	{ WANCONFIG_PPP,	ppp_conftab	},
	{ WANCONFIG_FR,		fr_conftab	},
	{ WANCONFIG_X25,	x25_conftab	},
	{ WANCONFIG_CHDLC,	chdlc_conftab	},
	{ 0,			NULL		}
};

look_up_t	config_id_str[] =
{
	{ WANCONFIG_PPP,	"WAN_PPP"	},
	{ WANCONFIG_FR,		"WAN_FR"	},
	{ WANCONFIG_X25,	"WAN_X25"	},
	{ WANCONFIG_CHDLC,	"WAN_CHDLC"	},
        { WANCONFIG_BSC,        "WAN_BSC"       },
        { WANCONFIG_HDLC,       "WAN_HDLC"      },
	{ WANCONFIG_MPPP,       "WAN_MULTPPP"   },
	{ WANCONFIG_BITSTRM, 	"WAN_BITSTRM"	},
	{ WANCONFIG_EDUKIT, 	"WAN_EDU_KIT"	},
	{ 0,			NULL,		}
};

/*
 * Configuration options values and their symbolic names.
 */
look_up_t	sym_table[] =
{
	/*----- General -----------------------*/
	{ WANOPT_OFF,		"OFF"		}, 
	{ WANOPT_ON,		"ON"		}, 
	{ WANOPT_NO,		"NO"		}, 
	{ WANOPT_YES,		"YES"		}, 
	/*----- Interface type ----------------*/
	{ WANOPT_RS232,		"RS232"		},
	{ WANOPT_V35,		"V35"		},
	/*----- Data encoding -----------------*/
	{ WANOPT_NRZ,		"NRZ"		}, 
	{ WANOPT_NRZI,		"NRZI"		}, 
	{ WANOPT_FM0,		"FM0"		}, 
	{ WANOPT_FM1,		"FM1"		}, 
	/*----- Idle Line ----------------------*/
	{ WANOPT_IDLE_FLAG,	"FLAG"	}, 
	{ WANOPT_IDLE_MARK,	"MARK"	},
	/*----- Link type ---------------------*/
	{ WANOPT_POINTTOPOINT,	"POINTTOPOINT"	},
	{ WANOPT_MULTIDROP,	"MULTIDROP"	},
	/*----- Clocking ----------------------*/
	{ WANOPT_EXTERNAL,	"EXTERNAL"	}, 
	{ WANOPT_INTERNAL,	"INTERNAL"	}, 
	/*----- Station -----------------------*/
	{ WANOPT_DTE,		"DTE"		}, 
	{ WANOPT_DCE,		"DCE"		}, 
	{ WANOPT_CPE,		"CPE"		}, 
	{ WANOPT_NODE,		"NODE"		}, 
	{ WANOPT_SECONDARY,	"SECONDARY"	}, 
	{ WANOPT_PRIMARY,	"PRIMARY"	}, 
	/*----- Connection options ------------*/
	{ WANOPT_PERMANENT,	"PERMANENT"	}, 
	{ WANOPT_SWITCHED,	"SWITCHED"	}, 
	{ WANOPT_ONDEMAND,	"ONDEMAND"	}, 
	/*----- Frame relay in-channel signalling */
	{ WANOPT_FR_ANSI,	"ANSI"		}, 
	{ WANOPT_FR_Q933,	"Q933"		}, 
	{ WANOPT_FR_LMI,	"LMI"		}, 
	/*----- PPP IP Mode Options -----------*/
	{ WANOPT_PPP_STATIC,	"STATIC"	}, 
	{ WANOPT_PPP_HOST,	"HOST"		}, 
	{ WANOPT_PPP_PEER,	"PEER"		}, 
	/*----- CHDLC Protocol Options --------*/
/* DF for now	{ WANOPT_CHDLC_NO_DCD,	"IGNORE_DCD"	},
	{ WANOPT_CHDLC_NO_CTS,	"IGNORE_CTS"	}, 
	{ WANOPT_CHDLC_NO_KEEP,	"IGNORE_KEEPALIVE"}, 
*/
	{ WANOPT_PRI,           "PRI"           },
	{ WANOPT_SEC,           "SEC"           },
	
        /*------Read Mode ---------------------*/
        { WANOPT_INTR,          "INT"           },
        { WANOPT_POLL,          "POLL"          },
	/*------- Async Options ---------------*/
	{ WANOPT_ONE,           "ONE"          	},
	{ WANOPT_TWO,           "TWO"          	},
	{ WANOPT_ONE_AND_HALF,  "ONE_AND_HALF" 	},
	{ WANOPT_NONE,          "NONE"    	},
	{ WANOPT_EVEN,          "EVEN"    	},
	{ WANOPT_ODD,           "ODD"    	},

	{ WANOPT_TTY_SYNC,	"SYNC"		},
	{ WANOPT_TTY_ASYNC,     "ASYNC"		},
	/* TE1 T1/E1 defines */
        /*------TE Cofngiuration --------------*/
        { WANOPT_MEDIA_T1,      "T1"            },
        { WANOPT_MEDIA_E1,      "E1"            },
	{ WANOPT_MEDIA_56K,     "56K"           },
        { WANOPT_LC_AMI,     	"AMI"           },
        { WANOPT_LC_B8ZS,       "B8ZS"          },
        { WANOPT_LC_HDB3,       "HDB3"          },
        { WANOPT_FR_D4,         "D4"            },
        { WANOPT_FR_ESF,        "ESF"           },
        { WANOPT_FR_NCRC4,      "NCRC4"         },
        { WANOPT_FR_CRC4,       "CRC4"          },
        { WANOPT_T1_LBO_0_DB,  	"0DB"           },
        { WANOPT_T1_LBO_75_DB,  "7.5DB"         },
        { WANOPT_T1_LBO_15_DB,  "15DB"          },
        { WANOPT_T1_LBO_225_DB, "22.5DB"        },
        { WANOPT_T1_0_110,  	"0-110FT"       },
        { WANOPT_T1_110_220,  	"110-220FT"     },
        { WANOPT_T1_220_330,  	"220-330FT"     },
        { WANOPT_T1_330_440,  	"330-440FT"     },
        { WANOPT_T1_440_550,  	"440-550FT"     },
        { WANOPT_T1_550_660,  	"550-660FT"     },
        { WANOPT_NORMAL_CLK,   	"NORMAL"        },
        { WANOPT_MASTER_CLK,   	"MASTER"        },
	
	{ WANOPT_NO,		"MP_PPP"	}, 
	{ WANOPT_YES,		"MP_CHDLC"	}, 

	
	/*----- End ---------------------------*/
	{ 0,			NULL		}, 
};

/****** Entry Point *********************************************************/

int main (int argc, char *argv[])
{
	int err = 0;	/* return code */
	int c;
	
	/* Process command line switches */
	action = DO_SHOW_STATUS;
	while(1) {
		c = getopt(argc, argv, "hvwUf:y:zd:y:z;V");

		if( c == -1)
			break;

		switch(c) {
	

		case 'd':
			conf_file = optarg;
			set_action(DO_STOP);
			break;
	
		case 'f':
			conf_file = optarg;
			set_action(DO_START);
			break;

		case 'v':
			verbose = 1;
			break;
		
		case 'w':
			weanie_flag = 1;
			break;
			
		case 'U':
			set_action(DO_ARG_CONFIG);
			break;
				
		case 'y':
			verbose_log = optarg;
			break;
			
		case 'z':
			krnl_log_file = optarg;
			break;
			
		case 'h':
			show_help();
			break;
		case 'V':
			printf("wanconfig version: %s\n\n",WANPIPE_VERSION);
			show_usage();
			break;

		case '?':
		case ':':
		default:
			show_usage();
			break;
		}	
	}
	
	/* Check that the no action is given with non-swtiched arguments 
	 */
	
	if( action != DO_SHOW_STATUS && action != DO_ARG_CONFIG ) {
		if(optind < argc)
			show_usage();
		
	}

	/* Process interface control command arguments */
	if( action == DO_SHOW_STATUS && optind  < argc ) {
		
		for( ; optind < argc; optind++ ){
			
			if( strcmp( argv[optind], "wan" ) == 0 \
			 	|| strcmp( argv[optind], "card") == 0 ) {
				
				/* Check that card name arg is given */
				if((optind+1) >= argc ) show_usage();
			
				card_name = argv[optind+1];
				optind++;
			}
			
			else if( strcmp( argv[optind], "dev" ) == 0 ) {
				
				/* Make sure nodev argument is not given */
				if(nodev_flag) show_usage();
				
				/* Make sure card argument is given */
				if(card_name == NULL) show_usage();
				
				/* Check that interface name arg is given */
				if((optind+1) >= argc ) show_usage(); 

				dev_name = argv[optind+1];
				optind++;
			}
			
			else if( strcmp( argv[optind], "nodev" ) == 0 ) {
				
				/* Make sure dev argument is not given */
				if(dev_name != NULL) show_usage();
				
				nodev_flag = 1;
			}
			
			else if( strcmp( argv[optind], "start" ) == 0 
				|| strcmp( argv[optind], "up") == 0
				|| strcmp( argv[optind], "add") == 0 ) {
				set_action(DO_START);
			}
			
			else if( strcmp( argv[optind], "stop" ) == 0
				|| strcmp( argv[optind], "del" ) == 0	
				|| strcmp( argv[optind], "delete" ) == 0	
				|| strcmp( argv[optind], "down") == 0 ) {
				set_action(DO_STOP);
			}
			
			else if( strcmp( argv[optind], "restart" ) == 0 
				|| strcmp( argv[optind], "reload") == 0 ) {
				set_action(DO_RESTART);
			}
			
			else if( strcmp( argv[optind], "status" ) == 0
				|| strcmp( argv[optind], "stat" ) == 0
				|| strcmp( argv[optind], "show" ) == 0 ) {
				set_action(DO_SHOW_STATUS);
			}
			
			else if( strcmp( argv[optind], "config" ) == 0 ) {
				set_action(DO_SHOW_CONFIG);
			}
			
			else if( strcmp( argv[optind], "help" ) == 0 ) {
				show_help();
			}
			
			else {
				show_usage();
			}
			
		}
	}

	/* Check length of names */
	if( card_name != NULL && strlen(card_name) > WAN_DRVNAME_SZ) {
	       	fprintf(stderr, "%s: WAN device names must be %d characters long or less\n",
			progname, WAN_DRVNAME_SZ);
		exit(1);
	}
		
	if( dev_name != NULL && strlen(dev_name) > WAN_IFNAME_SZ) {
	       	fprintf(stderr, "%s: Interface names must be %d characters long or less\n",
			progname, WAN_IFNAME_SZ);
		exit(1);
	}
 		
	/* Perform requested action */
	if (verbose) puts(banner);
	if (!err)
	switch (action) {
		char statbuf[80];

	case DO_START:
		err = startup();
		break;

	case DO_STOP:
		err = shutdown();
		break;

	case DO_RESTART:
		err = shutdown();
		if(err)
			break;
		sleep(SLEEP_TIME);
		err = startup();
		break;
		
	case DO_ARG_CONFIG:
		conf_file = "wanpipe_temp.conf";
		argc = argc - optind;
		argv = &argv[optind];
		err = get_config_data(argc,argv);
		if (err)
			break;
		err = configure();
		remove(conf_file);
		break;

	case DO_ARG_DOWN:
		printf("DO_ARG_DOWN\n");
		shutdown();
		break;

	case DO_HELP:
		show_help();
		break;
	
	case DO_SHOW_STATUS:
		if( card_name != NULL) {
			snprintf(statbuf, sizeof(statbuf), "%s/%s", router_dir, card_name);
			gencat(statbuf);
		}
		else {
			snprintf(statbuf, sizeof(statbuf), "%s/%s", router_dir, "status");
			gencat(statbuf);
		}
		exit(0);
		break;
		
	case DO_SHOW_CONFIG:
		snprintf(statbuf, sizeof(statbuf), "%s/%s", router_dir, "config");
		gencat(statbuf);
		exit(0);
		break;

	default:
		err = ERR_SYNTAX;
		break;
	}

	
	/* Always a good idea to sleep a bit - easier on system and link
	 * we could be operating 2 cards and 1st one improperly configured
	 * so dwell a little to make things easier before wanconfig reinvoked.
	 */
	if( !err || err == ERR_SYSTEM ) sleep(SLEEP_TIME);
	if (err == ERR_SYNTAX) fprintf(stderr, usagetext);
	return err;
}

/*============================================================================
 * Show help text
 */
void show_help(void) {
	puts(helptext);
	exit(1);
}

/*============================================================================
 * Show usage text
 */
void show_usage(void) {
	fprintf(stderr, usagetext);
	exit(1);
}

/*============================================================================
 * set action
 */
void set_action(int new_action) {
	if(action == DO_SHOW_STATUS ) 
		action = new_action;
	else
		show_usage();
}

/*============================================================================
 * Show error message.
 */
void show_error (int err)
{
	if (err == ERR_SYSTEM){ 
		fprintf(stderr, "%s: SYSTEM ERROR %d: %s!\n",
	 		progname, errno, strerror(errno));
	}else{
		fprintf(stderr, "%s: ERROR: %s : %s!\n", progname,
			err_messages[min(err, ERR_LIMIT) - 2], conf_file);
	}
}

/*============================================================================
 * cat a given  file to stdout - this is used for status and config
 */
int gencat (char *filename) 
{
	FILE *file;
	char buf[256];
	
	file = fopen(filename, "r");
	if( file == NULL ) {
		fprintf( stderr, "%s: cannot open %s\n",
				progname, filename);
		show_error(ERR_SYSTEM);
		exit(ERR_SYSTEM);
	}
	
	while(fgets(buf, sizeof(buf) -1, file)) {
		printf(buf);
	}	
	
	return(0);
}

/*============================================================================
 * Start up - This is a shell function to select the configuration source
 */
int startup(void)
{
	char buf[256];
	int err = 0;
	int res = 0;
        int found = 0;	
	DIR *dir;
	struct dirent *dentry;
	struct stat file_stat;

	if( conf_file != NULL ) {
		/* Method I. Observe given config file name
		 */
		return(configure());
	}
	else {
     		/* Method II. Scan /proc/net/wanrouter for device names
     		 */
		
     		/* Open directory */
                dir = opendir(router_dir);

                if(dir == NULL) {
	                        err = ERR_SYSTEM;
	                        show_error(err);
	                        return(err);
	                }


		while( (dentry = readdir(dir)) ) {
	
	                        if( strlen(dentry->d_name) > WAN_DRVNAME_SZ)
	                                continue;
	
	               		/* skip directory dots */
		       		if( strcmp(dentry->d_name, ".") == 0 \
					|| strcmp(dentry->d_name, "..") == 0)
	                                continue;

				/* skip /prroc/net/wanrouter/{status,config} */
	                        if( strcmp(dentry->d_name, "status") == 0 \
					|| strcmp(dentry->d_name, "config") == 0)
	                                continue;
				
				/* skip interfaces we are not configuring */
				if( card_name != NULL && strcmp(dentry->d_name, card_name) != 0 )
					continue;
				
				snprintf(buf, sizeof(buf), "%s/%s.conf", conf_dir, dentry->d_name);
				
				/* Stat config file to see if it is there */
				res = stat(buf, &file_stat);
				if( res < 0 ) {
					if( errno == ENOENT )
						continue;
					
					show_error(ERR_SYSTEM);
					exit(ERR_SYSTEM);
				}			

				found = 1;
				conf_file = buf;	
	                        err = configure();
	
		}

	closedir(dir);

}

if( !found ) {
	/* Method III. Use default configuration file
	*/
	conf_file = def_conf_file;
	err = configure();
}
return(err);
}

/*============================================================================
* Shut down link.
*/
int shutdown (void)
{
	int err = 0;
	DIR *dir;
	struct dirent *dentry;

	if (conf_file){
		err = conf_file_down();

	}else if( card_name != NULL && dev_name == NULL ) {
		err = router_down(card_name, 0);

	}else if( card_name != NULL && dev_name != NULL ) {
		err = router_ifdel(card_name, dev_name);
	}

	else {
		/* Open directory */
		dir = opendir(router_dir);

		if(dir == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
			return(err);
		}

		while( (dentry = readdir(dir)) ) {
			int res = 0;
			
			if( strlen(dentry->d_name) > WAN_DRVNAME_SZ)
				continue;

			if( strcmp(dentry->d_name, ".") == 0 \
				|| strcmp(dentry->d_name, "..") == 0)
				continue;

			res = router_down(dentry->d_name, 1);	
			if(res) 
				err = res;
		}
        	
		closedir(dir);
	
	}
		
	return err;

}


int router_down (char *devname, int ignore_error)
{
	char filename[sizeof(router_dir) + WAN_DRVNAME_SZ + 2];
        int dev, err=0;

	if (verbose)
        	printf(" * Shutting down WAN device %s ...\n", devname);                
	if (strlen(devname) > WAN_DRVNAME_SZ) {
                show_error(ERR_SYNTAX);
                return ERR_SYNTAX;
	}

        snprintf(filename, sizeof(filename), "%s/%s", router_dir, devname);
        dev = open(filename, O_RDONLY);
        if ((dev < 0) || (ioctl(dev, ROUTER_DOWN, NULL) < 0 
				&& ( !ignore_error || errno != ENOTTY))) {
		err = ERR_SYSTEM;
		fprintf(stderr, "\n\n\t%s: WAN device %s did not shutdown\n", progname, devname);
		fprintf(stderr, "\t%s: ioctl(%s,ROUTER_DOWN) failed:\n", 
				progname_sp, devname);
		fprintf(stderr, "\t%s:\t%d - %s\n", progname_sp, errno, strerror(errno));
		if( weanie_flag ) {
			fprintf(stderr, "\n\tIf you router was not running ignore this message\n !!");
			fprintf(stderr, "\tOtherwise, check the %s and \n", verbose_log);
			fprintf(stderr, "\t%s for errors\n", krnl_log_file);
		}
 		fprintf( stderr, "\n" );
 		if(dev >=0) close (dev);
 		return err;
       	}

        if (dev >= 0) close(dev);
	return 0;

}


int router_ifdel (char *card_name, char *dev_name)
{
	char filename[sizeof(router_dir) + WAN_DRVNAME_SZ + 2];
	char devicename[WAN_IFNAME_SZ +2];
	int dev, err=0;

	if (verbose)
        	printf(" * Shutting down WAN device %s ...\n", card_name);                
	if (strlen(card_name) > WAN_DRVNAME_SZ) {
                show_error(ERR_SYNTAX);
                return ERR_SYNTAX;
	}
	snprintf(filename, sizeof(filename), "%s/%s", router_dir, card_name);
	if (strlen(dev_name) > WAN_IFNAME_SZ) {
		show_error(ERR_SYNTAX);
		return ERR_SYNTAX;
	}
	snprintf(devicename, sizeof(devicename), "%s", dev_name);
	
        dev = open(filename, O_RDONLY);
        if ((dev < 0) || (ioctl(dev, ROUTER_IFDEL, devicename) < 0)) {
		err = ERR_SYSTEM;
		fprintf(stderr, "\n\n\t%s: Interface %s not destroyed\n", progname, devicename);
		fprintf(stderr, "\t%s: ioctl(%s,ROUTER_IFDEL,%s) failed:\n", 
				progname_sp, card_name, devicename);
		fprintf(stderr, "\t%s:\t%d - %s\n", progname_sp, errno, strerror(errno));
		if( weanie_flag ) {
			fprintf(stderr, "\n\tIf you router was not running ignore this message\n !!");
			fprintf(stderr, "\tOtherwise, check the %s and \n", verbose_log);
			fprintf(stderr, "\t%s for errors\n", krnl_log_file);
		}
 		fprintf( stderr, "\n" );
 		if(dev >=0) close (dev);
                return err;
       	}

        if (dev >= 0) close(dev);
	return 0;

}

/*============================================================================
 * Configure router.
 * o parse configuration file
 * o configure links
 * o configure logical channels (interfaces)
 */
int configure (void)
{
	int err = 0;
	link_def_t* linkdef;

	/* Parse configuration file */
	if (verbose)
		printf(" * Parsing configuration file %s ...\n", conf_file);

	err = parse_conf_file(conf_file);
	if (err) return err;

	if (link_defs == NULL) {
		if (verbose) printf(" * No link definitions found...\n");
		return 0;
	}

	/* Configure links */
	for (linkdef = link_defs; linkdef; linkdef = linkdef->next) {
		int res;

		if( card_name != NULL && strcmp(card_name, linkdef->name) != 0 )
			continue;

		if (verbose) printf(
			" * Configuring device %s (%s)\n",
			linkdef->name,
			linkdef->descr ? linkdef->descr : "no description");
		res = configure_link(linkdef);
		if (res) {
			return res;
		}
	}

	/* Clear definition list */
	while (link_defs != NULL) {
		link_def_t* linkdef = link_defs;

		while (linkdef->chan) {
			chan_def_t* chandef = linkdef->chan;

			if (chandef->conf) free(chandef->conf);
			if (chandef->addr) free(chandef->addr);
			if (chandef->usedby) free(chandef->usedby);
			if (chandef->descr) free(chandef->descr);
			linkdef->chan = chandef->next;
			free(chandef);
		}
		if (linkdef->conf) free(linkdef->conf);
		if (linkdef->descr) free(linkdef->descr);
		link_defs = linkdef->next;
		free(linkdef);
	}
	return err;
}

/*============================================================================
 * Parse configuration file.
 *	Read configuration file and create lists of link and channel
 *	definitions.
 */
int parse_conf_file (char* fname)
{
	int err = 0;
	FILE* file;

	file = fopen(fname, "r");
	if (file == NULL) {
		fprintf(stderr, "\nError: %s not found in %s directory\n",
				fname, conf_dir);
		show_error(ERR_SYSTEM);
		return ERR_SYSTEM;
	}

	/* Build a list of link and channel definitions */
	err = build_linkdef_list(file);
	if (!err && link_defs)
		err = build_chandef_list(file);
	fclose(file);
	return err;
}

/*============================================================================
 * Build a list of link definitions.
 */
int build_linkdef_list (FILE* file)
{
	int err = 0;
	char* conf;		/* -> section buffer */
	char* key;		/* -> current configuration record */
	int len;		/* record length */

	/* Read [devices] section */
	conf = read_conf_section(file, "devices");
	if (conf == NULL) return ERR_CONFIG;

	/* For each record in [devices] section create a link definition
	 * structure and link it to the list of link definitions.
	 */
	for (key = conf; !err && *key; key += len) {
		int toknum;
		char* token[MAX_TOKENS];
		link_def_t* linkdef;	/* -> link definition */
		int config_id = 0;

		len = strlen(key) + 1;
		toknum = tokenize(key, token);
		if (toknum < 2) continue;

		strupcase(token[1]);
		config_id = name2val(token[1], config_id_str);
		if (!config_id) {
			if (verbose) printf(
				" * Media ID %s is invalid!\n", token[1]);
			err = ERR_CONFIG;
			show_error(err);
			break;
		}
		linkdef = calloc(1, sizeof(link_def_t));
		if (linkdef == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
			break;
		}

	 	strncpy(linkdef->name, token[0], WAN_DRVNAME_SZ);
		linkdef->config_id = config_id;
	if ((toknum > 2) && token[2])
		linkdef->descr = strdup(token[2]);

		linkdef->conf = read_conf_section(file, linkdef->name);
                if (link_defs) {
			link_def_t* last;
			for (last = link_defs; last->next; last = last->next);
			last->next = linkdef;
		}
		else link_defs = linkdef;
	}
	free(conf);
	return err;
}

/*============================================================================
 * Build a list of channel definitions.
 */
int build_chandef_list (FILE* file)
{
	int err = 0;
	char* conf;		/* -> section buffer */
	char* key;		/* -> current configuration record */
	int len;		/* record length */

	/* Read [interfaces] section */
	conf = read_conf_section(file, "interfaces");
	if (conf == NULL) return 0;

	/* For each record in [interfaces] section create channel definition
	 * structure and link it to the list of channel definitions for
	 * appropriate link.
	 */
	for (key = conf; !err && *key; key += len) {
		int toknum;
		char* token[MAX_TOKENS];
		chan_def_t* chandef;		/* -> channel definition */
		link_def_t* linkdef;		/* -> channel definition */

		len = strlen(key) + 1;
		toknum = tokenize(key, token);
		if (toknum < 2) continue;

		/* find a WAN link definition for this channel */
		for (linkdef = link_defs;
		     linkdef && strcmp(token[1], linkdef->name);
		     linkdef = linkdef->next)
		;
		if (linkdef == NULL) continue;

		/* allocate and initialize channel definition structure */
		chandef = calloc(1, sizeof(chan_def_t));
		if (chandef == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
			break;
		}
	 	strncpy(chandef->name, token[0], WAN_IFNAME_SZ);
		if ((toknum > 2) && token[2])
			chandef->addr = strdup(token[2]);
		if ((toknum > 3) && token[3])
			chandef->usedby = strdup(token[3]);

		if (!(chandef->usedby) ||(( strcmp(chandef->usedby, "WANPIPE")     != 0 ) &&
					  ( strcmp(chandef->usedby, "API")         != 0 ) && 
					  ( strcmp(chandef->usedby, "BRIDGE")      != 0 ) &&
					  ( strcmp(chandef->usedby, "BRIDGE_NODE") != 0 )))
		{

			if (verbose) printf(" * %s to be used by an Invalid entity: %s\n", 
									chandef->name, chandef->usedby);
			return ERR_CONFIG;
		}

		if ((toknum > 4) && token[4])
			chandef->descr = strdup(token[3]);

		if (verbose)
			printf(" * %s to used by %s\n",
				chandef->name, chandef->usedby);

		chandef->conf = read_conf_section(file, token[0]);

		/* append channel definition structure to the list */
		if (linkdef->chan) {
			chan_def_t* last;

			for (last = linkdef->chan;
			     last->next;
			     last = last->next)
			;
			last->next = chandef;
		}
		else linkdef->chan = chandef;
	}
	free(conf);
	return err;
}

/*============================================================================
 * Configure WAN link.
 */
int configure_link (link_def_t* def)
{
	int err = 0;
	int len = 0;
	int i;
	key_word_t* conf_table = lookup(def->config_id, conf_def_tables);
	char filename[sizeof(router_dir) + WAN_DRVNAME_SZ + 2];
	chan_def_t* chandef;
	char* conf_rec;
	int dev;
	unsigned short hi_DLCI = 15;

	if (def->conf == NULL) {
		show_error(ERR_CONFIG);
		return ERR_CONFIG;
	}

	/* Clear configuration structure */
	memset(&u.linkconf, 0, sizeof(u.linkconf));
	u.linkconf.magic = ROUTER_MAGIC;
	u.linkconf.config_id = def->config_id;

	/* Parse link configuration */
	for (conf_rec = def->conf; *conf_rec; conf_rec += len) {
		int toknum;
		char* tok[MAX_TOKENS];

		len = strlen(conf_rec) + 1;
		toknum = tokenize(conf_rec, tok);
		if (toknum < 2) {
			show_error(ERR_CONFIG);
			return ERR_CONFIG;
		}

		/* Look up a keyword first in common, then in media-specific
		 * configuration definition tables.
		 */
		strupcase(tok[0]);
		err = set_conf_param(
			tok[0], tok[1], common_conftab, &u.linkconf);
		if ((err < 0) && (conf_table != NULL))
			err = set_conf_param(
				tok[0], tok[1], conf_table, &u.linkconf.u);
		if (err < 0) {
			printf(" * Unknown parameter %s\n", tok[0]);
			fprintf(stderr, "\n\n\tERROR in %s !!\n",conf_file);
			if(weanie_flag) fprintf(stderr, "\tPlease check %s for errors\n", verbose_log);
			fprintf(stderr, "\n");
			return ERR_CONFIG;
		}
		if (err){ 
			fprintf(stderr,"\n\tError parsing %s configuration file\n",conf_file);
			if(weanie_flag) fprintf(stderr,"\tPlease check %s for errors\n", verbose_log);
			fprintf(stderr, "\n");
			return err;
		}
	}

	/* Open SDLA device and perform link configuration */
	sprintf(filename, "%s/%s", router_dir, def->name);
	
	/* prepare a list of DLCI(s) and place it in the wandev_conf_t structure
	 * This is done so that we have a list of DLCI(s) available when we 
	 * call the wpf_init() routine in sdla_fr.c (triggered by the ROUTER_
	 * SETUP ioctl call) for running global SET_DLCI_CONFIGURATION cmd. This
	 * is only relevant for a station being a NODE, for CPE it polls the 
	 * NODE (auto config DLCI)
	 */
	
	if ((u.linkconf.config_id == WANCONFIG_FR)){
	   i = 0;
	
	   if(!err) for (chandef = def->chan; chandef; chandef = chandef->next){
 		if (verbose) printf(
			" * Reading DLCI(s) Included : %s\n",
			chandef->addr ? chandef->addr : "not specified");
		u.linkconf.u.fr.dlci[i] = dec_to_uint(chandef->addr,0);
		if(u.linkconf.u.fr.dlci[i] > hi_DLCI) {
			hi_DLCI = u.linkconf.u.fr.dlci[i];
			i++;
		}
		else {
			fprintf(stderr,"\n\tDLCI(s) specified in %s are not in order\n",conf_file); 
			fprintf(stderr,"\n\tor duplicate DLCI(s) assigned!\n");
			err = ERR_SYSTEM;
		}
		
	    }
	}

	dev = open(filename, O_RDONLY);

	{
		char wanpipe_version[50];
		memset(wanpipe_version,0,sizeof(wanpipe_version));
		if ((dev < 0) || (ioctl(dev, ROUTER_VER, wanpipe_version))){
			fprintf(stderr, "\n\n\t%s: WAN device %s driver load failed !!\n", progname, def->name);
			fprintf(stderr, "\t%s: ioctl(%s,ROUTER_VER) failed:\n", 
				progname_sp, def->name);
			fprintf(stderr, "\t%s:\t%d - %s\n", progname_sp, errno, strerror(errno));

			fprintf(stderr, "\n\t%s: Wanpipe version mismatch between utilites\n",
					progname_sp);
			fprintf(stderr, "\t%s: and kernel drivers:\n",
					progname_sp);
			fprintf(stderr, "\t%s: \tUtil Ver=%s, Driver Ver=%s\n",
					progname_sp,WANPIPE_VERSION,wanpipe_version);
	
			fprintf(stderr, "\n");
	 		if(dev >=0) close (dev);
			return ERR_SYSTEM;

		}

		if (strcmp(wanpipe_version,WANPIPE_VERSION)){
			fprintf(stderr, "\n\n\t%s: WAN device %s driver load failed !!\n", progname, def->name);
			fprintf(stderr, "\n\t%s: Wanpipe version mismatch between utilites\n",
					progname_sp);
			fprintf(stderr, "\t%s: and kernel drivers:\n",
					progname_sp);
			fprintf(stderr, "\t%s: \tUtil Ver=%s, Driver Ver=%s\n",
					progname_sp,WANPIPE_VERSION,wanpipe_version);
		
			fprintf(stderr, "\n");
 			if(dev >=0) close (dev);
			return ERR_SYSTEM;
		}
	}

	
	if (strstr(conf_file, "ft1.conf") != NULL){
		u.linkconf.ft1 = 1;
	}else{
		u.linkconf.ft1 = 0;
	}

	if ((dev < 0) || (ioctl(dev, ROUTER_SETUP, &u.linkconf) < 0
				&& ( dev_name == NULL || errno != EBUSY ))) {
                fprintf(stderr, "\n\n\t%s: WAN device %s driver load failed !!\n", progname, def->name);
		fprintf(stderr, "\t%s: ioctl(%s,ROUTER_SETUP) failed:\n", 
				progname_sp, def->name);
		fprintf(stderr, "\t%s:\t%d - %s\n", progname_sp, errno, strerror(errno));
		if( weanie_flag ) {
			fprintf(stderr, "\n\tWanpipe driver did not load properly\n");
			fprintf(stderr, "\tPlease check %s and\n", verbose_log);
			fprintf(stderr, "\t%s for errors\n", krnl_log_file); 
		}
		fprintf(stderr, "\n");
 		if(dev >=0) close (dev);
		return ERR_SYSTEM;
	}
	if (u.linkconf.data) free(u.linkconf.data);

	/* Configure logical channels */
	if( !err && !nodev_flag ) 
		for (chandef = def->chan; chandef; chandef = chandef->next) {

		if( dev_name != NULL && strcmp(dev_name, chandef->name) != 0 )
			continue;
		
		if (verbose) printf(
			" * Configuring channel %s (%s). Media address: %s\n",
			chandef->name,
			chandef->descr ? chandef->descr : "no description",
			chandef->addr ? chandef->addr : "not specified");
		memset(&u.chanconf, 0, sizeof(u.chanconf));
		u.chanconf.magic = ROUTER_MAGIC;
		u.chanconf.config_id = def->config_id;
		err=configure_chan(dev, chandef);
		if (err){
			break;
		}
	}
	/* clean up */
	if (dev > 0) close(dev);
	return err;
}

/*============================================================================
 * Configure WAN logical channel.
 */
int configure_chan (int dev, chan_def_t* def)
{
	int err = 0;
	int len = 0;
	char* conf_rec;

	/* Prepare configuration data */
	strncpy(u.chanconf.name, def->name, WAN_IFNAME_SZ);
	if (def->addr)
		strncpy(u.chanconf.addr, def->addr, WAN_ADDRESS_SZ);

	if (def->usedby)
		strncpy(u.chanconf.usedby, def->usedby, USED_BY_FIELD);

	if (def->conf) for (conf_rec = def->conf; *conf_rec; conf_rec += len) {
		int toknum;
		char* tok[MAX_TOKENS];

		len = strlen(conf_rec) + 1;
		toknum = tokenize(conf_rec, tok);
		if (toknum < 2) {
			show_error(ERR_CONFIG);
			return ERR_CONFIG;
		}

		/* Look up a keyword first in common, then media-specific
		 * configuration definition tables.
		 */
		strupcase(tok[0]);
		if (set_conf_param(tok[0], tok[1], chan_conftab, &u.chanconf)) {
			printf("Invalid parameter %s\n", tok[0]);
			show_error(ERR_CONFIG);
			return ERR_CONFIG;
		}
	}

	if (ioctl(dev, ROUTER_IFNEW, &u.chanconf) < 0) {
                fprintf(stderr, "\n\n\t%s: Interface %s setup failed\n", progname, def->name);
                fprintf(stderr, "\t%s: ioctl(ROUTER_IFNEW,%s) failed:\n", 
				progname_sp, def->name);
		fprintf(stderr, "\t%s:\t%d - %s\n", progname_sp, errno, strerror(errno));
		if( weanie_flag ) {
			fprintf(stderr, "\n\tWanpipe drivers could not setup network interface\n");
			fprintf(stderr, "\tPlease check %s and\n", verbose_log);
			fprintf(stderr, "\t%s for errors\n", krnl_log_file); 
		}
		fprintf(stderr, "\n");
		return ERR_SYSTEM;
	}
	return err;
}

/*============================================================================
 * Set configuration parameter.
 *	Look up a keyword in configuration description table to find offset
 *	and data type of the configuration parameter into configuration data
 *	buffer.  Convert parameter to apporriate data type and store it the
 *	configuration data buffer.
 *
 *	Return:	0 - success
 *		1 - error
 *		-1 - not found
 */
int set_conf_param (char* key, char* val, key_word_t* dtab, void* conf)
{
	ulong tmp = 0;

	/* Search a keyword in configuration definition table */
	for (; dtab->keyword && strcmp(dtab->keyword, key); ++dtab);
	if (dtab->keyword == NULL) return -1;	/* keyword not found */

	/* Interpert parameter value according to its data type */

	if (dtab->dtype == DTYPE_FILENAME) {
		return read_data_file(
			val, (void*)((char*)conf + dtab->offset));
	}

	if (verbose) printf(" * Setting %s to %s\n", key, val);
	if (dtab->dtype == DTYPE_STR) {
		strcpy((char*)conf + dtab->offset, val);
		return 0;
	}

	if (!isdigit(*val) || 
	    strcmp(key, "ACTIVE_CH") == 0 ||
	    strcmp(key, "LBO") == 0 ||
	    strcmp(key, "MEDIA") == 0) {
		look_up_t* sym;

		strupcase(val);
		for (sym = sym_table;
		     sym->ptr && strcmp(sym->ptr, val);
		     ++sym)
		;
		if (sym->ptr == NULL) {
			/* TE1 */
			if (strcmp(key, "ACTIVE_CH")) {		
				if (verbose) printf(" * invalid term %s ...\n", val);
				return 1;
			}
			/* TE1 Convert active channel string to ULONG */
		 	tmp = parse_active_channel(val);	
		}
		else tmp = sym->val;
	}
	else tmp = strtoul(val, NULL, 0);

	switch (dtab->dtype) {

	case DTYPE_INT:
		*(int*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_UINT:
		*(uint*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_LONG:
		*(long*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_ULONG:
		*(ulong*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_SHORT:
		*(short*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_USHORT:
		*(ushort*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_CHAR:
		*(char*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_UCHAR:
		*(unsigned char*)((char*)conf + dtab->offset) = tmp;
		break;

	case DTYPE_PTR:
		*(void**)((char*)conf + dtab->offset) = (void*)tmp;
		break;
	}
	return 0;
}

/*============================================================================
 * Read configuration file section.
 *	Return a pointer to memory filled with key entries read from given
 *	section or NULL if section is not found. Each key is a zero-terminated
 *	ASCII string with no leading spaces and comments. The last key is
 *	followed by an empty string.
 *	Note:	memory is allocated dynamically and must be released by the
 *		caller.
 */
char* read_conf_section (FILE* file, char* section)
{
	char key[MAX_CFGLINE];		/* key buffer */
	char* buf = NULL;
	int found = 0, offs = 0, len;

	rewind(file);
	while ((len = read_conf_record(file, key)) > 0) {
		char* tmp;

		if (found) {
			if (*key == '[') break;	/* end of section */

			if (buf) tmp = realloc(buf, offs + len + 1);
			else tmp = malloc(len + 1);
			if (tmp) {
				buf = tmp;
				strcpy(&buf[offs], key);
				offs += len;
				buf[offs] = '\0';
			}
			else {	/* allocation failed */
				show_error(ERR_SYSTEM);
 				break;
			}
		}
		else if (*key == '[') {
			tmp = strchr(key, ']');
			if (tmp != NULL) {
				*tmp = '\0';
				if (strcmp(&key[1], section) == 0) {
 					if (verbose) printf(
						" * Reading section [%s]...\n",
						section);
					found = 1;
				}
			}
		}
	}
 	if (verbose && !found) printf(
		" * section [%s] not found!\n", section);
	return buf;
 }

/*============================================================================
 * Read a single record from the configuration file.
 *	Read configuration file stripping comments and leading spaces until
 *	non-empty line is read.  Copy it to the destination buffer.
 *
 * Return string length (incl. terminating zero) or 0 if end of file has been
 * reached.
 */
int read_conf_record (FILE* file, char* key)
{
	char buf[MAX_CFGLINE];		/* line buffer */

	while (fgets(buf, MAX_CFGLINE, file)) {
		char* str;
		int len;

		/* Strip leading spaces and comments */
		for (str = buf; *str && strchr(" \t", *str); ++str);
		len = strcspn(str, "#;\n\r");
		if (len) {
			str[len] = '\0';
			strcpy(key, str);
			return len + 1;
		}
	}
	return 0;
}

/*============================================================================
 * Tokenize string.
 *	Parse a string of the following syntax:
 *		<tag>=<arg1>,<arg2>,...
 *	and fill array of tokens with pointers to string elements.
 *
 *	Return number of tokens.
 */
int tokenize (char* str, char **tokens)
{
	int cnt = 0;

	tokens[0] = strtok(str, "=");
	while (tokens[cnt] && (cnt < MAX_TOKENS - 1)) {
		tokens[cnt] = strstrip(tokens[cnt], " \t");
		tokens[++cnt] = strtok(NULL, ",");
	}
	return cnt;
}

/*============================================================================
 * Strip leading and trailing spaces off the string str.
 */
char* strstrip (char* str, char* s)
{
	char* eos = str + strlen(str);		/* -> end of string */

	while (*str && strchr(s, *str))
		++str;				/* strip leading spaces */
	while ((eos > str) && strchr(s, *(eos - 1)))
		--eos;				/* strip trailing spaces */
	*eos = '\0';
	return str;
}

/*============================================================================
 * Uppercase string.
 */
char* strupcase (char* str)
{
	char* s;

	for(s = str; *s; ++s) *s = toupper(*s);
	return str;
}

/*============================================================================
 * Get a pointer from a look-up table.
 */
void* lookup (int val, look_up_t* table)
{
	while (table->val && (table->val != val)) table++;
	return table->ptr;
}

/*============================================================================
 * Look up a symbolic name in name table.
 *	Return a numeric value associated with that name or zero, if name was
 *	not found.
 */
int name2val (char* name, look_up_t* table)
{
	while (table->ptr && strcmp(name, table->ptr)) table++;
	return table->val;
}

/*============================================================================
 * Create memory image of a file.
 */
int read_data_file (char* name, data_buf_t* databuf)
{
	int err = 0;
	FILE* file;
	void* buf;
	unsigned long fsize;

	databuf->data = NULL;
	databuf->size = 0;
	file = fopen(name, "rb");
	if (file == NULL) {
		if (verbose) printf(" * Can't open data file %s!\n", name);
		fprintf(stderr, "%s: Can't open data file %s!\n", progname, name);
		show_error(ERR_SYSTEM);
		return ERR_SYSTEM;
	}

	fsize = filesize(file);
	if (!fsize) {
		if (verbose) printf(" * Data file %s is empty!\n", name);
		err = ERR_CONFIG;
		show_error(err);
	}
	else {
		if (verbose) printf(
			" * Reading %lu bytes from %s ...\n",
			fsize, name);
		buf = malloc(fsize);
		if (buf == NULL) {
			err = ERR_SYSTEM;
			show_error(err);
		}
		else if (fread(buf, 1, fsize, file) < fsize) {
			free(buf);
			err = ERR_SYSTEM;
			show_error(err);
		}
		else {
			databuf->data = buf;
			databuf->size = fsize;
		}
	}
	fclose(file);
	return err;
}

/*============================================================================
 * Get file size
 *	Return file length or 0 if error.
 */
unsigned long filesize (FILE* file)
{
	unsigned long size = 0;
	unsigned long cur_pos;

	cur_pos = ftell(file);
	if ((cur_pos != -1L) && !fseek(file, 0, SEEK_END)) {
		size = ftell(file);
		fseek(file, cur_pos, SEEK_SET);
	}
	return size;
}

/*============================================================================
 * Convert decimal string to unsigned integer.
 * If len != 0 then only 'len' characters of the string are converted.
 */
unsigned int dec_to_uint (unsigned char* str, int len)
{
	unsigned val;

	if (!len) len = strlen(str);
	for (val = 0; len && is_digit(*str); ++str, --len)
		val = (val * 10) + (*str - (unsigned)'0')
	;
	return val;
}

unsigned int get_config_data (int argc, char **argv)
{

	FILE *fp;
	char *device_name=NULL;

	fp = fopen (conf_file,"w");
	if (fp == NULL){
		printf("Could not open file\n");
		return 1;
	}

	get_devices(fp,&argc,&argv,&device_name);
	get_interfaces(fp,&argc,&argv,device_name);
	get_hardware(fp,&argc,&argv,device_name);
	get_intr_setup (fp,&argc,&argv);

	fclose(fp);	
	return 0;

}

int get_devices (FILE *fp, int *argc_ptr, char ***argv_ptr, char **device_name){

	int i;
	int start=0, stop=0;
	int argc = *argc_ptr;
	char **argv = *argv_ptr;

	for (i=0;i<argc;i++){
		if (!strcmp(argv[i],"[devices]")){
			start = i;
			continue;
		}
		if (strstr(argv[i],"[") != NULL){
			stop = i;
			break;
		}	
	}

	if (!stop){
		printf("ERROR: No devices found, Incomplete Argument List\n");
		return 1;
	}

	if (stop == 3){
		fprintf(fp,"%s\n",argv[start]);
		*device_name = argv[start+1];					
		fprintf(fp,"%s = %s\n",argv[start+1],argv[start+2]);	
	}else if (stop > 3) {
		fprintf(fp,"%s\n",argv[start]);
		fprintf(fp,"%s = %s, %s\n",argv[start+1],argv[start+2],argv[start+3]);		
	}else{
		printf("ERROR: No devices found, Too many devices arguments\n");
		return 1;
	}		
	
	*argv_ptr += stop;
	*argc_ptr -= stop;

	//???????????????
	//printf("Start is %i and Stop is %i\n",start,stop);
	return 0;

}

int get_interfaces (FILE *fp, int *argc_ptr, char ***argv_ptr, char *device_name){

	char **argv = *argv_ptr;
	int argc = *argc_ptr;
	int i, start=0, stop=0;
	
	for (i=0;i<argc;i++){
		
		//?????????????????????
		//printf("INTR: Argv %i is %s\n",i,argv[i]);
		if (!strcmp(argv[i],"[interfaces]")){
			start = i;
			continue;
		}
		if (strstr(argv[i],"[") != NULL){
			stop = i;
			break;
		}	
	}
	
	if (!stop){
		printf("ERROR: No interfaces found, Incomplete Argument List\n");
		return 1;
	}

	if ((stop-1)%4){
		printf("ERROR: Insuficient/Too many Interface arguments\n");
		return 1;
	}

	fprintf(fp, "\n%s\n", argv[start]);
	for (i=(start+1);i<stop;i+=4){
		fprintf(fp, "%s = %s, ",argv[i],argv[i+1]);
		if (!strcmp(argv[i+2],"-")){
			fprintf(fp, " ,%s\n",argv[i+3]);
		}else{
			fprintf(fp, "%s, %s\n",argv[i+2],argv[i+3]);
		}
	}

	*argv_ptr += stop;
	*argc_ptr -= stop;

	//???????????????
	//printf("Interface Start is %i and Stop is %i\n",start,stop);
	return 0;
}

int get_hardware (FILE *fp, int *argc_ptr, char ***argv_ptr, char *device_name)
{

	char **argv = *argv_ptr;
	int argc = *argc_ptr;
	int i, start=0, stop=0;

	for (i=0;i<argc;i++){
		
		//?????????????????????
		//printf("HRDW: Argv %i is %s\n",i,argv[i]);
		if (strstr(argv[i],"[wanpipe") != NULL){
			start = i;
			continue;
		}
		if (strstr(argv[i],"[") != NULL){
			stop = i;
			break;
		}	
	}

	if (!stop){
		stop = argc;
	}

	if ((stop-1)%2){
		printf("ERROR: Insuficient/Too many Hardware arguments\n");
		return 1;
	}

	fprintf(fp, "\n%s\n", argv[start]);

	for (i=(start+1);i<stop;i+=2){
		fprintf(fp, "%s = %s\n",argv[i],argv[i+1]);
	}

	*argv_ptr += stop;
	*argc_ptr -= stop;

	//?????????????????
	//printf("Hardware Start is %i and Stop is %i\n",start,stop);
	return 0;
}

int get_intr_setup (FILE *fp, int *argc_ptr, char ***argv_ptr)
{

	char **argv = *argv_ptr;
	int argc = *argc_ptr;
	int i, start=0, stop=0, sfound=0;

	for (i=0;i<argc;i++){
		
		//?????????????????????
		//printf("INTR SETUP: Argv %i is %s\n",i,argv[i]);
		if ((strstr(argv[i],"[") != NULL) && !sfound){
			start = i;
			sfound = 1;
			continue;
		}
		if (strstr(argv[i],"[") != NULL){
			stop = i;
			break;
		}	
	}

	if (!stop){
		stop=argc;
	}

	if ((stop-1)%2){
		printf("ERROR: Insuficient/Too many Interface Setup arguments\n");
		return 1;
	}

	fprintf(fp, "\n%s\n", argv[start]);

	for (i=(start+1);i<stop;i+=2){
		fprintf(fp, "%s = %s\n",argv[i],argv[i+1]);
	}

	*argv_ptr += stop;
	*argc_ptr -= stop;

	//?????????????????
	//printf("Interface SETUP Start is %i and Stop is %i\n",start,stop);

	if (stop != argc){
		get_intr_setup (fp, argc_ptr, argv_ptr);
	}

	return 0;
}

int conf_file_down (void)
{
	FILE* file;
	char* conf;             /* -> section buffer */
	char devname[WAN_DRVNAME_SZ];
	int toknum, err;
        char* token[MAX_TOKENS];


	//printf("Shutting down Device %s\n",conf_file);
        file = fopen(conf_file, "r");
        if (file == NULL) {
                show_error(ERR_SYSTEM);
                return ERR_SYSTEM;
        }

	/* Read [devices] section */
        conf = read_conf_section(file, "devices");
        if (conf == NULL) return ERR_CONFIG;

        toknum = tokenize(conf, token);
        if (toknum < 2) return 1;

        strncpy(devname, token[0], WAN_DRVNAME_SZ);

	err = router_down (devname,0);
	if (!err && verbose)
		printf ("Done\n");

	free(conf);

	return err;
}

/*============================================================================
 * TE1
 * Parse active channel string.
 *
 * Return ULONG value, that include 1 in position `i` if channels i is active.
 */
ulong parse_active_channel(char* val)
{
#define SINGLE_CHANNEL	0x2
#define RANGE_CHANNEL	0x1
	int channel_flag = 0;
	char* ptr = val;
	int channel = 0, start_channel = 0;
	ulong tmp = 0;

	if (strcmp(val,"ALL") == 0)
		return 0xFFFFFFFFl;
	while(*ptr != '\0') {
		if (isdigit(*ptr)) {
			channel = strtoul(ptr, &ptr, 10);
			channel_flag |= SINGLE_CHANNEL;
		} else {
			if (*ptr == '-') {
				channel_flag |= RANGE_CHANNEL;
				start_channel = channel;
			} else {
				tmp |= get_active_channels(channel_flag, start_channel, channel);
				channel_flag = 0;
			}
			ptr++;
		}
	}
	if (channel_flag)
		tmp |= get_active_channels(channel_flag, start_channel, channel);
	return tmp;
}

/*============================================================================
 * TE1
 */
ulong get_active_channels(int channel_flag, int start_channel, int stop_channel)
{
	int i = 0;
	ulong tmp = 0, mask = 0;

	if ((channel_flag & (SINGLE_CHANNEL | RANGE_CHANNEL)) == 0)
		return tmp;
	if (channel_flag & RANGE_CHANNEL) { /* Range of channels */
		for(i = start_channel; i <= stop_channel; i++) {
			mask = 1 << (i - 1);
			tmp |=mask;
		}
	} else { /* Single channel */ 
		mask = 1 << (stop_channel - 1);
		tmp |= mask; 
	}
	return tmp;
}
/****** End *****************************************************************/

