/*****************************************************************************
* ppipemon.c	PPP Monitor.
*
* Author:	Nenad Corbic <ncorbic@sangoma.com>		
*
* Copyright:	(c) 1995-1999 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Oct 18, 1999  Nenad Corbic    Added new features for 2.1.0 release version
* Aug 13, 1998	Jaspreet Singh	Improved line tracing code
* Dec 19, 1997	Jaspreet Singh	Added 'irq timeouts' stat in 
*				'READ_COMMS_ERR_STATISTICS'.
* Nov 24, 1997  Jaspreet Singh  Added new stats for driver statistics
* Oct 20, 1997	Jaspreet Singh	Added new cblock.commands for Driver statistics and
*				Router Up time.
* Jul 28, 1997	Jaspreet Singh	Added a new cblock.command for running line trace 
*				displaying RAW data. 
* Jun 24, 1997  Jaspreet Singh	S508/FT1 test cblock.commands
* Apr 25, 1997	Farhan Thawar	Initial version based on ppipemon for WinNT.
*****************************************************************************/

/******************************************************************************
 * 			INCLUDE FILES					      *
 *****************************************************************************/
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <linux/wanpipe.h>
#include <linux/sdla_ppp.h>
#include "ft1_lib.h" 

/******************************************************************************
 * 			DEFINES/MACROS					      *
 *****************************************************************************/
#define TIMEOUT 1
#define TRUE 1
#define FALSE 0
#define MDATALEN 2024

#define CB_SIZE sizeof(wp_mgmt_t)+sizeof(cblock_t)+1

#define DO_COMMAND(packet)	DoCommand(&packet, CB_SIZE + packet.cblock.length)

/******************************************************************************
 * 			TYPEDEF/STRUCTURE				      *
 *****************************************************************************/
#pragma pack(1)
typedef struct {
	wp_mgmt_t	wp_mgmt PACKED;
	cblock_t	cblock  PACKED;
        unsigned char data[MDATALEN] PACKED;
//} CBLOCK;
} udp_mgmt_pkt_t;
#pragma pack()


/******************************************************************************
 * 			GLOBAL VARIABLES				      *
 *****************************************************************************/
/* storage for FT1 LEDs */
FT1_LED_STATUS FT1_LED;

/* global for now */
int sock;
//CBLOCK cb, cbconfig;
udp_mgmt_pkt_t	ppp_udp;
char codeversion[10];
unsigned int frame_count;
int is_508;
int raw_data = FALSE;

/* for S508/FT1 test cblock.commands */
static int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int loop_counter;

/* defines for now */
char ipaddress[16];
char cmd[5];
int udp_port = 9000;

/******************************************************************************
 * 			FUNCTION PROTOTYPES				      *
 *****************************************************************************/
void 		ResetCB(udp_mgmt_pkt_t *c1);
int 		MakeConnection( void );
int 		ObtainConfiguration( void );
void 		SaveCB(udp_mgmt_pkt_t *dest, udp_mgmt_pkt_t *src );
void 		init( int argv, char *argc[]);
unsigned long 	decode_bps( unsigned char bps );
char*		get_ip_addr(char *if_name);
void 		banner(char *);
void 		set_FT1_monitor_status( unsigned char);
void 		set_FT1_mode(int verbose);
void 		read_FT1_status( void );
void 	set_linelb(int);
void 	set_paylb(int);
void 	set_ddlb(int);
void 	send_lb(int);
void 	read_te1_56k_stat(void);
void 	flush_te1_pmon(void);
void read_ft1_te1_config (void);

extern unsigned char DoCommand(void*, int);
extern int MakeUdpConnection( void );
#ifdef LINUX_2_1
extern int MakeRawConnection( void );
#endif

/******************************************************************************
 * 			FUNCTION DEFINITION				      *
 *****************************************************************************/
//void ResetCB(CBLOCK *c1)
void ResetCB(udp_mgmt_pkt_t *ppp_udp)
{
	memset((void *)&ppp_udp->cblock.opp_flag, 0, sizeof(udp_mgmt_pkt_t)-sizeof(wp_mgmt_t));
}

int ObtainConfiguration( void ) 
{
	unsigned char x;
   
	x = 0;
	ppp_udp.cblock.command = PPP_READ_CONFIG;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;

	while (DO_COMMAND(ppp_udp) != 0 && ++x < 4) {
		if (ppp_udp.cblock.result == 0xaa) {
			printf("Error: Command timeout occurred"); 
			return(FALSE);
		}

		if (ppp_udp.cblock.result == 0xCC ) return(FALSE);
	}

	if (x >= 4) return(FALSE);

	if( ppp_udp.cblock.length == 0x73 ) {
		is_508 = TRUE;
	} else {
		is_508 = FALSE;
	} 
   
	strcpy(codeversion, "?.??");
   
	ppp_udp.cblock.command = PPP_READ_CODE_VERSION;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if (ppp_udp.cblock.result == 0) {
		ppp_udp.data[ppp_udp.cblock.length] = 0;
		strcpy(codeversion, ppp_udp.data);
	}

	return(TRUE);
}; /* ObtainConfiguration */


void init( int argc, char *argv[])
{
	int i, i_cnt=0, u_cnt=0, c_cnt=0, if_found=0;
	char * ip_addr;
	struct in_addr *ip_str=NULL;
	ppp_udp.wp_mgmt.id = 0;

	for (i=0;i<argc;i++){
		if (!strcmp(argv[i],"-i")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid IP or Interface Name, Type ppipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(ipaddress,argv[i+1]);
			if (inet_aton(ipaddress,ip_str) != 0 ){
			}else{
				if_found=1;
			}
			i_cnt=1;
		}else if (!strcmp(argv[i],"-u")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid UDP PORT, Type ppipemon <cr> for help\n\n");
				exit(0);
			}

		 	if(isdigit(argv[i+1][0])){
				udp_port = atoi(argv[i+1]);
			}else{
				printf("ERROR: Invalid UDP Port, Type ppipemon <cr> for help\n\n");
				exit(0);
			}
			u_cnt=1;
		}else if (!strcmp(argv[i],"-c")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type ppipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(cmd,argv[i+1]);
			c_cnt=1;
		}
	}

	if (!i_cnt){
		printf("ERROR: No IP address or Interface Name, Type ppipemon <cr> for help\n\n");
		exit(0);
	}
	if (!c_cnt){
		printf("ERROR: No Command, Type ppipemon <cr> for help\n\n");
		exit(0);
	}

	if (if_found){
		ip_addr = get_ip_addr(ipaddress);
		strcpy(ipaddress,ip_addr);
	}

	/* signature for UDP Management */
	strcpy( ppp_udp.wp_mgmt.signature, "PTPIPEAB");
};

char * get_ip_addr(char *if_name)
{

	int skfd;
	struct sockaddr_in *sin;
	struct ifreq ifr;

    	if ((skfd = socket(AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0) {
		perror("socket");
		exit(1);
    	}

	strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));

        if (ioctl(skfd, SIOCGIFDSTADDR , &ifr) < 0) {
		fprintf(stderr, "Error: Unknown interface: %s\n",if_name);
		exit(0);
    	}

	sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_dstaddr;
	sin->sin_family = AF_INET;
	sin->sin_port = 0;

	close (skfd);

	return (inet_ntoa(sin->sin_addr));

} 

unsigned long decode_bps( unsigned char bps ) 
{
	unsigned long number;
 
	switch( bps ) {
		case 0x00:
			number = 0;
			break;
		case 0x01:
			number = 1200;
			break;
		case 0x02:
			number = 2400;
			break;
		case 0x03:
			number = 4800;
			break;
		case 0x04:
			number = 9600;
			break;
		case 0x05:
			number = 19200;
			break;
		case 0x06:
			number = 38400;
			break;
		case 0x07:
			number = 45000;
			break;
		case 0x08:
			number = 56000;
			break;
		case 0x09:
			number = 64000;
			break;
		case 0x0A:
			number = 74000;
			break;
		case 0x0B:
			number = 112000;
			break;
		case 0x0C:
			number = 128000;
			break;
		case 0x0D:
			number = 156000;
			break;
		default:
			number = 0;
			break;
	} /* switch */

	return number;
}; /* decode_bps */

void error( void ) 
{

	printf("Error: Command failed!\n");

}; /* error */

void general_conf( void ) 
{
	unsigned long baud;
	unsigned short s_number, mtu, mru;
	unsigned char misc;
 
	ppp_udp.cblock.command = PPP_READ_CONFIG;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		if( is_508 == TRUE ) {
			banner("GENERAL CONFIGURATION 508 Board");
			memcpy(&baud,&ppp_udp.data[0],4);
			memcpy(&s_number,&ppp_udp.data[4],2);
			printf("Transmit buffer allocation in percent: %d\n",
				s_number);
			misc = ppp_udp.data[6];
			printf("Use alternate adapter frequency: ");
			( misc & 0x10 ) ? printf("Yes\n") : printf("No\n");
			printf("Set interface type to RS-232: ");
			( misc & 0x20 ) ? printf("Yes\n") : printf("No\n");
			memcpy(&mtu, &ppp_udp.data[8],2);
			memcpy(&mru, &ppp_udp.data[10],2);
		} else {
			banner("GENERAL CONFIGURATION 502 Board");
			baud = decode_bps(ppp_udp.data[0]);
			misc = ppp_udp.data[3];
			printf("Discard transmit-aborted frames: ");
			( misc & 0x01 ) ? printf("Yes\n") : printf("No\n");
			memcpy(&mtu, &ppp_udp.data[5],2);
			memcpy(&mru, &ppp_udp.data[7],2);
		} 

		printf("Baud rate in bps: %lu\n",baud);
		printf("Update transmit statistics( user data ): ");
		( misc & 0x02 ) ? printf("Yes\n") : printf("No\n");
		printf("Update receive statistics( user data ): ");
		( misc & 0x04 ) ? printf("Yes\n") : printf("No\n");
		printf("Timestamp received packets: ");
		( misc & 0x08 ) ? printf("Yes\n") : printf("No\n");
		printf("Maximum Receive/Transmit Unit(MRU/MTU): %d\n",mtu);
		printf("Minimum remote MRU required on connection: %d\n",mtu);
	} else {
		error();
	} 
}; /* general_conf */

void timers( void ) 
{
	int i;
	unsigned short tmp;
   
	ppp_udp.cblock.command = PPP_READ_CONFIG;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("PPP TIMERS");

		if( is_508 == TRUE ) {
			i = 12;
		} else {
			i = 9;
		} 

		memcpy(&tmp,&ppp_udp.data[i],2);
		printf("Restart timer: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+2],2);
		printf("Authentication restart timer: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+4],2);
		printf("Authentication wait timer: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+6],2);
		printf("DCD/CTS failure detection timer: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+8],2);
		printf("Drop DTR duration timer: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+10],2);
		printf("Connection attempt timeout: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+12],2);
		printf("Max-Configure counter: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+14],2);
		printf("Max-Terminate counter: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+16],2);
		printf("Max-Failure counter: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[i+18],2);
		printf("Max-Authenticate counter: %d\n",tmp);

	} else {
		error();
	} 
}; /* timers */

void authent( void ) 
{
	unsigned char tmp;
   
	ppp_udp.cblock.command = PPP_READ_CONFIG;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {
	
		banner("PPP AUTHENTICATION");

		if( is_508 == TRUE ) {
			tmp = ppp_udp.data[32];
		} else {
			tmp = ppp_udp.data[29];
		} 
	
		if( tmp & 0x01 ) {
		     printf("Allow the use of PAP for inbound/outbound: Yes\n");
			if( tmp & 0x80 ) {
				printf("Using inbound authentication.\n");
			} else {
				printf("Using outbound authentication.\n");
			} //if
		} else {
			printf("Allow the use of PAP for inbound/outbound: No\n"				);
		} //if

		if( tmp & 0x02 ) {
	 	    printf("Allow the use of CHAP for inbound/outbound: Yes\n");
			if( tmp & 0x80 ) {
				printf("Using inbound authentication.\n");
			} else {
				printf("Using outbound authentication.\n");
			} //if
		} else {
		     printf("Allow the use of CHAP for inbound/outbound: No\n");
		} //if

	} else {
		error();
	} //if
}; /* authent */

void ip_config( void ) 
{
	int i;
   
	ppp_udp.cblock.command = PPP_READ_CONFIG;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;

	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("PPP IP CONFIGURATION");

		if( is_508 == TRUE ) {
			i = 33;
		} else {
			i = 30;
		} //if
		printf("Enable the use of IP: ");
		( ppp_udp.data[i] & 0x80 ) ? printf("Yes\n") : printf("No\n");
		printf("Notify remote of locally-configure address: ");
		( ppp_udp.data[i] & 0x01 ) ? printf("Yes\n") : printf("No\n");
		printf("Local IP address( 0.0.0.0 = request ): %d.%d.%d.%d\n",
			ppp_udp.data[i+1],ppp_udp.data[i+2],ppp_udp.data[i+3],ppp_udp.data[i+4]);
		printf("Request remote to provide local address: ");
		( ppp_udp.data[i] & 0x02 ) ? printf("Yes\n") : printf("No\n");
		printf("Provide remote with pre-configured address: ");
		( ppp_udp.data[i] & 0x04 ) ? printf("Yes\n") : printf("No\n");
		printf("Remote IP address: %d.%d.%d.%d\n",ppp_udp.data[i+5],
			ppp_udp.data[i+6],ppp_udp.data[i+7],ppp_udp.data[i+8]);
		printf("Require that remote provide an address: ");
		( ppp_udp.data[i] & 0x08 ) ? printf("Yes\n") : printf("No\n");
	} else {
		error();
	} 
}; /* ip_config */

void ipx_config( void )
{
	int i;
   
	ppp_udp.cblock.command = PPP_READ_CONFIG;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;

	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {
	
		banner("PPP IPX CONFIGURATION");

		if( is_508 == TRUE ) {
			i = 42;
		} else {
			i = 39;
		} //if

		printf("Enable the use of IPX: ");
		( ppp_udp.data[i] & 0x80 ) ? printf("Yes\n") : printf("No\n");
		printf("Include network number in Config-Request: ");
		( ppp_udp.data[i] & 0x01 ) ? printf("Yes\n") : printf("No\n");
		printf("Network number( 00000000 = request ): %02X %02X %02X %02X\n",ppp_udp.data[i+1],ppp_udp.data[i+2],ppp_udp.data[i+3],ppp_udp.data[i+4]);
		printf("Include local node # in Config-Request: ");
		( ppp_udp.data[i] & 0x02 ) ? printf("Yes\n") : printf("No\n");
		printf("Local node number( 000000000000 = request ): %02X %02X %02X %02X %02X %02X\n",ppp_udp.data[i+5],ppp_udp.data[i+6],ppp_udp.data[i+7],ppp_udp.data[i+8],ppp_udp.data[i+9],ppp_udp.data[i+10]);
		printf("Force remote to accept remote node number: ");
		( ppp_udp.data[i] & 0x04 ) ? printf("Yes\n") : printf("No\n");
		printf("Remote node number: %02X %02X %02X %02X %02X %02X\n",
			ppp_udp.data[i+11],ppp_udp.data[i+12],ppp_udp.data[i+13],ppp_udp.data[i+14],
			ppp_udp.data[i+15],ppp_udp.data[i+16]);
		printf("Include config-complete in Config-Request: ");
		( ppp_udp.data[i] & 0x40 ) ? printf("Yes\n") : printf("No\n");

		if( ppp_udp.data[i] & 0x20 ) {
			printf("Routing protocol: Request default( normally RIP/SAP )\n");
		} else if( ppp_udp.data[i] & 0x18 ){
			printf("Routing protocol: Use either RIP/SAP or NLSP\n");
		} else if( ppp_udp.data[i] & 0x08 ){
			printf("Routing protocol: Use RIP/SAP only\n");
		} else if( ppp_udp.data[i] & 0x10 ){
			printf("Routing protocol: Use NLSP only\n");
		} else {
			printf("Routing protocol: No routing protocol\n");
		} //if
		printf("Local router name( max. 47 characters ): %s\n",
			&ppp_udp.data[i+17]);
	} else {
		error();
	} //if
}; /* ipx_config */

void set_state( unsigned char tmp ) 
{
	switch( tmp ) {
		case 0:
			printf("Initial\n");
			break;
		case 1:
			printf("Starting\n");
			break;
		case 2:
			printf("Closed\n");
			break;
		case 3:
			printf("Stopped\n");
			break;
		case 4:
			printf("Closing\n");
			break;
		case 5:
			printf("Stopping\n");
			break;
		case 6:
			printf("Request Sent\n");
			break;
		case 7:
			printf("Ack Received\n");
			break;
		case 8:
			printf("Ack Sent\n");
			break;
		case 9:
			printf("Opened\n");
			break;
		default:
			printf("Unknown\n");
			break;
	} //switch

}; /* set_state */

void state( void ) 
{
	ppp_udp.cblock.command = PPIPE_GET_IBA_DATA;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("PPP LINK STATE");

		switch( ppp_udp.data[2] ) {
			case 0:
				printf("PPP Phase: Link Dead\n");
				break;
			case 1:
				printf("PPP Phase: Establishment( LCP )\n");
				break;
			case 2:
				printf("PPP Phase: Authentication\n");
				break;
			case 3:
				printf("PPP Phase: Network Layer\n");
				break;
			case 4:
				printf("PPP Phase: Link Termination\n");
				break;
			default:
				printf("PPP Phase: Unknown\n");
				break;
		} 

		printf("LCP State: ");
		set_state( ppp_udp.data[1] );
		printf("IPCP State: ");
		set_state( ppp_udp.data[3] );
		printf("IPXCP State: ");
		set_state( ppp_udp.data[4] );
		switch( ppp_udp.data[5] ) {
			case 0:
				printf("PAP State: Initial( Inactive )\n");
				break;
			case 1:
				printf("PAP State: Failed\n");
				break;
			case 2:
				printf("PAP State: Request Sent\n");
				break;
			case 3:
				printf("PAP State: Waiting\n");
				break;
			case 4:
				printf("PAP State: Opened( Success )\n");
				break;
			default:
				printf("PAP State: Unknown\n");
				break;
		} 

		switch( ppp_udp.data[5] ) {
			case 0:
				printf("CHAP State: Initial( Inactive )\n");
				break;
			case 1:
				printf("CHAP State: Failed\n");
				break;
			case 2:
				printf("CHAP State: Challenge Sent\n");
				break;
			case 3:
				printf("CHAP State: Waiting for Challenge\n");
				break;
			case 4:
				printf("CHAP State: Response Sent\n");
				break;
			case 5:
				printf("CHAP State: Opened( Success )\n");
				break;
			default:
				printf("CHAP State: Unknown\n");
				break;
		} 
	} else {
		error();
	} 
}; /* state */

void negot( void ) {
	unsigned short mru;
   
	ppp_udp.cblock.command = PPP_GET_CONNECTION_INFO;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("PPP NEGOTIATIONS");

		memcpy(&mru,&ppp_udp.data[0],2);
		printf("Remote Maximum Receive Unit: %d\n",mru);
		printf("Negotiated IP options: %02X",ppp_udp.data[2]);
		( ppp_udp.data[2] & 0x80 ) ? printf("( IP Enabled )\n") : 
					printf("( IP Disabled )\n");
		printf("Local IP address: %d.%d.%d.%d\n",ppp_udp.data[3],ppp_udp.data[4],
			ppp_udp.data[5],ppp_udp.data[6]);
		printf("Remote IP address: %d.%d.%d.%d\n",ppp_udp.data[7],ppp_udp.data[8],
			ppp_udp.data[9],ppp_udp.data[10]);
		printf("Negotiated IPX options: %02X",ppp_udp.data[11]);
		( ppp_udp.data[11] & 0x80 ) ? printf("( IPX Enabled )\n") : 
					printf("( IPX Disabled )\n");
		printf("IPX network number: %02X %02X %02X %02X\n",ppp_udp.data[12],
			ppp_udp.data[13],ppp_udp.data[14],ppp_udp.data[15]);
		printf("Local IPX node number: %02X %02X %02X %02X %02X %02X\n",
			ppp_udp.data[16],ppp_udp.data[17],ppp_udp.data[18],ppp_udp.data[19],
			ppp_udp.data[20],ppp_udp.data[21]);
		printf("Remote IPX node number: %02X %02X %02X %02X %02X %02X\n"
			,ppp_udp.data[22],ppp_udp.data[23],ppp_udp.data[24],ppp_udp.data[25],
			ppp_udp.data[26],ppp_udp.data[27]);
		printf("Remote IPX router name: %s\n",&ppp_udp.data[28]);
		
		switch( ppp_udp.data[76] ) {
			case 0:
				printf("Authentication status: No inbound authentication negotiated\n");
				break;
			case 1:
				printf("Authentication status: PeerID valid, Password Incorrect\n");
				break;
			case 2:
				printf("Authentication status: PeerID was invalid\n");
				break;
			case 3:
				printf("Authentication status: PeerID and Password were correct\n");
				break;
			default:
				printf("Authentication status: Unknown\n");
				break;
		} 

		printf("Inbound PeerID: %s\n",&ppp_udp.data[77]);
		
	} else {
		error();
	} //if
}; /* negot */

void cause( void ) 
{
	unsigned short disc;
  
	ppp_udp.cblock.command = PPIPE_GET_IBA_DATA;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("LAST CAUSE OF LINK FAILURE");

		memcpy(&disc,&ppp_udp.data[7],2);
		printf("Local request by termination phase: ");
		(disc & 0x0100) ? printf("Yes\n") : printf("No\n");
		printf("DCD and/or CTS dropped: ");
		(disc & 0x0200) ? printf("Yes\n") : printf("No\n");
		printf("Disabled communications locally: ");
		(disc & 0x0400) ? printf("Yes\n") : printf("No\n");
		printf("Inbound/Outbound authentication failed: ");
		(disc & 0x0800) ? printf("Yes\n") : printf("No\n");
		printf("Failed to negotiate inbound auth. protocol with peer:");
		(disc & 0x1000) ? printf(" Yes\n") : printf(" No\n");
		printf("Rejected peers request for authentication: ");
		(disc & 0x2000) ? printf("Yes\n") : printf("No\n");
		printf("Peer rejected MRU option of config-request: ");
		(disc & 0x4000) ? printf("Yes\n") : printf("No\n");
		printf("MRU of peer was below required minumum: ");
		(disc & 0x8000) ? printf("Yes\n") : printf("No\n");
		printf("Rejected peers LCP option(s) too many times: ");
		(disc & 0x0001) ? printf("Yes\n") : printf("No\n");
		printf("Rejected peers IPCP option(s) too many times: ");
		(disc & 0x0002) ? printf("Yes\n") : printf("No\n");
		printf("Rejected peers IPXCP option(s) too many times: ");
		(disc & 0x0004) ? printf("Yes\n") : printf("No\n");
	} else {
		error();
	} //if
}; /* cause */

void modem( void )
{
	unsigned char cts_dcd;
   
	ppp_udp.cblock.command = PPIPE_GET_IBA_DATA;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("MODEM STATUS");

		memcpy(&cts_dcd,&ppp_udp.data[0],1);
		printf("DCD: ");
		(cts_dcd & 0x08) ? printf("High\n") : printf("Low\n");
		printf("CTS: ");
		(cts_dcd & 0x20) ? printf("High\n") : printf("Low\n");
	} else {
		error();
	} //if
}; /* modem */

void general_stats( void ) 
{
	unsigned short tmp;
	unsigned long l_tmp;
   
	ppp_udp.cblock.command = PPP_READ_STATISTICS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner ("GENERAL STATISTICS");

		if( is_508 == TRUE ) {
			memcpy(&tmp,&ppp_udp.data[2],2);
			printf("Number of received frames discarded due to bad length: %d\n",tmp);
		} else {
			memcpy(&tmp,&ppp_udp.data[0],2);
			printf("Number of received frames discarded at the interrupt level: %d\n",tmp);
			memcpy(&tmp,&ppp_udp.data[2],2);
			printf("Number of received frames discarded at the application level: %d\n",tmp);
			memcpy(&tmp,&ppp_udp.data[4],2);
			printf("Number of received retransmitted due to aborts: %d\n",tmp);
		} 

		memcpy(&l_tmp,&ppp_udp.data[6],4);
		printf("Number of user frames transmitted: %lu\n",l_tmp);
		memcpy(&l_tmp,&ppp_udp.data[10],4);
		printf("Number of user bytes transmitted: %lu\n",l_tmp);
		memcpy(&l_tmp,&ppp_udp.data[14],4);
		printf("Number of user frames received: %lu\n",l_tmp);
		memcpy(&l_tmp,&ppp_udp.data[18],4);
		printf("Number of user bytes received: %lu\n",l_tmp);
	} else {
		error();
	} 
}; /* general_stats */

void flush_general_stats( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_STATISTICS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result != 0 ) {
		error();
	} //if

}; /* flush_general_stats */

void comm_err( void ) 
{
	ppp_udp.cblock.command = PPP_READ_ERROR_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("COMMUNICATION ERROR STATISTICS");

		if( is_508 == TRUE ) {
			printf("Number of times receiver was halted due to full buffers: %d\n",ppp_udp.data[3]);
		} else {
			printf("Number of frames discarded at the interrupt level due to frame being too long: %d\n",ppp_udp.data[3]);
			printf("Number of transmit underruns: %d\n",ppp_udp.data[5]);
		} //if
		printf("Number of receiver overrun errors: %d\n",ppp_udp.data[0]);
		printf("Number of receiver CRC errors: %d\n",ppp_udp.data[1]);
		printf("Number of abort frames received: %d\n",ppp_udp.data[2]);
		printf("Number of abort frames transmitted: %d\n",ppp_udp.data[4]);
		printf("Number of missed transmit underrun interrupts: %d\n",ppp_udp.data[6]);
		printf("Number of IRQ timeouts: %d\n",ppp_udp.data[7]);
		printf("Number of times DCD changed state: %d\n",ppp_udp.data[8]);
		printf("Number of times CTS changed state: %d\n",ppp_udp.data[9]);
	} else {
		error();
	} //if
}; /* comm_err */

void flush_comm_err( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_ERROR_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result != 0 ) {
		error();
	} 
}; /* flush_general_stats */

void packet( void ) 
{
	unsigned short tmp, tmp2;
   
	ppp_udp.cblock.command = PPP_READ_PACKET_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("PACKET STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Number discards( bad header ): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[2],2);
		printf("Number discards( unknown/unsupported protocol ): %d\n",
			tmp);
		memcpy(&tmp,&ppp_udp.data[4],2);
		printf("Number discards(unknown/unsupported protocol+too large for Protocol-Reject): %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&ppp_udp.data[6],2);
		memcpy(&tmp2,&ppp_udp.data[8],2);
		printf("Number of LCP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[10],2);
		memcpy(&tmp2,&ppp_udp.data[12],2);
		printf("Number of IPCP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[14],2);
		memcpy(&tmp2,&ppp_udp.data[16],2);
		printf("Number of IPXCP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[18],2);
		memcpy(&tmp2,&ppp_udp.data[20],2);
		printf("Number of PAP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[22],2);
		memcpy(&tmp2,&ppp_udp.data[24],2);
		printf("Number of CHAP packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[26],2);
		memcpy(&tmp2,&ppp_udp.data[28],2);
		printf("Number of LQR packets: %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[30],2);
		memcpy(&tmp2,&ppp_udp.data[32],2);
		printf("Number of IP packets:  %d\t\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[34],2);
		memcpy(&tmp2,&ppp_udp.data[36],2);
		printf("Number of IPX packets: %d\t\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* packet */

void flush_packet( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_PACKET_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result != 0 ) {
		error();
	} 
}; /* flush_packet */

void lcp( void ) 
{
	unsigned short tmp, tmp2;
 
	ppp_udp.cblock.command = PPP_READ_LCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("LCP STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Packets discarded (unknown LCP code): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[48],2);
		printf("Received LCP packets too large: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[50],2);
		printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[52],2);
		printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[54],2);
		printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&ppp_udp.data[2],2);
		memcpy(&tmp2,&ppp_udp.data[26],2);
		printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[4],2);
		memcpy(&tmp2,&ppp_udp.data[28],2);
		printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[6],2);
		memcpy(&tmp2,&ppp_udp.data[30],2);
		printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[8],2);
		memcpy(&tmp2,&ppp_udp.data[32],2);
		printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[10],2);
		memcpy(&tmp2,&ppp_udp.data[34],2);
		printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[12],2);
		memcpy(&tmp2,&ppp_udp.data[36],2);
		printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[14],2);
		memcpy(&tmp2,&ppp_udp.data[38],2);
		printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[16],2);
		memcpy(&tmp2,&ppp_udp.data[40],2);
		printf("Number of Protocol-Rej packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[18],2);
		memcpy(&tmp2,&ppp_udp.data[42],2);
		printf("Number of Echo-Request packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[20],2);
		memcpy(&tmp2,&ppp_udp.data[44],2);
		printf("Number of Echo-Reply packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[22],2);
		memcpy(&tmp2,&ppp_udp.data[46],2);
		printf("Number of Discard-Request packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* lcp */

void flush_lcp( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_LCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;

	DO_COMMAND(ppp_udp);
	if( ppp_udp.cblock.result != 0 ) {
		error();
	} 
}; /* flush_packet */

void loopback( void ) 
{
	unsigned short tmp;
   
	ppp_udp.cblock.command = PPP_READ_LCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	
	if( ppp_udp.cblock.result == 0 ) {

		banner("LOOPBACK STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Looped-back link possible given Config-Req/Nak/Rej: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[2],2);
		printf("Looped-back link detected with Echo-Request: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[4],2);
		printf("Echo-Request received from bad source: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[6],2);
		printf("Looped-back link detected with Echo-Reply: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[8],2);
		printf("Echo-Reply received from bad source: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[10],2);
		printf("Looped-back link detected with Discard-Request: %d\n",
			tmp);
		memcpy(&tmp,&ppp_udp.data[12],2);
		printf("Discard-Request received from bad source: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[14],2);
		printf("Echo-Reply discarded( transmit collision ): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[16],2);
		printf("Echo-Reply discarded( receive collision ): %d\n",tmp);
	
	} else {
		error();
	} 
}; /* loopback */

void flush_loopback( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_LPBK_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result != 0 ) {
		error();
	}
 
}; /* flush_loopback */

void ipcp( void ) 
{
	unsigned short tmp, tmp2;
   
	ppp_udp.cblock.command = PPP_READ_IPCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("IPCP STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Packets discarded (unknown IPCP code): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[32],2);
		printf("Received IPCP packets too large: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[34],2);
		printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[36],2);
		printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[38],2);
		printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&ppp_udp.data[2],2);
		memcpy(&tmp2,&ppp_udp.data[18],2);
		printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[4],2);
		memcpy(&tmp2,&ppp_udp.data[20],2);
		printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[6],2);
		memcpy(&tmp2,&ppp_udp.data[22],2);
		printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[8],2);
		memcpy(&tmp2,&ppp_udp.data[24],2);
		printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[10],2);
		memcpy(&tmp2,&ppp_udp.data[26],2);
		printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[12],2);
		memcpy(&tmp2,&ppp_udp.data[28],2);
		printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[14],2);
		memcpy(&tmp2,&ppp_udp.data[30],2);
		printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* ipcp */

void flush_ipcp( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_IPCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result != 0 ) {
		error();
	} 
}; /* flush_ipcp */

void ipxcp( void ) 
{
	unsigned short tmp, tmp2;
  
	ppp_udp.cblock.command = PPP_READ_IPXCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if( ppp_udp.cblock.result == 0 ) {

		banner("IPXCP STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Packets discarded (unknown IPXCP code): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[32],2);
		printf("Received IPXCP packets too large: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[34],2);
		printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[36],2);
		printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[38],2);
		printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&ppp_udp.data[2],2);
		memcpy(&tmp2,&ppp_udp.data[18],2);
		printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[4],2);
		memcpy(&tmp2,&ppp_udp.data[20],2);
		printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[6],2);
		memcpy(&tmp2,&ppp_udp.data[22],2);
		printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[8],2);
		memcpy(&tmp2,&ppp_udp.data[24],2);
		printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[10],2);
		memcpy(&tmp2,&ppp_udp.data[26],2);
		printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[12],2);
		memcpy(&tmp2,&ppp_udp.data[28],2);
		printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[14],2);
		memcpy(&tmp2,&ppp_udp.data[30],2);
		printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} //if
}; /* ipxcp */

void flush_ipxcp( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_IPXCP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if( ppp_udp.cblock.result != 0 ) {
		error();
	}
}; /* flush_ipxcp */

void pap( void ) 
{
	unsigned short tmp, tmp2;
   
	ppp_udp.cblock.command = PPP_READ_PAP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {


		banner("PAP STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Packets discarded (unknown PAP code): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[16],2);
		printf("Received PAP packets too large: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[18],2);
		printf("Received packets invalid inbound PeerID: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[20],2);
		printf("Received packets invalid inbound Password: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&ppp_udp.data[2],2);
		memcpy(&tmp2,&ppp_udp.data[10],2);
		printf("Number of Authent-Request pkts: %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[4],2);
		memcpy(&tmp2,&ppp_udp.data[12],2);
		printf("Number of Authent-Ack packets : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[6],2);
		memcpy(&tmp2,&ppp_udp.data[14],2);
		printf("Number of Authent-Nack packets: %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* pap */

void flush_pap( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_PAP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if( ppp_udp.cblock.result != 0 ) {
		error();
	} 
}; /* flush_pap */

void chap( void ) 
{
	unsigned short tmp, tmp2;
   
	ppp_udp.cblock.command = PPP_READ_CHAP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) {

		banner("CHAP STATISTICS");

		memcpy(&tmp,&ppp_udp.data[0],2);
		printf("Packets discarded (unknown CHAP code): %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[20],2);
		printf("Received CHAP packets too large: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[22],2);
		printf("Received packets invalid inbound PeerID: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[24],2);
		printf("Received packets invalid inbound Password/Secret: %d\n",
			tmp);
		memcpy(&tmp,&ppp_udp.data[26],2);
		printf("Received packets invalid inbound MD5 message digest format: %d\n",tmp);
		memcpy(&tmp,&ppp_udp.data[28],2);
		printf("Invalid inbound ID or out-of-order or unelicited responses: %d\n",tmp);
		printf("\n\t\t\tReceived\tTransmitted\n");
		memcpy(&tmp,&ppp_udp.data[2],2);
		memcpy(&tmp2,&ppp_udp.data[12],2);
		printf("Number of Challenge packets  : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[4],2);
		memcpy(&tmp2,&ppp_udp.data[14],2);
		printf("Number of Response packets   : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[6],2);
		memcpy(&tmp2,&ppp_udp.data[16],2);
		printf("Number of Success packets    : %d\t%d\n",tmp,tmp2);
		memcpy(&tmp,&ppp_udp.data[8],2);
		memcpy(&tmp2,&ppp_udp.data[18],2);
		printf("Number of Failure packets    : %d\t%d\n",tmp,tmp2);
	} else {
		error();
	} 
}; /* chap */

void flush_chap( void ) 
{
	ppp_udp.cblock.command = PPP_FLUSH_CHAP_STATS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result != 0 ) {
		error();
	} 
}; /* flush_chap */

void usage( void ) 
{
	printf("ppipemon: Wanpipe PPP Debugging Utility\n\n");
	printf("Usage:\n");
	printf("-----\n\n");
	printf("ppipemon -i <ip-address or interface name> -u <port> -c <command>\n\n");
	printf("\tOption -i: \n");
	printf("\t\tWanpipe remote IP address must be supplied\n");
	printf("\t\t<or> Wanpipe network interface name (ex: wp1_ppp)\n");   
	printf("\tOption -u: (Optional, default: 9000)\n");
	printf("\t\tWanpipe UDPPORT specified in /etc/wanpipe#.conf\n");
	printf("\tOption -c: \n");
	printf("\t\tCommand is split into two parts:\n"); 
	printf("\t\t\tFirst letter is a command and the rest are options:\n"); 
	printf("\t\t\tex: xm = View Modem Status\n\n");
	printf("\tSupported Commands: x=status : s=statistics : t=trace \n");
	printf("\t                    c=config : T=FT1 stats  : f=flush\n\n");
	printf("\tCommand:  Options:   Description \n");	
	printf("\t-------------------------------- \n\n");    
	printf("\tCard Status\n");
	printf("\t   x         m       Modem Status\n");
	printf("\t             n       Parameters Negotiated on Last Connection/Attempt\n");
	printf("\t             ru      Display Router UP time\n");
	printf("\t             u       PPP Timers and Counters\n");
	printf("\t             s       PPP FSM Current State\n");
	printf("\t             c       Cause for Last Disconnection\n");
	printf("\tCard Configuration\n");
	printf("\t   c         g       PPP General Configuration\n");
	printf("\t             a       Authentication Configuration\n");
	printf("\t             i       IP Configuration\n");
	printf("\t             x       IPX Configuration\n");
	printf("\tCard Statistics\n");
	printf("\t   s         g       Global Statistics\n");
	printf("\t             c       Communication Error Statistics\n");
	printf("\t             p       Packet Statistics\n");
	printf("\t             lpc     LCP Statistics\n");
	printf("\t             lo      Loopback Detection / LCP Error Statistics\n");
	printf("\t             ipc     IP Control Protocol( IPCP )Statistics\n");
	printf("\t             xpc     IPX Control Protocol( IPXCP )Statistics\n");
	printf("\t             pap     Password Authentication (PAP) Statistics\n");
	printf("\t             chp     Challenge-Handshake Auth.(CHAP) Statistics\n");
	printf("\tTrace Data\n");
	printf("\t   t         i       Trace and Interpret ALL frames\n");
	printf("\t             ip      Trace and Interpret PROTOCOL frames only\n");
	printf("\t             id      Trace and Interpret DATA frames only\n");
	printf("\t             r       Trace ALL frames, in RAW format\n");
	printf("\t             rp      Trace PROTOCOL frames only, in RAW format\n");
	printf("\t             rd      Trace DATA frames only, in RAW format\n");
	printf("\tFT1/T1/E1/56K Configuration/Statistics\n");
	printf("\t   T         v       View Status\n");
	printf("\t             s       Self Test\n");
	printf("\t             l       Line Loop Test\n");
	printf("\t             d       Digital Loop Test\n");
	printf("\t             r       Remote Test\n");
	printf("\t             o       Operational Mode\n");
	printf("\t             read    Read CSU/DSU configuration (FT1/T1/E1 card)\n");  
	printf("\t             allb    Active Line Loopback mode (T1/E1 card only)\n");  
	printf("\t             dllb    Deactive Line Loopback mode (T1/E1 card only)\n");  
	printf("\t             aplb    Active Payload Loopback mode (T1/E1 card only)\n");  
	printf("\t             dplb    Deactive Payload Loopback mode (T1/E1 card only)\n");  
	printf("\t             adlb    Active Diagnostic Digital Loopback mode (T1/E1 card only)\n");  
	printf("\t             ddlb    Deactive Diagnostic Digital Loopback mode (T1/E1 card only)\n");  
	printf("\t             salb    Send Loopback Activate Code (T1/E1 card only)\n");  
	printf("\t             sdlb    Send Loopback Deactive Code (T1/E1 card only)\n");  
	printf("\t             a       Read T1/E1/56K alarms\n");  
	printf("\tFlush Statistics\n");
	printf("\t   f         g       Global Statistics\n");
	printf("\t             c       Communication Error Statistics\n");
	printf("\t             p       Packet Statistics\n");
	printf("\t             lpc     LCP Statistics\n");
	printf("\t             lo      Loopback Detection / LCP Error Statistics\n");
	printf("\t             ipc     IP Control Protocol( IPCP )Statistics\n");
	printf("\t             xpc     IPX Control Protocol( IPXCP )Statistics\n");
	printf("\t             pap     Password Authentication (PAP) Statistics\n");
	printf("\t             chp     Challenge-Handshake Auth.(CHAP) Statistics\n");
	printf("\t             pm      Flush T1/E1 performance monitoring counters\n");
	printf("\tDriver Statistics\n");
	printf("\t   d         ds      Display If_Send Driver Statistics\n");
	printf("\t             di      Display Interrupt Driver Statistics\n");
	printf("\t             dg      Display General Driver Statistics\n"); 
	printf("\n\tExamples:\n");
	printf("\t--------\n\n");
	printf("\tex: ppipemon -i wp1_ppp -u 9000 -c xm    :View Modem Status \n");
	printf("\tex: ppipemon -i 201.1.1.2 -u 9000 -c ti  :Trace and Interpret ALL frames\n\n");
}

void line_trace(int trace_mode) 
{
	unsigned char num_frames, num_chars;
	unsigned short curr_pos = 0;
	trace_pkt_t* frame_info;
	unsigned int i, j;
	unsigned char outstr[2000];
	int recv_buff = sizeof(udp_mgmt_pkt_t) + 100;
	fd_set ready;
	struct timeval to;
	unsigned short tmp;
 
	setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );

        ppp_udp.cblock.command = PPIPE_DISABLE_TRACING;
	ppp_udp.cblock.result = 0xaa;
        ppp_udp.cblock.length = 1;
        ppp_udp.data[0] = DISABLE_TRACING;
        DO_COMMAND(ppp_udp);   	

	ppp_udp.cblock.command = PPIPE_ENABLE_TRACING;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 1;
	ppp_udp.data[0]=0;
	if(trace_mode == TRACE_PROT){
                ppp_udp.data[0] = TRACE_SIGNALLING_FRAMES;
        }else if(trace_mode == TRACE_DATA){
                ppp_udp.data[0] = TRACE_DATA_FRAMES;
        }else{
                ppp_udp.data[0] = TRACE_SIGNALLING_FRAMES | 
			           TRACE_DATA_FRAMES; 
	}
	DO_COMMAND(ppp_udp);

	if( ppp_udp.cblock.result == 0 ) { 
		printf("Starting trace...(Press ENTER to exit)\n");
		fflush(stdout);
	} else if( ppp_udp.cblock.result == 0xCD ) {
		printf("Cannot Enable Line Tracing from Underneath.\n");
		fflush(stdout);
		return;
	} else if( ppp_udp.cblock.result == 0x01 ) {
		printf("Starting trace...(although it's already enabled!)\n");
		printf("Press ENTER to exit.\n");
		fflush(stdout);
	} else {
		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", 
			ppp_udp.cblock.result );
		fflush(stdout);
		return;
	}

	for(;;) {
		FD_ZERO(&ready);
		FD_SET(0,&ready);
		to.tv_sec = 1;
		to.tv_usec = 0;
	
		if(select(1,&ready, NULL, NULL, &to)) {
			break;
		} /* if */

		ppp_udp.cblock.command = PPIPE_GET_TRACE_INFO;
		ppp_udp.cblock.result = 0xaa;
		ppp_udp.cblock.length = 0;
		DO_COMMAND(ppp_udp);

		if (ppp_udp.cblock.result == 0 && ppp_udp.cblock.length) { 
			/* if we got something back then get number of frames */
			num_frames = ppp_udp.data[0] >> 2;
			for ( i=0; i<num_frames; i++) {
				
				frame_info = (trace_pkt_t*)&ppp_udp.data[curr_pos];

				/*  frame type */
				if (frame_info->status & 0x01) {
					sprintf(outstr,"OUTGOING\t");
				} else {
					sprintf(outstr,"INCOMING\t");
				}

				/* real length and time stamp */
				sprintf(outstr+strlen(outstr), "%d\t%d\t", 
					frame_info->real_length, 
					frame_info->time_stamp);

				/* first update curr_pos */
				curr_pos += sizeof(trace_pkt_t);
				
				if (frame_info->data_avail == 0) {
					sprintf( outstr+strlen(outstr), "the frame data is not available" );
				} else {
			 	   /* update curr_pos again */
				   curr_pos += (frame_info->real_length-1);
				   num_chars = (unsigned char)
					((frame_info->real_length <= 25)
					?frame_info->real_length:25);

				   if (raw_data) { /*  show raw data */

					for( j=0; j<num_chars; j++ ) {
						sprintf(outstr+strlen(outstr), "%02X ", (unsigned char)frame_info->data[j]);
					}
					outstr[strlen(outstr)-1] = '\0';

				   } else { /* show int data */
					memcpy(&tmp,&frame_info->data[2] ,2);

					switch(tmp) {

					case 0x2100: //IP Packet
						sprintf(outstr+strlen(outstr),"IP data packet from %d.%d.%d.%d to %d.%d.%d.%d",frame_info->data[16],frame_info->data[17],frame_info->data[18],frame_info->data[19],frame_info->data[20],frame_info->data[21],frame_info->data[22],frame_info->data[23]);
						break;

					case 0x2B00: //IPX Packet
						sprintf(outstr+strlen(outstr),"IPX Data packet");
						break;
					case 0x5780:
						sprintf(outstr+strlen(outstr),"IPCP-v6 packet - ");
						goto decode_control_frame;
					case 0x2180: //IPCP Packet
						sprintf(outstr+strlen(outstr),"IPCP-v4 packet - ");
						goto decode_control_frame;
					case 0x2B80: //IPXCP Packet
						sprintf(outstr+strlen(outstr),"IPXCP packet - ");
					case 0x21C0: //LCP Packet
						sprintf(outstr+strlen(outstr),"LCP packet - ");
decode_control_frame:
						switch(frame_info->data[4]){
						case 1:
							sprintf(outstr+strlen(outstr),"Configure REQUEST");
							break;
						case 2:
							sprintf(outstr+strlen(outstr),"Configure ACK");
							break;
						case 3:
							sprintf(outstr+strlen(outstr),"Configure NACK");
							break;
						case 4:
							sprintf(outstr+strlen(outstr),"Configure REJECT");
							break;
						case 5:
							sprintf(outstr+strlen(outstr),"Terminate REQUEST");
							break;
						case 6:
							sprintf(outstr+strlen(outstr),"Terminate ACK");
							break;
						case 7:
							sprintf(outstr+strlen(outstr),"Code REJECT");
							break;
						case 8:
							sprintf(outstr+strlen(outstr),"Protocol REJECT");
							break;
						case 9:
							sprintf(outstr+strlen(outstr),"Echo REQUEST");
							break;
						case 10:
							sprintf(outstr+strlen(outstr),"Echo REPLY");
							break;
						case 11:
							sprintf(outstr+strlen(outstr),"Discard REQUEST");
							break;
						default:
							sprintf(outstr+strlen(outstr),"Unknown type");
							break;
						} 
						break;

					case 0x23C0: //PAP
						sprintf(outstr+strlen(outstr),"PAP packet - ");
						switch(frame_info->data[4]){
						case 1:
						sprintf(outstr+strlen(outstr),"Authenticate REQUEST");
						break;
						case 2:
						sprintf(outstr+strlen(outstr),"Authenticate ACK");
						break;
						case 3:
						sprintf(outstr+strlen(outstr),"Authenticate NACK");
						break;
						default:
						sprintf(outstr+strlen(outstr),"Unknown type");
						break;
						} //switch
						break;
					case 0x25C0: //LQR
						sprintf(outstr+strlen(outstr),"Link Quality Report");
						break;
					case 0x23C2: //PAP
						sprintf(outstr+strlen(outstr),"CPAP packet - ");
						switch(frame_info->data[4]){
						case 1:
							sprintf(outstr+strlen(outstr),"Challenge");
							break;
						case 2:
							sprintf(outstr+strlen(outstr),"Response");
							break;
						case 3:
							sprintf(outstr+strlen(outstr),"Success");
							break;
						case 4:
							sprintf(outstr+strlen(outstr),"Failure");
							break;
						default:
							sprintf(outstr+strlen(outstr),"Unknown type");
							break;
						} //switch
						break;
					default:
						sprintf(outstr+strlen(outstr),"Unknown type");
						break;
					} 
				   } //if
				} //if

				/*  we have to stay on even boundary here, so 
				    update the curr_pos if needed. It's done 
				    exactly the same way in driver.
 			 	 */
 
				if (curr_pos & 0x0001) curr_pos++;
				printf("%s\n",outstr);
				fflush(stdout);
			} //for
		} //if
		curr_pos = 0;

		if (!(ppp_udp.data[0] &= 0x02)) { 
			sleep(TIMEOUT);
		}
	}

	ppp_udp.cblock.command = PPIPE_DISABLE_TRACING;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
}; /* line_trace */


/* Subroutine for changing modes on a FT1 boards */
void set_FT1_mode(int verbose)
{
	int cnt=0;
	for(;;) { 
		ppp_udp.cblock.command= SET_FT1_MODE;
		ppp_udp.cblock.result = 0xaa;
		ppp_udp.cblock.length = 0;
		DO_COMMAND(ppp_udp);
		if(ppp_udp.cblock.result == 0) {
			if(verbose) {
		                printf(".");
        		        fflush(stdout);
			}
			break;
		}
		if (++cnt == MAX_FT1_RETRY){
			break;
		}
	}
} 

/* This subroutine enables the FT1 monitor on the board */

void set_FT1_monitor_status( unsigned char status) 
{
	fail = 0;
	ppp_udp.cblock.command = FT1_MONITOR_STATUS_CTRL;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 1;
	ppp_udp.data[0] = status; 	
	DO_COMMAND(ppp_udp);
	
	if( ppp_udp.cblock.result != 0 && status){
		fail = 1;
		printf("This cblock.command is only possible with S508/FT1 board!");
	}
} /* set_FT1_monitor_status */



/* Read the data for status of all the lights */

void read_FT1_status( void )
{

	ppp_udp.cblock.command= PPIPE_FT1_READ_STATUS;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp); 
	if(ppp_udp.cblock.result == 0) {
		par_port_A_byte = ppp_udp.data[0];
		par_port_B_byte = ppp_udp.data[1];

		if(!(par_port_A_byte & PP_A_RT_NOT_RED)) {
			FT1_LED.RT_red ++;
		}
		if(!(par_port_A_byte & PP_A_RT_NOT_GREEN)) {
			FT1_LED.RT_green ++;
 		}
		if((par_port_A_byte & (PP_A_RT_NOT_GREEN | PP_A_RT_NOT_RED))
			== (PP_A_RT_NOT_GREEN | PP_A_RT_NOT_RED)) {
			FT1_LED.RT_off ++;
		}
 		if(!(par_port_A_byte & PP_A_LL_NOT_RED)) {
			FT1_LED.LL_red ++;
		}
		else {
			FT1_LED.LL_off ++;
		}
		if(!(par_port_A_byte & PP_A_DL_NOT_RED)) {
			FT1_LED.DL_red ++;
		}
		else {
			FT1_LED.DL_off ++;
		}
		if(!(par_port_B_byte & PP_B_RxD_NOT_GREEN)) {
			FT1_LED.RxD_green ++;
		}
		if(!(par_port_B_byte & PP_B_TxD_NOT_GREEN)) {
			FT1_LED.TxD_green ++;
		}
		if(!(par_port_B_byte & PP_B_ERR_NOT_GREEN)) {
			FT1_LED.ERR_green ++;
		}
		if(!(par_port_B_byte & PP_B_ERR_NOT_RED)) {
			FT1_LED.ERR_red ++;
		}
		if((par_port_B_byte & (PP_B_ERR_NOT_GREEN | PP_B_ERR_NOT_RED))
			== (PP_B_ERR_NOT_GREEN | PP_B_ERR_NOT_RED)) {
			FT1_LED.ERR_off ++;
		}
		if(!(par_port_B_byte & PP_B_INS_NOT_RED)) {
			FT1_LED.INS_red ++;
		}
		if(!(par_port_B_byte & PP_B_INS_NOT_GREEN)) {
			FT1_LED.INS_green ++;
		}
		if((par_port_B_byte & (PP_B_INS_NOT_GREEN | PP_B_INS_NOT_RED))
			== (PP_B_INS_NOT_GREEN | PP_B_INS_NOT_RED)) {
			FT1_LED.INS_off ++;
		}
		if(!(par_port_B_byte & PP_B_ST_NOT_GREEN)) {
			FT1_LED.ST_green ++;
		}
		if(!(par_port_B_byte & PP_B_ST_NOT_RED)) {
			FT1_LED.ST_red ++;
		}
		if((par_port_B_byte & (PP_B_ST_NOT_GREEN | PP_B_ST_NOT_RED))
			== (PP_B_ST_NOT_GREEN | PP_B_ST_NOT_RED)) {
			FT1_LED.ST_off ++;
		}
	}
} /* read_FT1_status */
 

void ppp_driver_stat_ifsend( void )
{
	if_send_stat_t *if_stats;
	ppp_udp.cblock.command = PPIPE_DRIVER_STAT_IFSEND;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	ppp_udp.data[0] = 0;
	DO_COMMAND(ppp_udp);
      	if_stats = (if_send_stat_t *)&ppp_udp.data[0];


      	printf("                                 Total number of If_Send entries:  %ld\n", 
  		if_stats->if_send_entry);
      	printf("                       Number of If_Send entries with SKB = NULL:  %ld\n",
 		if_stats->if_send_skb_null);
      	printf("Number of If_Send entries with broadcast addressed pkt discarded:  %ld\n",
		if_stats->if_send_broadcast);
      	printf("Number of If_Send entries with multicast addressed pkt discarded:  %ld\n",
		if_stats->if_send_multicast);
      	printf("             Number of If_Send entries with CRITICAL_RX_INTR set:  %ld\n",
		if_stats->if_send_critical_ISR);
      	printf("   Number of If_Send entries with Critical set and pkt discarded:  %ld\n",
		if_stats->if_send_critical_non_ISR);
      	printf("                  Number of If_Send entries with Device Busy set:  %ld\n",
		if_stats->if_send_tbusy);
      	printf("              Number of If_Send entries with Device Busy Timeout:  %ld\n",
		if_stats->if_send_tbusy_timeout);
      	printf("           Number of If_Send entries with PTPIPE MONITOR Request:  %ld\n",
		if_stats->if_send_PIPE_request);
      	printf("                 Number of If_Send entries with WAN Disconnected:  %ld\n",
		if_stats->if_send_wan_disconnected);
      	printf("                   Number of If_Send entries with Protocol Error:  %ld\n",
		if_stats->if_send_protocol_error);
      	printf("       Number of If_Send entries with Transmit Interrupt Enabled:  %ld\n",
		if_stats->if_send_tx_int_enabled);
	printf("              Number of If_Send entries with Adapter Send Failed:  %ld\n",
		if_stats->if_send_bfr_not_passed_to_adptr);
      	printf("              Number of If_Send entries with Adapter Send passed:  %ld\n",
		if_stats->if_send_bfr_passed_to_adptr);
// 	printf("               Number of times host irq left disabled in If_Send:  %d\n", 
//		if_stats. ); 

} /* ppp_driver_stat_ifsend */

void ppp_driver_stat_intr( void )
{
	rx_intr_stat_t *rx_stat;
	global_stats_t *g_stat;
      	ppp_udp.cblock.command = PPIPE_DRIVER_STAT_INTR;
	ppp_udp.cblock.result = 0xaa;
     	ppp_udp.cblock.length = 0;
      	ppp_udp.data[0] = 0;
      	DO_COMMAND(ppp_udp);
	g_stat = (global_stats_t *)&ppp_udp.data[0];
	rx_stat = (rx_intr_stat_t *)&ppp_udp.data[sizeof(global_stats_t)];
	
      
      	printf("                                    Number of ISR entries:   %ld\n", 
		g_stat->isr_entry );
      	printf("                  Number of ISR entries with Critical Set:   %ld\n", 
		g_stat->isr_already_critical );
      	printf("                              Number of Receive Interrupt:   %ld\n", 
		g_stat->isr_rx );
      	printf("                             Number of Transmit Interrupt:   %ld\n", 
		g_stat->isr_tx );
      	printf("              Number of ISR entries for Interrupt Testing:   %ld\n", 
		g_stat->isr_intr_test );
      	printf("                             Number of Spurious Interrupt:   %ld\n", 
		g_stat->isr_spurious );
      	printf("       Number of times Transmit Interrupts Enabled in ISR:   %ld\n", 
		g_stat->isr_enable_tx_int );
      	printf("         Number of Receive Interrupts with Corrupt Buffer:   %ld\n", 
		g_stat->rx_intr_corrupt_rx_bfr );
      	printf("              Number of Receive Interrupts with No socket:   %ld\n", 
		rx_stat->rx_intr_no_socket );
      	printf("  Number of Receive Interrupts for PTPIPE MONITOR Request:   %ld\n",
		rx_stat->rx_intr_PIPE_request);
      	printf(" Number of Receive Interrupts with Buffer Passed to Stack:   %ld\n",
		rx_stat->rx_intr_bfr_passed_to_stack);
        printf(" Number of Receive Inter. with Buffer NOT Passed to Stack:   %ld\n",
		rx_stat->rx_intr_bfr_not_passed_to_stack);
      	printf("     Number of Receive Interrupts with Device not started:   %ld\n", 
		g_stat->rx_intr_dev_not_started );
      	printf("    Number of Transmit Interrupts with Device not started:   %ld\n", 
		g_stat->tx_intr_dev_not_started );
     
}


void ppp_driver_stat_gen( void )
{
	pipe_mgmt_stat_t *p_stat;
	global_stats_t *g_stat;
      	ppp_udp.cblock.command = PPIPE_DRIVER_STAT_GEN;
	ppp_udp.cblock.result = 0xaa;
      	ppp_udp.cblock.length = 0;
      	ppp_udp.data[0] = 0;
      	DO_COMMAND(ppp_udp);
	p_stat = (pipe_mgmt_stat_t *)&ppp_udp.data[0];
	g_stat = (global_stats_t *)&ppp_udp.data[sizeof(pipe_mgmt_stat_t)];
     
      	printf("           Number of PTPIPE Monitor call with Kmalloc error:  %ld\n",
		p_stat->UDP_PIPE_mgmt_kmalloc_err );
      	printf("      Number of PTPIPE Monitor call with Adapter Type error:  %ld\n",
		p_stat->UDP_PIPE_mgmt_adptr_type_err );
      	printf("         Number of PTPIPE Monitor call with Direction Error:  %ld\n",
		p_stat->UDP_PIPE_mgmt_direction_err );
      	printf(" Number of PTPIPE Monitor call with Adapter Command Timeout:  %ld\n",
		p_stat->UDP_PIPE_mgmt_adptr_cmnd_timeout );
      	printf("      Number of PTPIPE Monitor call with Adapter Command OK:  %ld\n",
		p_stat->UDP_PIPE_mgmt_adptr_cmnd_OK );
      	printf("  Number of PTPIPE Monitor call with pkt passed to Adapter:   %ld\n",
		p_stat->UDP_PIPE_mgmt_passed_to_adptr );
      	printf("    Number of PTPIPE Monitor call with pkt passed to Stack:   %ld\n",
		p_stat->UDP_PIPE_mgmt_passed_to_stack);
      	printf("              Number of PTPIPE Monitor call with no socket:   %ld\n",
		p_stat->UDP_PIPE_mgmt_no_socket);
      	printf("                                    Number of Poll Entries:   %ld\n",
		g_stat->poll_entry);
      	printf("                  Number of Poll Entries with Critical set:   %ld\n",
		g_stat->poll_already_critical);
      	printf("                          Number of Poll Entries Processed:   %ld\n",
		g_stat->poll_processed);
      	printf("            Number of times host irq left disabled in Poll:   %ld\n", 
		g_stat->poll_host_disable_irq);

} /* ppp_driver_stat_ifsend */

void flush_driver_stats( void )
{
      	ppp_udp.cblock.command = PPIPE_FLUSH_DRIVER_STATS;
	ppp_udp.cblock.result = 0xaa;
      	ppp_udp.cblock.length = 0;
      	ppp_udp.data[0] = 0;
      	DO_COMMAND(ppp_udp);

      	printf("All Driver Statistics are Flushed.\n");

}

void ppp_router_up_time( void )
{
     	unsigned long time;
     	ppp_udp.cblock.command = PPIPE_ROUTER_UP_TIME;
	ppp_udp.cblock.result = 0xaa;
     	ppp_udp.cblock.length = 0;
     	ppp_udp.data[0] = 0;
     	DO_COMMAND(ppp_udp);
     
     	time = *(unsigned long*)&ppp_udp.data[0];

     	if (time < 3600) {
		if (time<60) 
     			printf("    Router UP Time:  %ld seconds\n", time);
		else
     			printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     	}else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
}

void set_linelb(int mode)
{
	ppp_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_LINELB_MODE : 
			DEACTIVATE_LINELB_MODE;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to %s line loopback mode.\n",
			(mode == ACTIVATE_LOOPBACK) ? "activate" : "deactivate");
	}else{
		printf("Line loopback mode is %s!\n",
			(mode == ACTIVATE_LOOPBACK) ? "activated" : "deactivated");
	}
}

void set_paylb(int mode)
{
	ppp_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_PAYLB_MODE : 
			DEACTIVATE_PAYLB_MODE;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to %s payload loopback mode.\n",
			(mode == ACTIVATE_LOOPBACK) ? "activate" : "deactivate");
	}else{
		printf("Payload loopback mode is %s!\n",
			(mode == ACTIVATE_LOOPBACK) ? "activated" : "deactivated");
	}
}

void set_ddlb(int mode)
{
	ppp_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_DDLB_MODE : 
			DEACTIVATE_DDLB_MODE;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to %s digital loopback mode.\n",
			(mode == ACTIVATE_LOOPBACK) ? "activate" : "deactivate");
	}else{
		printf("Digital loopback mode is %s!\n",
			(mode == ACTIVATE_LOOPBACK) ? "activated" : "deactivated");
	}
}

void send_lb(int mode)
{
	ppp_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_LB_MODE : 
			DEACTIVATE_LB_MODE;
	ppp_udp.cblock.result = 0xaa;
	ppp_udp.cblock.length = 0;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to send Loopback %s code.\n",
			(mode == ACTIVATE_LOOPBACK) ? "Activate" : "Deactivate");
	}else{
		printf("The Loopback %s code is sent!\n",
			(mode == ACTIVATE_LOOPBACK) ? "Activate" : "Deactivate");
	}
}

void read_te1_56k_stat(void)
{
	unsigned long 	alarms = 0x00;
	pmc_pmon_t*	pmon = NULL;
	unsigned char	adapter_type = 0x00;

	/* Read Adapter Type */
	ppp_udp.cblock.command = PPIPE_GET_MEDIA_TYPE;
	ppp_udp.cblock.length = 0;
    	ppp_udp.cblock.result = 0xaa;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to read Adapter Type.\n");
		return;
	}
	adapter_type = ppp_udp.data[0];
	if (!adapter_type){
		return;
	}

	/* Read T1/E1/56K alarms and T1/E1 performance monitoring counters */
	ppp_udp.cblock.command = PPIPE_TE1_56K_STAT;
	ppp_udp.cblock.length = 0;
    	ppp_udp.cblock.result = 0xaa;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to read T1/E1/56K alarms.\n");
		return;
	}
	alarms = *(unsigned long*)&ppp_udp.data[0];
	if (adapter_type == WANOPT_MEDIA_T1 || adapter_type == WANOPT_MEDIA_E1){
		printf("***** %s Alarms *****\n\n",
			(adapter_type == WANOPT_MEDIA_T1) ? "T1" : "E1");
		printf("ALOS:\t%s\t| LOS:\t%s\n", 
				ALOS_ALARM(alarms), 
				LOS_ALARM(alarms));
		printf("RED:\t%s\t| AIS:\t%s\n", 
				RED_ALARM(alarms), 
				AIS_ALARM(alarms));
		if (adapter_type == WANOPT_MEDIA_T1){ 
			printf("YEL:\t%s\t| OOF:\t%s\n", 
					YEL_ALARM(alarms), 
					OOF_ALARM(alarms));
		}else{
			printf("OOF:\t%s\n", 
					OOF_ALARM(alarms));
		}
	}else{
		printf("***** 56K CSU/DSU Alarms *****\n\n\n");
	 	printf("In Service:\t\t%s\tData mode idle:\t\t%s\n",
			 	INS_ALARM_56K(alarms), 
			 	DMI_ALARM_56K(alarms));
		
	 	printf("Zero supp. code:\t%s\tCtrl mode idle:\t\t%s\n",
			 	ZCS_ALARM_56K(alarms), 
			 	CMI_ALARM_56K(alarms));

	 	printf("Out of service code:\t%s\tOut of frame code:\t%s\n",
			 	OOS_ALARM_56K(alarms), 
			 	OOF_ALARM_56K(alarms));
		
	 	printf("Valid DSU NL loopback:\t%s\tUnsigned mux code:\t%s\n",
			 	DLP_ALARM_56K(alarms), 
			 	UMC_ALARM_56K(alarms));

	 	printf("Rx loss of signal:\t%s\t\n",
			 	RLOS_ALARM_56K(alarms)); 
	}

	if (adapter_type != WANOPT_MEDIA_T1 && adapter_type != WANOPT_MEDIA_E1){
		return;
	}
	pmon = (pmc_pmon_t*)&ppp_udp.data[sizeof(unsigned long)];
	printf("\n\n***** %s Performance Monitoring Counters *****\n\n",
			(adapter_type == WANOPT_MEDIA_T1) ? "T1" : "E1");
	printf("Framing Bit Error:\t%ld\tLine Code Violation:\t%ld\n", 
			pmon->frm_bit_error,
			pmon->lcv);
	if (adapter_type == WANOPT_MEDIA_T1){
		printf("Out of Frame Errors:\t%ld\tBit Errors:\t\t%ld\n", 
				pmon->oof_errors,
				pmon->bit_errors);
	}else{
		printf("Far End Block Errors:\t%ld\tCRC Errors:\t%ld\n", 
				pmon->far_end_blk_errors,
				pmon->crc_errors);
	}
	return;
}

void flush_te1_pmon(void)
{
	unsigned char	adapter_type = 0x00;

	/* Read Adapter Type */
	ppp_udp.cblock.command = PPIPE_GET_MEDIA_TYPE;
	ppp_udp.cblock.length = 0;
    	ppp_udp.cblock.result = 0xaa;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to read Adapter Type.\n");
		return;
	}
	adapter_type = ppp_udp.data[0];
	if (!adapter_type || adapter_type == WANOPT_MEDIA_56K){
		return;
	}

	/* Read T1/E1/56K alarms */
	ppp_udp.cblock.command = PPIPE_FLUSH_TE1_PMON;
	ppp_udp.cblock.length = 0;
    	ppp_udp.cblock.result = 0xaa;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to flush T1/E1 Performance Monitoring counters.\n");
		return;
	}
	return;
}

void read_ft1_te1_config (void)
{
	unsigned char	adapter_type = 0x00;

	/* Read Adapter Type */
	ppp_udp.cblock.command = PPIPE_GET_MEDIA_TYPE;
	ppp_udp.cblock.length = 0;
        ppp_udp.cblock.result = 0xaa;
	DO_COMMAND(ppp_udp);
	if (ppp_udp.cblock.result != 0){
		printf("Failed to read Adapter Type.\n");
		return;
	}
	adapter_type = ppp_udp.data[0];

	if (adapter_type == WANOPT_MEDIA_T1 || adapter_type == WANOPT_MEDIA_E1){
		int num_of_chan = (adapter_type == WANOPT_MEDIA_T1) ? 
						NUM_OF_T1_CHANNELS : 
						NUM_OF_E1_CHANNELS;
		int i = 0, start_chan = 0;

		/* T1/E1 card */
		ppp_udp.cblock.command = PPIPE_TE1_CFG;
		ppp_udp.cblock.length = 0;
		ppp_udp.cblock.result = 0xaa;

		DO_COMMAND(ppp_udp);
		if (ppp_udp.cblock.result != 0){
			printf("CSU/DSU Read Configuration Failed");
		}else{
			sdla_te_cfg_t *te_cfg = (sdla_te_cfg_t *)&ppp_udp.data[0];
			printf("CSU/DSU Conriguration:\n");
			printf("\tMedia type\t%s\n",
				(te_cfg->media == WANOPT_MEDIA_T1) ? "T1" : "E1");
			printf("\tFraming\t\t%s\n",
				(te_cfg->frame == WANOPT_FR_ESF) ? "ESF" : 
				(te_cfg->frame == WANOPT_FR_D4) ? "D4" :
				(te_cfg->frame == WANOPT_FR_CRC4) ? "CRC4" :
				(te_cfg->frame == WANOPT_FR_NCRC4) ? "non-CRC4" :
				(te_cfg->frame == WANOPT_FR_ESF_JAPAN) ? "ESF-Japan" :
									"Unknown");
			printf("\tEncoding\t%s\n",
				(te_cfg->lcode == WANOPT_LC_AMI) ? "AMI" : 
				(te_cfg->lcode == WANOPT_LC_B8ZS) ? "B8ZS" :
				(te_cfg->lcode == WANOPT_LC_HDB3) ? "HDB3" :
									"Unknown");
			if (adapter_type == WANOPT_MEDIA_T1){
				printf("\tLine Build\t%s\n",
					(te_cfg->lbo == WANOPT_T1_LBO_0_DB) ? "0dB" :
					(te_cfg->lbo == WANOPT_T1_LBO_75_DB) ? "7.5dB" :
					(te_cfg->lbo == WANOPT_T1_LBO_15_DB) ? "15dB" :
					(te_cfg->lbo == WANOPT_T1_LBO_225_DB) ? "22.5dB" :
					(te_cfg->lbo == WANOPT_T1_0_110) ? "0-110 ft." :
					(te_cfg->lbo == WANOPT_T1_110_220) ? "110-220 ft." :
					(te_cfg->lbo == WANOPT_T1_220_330) ? "220-330 ft." :
					(te_cfg->lbo == WANOPT_T1_330_440) ? "330-440 ft." :
					(te_cfg->lbo == WANOPT_T1_440_550) ? "440-550 ft." :
					(te_cfg->lbo == WANOPT_T1_550_660) ? "550-660 ft." :
										"Unknown");
			}
			printf("\tChannel Base\t");
			for (i = 0, start_chan = 0; i < num_of_chan; i++){
				if (te_cfg->active_ch & (1 << i)){
					if (!start_chan){
						start_chan = i+1;
					}
					//printf("%d ", i+1);
				}else{
					if (start_chan){
						if (start_chan != i + 1){
							printf("%d-%d ", start_chan, i);
						}else{
							printf("%d ", start_chan);
						}
						start_chan = 0;
					}
				}
			} 
			if (start_chan){
				if (start_chan != num_of_chan){
					printf("%d-%d ", start_chan, num_of_chan);
				}else{
					printf("%d ", start_chan);
				}
			}
			printf("\n");
			printf("\tClock Mode\t%s\n",
				(te_cfg->te_clock == WANOPT_NORMAL_CLK) ? "Normal" :
				(te_cfg->te_clock == WANOPT_MASTER_CLK) ? "Master" :
										"Unknown");
		}
	}
	return;
}

void banner (char *title){
	
	int len,i;
	
	len = strlen(title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\t\t%s",title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\n");
	

}

int main(int argc, char* argv[])
{
	int proceed;
	char command;
	char *opt;	

   	printf("\n");
   	if( argc > 2 ) {
     
		init( argc, argv);
     		proceed = MakeUdpConnection();
		command = cmd[0];
		opt   = (char *) &cmd[1];
     		
		if (proceed == TRUE && !ObtainConfiguration()) {
      			printf("Error: Unable to obtain PPP information.\n");
			printf("Make sure the IP and UDP port are correct.\n");
			proceed = FALSE;
	   	} /* if */   

		if(proceed == TRUE){
      			switch(command) {
				
      				case 'x':
					if (!strcmp(opt,"m")){
						modem();
					}else if (!strcmp(opt,"u")){
						timers();	
					}else if (!strcmp(opt,"n")){
						negot();
					}else if (!strcmp(opt,"c")){
						cause();
					}else if (!strcmp(opt,"s")){
						state();
					}else if (!strcmp(opt,"ru")){
	 					ppp_router_up_time();
					}else{
						printf("ERROR: Invalid Status Command 'x', Type ppipemon <cr> for help\n\n");
					}
	 				break;


				case 'c':
					if (!strcmp(opt,"i")){
						ip_config();	
					}else if (!strcmp(opt,"x")){
						ipx_config();
					}else if (!strcmp(opt,"a")){
						authent();
					}else if (!strcmp(opt,"g")){
						general_conf();	
					}else{
						printf("ERROR: Invalid Configuration Command 'c', Type ppipemon <cr> for help\n\n");		
					}
					break;

      				case 's':
					if (!strcmp(opt,"g")){
	 					general_stats();
					}else if (!strcmp(opt,"c")){
						comm_err();
					}else if (!strcmp(opt,"p")){
						packet();
					}else if (!strcmp(opt,"lo")){
						loopback();
					}else if (!strcmp(opt,"pap")){
						pap();
					}else if (!strcmp(opt,"chp")){
						chap();			
					}else if (!strcmp(opt,"ipc")){
						ipcp();
					}else if (!strcmp(opt,"xpc")){
						ipxcp();			
					}else if (!strcmp(opt,"lpc")){
						lcp();
					}else{
						printf("ERROR: Invalid Statistics Command 's', Type ppipemon <cr> for help\n\n");
					}
	 				break;

      				case 't':
	 				if(!strcmp(opt,"i" )){
						line_trace(TRACE_ALL);
					}else if (!strcmp(opt, "ip")){
						line_trace(TRACE_PROT);
					}else if (!strcmp(opt, "id")){
						line_trace(TRACE_DATA);
					}else if (!strcmp(opt, "r")){
	   					raw_data = TRUE;
						line_trace(TRACE_ALL);
					}else if (!strcmp(opt, "rp")){
						raw_data = TRUE;
						line_trace(TRACE_PROT);
					}else if (!strcmp(opt, "rd")){
						raw_data = TRUE;
						line_trace(TRACE_DATA);
					}else{
						printf("ERROR: Invalid Trace Command 't', Type ppipemon <cr> for help\n\n");
					}
	 				break;
      				case 'd':
	 				if(!strcmp(opt,"s" )){
	 					ppp_driver_stat_ifsend();
	 				}else if (!strcmp(opt,"i" )){ 
						ppp_driver_stat_intr();	 
	 				}else if (!strcmp(opt,"g" )){ 
						ppp_driver_stat_gen();	 
					}else{
						printf("ERROR: Invalid Driver Stat Command 't', Type ppipemon <cr> for help\n\n");
					}
					break;
      				case 'f':
	 				if(!strcmp(opt,"g" )){	
	       					flush_general_stats();
	      	 				general_stats();
					}else if (!strcmp(opt,"c" )){
	       					flush_comm_err();
	       					comm_err();
					}else if (!strcmp(opt,"p" )){
	       					flush_packet();
	       					packet();
					}else if (!strcmp(opt,"l" )){
						flush_lcp();
	       					lcp();
					}else if (!strcmp(opt,"lo" )){
	       					flush_loopback();
	       					loopback();
					}else if (!strcmp(opt,"icp" )){
	       					flush_ipcp();
	       					ipcp();
					}else if (!strcmp(opt,"ixp" )){
	       					flush_ipxcp();
	       					ipxcp();
					}else if (!strcmp(opt,"pap" )){
	       					flush_pap();
	       					pap();
					}else if (!strcmp(opt,"chp" )){
	       					flush_chap();
	       					chap();
					}else if (!strcmp(opt,"d" )){
	       					flush_driver_stats();
					}else if (!strcmp(opt,"pm" )){
	       					flush_te1_pmon();
	 				} else {
						printf("ERROR: Invalid Flush Command 'f', Type ppipemon <cr> for help\n\n");
	 				}
	 				break;
      				case 'T':
					if (!strcmp(opt,"v" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							view_FT1_status();
	    					 }
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"s" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){	 	
							FT1_self_test();
	    	 				}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"l" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_local_loop_mode();
             					}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"d" )){
             					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_digital_loop_mode();
	     					}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"r" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_remote_test();
	     					}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"o" )){
	     					set_FT1_monitor_status(0x01);
	     					if(!fail){
							FT1_operational_mode();
	    		 			}
						set_FT1_monitor_status(0x00);
					}else if (!strcmp(opt,"read")){
						read_ft1_te1_config();
					}else if (!strcmp(opt,"allb")){
						set_linelb(ACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"dllb")){
						set_linelb(DEACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"aplb")){
						set_paylb(ACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"dplb")){
						set_paylb(DEACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"adlb")){
						set_ddlb(ACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"ddlb")){
						set_ddlb(DEACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"salb")){
						send_lb(ACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"sdlb")){
						send_lb(DEACTIVATE_LOOPBACK);
					}else if (!strcmp(opt,"a")){
						read_te1_56k_stat();
					} else{
						printf("ERROR: Invalid FT1 Command 'T', Type ppipemon <cr> for help\n\n");
					} 
        				break; 
      				default:
					printf("ERROR: Invalid Command, Type ppipemon <cr> for help\n\n");

      			} //switch
     		} 
     		close( sock );
   	} else {
      		usage();
   	} //if
   printf("\n");
   fflush(stdout);
   return 0;
}; //main

/*
 * EOF ppipemon.c
 */
