/*****************************************************************************
* fpipemon.c	Frame Relay Monitor.
*
* Authors:	Nenad Corbic
* 		Jaspreet Singh	
*
* Copyright:	(c) 1995-2000 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Oct 24, 2001  Nenad Corbic	Added the -full option to display all trace
*                               packets. i.e. disable the default 25 byte 
*                               cutoff.
* Oct 15, 2001  Nenad Corbic    Added the -x option: to format all output in
*                               XML format.
* Mar 14, 2000	Nenad Corbic	Added Raw Socket API support. No IP addresses.
* Mar 22, 1997	Jaspreet Singh	Improved Error handling
* Nov 24, 1997	Jaspreet Singh	Added new stats for driver statistics
* Nov 13, 1997	Jaspreet Singh	Fixed descriptions of Global Error Statistics
* Oct 20, 1997 	Jaspreet Singh	Added new commands for driver specific stats
*				and router up time.
* Jul 28, 1997	Jaspreet Singh	Added a new command for running line trace 
*				displaying RAW data.
* Jul 25, 1997	Jaspreet Singh	Added commands for viewing specific DLCI data 
*				including FECN and BECN. 
* Jun 24, 1997	Jaspreet Singh	S508/FT1 test commands		
* Apr 25, 1997	Farhan Thawar	Initial version based on fpipemon for WinNT.
*****************************************************************************/

/******************************************************************************
 * 			INCLUDE FILES					      *
 *****************************************************************************/
#include <linux/version.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <linux/wanpipe.h>
#include <linux/sdla_fr.h>
#include <linux/if_packet.h>
#include <linux/if_wanpipe.h>
#include <linux/if_ether.h>
#include "ft1_lib.h"


/******************************************************************************
 * 			DEFINES/MACROS					      *
 *****************************************************************************/
#if LINUX_VERSION_CODE >= 0x020100
#define LINUX_2_1
#endif

#define TIMEOUT 1
#define MDATALEN 2024
#define MAX_TRACE_BUF MDATALEN+100
#define TRUE 1
#define FALSE 0
#define MAX_CMD_ARG 10

#define HDR_SIZE sizeof(fr_encap_hdr_t)+sizeof(ip_pkt_t)+sizeof(udp_pkt_t) 
#define CB_SIZE sizeof(wp_mgmt_t)+sizeof(cblock_t)+1

#define DO_COMMAND(packet)	DoCommand(&packet, CB_SIZE + packet.cblock.length)

/******************************************************************************
 * 			TYPEDEF/STRUCTURE				      *
 *****************************************************************************/
typedef struct {
	wp_mgmt_t	wp_mgmt		PACKED;
	cblock_t        cblock          PACKED;
	unsigned char	data[MDATALEN]      PACKED;
} udp_mgmt_pkt_t;

/******************************************************************************
 * 			GLOBAL VARIABLES				      *
 *****************************************************************************/
/* global for now */
int sock;
//struct sockaddr_in soin;
//CBLOCK cb, cbconfig;
udp_mgmt_pkt_t fr_udp;
char codeversion[10];
unsigned int frame_count;
unsigned char station_config;
int raw_data = FALSE;
int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int loop_counter;
int xml_output=0;
int trace_all_data=0;

/* defines for now */
char ipaddress[16];
char src_ip[16];
int udp_port = 9000;
char *cmd[MAX_CMD_ARG];
unsigned short dlci_number = 0;
//char raw_sock=0;
int start_xml_router=0;
int stop_xml_router=1;

/* storage for FT1 LEDs */
FT1_LED_STATUS FT1_LED;

extern char i_name[];
/******************************************************************************
 * 			FUNCTION PROTOTYPES				      *
 *****************************************************************************/
/* Prototypes */
void 	ResetCB(cblock_t *c1);
int 	ObtainConfiguration( void );
void 	init( int , char **);
void 	error( char return_code);
void 	get_ip_addr(char *);
void	banner(char *, int);
int 	output_start_xml_router (void);
int 	output_stop_xml_router (void);
int 	output_start_xml_header (char *hdr);
int 	output_stop_xml_header (void);
void 	output_xml_val_data (char *value_name, int value);
void 	output_xml_val_asc (char *value_name, char * value);
void 	output_error(char *value);
void 	set_FT1_monitor_status( unsigned char);
void 	set_FT1_mode(int verbose);
void 	read_FT1_status( void );
void 	set_linelb(int);
void 	set_paylb(int);
void 	set_ddlb(int);
void 	send_lb(int);
void 	read_te1_56k_stat(void);
void	flush_te1_pmon(void);
void read_ft1_te1_config (void);

extern int MakeUdpConnection(void);
#ifdef LINUX_2_1
extern int MakeRawConnection(void);
#endif
extern unsigned char DoCommand(void*, int);

/******************************************************************************
 * 			FUNCTION DEFINITION				      *
 *****************************************************************************/
void ResetCB(cblock_t *cblock)
{
	memset((void *)&cblock->opp_flag, 0, sizeof(cblock_t));
}; //ResetCB

int ObtainConfiguration( void ) 
{
   	unsigned char x;
   
   	x = 0;
   	fr_udp.cblock.command = FR_READ_CONFIG;
   	fr_udp.cblock.length = 0;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.dlci = 0;
   	while (DO_COMMAND(fr_udp) != 0 && ++x < 4) {
      		if (fr_udp.cblock.result == 0xaa) {
	 		printf("Error: Command timeout occurred\n"); 
	 		return(FALSE);
      		}
      		if (fr_udp.cblock.result == 0xCC ) return(FALSE);
   	}
   
	if (x >= 4) return(FALSE);
   	station_config = fr_udp.data[0];
   
   	strcpy(codeversion, "?.??");
   
   	fr_udp.cblock.command = FR_READ_CODE_VERSION;
      	fr_udp.cblock.result = 0xaa;
  	fr_udp.cblock.length = 0;
   	DO_COMMAND(fr_udp);

   	if (fr_udp.cblock.result == 0) {
      		fr_udp.data[fr_udp.cblock.length] = 0;
      		strcpy(codeversion, fr_udp.data);
   	}
   	return(TRUE);
}; /* ObtainConfiguration */


void init( int argc, char *argv[]){

	int i,j, i_cnt=0, d_cnt=0, u_cnt=0, c_cnt=0, if_found=0;
	struct in_addr ip_str;
	fr_udp.wp_mgmt.id = 0;

	for (i=0;i<argc;i++){
		if (!strcmp(argv[i],"-i")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid IP or Interface Name, Type fpipemon <cr> for help\n\n");
				exit(0);
			}

			strcpy(ipaddress,argv[i+1]);
			if (inet_aton(ipaddress,&ip_str) != 0 ){
			}else{
				strcpy(i_name,ipaddress);
				if_found=1;
			}
			i_cnt=1;
		}else if (!strcmp(argv[i],"-u")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid UDP PORT, Type fpipemon <cr> for help\n\n");
				exit(0);
			}

		 	if(isdigit(argv[i+1][0])){
				udp_port = atoi(argv[i+1]);
			}else{
				printf("ERROR: Invalid UDP Port, Type fpipemon <cr> for help\n\n");
				exit(0);
			}
			u_cnt=1;
		}else if (!strcmp(argv[i],"-c")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type fpipemon <cr> for help\n\n");
				exit(0);
			}

			for(j=1;(j-1)<MAX_CMD_ARG;j++){
				if (i+j > argc-1)
					break;
				cmd[j-1]=strdup(argv[i+j]);
				c_cnt=1;
			}
		}else if (!strcmp(argv[i], "-d")){

			if (i+1 > argc-1){
				printf("ERROR: Invalid Command, Type fpipemon <cr> for help\n\n");
				exit(0);
			}
			if (isdigit(argv[i+1][0])){
				dlci_number = atoi(argv[i+1]);
			}else{
				printf("ERROR: DLCI must be an integer,Type fpipemon <cr> for help\n\n");
				exit(0);
			} 
			d_cnt=1;
		
		}else if (!strcmp(argv[i], "-x")){
			xml_output=1;

		}else if (!strcmp(argv[i], "-full")){
			trace_all_data=1;
		}
	}

	if (!i_cnt){
		printf("ERROR: No IP address or Interface Name, Type fpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!c_cnt){
		printf("ERROR: No Command, Type fpipemon <cr> for help\n\n");
		exit(0);
	}
	if (!d_cnt){
		/* Default DLCI Number to 0 */
		dlci_number = 0;
	}
	
	if (if_found){
		get_ip_addr(ipaddress);
	}

	strcpy(fr_udp.wp_mgmt.signature, "FPIPE8ND");
};

void get_ip_addr(char *if_name){

	int skfd;
	struct sockaddr_in *sin;
	struct ifreq ifr;

    	if ((skfd = socket(AF_INET, SOCK_STREAM, IPPROTO_IP)) < 0) {
		perror("socket");
		exit(1);
    	}

	strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));

        if (ioctl(skfd, SIOCGIFFLAGS , &ifr) < 0) {
		fprintf(stderr, "Error: Unknown interface: %s\n",if_name);
		exit(0);
    	}

        if (ioctl(skfd, SIOCGIFDSTADDR , &ifr) < 0) {

		strcpy(ipaddress,"0.0.0.0");
	}else{
		sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_dstaddr;
		strcpy(ipaddress,inet_ntoa(sin->sin_addr));
	}

	if (ioctl(skfd, SIOCGIFADDR, &ifr) < 0) {
		
		strcpy(src_ip,"0.0.0.0");
	}else{
		sin = (struct sockaddr_in *)&ifr.ifr_ifru.ifru_addr;
		strcpy(src_ip,inet_ntoa(sin->sin_addr));
	}	
	
	close (skfd);

	return;

}

void error( char return_code ) 
{
	switch( return_code ){
		case 0x04:
			printf("Error: An invalid DLCI was selected\n");
			break;
		case 0x10:
			printf("Error: A modem failure occurred - DCD and/or CTS were found to be unexpectedly low\n");
			break;
		case 0x11:
			printf("Error: The Channel moved from Operative to being Inoperative\n");
			break;
		case 0x12:
			printf("Error: The Channel moved from Inoperative to being Operative\n");
			break;
		case 0x13:
			printf("Error: The Access Node has reported a change in the status of a DLCI or a number of DLCIs\n");
			break;
		case 0x14:
			printf("Error: A Full Status Report included a DLCI or a number of DLCIis which were not included before\n"); 
			break;
		case 0x1F:
			printf("Error: The frame relay command is invalid\n");
			break;
		default:
			break;
   	}		
}; /* error */

void link_status( void ) 
{
	fr_udp.cblock.command = FR_READ_STATUS;
	fr_udp.cblock.length = 0;
      	fr_udp.cblock.result = 0xaa;
	fr_udp.cblock.dlci = 0;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0) {
		error(fr_udp.cblock.result);
		return;
	}

	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("Physical Link Status");
		if (fr_udp.data[0])
			output_xml_val_asc("Channel status","OPERATIVE");
		else
	 		output_xml_val_asc("Channel status", "INOPERATIVE");
		output_stop_xml_header();
		output_stop_xml_router();
	}else{
		banner("PHYSICAL LINK STATUS", 0);

		if (fr_udp.data[0])
			printf("Channel status: OPERATIVE\n");
		else
	 		printf("Channel status: INOPERATIVE\n");
	}
}; /* link_status */
 
void modem_status( void ) 
{
   	fr_udp.cblock.command = FR_READ_MODEM_STATUS;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
   	DO_COMMAND(fr_udp);
   
	if (fr_udp.cblock.result != 0){
		error(fr_udp.cblock.result);
		return;
	}

	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("Modem Status");

      		if (fr_udp.data[0] & 0x08) 
	 		output_xml_val_asc("DCD", "HIGH");
      		else
	 		output_xml_val_asc("DCD", "LOW");
      	
      		if( fr_udp.data[0] & 0x20) 
	 		output_xml_val_asc("CTS", "HIGH");
      		else 
	 		output_xml_val_asc("CTS", "LOW");

		output_stop_xml_header();
		output_stop_xml_router();
		
	}else{
		banner("MODEM STATUS", 0);

      		if (fr_udp.data[0] & 0x08) 
	 		printf("DCD: HIGH\n");
      		else
	 		printf("DCD: LOW\n");
      	
      		if( fr_udp.data[0] & 0x20) 
	 		printf("CTS: HIGH\n");
      		else 
	 		printf("CTS: LOW\n");
      	} 

}; //modem_status
 

void comm_err() 
{
	ResetCB(&fr_udp.cblock);
   	fr_udp.cblock.command = FR_READ_ERROR_STATS;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
   	fr_udp.cblock.dlci = 0;	// for supervisor display
   
	DO_COMMAND(fr_udp);
   
	if (fr_udp.cblock.result == 0 && fr_udp.cblock.length == 0x0A) {
		
		if (xml_output){
			output_start_xml_router();
			output_start_xml_header("Communication Error Statistics");
			output_xml_val_data("Number of receiver overrun errors",fr_udp.data[0]); 
			output_xml_val_data("Number of receiver CRC errors",fr_udp.data[1]);
			output_xml_val_data("Number of abort frames received",fr_udp.data[2]);
			output_xml_val_data("Number of times receiver disabled (buffers full)",fr_udp.data[3]);
			output_xml_val_data("Number of abort frames transmitted",fr_udp.data[5]);
			output_xml_val_data("Number of transmit underrun interrupts missed",fr_udp.data[6]);
			output_xml_val_data("Number of times DCD dropped unexpectedly",fr_udp.data[8]);
			output_xml_val_data("Number of times CTS dropped unexpectedly",fr_udp.data[9]);
			output_stop_xml_header();
			output_stop_xml_router();
		}else{
			banner("COMMUNICATION ERROR STATISTICS",0);

			printf("               Number of receiver overrun errors: %d\n",fr_udp.data[0]);
			printf("                   Number of receiver CRC errors: %d\n",fr_udp.data[1]);
			printf("                 Number of abort frames received: %d\n",fr_udp.data[2]);
			printf("Number of times receiver disabled (buffers full): %d\n",fr_udp.data[3]);
			printf("              Number of abort frames transmitted: %d\n",fr_udp.data[5]);
			printf("   Number of transmit underrun interrupts missed: %d\n",fr_udp.data[6]);
			printf("        Number of times DCD dropped unexpectedly: %d\n",fr_udp.data[8]);
			printf("        Number of times CTS dropped unexpectedly: %d\n",fr_udp.data[9]);
		}
   	} else {
      		error(fr_udp.cblock.result);
   	} 
}; /* comm_err(); */

void flush_comm_err( void ) 
{
	fr_udp.cblock.command = FR_FLUSH_ERROR_STATS;
      	fr_udp.cblock.result = 0xaa;
	fr_udp.cblock.length = 0;
	fr_udp.cblock.dlci = 0;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0) 
		error(fr_udp.cblock.result);
}; /* flush_comm_err */

void global_stats( void ) 
{
   	ResetCB(&fr_udp.cblock);
   	fr_udp.cblock.command = FR_READ_STATISTICS;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
   	fr_udp.cblock.dlci = 0;
   	DO_COMMAND(fr_udp);
   
	if (fr_udp.cblock.result != 0){
		error(fr_udp.cblock.result);	
		return;
	}
	
	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("Global Statistics");
      		if( station_config == 0 ) {
	 		output_xml_val_data("Full Status Enquiry messages sent", *(unsigned short*)&fr_udp.data[12]);
	 		output_xml_val_data("Link Integrity Verification Status Enquiry messages sent", *(unsigned short*)&fr_udp.data[14]);
	 		output_xml_val_data("Full Status messages received", *(unsigned short*)&fr_udp.data[16]);
	 		output_xml_val_data("Link Integrity Verification Status messages received", *(unsigned short*)&fr_udp.data[18]);
      		} else {
	 		output_xml_val_data("Full Status Enquiry messages sent", *(unsigned short*)&fr_udp.data[20]);
	 		output_xml_val_data("Link Integrity Verification Status Enquiry messages sent", *(unsigned short*)&fr_udp.data[22]);
	 		output_xml_val_data("Full Status messages received", *(unsigned short*)&fr_udp.data[24]);
	 		output_xml_val_data("Link Integrity Verification Status messages received", *(unsigned short*)&fr_udp.data[26]);
      		} //if
      		output_xml_val_data("CPE initializations", *(unsigned short*)&fr_udp.data[42]);
      		output_xml_val_data("current Send Sequence Number", *(unsigned short*)&fr_udp.data[44]);
      		output_xml_val_data("current Receive Sequence Number", *(unsigned short*)&fr_udp.data[46]);
      		output_xml_val_data("current N392 count", *(unsigned short*)&fr_udp.data[52]);
      		output_xml_val_data("current N393 count", *(unsigned short*)&fr_udp.data[54]);
		output_stop_xml_header();
		output_stop_xml_router();
	}else{
	
		banner("GLOBAL STATISTICS",0);

      		if( station_config == 0 ) {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&fr_udp.data[12]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&fr_udp.data[14]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&fr_udp.data[16]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&fr_udp.data[18]);
      		} else {
	 		printf("                       Full Status Enquiry messages sent: %d\n", *(unsigned short*)&fr_udp.data[20]);
	 		printf("Link Integrity Verification Status Enquiry messages sent: %d\n", *(unsigned short*)&fr_udp.data[22]);
	 		printf("                           Full Status messages received: %d\n", *(unsigned short*)&fr_udp.data[24]);
	 		printf("    Link Integrity Verification Status messages received: %d\n", *(unsigned short*)&fr_udp.data[26]);
      		} //if
      		printf("                                     CPE initializations: %d\n", *(unsigned short*)&fr_udp.data[42]);
      		printf("                            current Send Sequence Number: %d\n", *(unsigned short*)&fr_udp.data[44]);
      		printf("                         current Receive Sequence Number: %d\n", *(unsigned short*)&fr_udp.data[46]);
      		printf("                                      current N392 count: %d\n", *(unsigned short*)&fr_udp.data[52]);
      		printf("                                      current N393 count: %d\n", *(unsigned short*)&fr_udp.data[54]);
   	}
}; /* global_stats */

void flush_global_stats( void ) 
{
   	fr_udp.cblock.command = FR_FLUSH_STATISTICS;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 1;
   	fr_udp.cblock.dlci = 0;
	fr_udp.data[0] = 0x01;
   	DO_COMMAND(fr_udp);
   
	if (fr_udp.cblock.result != 0) { 
		switch(fr_udp.cblock.result){
			case 0x06:
				printf("Error: Global Statistics not flushed\n");
				break;
			default:
				error(fr_udp.cblock.result);
				break;	  
		}
	}	
}; /* flush_global_stats */

void error_stats( void ) 
{
	ResetCB(&fr_udp.cblock);
	fr_udp.cblock.command = FR_READ_STATISTICS;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
   	fr_udp.cblock.dlci = 0;
   	DO_COMMAND(fr_udp);

	if (fr_udp.cblock.result != 0)
		return;
	

	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("Error Statistics");
		output_xml_val_data("I-frames not transmitted after a tx. int. due to exessive frame length",*(unsigned short*)&fr_udp.data[0]);
      		output_xml_val_data("I-frames not transmitted after a tx. int. due to excessive throughput",*(unsigned short*)&fr_udp.data[2]);
      		output_xml_val_data("Received frames discarded as they were either too short or too long",*(unsigned short*)&fr_udp.data[4]);
      		output_xml_val_data("discarded I-frames with unconfigured DLCI",*(unsigned short*)&fr_udp.data[6]);
      		output_xml_val_data("discarded I-frames due to a format error",*(unsigned short*)&fr_udp.data[8]);
      		output_xml_val_data("App. didn't respond to the triggered IRQ within the given timeout period",*(unsigned short*)&fr_udp.data[10]);
      		output_xml_val_data("discarded In-channel Signalling frames due to a format error",*(unsigned short*)&fr_udp.data[28]);
      		output_xml_val_data("In-channel frames received with an invalid Send Seq. Numbers received",*(unsigned short*)&fr_udp.data[32]);
      		output_xml_val_data("In-channel frames received with an invalid Receive Seq. Numbers received",*(unsigned short*)&fr_udp.data[34]);
      		if( station_config == 0 ) {
	 		output_xml_val_data("Number of unsolicited responses from the Access Node",*(unsigned short*)&fr_udp.data[30]);
	 		output_xml_val_data("timeouts on the T391 timer",*(unsigned short*)&fr_udp.data[36]);
	 		output_xml_val_data("consecutive timeouts on the T391 timer",*(unsigned short*)&fr_udp.data[48]);
      		} else {
	 		output_xml_val_data("timeouts on the T392 timer",*(unsigned short*)&fr_udp.data[38]);
	 		output_xml_val_data("consecutive timeouts on the T392 timer",*(unsigned short*)&fr_udp.data[50]);
      		} 
      		output_xml_val_data("times that N392 error threshold was reached during N393 monitored events",*(unsigned short*)&fr_udp.data[40]);
		output_stop_xml_header();
		output_stop_xml_router();
	}else{
	
		banner("ERROR STATISTICS",0);

      		printf("  I-frames not transmitted after a tx. int. due to exessive frame length: %d\n",*(unsigned short*)&fr_udp.data[0]);
      		printf("   I-frames not transmitted after a tx. int. due to excessive throughput: %d\n",*(unsigned short*)&fr_udp.data[2]);
      		printf("     Received frames discarded as they were either too short or too long: %d\n",*(unsigned short*)&fr_udp.data[4]);
      		printf("                               discarded I-frames with unconfigured DLCI: %d\n",*(unsigned short*)&fr_udp.data[6]);
      		printf("                                discarded I-frames due to a format error: %d\n",*(unsigned short*)&fr_udp.data[8]);
      		printf("App. didn't respond to the triggered IRQ within the given timeout period: %d\n",*(unsigned short*)&fr_udp.data[10]);
      		printf("            discarded In-channel Signalling frames due to a format error: %d\n",*(unsigned short*)&fr_udp.data[28]);
      		printf("   In-channel frames received with an invalid Send Seq. Numbers received: %d\n",*(unsigned short*)&fr_udp.data[32]);
      		printf("In-channel frames received with an invalid Receive Seq. Numbers received: %d\n",*(unsigned short*)&fr_udp.data[34]);
      		if( station_config == 0 ) {
	 		printf("                    Number of unsolicited responses from the Access Node: %d\n",*(unsigned short*)&fr_udp.data[30]);
	 		printf("                                              timeouts on the T391 timer: %d\n",*(unsigned short*)&fr_udp.data[36]);
	 		printf("                                  consecutive timeouts on the T391 timer: %d\n",*(unsigned short*)&fr_udp.data[48]);
      		} else {
	 		printf("                                              timeouts on the T392 timer: %d\n",*(unsigned short*)&fr_udp.data[38]);
	 		printf("                                  consecutive timeouts on the T392 timer: %d\n",*(unsigned short*)&fr_udp.data[50]);
      		} 
      		printf("times that N392 error threshold was reached during N393 monitored events: %d\n",*(unsigned short*)&fr_udp.data[40]);
	}

}; /* error_stats */

void line_trace(int trace_mode) 
{
	unsigned char num_frames;
	unsigned int num_chars;
   	unsigned short curr_pos = 0, dlci;
	fpipemon_trc_t *fpipemon_trc;
   	unsigned int i, j;
   	unsigned char outstr[MAX_TRACE_BUF];
   	int recv_buff = sizeof(udp_mgmt_pkt_t) + 100;
  	fd_set ready;
   	struct timeval to;
   
   	setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );

	fr_udp.cblock.command = FPIPE_DISABLE_TRACING;
      	fr_udp.cblock.result = 0xaa;
	fr_udp.cblock.length = 0;
	fr_udp.data[0] = 0;
	DO_COMMAND(fr_udp);

   	fr_udp.cblock.command = FPIPE_ENABLE_TRACING;
      	fr_udp.cblock.result = 0xaa;
	fr_udp.cblock.length = 1;
	fr_udp.data[0] = 0;

	if(trace_mode == TRACE_PROT){
		fr_udp.data[0] |= TRC_SIGNALLING_FRMS | TRC_ACTIVE; 
	}else if(trace_mode == TRACE_DATA){
		fr_udp.data[0] = TRC_INFO_FRMS | TRC_ACTIVE; 
	}else{
		fr_udp.data[0] |= TRC_SIGNALLING_FRMS | TRC_INFO_FRMS | TRC_ACTIVE;
	}
   	DO_COMMAND(fr_udp);

   	if( fr_udp.cblock.result == 0 ) { 
      		printf("Starting trace...(Press ENTER to exit)\n");
		fflush(stdout);
   	} else if (fr_udp.cblock.result == 0x1F) {
      		printf("Line Tracing is possible only with S508 board.\n");
		fflush(stdout);
      		return;
   	} else if( fr_udp.cblock.result == 0xCD ) {
      		printf("Cannot Enable Line Tracing from Underneath.\n");
		fflush(stdout);
      		return;
   	} else if( fr_udp.cblock.result == 0x01 ) {
      		printf("Starting trace...(although it's already enabled!)\n");
      		printf("Press ENTER to exit.\n");
   	} else {
      		printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", fr_udp.cblock.result );
		fflush(stdout);
      		return;
   	}
   
	for(;;) {
      	    FD_ZERO(&ready);
      	    FD_SET(0,&ready);
      	    to.tv_sec = 1;
      	    to.tv_usec = 0;
     
	    if(select(1,&ready, NULL, NULL, &to)) {
	 	break;
      	    }
      
	    fr_udp.cblock.command = FPIPE_GET_TRACE_INFO;
      	    fr_udp.cblock.result = 0xaa;
      	    fr_udp.cblock.length = 0;
      	    DO_COMMAND(fr_udp);
    
	    if (fr_udp.cblock.result == 0 && fr_udp.cblock.length) { 

		/*  get number of frames */
 		num_frames = (fr_udp.data[0] >> 1) & MAX_FRMS_TRACED;

		for (i = 0; i < num_frames; i++) {

			fpipemon_trc = (fpipemon_trc_t *)(&fr_udp.data[curr_pos]);
    		   /*  frame type */
    		   if (fpipemon_trc->fpipemon_trc_hdr.status & TRC_OUTGOING_FRM) {
       		   	sprintf(outstr,"OUTGOING\t");
    		   } else {
       		   	sprintf(outstr, "INCOMING\t");
    		   }

    		   /* real length and time stamp */
    		   sprintf(outstr+strlen(outstr), "%d\t%d\t", fpipemon_trc->fpipemon_trc_hdr.length, fpipemon_trc->fpipemon_trc_hdr.tmstamp);
	    			
		   /*  first update curr_pos */
	    	   curr_pos += sizeof(fpipemon_trc_hdr_t);

			if(fpipemon_trc->fpipemon_trc_hdr.status &
			(TRC_ABORT_ERROR | TRC_CRC_ERROR | TRC_OVERRUN_ERROR)) {
				sprintf(outstr+strlen(outstr), "Receive error - ");

				if(fpipemon_trc->fpipemon_trc_hdr.status & TRC_ABORT_ERROR)
					sprintf(outstr+strlen(outstr), "Abort");
				else
					sprintf(outstr+strlen(outstr),(fpipemon_trc->fpipemon_trc_hdr.status & TRC_CRC_ERROR) ? "Bad CRC" : "Overrun");

			}	
	    	   else if (fpipemon_trc->fpipemon_trc_hdr.length == 0) {
	       	   	sprintf( outstr+strlen(outstr), "the frame data is not available" );
	    	   } else {
	      		/*  update curr_pos again */
	       	  	curr_pos += fpipemon_trc->fpipemon_trc_hdr.length;
			if (!trace_all_data){
	       			num_chars = ((fpipemon_trc->fpipemon_trc_hdr.length <= 25)? fpipemon_trc->fpipemon_trc_hdr.length:25);
			}else{
				num_chars = fpipemon_trc->fpipemon_trc_hdr.length;
			}
			
       			if (raw_data) {
			   /* show raw data */
	 		   for( j=0; j<num_chars; j++ ) {
	     			sprintf(outstr+strlen(outstr), "%02X ", (unsigned char)fpipemon_trc->data[j]);
	  		   }
	  		   outstr[strlen(outstr)-1] = '\0';
	      	  	} else { 
			   /* show intrptd. data */
	  		   dlci = (unsigned short)fpipemon_trc->data[0];
	  		   dlci = (dlci << 2) | (fpipemon_trc->data[1] >> 4);
	  		   if (dlci != 0) {
				/* An Inverse Arp packet */
				if (fpipemon_trc->data[2] == 0x03 &&
				    fpipemon_trc->data[3] == 0x00 &&
				    fpipemon_trc->data[4] == 0x80 &&
				    fpipemon_trc->data[8] == 0x08 &&
				    fpipemon_trc->data[9] == 0x06) {
					/* Request packet */
					if(fpipemon_trc->data[17] == 0x08)
						sprintf(outstr+strlen(outstr), "Inverse Arp \tRequest on DLCI %d", dlci);
					else if(fpipemon_trc->data[17] == 0x09)
						sprintf(outstr+strlen(outstr), "Inverse Arp \tReply on DLCI %d", dlci);
				} else 
	     				sprintf(outstr+strlen(outstr), "Data Frame on DLCI %d", dlci);
	  		   } else {
	     			if(fpipemon_trc->data[9])
		 		    sprintf(outstr+strlen(outstr), "Link Verification");
	     			else
		 	 	    sprintf(outstr+strlen(outstr), "Full Status");
	     			if (fpipemon_trc->data[5] == 0x75)
		 		    sprintf(outstr+strlen(outstr), "\tRequest");
	    	 		else
		 		    sprintf(outstr+strlen(outstr), "\tReply");
	     			
				sprintf(outstr+strlen(outstr), "\tSx %02X  Rx %02X  ", (unsigned char)fpipemon_trc->data[0x0C], (unsigned char)fpipemon_trc->data[0x0D] );
	     			if ( !fpipemon_trc->data[9] && (fpipemon_trc->data[5]==0x7D) ) {
				   /* full status reply */
				   for( j=0x0E; ( ((j+5)<fpipemon_trc->fpipemon_trc_hdr.length) && (j<18)); j+=5 ) {
		   			dlci = (unsigned short)fpipemon_trc->data[j+2];
		   			dlci = (dlci << 4) | ((fpipemon_trc->data[j+3]>>3) & 0x0F);
		   			sprintf(outstr+strlen(outstr), "DLCI %d ", dlci);
		   			if (fpipemon_trc->data[j+4] & 0x08)
		       			   sprintf(outstr+strlen(outstr), "NEW %d", dlci);
		   			else
		       			   sprintf(outstr+strlen(outstr), "PRS %d", dlci);
		   			if (fpipemon_trc->data[j+4] & 0x02)
		       			   sprintf(outstr+strlen(outstr), "ACT\t%d", dlci);
		   			else
		       		  	   sprintf(outstr+strlen(outstr), "INACT\t%d", dlci);
				  }
	     			}
	  		   }
		 	}
	  	   }

		printf("%s\n", outstr);
		fflush(stdout);
		
	 	} /* end of for */
      	   } else if (fr_udp.cblock.result != 0) 
		error(fr_udp.cblock.result);
      	
	   curr_pos = 0;
      	   if (!(fr_udp.data[0] & MORE_TRC_DATA)) {
		 sleep(TIMEOUT);
      	   } 
  	}
   	fr_udp.cblock.command = FPIPE_DISABLE_TRACING;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
   	DO_COMMAND(fr_udp);
}; //line_trace

void list_all_dlcis(void)
{
	int i;
   	fr_udp.cblock.command = FR_READ_STATUS;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
	fr_udp.cblock.dlci = 0;
   	DO_COMMAND(fr_udp);

	if (fr_udp.cblock.result != 0) {
		error(fr_udp.cblock.result);
		return;
	}

	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("Global DLCI Status");
		output_xml_val_asc("Channel Status",fr_udp.data[0] ? "OPERATIVE" : "INOPERATIVE");	
		if (fr_udp.cblock.length > 2){
			unsigned char str_val[50];
			int cnt_val=0;
			for (i=1;i<fr_udp.cblock.length;){
				output_xml_val_data("DLCI",*((unsigned short*)&fr_udp.data[i]));
				if (fr_udp.data[i+2] & 0x40){
					cnt_val=sprintf(str_val,"Included, ");
				}else{
					cnt_val=sprintf(str_val,"Excluded, ");
				}		
			
				cnt_val= sprintf((str_val+cnt_val),"%s",
					(fr_udp.data[i+2] & 0x01) ? "Deleted" :
					(fr_udp.data[i+2] & 0x02) ? "Active" :
					(fr_udp.data[i+2] & 0x04) ? "Waiting" :
					(fr_udp.data[i+2] & 0x08) ? "New" : "Inactive");
				
				*(str_val+cnt_val)='\0';
				
				output_xml_val_asc("Status",str_val);
			}
		}
		output_stop_xml_header();
		output_stop_xml_router();
	}else{
		banner("GLOBAL DLCI STATUS",0);
		
		printf("  Channel Status: %s\n",fr_udp.data[0] ? "OPERATIVE" : "INOPERATIVE");

		if (fr_udp.cblock.length < 2){
			printf("\n  No DLCIs reported by switch!\n");
			return;
		}
		

		printf("\n  DLCI\t\tSTATUS\n");
		for (i=1;i<fr_udp.cblock.length;){

			printf("   %d",
				*((unsigned short*)&fr_udp.data[i]));

			if (fr_udp.data[i+2] & 0x40){
				printf("\t\tIncluded, ");
			}else{
				printf("\t\tExcluded, ");
			}

			printf("%s\n",
				(fr_udp.data[i+2] & 0x01) ? "Deleted" :
				(fr_udp.data[i+2] & 0x02) ? "Active" :
				(fr_udp.data[i+2] & 0x04) ? "Waiting" :
				(fr_udp.data[i+2] & 0x08) ? "New" : "Inactive");
			i+=3;
		}
	}
}


void list_dlcis( void )
{
   	int i,cnt;

   	fr_udp.cblock.command = FR_LIST_ACTIVE_DLCI;
      	fr_udp.cblock.result = 0xaa;
   	fr_udp.cblock.length = 0;
   	DO_COMMAND(fr_udp);
 
	if( fr_udp.cblock.result != 0){
		error(fr_udp.cblock.result);
		return;
   	}	

	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("LIST OF ACTIVE DLCIs");
	
		cnt = fr_udp.cblock.length;
     		if( cnt != 0 ){
       			for(i=0; i< cnt; i+=2){
				output_xml_val_data("DLCI", 
					*(unsigned short*)&fr_udp.data[i]);
       			}
     		}
		output_stop_xml_header();
		output_stop_xml_router();
	}else{
	
	
		banner("LIST OF ACTIVE DLCIs",0);
 
		cnt = fr_udp.cblock.length;
     		if( cnt != 0 ){
       			printf("ACTIVE DLCIs\n");
       			for(i=0; i< cnt; i+=2){
       	  			printf("DLCI: %u\n", *(unsigned short*)&fr_udp.data[i]);	
       			}
     		}else{
       			printf("There are NO ACTIVE DLCIs\n"); 
     		}
   	} 
} /* list_dlcis */

void read_dlci_stat( void )
{
     	fr_udp.cblock.command = FR_READ_STATISTICS;
      	fr_udp.cblock.result = 0xaa;
     	fr_udp.cblock.dlci = dlci_number;
     	fr_udp.cblock.length = 1;
     	fr_udp.data[0] = 0;
     
	DO_COMMAND(fr_udp);
     	if( fr_udp.cblock.result != 0 ){
		error(fr_udp.cblock.result);
		return;
     	}


	if (xml_output){
		char tmp[100];
		sprintf(tmp,"Statistics for dlci %i",dlci_number);
		output_start_xml_router();
		output_start_xml_header(tmp);
	
		if( (fr_udp.cblock.result == 0) && (fr_udp.cblock.length == 0x20)){
			output_xml_val_data("Information frames transmitted", *(unsigned long*)&fr_udp.data[0]);
			output_xml_val_data("Information bytes transmitted", *(unsigned long*)&fr_udp.data[4]);
			output_xml_val_data("Information frames received", *(unsigned long*)&fr_udp.data[8]);
			output_xml_val_data("Information bytes received", *(unsigned long*)&fr_udp.data[12]);
			output_xml_val_data("Received I-frames discarded due to inactive DLCI", *(unsigned long*)&fr_udp.data[20]);
			output_xml_val_data("I-frames received with Discard Eligibility (DE) indicator set", *(unsigned long*)&fr_udp.data[28]); 
		}
	
		fr_udp.cblock.command = FR_READ_ADD_DLC_STATS;
      		fr_udp.cblock.result = 0xaa;
		fr_udp.cblock.dlci = dlci_number;
		fr_udp.cblock.length = 0;
		fr_udp.data[0] = 0;
		DO_COMMAND(fr_udp);
	     
		if( fr_udp.cblock.result == 0 ){
			output_xml_val_data("I-frames received with the FECN bit set", *(unsigned short*)&fr_udp.data[0]);
			output_xml_val_data("I-frames received with the BECN bit set", *(unsigned short*)&fr_udp.data[2]);
		} 
		output_stop_xml_header();
		output_stop_xml_router();
	}else{
	
		banner("STATISTICS FOR DLCI",dlci_number);

		if( (fr_udp.cblock.result == 0) && (fr_udp.cblock.length == 0x20)){
			printf("                                Information frames transmitted: %ld\n", *(unsigned long*)&fr_udp.data[0]);
			printf("                                 Information bytes transmitted: %ld\n", *(unsigned long*)&fr_udp.data[4]);
			printf("                                   Information frames received: %ld\n", *(unsigned long*)&fr_udp.data[8]);
			printf("                                    Information bytes received: %ld\n", *(unsigned long*)&fr_udp.data[12]);
			printf("              Received I-frames discarded due to inactive DLCI: %ld\n", *(unsigned long*)&fr_udp.data[20]);
			printf(" I-frames received with Discard Eligibility (DE) indicator set: %ld\n", *(unsigned long*)&fr_udp.data[28]); 
		}
	     
		fr_udp.cblock.command = FR_READ_ADD_DLC_STATS;
      		fr_udp.cblock.result = 0xaa;
		fr_udp.cblock.dlci = dlci_number;
		fr_udp.cblock.length = 0;
		fr_udp.data[0] = 0;
		DO_COMMAND(fr_udp);
	     
		if( fr_udp.cblock.result == 0 ){
			printf("                       I-frames received with the FECN bit set: %d\n", *(unsigned short*)&fr_udp.data[0]);
			printf("                       I-frames received with the BECN bit set: %d\n", *(unsigned short*)&fr_udp.data[2]);
		
		} else { 
			printf("Error: Please enter a non-zero DLCI\n");
		}
	}
	
} /* read_dlci_stat */

void flush_dlci_stats( void )
{
     	fr_udp.cblock.command = FR_FLUSH_STATISTICS;
      	fr_udp.cblock.result = 0xaa;
     	fr_udp.cblock.dlci = dlci_number;
     	fr_udp.cblock.length = 0;
     	DO_COMMAND(fr_udp);
     	
	if( fr_udp.cblock.result != 0 ) {
		switch( fr_udp.cblock.result ){
			case 0x06:
				printf("DLCI Statistics are not flushed\n");
				break;
			default:
				error(fr_udp.cblock.result);
		}
	}
} /* flush_dlci_stats */

void set_FT1_monitor_status( unsigned char status) 
{
	fail = 0;
     	fr_udp.cblock.command = FR_FT1_STATUS_CTRL;
      	fr_udp.cblock.result = 0xaa;
      	fr_udp.cblock.length = 1;
      	fr_udp.data[0] = status; 	
      	DO_COMMAND(fr_udp);
      
	if( fr_udp.cblock.result != 0 && status){
		fail = 1;
		if( fr_udp.cblock.result == 0xCD )
                	printf("Error:  Cannot run this command from Underneath.\n");
		else 
			printf("Error:  This command is only possible with S508/FT1 board!");
      	}

} /* set_FT1_monitor_status */

void set_FT1_mode(int verbose)
{
	int cnt=0;

	for(;;) { 
		fr_udp.cblock.command= FR_SET_FT1_MODE;
      	fr_udp.cblock.result = 0xaa;
		fr_udp.cblock.length = 0;
		DO_COMMAND(fr_udp);
		if(fr_udp.cblock.result == 0) {
			if(verbose) {
		                printf(".");
        		        fflush(stdout);
			}
			break;
		}else if( fr_udp.cblock.result == 0xCD ){
                	printf("Error: Cannot run this command from Underneath.\n");
			exit(1);
		}
		if (++cnt == MAX_FT1_RETRY){
			//????????
			printf("MAX RETRY REACHED !!!!\n");
			break;
		}
	}
} 

/* Read the data for status of all the lights */

void read_FT1_status( void )
{

	fr_udp.cblock.command= FPIPE_FT1_READ_STATUS;
      	fr_udp.cblock.result = 0xaa;
	fr_udp.cblock.length = 0;
	DO_COMMAND(fr_udp); 
	if(fr_udp.cblock.result == 0) {
		par_port_A_byte = fr_udp.data[0];
		par_port_B_byte = fr_udp.data[1];

		if(!(par_port_A_byte & PP_A_RT_NOT_RED)) {
			FT1_LED.RT_red ++;
		}
		if(!(par_port_A_byte & PP_A_RT_NOT_GREEN)) {
			FT1_LED.RT_green ++;
 		}
		if((par_port_A_byte & (PP_A_RT_NOT_GREEN | PP_A_RT_NOT_RED))
			== (PP_A_RT_NOT_GREEN | PP_A_RT_NOT_RED)) {
			FT1_LED.RT_off ++;
		}
 		if(!(par_port_A_byte & PP_A_LL_NOT_RED)) {
			FT1_LED.LL_red ++;
		}
		else {
			FT1_LED.LL_off ++;
		}
		if(!(par_port_A_byte & PP_A_DL_NOT_RED)) {
			FT1_LED.DL_red ++;
		}
		else {
			FT1_LED.DL_off ++;
		}
		if(!(par_port_B_byte & PP_B_RxD_NOT_GREEN)) {
			FT1_LED.RxD_green ++;
		}
		if(!(par_port_B_byte & PP_B_TxD_NOT_GREEN)) {
			FT1_LED.TxD_green ++;
		}
		if(!(par_port_B_byte & PP_B_ERR_NOT_GREEN)) {
			FT1_LED.ERR_green ++;
		}
		if(!(par_port_B_byte & PP_B_ERR_NOT_RED)) {
			FT1_LED.ERR_red ++;
		}
		if((par_port_B_byte & (PP_B_ERR_NOT_GREEN | PP_B_ERR_NOT_RED))
			== (PP_B_ERR_NOT_GREEN | PP_B_ERR_NOT_RED)) {
			FT1_LED.ERR_off ++;
		}
		if(!(par_port_B_byte & PP_B_INS_NOT_RED)) {
			FT1_LED.INS_red ++;
		}
		if(!(par_port_B_byte & PP_B_INS_NOT_GREEN)) {
			FT1_LED.INS_green ++;
		}
		if((par_port_B_byte & (PP_B_INS_NOT_GREEN | PP_B_INS_NOT_RED))
			== (PP_B_INS_NOT_GREEN | PP_B_INS_NOT_RED)) {
			FT1_LED.INS_off ++;
		}
		if(!(par_port_B_byte & PP_B_ST_NOT_GREEN)) {
			FT1_LED.ST_green ++;
		}
		if(!(par_port_B_byte & PP_B_ST_NOT_RED)) {
			FT1_LED.ST_red ++;
		}
		if((par_port_B_byte & (PP_B_ST_NOT_GREEN | PP_B_ST_NOT_RED))
			== (PP_B_ST_NOT_GREEN | PP_B_ST_NOT_RED)) {
			FT1_LED.ST_off ++;
		}
	}
} /* read_FT1_status */
 

void fr_driver_stat_ifsend( void )
{
	if_send_stat_t *stats;
      	fr_udp.cblock.command = FPIPE_DRIVER_STAT_IFSEND;
      	fr_udp.cblock.result = 0xaa;
      	fr_udp.cblock.length = 0;
      	fr_udp.data[0] = 0;
      	DO_COMMAND(fr_udp);
    
	
	stats = (if_send_stat_t *)&fr_udp.data[0];	


	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("Driver if_send statistics");
		output_xml_val_data("Total Number of If_Send entries",
				stats->if_send_entry);
		output_xml_val_data("Number of If_Send entries with SKB = NULL",
				stats->if_send_skb_null);
		output_xml_val_data("Number of If_Send entries with broadcast addressed packet discarded",
				stats->if_send_broadcast);
		output_xml_val_data("Number of If_Send entries with multicast addressed packet discarded",
			 stats->if_send_multicast);
		output_xml_val_data("Number of If_Send entries with CRITICAL_RX_INTR set", 
			stats->if_send_critical_ISR);
		output_xml_val_data("Number of If_Send entries with Critical set and packet discarded", 
			stats->if_send_critical_non_ISR);
		output_xml_val_data("Number of If_Send entries with Device Busy set", 
			stats->if_send_tbusy);
		output_xml_val_data("Number of If_Send entries with Device Busy Timeout", 
			stats->if_send_tbusy_timeout);
		output_xml_val_data("Number of If_Send entries with FPIPE MONITOR Request", 
			stats->if_send_PIPE_request);
		output_xml_val_data("Number of If_Send entries with WAN Disconnected", 
			stats->if_send_wan_disconnected);
		output_xml_val_data("Number of If_Send entries with DLCI Disconnected", 
			stats->if_send_dlci_disconnected);
		output_xml_val_data("Number of If_Send entries with check for Buffers failed", 
			stats->if_send_no_bfrs);
		output_xml_val_data("Number of If_Send entries with Send failed", 
			stats->if_send_adptr_bfrs_full);
		output_xml_val_data("Number of If_Send entries with Send passed", 
			stats->if_send_bfr_passed_to_adptr);
		output_xml_val_data("Number of Consecutive send failures for a packet", 
			stats->if_send_consec_send_fail);

		output_stop_xml_header();
		output_stop_xml_router();
	}else{
	
		banner("DRIVER IF_SEND STATISTICS",0);

		printf("                                    Total Number of If_Send entries:  %ld\n",
			stats->if_send_entry);
		printf("                          Number of If_Send entries with SKB = NULL:  %ld\n", 
			stats->if_send_skb_null);
		printf("Number of If_Send entries with broadcast addressed packet discarded:  %ld\n",
			 stats->if_send_broadcast);
		printf("Number of If_Send entries with multicast addressed packet discarded:  %ld\n",
			 stats->if_send_multicast);
		printf("                Number of If_Send entries with CRITICAL_RX_INTR set:  %ld\n", 
			stats->if_send_critical_ISR);
		printf("   Number of If_Send entries with Critical set and packet discarded:  %ld\n", 
			stats->if_send_critical_non_ISR);
		printf("                     Number of If_Send entries with Device Busy set:  %ld\n", 
			stats->if_send_tbusy);
		printf("                 Number of If_Send entries with Device Busy Timeout:  %ld\n", 
			stats->if_send_tbusy_timeout);
		printf("               Number of If_Send entries with FPIPE MONITOR Request:  %ld\n", 
			stats->if_send_PIPE_request);
		printf("                    Number of If_Send entries with WAN Disconnected:  %ld\n", 
			stats->if_send_wan_disconnected);
		printf("                   Number of If_Send entries with DLCI Disconnected:  %ld\n", 
			stats->if_send_dlci_disconnected);
		printf("            Number of If_Send entries with check for Buffers failed:  %ld\n", 
			stats->if_send_no_bfrs);
		printf("                         Number of If_Send entries with Send failed:  %ld\n", 
			stats->if_send_adptr_bfrs_full);
		printf("                         Number of If_Send entries with Send passed:  %ld\n", 
			stats->if_send_bfr_passed_to_adptr);
		printf("                   Number of Consecutive send failures for a packet:  %ld\n", 
			stats->if_send_consec_send_fail);
	}
} /* fr_driver_stat_ifsend */

void fr_driver_stat_intr( void )
{

	global_stats_t *global_stats;
	rx_intr_stat_t *rx_intr_stats;	 	
      	fr_udp.cblock.command = FPIPE_DRIVER_STAT_INTR;
      	fr_udp.cblock.result = 0xaa;
      	fr_udp.cblock.length = 0;
      	fr_udp.data[0] = 0;
      	DO_COMMAND(fr_udp);

	
     	global_stats = (global_stats_t *)&fr_udp.data[0];
	rx_intr_stats = (rx_intr_stat_t *)&fr_udp.data[sizeof(global_stats_t)];

	if (xml_output){
		
		output_start_xml_router();
		output_start_xml_header("Driver interrupt statistics");
		
		output_xml_val_data("Number of ISR entries" , 
			global_stats->isr_entry);
		output_xml_val_data("Number of ISR entries with Critical Set" , 
			global_stats->isr_already_critical);
		output_xml_val_data("Number of Receive Interrupt" , 
			global_stats->isr_rx);
		output_xml_val_data("Number of Transmit Interrupt" , 
			global_stats->isr_tx);
		output_xml_val_data("Number of ISR entries for Interrupt Testing" , 
			global_stats->isr_intr_test);
		output_xml_val_data("Number of Spurious Interrupt" , 
			global_stats->isr_spurious);
		output_xml_val_data("Number of Times Transmit Interrupts Enabled in ISR" ,
			global_stats->isr_enable_tx_int);
		output_xml_val_data("Number of Transmit Interrupts with Device Not Started" , 
			global_stats->tx_intr_dev_not_started);
		output_xml_val_data("Number of Receive Interrupts with Corrupt Buffer" , 
			global_stats->rx_intr_corrupt_rx_bfr);
		output_xml_val_data("Number of Receive Interrupts with orphaned DLCI" , 
			global_stats->rx_intr_on_orphaned_DLCI);
		output_xml_val_data("Number of Receive Interrupts with No socket" , 
			rx_intr_stats->rx_intr_no_socket);
		output_xml_val_data("Number of Receive Interrupts with Device Not Started" ,
			rx_intr_stats->rx_intr_dev_not_started);
		output_xml_val_data("Number of Receive Interrupts for FPIPE MONITOR Request" , 
			rx_intr_stats->rx_intr_PIPE_request);
		output_xml_val_data("Number of Receive Interrupts with Buffer discarded" , 
			rx_intr_stats->rx_intr_bfr_not_passed_to_stack);
		output_xml_val_data("Number of Receive Interrupts with Buffer Passed to Stack" , 
			rx_intr_stats->rx_intr_bfr_passed_to_stack);
		
		output_stop_xml_header();
		output_stop_xml_router();

	}else{
		banner ("DRIVER INTERRUPT STATISTICS",0);      
	 
		printf("                                   Number of ISR entries:    %ld\n" , 
			global_stats->isr_entry);
		printf("                 Number of ISR entries with Critical Set:    %ld\n" , 
			global_stats->isr_already_critical);
		printf("                             Number of Receive Interrupt:    %ld\n" , 
			global_stats->isr_rx);
		printf("                            Number of Transmit Interrupt:    %ld\n" , 
			global_stats->isr_tx);
		printf("             Number of ISR entries for Interrupt Testing:    %ld\n" , 
			global_stats->isr_intr_test);
		printf("                            Number of Spurious Interrupt:    %ld\n" , 
			global_stats->isr_spurious);
		printf("      Number of Times Transmit Interrupts Enabled in ISR:    %ld\n" ,
			global_stats->isr_enable_tx_int);
		printf("   Number of Transmit Interrupts with Device Not Started:    %ld\n" , 
			global_stats->tx_intr_dev_not_started);
		printf("        Number of Receive Interrupts with Corrupt Buffer:    %ld\n" , 
			global_stats->rx_intr_corrupt_rx_bfr);
		printf("         Number of Receive Interrupts with orphaned DLCI:    %ld\n" , 
			global_stats->rx_intr_on_orphaned_DLCI);
		printf("             Number of Receive Interrupts with No socket:    %ld\n" , 
			rx_intr_stats->rx_intr_no_socket);
		printf("    Number of Receive Interrupts with Device Not Started:    %ld\n" ,
			rx_intr_stats->rx_intr_dev_not_started);
		printf("  Number of Receive Interrupts for FPIPE MONITOR Request:    %ld\n" , 
			rx_intr_stats->rx_intr_PIPE_request);
		printf("      Number of Receive Interrupts with Buffer discarded:    %ld\n" , 
			rx_intr_stats->rx_intr_bfr_not_passed_to_stack);
		printf("Number of Receive Interrupts with Buffer Passed to Stack:    %ld\n" , 
			rx_intr_stats->rx_intr_bfr_passed_to_stack);
     	}

} /* fr_driver_stat_intr */

void fr_driver_stat_gen( void )
{
	pipe_mgmt_stat_t *pipe_mgmt_stats;
	global_stats_t *global_stats;
      	fr_udp.cblock.command = FPIPE_DRIVER_STAT_GEN;
      	fr_udp.cblock.result = 0xaa;
      	fr_udp.cblock.length = 0;
      	fr_udp.data[0] = 0;
      	DO_COMMAND(fr_udp);
      
	pipe_mgmt_stats = (pipe_mgmt_stat_t *)&fr_udp.data[0];
	global_stats = (global_stats_t *)&fr_udp.data[sizeof(pipe_mgmt_stat_t)];


	if (xml_output){
		output_start_xml_router();
		output_start_xml_header("General Driver Statistics");

		output_xml_val_data("Number of FPIPE Monitor call with kmalloc error", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_kmalloc_err);
		output_xml_val_data("Number of FPIPE Monitor call with Adapter Type error", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_type_err);
		output_xml_val_data("Number of FPIPE Monitor call with Direction Error", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_direction_err);
		output_xml_val_data("Number of FPIPE Monitor call with Adapter Command Timeout", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_cmnd_timeout);
		output_xml_val_data("Number of FPIPE Monitor call with Adapter Command OK", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_cmnd_OK);
		output_xml_val_data("Number of FPIPE Monitor call with Adapter Send passed", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_send_passed);
		output_xml_val_data("Number of FPIPE Monitor call with Adapter Send failed", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_send_failed);
		output_xml_val_data("Number of FPIPE Monitor call with no socket", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_no_socket);
		output_xml_val_data("Number of FPIPE Monitor call with pkt not passed to stack", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_not_passed_to_stack);
		output_xml_val_data("Number of FPIPE Monitor call with pkt passed to stack", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_passed_to_stack);
		output_xml_val_data("Number of Poll Entries", 
			global_stats->poll_entry);
		output_xml_val_data("Number of Poll Entries with Critical set", 
			global_stats->poll_already_critical);
		output_xml_val_data("Number of Poll Entries Processed", 
			global_stats->poll_processed);
		output_xml_val_data("Number of times host irq left disabled in Poll", 
			global_stats->poll_host_disable_irq);
		output_stop_xml_header();
		output_stop_xml_router();

	}else{
		banner("GENERAL DRIVER STATISTICS",0);

		printf("            Number of FPIPE Monitor call with kmalloc error:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_kmalloc_err);
		printf("       Number of FPIPE Monitor call with Adapter Type error:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_type_err);
		printf("          Number of FPIPE Monitor call with Direction Error:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_direction_err);
		printf("  Number of FPIPE Monitor call with Adapter Command Timeout:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_cmnd_timeout);
		printf("       Number of FPIPE Monitor call with Adapter Command OK:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_cmnd_OK);
		printf("      Number of FPIPE Monitor call with Adapter Send passed:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_send_passed);
		printf("      Number of FPIPE Monitor call with Adapter Send failed:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_adptr_send_failed);
		printf("                Number of FPIPE Monitor call with no socket:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_no_socket);
		printf("  Number of FPIPE Monitor call with pkt not passed to stack:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_not_passed_to_stack);
		printf("      Number of FPIPE Monitor call with pkt passed to stack:  %ld\n", 
			pipe_mgmt_stats->UDP_PIPE_mgmt_passed_to_stack);
		printf("                                     Number of Poll Entries:  %ld\n", 
			global_stats->poll_entry);
		printf("                   Number of Poll Entries with Critical set:  %ld\n", 
			global_stats->poll_already_critical);
		printf("                           Number of Poll Entries Processed:  %ld\n", 
			global_stats->poll_processed);
		printf("             Number of times host irq left disabled in Poll:  %ld\n", 
			global_stats->poll_host_disable_irq);
	}
	
} /* fr_driver_stat_gen */

void flush_driver_stats( void )
{

      fr_udp.cblock.command = FPIPE_FLUSH_DRIVER_STATS;
      fr_udp.cblock.result = 0xaa;
      fr_udp.cblock.length = 0;
      fr_udp.data[0] = 0;
      DO_COMMAND(fr_udp);

      printf("All Driver Statistics are Flushed.\n");

} /* flush_driver_stats */

void fr_router_up_time( void )
{
     unsigned long time;
     fr_udp.cblock.command = FPIPE_ROUTER_UP_TIME;
     fr_udp.cblock.result = 0xaa;
     fr_udp.cblock.length = 0;
     fr_udp.data[0] = 0;
     DO_COMMAND(fr_udp);
    
     time = fr_udp.data[0] + (fr_udp.data[1]*256) + (fr_udp.data[2]*65536) + 
					(fr_udp.data[3]*16777216);

     if (xml_output){
	output_start_xml_router();
	output_start_xml_header("Router up time");
	output_xml_val_data("Router UP Time",time);
	output_stop_xml_header();
	output_stop_xml_router();
     }else{ 
	     banner("ROUTER UP TIME",0);
	     
	     if (time < 3600) {
		if (time<60){ 
			printf("    Router UP Time:  %ld seconds\n", time);
		}else{
			printf("    Router UP Time:  %ld minute(s)\n", (time/60));
		}
	     }else{
			printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
	     }
     }	
      
} /* fr_router_up_time */

void usage( void ) {

	printf("fpipemon: Wanpipe Frame Relay Debugging Utility\n\n");
	printf("Usage:\n");
	printf("-----\n\n");
	printf("fpipemon -i <ip-addr or interface> -u <port> -c <command> -d <dlci num>\n\n");
	printf("\tOption -i: \n");
	printf("\t\tWanpipe remote IP address or\n");
	printf("\t\tWanpipe network interface name (ex: wp1_fr16)\n");   
	printf("\tOption -u: (optional, defaults to 9000)\n");
	printf("\t\tWanpipe UDPPORT specified in /etc/wanpipe#.conf\n");
	printf("\tOption -d: \n");
	printf("\t\tDLCI Number: (optional, defaults to 0) \n");
	printf("\tOption -x: (optional)\n");
	printf("\t\tDisplay all output/results in XML format\n");
	printf("\tOption -full: (optional trace command)\n");
	printf("\t\tDisplay all raw packets in full: default trace pkt len=25bytes\n");
	printf("\tOption -c: \n");
	printf("\t\tFpipemon Command\n"); 
	printf("\t\t\tFirst letter represents a command group:\n"); 
	printf("\t\t\tex: xm = View Modem Status\n");
	printf("\tSupported Commands: x=status : s=statistics : t=trace \n");
	printf("\t                    c=config : T=FT1 stats  : f=flush\n\n");
	printf("\tCommand:  Options:   Description \n");	
	printf("\t-------------------------------- \n\n");    
	printf("\tCard Status\n");
	printf("\t   x         m       Modem Status\n");
	printf("\t             l       Link Status\n");
	printf("\t             ru      Display Router UP time\n");
	printf("\tCard Configuration\n");
	printf("\t   c         l       List Active DLCIs\n");
	printf("\t             lr      List All Reported DLCIs\n");
	printf("\tCard Statistics\n");
	printf("\t   s         g       Global Statistics\n");
	printf("\t             c       Communication Error Statistics\n");
	printf("\t             e       Error Statistics\n");
	printf("\t             d       Read Statistics for a specific DLCI\n");
	printf("\tTrace Data \n");
	printf("\t   t         i       Trace and Interpret frames\n");
	printf("\t             ip      Trace and Interpret PROTOCOL frames only\n");
	printf("\t             id      Trace and Interpret DATA frames only\n");
	printf("\t             r       Trace ALL frames, in RAW format\n");
	printf("\t             rp      Trace PROTOCOL frames only, in RAW format\n");
	printf("\t             rd      Trace DATA frames only, in RAW format\n");
	printf("\tFT1/T1/E1/56K Configuration/Statistics\n");
	printf("\t   T         v       View Status \n");
	printf("\t             s       Self Test\n");
	printf("\t             l       Line Loop Test\n");
	printf("\t             d       Digital Loop Test\n");
	printf("\t             r       Remote Test\n");
	printf("\t             o       Operational Mode\n");
	printf("\t             read    Read CSU/DSU configuration (FT1/T1/E1 card)\n");  
	printf("\t             allb    Active Line Loopback mode (T1/E1 card only)\n");  
	printf("\t             dllb    Deactive Line Loopback mode (T1/E1 card only)\n");  
	printf("\t             aplb    Active Payload Loopback mode (T1/E1 card only)\n");  
	printf("\t             dplb    Deactive Payload Loopback mode (T1/E1 card only)\n");  
	printf("\t             adlb    Active Diagnostic Digital Loopback mode (T1/E1 card only)\n");  
	printf("\t             ddlb    Deactive Diagnostic Digital Loopback mode (T1/E1 card only)\n");  
	printf("\t             salb    Send Loopback Activate Code (T1/E1 card only)\n");  
	printf("\t             sdlb    Send Loopback Deactive Code (T1/E1 card only)\n");  
	printf("\t             a       Read T1/E1/56K alrams\n");  
	printf("\tDriver Statistics\n");
	printf("\t   d         s       Display If_Send Driver Statistics\n");
	printf("\t             i       Display Interrupt Driver Statistics\n");
	printf("\t             g       Display General Driver Statistics\n");	
	printf("\tFlush Statistics\n");
	printf("\t   f         g       Flush Global Statistics\n");
	printf("\t             c       Flush Communication Error Statistics\n");
	printf("\t             e       Flush Error Statistics\n");
	printf("\t             i       Flush DLCI Statistics\n");
	printf("\t             d       Flush Driver Statistics\n");
	printf("\t             p       Flush T1/E1 performance monitoring counters\n");
	printf("\tExamples:\n");
	printf("\t--------\n\n");
	printf("\tex: fpipemon -i wp1_fr16 -u 9000 -c xm   :View Modem Status \n");
	printf("\tex: fpipemon -i 201.1.1.2 -u 9000 -c ti  :Trace and Interpret frames\n");
	printf("\tex: fpipemon -i fr17 -u 9000 -c sd -d 16 :Statistics for DLCI 16 \n\n");

}; //usage

void read_ft1_te1_config (void)
{
	unsigned char	adapter_type = 0x00;

	/* Read Adapter Type */
	fr_udp.cblock.command = FPIPE_GET_MEDIA_TYPE;
	fr_udp.cblock.length = 0;
	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to read Adapter Type.\n");
		return;
	}
	adapter_type = fr_udp.data[0];

	if (adapter_type == WANOPT_MEDIA_T1 || adapter_type == WANOPT_MEDIA_E1){
		int num_of_chan = (adapter_type == WANOPT_MEDIA_T1) ? 
						NUM_OF_T1_CHANNELS : 
						NUM_OF_E1_CHANNELS;
		int i = 0, start_chan = 0;

		/* T1/E1 card */
		fr_udp.cblock.command = FPIPE_TE1_CFG;
		fr_udp.cblock.length = 0;
		fr_udp.cblock.result = 0xaa;

		DO_COMMAND(fr_udp);
		if (fr_udp.cblock.result != 0){
			printf("CSU/DSU Read Configuration Failed");
		}else{
			sdla_te_cfg_t *te_cfg = (sdla_te_cfg_t *)&fr_udp.data[0];
			printf("CSU/DSU Conriguration:\n");
			printf("\tMedia type\t%s\n",
				(te_cfg->media == WANOPT_MEDIA_T1) ? "T1" : "E1");
			printf("\tFraming\t\t%s\n",
				(te_cfg->frame == WANOPT_FR_ESF) ? "ESF" : 
				(te_cfg->frame == WANOPT_FR_D4) ? "D4" :
				(te_cfg->frame == WANOPT_FR_CRC4) ? "CRC4" :
				(te_cfg->frame == WANOPT_FR_NCRC4) ? "non-CRC4" :
				(te_cfg->frame == WANOPT_FR_ESF_JAPAN) ? "ESF-Japan" :
									"Unknown");
			printf("\tEncoding\t%s\n",
				(te_cfg->lcode == WANOPT_LC_AMI) ? "AMI" : 
				(te_cfg->lcode == WANOPT_LC_B8ZS) ? "B8ZS" :
				(te_cfg->lcode == WANOPT_LC_HDB3) ? "HDB3" :
									"Unknown");
			if (adapter_type == WANOPT_MEDIA_T1){
				printf("\tLine Build\t%s\n",
					(te_cfg->lbo == WANOPT_T1_LBO_0_DB) ? "0dB" :
					(te_cfg->lbo == WANOPT_T1_LBO_75_DB) ? "7.5dB" :
					(te_cfg->lbo == WANOPT_T1_LBO_15_DB) ? "15dB" :
					(te_cfg->lbo == WANOPT_T1_LBO_225_DB) ? "22.5dB" :
					(te_cfg->lbo == WANOPT_T1_0_110) ? "0-110 ft." :
					(te_cfg->lbo == WANOPT_T1_110_220) ? "110-220 ft." :
					(te_cfg->lbo == WANOPT_T1_220_330) ? "220-330 ft." :
					(te_cfg->lbo == WANOPT_T1_330_440) ? "330-440 ft." :
					(te_cfg->lbo == WANOPT_T1_440_550) ? "440-550 ft." :
					(te_cfg->lbo == WANOPT_T1_550_660) ? "550-660 ft." :
										"Unknown");
			}
			printf("\tChannel Base\t");
			for (i = 0, start_chan = 0; i < num_of_chan; i++){
				if (te_cfg->active_ch & (1 << i)){
					if (!start_chan){
						start_chan = i+1;
					}
					//printf("%d ", i+1);
				}else{
					if (start_chan){
						if (start_chan != i + 1){
							printf("%d-%d ", start_chan, i);
						}else{
							printf("%d ", start_chan);
						}
						start_chan = 0;
					}
				}
			} 
			if (start_chan){
				if (start_chan != num_of_chan){
					printf("%d-%d ", start_chan, num_of_chan);
				}else{
					printf("%d ", start_chan);
				}
			}
			printf("\n");
			printf("\tClock Mode\t%s\n",
				(te_cfg->te_clock == WANOPT_NORMAL_CLK) ? "Normal" :
				(te_cfg->te_clock == WANOPT_MASTER_CLK) ? "Master" :
										"Unknown");
		}
	}
	return;
}

void banner (char *title, int dlci){
	
	int len,i;
	
	len = strlen(title);
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	if (dlci){
		printf("\n\t\t%s %i",title,dlci);
	}else{
		printf("\n\t\t%s",title);
	}
	printf("\n\t");
	for (i=0;i<(len+16);i++)
		printf("-");
	printf("\n\n");
	

}

int output_start_xml_router (void)
{
	FILE* pipe_fd;
	char host[50];
	char tmp_time[10];
	time_t time_val;
	struct tm *time_tm;

	if (start_xml_router)
		return 0;
	
	/* Get the host name of the router */
	pipe_fd = popen("cat /etc/hostname","r");
	if (!pipe_fd)
		return 1;
	
	fgets(host,sizeof(host)-1,pipe_fd);
	pclose(pipe_fd);
	host[strlen(host)-1]='\0';

	/* Parse time and date */
	time(&time_val);
	time_tm = localtime(&time_val);
	
	printf("<router name=\"%s\">\n",host);

	printf("<date>\n");
	
	strftime(tmp_time,sizeof(tmp_time),"%Y",time_tm);
	printf("<year>%s</year>\n",tmp_time);

	strftime(tmp_time,sizeof(tmp_time),"%m",time_tm);
	printf("<month>%s</month>\n",tmp_time);

	strftime(tmp_time,sizeof(tmp_time),"%d",time_tm);
	printf("<day>%s</day>\n",tmp_time);

	strftime(tmp_time,sizeof(tmp_time),"%H",time_tm);
	printf("<hour>%s</hour>\n",tmp_time);

	strftime(tmp_time,sizeof(tmp_time),"%M",time_tm);
	printf("<minute>%s</minute>\n",tmp_time);

	strftime(tmp_time,sizeof(tmp_time),"%S",time_tm);
	printf("<sec>%s</sec>\n",tmp_time);
	
	printf("</date>\n<output>\n");
	
	return 0;
}

int output_stop_xml_router(void)
{
	if (stop_xml_router)
		return 0;

	printf("</output>\n</router>\n");
	return 0;
}

int output_start_xml_header(char *hdr)
{
	printf(" <header name=\"%s\">\n",hdr);
	return 0;		
}

int output_stop_xml_header(void)
{
	printf(" </header>\n");
	return 0;
}

void output_xml_val_data (char *value_name, int value)
{
	printf("    <value name=\"%s\">\n      %i\n    </value>\n",value_name,value);
}

void output_xml_val_asc(char *value_name, char *value)
{
	printf("    <value name=\"%s\">\n      %s\n    </value>\n",value_name,value);
}

void output_error(char *value)
{
	if (xml_output){
		output_start_xml_router();	
		printf("    <error>\n      %s\n    </error>\n",value);
		output_stop_xml_router();
	}else{
		printf("ERROR: %s\n",value);
	}
}


void sig_handler(int sigint)
{
	fr_udp.cblock.command = FPIPE_DISABLE_TRACING;
      	fr_udp.cblock.result = 0xaa;
	fr_udp.cblock.length = 0;
	fr_udp.data[0] = 0;
	DO_COMMAND(fr_udp);
}


void set_linelb(int mode)
{
	fr_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_LINELB_MODE : 
			DEACTIVATE_LINELB_MODE;
	fr_udp.cblock.length = 0;
      	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to %s line loopback mode.\n",
			(mode == ACTIVATE_LOOPBACK) ? "activate" : "deactivate");
	}else{
		printf("Line loopback mode is %s!\n",
			(mode == ACTIVATE_LOOPBACK) ? "activated" : "deactivated");
	}
}

void set_paylb(int mode)
{
	fr_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_PAYLB_MODE : 
			DEACTIVATE_PAYLB_MODE;
	fr_udp.cblock.length = 0;
      	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to %s payload loopback mode.\n",
			(mode == ACTIVATE_LOOPBACK) ? "activate" : "deactivate");
	}else{
		printf("Payload loopback mode is %s!\n",
			(mode == ACTIVATE_LOOPBACK) ? "activated" : "deactivated");
	}
}

void set_ddlb(int mode)
{
	fr_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_DDLB_MODE : 
			DEACTIVATE_DDLB_MODE;
	fr_udp.cblock.length = 0;
      	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to %s digital loopback mode.\n",
			(mode == ACTIVATE_LOOPBACK) ? "activate" : "deactivate");
	}else{
		printf("Digital loopback mode is %s!\n",
			(mode == ACTIVATE_LOOPBACK) ? "activated" : "deactivated");
	}
}

void send_lb(int mode)
{
	fr_udp.cblock.command = 
		(mode == ACTIVATE_LOOPBACK) ? 
			ACTIVATE_LB_MODE : 
			DEACTIVATE_LB_MODE;
	fr_udp.cblock.length = 0;
      	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to send Loopback %s code.\n",
			(mode == ACTIVATE_LOOPBACK) ? "Activate" : "Deactivate");
	}else{
		printf("The Loopback %s code is sent!\n",
			(mode == ACTIVATE_LOOPBACK) ? "Activate" : "Deactivate");
	}
}

void read_te1_56k_stat(void)
{
	unsigned long 	alarms = 0x00;
	pmc_pmon_t*	pmon = NULL;
	unsigned char	adapter_type = 0x00;

	/* Read Adapter Type */
	fr_udp.cblock.command = FPIPE_GET_MEDIA_TYPE;
	fr_udp.cblock.length = 0;
    	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to read Adapter Type.\n");
		return;
	}
	adapter_type = fr_udp.data[0];
	if (!adapter_type){
		return;
	}

	/* Read T1/E1/56K alarms and T1/E1 performance monitoring counters */
	fr_udp.cblock.command = FPIPE_TE1_56K_STAT;
	fr_udp.cblock.length = 0;
    	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to read T1/E1/56K alarms.\n");
		return;
	}
	alarms = *(unsigned long*)&fr_udp.data[0];
	if (adapter_type == WANOPT_MEDIA_T1 || adapter_type == WANOPT_MEDIA_E1){
		printf("***** %s Alarms *****\n\n",
			(adapter_type == WANOPT_MEDIA_T1) ? "T1" : "E1");
		printf("ALOS:\t%s\t| LOS:\t%s\n", 
				ALOS_ALARM(alarms), 
				LOS_ALARM(alarms));
		printf("RED:\t%s\t| AIS:\t%s\n", 
				RED_ALARM(alarms), 
				AIS_ALARM(alarms));
		if (adapter_type == WANOPT_MEDIA_T1){ 
			printf("YEL:\t%s\t| OOF:\t%s\n", 
					YEL_ALARM(alarms), 
					OOF_ALARM(alarms));
		}else{
			printf("OOF:\t%s\n", 
					OOF_ALARM(alarms));
		}
	}else{
		printf("***** 56K CSU/DSU Alarms *****\n\n\n");
	 	printf("In Service:\t\t%s\tData mode idle:\t\t%s\n",
			 	INS_ALARM_56K(alarms), 
			 	DMI_ALARM_56K(alarms));
	
	 	printf("Zero supp. code:\t%s\tCtrl mode idle:\t\t%s\n",
			 	ZCS_ALARM_56K(alarms), 
			 	CMI_ALARM_56K(alarms));

	 	printf("Out of service code:\t%s\tOut of frame code:\t%s\n",
			 	OOS_ALARM_56K(alarms), 
			 	OOF_ALARM_56K(alarms));
		
	 	printf("Valid DSU NL loopback:\t%s\tUnsigned mux code:\t%s\n",
			 	DLP_ALARM_56K(alarms), 
			 	UMC_ALARM_56K(alarms));

	 	printf("Rx loss of signal:\t%s\t\n",
			 	RLOS_ALARM_56K(alarms)); 
	}

	if (adapter_type != WANOPT_MEDIA_T1 && adapter_type != WANOPT_MEDIA_E1){
		return;
	}
	
	pmon = (pmc_pmon_t*)&fr_udp.data[sizeof(unsigned long)];
	printf("\n\n***** %s Performance Monitoring Counters *****\n\n",
			(adapter_type == WANOPT_MEDIA_T1) ? "T1" : "E1");
	printf("Framing Bit Error:\t%ld\tLine Code Violation:\t%ld\n", 
			pmon->frm_bit_error,
			pmon->lcv);
	if (adapter_type == WANOPT_MEDIA_T1){
		printf("Out of Frame Errors:\t%ld\tBit Errors:\t\t%ld\n", 
				pmon->oof_errors,
				pmon->bit_errors);
	}else{
		printf("Far End Block Errors:\t%ld\tCRC Errors:\t%ld\n", 
				pmon->far_end_blk_errors,
				pmon->crc_errors);
	}
	return;
}

void flush_te1_pmon(void)
{
	unsigned char	adapter_type = 0x00;

	/* Read Adapter Type */
	fr_udp.cblock.command = FPIPE_GET_MEDIA_TYPE;
	fr_udp.cblock.length = 0;
    	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to read Adapter Type.\n");
		return;
	}
	adapter_type = fr_udp.data[0];
	if (!adapter_type || adapter_type == WANOPT_MEDIA_56K){
		return;
	}

	/* Flush perfomance mono\itoring counters */
	fr_udp.cblock.command = FPIPE_FLUSH_TE1_PMON;
	fr_udp.cblock.length = 0;
    	fr_udp.cblock.result = 0xaa;
	DO_COMMAND(fr_udp);
	if (fr_udp.cblock.result != 0){
		printf("Failed to flush Perfomance Monitoring counters.\n");
		return;
	}
	return;
}

int main(int argc, char* argv[])
{
	int proceed=0;
	char command;
	char *opt;
	char *cmd_tmp;
	int i,err=0;

	memset(cmd,0,sizeof(cmd));

	signal(SIGINT,&sig_handler);
	signal(SIGTERM,&sig_handler);
	
	if( argc < 3) {
		usage();
		return -1;
	}
	
	init( argc, argv);

	if (!xml_output){
		printf("\n");
	}
	
	if (!strcmp(ipaddress,"0.0.0.0")){
		#ifdef LINUX_2_1
		proceed = MakeRawConnection();
		#else
		printf("ERROR: Raw API is not supported for kernels lower than 2.2.X\n");
		printf("       Upgrate to 2.2.X kernel for Raw API support\n");
		exit(1);
		#endif
	}else{
		proceed = MakeUdpConnection();
	}

	if (proceed == TRUE && !ObtainConfiguration()) {
      		printf("Error: Unable to obtain Frame Relay information.\n");
		printf("Make sure the IP and UDP port are correct.\n");
		proceed = FALSE;
   	} /* if */   


	if (proceed != TRUE)
		return -1;
	
	if (xml_output){
		start_xml_router=0;
		output_start_xml_router();	
		start_xml_router=1;
	}
	
	for (i=0;i<MAX_CMD_ARG;i++){

		cmd_tmp=cmd[i];
		if (!cmd_tmp){
			err=-1;
			break;
		}
			
		command = cmd_tmp[0];
		opt	= (char *) &cmd_tmp[1];

		if (command == '-'){
			err = -1;
			break;
		}

		switch(command) {
			case 'x':
				if (!strcmp(opt,"m")){
					modem_status();
				}else if (!strcmp(opt, "l")){
					link_status();
				}else if (!strcmp(opt, "ru")){
					fr_router_up_time();
				}else{
					output_error("Invalid Status Command 'x', Type fpipemon <cr> for help");
					err=-1;
				}
				break;
			case 's':
				if (!strcmp(opt,"c")){
					comm_err();
				}else if (!strcmp(opt,"g")){
					global_stats();
				}else if (!strcmp(opt,"e")){
					error_stats();
				}else if (!strcmp(opt,"d")){
					read_dlci_stat();
				}else {
					output_error("Invalid Status Command 's', Type fpipemon <cr> for help");
					err=-1;
				}	
				break;
			case 't':
				if(!strcmp(opt,"i" )){
					line_trace(TRACE_ALL);
				}else if (!strcmp(opt, "ip")){
					line_trace(TRACE_PROT);
				}else if (!strcmp(opt, "id")){
					line_trace(TRACE_DATA);
				}else if (!strcmp(opt, "r")){
					raw_data = TRUE;
					line_trace(TRACE_ALL);
				}else if (!strcmp(opt, "rp")){
					raw_data = TRUE;
					line_trace(TRACE_PROT);
				}else if (!strcmp(opt, "rd")){
					raw_data = TRUE;
					line_trace(TRACE_DATA);
				}else{
					output_error("Invalid Status Command 't', Type fpipemon <cr> for help");
					err=-1;
				}
				break;

			case 'c':
				if (!strcmp(opt, "l")){
					list_dlcis();
					
				}else if (!strcmp(opt, "lr")){
					list_all_dlcis();

				}else{
					output_error("Invalid Status Command 'c', Type fpipemon <cr> for help");
					err=-1;
				}
				break;
			case 'd':
					/* Different signature for Driver Statistics */
				if(!strcmp(opt, "s")){
					fr_driver_stat_ifsend();
				}else if (!strcmp(opt, "i")){
					fr_driver_stat_intr();
				}else if (!strcmp(opt, "g")){
					fr_driver_stat_gen();
				}else{
					output_error("Invalid Status Command 'd', Type fpipemon <cr> for help");
					err=-1;
				}
				break;
			case 'f':
				if (!strcmp(opt, "c")){
					flush_comm_err();
					comm_err();
				}else if (!strcmp(opt, "g")){
					flush_global_stats();
					global_stats();
				}else if (!strcmp(opt, "e")){
					flush_global_stats();
					error_stats();
				}else if (!strcmp(opt, "i")){
					flush_dlci_stats();
					read_dlci_stat();
				}else if (!strcmp(opt, "d")){
					flush_driver_stats();
				}else{
					output_error("Invalid Status Command 'f', Type fpipemon <cr> for help");
					err=-1;
				}
				break;
			case 'T':
				if (!strcmp(opt, "v")){
					set_FT1_monitor_status(0x01);
					if(!fail){
						view_FT1_status();
					}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "s")){
					set_FT1_monitor_status(0x01);
					if(!fail){
						FT1_self_test();
					}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "l")){
					set_FT1_monitor_status(0x01);
					if(!fail){
						FT1_local_loop_mode();
					}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "d")){
					set_FT1_monitor_status(0x01);
					if(!fail){
						FT1_digital_loop_mode();
					}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt, "r")){
					set_FT1_monitor_status(0x01);
					if(!fail){
						FT1_remote_test();		
					}
					set_FT1_monitor_status(0x00);	
				}else if (!strcmp(opt, "o")){
					set_FT1_monitor_status(0x01);
					if(!fail){
						FT1_operational_mode();
					}
					set_FT1_monitor_status(0x00);
				}else if (!strcmp(opt,"read")){
					read_ft1_te1_config();
				}else if (!strcmp(opt,"allb")){
					set_linelb(ACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"dllb")){
					set_linelb(DEACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"aplb")){
					set_paylb(ACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"dplb")){
					set_paylb(DEACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"adlb")){
					set_ddlb(ACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"ddlb")){
					set_ddlb(DEACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"salb")){
					send_lb(ACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"sdlb")){
					send_lb(DEACTIVATE_LOOPBACK);
				}else if (!strcmp(opt,"a")){
					read_te1_56k_stat();
				}else{
					output_error("Invalid Status Command 'T', Type fpipemon <cr> for help");
					err=-1;
				}
				break;	
			default:
				output_error("ERROR: Invalid Command, Type fpipemon <cr> for help");
				err=-1;
				break;

		} //switch
	} //for

	if (xml_output){
		stop_xml_router=0;
		output_stop_xml_router();	
		stop_xml_router=1;
	}

	
	close(sock);

	for (i=0;i<MAX_CMD_ARG;i++){
		if (!cmd[i])
			break;
		free(cmd[i]);
	}

	if (!xml_output){
   		printf("\n");
	}

	fflush(stdout);
	return err;
}; //main

/*
 * EOF fpipemon.c
 */
