/*****************************************************************************
* wanproc.c	WAN Router Module. /proc filesystem interface.
*
*		This module is completely hardware-independent and provides
*		access to the router using Linux /proc filesystem.
*
* Author: 	Gideon Hack	
*
* Copyright:	(c) 1995-1999 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ============================================================================
* Aug 20, 2001  Alex Feldman  	Support SNMP.
* May 25, 2001  Alex Feldman  	Added T1/E1 support (TE1).
* May 23, 2001  Nenad Corbic 	Bug fix supplied by Akash Jain. If the user
*                               copy fails free up allocated memory.
* Jun 02, 1999  Gideon Hack	Updates for Linux 2.2.X kernels.
* Jun 29, 1997	Alan Cox	Merged with 1.0.3 vendor code
* Jan 29, 1997	Gene Kozin	v1.0.1. Implemented /proc read routines
* Jan 30, 1997	Alan Cox	Hacked around for 2.1
* Dec 13, 1996	Gene Kozin	Initial version (based on Sangoma's WANPIPE)
*****************************************************************************/

#include <linux/version.h>
#include <linux/config.h>
#include <linux/stddef.h>	/* offsetof(), etc. */
#include <linux/errno.h>	/* return codes */
#include <linux/kernel.h>
#include <linux/slab.h>		/* kmalloc(), kfree() */
#include <linux/mm.h>		/* verify_area(), etc. */
#include <linux/string.h>	/* inline mem*, str* functions */
#include <asm/byteorder.h>	/* htons(), etc. */
#include <asm/io.h>
#include <linux/wanrouter.h>	/* WAN router API definitions */
#include <linux/sdladrv.h>
#include <linux/wanproc.h>


#if defined(LINUX_2_1) || defined(LINUX_2_4) 
 #include <linux/init.h>	/* __initfunc et al. */
 #include <asm/uaccess.h>       /* copy_to_user */
 #define PROC_STATS_1_FORMAT "%25s: %10lu\n"
 #define PROC_STATS_2_FORMAT "%25s: %10lu %25s: %10lu\n"
 #define PROC_STATS_ALARM_FORMAT "%25s: %10s %25s: %10s\n"
 #define PROC_STATS_STR_FORMAT "%25s: %10s\n"
 #define PROC_STATS_PMON_FORMAT "%25s: %10lu %25s: %10lu\n"

 #ifndef proc_mkdir
 #define proc_mkdir(buf, usbdir) create_proc_entry(buf, S_IFDIR, usbdir)
 #endif

#else
 #define PROC_STATS_1_FORMAT "%25s: %10lu\n"
 #define PROC_STATS_2_FORMAT "%25s: %10lu %25s: %10lu\n"
 #define PROC_STATS_ALARM_FORMAT "%25s: %10s %25s: %10s\n"
 #define PROC_STATS_STR_FORMAT "%25s: %10s\n"
 #define PROC_STATS_PMON_FORMAT "%25s: %10lu %25s: %10lu\n"
 #include <asm/segment.h>	/* kernel <-> user copy */
#endif



/****** Defines and Macros **************************************************/

#ifndef	wp_min
#define wp_min(a,b) (((a)<(b))?(a):(b))
#endif
#ifndef	wp_max
#define wp_max(a,b) (((a)>(b))?(a):(b))
#endif

#define	PROC_BUFSZ	4000	/* buffer size for printing proc info */

#define PROT_UNKNOWN	"Unknown"
#define PROT_DECODE(prot,cap) ((prot == WANCONFIG_FR) ? ((cap)?"FR":"fr") :\
			        (prot == WANCONFIG_X25) ? ((cap)?"X25":"x25") : \
			          (prot == WANCONFIG_PPP) ? ((cap)?"PPP":"ppp") : \
				    (prot == WANCONFIG_CHDLC) ? ((cap)?"CHDLC":"chdlc") :\
				      (prot == WANCONFIG_MPPP) ? ((cap)?"MPPP":"mppp") : \
				        PROT_UNKNOWN )
	
// NEW_PROC
#define CARD_DECODE(wandev)	(!wandev->S514_cpu_no[0]) ? "S508" : 	\
				(IS_TE1(wandev->te_cfg)) ? "S514 TE1" :	\
				(IS_56K(wandev->te_cfg)) ? "S514 56K" : "S514"

/****** Data Types **********************************************************/

typedef struct wan_stat_entry
{
	struct wan_stat_entry *next;
	char *description;		/* description string */
	void *data;			/* -> data */
	unsigned data_type;		/* data type */
} wan_stat_entry_t;

typedef struct wan_proc_entry
{
	struct proc_dir_entry *protocol_entry;
	int count;
} wan_proc_entry_t;

/****** Function Prototypes *************************************************/

extern wanpipe_hw_probe_t *wanpipe_hw_probe_buf;

#ifdef CONFIG_PROC_FS

#define CONF_PCI_FRM	 "%-12s| %-13s| %-9s| %-4u| %-8u| %-5u| %-4s| %-10u|\n"
#define CONF_ISA_FRM	 "%-12s| %-13s| %-9s| %-4u| 0x%-6X| %-5u| %-4s| %-10u|\n"
#define STAT_FRM	 "%-12s| %-9s| %-8s| %-14s|\n"
#define MAP_FRM		 "%-12s| %-50s\n"
#define INTERFACES_FRM	 "%-15s| %-12s|\n"
#define PROC_FR_FRM	 "%-30s\n"
#define PROC_FR_CFG_FRM	 "%-15s| %-12s| %-5u|\n"
#define PROC_FR_STAT_FRM "%-15s| %-12s| %-14s|\n"

// NEW_PROC
/* Strings */
static char conf_hdr[] =
	"Device name | Protocol Map | Adapter  | IRQ | Slot/IO "
	"| If's | CLK | Baud rate |\n";
static char map_hdr[] =
	"Device name | Protocol Map \n";
static char stat_hdr[] =
	"Device name | Protocol | Station | Status        |\n";

static char interfaces_hdr[] =
	"Interface name | Device name |\n";


/* Proc filesystem interface */
static int router_proc_perms(struct inode *, int);
static ssize_t router_proc_read(struct file*, char*, size_t,loff_t*);
static ssize_t router_proc_write(struct file*, const char*, size_t,loff_t*);

/* Methods for preparing data for reading proc entries */

#ifdef LINUX_2_4
static int config_get_info(char* buf, char** start, off_t offs, int len);
static int status_get_info(char* buf, char** start, off_t offs, int len);
static int probe_get_info(char* buf, char** start, off_t offs, int len);
static int wandev_get_info(char* buf, char** start, off_t offs, int len);

static int map_get_info(char* buf, char** start, off_t offs, int len);
static int interfaces_get_info(char* buf, char** start, off_t offs, int len);

static int get_dev_config_info(char* buf, char** start, off_t offs, int len);
static int get_dev_status_info(char* buf, char** start, off_t offs, int len);

#else
static int config_get_info(char* buf, char** start, off_t offs, int len, int dummy);
static int status_get_info(char* buf, char** start, off_t offs, int len, int dummy);
static int probe_get_info(char* buf, char** start, off_t offs, int len, int dummy);
static int wandev_get_info(char* buf, char** start, off_t offs, int len, int dummy);

static int map_get_info(char* buf, char** start, off_t offs, int len, int dummy);
static int interfaces_get_info(char* buf, char** start, off_t offs, int len, int dummy);

static int get_dev_config_info(char* buf, char** start, off_t offs, int len, int dummy);
static int get_dev_status_info(char* buf, char** start, off_t offs, int len, int dummy);
#endif


/* Miscellaneous */

/*
 *	Structures for interfacing with the /proc filesystem.
 *	Router creates its own directory /proc/net/wanrouter with the folowing
 *	entries:
 *	config		device configuration
 *	status		global device statistics
 *	<device>	entry for each WAN device
 */

/*
 *	Generic /proc/net/wanrouter/<file> file and inode operations 
 */



static struct proc_dir_entry *proc_router;

#ifdef LINUX_2_4


static struct file_operations router_fops =
{
	read:		router_proc_read,
	write:		router_proc_write,
};

static struct inode_operations router_inode =
{
	permission:	router_proc_perms,
};

/*
 *	/proc/net/wanrouter/<device> file operations
 */

static struct file_operations wandev_fops =
{
	read:		router_proc_read,
	ioctl:		wanrouter_ioctl,
};

/*
 *	/proc/net/wanrouter 
 */


#else

static struct file_operations router_fops =
{
	NULL,			/* lseek   */
	router_proc_read,	/* read	   */
	router_proc_write,	/* write   */
	NULL,			/* readdir */
	NULL,			/* select  */
	NULL,			/* ioctl   */
	NULL,			/* mmap	   */
	NULL,			/* no special open code	   */
	NULL,			/* flush */
	NULL,			/* no special release code */
	NULL			/* can't fsync */
};

static struct inode_operations router_inode =
{
	&router_fops,
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* follow link */
	NULL,			/* readlink */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	router_proc_perms
};


static struct file_operations wandev_fops =
{
	NULL,			/* lseek   */
	router_proc_read,	/* read	   */
	NULL,			/* write   */
	NULL,			/* readdir */
	NULL,			/* select  */
	wanrouter_ioctl,	/* ioctl   */
	NULL,			/* mmap	   */
	NULL,			/* no special open code	   */
	NULL,			/* flush */
	NULL,			/* no special release code */
	NULL			/* can't fsync */
};

static struct inode_operations wandev_inode =
{
	&wandev_fops,
	NULL,			/* create */
	NULL,			/* lookup */
	NULL,			/* link */
	NULL,			/* unlink */
	NULL,			/* symlink */
	NULL,			/* mkdir */
	NULL,			/* rmdir */
	NULL,			/* mknod */
	NULL,			/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	router_proc_perms
};

#endif


/*
 *	/proc/net/wanrouter/<protocol>
 */
wan_proc_entry_t proc_router_fr;
wan_proc_entry_t proc_router_chdlc;
wan_proc_entry_t proc_router_ppp;
wan_proc_entry_t proc_router_x25;
wan_proc_entry_t proc_router_mppp;


/*
 *	Interface functions
 */

/*
 *	Initialize router proc interface.
 */

int wanrouter_proc_init (void)
{
	struct proc_dir_entry *p;
	proc_router = proc_mkdir(ROUTER_NAME, proc_net);
	if (!proc_router)
		goto fail;
	
	p = create_proc_entry("config",0,proc_router);
	if (!p)
		goto fail_config;
#ifdef LINUX_2_4
	p->proc_fops = &router_fops;
	p->proc_iops = &router_inode;
#else
	p->ops = &router_inode;
	p->nlink = 1;
#endif
	p->get_info = config_get_info;
	
	p = create_proc_entry("status",0,proc_router);
	if (!p)
		goto fail_stat;

#ifdef LINUX_2_4
	p->proc_fops = &router_fops;
	p->proc_iops = &router_inode;
#else
	p->ops = &router_inode;
	p->nlink = 1;
#endif
	p->get_info = status_get_info;
	
	p = create_proc_entry("hwprobe",0,proc_router);
	if (!p)
		goto fail_probe;

#ifdef LINUX_2_4
	p->proc_fops = &router_fops;
	p->proc_iops = &router_inode;
#else
	p->ops = &router_inode;
	p->nlink = 1;
#endif
	p->get_info = probe_get_info;

	// NEW_PROC
	p = create_proc_entry("map",0,proc_router);
	if (!p)
		goto fail_map;
#ifdef LINUX_2_4
	p->proc_fops = &router_fops;
	p->proc_iops = &router_inode;
#else
	p->ops = &router_inode;
	p->nlink = 1;
#endif
	p->get_info = map_get_info;

	// NEW_PROC
	p = create_proc_entry("interfaces",0,proc_router);
	if (!p)
		goto fail_interfaces;

#ifdef LINUX_2_4
	p->proc_fops = &router_fops;
	p->proc_iops = &router_inode;
#else
	p->ops = &router_inode;
	p->nlink = 1;
#endif
	p->get_info = interfaces_get_info;

	/* Initialize protocol proc fs. */
	proc_router_chdlc.count = 0;
	proc_router_chdlc.protocol_entry = NULL;
	proc_router_fr.count = 0;
	proc_router_fr.protocol_entry = NULL;
	proc_router_ppp.count = 0;
	proc_router_ppp.protocol_entry = NULL;
	proc_router_x25.count = 0;
	proc_router_x25.protocol_entry = NULL;
	proc_router_mppp.count = 0;
	proc_router_mppp.protocol_entry = NULL;
	return 0;

fail_interfaces:	
	remove_proc_entry("map", proc_router);
fail_map:	
	remove_proc_entry("hwprobe", proc_router);
fail_probe:
	remove_proc_entry("status", proc_router);
fail_stat:
	remove_proc_entry("config", proc_router);
fail_config:
	remove_proc_entry(ROUTER_NAME, proc_net);
fail:
	return -ENOMEM;
}


/*
 *	Clean up router proc interface.
 */
void wanrouter_proc_cleanup (void)
{
	remove_proc_entry("config", proc_router);
	remove_proc_entry("status", proc_router);
	remove_proc_entry("hwprobe", proc_router);
	remove_proc_entry("map", proc_router);
	remove_proc_entry("interfaces", proc_router);
	remove_proc_entry(ROUTER_NAME,proc_net);
}

/*
 *	Add directory entry for WAN device.
 */

int wanrouter_proc_add (wan_device_t* wandev)
{	
	int err=0;
	
	if (wandev->magic != ROUTER_MAGIC)
		return -EINVAL;


	wandev->dent = create_proc_entry(wandev->name, 0, proc_router);
	if (!wandev->dent)
		return -ENOMEM;

#ifdef LINUX_2_4
	wandev->dent->proc_fops	= &wandev_fops;
	wandev->dent->proc_iops	= &router_inode;
#else
	wandev->dent->ops = &wandev_inode;
	wandev->dent->nlink	= 1;
#endif
	wandev->dent->get_info	= wandev_get_info;
	wandev->dent->data	= wandev;
	
	return err;
}

/*
 *	Delete directory entry for WAN device.
 */
 
int wanrouter_proc_delete(wan_device_t* wandev)
{
	if (wandev->magic != ROUTER_MAGIC)
		return -EINVAL;

	remove_proc_entry(wandev->name, proc_router);
	return 0;
}
/*
 *	Add directory entry for Protocol.
 */

int wanrouter_proc_add_protocol(wan_device_t* wandev)
{
	struct proc_dir_entry*	p = NULL;
	//struct proc_dir_entry**	proc_protocol;
	wan_proc_entry_t*	proc_protocol;

	if (wandev->magic != ROUTER_MAGIC)
		return -EINVAL;

	switch(wandev->config_id){
		case WANCONFIG_FR:
			proc_protocol = &proc_router_fr;
			break;
			
		case WANCONFIG_CHDLC:
			proc_protocol = &proc_router_chdlc;
			break;
			
		case WANCONFIG_PPP:
			proc_protocol = &proc_router_ppp;
			break;
			
		case WANCONFIG_X25:
			proc_protocol = &proc_router_x25;
			break;
			
		case WANCONFIG_MPPP:
			proc_protocol = &proc_router_mppp;
			break;
			
		default:
			proc_protocol = NULL;
			return 0;
			break;
	}

	if (proc_protocol->protocol_entry == NULL){

		proc_protocol->count=0;
		
		/* Create /proc/net/wanrouter/<protocol> directory */
		proc_protocol->protocol_entry = 
			proc_mkdir(PROT_DECODE(wandev->config_id,0), proc_router);
		
		if (!proc_protocol->protocol_entry)
			goto fail;
	
		/* Create /proc/net/wanrouter/<protocol>/config directory */
		p = create_proc_entry("config",0,proc_protocol->protocol_entry);
		if (!p)
			goto fail_config;
#ifdef LINUX_2_4
		p->proc_fops 	= &router_fops;
		p->proc_iops 	= &router_inode;
#else

		p->ops = &router_inode;
		p->nlink = 1;
#endif
		p->get_info 	= get_dev_config_info;
		p->data 	= (void*)wandev->config_id;
	
		/* Create /proc/net/wanrouter/<protocol>/status directory */
		p = create_proc_entry("status",0,proc_protocol->protocol_entry);
		if (!p)
			goto fail_stat;
#ifdef LINUX_2_4
		p->proc_fops 	= &router_fops;
		p->proc_iops 	= &router_inode;
#else
		p->ops = &router_inode;
		p->nlink = 1;
#endif
		p->get_info 	= get_dev_status_info;
		p->data 	= (void*)wandev->config_id;
	}	
	
	/* Create /proc/net/wanrouter/<protocol>/<wanpipe#> directory */
	wandev->link = proc_mkdir(wandev->name, proc_protocol->protocol_entry);
	if (!wandev->link)
		goto fail_link;

	/* Create /proc/net/wanrouter/<protocol>/<wanpipe#>config directory */
	p = create_proc_entry("config",0,wandev->link);
	if (!p)
		goto fail_link_config;
#ifdef LINUX_2_4
	p->proc_fops 	= &router_fops;
	p->proc_iops 	= &router_inode;
#else
	p->ops = &router_inode;
	p->nlink = 1;
#endif
	p->get_info 	= wandev->get_dev_config_info;
	p->write_proc 	= wandev->set_dev_config;
	p->data 	= wandev;

	proc_protocol->count ++;
	return 0;

fail_link_config:
	remove_proc_entry(wandev->name, proc_protocol->protocol_entry);
fail_link:
	remove_proc_entry("status", proc_protocol->protocol_entry);
	return -ENOMEM;

fail_stat:
	remove_proc_entry("config", proc_protocol->protocol_entry);
fail_config:
	remove_proc_entry(PROT_DECODE(wandev->config_id,0), proc_router);
	proc_protocol->protocol_entry = NULL;
fail:
	return -ENOMEM;
}

/*
 *	Delete directory entry for Protocol.
 */
 
int wanrouter_proc_delete_protocol(wan_device_t* wandev)
{
	wan_proc_entry_t*	proc_protocol = NULL;

	if (wandev->magic != ROUTER_MAGIC)
		return -EINVAL;

	switch(wandev->config_id){
		case WANCONFIG_FR:
			proc_protocol = &proc_router_fr;
			break;
			
		case WANCONFIG_CHDLC:
			proc_protocol = &proc_router_chdlc;
			break;
			
		case WANCONFIG_PPP:
			proc_protocol = &proc_router_ppp;
			break;
			
		case WANCONFIG_X25:
			proc_protocol = &proc_router_x25;
			break;
			
		case WANCONFIG_MPPP:
			proc_protocol = &proc_router_mppp;
			break;
			
		default:
			proc_protocol = NULL;
			return 0;
			break;
	}

	remove_proc_entry("config", wandev->link);
	remove_proc_entry(wandev->name, proc_protocol->protocol_entry);
	proc_protocol->count --;
	if (!proc_protocol->count){
		remove_proc_entry("config", proc_protocol->protocol_entry);
		remove_proc_entry("status", proc_protocol->protocol_entry);
		remove_proc_entry(PROT_DECODE(wandev->config_id,0), proc_router);
		proc_protocol->protocol_entry = NULL;
	}
	return 0;
}

/*
 *	Add directory entry for interface.
 */

int wanrouter_proc_add_interface(wan_device_t* wandev, 
				 struct proc_dir_entry** dent,
				 char* if_name,
				 void* priv)
{
	if (wandev->magic != ROUTER_MAGIC)
		return -EINVAL;
	if (wandev->link == NULL || wandev->get_if_info == NULL)
		return -ENODEV;

	*dent = create_proc_entry(if_name, 0, wandev->link);
	if (!*dent)
		return -ENOMEM;
#ifdef LINUX_2_4
	(*dent)->proc_fops	= &router_fops;
	(*dent)->proc_iops	= &router_inode;
#else
	(*dent)->ops = &router_inode;
	(*dent)->nlink = 1;
#endif
	(*dent)->get_info	= wandev->get_if_info;
	(*dent)->write_proc	= wandev->set_if_info;
	(*dent)->data		= priv;

	return 0;
}

/*
 *	Delete directory entry for interface.
 */
 
int wanrouter_proc_delete_interface(wan_device_t* wandev, char* if_name)
{

	if (wandev->magic != ROUTER_MAGIC)
		return -EINVAL;
	if (wandev->link == NULL)
		return -ENODEV;

	remove_proc_entry(if_name, wandev->link);
	return 0;
}

/****** Proc filesystem entry points ****************************************/

/*
 *	Verify access rights.
 */

static int router_proc_perms (struct inode* inode, int op)
{
	return 0;
}

/*
 *	Read router proc directory entry.
 *	This is universal routine for reading all entries in /proc/net/wanrouter
 *	directory.  Each directory entry contains a pointer to the 'method' for
 *	preparing data for that entry.
 *	o verify arguments
 *	o allocate kernel buffer
 *	o call get_info() to prepare data
 *	o copy data to user space
 *	o release kernel buffer
 *
 *	Return:	number of bytes copied to user space (0, if no data)
 *		<0	error
 */

static ssize_t router_proc_read(struct file* file, char* buf, size_t count,
				loff_t *ppos)
{
	struct inode *inode = file->f_dentry->d_inode;
	struct proc_dir_entry* dent;
	char* page;
	int len;

	if (count <= 0)
		return 0;
		
	dent = inode->u.generic_ip;
	if ((dent == NULL) || (dent->get_info == NULL))
		return 0;
		
	page = kmalloc(count, GFP_KERNEL);
	if (page == NULL)
		return -ENOBUFS;
#ifdef LINUX_2_4
	len = dent->get_info(page, dent->data, file->f_pos, count);
#else
	len = dent->get_info(page, dent->data, file->f_pos, count, 0);
#endif
	if (len) {
		if(copy_to_user(buf, page, len)){
			kfree(page);
			return -EFAULT;
		}
		file->f_pos += len;
	}
	kfree(page);
	return len;
}

/*
 *	Write router proc directory entry.
 *	This is universal routine for writing all entries in /proc/net/wanrouter
 *	directory.  Each directory entry contains a pointer to the 'method' for
 *	preparing data for that entry.
 *	o verify arguments
 *	o allocate kernel buffer
 *	o copy data from user space
 *	o call write_info()
 *	o release kernel buffer
 *
 *	Return:	number of bytes copied to user space (0, if no data)
 *		<0	error
 */
static ssize_t router_proc_write (struct file *file, const char *buf, size_t count, 
					loff_t *ppos)
{
	struct inode*		inode = file->f_dentry->d_inode;
	struct proc_dir_entry* 	dent = NULL;
	char* 			page = NULL;
	//int pos, offs, len;

	if (count <= 0)
		return 0;
		
	dent = inode->u.generic_ip;
	if ((dent == NULL) || (dent->write_proc == NULL))
		return count;
		
	page = kmalloc(count, GFP_KERNEL);
	if (page == NULL)
		return -ENOBUFS;
	if (copy_from_user(page, buf, count)){
		kfree(page);
		return -EINVAL;
	}
	page[count-1] = '\0';
		
	// Add supporting Write to proc fs
	count = dent->write_proc(file, page, count, dent->data);
	kfree(page);
	return count;
}

/*
 *	Prepare data for reading 'Config' entry.
 *	Return length of data.
 */
#ifdef LINUX_2_4
static int config_get_info(char* buf, char** start, off_t offs, int len)
#else
static int config_get_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif
{
	int cnt = 0, size = 0;
	wan_device_t* wandev = NULL;
	PROC_ADD_DECL(stop_cnt);

	PROC_ADD_INIT(offs, stop_cnt);

	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, conf_hdr));
	for (wandev = router_devlist; wandev; wandev = wandev->next){
		if (wandev->state){ 
			PROC_ADD_LINE(cnt, 
			      (buf, &cnt, len, offs, &stop_cnt, &size,
			       (wandev->S514_cpu_no[0])?CONF_PCI_FRM:CONF_ISA_FRM,
				wandev->name,
				"N/A",	// FIXME
				CARD_DECODE(wandev),
				wandev->irq,
				(wandev->S514_cpu_no[0])?wandev->S514_slot_no:wandev->ioport,
				wandev->ndev,
				CLK_DECODE(wandev->clocking),
				wandev->bps));
		}
	}

	PROC_ADD_RET(cnt, offs, stop_cnt);
}


/*
 *	Prepare data for reading 'Status' entry.
 *	Return length of data.
 */

#ifdef LINUX_2_4
static int status_get_info(char* buf, char** start, off_t offs, int len)
#else
static int status_get_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif

{
	int cnt = 0, size = 0;
	wan_device_t* wandev;
	PROC_ADD_DECL(stop_cnt);

	PROC_ADD_INIT(offs, stop_cnt);

	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, stat_hdr));

	for (wandev = router_devlist; wandev; wandev = wandev->next){
		if (!wandev->state) continue;
		PROC_ADD_LINE(cnt, 
			      (buf, &cnt, len, offs, &stop_cnt, &size,
			       STAT_FRM,
			       wandev->name,
			       PROT_DECODE(wandev->config_id,1),
			       wandev->config_id == WANCONFIG_FR ? 
					FR_STATION_DECODE(wandev->station) :
					(wandev->config_id == WANCONFIG_X25 ?
			       X25_STATION_DECODE(wandev->station) : ("N/A")),
			       STATE_DECODE(wandev->state)));
	}
	PROC_ADD_RET(cnt, offs, stop_cnt);
}


/*
 *	Prepare data for reading 'MAP' entry.
 *	Return length of data.
 */
#ifdef LINUX_2_4
static int map_get_info(char* buf, char** start, off_t offs, int len)
#else
static int map_get_info(char* buf, char** start, off_t offs, int len, int dummy)	
#endif
{
	int cnt = 0, size = 0;
	wan_device_t* wandev;
	PROC_ADD_DECL(stop_cnt);

	PROC_ADD_INIT(offs, stop_cnt);
	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, map_hdr));

	for (wandev = router_devlist; wandev; wandev = wandev->next){
		if (!wandev->state) continue;
		PROC_ADD_LINE(cnt, 
			      (buf, &cnt, len, offs, &stop_cnt, &size,
			       MAP_FRM,
			       wandev->name,
			       "N/A")); 	//FIXME
	}
	PROC_ADD_RET(cnt, offs, stop_cnt);
}

/*
 *	Prepare data for reading 'Interfaces' entry.
 *	Return length of data.
 */

#ifdef LINUX_2_4
static int interfaces_get_info(char* buf, char** start, off_t offs, int len)
#else
static int interfaces_get_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif
{
	int 		cnt = 0, size = 0;
	wan_device_t* 	wandev = NULL;
	netdevice_t*	dev = NULL;
	PROC_ADD_DECL(stop_cnt);

	PROC_ADD_INIT(offs, stop_cnt);
	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, interfaces_hdr));

	for (wandev = router_devlist;
	     wandev && (cnt < (len - 80));
	     wandev = wandev->next) {
		if (!wandev->state) continue;
		for(dev = wandev->dev; dev; dev=*((netdevice_t**)dev->priv)){
			PROC_ADD_LINE(cnt, 
			      (buf, &cnt, len, offs, &stop_cnt, &size,
			       INTERFACES_FRM,
			       dev->name, wandev->name));
		}
	}
	PROC_ADD_RET(cnt, offs, stop_cnt);
}

#ifdef LINUX_2_4
static int probe_get_info(char* buf, char** start, off_t offs, int len)
#else
static int probe_get_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif
{
	int cnt = 0, i=0, size = 0;
	wanpipe_hw_probe_t *hw_probe;
	PROC_ADD_DECL(stop_cnt);

	PROC_ADD_INIT(offs, stop_cnt);
	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, "-------------------------------\n"));
	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, "| Wanpipe Hardware Probe Info |\n"));
	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, &stop_cnt, &size, "-------------------------------\n"));
	
	for (hw_probe = wanpipe_hw_probe_buf;
	     hw_probe;
	     hw_probe = hw_probe->next) {

		i++;
		PROC_ADD_LINE(cnt, 
		      (buf, &cnt, len, offs, &stop_cnt, &size,
		       "%i. %s\n", i, hw_probe->hw_type)); 
	}
	PROC_ADD_RET(cnt, offs, stop_cnt);
}


/*
 *	Prepare data for reading <device> entry.
 *	Return length of data.
 *
 *	On entry, the 'start' argument will contain a pointer to WAN device
 *	data space.
 */

#ifdef LINUX_2_4
static int wandev_get_info(char* buf, char** start, off_t offs, int len)
#else
static int wandev_get_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif
{
	wan_device_t* wandev = (void*)start;
	int cnt = 0, size = 0;
	int rslt = 0;
	PROC_ADD_DECL(stop_cnt);

	if ((wandev == NULL) || (wandev->magic != ROUTER_MAGIC))
		return 0;
	PROC_ADD_INIT(offs, stop_cnt);
	if (!wandev->state){
		PROC_ADD_LINE(cnt, 
		      (buf, &cnt, len, offs, &stop_cnt, &size, "device is not configured!\n"));
		goto wandev_get_info_end;
	}

	/* Update device statistics */
	if (wandev->update) {

		rslt = wandev->update(wandev);
		if(rslt) {
			switch (rslt) {
			case -EAGAIN:
				PROC_ADD_LINE(cnt, 
					(buf, &cnt, len, offs, &stop_cnt, &size, "Device is busy!\n"));
				break;
					

			default:
				PROC_ADD_LINE(cnt, 
					(buf, &cnt, len, offs, &stop_cnt, &size, "Device is not configured!\n"));
				break;
			}
			goto wandev_get_info_end;
		}
	}

	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"total rx packets", wandev->stats.rx_packets,
		"total tx packets", wandev->stats.tx_packets));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"total rx bytes", wandev->stats.rx_bytes, 
		"total tx bytes", wandev->stats.tx_bytes));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"bad rx packets", wandev->stats.rx_errors, 
		"packet tx problems", wandev->stats.tx_errors));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"rx frames dropped", wandev->stats.rx_dropped,
		"tx frames dropped", wandev->stats.tx_dropped));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"multicast rx packets", wandev->stats.multicast,
		"tx collisions", wandev->stats.collisions));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"rx length errors", wandev->stats.rx_length_errors,
		"rx overrun errors", wandev->stats.rx_over_errors));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"CRC errors", wandev->stats.rx_crc_errors,
		"abort frames", wandev->stats.rx_frame_errors));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_2_FORMAT,
		"rx fifo overrun", wandev->stats.rx_fifo_errors,
		"rx missed packet", wandev->stats.rx_missed_errors));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_1_FORMAT,
		"aborted tx frames", wandev->stats.tx_aborted_errors));
	
	/* TE1 Update T1/E1 alarms and perfomance counters */
	if (wandev->te_cfg.media == WANOPT_MEDIA_T1 ||
	    wandev->te_cfg.media == WANOPT_MEDIA_E1) {
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size, 
			   "=============================== %s Alarms ===============================\n",
		   (wandev->te_cfg.media == WANOPT_MEDIA_T1) ? "T1" : "E1"));
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_ALARM_FORMAT,
			"ALOS", ALOS_ALARM(wandev->cur_te_alarm), 
			"LOS", LOS_ALARM(wandev->cur_te_alarm)));
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_ALARM_FORMAT,
			"RED", RED_ALARM(wandev->cur_te_alarm), 
			"AIS", AIS_ALARM(wandev->cur_te_alarm)));
		if (wandev->te_cfg.media == WANOPT_MEDIA_T1){
			PROC_ADD_LINE(cnt, 
				(buf, &cnt, len, offs, &stop_cnt, &size, 
				 PROC_STATS_ALARM_FORMAT,
				 "YEL", YEL_ALARM(wandev->cur_te_alarm),
				 "OOF", OOF_ALARM(wandev->cur_te_alarm)));
		}else{ 
			PROC_ADD_LINE(cnt, 
				(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_ALARM_FORMAT,
				"OOF", OOF_ALARM(wandev->cur_te_alarm), 
				"", ""));
		}
		
#if 0
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_ALARM_FORMAT,
			"ALTLOS", ALTLOS_ALARM(wandev->cur_te_alarm), 
			"", ""));
		if (wandev->te_cfg.media == WANOPT_MEDIA_E1){
			PROC_ADD_LINE(cnt, 
				(buf, &cnt, len, offs, &stop_cnt, &size,
				 PROC_STATS_ALARM_FORMAT,
				 "OOSMFV", OOSMFV_ALARM(wandev->cur_te_alarm),  
				 "OOCMFV", OOCMFV_ALARM(wandev->cur_te_alarm)));
			PROC_ADD_LINE(cnt, 
				(buf, &cnt, len, offs, &stop_cnt, &size,
				 PROC_STATS_ALARM_FORMAT,
				 "OOOFV", OOOFV_ALARM(wandev->cur_te_alarm), 
				 "RAIV", RAIV_ALARM(wandev->cur_te_alarm)));
		}
#endif
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size, 
			 "=========================== %s PMON counters ============================\n",
			 (wandev->te_cfg.media == WANOPT_MEDIA_T1) ? "T1" : "E1"));
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size, PROC_STATS_PMON_FORMAT,
			 "Framing Bit Error", wandev->te_pmon.frm_bit_error,
			 "Line Code Violation", wandev->te_pmon.lcv));
		if (wandev->te_cfg.media == WANOPT_MEDIA_T1) {
			PROC_ADD_LINE(cnt, 
				(buf, &cnt, len, offs, &stop_cnt, &size,
				 PROC_STATS_PMON_FORMAT,
				 "Out of Frame Errors", wandev->te_pmon.bit_errors,
				 "Bit Errors", wandev->te_pmon.bit_errors));
		} else { 
			PROC_ADD_LINE(cnt, 
				(buf, &cnt, len, offs, &stop_cnt, &size,
				 PROC_STATS_PMON_FORMAT,
				 "Far End Block Errors", wandev->te_pmon.far_end_blk_errors,
				 "CRC Errors", wandev->te_pmon.crc_errors));
		}
	
		
	}else if (wandev->te_cfg.media == WANOPT_MEDIA_56K){

		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size,
			 "\n====================== Rx 56K CSU/DSU Alarms ==============================\n"));
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size,
			 PROC_STATS_ALARM_FORMAT,
			 "In service", INS_ALARM_56K(wandev->k56_alarm), 
			 "Data mode idle", DMI_ALARM_56K(wandev->k56_alarm)));
		
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size,
			 PROC_STATS_ALARM_FORMAT,
			 "Zero supp. code", ZCS_ALARM_56K(wandev->k56_alarm), 
			 "Ctrl mode idle", CMI_ALARM_56K(wandev->k56_alarm)));

		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size,
			 PROC_STATS_ALARM_FORMAT,
			 "Out of service code", OOS_ALARM_56K(wandev->k56_alarm), 
			 "Out of frame code", OOF_ALARM_56K(wandev->k56_alarm)));
		
		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size,
			 PROC_STATS_ALARM_FORMAT,
			 "Valid DSU NL loopback", DLP_ALARM_56K(wandev->k56_alarm), 
			 "Unsigned mux code", UMC_ALARM_56K(wandev->k56_alarm)));

		PROC_ADD_LINE(cnt, 
			(buf, &cnt, len, offs, &stop_cnt, &size,
			 PROC_STATS_ALARM_FORMAT,
			 "Rx loss of signal", RLOS_ALARM_56K(wandev->k56_alarm), 
			 "N/A", "N/A"));

	}

wandev_get_info_end:
	PROC_ADD_RET(cnt, offs, stop_cnt);
}

/*
 *	Prepare data for reading FR 'Config' entry.
 *	Return length of data.
 */

#ifdef LINUX_2_4
static int get_dev_config_info(char* buf, char** start, off_t offs, int len)
#else
static int get_dev_config_info(char* buf, char** start, off_t offs, int len,int dummy)
#endif
{
	wan_device_t*	wandev = NULL;
	netdevice_t*	dev = NULL;
	int 		cnt = 0;
	PROC_ADD_DECL(stop_cnt);

	PROC_ADD_INIT(offs, stop_cnt);
	for (wandev = router_devlist;
	     wandev && (cnt < (PROC_BUFSZ - 80));
	     wandev = wandev->next) {
		if ((wandev->config_id != (unsigned)start) || !wandev->state) continue;
		for(dev = wandev->dev; dev; dev=*((netdevice_t**)dev->priv)){
			cnt = wandev->get_config_info(dev->priv, buf, cnt, len, offs, &stop_cnt); 
		}
	}

	PROC_ADD_RET(cnt,offs,stop_cnt);
}	

/*
 *	Prepare data for reading FR 'Status' entry.
 *	Return length of data.
 */
#ifdef LINUX_2_4
static int get_dev_status_info(char* buf, char** start, off_t offs, int len)
#else
static int get_dev_status_info(char* buf, char** start, off_t offs, int len, int dummy)	
#endif
{
	wan_device_t*	wandev = NULL;
	netdevice_t*	dev = NULL;
	int		cnt = 0;
	PROC_ADD_DECL(stop_cnt);
	
	PROC_ADD_INIT(offs, stop_cnt);
	for (wandev = router_devlist;
	     wandev && (cnt < (PROC_BUFSZ - 80));
	     wandev = wandev->next) {
		if ((wandev->config_id != (unsigned)start) || !wandev->state) continue;
		for(dev = wandev->dev; dev; dev=*((netdevice_t**)dev->priv)){
			cnt = wandev->get_status_info(dev->priv, buf, cnt, len, offs, &stop_cnt); 
		}
	}
	PROC_ADD_RET(cnt,offs,stop_cnt);
}

#else

/*
 *	No /proc - output stubs
 */

int wanrouter_proc_init(void)
{
	return 0;
}

void wanrouter_proc_cleanup(void)
{
	return;
}

int wanrouter_proc_add(wan_device_t *wandev)
{
	return 0;
}

int wanrouter_proc_delete(wan_device_t *wandev)
{
	return 0;
}

int wanrouter_proc_add_protocol(wan_device_t *wandev)
{
	return 0;
}

int wanrouter_proc_delete_protocol(wan_device_t *wandev)
{
	return 0;
}

int wanrouter_proc_add_interface(wan_device_t* wandev, 
				 struct proc_dir_entry** dent,
				 char* if_name,
				 void* priv)
{
	return 0;
}

int wanrouter_proc_delete_interface(wan_device_t* wandev, char* if_name)
{
	return 0;
}

#endif

/*============================================================================
 * Write WAN device ???.
 * o Find WAN device associated with this node
 */
#ifdef LINUX_2_0
static int device_write(
        struct inode* inode, struct file* file, const char* buf, int count)
{
        int err = verify_area(VERIFY_READ, buf, count);
        struct proc_dir_entry* dent;
        wan_device_t* wandev;

        if (err) return err;

        dent = inode->u.generic_ip;
        if ((dent == NULL) || (dent->data == NULL))
                return -ENODATA;

        wandev = dent->data;

        printk(KERN_ERR "%s: writing %d bytes to %s...\n",
                name_root, count, dent->name);
        
	return 0;
}
#endif

/*
 *	End
 */
 
