/*****************************************************************************
* sdla_ppp.c	WANPIPE(tm) Multiprotocol WAN Link Driver. PPP module.
*
* Author: 	Nenad Corbic <ncorbic@sangoma.com>
*
* Copyright:	(c) 1995-2001 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ============================================================================
* Jan 12, 2002  Nenad Corbic	o Removed the 2.0.X support
* 				  Added front end states
* Dec 03, 2001  Gideon Hack	o Updated for S514-5 56K adapter. 
* Sep 20, 2001  Nenad Corbic    o The min() function has changed for 2.4.9 
* 				  kernel. Thus using the wp_min() defined in
* 				  wanpipe.h
* Sept 6, 2001  Alex Feldman    o Add SNMP support.
* Aug 23, 2001  Nenad Corbic    o Removed the if_header and set the hard_header
* 			          length to zero. Caused problems with Checkpoint
* 			          firewall.
* Jul 22, 2001  Nenad Corbic    o Buf Fix make sure that in_dev ptr exists
*                                 before trying to reference it.
* Jun 07, 2001  Nenad Corbic  	o Bug Fix in retrigger_comm(): 
* 				  On a noisy line, if the 
* 				  protocol toggles UP and DOWN fast 
* 				  enough it can lead to a race condition 
* 				  that will cause it to stop working.
* May 25, 2001  Alex Feldman 	o Added T1/E1 support 
* Feb 28, 2001  Nenad Corbic	o Updated if_tx_timeout() routine for 
* 				  2.4.X kernels.
* Nov 29, 2000  Nenad Corbic	o Added the 2.4.x kernel support:
* 				  get_ip_address() function has moved
* 				  into the ppp_poll() routine. It cannot
* 				  be called from an interrupt.
* Nov 07, 2000  Nenad Corbic	o Added security features for UDP debugging:
*                                 Deny all and specify allowed requests.
* May 02, 2000  Nenad Corbic	o Added the dynamic interface shutdown
*                                 option. When the link goes down, the
*                                 network interface IFF_UP flag is reset.
* Mar 06, 2000  Nenad Corbic	o Bug Fix: corrupted mbox recovery.
* Feb 25, 2000  Nenad Corbic    o Fixed the FT1 UDP debugger problem.
* Feb 09, 2000  Nenad Coribc    o Shutdown bug fix. update() was called
*                                 with NULL dev pointer: no check.
* Jan 24, 2000  Nenad Corbic    o Disabled use of CMD complete inter.
* Dev 15, 1999  Nenad Corbic    o Fixed up header files for 2.0.X kernels
* Oct 25, 1999  Nenad Corbic    o Support for 2.0.X kernels
*                                 Moved dynamic route processing into 
*                                 a polling routine.
* Oct 07, 1999  Nenad Corbic    o Support for S514 PCI card.  
*               Gideon Hack     o UPD and Updates executed using timer interrupt
* Sep 10, 1999  Nenad Corbic    o Fixed up the /proc statistics
* Jul 20, 1999  Nenad Corbic    o Remove the polling routines and use 
*                                 interrupts instead.
* Sep 17, 1998	Jaspreet Singh	o Updates for 2.2.X Kernels.
* Aug 13, 1998	Jaspreet Singh	o Improved Line Tracing.
* Jun 22, 1998	David Fong	o Added remote IP address assignment
* Mar 15, 1998	Alan Cox	o 2.1.8x basic port.
* Apr 16, 1998	Jaspreet Singh	o using htons() for the IPX protocol.
* Dec 09, 1997	Jaspreet Singh	o Added PAP and CHAP.
*				o Implemented new routines like 
*				  ppp_set_inbnd_auth(), ppp_set_outbnd_auth(),
*				  tokenize() and strstrip().
* Nov 27, 1997	Jaspreet Singh	o Added protection against enabling of irqs 
*				  while they have been disabled.
* Nov 24, 1997  Jaspreet Singh  o Fixed another RACE condition caused by
*                                 disabling and enabling of irqs.
*                               o Added new counters for stats on disable/enable
*                                 IRQs.
* Nov 10, 1997	Jaspreet Singh	o Initialized 'skb->mac.raw' to 'skb->data'
*				  before every netif_rx().
*				o Free up the device structure in del_if().
* Nov 07, 1997	Jaspreet Singh	o Changed the delay to zero for Line tracing
*				  command.
* Oct 20, 1997 	Jaspreet Singh	o Added hooks in for Router UP time.
* Oct 16, 1997	Jaspreet Singh  o The critical flag is used to maintain flow
*				  control by avoiding RACE conditions.  The 
*				  cli() and restore_flags() are taken out.
*				  A new structure, "ppp_private_area", is added 
*				  to provide Driver Statistics.   
* Jul 21, 1997 	Jaspreet Singh	o Protected calls to sdla_peek() by adding 
*				  save_flags(), cli() and restore_flags().
* Jul 07, 1997	Jaspreet Singh  o Added configurable TTL for UDP packets
*				o Added ability to discard mulitcast and
*				  broacast source addressed packets.
* Jun 27, 1997 	Jaspreet Singh	o Added FT1 monitor capabilities
*				  New case (0x25) statement in if_send routine.
*				  Added a global variable rCount to keep track
*				  of FT1 status enabled on the board.
* May 22, 1997	Jaspreet Singh	o Added change in the PPP_SET_CONFIG command for
*				508 card to reflect changes in the new 
*				ppp508.sfm for supporting:continous transmission
*				of Configure-Request packets without receiving a
*				reply 				
*				OR-ed 0x300 to conf_flags 
*			        o Changed connect_tmout from 900 to 0
* May 21, 1997	Jaspreet Singh  o Fixed UDP Management for multiple boards
* Apr 25, 1997  Farhan Thawar    o added UDP Management stuff
* Mar 11, 1997  Farhan Thawar   Version 3.1.1
*                                o fixed (+1) bug in rx_intr()
*                                o changed if_send() to return 0 if
*                                  wandev.critical() is true
*                                o free socket buffer in if_send() if
*                                  returning 0 
* Jan 15, 1997	Gene Kozin	Version 3.1.0
*				 o implemented exec() entry point
* Jan 06, 1997	Gene Kozin	Initial version.
*****************************************************************************/

#include <linux/config.h>
#include <linux/version.h>
#include <linux/kernel.h>	/* printk(), and other useful stuff */
#include <linux/stddef.h>	/* offsetof(), etc. */
#include <linux/errno.h>	/* return codes */
#include <linux/string.h>	/* inline memset(), etc. */
#include <linux/slab.h>		/* kmalloc(), kfree() */
#include <linux/wanrouter.h>	/* WAN router definitions */
#include <linux/wanpipe.h>	/* WANPIPE common user API definitions */
#include <linux/if_arp.h>	/* ARPHRD_* defines */
#include <asm/byteorder.h>	/* htons(), etc. */
#include <linux/in.h>		/* sockaddr_in */
#include <linux/inet.h>		/* in_aton(), in_ntoa() prototypes */

 #include <asm/uaccess.h>
 #include <linux/inetdevice.h>
 #include <linux/netdevice.h>

#include <linux/if.h>
#include <linux/wanproc.h>
#include <linux/sdla_ppp.h>		/* PPP firmware API definitions */
#include <linux/sdlasfm.h>		/* S514 Type Definition */

#include <linux/sdla_front_end.h>
/****** Defines & Macros ****************************************************/

#define	PPP_DFLT_MTU	1500		/* default MTU */
#define	PPP_MAX_MTU	4000		/* maximum MTU */
#define PPP_HDR_LEN	1

#define MAX_IP_ERRORS 100 

#define	CONNECT_TIMEOUT	(90*HZ)		/* link connection timeout */
#define	HOLD_DOWN_TIME	(5*HZ)		/* link hold down time : Changed from 30 to 5 */

/* For handle_IPXWAN() */
#define CVHexToAscii(b) (((unsigned char)(b) > (unsigned char)9) ? ((unsigned char)'A' + ((unsigned char)(b) - (unsigned char)10)) : ((unsigned char)'0' + (unsigned char)(b)))

/* Macro for enabling/disabling debugging comments */
#undef NEX_DEBUG
#ifdef NEX_DEBUG
#define NEX_PRINTK(format, a...) printk(format, ## a)
#else
#define NEX_PRINTK(format, a...)
#endif /* NEX_DEBUG */ 

#define DCD(a)   ( a & 0x08 ? "HIGH" : "LOW" )
#define CTS(a)   ( a & 0x20 ? "HIGH" : "LOW" )
#define LCP(a)   ( a == 0x09 ? "OPEN" : "CLOSED" )
#define IP(a)    ( a == 0x09 ? "ENABLED" : "DISABLED" )

#define TMR_INT_ENABLED_UPDATE  	0x01
#define TMR_INT_ENABLED_PPP_EVENT	0x02
#define TMR_INT_ENABLED_UDP		0x04
#define TMR_INT_ENABLED_CONFIG		0x20
#define TMR_INT_ENABLED_TE		0x40

/* Set Configuraton Command Definitions */
#define PERCENT_TX_BUFF			60
#define TIME_BETWEEN_CONF_REQ  		120
#define TIME_BETWEEN_PAP_CHAP_REQ	120
#define WAIT_PAP_CHAP_WITHOUT_REPLY     1200
#define WAIT_AFTER_DCD_CTS_LOW          20
#define TIME_DCD_CTS_LOW_AFTER_LNK_DOWN 40
#define WAIT_DCD_HIGH_AFTER_ENABLE_COMM 3400
#define MAX_CONF_REQ_WITHOUT_REPLY      40
#define MAX_TERM_REQ_WITHOUT_REPLY      10
#define NUM_CONF_NAK_WITHOUT_REPLY      20
#define NUM_AUTH_REQ_WITHOUT_REPLY      40


#define END_OFFSET 0x1F0
#if LINUX_VERSION_CODE < 0x020125
#define test_and_set_bit set_bit
#define net_ratelimit() 1
#endif

/* Number of times we'll retry to 
 * enable comunications.  If it fails
 * the link should be restarted */
#define MAX_COMM_BUSY_RETRY	10

/******Data Structures*****************************************************/

/* This structure is placed in the private data area of the device structure.
 * The card structure used to occupy the private area but now the following 
 * structure will incorporate the card structure along with PPP specific data
 */
  
typedef struct ppp_private_area
{
	netdevice_t *slave;
	sdla_t* card;	
	unsigned long router_start_time;	/*router start time in sec */
	unsigned long tick_counter;		/*used for 5 second counter*/
	unsigned mc;				/*multicast support on or off*/
	unsigned char enable_IPX;
	unsigned long network_number;
	unsigned char pap;
	unsigned char chap;
	unsigned char sysname[31];		/* system name for in-bnd auth*/
	unsigned char userid[511];		/* list of user ids */
	unsigned char passwd[511];		/* list of passwords */
	unsigned protocol;			/* SKB Protocol */
	u32 ip_local;				/* Local IP Address */
	u32 ip_remote;				/* remote IP Address */

	u32 ip_local_tmp;
	u32 ip_remote_tmp;
	
	unsigned char timer_int_enabled;	/* Who enabled the timer inter*/
	unsigned char update_comms_stats;	/* Used by update function */
	unsigned long curr_trace_addr;		/* Trace information */
	unsigned long start_trace_addr;
	unsigned long end_trace_addr;

	unsigned char interface_down;		/* Brind down interface when channel 
                                                   goes down */
	unsigned long config_wait_timeout;	/* After if_open() if in dynamic if mode,
						   wait a few seconds before configuring */
	
	unsigned short udp_pkt_lgth;
	char  udp_pkt_src;
      	char  udp_pkt_data[MAX_LGTH_UDP_MGNT_PKT];

	/* PPP specific statistics */

	if_send_stat_t if_send_stat;
	rx_intr_stat_t rx_intr_stat;
	pipe_mgmt_stat_t pipe_mgmt_stat;

	unsigned long router_up_time; 

	/* Polling task queue. Each interface
         * has its own task queue, which is used
         * to defer events from the interrupt */
	struct tq_struct poll_task;
	struct timer_list poll_delay_timer;

	u8 gateway;
	u8 config_ppp;
	u8 ip_error;
	
	/* Entry in proc fs per each interface */
	struct proc_dir_entry* 	dent;

	unsigned char comm_busy_retry;
	unsigned char ppp_state;

}ppp_private_area_t;

/* variable for keeping track of enabling/disabling FT1 monitor status */
static int rCount = 0;

extern void disable_irq(unsigned int);
extern void enable_irq(unsigned int);

/****** Function Prototypes *************************************************/

/* WAN link driver entry points. These are called by the WAN router module. */
static int update(wan_device_t *wandev);
static int new_if(wan_device_t *wandev, netdevice_t *dev, wanif_conf_t *conf);
static int del_if(wan_device_t *wandev, netdevice_t *dev);

/* WANPIPE-specific entry points */
static int wpp_exec (struct sdla *card, void *u_cmd, void *u_data);

/* Network device interface */
static int if_init(netdevice_t *dev);
static int if_open(netdevice_t *dev);
static int if_close(netdevice_t *dev);

#ifdef LINUX_2_4
static void if_tx_timeout (netdevice_t *dev);
#endif

static struct net_device_stats *if_stats(netdevice_t *dev);
static int if_send(struct sk_buff *skb, netdevice_t *dev);
static void ppp_handle_front_end_state(sdla_t *card);


/* PPP firmware interface functions */
static int ppp_read_version(sdla_t *card, char *str);
static int ppp_set_outbnd_auth(sdla_t *card, ppp_private_area_t *ppp_priv_area);
static int ppp_set_inbnd_auth(sdla_t *card, ppp_private_area_t *ppp_priv_area);
static int ppp_configure(sdla_t *card, void *data);
static int ppp_set_intr_mode(sdla_t *card, unsigned char mode);
static int ppp_comm_enable(sdla_t *card);
static int ppp_comm_disable(sdla_t *card);
static int ppp_comm_disable_shutdown(sdla_t *card);
static int ppp_get_err_stats(sdla_t *card);
static int ppp_send(sdla_t *card, void *data, unsigned len, unsigned proto);
static int ppp_error(sdla_t *card, int err, ppp_mbox_t *mb);

static int set_adapter_config (sdla_t* card);


static void wpp_isr(sdla_t *card);
static void rx_intr(sdla_t *card);
static void event_intr(sdla_t *card);
static void timer_intr(sdla_t *card);

/* Background polling routines */
static void process_route(sdla_t *card);
static int retrigger_comm(sdla_t *card);

/* Miscellaneous functions */
static int read_info( sdla_t *card );
static int read_connection_info (sdla_t *card);
static void remove_route( sdla_t *card );
static int config508(netdevice_t *dev, sdla_t *card);
static void show_disc_cause(sdla_t * card, unsigned cause);
static int reply_udp( unsigned char *data, unsigned int mbox_len );
static void process_udp_mgmt_pkt(sdla_t *card, netdevice_t *dev, 
				ppp_private_area_t *ppp_priv_area);
static void init_ppp_tx_rx_buff( sdla_t *card );
static int intr_test( sdla_t *card );
static int udp_pkt_type( struct sk_buff *skb , sdla_t *card);
static void init_ppp_priv_struct( ppp_private_area_t *ppp_priv_area);
static void init_global_statistics( sdla_t *card );
static int tokenize(char *str, char **tokens);
static char* strstrip(char *str, char *s);
static int chk_bcast_mcast_addr(sdla_t* card, netdevice_t* dev,
				struct sk_buff *skb);

static int config_ppp (sdla_t *);
static void ppp_poll(netdevice_t *);
static void trigger_ppp_poll(netdevice_t *);
static void ppp_poll_delay (unsigned long dev_ptr);


static int Read_connection_info;
static int Intr_test_counter;
static unsigned short available_buffer_space;


/* IPX functions */
static void switch_net_numbers(unsigned char *sendpacket, unsigned long network_number, 
			       unsigned char incoming);
static int handle_IPXWAN(unsigned char *sendpacket, char *devname, unsigned char enable_PX, 
			 unsigned long network_number, unsigned short proto);

/* Lock Functions */
static void s508_lock (sdla_t *card, unsigned long *smp_flags);
static void s508_unlock (sdla_t *card, unsigned long *smp_flags);

static int store_udp_mgmt_pkt(char udp_pkt_src, sdla_t* card,
                                struct sk_buff *skb, netdevice_t* dev,
                                ppp_private_area_t* ppp_priv_area );
static unsigned short calc_checksum (char *data, int len);
static void disable_comm (sdla_t *card);
static int detect_and_fix_tx_bug (sdla_t *card);

static int ppp_get_config_info(void* priv, char* buf, int cnt, int, int, int*);
static int ppp_get_status_info(void* priv, char* buf, int cnt, int, int, int*);
#ifdef LINUX_2_4
static int ppp_get_dev_config_info(char* buf, char** start, off_t offs, int len);
static int ppp_get_if_info(char* buf, char** start, off_t offs, int len);
#else
static int ppp_get_dev_config_info(char* buf, char** start, off_t offs, int len, int dummy);
static int ppp_get_if_info(char* buf, char** start, off_t offs, int len, int dummy);
#endif
static int ppp_set_dev_config(struct file*, const char*, unsigned long, void *);
static int ppp_set_if_info(struct file*, const char*, unsigned long, void *);

static WRITE_FRONT_END_REG_T 	write_front_end_reg;
static READ_FRONT_END_REG_T	read_front_end_reg;
static void ppp_enable_timer(void* card_id);
/****** Public Functions ****************************************************/

/*============================================================================
 * PPP protocol initialization routine.
 *
 * This routine is called by the main WANPIPE module during setup.  At this
 * point adapter is completely initialized and firmware is running.
 *  o read firmware version (to make sure it's alive)
 *  o configure adapter
 *  o initialize protocol-specific fields of the adapter data space.
 *
 * Return:	0	o.k.
 *		< 0	failure.
 */
int wpp_init(sdla_t *card, wandev_conf_t *conf)
{
	ppp_flags_t *flags;
	union
	{
		char str[80];
	} u;

	/* Verify configuration ID */
	if (conf->config_id != WANCONFIG_PPP) {
		
		printk(KERN_INFO "%s: invalid configuration ID %u!\n",
			card->devname, conf->config_id);
		return -EINVAL;

	}

	/* Initialize miscellaneous pointers to structures on the adapter */
	switch (card->hw.type) {

		case SDLA_S508:
			card->mbox =(void*)(card->hw.dpmbase + PPP508_MB_OFFS);
			card->flags=(void*)(card->hw.dpmbase + PPP508_FLG_OFFS);
			break;
		
		case SDLA_S514:
			card->mbox =(void*)(card->hw.dpmbase + PPP514_MB_OFFS);
			card->flags=(void*)(card->hw.dpmbase + PPP514_FLG_OFFS);
			break;

		default:
			return -EINVAL;

	}
	flags = card->flags;

	card->wandev.ignore_front_end_status = conf->ignore_front_end_status;

	/* TE1 Make special hardware initialization for T1/E1 board */
	if (IS_TE1(conf->te_cfg)){
		
		if (card->hw.pci_adapter_type != S5144_ADPTR_1_CPU_T1E1){
			printk(KERN_INFO
				"%s: Error: Card type mismatch: User=T1/E1 Actual=%s\n",
				card->devname,
				DECODE_PCI_TYPE(card->hw.pci_adapter_type));
			return -EIO;
		}
		
		memcpy(&card->wandev.te_cfg, &conf->te_cfg, sizeof(sdla_te_cfg_t));
		card->wandev.write_front_end_reg = write_front_end_reg;
		card->wandev.read_front_end_reg = read_front_end_reg;
		card->wandev.te_enable_timer = ppp_enable_timer;
		conf->interface = 
			(card->wandev.te_cfg.media == WANOPT_MEDIA_T1) ? WANOPT_V35 : WANOPT_RS232;
		conf->clocking = WANOPT_EXTERNAL;

	}else if (IS_56K(conf->te_cfg)){

		if (card->hw.pci_adapter_type != S5145_ADPTR_1_CPU_56K){
			printk(KERN_INFO
				"%s: Error: Card type mismatch: User=56K Actual=%s\n",
				card->devname,
				DECODE_PCI_TYPE(card->hw.pci_adapter_type));
			return -EIO;
		}
		
		memcpy(&card->wandev.te_cfg, &conf->te_cfg, sizeof(sdla_te_cfg_t));
		card->wandev.write_front_end_reg = write_front_end_reg;
		card->wandev.read_front_end_reg = read_front_end_reg;
		
		conf->clocking = WANOPT_EXTERNAL;
	}else{

		if (card->hw.pci_adapter_type == S5145_ADPTR_1_CPU_56K ||
		    card->hw.pci_adapter_type == S5144_ADPTR_1_CPU_T1E1){
			printk(KERN_INFO
				"%s: Error: Card type mismatch: User=S514(1/2/3) Actual=%s\n",
				card->devname,
				DECODE_PCI_TYPE(card->hw.pci_adapter_type));
			return -EIO;
		}

		card->wandev.front_end_status = FE_CONNECTED;	
	}

	if (card->wandev.ignore_front_end_status == WANOPT_NO){
		printk(KERN_INFO 
		  "%s: Enabling front end link monitor\n",
				card->devname);
	}else{
		printk(KERN_INFO 
		"%s: Disabling front end link monitor\n",
				card->devname);
	}

	/* Read firmware version.  Note that when adapter initializes, it
	 * clears the mailbox, so it may appear that the first command was
	 * executed successfully when in fact it was merely erased. To work
	 * around this, we execute the first command twice.
	 */
	if (ppp_read_version(card, NULL) || ppp_read_version(card, u.str))
		return -EIO;
	
	printk(KERN_INFO "%s: running PPP firmware v%s\n",card->devname, u.str); 


	if (set_adapter_config(card)) {
		return -EIO;
	}

	
	/* Adjust configuration and set defaults */
	card->wandev.mtu = (conf->mtu) ?
		wp_min(conf->mtu, PPP_MAX_MTU) : PPP_DFLT_MTU;

	card->wandev.bps	= conf->bps;
	card->wandev.interface	= conf->interface;
	card->wandev.clocking	= conf->clocking;
	card->wandev.station	= conf->station;
	card->isr		= &wpp_isr;
	card->poll		= NULL; 
	card->exec		= &wpp_exec;
	card->wandev.update	= &update;
	card->wandev.new_if	= &new_if;
	card->wandev.del_if	= &del_if;
        card->wandev.udp_port   = conf->udp_port;
	card->wandev.ttl	= conf->ttl;
	card->wandev.state      = WAN_DISCONNECTED;
	card->disable_comm	= &disable_comm;
	card->irq_dis_if_send_count = 0;
        card->irq_dis_poll_count = 0;
	card->u.p.authenticator = conf->u.ppp.authenticator;
	card->u.p.ip_mode 	= conf->u.ppp.ip_mode ?
				 conf->u.ppp.ip_mode : WANOPT_PPP_STATIC;
        card->TracingEnabled    = 0;
	Read_connection_info    = 1;

	// Proc fs functions
	card->wandev.get_config_info 	= &ppp_get_config_info;
	card->wandev.get_status_info 	= &ppp_get_status_info;
	card->wandev.get_dev_config_info= &ppp_get_dev_config_info;
	card->wandev.get_if_info     	= &ppp_get_if_info;
	card->wandev.set_dev_config    	= &ppp_set_dev_config;
	card->wandev.set_if_info     	= &ppp_set_if_info;

	/* initialize global statistics */
	init_global_statistics( card );



	if (!card->configured){
		int err;

		Intr_test_counter = 0;
		err = intr_test(card);

		if(err || (Intr_test_counter < MAX_INTR_TEST_COUNTER)) {
			printk("%s: Interrupt Test Failed, Counter: %i\n", 
				card->devname, Intr_test_counter);
			printk( "%s: Please choose another interrupt\n",card->devname);
			return -EIO;
		}
		
		printk(KERN_INFO "%s: Interrupt Test Passed, Counter: %i\n", 
			card->devname, Intr_test_counter);
		card->configured = 1;
	}

	ppp_set_intr_mode(card, PPP_INTR_TIMER); 

	/* Turn off the transmit and timer interrupt */
	flags->imask &= ~PPP_INTR_TIMER;

	printk(KERN_INFO "\n");

	return 0;
}

/******* WAN Device Driver Entry Points *************************************/

/*============================================================================
 * Update device status & statistics.
 */
static int update(wan_device_t *wandev)
{
	sdla_t* card = wandev->private;
 	netdevice_t* dev;
        volatile ppp_private_area_t *ppp_priv_area;
	ppp_flags_t *flags = card->flags;
	unsigned long timeout;

	/* sanity checks */
	if ((wandev == NULL) || (wandev->private == NULL))
		return -EFAULT;
	
	if (wandev->state == WAN_UNCONFIGURED)
		return -ENODEV;
	
	/* Shutdown bug fix. This function can be
         * called with NULL dev pointer during
         * shutdown 
	 */
	if ((dev=card->wandev.dev) == NULL){
		return -ENODEV;
	}

	if ((ppp_priv_area=dev->priv) == NULL){
		return -ENODEV;
	}
	
	/* TE1 Read T1/E1 alarms and PMON counters in one timer interrupt */
	ppp_priv_area->update_comms_stats = 1;
	ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_UPDATE;
	flags->imask |= PPP_INTR_TIMER;	
	
	/* wait a maximum of 1 second for the statistics to be updated */ 
        timeout = jiffies;
        for(;;) {
		if(ppp_priv_area->update_comms_stats == 0){
			break;
		}
                if ((jiffies - timeout) > (1 * HZ)){
    			ppp_priv_area->update_comms_stats = 0;
 			ppp_priv_area->timer_int_enabled &=
				~TMR_INT_ENABLED_UPDATE; 
 			return -EAGAIN;
		}
        }

	return 0;
}

/*============================================================================
 * Create new logical channel.
 * This routine is called by the router when ROUTER_IFNEW IOCTL is being
 * handled.
 * o parse media- and hardware-specific configuration
 * o make sure that a new channel can be created
 * o allocate resources, if necessary
 * o prepare network device structure for registaration.
 *
 * Return:	0	o.k.
 *		< 0	failure (channel will not be created)
 */
static int new_if(wan_device_t *wandev, netdevice_t *dev, wanif_conf_t *conf)
{
	sdla_t *card = wandev->private;
	ppp_private_area_t *ppp_priv_area;
	int err = 0;

	if (wandev->ndev)
		return -EEXIST;
	

	printk(KERN_INFO "%s: Configuring Interface: %s\n",
			card->devname, conf->name);

	if ((conf->name[0] == '\0') || (strlen(conf->name) > WAN_IFNAME_SZ)) {

		printk(KERN_INFO "%s: Invalid interface name!\n",
			card->devname);
		return -EINVAL;

	}

	/* allocate and initialize private data */
	ppp_priv_area = kmalloc(sizeof(ppp_private_area_t), GFP_KERNEL);
	
	if( ppp_priv_area == NULL )
		return	-ENOMEM;
	
	memset(ppp_priv_area, 0, sizeof(ppp_private_area_t));
	
	ppp_priv_area->card = card; 
	
	/* initialize data */
	strcpy(card->u.p.if_name, conf->name);

	/* initialize data in ppp_private_area structure */
	
	init_ppp_priv_struct( ppp_priv_area );

	ppp_priv_area->mc = conf->mc;
	ppp_priv_area->pap = conf->pap;
	ppp_priv_area->chap = conf->chap;

	/* Option to bring down the interface when 
         * the link goes down */
	if (conf->if_down){
		set_bit(DYN_OPT_ON,&ppp_priv_area->interface_down);
		printk("%s: Dynamic interface configuration enabled\n",
			card->devname);
	} 

	/* If no user ids are specified */
	if(!strlen(conf->userid) && (ppp_priv_area->pap||ppp_priv_area->chap)){
		kfree(ppp_priv_area);
		return -EINVAL;
	}

	/* If no passwords are specified */
	if(!strlen(conf->passwd) && (ppp_priv_area->pap||ppp_priv_area->chap)){
		kfree(ppp_priv_area);
		return -EINVAL;
	}

	if(strlen(conf->sysname) > 31){
		kfree(ppp_priv_area);
		return -EINVAL;
	}

	/* If no system name is specified */
	if(!strlen(conf->sysname) && (card->u.p.authenticator)){
		kfree(ppp_priv_area);
		return -EINVAL;
	}

	/* copy the data into the ppp private structure */
	memcpy(ppp_priv_area->userid, conf->userid, strlen(conf->userid));
	memcpy(ppp_priv_area->passwd, conf->passwd, strlen(conf->passwd));
	memcpy(ppp_priv_area->sysname, conf->sysname, strlen(conf->sysname));

	
	ppp_priv_area->enable_IPX = conf->enable_IPX;
	if (conf->network_number){
		ppp_priv_area->network_number = conf->network_number;
	}else{
		ppp_priv_area->network_number = 0xDEADBEEF;
	}

	/* Tells us that if this interface is a
         * gateway or not */
	if ((ppp_priv_area->gateway = conf->gateway) == WANOPT_YES){
		printk(KERN_INFO "%s: Interface %s is set as a gateway.\n",
			card->devname,card->u.p.if_name);
	}

	/* prepare network device data space for registration */
#ifdef LINUX_2_4
 	strcpy(dev->name,card->u.p.if_name);
#else
	dev->name = (char *)kmalloc(strlen(card->u.p.if_name) + 2, GFP_KERNEL); 
	if(dev->name == NULL)
	{
		kfree(ppp_priv_area);
		return -ENOMEM;
	}
	sprintf(dev->name, "%s", card->u.p.if_name);
#endif
	
	dev->init = &if_init;
	dev->priv = ppp_priv_area;
	dev->mtu = wp_min(dev->mtu, card->wandev.mtu);

	/* Initialize the polling task routine */
#ifndef LINUX_2_4
	ppp_priv_area->poll_task.next = NULL;
#endif
	ppp_priv_area->poll_task.sync=0;
	ppp_priv_area->poll_task.routine = (void*)(void*)ppp_poll;
	ppp_priv_area->poll_task.data = dev;

	/* Initialize the polling delay timer */
	init_timer(&ppp_priv_area->poll_delay_timer);
	ppp_priv_area->poll_delay_timer.data = (unsigned long)dev;
	ppp_priv_area->poll_delay_timer.function = ppp_poll_delay;
	
	/*
	 * Create interface file in proc fs.
	 */
	err = wanrouter_proc_add_interface(wandev, 
					   &ppp_priv_area->dent, 
					   card->u.p.if_name, 
					   dev);
	if (err){	
		printk(KERN_INFO
			"%s: can't create /proc/net/router/ppp/%s entry!\n",
			card->devname, card->u.p.if_name);
		return err;
	}
	
	ppp_priv_area->ppp_state=WAN_DISCONNECTED;
	
	/* Since we start with dummy IP addresses we can say
	 * that route exists */
	printk(KERN_INFO "\n");

	return 0;
}

/*============================================================================
 * Delete logical channel.
 */
static int del_if(wan_device_t *wandev, netdevice_t *dev)
{
	ppp_private_area_t* 	ppp_priv_area = dev->priv;
	sdla_t*			card = ppp_priv_area->card;
	
	/* Delete interface name from proc fs. */
	wanrouter_proc_delete_interface(wandev, card->u.p.if_name);
	return 0;
}

static void disable_comm (sdla_t *card)
{
	ppp_comm_disable_shutdown(card);
	return;
}

/****** WANPIPE-specific entry points ***************************************/

/*============================================================================
 * Execute adapter interface command.
 */

//FIXME: Why do we need this ????
static int wpp_exec(struct sdla *card, void *u_cmd, void *u_data)
{
	ppp_mbox_t *mbox = card->mbox;
	int len;

	if (copy_from_user((void*)&mbox->cmd, u_cmd, sizeof(ppp_cmd_t)))
		return -EFAULT;

	len = mbox->cmd.length;

	if (len) {

		if( copy_from_user((void*)&mbox->data, u_data, len))
			return -EFAULT;

	}

	/* execute command */
	if (!sdla_exec(mbox))
		return -EIO;

	/* return result */
	if( copy_to_user(u_cmd, (void*)&mbox->cmd, sizeof(ppp_cmd_t)))
		return -EFAULT;
	len = mbox->cmd.length;

	if (len && u_data && copy_to_user(u_data, (void*)&mbox->data, len))
		return -EFAULT;

	return 0;
}

/****** Network Device Interface ********************************************/

/*============================================================================
 * Initialize Linux network interface.
 *
 * This routine is called only once for each interface, during Linux network
 * interface registration.  Returning anything but zero will fail interface
 * registration.
 */
static int if_init(netdevice_t *dev)
{
	ppp_private_area_t *ppp_priv_area = dev->priv;
	sdla_t *card = ppp_priv_area->card;
	wan_device_t *wandev = &card->wandev;
#ifdef LINUX_2_0
	int i;
#endif

	/* Initialize device driver entry points */
	dev->open		= &if_open;
	dev->stop		= &if_close;
	dev->hard_header	= NULL;
	dev->rebuild_header	= NULL;
	dev->hard_start_xmit	= &if_send;
	dev->get_stats		= &if_stats;
#ifdef LINUX_2_4
	dev->tx_timeout		= &if_tx_timeout;
	dev->watchdog_timeo	= TX_TIMEOUT;
#endif

	/* Initialize media-specific parameters */
	dev->type		= ARPHRD_PPP;	/* ARP h/w type */
	dev->flags		|= IFF_POINTOPOINT;
	dev->flags		|= IFF_NOARP;

	/* Enable Mulitcasting if specified by user*/
	if (ppp_priv_area->mc == WANOPT_YES){
		dev->flags	|= IFF_MULTICAST;
	}

#ifdef LINUX_2_0
	dev->family		= AF_INET;
#endif	
	dev->mtu		= wandev->mtu;
	dev->hard_header_len	= 0; 	/* media header length */

	/* Initialize hardware parameters (just for reference) */
	dev->irq		= wandev->irq;
	dev->dma		= wandev->dma;
	dev->base_addr		= wandev->ioport;
	dev->mem_start		= wandev->maddr;
	dev->mem_end		= wandev->maddr + wandev->msize - 1;

        /* Set transmit buffer queue length */
        dev->tx_queue_len = 100;
   
	return 0;
}

/*============================================================================
 * Open network interface.
 * o enable communications and interrupts.
 * o prevent module from unloading by incrementing use count
 *
 * Return 0 if O.k. or errno.
 */
static int if_open (netdevice_t *dev)
{
	ppp_private_area_t *ppp_priv_area = dev->priv;
	sdla_t *card = ppp_priv_area->card;
	struct timeval tv;

	if (is_dev_running(dev))
		return -EBUSY;

	wanpipe_open(card);

#ifdef LINUX_2_4
	netif_start_queue(dev);
#else
	dev->interrupt = 0;
	dev->tbusy = 0;
	dev->start = 1;
#endif
	
	do_gettimeofday( &tv );
	ppp_priv_area->router_start_time = tv.tv_sec;

	/* We cannot configure the card here because we don't
	 * have access to the interface IP addresses.
         * Once the interface initilization is complete, we will be
         * able to access the IP addresses.  Therefore,
         * configure the ppp link in the poll routine */
	set_bit(0,&ppp_priv_area->config_ppp);
	ppp_priv_area->config_wait_timeout=jiffies;

	/* Start the PPP configuration after 1sec delay.
	 * This will give the interface initilization time
	 * to finish its configuration */
	del_timer(&ppp_priv_area->poll_delay_timer);
	ppp_priv_area->poll_delay_timer.expires = jiffies+HZ;
	add_timer(&ppp_priv_area->poll_delay_timer);
	return 0;
}

/*============================================================================
 * Close network interface.
 * o if this is the last open, then disable communications and interrupts.
 * o reset flags.
 */
static int if_close(netdevice_t *dev)
{
	ppp_private_area_t *ppp_priv_area = dev->priv;
	sdla_t *card = ppp_priv_area->card;

	stop_net_queue(dev);
#ifndef LINUX_2_4
	dev->start=0;
#endif
	wanpipe_close(card);

	del_timer (&ppp_priv_area->poll_delay_timer);
	/* TE1 unconfiging */
	if (IS_TE1(card->wandev.te_cfg)) {
		sdla_te_unconfig(card);
	}
	return 0;
}

#ifdef LINUX_2_4
/*============================================================================
 * Handle transmit timeout event from netif watchdog
 */
static void if_tx_timeout (netdevice_t *dev)
{
    	ppp_private_area_t* chan = dev->priv;
	sdla_t *card = chan->card;
	
	/* If our device stays busy for at least 5 seconds then we will
	 * kick start the device by making dev->tbusy = 0.  We expect
	 * that our device never stays busy more than 5 seconds. So this                 
	 * is only used as a last resort.
	 */

	++ chan->if_send_stat.if_send_tbusy;
	++card->wandev.stats.collisions;

	printk (KERN_INFO "%s: Transmit timed out on %s\n", card->devname,dev->name);
	++chan->if_send_stat.if_send_tbusy_timeout;
	netif_wake_queue (dev);
}
#endif



/*============================================================================
 * Send a packet on a network interface.
 * o set tbusy flag (marks start of the transmission) to block a timer-based
 *   transmit from overlapping.
 * o check link state. If link is not up, then drop the packet.
 * o execute adapter send command.
 * o free socket buffer
 *
 * Return:	0	complete (socket buffer must be freed)
 *		non-0	packet may be re-transmitted (tbusy must be set)
 *
 * Notes:
 * 1. This routine is called either by the protocol stack or by the "net
 *    bottom half" (with interrupts enabled).
 * 2. Setting tbusy flag will inhibit further transmit requests from the
 *    protocol stack and can be used for flow control with protocol layer.
 */
static int if_send (struct sk_buff *skb, netdevice_t *dev)
{
	ppp_private_area_t *ppp_priv_area = dev->priv;
	sdla_t *card = ppp_priv_area->card;
	unsigned char *sendpacket;
	unsigned long smp_flags;
	ppp_flags_t *flags = card->flags;
	int udp_type;
	int err=0;
	
	++ppp_priv_area->if_send_stat.if_send_entry;

#ifdef LINUX_2_4
	netif_stop_queue(dev);
#endif
	
	if (skb == NULL) {

		/* If we get here, some higher layer thinks we've missed an
		 * tx-done interrupt.
		 */
		printk(KERN_INFO "%s: interface %s got kicked!\n",
			card->devname, dev->name);
		
		++ppp_priv_area->if_send_stat.if_send_skb_null;
	
		wake_net_dev(dev);
		return 0;
	}

#ifndef LINUX_2_4
	if (dev->tbusy) {

		/* If our device stays busy for at least 5 seconds then we will
		 * kick start the device by making dev->tbusy = 0.  We expect 
		 * that our device never stays busy more than 5 seconds. So this
		 * is only used as a last resort. 
		 */
              
		++ppp_priv_area->if_send_stat.if_send_tbusy;
        	++card->wandev.stats.collisions;

		if ((jiffies - ppp_priv_area->tick_counter) < (5*HZ)) {
			return 1;
		}

		printk (KERN_INFO "%s: Transmit times out on %s\n",card->devname,dev->name);
	
		++ppp_priv_area->if_send_stat.if_send_tbusy_timeout;
		++card->wandev.stats.collisions;

		/* unbusy the card (because only one interface per card)*/
		dev->tbusy = 0;
	}	
#endif
	
	sendpacket = skb->data;

	udp_type = udp_pkt_type( skb, card );


	if (udp_type == UDP_PTPIPE_TYPE){
		if(store_udp_mgmt_pkt(UDP_PKT_FRM_STACK, card, skb, dev,
                	              ppp_priv_area)){
	               	flags->imask |= PPP_INTR_TIMER;
		}
		++ppp_priv_area->if_send_stat.if_send_PIPE_request;
		start_net_queue(dev);
		return 0;
	}

	/* Check for broadcast and multicast addresses 
	 * If found, drop (deallocate) a packet and return.
	 */
	if(chk_bcast_mcast_addr(card, dev, skb)){
		++card->wandev.stats.tx_dropped;
		wan_dev_kfree_skb(skb,FREE_WRITE);
		start_net_queue(dev);
		return 0;
	}


 	if(card->hw.type != SDLA_S514){
		s508_lock(card,&smp_flags);
	}

    	if (test_and_set_bit(SEND_CRIT, (void*)&card->wandev.critical)) {

		printk(KERN_INFO "%s: Critical in if_send: %lx\n",
				card->wandev.name,card->wandev.critical);
		
		++card->wandev.stats.tx_dropped;
		++ppp_priv_area->if_send_stat.if_send_critical_non_ISR;
		start_net_queue(dev);
		goto if_send_exit_crit;
	}

	if (card->wandev.state != WAN_CONNECTED) {

		++ppp_priv_area->if_send_stat.if_send_wan_disconnected;
        	++card->wandev.stats.tx_dropped;
		start_net_queue(dev);
		
     	} else if (!skb->protocol) {
		++ppp_priv_area->if_send_stat.if_send_protocol_error;
        	++card->wandev.stats.tx_errors;
		start_net_queue(dev);
		
	} else {

		/*If it's IPX change the network numbers to 0 if they're ours.*/
		if( skb->protocol == htons(ETH_P_IPX) ) {
			if(ppp_priv_area->enable_IPX) {
				switch_net_numbers( skb->data, 
					ppp_priv_area->network_number, 0);
			} else {
				++card->wandev.stats.tx_dropped;
				start_net_queue(dev);
				goto if_send_exit_crit;
			}
		}

		if (ppp_send(card, skb->data, skb->len, skb->protocol)) {
			stop_net_queue(dev);
			++ppp_priv_area->if_send_stat.if_send_adptr_bfrs_full;
			++ppp_priv_area->if_send_stat.if_send_tx_int_enabled;
		} else {
			++ppp_priv_area->if_send_stat.if_send_bfr_passed_to_adptr;
			++card->wandev.stats.tx_packets;
			card->wandev.stats.tx_bytes += skb->len;
			start_net_queue(dev);
#ifdef LINUX_2_4
			dev->trans_start = jiffies;
#endif
		}
    	}
	
if_send_exit_crit:
	
	if (!(err=is_queue_stopped(dev))){
      		wan_dev_kfree_skb(skb, FREE_WRITE);
	}else{
		ppp_priv_area->tick_counter = jiffies;
		flags->imask |= PPP_INTR_TXRDY;	/* unmask Tx interrupts */
	}
	
	clear_bit(SEND_CRIT,&card->wandev.critical);
	if(card->hw.type != SDLA_S514){	
		s508_unlock(card,&smp_flags);
	}

	return err;
}


/*=============================================================================
 * Store a UDP management packet for later processing.
 */

static int store_udp_mgmt_pkt(char udp_pkt_src, sdla_t* card,
                                struct sk_buff *skb, netdevice_t* dev,
                                ppp_private_area_t* ppp_priv_area )
{
	int udp_pkt_stored = 0;

	if(!ppp_priv_area->udp_pkt_lgth && (skb->len<=MAX_LGTH_UDP_MGNT_PKT)){
        	ppp_priv_area->udp_pkt_lgth = skb->len;
		ppp_priv_area->udp_pkt_src = udp_pkt_src;
       		memcpy(ppp_priv_area->udp_pkt_data, skb->data, skb->len);
		ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_UDP;
		ppp_priv_area->protocol = skb->protocol;
		udp_pkt_stored = 1;
	}else{
		if (skb->len > MAX_LGTH_UDP_MGNT_PKT){
			printk(KERN_INFO "%s: PIPEMON UDP request too long : %i\n",
				card->devname, skb->len);
		}else{
			printk(KERN_INFO "%s: PIPEMON UPD request already pending\n",
				card->devname);
		}
		ppp_priv_area->udp_pkt_lgth = 0;
	}

	if(udp_pkt_src == UDP_PKT_FRM_STACK){
		wan_dev_kfree_skb(skb, FREE_WRITE);
	}else{
                wan_dev_kfree_skb(skb, FREE_READ);
	}

	return(udp_pkt_stored);
}



/*============================================================================
 * Reply to UDP Management system.
 * Return length of reply.
 */
static int reply_udp( unsigned char *data, unsigned int mbox_len ) 
{
	unsigned short len, udp_length, temp, ip_length;
	unsigned long ip_temp;
	int even_bound = 0;
	ppp_udp_pkt_t *p_udp_pkt = (ppp_udp_pkt_t *)data;
 
	/* Set length of packet */
	len = sizeof(ip_pkt_t)+ 
	      sizeof(udp_pkt_t)+
	      sizeof(wp_mgmt_t)+
	      sizeof(cblock_t)+
	      mbox_len;

	/* fill in UDP reply */
  	p_udp_pkt->wp_mgmt.request_reply = UDPMGMT_REPLY; 

	/* fill in UDP length */
	udp_length = sizeof(udp_pkt_t)+ 
		     sizeof(wp_mgmt_t)+
		     sizeof(cblock_t)+
		     mbox_len; 
  
 
	/* put it on an even boundary */
	if ( udp_length & 0x0001 ) {
		udp_length += 1;
		len += 1;
		even_bound=1;
	} 
	
	temp = (udp_length<<8)|(udp_length>>8);
	p_udp_pkt->udp_pkt.udp_length = temp;		

 
	/* swap UDP ports */
	temp = p_udp_pkt->udp_pkt.udp_src_port;
	p_udp_pkt->udp_pkt.udp_src_port = 
			p_udp_pkt->udp_pkt.udp_dst_port; 
	p_udp_pkt->udp_pkt.udp_dst_port = temp;


	/* add UDP pseudo header */
	temp = 0x1100;
	*((unsigned short *)(p_udp_pkt->data+mbox_len+even_bound)) = temp;
	temp = (udp_length<<8)|(udp_length>>8);
	*((unsigned short *)(p_udp_pkt->data+mbox_len+even_bound+2)) = temp;
 
	/* calculate UDP checksum */
	p_udp_pkt->udp_pkt.udp_checksum = 0;
	p_udp_pkt->udp_pkt.udp_checksum = 
		calc_checksum(&data[UDP_OFFSET],udp_length+UDP_OFFSET);

	/* fill in IP length */
	ip_length = udp_length + sizeof(ip_pkt_t);
	temp = (ip_length<<8)|(ip_length>>8);
  	p_udp_pkt->ip_pkt.total_length = temp;
 
	/* swap IP addresses */
	ip_temp = p_udp_pkt->ip_pkt.ip_src_address;
	p_udp_pkt->ip_pkt.ip_src_address = p_udp_pkt->ip_pkt.ip_dst_address;
	p_udp_pkt->ip_pkt.ip_dst_address = ip_temp;

	/* fill in IP checksum */
	p_udp_pkt->ip_pkt.hdr_checksum = 0;
	p_udp_pkt->ip_pkt.hdr_checksum = calc_checksum(data,sizeof(ip_pkt_t));

	return len;

} /* reply_udp */

unsigned short calc_checksum (char *data, int len)
{
	unsigned short temp; 
	unsigned long sum=0;
	int i;

	for( i = 0; i <len; i+=2 ) {
		memcpy(&temp,&data[i],2);
		sum += (unsigned long)temp;
	}

	while (sum >> 16 ) {
		sum = (sum & 0xffffUL) + (sum >> 16);
	}

	temp = (unsigned short)sum;
	temp = ~temp;

	if( temp == 0 ) 
		temp = 0xffff;

	return temp;	
}

/*
   If incoming is 0 (outgoing)- if the net numbers is ours make it 0
   if incoming is 1 - if the net number is 0 make it ours 

*/
static void switch_net_numbers(unsigned char *sendpacket, unsigned long network_number, unsigned char incoming)
{
	unsigned long pnetwork_number;

	pnetwork_number = (unsigned long)((sendpacket[6] << 24) + 
			  (sendpacket[7] << 16) + (sendpacket[8] << 8) + 
			  sendpacket[9]);

	if (!incoming) {
		//If the destination network number is ours, make it 0
		if( pnetwork_number == network_number) {
			sendpacket[6] = sendpacket[7] = sendpacket[8] = 
					 sendpacket[9] = 0x00;
		}
	} else {
		//If the incoming network is 0, make it ours
		if( pnetwork_number == 0) {
			sendpacket[6] = (unsigned char)(network_number >> 24);
			sendpacket[7] = (unsigned char)((network_number & 
					 0x00FF0000) >> 16);
			sendpacket[8] = (unsigned char)((network_number & 
					 0x0000FF00) >> 8);
			sendpacket[9] = (unsigned char)(network_number & 
					 0x000000FF);
		}
	}


	pnetwork_number = (unsigned long)((sendpacket[18] << 24) + 
			  (sendpacket[19] << 16) + (sendpacket[20] << 8) + 
			  sendpacket[21]);

	if( !incoming ) {
		//If the source network is ours, make it 0
		if( pnetwork_number == network_number) {
			sendpacket[18] = sendpacket[19] = sendpacket[20] = 
					 sendpacket[21] = 0x00;
		}
	} else {
		//If the source network is 0, make it ours
		if( pnetwork_number == 0 ) {
			sendpacket[18] = (unsigned char)(network_number >> 24);
			sendpacket[19] = (unsigned char)((network_number & 
					 0x00FF0000) >> 16);
			sendpacket[20] = (unsigned char)((network_number & 
					 0x0000FF00) >> 8);
			sendpacket[21] = (unsigned char)(network_number & 
					 0x000000FF);
		}
	}
} /* switch_net_numbers */

/*============================================================================
 * Get ethernet-style interface statistics.
 * Return a pointer to struct net_device_stats.
 */
static struct net_device_stats *if_stats(netdevice_t *dev)
{

	ppp_private_area_t *ppp_priv_area = dev->priv;
	sdla_t* card;
	
	if( ppp_priv_area == NULL )
		return NULL;

	card = ppp_priv_area->card;
	return &card->wandev.stats;
}


/****** PPP Firmware Interface Functions ************************************/

/*============================================================================
 * Read TE1/56K Front end registers
 */
static unsigned char read_front_end_reg (void* card1, unsigned short reg)
{
	sdla_t* card = (sdla_t*)card1;
        ppp_mbox_t* mb = card->mbox;
	char* data = mb->data;
        int err;

	((FRONT_END_REG_STRUCT *)data)->register_number = (unsigned short)reg;
	mb->cmd.length = sizeof(FRONT_END_REG_STRUCT);
        mb->cmd.command = READ_FRONT_END_REGISTER;
        err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
        if (err != CMD_OK)
                ppp_error(card,err,mb);

	return(((FRONT_END_REG_STRUCT *)data)->register_value);
}

/*============================================================================
 * Write to TE1/56K Front end registers 
 */
static unsigned char write_front_end_reg (void* card1, unsigned short reg, unsigned char value)
{
	sdla_t* card = (sdla_t*)card1;
        ppp_mbox_t* mb = card->mbox;
	char* data = mb->data;
        int err;
	
	((FRONT_END_REG_STRUCT *)data)->register_number = (unsigned short)reg;
	((FRONT_END_REG_STRUCT *)data)->register_value = value;
	mb->cmd.length = sizeof(FRONT_END_REG_STRUCT);
        mb->cmd.command = WRITE_FRONT_END_REGISTER;
        err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
        if (err != CMD_OK)
                ppp_error(card,err,mb);
        return err;
}


/*============================================================================
 * Enable timer interrupt  
 */
static void ppp_enable_timer (void* card_id)
{
	sdla_t* 		card = (sdla_t*)card_id;
	netdevice_t* 		dev = NULL;
	ppp_flags_t*		flags = card->flags;
	ppp_private_area_t*	ppp_priv_area;
	
	dev = card->wandev.dev;
	if ((ppp_priv_area=dev->priv) == NULL)
		return;
	ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_TE;
	flags->imask |= PPP_INTR_TIMER;
	return;
}

/*============================================================================
 * Read firmware code version.
 *	Put code version as ASCII string in str. 
 */
static int ppp_read_version(sdla_t *card, char *str)
{
	ppp_mbox_t *mb = card->mbox;
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	mb->cmd.command = PPP_READ_CODE_VERSION;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	if (err != CMD_OK)
 
		ppp_error(card, err, mb);

	else if (str) {

		int len = mb->cmd.length;

		memcpy(str, mb->data, len);
		str[len] = '\0';

	}

	return err;
}
/*===========================================================================
 * Set Out-Bound Authentication.
*/
static int ppp_set_outbnd_auth (sdla_t *card, ppp_private_area_t *ppp_priv_area)
{
	ppp_mbox_t *mb = card->mbox;
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	memset(&mb->data, 0, (strlen(ppp_priv_area->sysname) + strlen(ppp_priv_area->userid) + 
					strlen(ppp_priv_area->passwd) + 3));
	memcpy(mb->data, ppp_priv_area->sysname, strlen(ppp_priv_area->sysname));
	memcpy((mb->data + strlen(ppp_priv_area->sysname) + 1), 
			ppp_priv_area->userid, strlen(ppp_priv_area->userid));
	memcpy((mb->data + strlen(ppp_priv_area->sysname) + strlen(ppp_priv_area->userid) + 2), 
		ppp_priv_area->passwd, strlen(ppp_priv_area->passwd));	
	
	mb->cmd.length  = strlen(ppp_priv_area->sysname) + strlen(ppp_priv_area->userid) + 
					strlen(ppp_priv_area->passwd) + 3 ;
	
	mb->cmd.command = PPP_SET_OUTBOUND_AUTH;

	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	if (err != CMD_OK)
		ppp_error(card, err, mb);

	return err;
}

/*===========================================================================
 * Set In-Bound Authentication.
*/
static int ppp_set_inbnd_auth (sdla_t *card, ppp_private_area_t *ppp_priv_area)
{
	ppp_mbox_t *mb = card->mbox;
	int err, i;
	char* user_tokens[32];
	char* pass_tokens[32];
	int userids, passwds;
	int add_ptr;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	memset(&mb->data, 0, 1008);
	memcpy(mb->data, ppp_priv_area->sysname, 
						strlen(ppp_priv_area->sysname));
	
	/* Parse the userid string and the password string and build a string
	   to copy it to the data area of the command structure.   The string
	   will look like "SYS_NAME<NULL>USER1<NULL>PASS1<NULL>USER2<NULL>PASS2
	   ....<NULL> " 
	 */
	userids = tokenize( ppp_priv_area->userid, user_tokens);
	passwds = tokenize( ppp_priv_area->passwd, pass_tokens);
	
	if (userids != passwds){
		printk(KERN_INFO "%s: Number of passwords does not equal the number of user ids\n", card->devname);
		return 1;	
	}

	add_ptr = strlen(ppp_priv_area->sysname) + 1;
	for (i=0; i<userids; i++){
		memcpy((mb->data + add_ptr), user_tokens[i], 
							strlen(user_tokens[i]));
		memcpy((mb->data + add_ptr + strlen(user_tokens[i]) + 1), 
					pass_tokens[i], strlen(pass_tokens[i]));
		add_ptr = add_ptr + strlen(user_tokens[i]) + 1 + 
						strlen(pass_tokens[i]) + 1;
	}

	mb->cmd.length  = add_ptr + 1;
	mb->cmd.command = PPP_SET_INBOUND_AUTH;

	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	if (err != CMD_OK)
		ppp_error(card, err, mb);

	return err;
}


/*============================================================================
 * Tokenize string.
 *      Parse a string of the following syntax:
 *              <arg1>,<arg2>,...
 *      and fill array of tokens with pointers to string elements.
 *
 */
static int tokenize (char *str, char **tokens)
{
        int cnt = 0;

        tokens[0] = strtok(str, "/");
        while (tokens[cnt] && (cnt < 32 - 1))
        {
                tokens[cnt] = strstrip(tokens[cnt], " \t");
                tokens[++cnt] = strtok(NULL, "/");
        }
	return cnt;
}

/*============================================================================
 * Strip leading and trailing spaces off the string str.
 */
static char* strstrip (char *str, char* s)
{
        char *eos = str + strlen(str);          /* -> end of string */

        while (*str && strchr(s, *str))
                ++str                           /* strip leading spaces */
        ;
        while ((eos > str) && strchr(s, *(eos - 1)))
                --eos                           /* strip trailing spaces */
        ;
        *eos = '\0';
        return str;
}
/*============================================================================
 * Configure PPP firmware.
 */
static int ppp_configure(sdla_t *card, void *data)
{
	ppp_mbox_t *mb = card->mbox;
	int data_len = sizeof(ppp508_conf_t); 
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	memcpy(mb->data, data, data_len);
	mb->cmd.length  = data_len;
	mb->cmd.command = PPP_SET_CONFIG;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	if (err != CMD_OK) 
		ppp_error(card, err, mb);
	
	return err;
}

/*============================================================================
 * Set interrupt mode.
 */
static int ppp_set_intr_mode(sdla_t *card, unsigned char mode)
{
	ppp_mbox_t *mb = card->mbox;
        ppp_intr_info_t *ppp_intr_data = (ppp_intr_info_t *) &mb->data[0];
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	ppp_intr_data->i_enable = mode;

	ppp_intr_data->irq = card->hw.irq;
	mb->cmd.length = 2;

       /* If timer has been enabled, set the timer delay to 1sec */
       if (mode & 0x80){
       		ppp_intr_data->timer_len = 250; //5;//100; //250;
                mb->cmd.length = 4;
        }
	
	mb->cmd.command = PPP_SET_INTR_FLAGS;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
	
	if (err != CMD_OK) 
		ppp_error(card, err, mb);
 		

	return err;
}

/*============================================================================
 * Enable communications.
 */
static int ppp_comm_enable(sdla_t *card)
{
	ppp_mbox_t *mb = card->mbox;
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	mb->cmd.command = PPP_COMM_ENABLE;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
	
	if (err != CMD_OK) 
		ppp_error(card, err, mb);
	else	
		card->u.p.comm_enabled = 1;	

	return err;
}

/*============================================================================
 * Disable communications.
 */
static int ppp_comm_disable(sdla_t *card)
{
	ppp_mbox_t *mb = card->mbox;
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	mb->cmd.command = PPP_COMM_DISABLE;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
	if (err != CMD_OK) 
		ppp_error(card, err, mb);
	else
		card->u.p.comm_enabled = 0;

	return err;
}

static int ppp_comm_disable_shutdown(sdla_t *card)
{
	ppp_mbox_t *mb = card->mbox;
	ppp_intr_info_t *ppp_intr_data;
	int err;

	if (!mb){
		return 1;
	}
	
	ppp_intr_data = (ppp_intr_info_t *) &mb->data[0];
	
	/* Disable all interrupts */
	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	ppp_intr_data->i_enable = 0;

	ppp_intr_data->irq = card->hw.irq;
	mb->cmd.length = 2;

	mb->cmd.command = PPP_SET_INTR_FLAGS;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	/* Disable communicatinons */
	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	mb->cmd.command = PPP_COMM_DISABLE;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	card->u.p.comm_enabled = 0;

	return 0;
}



/*============================================================================
 * Get communications error statistics.
 */
static int ppp_get_err_stats(sdla_t *card)
{
	ppp_mbox_t *mb = card->mbox;
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	mb->cmd.command = PPP_READ_ERROR_STATS;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
	
	if (err == CMD_OK) {
		
		ppp_err_stats_t* stats = (void*)mb->data;
		card->wandev.stats.rx_over_errors    = stats->rx_overrun;
		card->wandev.stats.rx_crc_errors     = stats->rx_bad_crc;
		card->wandev.stats.rx_missed_errors  = stats->rx_abort;
		card->wandev.stats.rx_length_errors  = stats->rx_lost;
		card->wandev.stats.tx_aborted_errors = stats->tx_abort;
	
	} else 
		ppp_error(card, err, mb);
	
	return err;
}

/*============================================================================
 * Send packet.
 *	Return:	0 - o.k.
 *		1 - no transmit buffers available
 */
static int ppp_send (sdla_t *card, void *data, unsigned len, unsigned proto)
{
	ppp_buf_ctl_t *txbuf = card->u.p.txbuf;

	if (txbuf->flag)
                return 1;
	
	sdla_poke(&card->hw, txbuf->buf.ptr, data, len);

	txbuf->length = len;		/* frame length */
	
	if (proto == htons(ETH_P_IPX))
		txbuf->proto = 0x01;	/* protocol ID */
	else
		txbuf->proto = 0x00;	/* protocol ID */
	
	txbuf->flag = 1;		/* start transmission */

	/* Update transmit buffer control fields */
	card->u.p.txbuf = ++txbuf;

	if ((void*)txbuf > card->u.p.txbuf_last)
		card->u.p.txbuf = card->u.p.txbuf_base;

	return 0;
}

/****** Firmware Error Handler **********************************************/

/*============================================================================
 * Firmware error handler.
 *	This routine is called whenever firmware command returns non-zero
 *	return code.
 *
 * Return zero if previous command has to be cancelled.
 */
static int ppp_error(sdla_t *card, int err, ppp_mbox_t *mb)
{
	unsigned cmd = mb->cmd.command;

	switch (err) {

		case CMD_TIMEOUT:
			printk(KERN_INFO "%s: command 0x%02X timed out!\n",
				card->devname, cmd);
			break;

		default:
			printk(KERN_INFO "%s: command 0x%02X returned 0x%02X!\n"
				, card->devname, cmd, err);
	}

	return 0;
}

/****** Interrupt Handlers **************************************************/

/*============================================================================
 * PPP interrupt service routine.
 */
static void wpp_isr (sdla_t *card)
{
	ppp_flags_t *flags = card->flags;
	char *ptr = &flags->iflag;
	netdevice_t *dev = card->wandev.dev;
	int i;

	card->in_isr = 1;
	++card->statistics.isr_entry;

	if (!dev && flags->iflag != PPP_INTR_CMD){
		card->in_isr = 0;
		flags->iflag = 0;
		return;
	}
	
	if (test_bit(PERI_CRIT, (void*)&card->wandev.critical)) {
		card->in_isr = 0;
		flags->iflag = 0;
		return;
	}
	
	
	if(card->hw.type != SDLA_S514){
		if (test_bit(SEND_CRIT, (void*)&card->wandev.critical)) {
			++card->statistics.isr_already_critical;
			printk (KERN_INFO "%s: Critical while in ISR!\n",
					card->devname);
			card->in_isr = 0;
			flags->iflag = 0;
			return;
		}
	}

	switch (flags->iflag) {

		case PPP_INTR_RXRDY:	/* receive interrupt  0x01  (bit 0)*/
			++card->statistics.isr_rx;
			rx_intr(card);
			break;

		case PPP_INTR_TXRDY:	/* transmit interrupt  0x02 (bit 1)*/
			++card->statistics.isr_tx;
			flags->imask &= ~PPP_INTR_TXRDY;
			wake_net_dev(dev);
			break;

		case PPP_INTR_CMD:      /* interface command completed */
			++Intr_test_counter;
			++card->statistics.isr_intr_test;
			break;

		case PPP_INTR_MODEM:    /* modem status change (DCD, CTS) 0x04 (bit 2)*/
		case PPP_INTR_DISC:  	/* Data link disconnected 0x10  (bit 4)*/	
		case PPP_INTR_OPEN:   	/* Data link open 0x20  (bit 5)*/
		case PPP_INTR_DROP_DTR:	/* DTR drop timeout expired  0x40 bit 6 */
			event_intr(card);
			break;
	
		case PPP_INTR_TIMER:
			timer_intr(card);
			break;	 

		default:	/* unexpected interrupt */
			++card->statistics.isr_spurious;
			printk(KERN_INFO "%s: spurious interrupt 0x%02X!\n", 
				card->devname, flags->iflag);
			printk(KERN_INFO "%s: ID Bytes = ",card->devname);
	 		for(i = 0; i < 8; i ++){
				printk("0x%02X ", *(ptr + 0x28 + i));
			}
			printk("\n");	
	}
	
	card->in_isr = 0;
	flags->iflag = 0;
	return;
}

/*============================================================================
 * Receive interrupt handler.
 */
static void rx_intr(sdla_t *card)
{
	ppp_buf_ctl_t *rxbuf = card->rxmb;
	netdevice_t *dev = card->wandev.dev;
	ppp_private_area_t *ppp_priv_area;
	struct sk_buff *skb;
	unsigned len;
	void *buf;
	int i;
        ppp_flags_t *flags = card->flags;
        char *ptr = &flags->iflag;
	int udp_type;
	

	if (rxbuf->flag != 0x01) {

		printk(KERN_INFO 
			"%s: corrupted Rx buffer @ 0x%X, flag = 0x%02X!\n", 
			card->devname, (unsigned)rxbuf, rxbuf->flag);
	
		printk(KERN_INFO "%s: ID Bytes = ",card->devname);
	 	
		for(i = 0; i < 8; i ++)
			printk(KERN_INFO "0x%02X ", *(ptr + 0x28 + i));
		printk(KERN_INFO "\n");	
		
		++card->statistics.rx_intr_corrupt_rx_bfr;


		/* Bug Fix: Mar 6 2000
                 * If we get a corrupted mailbox, it means that driver 
                 * is out of sync with the firmware. There is no recovery.
                 * If we don't turn off all interrupts for this card
                 * the machine will crash. 
                 */
		printk(KERN_INFO "%s: Critical router failure ...!!!\n", card->devname);
		printk(KERN_INFO "Please contact Sangoma Technologies !\n");
		ppp_set_intr_mode(card,0);
		return;
	}
      
	if (dev && is_dev_running(dev) && dev->priv){
	
		len  = rxbuf->length;
		ppp_priv_area = dev->priv;

		/* Allocate socket buffer */
		skb = dev_alloc_skb(len);

		if (skb != NULL) {
		
			/* Copy data to the socket buffer */
			unsigned addr = rxbuf->buf.ptr;

			if ((addr + len) > card->u.p.rx_top + 1) {
			
				unsigned tmp = card->u.p.rx_top - addr + 1;
				buf = skb_put(skb, tmp);
				sdla_peek(&card->hw, addr, buf, tmp);
				addr = card->u.p.rx_base;
				len -= tmp;
			}
			buf = skb_put(skb, len);
			sdla_peek(&card->hw, addr, buf, len);

			/* Decapsulate packet */
        		switch (rxbuf->proto) {
	
				case 0x00:
					skb->protocol = htons(ETH_P_IP);
					break;

				case 0x01:
					skb->protocol = htons(ETH_P_IPX);
					break;
			}

			udp_type = udp_pkt_type( skb, card );

			if (udp_type == UDP_PTPIPE_TYPE){

				/* Handle a UDP Request in Timer Interrupt */
				if(store_udp_mgmt_pkt(UDP_PKT_FRM_NETWORK, card, skb, dev,
                	              			ppp_priv_area)){
	               			flags->imask |= PPP_INTR_TIMER;
				}
				++ppp_priv_area->rx_intr_stat.rx_intr_PIPE_request;


			} else if (handle_IPXWAN(skb->data,card->devname, 
						 ppp_priv_area->enable_IPX, 
						 ppp_priv_area->network_number, 
						 skb->protocol)) {
			
				/* Handle an IPXWAN packet */
				if( ppp_priv_area->enable_IPX) {
					
					/* Make sure we are not already sending */
					if (!test_bit(SEND_CRIT, &card->wandev.critical)){
					 	ppp_send(card, skb->data, skb->len, htons(ETH_P_IPX));
					}
					wan_dev_kfree_skb(skb,FREE_READ);

				} else {
					++card->wandev.stats.rx_dropped;
				}
			} else {
				/* Pass data up the protocol stack */
	    			skb->dev = dev;
				skb->mac.raw  = skb->data;

			    	++card->wandev.stats.rx_packets;
				card->wandev.stats.rx_bytes += skb->len;
		    		++ppp_priv_area->rx_intr_stat.rx_intr_bfr_passed_to_stack;	
				netif_rx(skb);
			}

		} else {
	
			if (net_ratelimit()){
				printk(KERN_INFO "%s: no socket buffers available!\n",
					card->devname);
			}
			++card->wandev.stats.rx_dropped;
			++ppp_priv_area->rx_intr_stat.rx_intr_no_socket;
		}

	} else {
		++card->statistics.rx_intr_dev_not_started;
	}

	/* Release buffer element and calculate a pointer to the next one */
	rxbuf->flag = 0x00;
	card->rxmb = ++rxbuf;
	if ((void*)rxbuf > card->u.p.rxbuf_last)
		card->rxmb = card->u.p.rxbuf_base;
}


void event_intr (sdla_t *card)
{
 	netdevice_t* dev = card->wandev.dev;
        ppp_private_area_t* ppp_priv_area = dev->priv;
	volatile ppp_flags_t *flags = card->flags;

	switch (flags->iflag){

		case PPP_INTR_MODEM:    /* modem status change (DCD, CTS) 0x04  (bit 2)*/

			if (IS_56K(card->wandev.te_cfg)) {

				FRONT_END_STATUS_STRUCT *FE_status = 
				(FRONT_END_STATUS_STRUCT *)((unsigned long)card->hw.dpmbase + 0xF020);
				card->wandev.RR8_reg_56k = 
					FE_status->FE_U.stat_56k.RR8_56k;	
				card->wandev.RRA_reg_56k = 
					FE_status->FE_U.stat_56k.RRA_56k;	
				card->wandev.RRC_reg_56k = 
					FE_status->FE_U.stat_56k.RRC_56k;	
				sdla_56k_alarm(card, 0);

				FE_status->opp_flag = 0x01;

				ppp_handle_front_end_state(card);	
				break;
			}

			if (IS_TE1(card->wandev.te_cfg)) {
				
				FRONT_END_STATUS_STRUCT *FE_status = 
				(FRONT_END_STATUS_STRUCT *)((unsigned long)card->hw.dpmbase + 0xF020);

				/* TE_INTR */
				sdla_te_intr(card);
				FE_status->opp_flag = 0x01;

				ppp_handle_front_end_state(card);
				break;
			}	

			if (flags->mstatus == 0x28){
				card->wandev.front_end_status = FE_CONNECTED;			
			}else{
				card->wandev.front_end_status = FE_DISCONNECTED;	
			}
		
			if (net_ratelimit()){
				printk (KERN_INFO "\n");
				printk (KERN_INFO "%s: Modem status: DCD=%s CTS=%s\n",
					card->devname, DCD(flags->mstatus), CTS(flags->mstatus));
			}

			ppp_handle_front_end_state(card);
			break;

		case PPP_INTR_DISC:  	/* Data link disconnected 0x10  (bit 4)*/	

			printk (KERN_INFO "\n");
			printk (KERN_INFO "%s: Data link disconnected intr\n",
					       card->devname);
			
			show_disc_cause(card, flags->disc_cause);

			if (flags->disc_cause &
				(PPP_LOCAL_TERMINATION |
				 PPP_REMOTE_TERMINATION)) {

				if (card->u.p.ip_mode == WANOPT_PPP_PEER) { 
					set_bit(0,&Read_connection_info);
				}
				ppp_priv_area->ppp_state = WAN_DISCONNECTED;
				wanpipe_set_state(card, WAN_DISCONNECTED);

				ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_PPP_EVENT;
				flags->imask |= PPP_INTR_TIMER;
				trigger_ppp_poll(dev);
			}
			break;

		case PPP_INTR_OPEN:   	/* Data link open 0x20  (bit 5)*/

			NEX_PRINTK (KERN_INFO "%s: PPP Link Open, LCP=%s IP=%s\n",
					card->devname,LCP(flags->lcp_state),
					IP(flags->ip_state));

			if (flags->lcp_state == 0x09 && 
                           (flags->ip_state == 0x09 || flags->ipx_state == 0x09)){

                                /* Initialize the polling timer and set the state
                                 * to WAN_CONNNECTED */


				/* BUG FIX: When the protocol restarts, during heavy 
                                 * traffic, board tx buffers and driver tx buffers
                                 * can go out of sync.  This checks the condition
                                 * and if the tx buffers are out of sync, the 
                                 * protocols are restarted. 
                                 * I don't know why the board tx buffer is out
                                 * of sync. It could be that a packets is tx
                                 * while the link is down, but that is not 
                                 * possible. The other possiblility is that the
                                 * firmware doesn't reinitialize properly.
                                 * FIXME: A better fix should be found.
                                 */ 
				if (detect_and_fix_tx_bug(card)){

					ppp_comm_disable(card);

					ppp_priv_area->ppp_state = WAN_DISCONNECTED;
					wanpipe_set_state(card, WAN_DISCONNECTED);

					ppp_priv_area->timer_int_enabled |= 
						TMR_INT_ENABLED_PPP_EVENT;
					flags->imask |= PPP_INTR_TIMER;
					break;	
				}

				card->state_tick = jiffies;
				ppp_priv_area->ppp_state = WAN_CONNECTED;

				if (card->wandev.ignore_front_end_status == WANOPT_YES || 
				    card->wandev.front_end_status == FE_CONNECTED){
					
					wanpipe_set_state(card, WAN_CONNECTED);

					NEX_PRINTK(KERN_INFO "CON: L Tx: %lx  B Tx: %lx || L Rx %lx B Rx %lx\n",
						(unsigned long)card->u.p.txbuf, *card->u.p.txbuf_next,
						(unsigned long)card->rxmb, *card->u.p.rxbuf_next);

					/* Tell timer interrupt that PPP event occured */
					ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_PPP_EVENT;
					flags->imask |= PPP_INTR_TIMER;

					/* If we are in PEER mode, we must first obtain the
					 * IP information and then go into the poll routine */
					if (card->u.p.ip_mode != WANOPT_PPP_PEER){	
						trigger_ppp_poll(dev);
					}
				}
			}
                   	break;

		case PPP_INTR_DROP_DTR:		/* DTR drop timeout expired  0x40 bit 6 */

			printk (KERN_INFO "\n");
			printk(KERN_INFO "%s: DTR Drop Timeout Interrupt \n",card->devname); 

			if (card->u.p.ip_mode == WANOPT_PPP_PEER) { 
				set_bit(0,&Read_connection_info);
			}
	
			ppp_priv_area->ppp_state = WAN_DISCONNECTED;
			wanpipe_set_state(card, WAN_DISCONNECTED);

			show_disc_cause(card, flags->disc_cause);
			ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_PPP_EVENT;
			flags->imask |= PPP_INTR_TIMER;
			trigger_ppp_poll(dev);
			break;
		
		default:
			printk(KERN_INFO "%s: Error, Invalid PPP Event\n",card->devname);
	}
}



/* TIMER INTERRUPT */

void timer_intr (sdla_t *card)
{

        netdevice_t* dev = card->wandev.dev;
        ppp_private_area_t* ppp_priv_area = dev->priv;
	ppp_flags_t *flags = card->flags;


	if (ppp_priv_area->timer_int_enabled & TMR_INT_ENABLED_CONFIG){
		config_ppp(card);
		ppp_priv_area->timer_int_enabled &= 
					~TMR_INT_ENABLED_CONFIG;	
	}

	/* Update statistics */
	if (ppp_priv_area->timer_int_enabled & TMR_INT_ENABLED_UPDATE){
		
		ppp_get_err_stats(card);
		
		if (IS_TE1(card->wandev.te_cfg)) {	
			card->wandev.cur_te_alarm = sdla_te_alarm(card, 0); 
			/* TE1 Update T1/E1 perfomance counters */
			sdla_te_pmon(card);
		}else if (IS_56K(card->wandev.te_cfg)) {
			/* 56K Update CSU/DSU alarms */
			card->wandev.k56_alarm = sdla_56k_alarm(card, 1);
		}

                if(!(--ppp_priv_area->update_comms_stats)){
			ppp_priv_area->timer_int_enabled &= 
				~TMR_INT_ENABLED_UPDATE;
		}
	}

	/* TE timer interrupt */
	if (ppp_priv_area->timer_int_enabled & TMR_INT_ENABLED_TE){
		sdla_te_polling(card);
		ppp_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_TE;
	}

	/* PPIPEMON UDP request */
	if (ppp_priv_area->timer_int_enabled & TMR_INT_ENABLED_UDP){
		process_udp_mgmt_pkt(card,dev, ppp_priv_area);
		ppp_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_UDP;
	}

	/* PPP Event */
	if (ppp_priv_area->timer_int_enabled & TMR_INT_ENABLED_PPP_EVENT){

		if (ppp_priv_area->ppp_state == WAN_DISCONNECTED){
			if (retrigger_comm(card) < 0){
				/* DTR has been dropped.  The card will 
				 * give us an interrupt and timer will
				 * be called to retry enabling communications. 
				 */
				ppp_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_PPP_EVENT;
			}
		}

		/* If the state is CONNECTING, it means that communicatins were
	 	 * enabled. When the remote side enables its comminication we
	 	 * should get an interrupt PPP_INTR_OPEN, thus turn off polling 
		 */

		else if (ppp_priv_area->ppp_state == WAN_CONNECTING){
			/* Turn off the timer interrupt */
			ppp_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_PPP_EVENT;
		}

		/* If state is connected and we are in PEER mode 
	 	 * poll for an IP address which will be provided by remote end.
		 *
		 * We must wait for the main state to become connected,
		 * because the front end might be disconnected still,
		 * and the interface might be down.
	 	 */
		else if ((card->wandev.state == WAN_CONNECTED && 
		  	  card->u.p.ip_mode == WANOPT_PPP_PEER) && 
		  	  test_bit(0,&Read_connection_info)){

			card->state_tick = jiffies;
			if (read_connection_info (card)){
				printk(KERN_INFO "%s: Failed to read PEER IP Addresses\n",
					card->devname);
			}else{
				clear_bit(0,&Read_connection_info);
				set_bit(1,&Read_connection_info);
				trigger_ppp_poll(dev);
			}
		}else{
			//FIXME Put the comment back int
			ppp_priv_area->timer_int_enabled &= ~TMR_INT_ENABLED_PPP_EVENT;
		}

	}/* End of PPP_EVENT */


	/* Only disable the timer interrupt if there are no udp, statistic */
	/* updates or events pending */
        if(!ppp_priv_area->timer_int_enabled) {
                flags->imask &= ~PPP_INTR_TIMER;
        }
}


static int handle_IPXWAN(unsigned char *sendpacket, char *devname, unsigned char enable_IPX, unsigned long network_number, unsigned short proto)
{
	int i;

	if( proto == htons(ETH_P_IPX) ) {
		//It's an IPX packet
		if(!enable_IPX) {
			//Return 1 so we don't pass it up the stack.
			return 1;
		}
	} else {
		//It's not IPX so pass it up the stack.
		return 0;
	}

	if( sendpacket[16] == 0x90 &&
	    sendpacket[17] == 0x04)
	{
		//It's IPXWAN

		if( sendpacket[2] == 0x02 &&
		    sendpacket[34] == 0x00)
		{
			//It's a timer request packet
			printk(KERN_INFO "%s: Received IPXWAN Timer Request packet\n",devname);

			//Go through the routing options and answer no to every
			//option except Unnumbered RIP/SAP
			for(i = 41; sendpacket[i] == 0x00; i += 5)
			{
				//0x02 is the option for Unnumbered RIP/SAP
				if( sendpacket[i + 4] != 0x02)
				{
					sendpacket[i + 1] = 0;
				}
			}

			//Skip over the extended Node ID option
			if( sendpacket[i] == 0x04 )
			{
				i += 8;
			}

			//We also want to turn off all header compression opt.
			for(; sendpacket[i] == 0x80 ;)
			{
				sendpacket[i + 1] = 0;
				i += (sendpacket[i + 2] << 8) + (sendpacket[i + 3]) + 4;
			}

			//Set the packet type to timer response
			sendpacket[34] = 0x01;

			printk(KERN_INFO "%s: Sending IPXWAN Timer Response\n",devname);
		}
		else if( sendpacket[34] == 0x02 )
		{
			//This is an information request packet
			printk(KERN_INFO "%s: Received IPXWAN Information Request packet\n",devname);

			//Set the packet type to information response
			sendpacket[34] = 0x03;

			//Set the router name
			sendpacket[51] = 'P';
			sendpacket[52] = 'T';
			sendpacket[53] = 'P';
			sendpacket[54] = 'I';
			sendpacket[55] = 'P';
			sendpacket[56] = 'E';
			sendpacket[57] = '-';
			sendpacket[58] = CVHexToAscii(network_number >> 28);
			sendpacket[59] = CVHexToAscii((network_number & 0x0F000000)>> 24);
			sendpacket[60] = CVHexToAscii((network_number & 0x00F00000)>> 20);
			sendpacket[61] = CVHexToAscii((network_number & 0x000F0000)>> 16);
			sendpacket[62] = CVHexToAscii((network_number & 0x0000F000)>> 12);
			sendpacket[63] = CVHexToAscii((network_number & 0x00000F00)>> 8);
			sendpacket[64] = CVHexToAscii((network_number & 0x000000F0)>> 4);
			sendpacket[65] = CVHexToAscii(network_number & 0x0000000F);
			for(i = 66; i < 99; i+= 1)
			{
				sendpacket[i] = 0;
			}

			printk(KERN_INFO "%s: Sending IPXWAN Information Response packet\n",devname);
		}
		else
		{
			printk(KERN_INFO "%s: Unknown IPXWAN packet!\n",devname);
			return 0;
		}

		//Set the WNodeID to our network address
		sendpacket[35] = (unsigned char)(network_number >> 24);
		sendpacket[36] = (unsigned char)((network_number & 0x00FF0000) >> 16);
		sendpacket[37] = (unsigned char)((network_number & 0x0000FF00) >> 8);
		sendpacket[38] = (unsigned char)(network_number & 0x000000FF);

		return 1;
	} else {
		//If we get here's its an IPX-data packet, so it'll get passed up the stack.

		//switch the network numbers
		switch_net_numbers(sendpacket, network_number, 1);	
		return 0;
	}
}

/****** Background Polling Routines  ****************************************/

/* All polling functions are invoked by the TIMER interrupt in the wpp_isr 
 * routine.  
 */

/*============================================================================
 * Monitor active link phase.
 */
static void process_route (sdla_t *card)
{
	ppp_flags_t *flags = card->flags;
	netdevice_t *dev = card->wandev.dev;
	ppp_private_area_t *ppp_priv_area = dev->priv;
	
	if ((card->u.p.ip_mode == WANOPT_PPP_PEER) &&
	    (flags->ip_state == 0x09)){ 

		/* We get ip_local from the firmware in PEER mode.
	         * Therefore, if ip_local is 0, we failed to obtain
         	 * the remote IP address. */
		if (ppp_priv_area->ip_local == 0) 
			return;
		
		printk(KERN_INFO "%s: IPCP State Opened.\n", card->devname);
		if (read_info( card )) {
   			printk(KERN_INFO 
				"%s: An error occurred in IP assignment.\n", 
				card->devname);
		} else {
			struct in_device *in_dev = dev->ip_ptr;
			if (in_dev != NULL ) {
				struct in_ifaddr *ifa = in_dev->ifa_list;
				if (ifa){
					printk(KERN_INFO "%s: Assigned Lcl. Addr: %s\n", 
						card->devname, in_ntoa(ifa->ifa_local));
					printk(KERN_INFO "%s: Assigned Rmt. Addr: %s\n", 
							card->devname, in_ntoa(ifa->ifa_address));
				}
			}else{
				printk(KERN_INFO 
				"%s: Error: Failed to add a route for PPP interface %s\n",
					card->devname,dev->name);	
			}
		}
	}
}

/*============================================================================
 * Monitor physical link disconnected phase.
 *  o if interface is up and the hold-down timeout has expired, then retry
 *    connection.
 */
static int retrigger_comm(sdla_t *card)
{
	netdevice_t *dev = card->wandev.dev;
	ppp_private_area_t *ppp_priv_area;
	int err=0;

	if (!dev)
		return 0;
		
	if (!(ppp_priv_area = dev->priv))
		return 0;

	if ((jiffies - card->state_tick) > HOLD_DOWN_TIME){

		/*Bug Fix: May 24 2001
		 * Only set state if the communicatins are
		 * enabled */
		err = ppp_comm_enable(card);
	
		switch (err){

		case 0x00: 
			/* Communications are back up, change
			 * state to connecting and reinitialize
			 * buffers */
			ppp_priv_area->ppp_state = WAN_CONNECTING;
			wanpipe_set_state(card, WAN_CONNECTING);
			init_ppp_tx_rx_buff( card );
			ppp_priv_area->comm_busy_retry=0;
			break;
			
		case 0x0A:  
			printk(KERN_INFO "%s: Comm enable busy, retrying in %i sec.\n",
					card->devname,HOLD_DOWN_TIME);
			ppp_comm_disable(card);	
			card->state_tick=jiffies;
			
			/* We are in process of handing up, retry
			 * again later by leaving the wanpipe  
			 * device in DISCONNECTED state */
			
			return -EINVAL;

		/* NB: 
		 * By setting wanpipe state to connecting, the
		 * communicatins will stay in disabled state, and the
		 * retries will fail */
			
		case 0x01:
			printk(KERN_INFO "%s: ERROR PPP Com-Enable failed: protocol not configured!\n",
					card->devname);
			return -EFAULT;

		case 0x09:
			printk(KERN_INFO "%s: ERROR PPP Com-Enable failed: already enabled!\n",
					card->devname);
			ppp_priv_area->ppp_state = WAN_CONNECTING;
			wanpipe_set_state(card, WAN_CONNECTING);
			break;

		default:
			printk(KERN_INFO "%s: ERROR PPP Com-Enable failed: unknown rc=0x%X!\n",
					card->devname,err);
			return -EINVAL;
		}		
	}
	return 0;
}

/****** Miscellaneous Functions *********************************************/

/*============================================================================
 * Configure S508 adapter.
 */
static int config508(netdevice_t *dev, sdla_t *card)
{
	ppp508_conf_t cfg;
	ppp_private_area_t *ppp_priv_area = dev->priv;

	/* Prepare PPP configuration structure */
	memset(&cfg, 0, sizeof(ppp508_conf_t));

	if (card->wandev.clocking)
		cfg.line_speed = card->wandev.bps;

	if (card->wandev.interface == WANOPT_RS232)
		cfg.conf_flags |= INTERFACE_LEVEL_RS232;


        cfg.conf_flags 	|= DONT_TERMINATE_LNK_MAX_CONFIG; /*send Configure-Request packets forever*/
	cfg.txbuf_percent	= PERCENT_TX_BUFF;	/* % of Tx bufs */
	cfg.mtu_local		= card->wandev.mtu;
	cfg.mtu_remote		= card->wandev.mtu;                  /*    Default   */
	cfg.restart_tmr		= TIME_BETWEEN_CONF_REQ;  	     /*    30 = 3sec */
	cfg.auth_rsrt_tmr	= TIME_BETWEEN_PAP_CHAP_REQ;         /*    30 = 3sec */
	cfg.auth_wait_tmr	= WAIT_PAP_CHAP_WITHOUT_REPLY;       /*   300 = 30s  */
	cfg.mdm_fail_tmr	= WAIT_AFTER_DCD_CTS_LOW;            /*     5 = 0.5s */
	cfg.dtr_drop_tmr	= TIME_DCD_CTS_LOW_AFTER_LNK_DOWN;   /*    10 = 1s   */
	cfg.connect_tmout	= WAIT_DCD_HIGH_AFTER_ENABLE_COMM;   /*   900 = 90s  */
	cfg.conf_retry		= MAX_CONF_REQ_WITHOUT_REPLY;        /*    10 = 1s   */
	cfg.term_retry		= MAX_TERM_REQ_WITHOUT_REPLY;	     /*     2 times  */
	cfg.fail_retry		= NUM_CONF_NAK_WITHOUT_REPLY;        /*     5 times  */
	cfg.auth_retry		= NUM_AUTH_REQ_WITHOUT_REPLY;        /*     10 times */   


	if( !card->u.p.authenticator ) {
		printk(KERN_INFO "%s: Device is not configured as an authenticator\n", 
				card->devname);
		cfg.auth_options = NO_AUTHENTICATION;
	}else{
		printk(KERN_INFO "%s: Device is configured as an authenticator\n", 
				card->devname);
		cfg.auth_options = INBOUND_AUTH;
	}

	if( ppp_priv_area->pap == WANOPT_YES){
		cfg.auth_options |=PAP_AUTH;
		printk(KERN_INFO "%s: Pap enabled\n", card->devname);
	}
	if( ppp_priv_area->chap == WANOPT_YES){
		cfg.auth_options |= CHAP_AUTH;
		printk(KERN_INFO "%s: Chap enabled\n", card->devname);
	}


	if (ppp_priv_area->enable_IPX == WANOPT_YES){
		printk(KERN_INFO "%s: Enabling IPX Protocol\n",card->devname);
		cfg.ipx_options		= ENABLE_IPX | ROUTING_PROT_DEFAULT;
	}else{
		cfg.ipx_options 	= DISABLE_IPX;
	}

	switch (card->u.p.ip_mode) {
	
		case WANOPT_PPP_STATIC:

			printk(KERN_INFO "%s: PPP IP Mode: STATIC\n",card->devname);
			cfg.ip_options		= L_AND_R_IP_NO_ASSIG | 
							    ENABLE_IP;
			cfg.ip_local		= ppp_priv_area->ip_local;
			cfg.ip_remote		= ppp_priv_area->ip_remote;

			/* Debugging code used to check that IP addresses
                         * obtained from the kernel are correct */

			{
			char laddr[20];
			char raddr[20];
			strcpy(laddr,in_ntoa(cfg.ip_local));
			strcpy(raddr,in_ntoa(cfg.ip_remote));

                        NEX_PRINTK(KERN_INFO "Local %s Remote %s Name %s\n",
					laddr,raddr, dev->name);
			
			}	
			break;

		case WANOPT_PPP_HOST:

			printk(KERN_INFO "%s: PPP IP Mode: HOST\n",card->devname);
			cfg.ip_options		= L_IP_LOCAL_ASSIG |
						  R_IP_LOCAL_ASSIG | 
						  ENABLE_IP;

			cfg.ip_local		= ppp_priv_area->ip_local;
			cfg.ip_remote		= ppp_priv_area->ip_remote;

			/* Debugging code used to check that IP addresses
                         * obtained from the kernel are correct */
			{
			char laddr[20];
			char raddr[20];
			strcpy(laddr,in_ntoa(cfg.ip_local));
			strcpy(raddr,in_ntoa(cfg.ip_remote));

                        NEX_PRINTK (KERN_INFO "Local %s Remote %s Name %s\n",
					laddr,raddr, dev->name);
			
			}	


			break;
	
		case WANOPT_PPP_PEER:

			printk(KERN_INFO "%s: PPP IP Mode: PEER\n",card->devname);
			cfg.ip_options		= L_IP_REMOTE_ASSIG | 
						  R_IP_REMOTE_ASSIG | 
							  ENABLE_IP;
			cfg.ip_local		= 0x00;
			cfg.ip_remote		= 0x00;
			break;

		default:
			printk(KERN_INFO "%s: ERROR: Unsuported PPP Mode Selected\n",
					card->devname);
			printk(KERN_INFO "%s:        PPP IP Modes: STATIC, PEER or HOST\n",
					card->devname);	
			return 1;
	}

	return ppp_configure(card, &cfg);
}

/*============================================================================
 * Show disconnection cause.
 */
static void show_disc_cause(sdla_t *card, unsigned cause)
{
	if (cause & 0x0802) 

		printk(KERN_INFO "%s: link terminated by peer\n", 
			card->devname);

	else if (cause & 0x0004) 

		printk(KERN_INFO "%s: link terminated by user\n", 
			card->devname);

	else if (cause & 0x0008) 

		printk(KERN_INFO "%s: authentication failed\n", card->devname);
	
	else if (cause & 0x0010) 

		printk(KERN_INFO 
			"%s: authentication protocol negotiation failed\n", 
			card->devname);

	else if (cause & 0x0020) 
		
		printk(KERN_INFO
		"%s: peer's request for authentication rejected\n",
		card->devname);

	else if (cause & 0x0040) 
	
		printk(KERN_INFO "%s: MRU option rejected by peer\n", 
		card->devname);

	else if (cause & 0x0080) 
	
		printk(KERN_INFO "%s: peer's MRU was too small\n", 
		card->devname);

	else if (cause & 0x0100) 

		printk(KERN_INFO "%s: failed to negotiate peer's LCP options\n",
		card->devname);

	else if (cause & 0x0200) 
		
		printk(KERN_INFO "%s: failed to negotiate peer's IPCP options\n"
		, card->devname);

	else if (cause & 0x0400) 

		printk(KERN_INFO 
			"%s: failed to negotiate peer's IPXCP options\n",
			card->devname);
}

/*=============================================================================
 * Process UDP call of type PTPIPEAB.
 */
static void process_udp_mgmt_pkt(sdla_t *card, netdevice_t *dev, 
				 ppp_private_area_t *ppp_priv_area ) 
{
	unsigned char buf2[5];
	unsigned char *buf;
	unsigned int frames, len;
	struct sk_buff *new_skb;
	unsigned short data_length, buffer_length, real_len;
	unsigned long data_ptr;
	int udp_mgmt_req_valid = 1;
	ppp_mbox_t *mbox = card->mbox;
	struct timeval tv;
	int err;
	ppp_udp_pkt_t *ppp_udp_pkt = (ppp_udp_pkt_t*)&ppp_priv_area->udp_pkt_data;

	memcpy(&buf2, &card->wandev.udp_port, 2 );


	if(ppp_priv_area->udp_pkt_src == UDP_PKT_FRM_NETWORK) {

		switch(ppp_udp_pkt->cblock.command) {

			case PPIPE_GET_IBA_DATA:
			case PPP_READ_CONFIG:
			case PPP_GET_CONNECTION_INFO:
			case PPIPE_ROUTER_UP_TIME:
			case PPP_READ_STATISTICS:
			case PPP_READ_ERROR_STATS:
			case PPP_READ_PACKET_STATS:
			case PPP_READ_LCP_STATS:
			case PPP_READ_IPCP_STATS:
			case PPP_READ_IPXCP_STATS:
			case PPP_READ_PAP_STATS:
			case PPP_READ_CHAP_STATS:
			case PPP_READ_CODE_VERSION:
			case PPIPE_GET_MEDIA_TYPE:
			case PPIPE_TE1_56K_STAT:
				udp_mgmt_req_valid = 1;
				break;
			   
			default:
				udp_mgmt_req_valid = 0;
				break;
		} 
	}
	
    	ppp_udp_pkt->cblock.opp_flag = 0x00;
  	if(!udp_mgmt_req_valid) {
	    
		/* set length to 0 */
    		ppp_udp_pkt->cblock.length = 0x00;

    		/* set return code */
    		ppp_udp_pkt->cblock.result = 0xCD; 
		++ppp_priv_area->pipe_mgmt_stat.UDP_PIPE_mgmt_direction_err;
	
		if (net_ratelimit()){	
			printk(KERN_INFO 
			"%s: Warning, Illegal UDP command attempted from network: %x\n",
			card->devname,ppp_udp_pkt->cblock.command);
		}
   	} else {
		/* Initialize the trace element */
		trace_element_t trace_element;		    

		switch (ppp_udp_pkt->cblock.command){

		/* PPIPE_ENABLE_TRACING */
    		case PPIPE_ENABLE_TRACING:
			if (!card->TracingEnabled) {
    			
				/* OPERATE_DATALINE_MONITOR */
    				mbox->cmd.command = PPP_DATALINE_MONITOR;
    				mbox->cmd.length = 0x01;
    				mbox->data[0] = ppp_udp_pkt->data[0];
	    			err = sdla_exec(mbox) ? 
					mbox->cmd.result : CMD_TIMEOUT;
	   
				if (err != CMD_OK) { 
	        			
					ppp_error(card, err, mbox);
	        			card->TracingEnabled = 0;
	        		
					/* set the return code */

		        		ppp_udp_pkt->cblock.result = mbox->cmd.result;
	        			mbox->cmd.length = 0;
	        			break;
	    			} 

				sdla_peek(&card->hw, 0xC000, &buf2, 2);
		    
				ppp_priv_area->curr_trace_addr = 0;
		    		memcpy(&ppp_priv_area->curr_trace_addr, &buf2, 2);
		    		ppp_priv_area->start_trace_addr = 
						ppp_priv_area->curr_trace_addr;
				ppp_priv_area->end_trace_addr = 
					ppp_priv_area->start_trace_addr + END_OFFSET;
		    	
				/* MAX_SEND_BUFFER_SIZE - 28 (IP header) 
				   - 32 (ppipemon CBLOCK) */
		    		available_buffer_space = MAX_LGTH_UDP_MGNT_PKT - 
							 sizeof(ip_pkt_t)-
							 sizeof(udp_pkt_t)-
							 sizeof(wp_mgmt_t)-
							 sizeof(cblock_t);
	       	  	}
	       	  	ppp_udp_pkt->cblock.result = 0;
	       	  	mbox->cmd.length = 0;
	       	  	card->TracingEnabled = 1;
	       	  	break;
	   
		/* PPIPE_DISABLE_TRACING */
		case PPIPE_DISABLE_TRACING:
	      		
			if(card->TracingEnabled) {
		   	
				/* OPERATE_DATALINE_MONITOR */
		    		mbox->cmd.command = 0x33;
		    		mbox->cmd.length = 1;
		    		mbox->data[0] = 0x00;
		    		err = sdla_exec(mbox) ? 
					mbox->cmd.result : CMD_TIMEOUT;
	       	  
			} 
		
			/*set return code*/
			ppp_udp_pkt->cblock.result = 0;
			mbox->cmd.length = 0;
			card->TracingEnabled = 0;
			break;
	   
		/* PPIPE_GET_TRACE_INFO */
		case PPIPE_GET_TRACE_INFO:

			if(!card->TracingEnabled) {
				/* set return code */
	    			ppp_udp_pkt->cblock.result = 1;
	    			mbox->cmd.length = 0;
			}		    

			buffer_length = 0;
			
			/* frames < 62, where 62 is the number of trace
			   information elements.  There is in total 496
			   bytes of space and each trace information
			   element is 8 bytes. 
			 */
			for ( frames=0; frames<62; frames++) {
	
				trace_pkt_t *trace_pkt = (trace_pkt_t *)
					&ppp_udp_pkt->data[buffer_length];

				/* Read the whole trace packet */
				sdla_peek(&card->hw, ppp_priv_area->curr_trace_addr, 
					  &trace_element, sizeof(trace_element_t));
	
				/* no data on board so exit */
				if( trace_element.opp_flag == 0x00 ) 
					break;
	      
				data_ptr = trace_element.trace_data_ptr;

				/* See if there is actual data on the trace buffer */
				if (data_ptr){
					data_length = trace_element.trace_length;
				}else{
					data_length = 0;
					ppp_udp_pkt->data[0] |= 0x02;
				}

				//FIXME: Do we need this check
				if ((available_buffer_space - buffer_length) 
				     < (sizeof(trace_element_t)+1)){
					
					/*indicate we have more frames 
					 * on board and exit 
					 */
					ppp_udp_pkt->data[0] |= 0x02;
					break;
				}
				
				trace_pkt->status = trace_element.trace_type;
				trace_pkt->time_stamp = trace_element.trace_time_stamp;
				trace_pkt->real_length = trace_element.trace_length;

				real_len = trace_element.trace_length;	
				
				if(data_ptr == 0){
					trace_pkt->data_avail = 0x00;
				}else{
					/* we can take it next time */
					if ((available_buffer_space - buffer_length)<
						(real_len + sizeof(trace_pkt_t))){
					
						ppp_udp_pkt->data[0] |= 0x02;
						break;
					} 
					trace_pkt->data_avail = 0x01;
				
					/* get the data */
					sdla_peek(&card->hw, data_ptr, 
						  &trace_pkt->data,
						  real_len);
				}	
				/* zero the opp flag to 
				   show we got the frame */
				buf2[0] = 0x00;
				sdla_poke(&card->hw, ppp_priv_area->curr_trace_addr,
					  &buf2, 1);

				/* now move onto the next 
				   frame */
				ppp_priv_area->curr_trace_addr += sizeof(trace_element_t);

				/* check if we passed the last address */
				if ( ppp_priv_area->curr_trace_addr >= 
					ppp_priv_area->end_trace_addr){

					ppp_priv_area->curr_trace_addr = 
						ppp_priv_area->start_trace_addr;
				}
 
				/* update buffer length and make sure its even */ 

				if ( trace_pkt->data_avail == 0x01 ) {
					buffer_length += real_len - 1;
				}
 
				/* for the header */
				buffer_length += sizeof(trace_pkt_t);

				if( buffer_length & 0x0001 )
					buffer_length += 1;
			}

			/* ok now set the total number of frames passed
			   in the high 5 bits */
			ppp_udp_pkt->data[0] |= (frames << 2);
	 
			/* set the data length */
			mbox->cmd.length = buffer_length;
			ppp_udp_pkt->cblock.length = buffer_length;
	 
			/* set return code */
			ppp_udp_pkt->cblock.result = 0;
	      	  	break;

   		/* PPIPE_GET_IBA_DATA */
		case PPIPE_GET_IBA_DATA:
	        
			mbox->cmd.length = 0x09;
		
			sdla_peek(&card->hw, 0xF003, &ppp_udp_pkt->data, 
					mbox->cmd.length);
	        
			/* set the length of the data */
			ppp_udp_pkt->cblock.length = 0x09;

			/* set return code */
			ppp_udp_pkt->cblock.result = 0x00;
			ppp_udp_pkt->cblock.result = 0;
			break;

		/* PPIPE_FT1_READ_STATUS */
		case PPIPE_FT1_READ_STATUS:
			sdla_peek(&card->hw, 0xF020, &ppp_udp_pkt->data[0], 2);
			ppp_udp_pkt->cblock.length = mbox->cmd.length = 2;
			ppp_udp_pkt->cblock.result = 0;
			break;
		
		case PPIPE_FLUSH_DRIVER_STATS:   
			init_ppp_priv_struct( ppp_priv_area );
			init_global_statistics( card );
			mbox->cmd.length = 0;
			ppp_udp_pkt->cblock.result = 0;
			break;

		
		case PPIPE_ROUTER_UP_TIME:

			do_gettimeofday( &tv );
			ppp_priv_area->router_up_time = tv.tv_sec - 
					ppp_priv_area->router_start_time;
			*(unsigned long *)&ppp_udp_pkt->data = ppp_priv_area->router_up_time;
			mbox->cmd.length = 4;
			ppp_udp_pkt->cblock.result = 0;
			break;

				/* PPIPE_DRIVER_STATISTICS */   
		case PPIPE_DRIVER_STAT_IFSEND:
			memcpy(&ppp_udp_pkt->data, &ppp_priv_area->if_send_stat, 
				sizeof(if_send_stat_t));


			ppp_udp_pkt->cblock.result = 0;
			ppp_udp_pkt->cblock.length = sizeof(if_send_stat_t);
			mbox->cmd.length = sizeof(if_send_stat_t);	
			break;

		case PPIPE_DRIVER_STAT_INTR:
			memcpy(&ppp_udp_pkt->data, &card->statistics, 
				sizeof(global_stats_t));

			memcpy(&ppp_udp_pkt->data+sizeof(global_stats_t),
				&ppp_priv_area->rx_intr_stat,
				sizeof(rx_intr_stat_t));

			ppp_udp_pkt->cblock.result = 0;
			ppp_udp_pkt->cblock.length = sizeof(global_stats_t)+
						     sizeof(rx_intr_stat_t);
			mbox->cmd.length = ppp_udp_pkt->cblock.length;
			break;

		case PPIPE_DRIVER_STAT_GEN:
			memcpy( &ppp_udp_pkt->data,
				&ppp_priv_area->pipe_mgmt_stat,
				sizeof(pipe_mgmt_stat_t));

			memcpy(&ppp_udp_pkt->data+sizeof(pipe_mgmt_stat_t), 
			       &card->statistics, sizeof(global_stats_t));

			ppp_udp_pkt->cblock.result = 0;
			ppp_udp_pkt->cblock.length = sizeof(global_stats_t)+
						     sizeof(rx_intr_stat_t);
			mbox->cmd.length = ppp_udp_pkt->cblock.length;
			break;


		/* FT1 MONITOR STATUS */
   		case FT1_MONITOR_STATUS_CTRL:
	
			/* Enable FT1 MONITOR STATUS */
	        	if( ppp_udp_pkt->data[0] == 1) {
			
				if( rCount++ != 0 ) {
		        		ppp_udp_pkt->cblock.result = 0;
	          			mbox->cmd.length = 1;
		  			break;
		    		}	
	      		}

	      		/* Disable FT1 MONITOR STATUS */
	      		if( ppp_udp_pkt->data[0] == 0) {

	      	   		if( --rCount != 0) {
		  			ppp_udp_pkt->cblock.result = 0;
		  			mbox->cmd.length = 1;
		  			break;
	   	    		} 
	      		} 	
			goto udp_dflt_cmd;
			
		case ACTIVATE_LINELB_MODE:
			/* Activate Line loopback mode */
			if (!sdla_te_linelb(card, ACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case DEACTIVATE_LINELB_MODE:
			/* Deactivate Line loopback mode */
			if (!sdla_te_linelb(card, DEACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case ACTIVATE_PAYLB_MODE:
			/* Activate Payload loopback mode */
			if (!sdla_te_paylb(card, ACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case DEACTIVATE_PAYLB_MODE:
			/* Deactivate Payload loopback mode */
			if (!sdla_te_paylb(card, DEACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case ACTIVATE_DDLB_MODE:
			/* Activate Diagnostic Digital loopback mode */
			if (!sdla_te_ddlb(card, ACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case DEACTIVATE_DDLB_MODE:
			/* Deactivate Diagnostic Digital loopback mode */
			if (!sdla_te_ddlb(card, DEACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case ACTIVATE_LB_MODE:
			/* Send activation loopback code */
			if (!sdla_te_lb(card, ACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case DEACTIVATE_LB_MODE:
			/* Send deactivation loopback code */
			if (!sdla_te_lb(card, DEACTIVATE_LOOPBACK)){
				ppp_udp_pkt->cblock.result = CMD_OK;
			}
			break;

		case PPIPE_GET_MEDIA_TYPE:
	 		ppp_udp_pkt->data[0] = 
				(IS_T1(card->wandev) ? WANOPT_MEDIA_T1 :
				 IS_E1(card->wandev) ? WANOPT_MEDIA_E1 :
				 IS_56K(card->wandev.te_cfg) ? WANOPT_MEDIA_56K : 
				 				WANOPT_MEDIA_NONE);
		    	ppp_udp_pkt->cblock.length = sizeof(unsigned char); 
			ppp_udp_pkt->cblock.result = CMD_OK;
			mbox->cmd.length = sizeof(unsigned char);
			break;

		case PPIPE_TE1_56K_STAT:
		    	if (IS_TE1(card->wandev.te_cfg)) {	
	 	    		/* TE1_56K Read T1/E1/56K alarms */
			  	*(unsigned long *)&ppp_udp_pkt->data = 
		                        sdla_te_alarm(card, 0);
				/* TE1 Update T1/E1 perfomance counters */
    				sdla_te_pmon(card);
	        		memcpy(&ppp_udp_pkt->data[sizeof(unsigned long)],
					&card->wandev.te_pmon,
					sizeof(pmc_pmon_t));
		        	ppp_udp_pkt->cblock.result = CMD_OK;
		    		ppp_udp_pkt->cblock.length = 
					sizeof(unsigned long) + sizeof(pmc_pmon_t); 
				mbox->cmd.length = ppp_udp_pkt->cblock.length;
			}else if (IS_56K(card->wandev.te_cfg)){
				/* 56K Update CSU/DSU alarms */
				card->wandev.k56_alarm = sdla_56k_alarm(card, 1); 
			 	*(unsigned long *)&ppp_udp_pkt->data = 
			                        card->wandev.k56_alarm;
				ppp_udp_pkt->cblock.result = CMD_OK;
	    			ppp_udp_pkt->cblock.length = sizeof(unsigned long);
				mbox->cmd.length = ppp_udp_pkt->cblock.length;
			}
		    	break;

 		case PPIPE_FLUSH_TE1_PMON:
 	    		/* TE1 Flush T1/E1 pmon counters */
	    		if (IS_TE1(card->wandev.te_cfg)){	
				sdla_flush_te1_pmon(card);
	        		ppp_udp_pkt->cblock.result = CMD_OK;
	    		}
	    		break;

		case PPIPE_TE1_CFG:
			/* Read T1/E1 configuration */
		    	if (IS_TE1(card->wandev.te_cfg)) {	
	        		memcpy(&ppp_udp_pkt->data[0],
					&card->wandev.te_cfg,
					sizeof(sdla_te_cfg_t));
		        	ppp_udp_pkt->cblock.result = CMD_OK;
		    		ppp_udp_pkt->cblock.length = sizeof(sdla_te_cfg_t); 
				mbox->cmd.length = ppp_udp_pkt->cblock.length;
			}
			break;

	/* WARNING: FIXME: This should be fixed.
		 * The FT1 Status Ctrl doesn't have a break
                 * statment.  Thus, no code must be inserted
                 * HERE: between default and above case statement */

		default:
udp_dflt_cmd:
	        
			/* it's a board command */
			mbox->cmd.command = ppp_udp_pkt->cblock.command;
			mbox->cmd.length = ppp_udp_pkt->cblock.length;
 
			if(mbox->cmd.length) {
				memcpy(&mbox->data,(unsigned char *)ppp_udp_pkt->data,
				       mbox->cmd.length);
	      		} 
	          
			/* run the command on the board */
			err = sdla_exec(mbox) ? mbox->cmd.result : CMD_TIMEOUT;
		
			if (err != CMD_OK) {
		
		    		ppp_error(card, err, mbox);
		    		++ppp_priv_area->pipe_mgmt_stat.
					 UDP_PIPE_mgmt_adptr_cmnd_timeout;
				break;
			}
	          
		  	++ppp_priv_area->pipe_mgmt_stat.UDP_PIPE_mgmt_adptr_cmnd_OK;
		
			/* copy the result back to our buffer */
			memcpy(&ppp_udp_pkt->cblock,mbox, sizeof(cblock_t));
	          
			if(mbox->cmd.length) {
				memcpy(&ppp_udp_pkt->data,&mbox->data,mbox->cmd.length);
			} 

		} /* end of switch */
     	} /* end of else */

     	/* Fill UDP TTL */
     	ppp_udp_pkt->ip_pkt.ttl = card->wandev.ttl; 
     	len = reply_udp(ppp_priv_area->udp_pkt_data, mbox->cmd.length);

     	if (ppp_priv_area->udp_pkt_src == UDP_PKT_FRM_NETWORK) {

		/* Make sure we are not already sending */
		if (!test_bit(SEND_CRIT,&card->wandev.critical)){
			++ppp_priv_area->pipe_mgmt_stat.UDP_PIPE_mgmt_passed_to_adptr;
			ppp_send(card,ppp_priv_area->udp_pkt_data,len,ppp_priv_area->protocol);
		}

	} else {	
	
		/* Pass it up the stack
    		   Allocate socket buffer */
		if ((new_skb = dev_alloc_skb(len)) != NULL) {
	    	
			/* copy data into new_skb */

  	    		buf = skb_put(new_skb, len);
  	    		memcpy(buf,ppp_priv_area->udp_pkt_data, len);

	    		++ppp_priv_area->pipe_mgmt_stat.UDP_PIPE_mgmt_passed_to_stack;
			
            		/* Decapsulate packet and pass it up the protocol 
			   stack */
	    		new_skb->protocol = htons(ETH_P_IP);
            		new_skb->dev = dev;
	    		new_skb->mac.raw  = new_skb->data;
			netif_rx(new_skb);
		
		} else {
	    	
			++ppp_priv_area->pipe_mgmt_stat.UDP_PIPE_mgmt_no_socket;
			printk(KERN_INFO "no socket buffers available!\n");
  		}
    	}	

	ppp_priv_area->udp_pkt_lgth = 0;
	
	return; 
}

/*=============================================================================
 * Initial the ppp_private_area structure.
 */
static void init_ppp_priv_struct( ppp_private_area_t *ppp_priv_area )
{

	memset(&ppp_priv_area->if_send_stat, 0, sizeof(if_send_stat_t));
	memset(&ppp_priv_area->rx_intr_stat, 0, sizeof(rx_intr_stat_t));
	memset(&ppp_priv_area->pipe_mgmt_stat, 0, sizeof(pipe_mgmt_stat_t));	
}

/*============================================================================
 * Initialize Global Statistics
 */
static void init_global_statistics( sdla_t *card )
{
	memset(&card->statistics, 0, sizeof(global_stats_t));
}

/*============================================================================
 * Initialize Receive and Transmit Buffers.
 */
static void init_ppp_tx_rx_buff( sdla_t *card )
{
	ppp508_buf_info_t* info;

	if (card->hw.type == SDLA_S514) {
		
		info = (void*)(card->hw.dpmbase + PPP514_BUF_OFFS);

       		card->u.p.txbuf_base = (void*)(card->hw.dpmbase +
			info->txb_ptr);

                card->u.p.txbuf_last = (ppp_buf_ctl_t*)card->u.p.txbuf_base +
                        (info->txb_num - 1);

                card->u.p.rxbuf_base = (void*)(card->hw.dpmbase +
                        info->rxb_ptr);

                card->u.p.rxbuf_last = (ppp_buf_ctl_t*)card->u.p.rxbuf_base +
                        (info->rxb_num - 1);

	} else {
		
		info = (void*)(card->hw.dpmbase + PPP508_BUF_OFFS);

		card->u.p.txbuf_base = (void*)(card->hw.dpmbase +
			(info->txb_ptr - PPP508_MB_VECT));

		card->u.p.txbuf_last = (ppp_buf_ctl_t*)card->u.p.txbuf_base +
			(info->txb_num - 1);

		card->u.p.rxbuf_base = (void*)(card->hw.dpmbase +
			(info->rxb_ptr - PPP508_MB_VECT));

		card->u.p.rxbuf_last = (ppp_buf_ctl_t*)card->u.p.rxbuf_base +
			(info->rxb_num - 1);
	}

	card->u.p.txbuf_next = (unsigned long*)&info->txb_nxt; 
	card->u.p.rxbuf_next = (unsigned long*)&info->rxb1_ptr;

	card->u.p.rx_base = info->rxb_base;
        card->u.p.rx_top  = info->rxb_end;
      
	card->u.p.txbuf = card->u.p.txbuf_base;
	card->rxmb = card->u.p.rxbuf_base;

}

/*=============================================================================
 * Read Connection Information (ie for Remote IP address assginment).
 * Called when ppp interface connected.
 */
static int read_info( sdla_t *card )
{
	netdevice_t *dev = card->wandev.dev;
	ppp_private_area_t *ppp_priv_area = dev->priv;
	int err;
	struct ifreq if_info;
	struct sockaddr_in *if_data1, *if_data2;
	mm_segment_t fs;


	
	/* Set Local and remote addresses */
	memset(&if_info, 0, sizeof(if_info));
	strcpy(if_info.ifr_name, dev->name);


	fs = get_fs();
	set_fs(get_ds());     /* get user space block */ 

	/* Change the local and remote ip address of the interface.
	 * This will also add in the destination route.
	 */	
	if_data1 = (struct sockaddr_in *)&if_info.ifr_addr;
	if_data1->sin_addr.s_addr = ppp_priv_area->ip_local;
	if_data1->sin_family = AF_INET;
	err = devinet_ioctl( SIOCSIFADDR, &if_info );
	if_data2 = (struct sockaddr_in *)&if_info.ifr_dstaddr;
	if_data2->sin_addr.s_addr = ppp_priv_area->ip_remote;
	if_data2->sin_family = AF_INET;
	err = devinet_ioctl( SIOCSIFDSTADDR, &if_info );

	set_fs(fs);           /* restore old block */
	
	if (err) {
		printk (KERN_INFO "%s: Adding of route failed: %i\n",
			card->devname,err);
		printk (KERN_INFO "%s:	Local : %s\n",
			card->devname,in_ntoa(ppp_priv_area->ip_local));
		printk (KERN_INFO "%s:	Remote: %s\n",
			card->devname,in_ntoa(ppp_priv_area->ip_remote));
	}
	return err;
}

/*=============================================================================
 * Remove Dynamic Route.
 * Called when ppp interface disconnected.
 */

static void remove_route( sdla_t *card )
{

	netdevice_t *dev = card->wandev.dev;
	long ip_addr;
	int err;
        mm_segment_t fs;
	struct ifreq if_info;
	struct sockaddr_in *if_data1;
        struct in_device *in_dev = dev->ip_ptr;
        struct in_ifaddr *ifa;	

	if (!in_dev)
		return;
	
	if (!(ifa = in_dev->ifa_list))
		return;

	ip_addr = ifa->ifa_local;

	/* Set Local and remote addresses */
	memset(&if_info, 0, sizeof(if_info));
	strcpy(if_info.ifr_name, dev->name);

	fs = get_fs();
       	set_fs(get_ds());     /* get user space block */ 


	/* Change the local ip address of the interface to 0.
	 * This will also delete the destination route.
	 */	
	if_data1 = (struct sockaddr_in *)&if_info.ifr_addr;
	if_data1->sin_addr.s_addr = 0;
	if_data1->sin_family = AF_INET;
	err = devinet_ioctl( SIOCSIFADDR, &if_info );

        set_fs(fs);           /* restore old block */

	
	if (err) {
		printk (KERN_INFO "%s: Deleting dynamic route failed %d!\n",
			 card->devname, err);
		return;
	}else{
		printk (KERN_INFO "%s: PPP Deleting dynamic route %s successfuly\n",
			card->devname, in_ntoa(ip_addr));
	}
	return;
}

/*=============================================================================
 * Perform the Interrupt Test by running the READ_CODE_VERSION command MAX_INTR
 * _TEST_COUNTER times.
 */
static int intr_test( sdla_t *card )
{
	ppp_mbox_t *mb = card->mbox;
	int err,i;

	err = ppp_set_intr_mode( card, 0x08 );
	
	if (err == CMD_OK) { 
		
		for (i = 0; i < MAX_INTR_TEST_COUNTER; i ++) {	
			/* Run command READ_CODE_VERSION */
			memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
			mb->cmd.length  = 0;
			mb->cmd.command = PPP_READ_CODE_VERSION;
			err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;
			if (err != CMD_OK) 
				ppp_error(card, err, mb);
		}
	}
	else return err;

	err = ppp_set_intr_mode( card, 0 );
	if (err != CMD_OK) 
		return err;

	return 0;
}

/*==============================================================================
 * Determine what type of UDP call it is. DRVSTATS or PTPIPEAB ?
 */
static int udp_pkt_type( struct sk_buff *skb, sdla_t *card )
{
	unsigned char *sendpacket;
	unsigned char buf2[5]; 
	ppp_udp_pkt_t *ppp_udp_pkt = (ppp_udp_pkt_t *)skb->data; 
	
	sendpacket = skb->data;
	memcpy(&buf2, &card->wandev.udp_port, 2);
	
	if( 	ppp_udp_pkt->ip_pkt.ver_inet_hdr_length  == 0x45 &&        /* IP packet */ 
		sendpacket[9]  == 0x11 &&        /* UDP packet */
		sendpacket[22] == buf2[1] &&     /* UDP Port */
		sendpacket[23] == buf2[0] &&
		sendpacket[36] == 0x01 ) {
	
		if (    sendpacket[28] == 0x50 &&    /* PTPIPEAB: Signature */ 
			sendpacket[29] == 0x54 &&      
			sendpacket[30] == 0x50 &&      
			sendpacket[31] == 0x49 &&      
			sendpacket[32] == 0x50 &&      
			sendpacket[33] == 0x45 &&      
			sendpacket[34] == 0x41 &&      
			sendpacket[35] == 0x42 ){ 

			return UDP_PTPIPE_TYPE;
	
		} else if(sendpacket[28] == 0x44 &&  /* DRVSTATS: Signature */
			sendpacket[29] == 0x52 &&      
      			sendpacket[30] == 0x56 &&      
      			sendpacket[31] == 0x53 &&      
      			sendpacket[32] == 0x54 &&      
      			sendpacket[33] == 0x41 &&      
      			sendpacket[34] == 0x54 &&      
      			sendpacket[35] == 0x53 ){
	
			return UDP_DRVSTATS_TYPE;

		} else
			return UDP_INVALID_TYPE;

	} else
		return UDP_INVALID_TYPE;

}

/*============================================================================
 * Check to see if the packet to be transmitted contains a broadcast or
 * multicast source IP address.
 */

static int chk_bcast_mcast_addr(sdla_t *card, netdevice_t* dev,
				struct sk_buff *skb)
{
	u32 src_ip_addr;
        u32 broadcast_ip_addr = 0;
        struct in_device *in_dev;
        /* read the IP source address from the outgoing packet */
        src_ip_addr = *(u32 *)(skb->data + 12);

	/* read the IP broadcast address for the device */
        in_dev = dev->ip_ptr;
        if(in_dev != NULL) {
                struct in_ifaddr *ifa= in_dev->ifa_list;
                if(ifa != NULL)
                        broadcast_ip_addr = ifa->ifa_broadcast;
                else
                        return 0;
        }
 
        /* check if the IP Source Address is a Broadcast address */
        if((dev->flags & IFF_BROADCAST) && (src_ip_addr == broadcast_ip_addr)) {
                printk(KERN_INFO "%s: Broadcast Source Address silently discarded\n",
				card->devname);
                return 1;
        } 

        /* check if the IP Source Address is a Multicast address */
        if((ntohl(src_ip_addr) >= 0xE0000001) &&
		(ntohl(src_ip_addr) <= 0xFFFFFFFE)) {
                printk(KERN_INFO "%s: Multicast Source Address silently discarded\n",
				card->devname);
                return 1;
        }

        return 0;
}

void s508_lock (sdla_t *card, unsigned long *smp_flags)
{
#if defined(CONFIG_SMP) || defined(LINUX_2_4)
	spin_lock_irqsave(&card->wandev.lock, *smp_flags);
#else
	disable_irq(card->hw.irq);
#endif                                                                     
}

void s508_unlock (sdla_t *card, unsigned long *smp_flags)
{
#if defined(CONFIG_SMP) || defined(LINUX_2_4)
        spin_unlock_irqrestore(&card->wandev.lock, *smp_flags);
#else
	enable_irq(card->hw.irq);
#endif           
}

static int read_connection_info (sdla_t *card)
{
	ppp_mbox_t *mb = card->mbox;
	netdevice_t *dev = card->wandev.dev;
	ppp_private_area_t *ppp_priv_area = dev->priv;
	ppp508_connect_info_t *ppp508_connect_info;
	int err;

	memset(&mb->cmd, 0, sizeof(ppp_cmd_t));
	mb->cmd.length  = 0;
	mb->cmd.command = PPP_GET_CONNECTION_INFO;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	if (err != CMD_OK) { 
		ppp_error(card, err, mb);
		ppp_priv_area->ip_remote = 0;
		ppp_priv_area->ip_local = 0;
	}
	else {
		ppp508_connect_info = (ppp508_connect_info_t *)mb->data;
		ppp_priv_area->ip_remote = ppp508_connect_info->ip_remote;
		ppp_priv_area->ip_local = ppp508_connect_info->ip_local;

		NEX_PRINTK(KERN_INFO "READ CONNECTION GOT IP ADDRESS %x, %x\n",
				ppp_priv_area->ip_remote,
				ppp_priv_area->ip_local);
	}

	return err;
}

/*===============================================================================
 * config_ppp
 *
 *	Configure the ppp protocol and enable communications.		
 *
 *   	The if_open function binds this function to the poll routine.
 *      Therefore, this function will run every time the ppp interface
 *      is brought up.  
 *      
 *	If the communications are not enabled, proceed to configure
 *      the card and enable communications.
 *
 *      If the communications are enabled, it means that the interface
 *      was shutdown by ether the user or driver. In this case, we 
 *      have to check that the IP addresses have not changed.  If
 *      the IP addresses changed, we have to reconfigure the firmware
 *      and update the changed IP addresses.  Otherwise, just exit.
 */
static int config_ppp (sdla_t *card)
{

	netdevice_t *dev = card->wandev.dev;
	ppp_flags_t *flags = card->flags;
	ppp_private_area_t *ppp_priv_area = dev->priv;

	if (card->u.p.comm_enabled){

		if (ppp_priv_area->ip_local_tmp != ppp_priv_area->ip_local ||
		    ppp_priv_area->ip_remote_tmp != ppp_priv_area->ip_remote){
			
			/* The IP addersses have changed, we must
                         * stop the communications and reconfigure
                         * the card. Reason: the firmware must know
                         * the local and remote IP addresses. */
			disable_comm(card);
			ppp_priv_area->ppp_state = WAN_DISCONNECTED;
			wanpipe_set_state(card, WAN_DISCONNECTED);
			printk(KERN_INFO 
				"%s: IP addresses changed!\n",
					card->devname);
			printk(KERN_INFO "%s: Restarting communications ...\n",
					card->devname);
		}else{ 
			/* IP addresses are the same and the link is up, 
                         * we dont have to do anything here. Therefore, exit */
			return 0;
		}
	}

	flags->imask = 0;
	flags->iflag = 0;

	if (IS_TE1(card->wandev.te_cfg)) {
		printk(KERN_INFO "%s: Configuring onboard %s CSU/DSU\n",
			card->devname, 
			(card->wandev.te_cfg.media==WANOPT_MEDIA_T1)?"T1":"E1");
		if (sdla_te_config(card)){
			printk(KERN_INFO "%s: Failed %s configuratoin!\n",
					card->devname,
					(card->wandev.te_cfg.media==WANOPT_MEDIA_T1)?"T1":"E1");
			return 0;
		}
		
	}else if (IS_56K(card->wandev.te_cfg)) {
		printk(KERN_INFO "%s: Configuring 56K onboard CSU/DSU\n",
			card->devname);

		if(sdla_56k_config(card)) {
			printk (KERN_INFO "%s: Failed 56K configuration!\n",
				card->devname);
			return 0;
		}
	}

	
	/* Record the new IP addreses */
	ppp_priv_area->ip_local = ppp_priv_area->ip_local_tmp;
	ppp_priv_area->ip_remote = ppp_priv_area->ip_remote_tmp;

	if (config508(dev, card)){
		printk(KERN_INFO "%s: Failed to configure PPP device\n",
			card->devname);
		return 0;
	}

	if (ppp_set_intr_mode(card, PPP_INTR_RXRDY|
			    		PPP_INTR_TXRDY|
				    	PPP_INTR_MODEM|
				    	PPP_INTR_DISC |
				    	PPP_INTR_OPEN |
				    	PPP_INTR_DROP_DTR |
					PPP_INTR_TIMER)) {

		printk(KERN_INFO "%s: Failed to configure board interrupts !\n", 
			card->devname);
		return 0;
	}

        /* Turn off the transmit and timer interrupt */
	flags->imask &= ~(PPP_INTR_RXRDY|
			    		PPP_INTR_TXRDY|
				    	PPP_INTR_MODEM|
				    	PPP_INTR_DISC |
				    	PPP_INTR_OPEN |
				    	PPP_INTR_DROP_DTR |
					PPP_INTR_TIMER);

	/* If you are not the authenticator and any one of the protocol is 
	 * enabled then we call the set_out_bound_authentication.
	 */
	if ( !card->u.p.authenticator  && (ppp_priv_area->pap || ppp_priv_area->chap)) {
		if ( ppp_set_outbnd_auth(card, ppp_priv_area) ){
			printk(KERN_INFO "%s: Outbound authentication failed !\n",
				card->devname);
			return 0;
		}
	} 
	
	/* If you are the authenticator and any one of the protocol is enabled
	 * then we call the set_in_bound_authentication.
	 */
	if (card->u.p.authenticator && (ppp_priv_area->pap || ppp_priv_area->chap)){
		if (ppp_set_inbnd_auth(card, ppp_priv_area)){
			printk(KERN_INFO "%s: Inbound authentication failed !\n",
				card->devname);	
			return 0;
		}
	}


	/* Turn on all masked interrupts except for
	 * tx and timer interrupts */
	flags->imask |= (PPP_INTR_RXRDY|
				    	PPP_INTR_MODEM|
				    	PPP_INTR_DISC |
				    	PPP_INTR_OPEN |
				    	PPP_INTR_DROP_DTR);

	/* If we fail to enable communications here it's OK,
	 * since the DTR timer will cause a disconnected, which
	 * will retrigger communication in timer_intr() */
	if (ppp_comm_enable(card) == CMD_OK) {
		ppp_priv_area->ppp_state = WAN_CONNECTING;
		wanpipe_set_state(card, WAN_CONNECTING);
		init_ppp_tx_rx_buff(card);
	}

	/* Manually poll the 56K CSU/DSU to get the status */
	if (IS_56K(card->wandev.te_cfg)) {
		/* 56K Update CSU/DSU alarms */
		sdla_56k_alarm(card, 1);
	}

	return 0; 
}

/*============================================================
 * ppp_poll
 *	
 * Rationale:
 * 	We cannot manipulate the routing tables, or
 *      ip addresses withing the interrupt. Therefore
 *      we must perform such actons outside an interrupt 
 *      at a later time. 
 *
 * Description:	
 *	PPP polling routine, responsible for 
 *     	shutting down interfaces upon disconnect
 *     	and adding/removing routes. 
 *      
 * Usage:        
 * 	This function is executed for each ppp  
 * 	interface through a tq_schedule bottom half.
 *      
 *      trigger_ppp_poll() function is used to kick
 *      the ppp_poll routine.  
 */
static void ppp_poll (netdevice_t *dev)
{
	ppp_private_area_t *ppp_priv_area; 	
	sdla_t *card;
	u8 check_gateway=0;
	ppp_flags_t *flags;

	if (!dev || (ppp_priv_area = dev->priv) == NULL)
		return;

	card = ppp_priv_area->card;
	flags = card->flags;

	/* Shutdown is in progress, stop what you are 
	 * doing and get out */
	if (test_bit(PERI_CRIT,&card->wandev.critical)){
		clear_bit(POLL_CRIT,&card->wandev.critical);
		return;
	}

	/* if_open() function has triggered the polling routine
	 * to determine the configured IP addresses.  Once the
	 * addresses are found, trigger the chdlc configuration */
	if (test_bit(0,&ppp_priv_area->config_ppp)){

		ppp_priv_area->ip_local_tmp  = get_ip_address(dev,WAN_LOCAL_IP);
		ppp_priv_area->ip_remote_tmp = get_ip_address(dev,WAN_POINTOPOINT_IP);

		if (ppp_priv_area->ip_local_tmp == ppp_priv_area->ip_remote_tmp && 
	            card->u.p.ip_mode == WANOPT_PPP_HOST){
			
			if (++ppp_priv_area->ip_error > MAX_IP_ERRORS){
				printk(KERN_INFO "\n");
				printk(KERN_INFO "%s: --- WARNING ---\n",
						card->devname);
				printk(KERN_INFO "%s: The local IP address is the same as the\n",
						card->devname);
				printk(KERN_INFO "%s: Point-to-Point IP address.\n",
						card->devname);
				printk(KERN_INFO "%s: --- WARNING ---\n\n",
						card->devname);
			}else{
				clear_bit(POLL_CRIT,&card->wandev.critical);
				ppp_priv_area->poll_delay_timer.expires = jiffies+HZ;
				add_timer(&ppp_priv_area->poll_delay_timer);
				return;
			}
		}

		ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_CONFIG;
		flags->imask |= PPP_INTR_TIMER;	
		ppp_priv_area->ip_error=0;	
		
		clear_bit(0,&ppp_priv_area->config_ppp);
		clear_bit(POLL_CRIT,&card->wandev.critical);
		return;
	}

	/* Dynamic interface implementation, as well as dynamic
	 * routing.  */
	
	switch (card->wandev.state) {
	
	case WAN_DISCONNECTED:

		/* If the dynamic interface configuration is on, and interface 
		 * is up, then bring down the netowrk interface */

		if (test_bit(DYN_OPT_ON,&ppp_priv_area->interface_down) &&
		    !test_bit(DEV_DOWN,&ppp_priv_area->interface_down)	&&	
		    card->wandev.dev->flags & IFF_UP){	

			printk(KERN_INFO "%s: Interface %s down.\n",
				card->devname,card->wandev.dev->name);
			change_dev_flags(card->wandev.dev,
					(card->wandev.dev->flags&~IFF_UP));
			set_bit(DEV_DOWN,&ppp_priv_area->interface_down);
		}else{
			/* We need to check if the local IP address is
               	   	 * zero. If it is, we shouldn't try to remove it.
                 	 * For some reason the kernel crashes badly if 
                 	 * we try to remove the route twice */

			if (card->wandev.dev->flags & IFF_UP && 
		    	    get_ip_address(card->wandev.dev,WAN_LOCAL_IP) &&
		    	    card->u.p.ip_mode == WANOPT_PPP_PEER){ 

				remove_route(card);
			}
		}
		break;

	case WAN_CONNECTED:
		
		/* In SMP machine this code can execute before the interface
		 * comes up.  In this case, we must make sure that we do not
		 * try to bring up the interface before dev_open() is finished */


		/* DEV_DOWN will be set only when we bring down the interface
		 * for the very first time. This way we know that it was us
		 * that brought the interface down */
		
		if (test_bit(DYN_OPT_ON,&ppp_priv_area->interface_down) &&
	            test_bit(DEV_DOWN,  &ppp_priv_area->interface_down) &&
 		    !(card->wandev.dev->flags & IFF_UP)){
			
			printk(KERN_INFO "%s: Interface %s up.\n",
				card->devname,card->wandev.dev->name);
			
			change_dev_flags(card->wandev.dev,(card->wandev.dev->flags|IFF_UP));
			clear_bit(DEV_DOWN,&ppp_priv_area->interface_down);
			check_gateway=1;
		}

		if ((card->u.p.ip_mode == WANOPT_PPP_PEER) && 
		    test_bit(1,&Read_connection_info)) { 
			
			process_route(card);
			clear_bit(1,&Read_connection_info);
			check_gateway=1;
		}

		if (ppp_priv_area->gateway && check_gateway)
			add_gateway(card,dev);

		break;
	}
	clear_bit(POLL_CRIT,&card->wandev.critical);
	return;
}

/*============================================================
 * trigger_ppp_poll
 *
 * Description:
 * 	Add a ppp_poll() task into a tq_scheduler bh handler
 *      for a specific interface.  This will kick
 *      the ppp_poll() routine at a later time. 
 *
 * Usage:
 * 	Interrupts use this to defer a taks to 
 *      a polling routine.
 *
 */	

static void trigger_ppp_poll (netdevice_t *dev)
{
	ppp_private_area_t *ppp_priv_area;
	if ((ppp_priv_area=dev->priv) != NULL){ 	
		
		sdla_t *card = ppp_priv_area->card;

		if (test_bit(PERI_CRIT,&card->wandev.critical)){
			return;
		}
		
		if (test_and_set_bit(POLL_CRIT,&card->wandev.critical)){
			return;
		}

#ifdef LINUX_2_4
		schedule_task(&ppp_priv_area->poll_task);
#else
		queue_task(&ppp_priv_area->poll_task, &tq_scheduler);
#endif
	}
	return;
}

static void ppp_poll_delay (unsigned long dev_ptr)
{
	netdevice_t *dev = (netdevice_t *)dev_ptr;
	trigger_ppp_poll(dev);
}

/*============================================================
 * detect_and_fix_tx_bug
 *
 * Description:
 *	On connect, if the board tx buffer ptr is not the same
 *      as the driver tx buffer ptr, we found a firmware bug.
 *      Report the bug to the above layer.  To fix the
 *      error restart communications again.
 *
 * Usage:
 *
 */	

static int detect_and_fix_tx_bug (sdla_t *card)
{
	if (((unsigned long)card->u.p.txbuf_base&0xFFF) != ((*card->u.p.txbuf_next)&0xFFF)){
		NEX_PRINTK(KERN_INFO "Major Error, Fix the bug\n");
		return 1;
	}
	return 0;
}


static int set_adapter_config (sdla_t* card)
{
	ppp_mbox_t * mb = card->mbox;
	char* data = mb->data;
	int err;

	((ADAPTER_CONFIGURATION_STRUCT *)data)->adapter_type = 
		card->hw.pci_adapter_type; 
	((ADAPTER_CONFIGURATION_STRUCT *)data)->adapter_config = 0x00; 
	((ADAPTER_CONFIGURATION_STRUCT *)data)->operating_frequency = 00; 
	mb->cmd.length = sizeof(ADAPTER_CONFIGURATION_STRUCT);
	mb->cmd.command = SET_ADAPTER_CONFIGURATION;
	err = sdla_exec(mb) ? mb->cmd.result : CMD_TIMEOUT;

	if(err != 0) {
		ppp_error(card,err,mb);
	}
	return (err);
}


/*
 * ******************************************************************
 * Proc FS function 
 */
#define PROC_CFG_FRM	"%-15s| %-12s|\n"
#define PROC_STAT_FRM	"%-15s| %-12s| %-14s|\n"
static char ppp_config_hdr[] =
	"Interface name | Device name |\n";
static char ppp_status_hdr[] =
	"Interface name | Device name | Status        |\n";

static int ppp_get_config_info(void* priv, char* buf, int cnt, int len, int offs, int* stop_cnt) 
{
	ppp_private_area_t*	ppp_priv_area = priv;
	sdla_t*			card = NULL;
	int 			size = 0;

	if (ppp_priv_area == NULL)
		return cnt;
	card = ppp_priv_area->card;

	if ((offs == 0 && cnt == 0) || (offs && offs == *stop_cnt)){
		PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, stop_cnt, &size, "%s", ppp_config_hdr));
	}

	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, stop_cnt, &size,
			    PROC_CFG_FRM, card->u.p.if_name, card->devname));
	return cnt;
}

static int ppp_get_status_info(void* priv, char* buf, int cnt, int len, int offs, int* stop_cnt)
{
	ppp_private_area_t*	ppp_priv_area = priv;
	sdla_t*			card = NULL;
	int 			size = 0;

	if (ppp_priv_area == NULL)
		return cnt;
	card = ppp_priv_area->card;

	if ((offs == 0 && cnt == 0) || (offs && offs == *stop_cnt)){
		PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, stop_cnt, &size, "%s", ppp_status_hdr));
	}

	PROC_ADD_LINE(cnt, (buf, &cnt, len, offs, stop_cnt, &size,
			    PROC_STAT_FRM, card->u.p.if_name, card->devname, STATE_DECODE(card->wandev.state)));
	return cnt;
}


#define PROC_DEV_PPP_S_FRM	"%-20s| %-14s|\n"
#define PROC_DEV_PPP_D_FRM	"%-20s| %-14d|\n"
#define PROC_DEV_SEPARATE	"=====================================\n"

#ifdef LINUX_2_4
static int ppp_get_dev_config_info(char* buf, char** start, off_t offs, int len)
#else
static int ppp_get_dev_config_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif
{
	int 		cnt = 0;
	wan_device_t*	wandev = (void*)start;
	sdla_t*		card = NULL;
	int 		size = 0;
	PROC_ADD_DECL(stop_cnt);

	if (wandev == NULL)
		return cnt;

	PROC_ADD_INIT(offs, stop_cnt);
	card = (sdla_t*)wandev->private;

	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_SEPARATE));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, 
		 "Configuration for %s device\n", wandev->name));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_SEPARATE));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_S_FRM,
		 "Interface", INT_DECODE(wandev->interface)));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_S_FRM,
		"Clocking", CLK_DECODE(wandev->clocking)));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_D_FRM,
		"BaudRate",wandev->bps));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_D_FRM,
		 "MTU", wandev->mtu));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_D_FRM,
		 "UDP Port",  wandev->udp_port));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_D_FRM,
		 "TTL", wandev->ttl));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_PPP_S_FRM,
		 "IP Mode", IP_MODE_DECODE(card->u.p.ip_mode)));
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_DEV_SEPARATE));

	PROC_ADD_RET(cnt, offs, stop_cnt);
}

static int ppp_set_dev_config(struct file *file, 
			      const char *buffer,
			      unsigned long count, 
			      void *data)
{
	int 		cnt = 0;
	wan_device_t*	wandev = (void*)data;
	sdla_t*		card = NULL; 

	if (wandev == NULL)
		return cnt;

	card = (sdla_t*)wandev->private;

	printk(KERN_INFO "%s: New device config (%s)\n",
			wandev->name, buffer);
	/* Parse string */

	return count;
}


#define PROC_IF_PPP_S_FRM	"%-30s\t%-14s\n"
#define PROC_IF_PPP_D_FRM	"%-30s\t%-14d\n"
#define PROC_IF_PPP_L_FRM	"%-30s\t%-14ld\n"
#define PROC_IF_SEPARATE	"===========================================\n"

#ifdef LINUX_2_4
static int ppp_get_if_info(char* buf, char** start, off_t offs, int len)
#else
static int ppp_get_if_info(char* buf, char** start, off_t offs, int len, int dummy)
#endif
{
	int 			cnt = 0;
	netdevice_t*		dev = (void*)start;
	ppp_private_area_t* 	ppp_priv_area = dev->priv;
	sdla_t*			card = ppp_priv_area->card;
	int 			size = 0;
	PROC_ADD_DECL(stop_cnt);

	goto get_if_info_end;
	/* Update device statistics */
	if (!offs && card->wandev.update) {
		int rslt = 0;
		rslt = card->wandev.update(&card->wandev);
		if(rslt) {
			switch (rslt) {
			case -EAGAIN:
				PROC_ADD_LINE(cnt, 
					(buf, &cnt, len, offs, &stop_cnt, &size, 
					 "Device is busy!\n"));
				break;

			default:
				PROC_ADD_LINE(cnt, 
					(buf, &cnt, len, offs, &stop_cnt, &size,
					 "Device is not configured!\n"));
				break;
			}
			goto get_if_info_end;
		}
	}

	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, PROC_IF_SEPARATE));
	
	PROC_ADD_LINE(cnt, 
		(buf, &cnt, len, offs, &stop_cnt, &size, 
		 "SNMP doesn't support for PPP protocol\n"));

get_if_info_end:
	PROC_ADD_RET(cnt, offs, stop_cnt);
}
	
	
static int ppp_set_if_info(struct file *file, 
			   const char *buffer,
			   unsigned long count, 
			   void *data)
{
	netdevice_t*		dev = (void*)data;
	ppp_private_area_t* 	ppp_priv_area = NULL;
	sdla_t*			card = NULL;

	if (dev == NULL || dev->priv == NULL)
		return count;
	ppp_priv_area = (ppp_private_area_t*)dev->priv;
	if (ppp_priv_area->card == NULL)
		return count;
	card = ppp_priv_area->card;

	printk(KERN_INFO "%s: New interface config (%s)\n",
			card->u.p.if_name, buffer);
	/* Parse string */

	return count;
}

static void ppp_handle_front_end_state(sdla_t *card)
{
	netdevice_t* dev = card->wandev.dev;
        ppp_private_area_t* ppp_priv_area = dev->priv;
	volatile ppp_flags_t *flags = card->flags;

	if (card->wandev.ignore_front_end_status == WANOPT_YES){
		return;
	}

	if (ppp_priv_area->ppp_state != WAN_CONNECTED){
		return;
	}
	
	if (card->wandev.front_end_status == FE_CONNECTED){

		if (ppp_priv_area->ppp_state == WAN_CONNECTED &&
		    card->wandev.state != WAN_CONNECTED){
		
			wanpipe_set_state(card, WAN_CONNECTED);

			ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_PPP_EVENT;
			flags->imask |= PPP_INTR_TIMER;
			trigger_ppp_poll(dev);
		}
	}else{
		wanpipe_set_state(card, WAN_DISCONNECTED);
		
		ppp_priv_area->timer_int_enabled |= TMR_INT_ENABLED_PPP_EVENT;
		flags->imask |= PPP_INTR_TIMER;
		if (card->u.p.ip_mode == WANOPT_PPP_PEER) { 
			set_bit(0,&Read_connection_info);
		}
		trigger_ppp_poll(dev);
	}
}

/****** End *****************************************************************/
