/*****************************************************************************
* ft1_config.c    X25 Test Menu Driven Application based on X25 API
*
* Author:       Nenad Corbic <ncorbic@sangoma.com>
*
* Copyright:    (c) 1999 Sangoma Technologies Inc.
*
*               This program is free software; you can redistribute it and/or
*               modify it under the terms of the GNU General Public License
*               as published by the Free Software Foundation; either version
*               2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
*
* Nov 25, 1999	Nenad Corbic	Inital Verstion
*				Supports Standard, Advanced and Auto config
*      				as well as reading CSU/DSU config.
*
*
*****************************************************************************/

/*****************************************************************************
*
* !!! WARNING !!!
*
* This program is used by 'ft1conf' script. It is not supposed to be 
* used by itself.  Please run the ft1conf script to use functions
* of this program.
*
* Usage:
*   ft1_config [{switches}] {device} ]
*
*   ex: ./ft1_config wanpipe1 <command> <options>
*
* Where:
*   {device}    WANPIPE device name in /proc/net/router directory
*
*****************************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/mman.h>

#include <linux/wanpipe.h>
#include <linux/sdla_chdlc.h>
/****** Defines *************************************************************/

#ifndef min
#define min(a,b)        (((a)<(b))?(a):(b))
#endif

/* Error exit codes */
enum ErrCodes
{
	ERR_SYSTEM = 1,         /* System error */
	ERR_SYNTAX,             /* Command line syntax error */
	ERR_LIMIT
};


/* Defaults */
#define TRY_AGAIN 	0x07

/****** Data Types **********************************************************/

/****** Function Prototypes *************************************************/

void show_error (int err);
extern  int close (int);
int advanced_config (int);
int auto_config (int);
int standard_config (int , char **);
int read_configuration (int);
int advanced_config (int);
int auto_config (int dev);
int kbdhit(int *);

/****** Global Data *********************************************************/

int dev; 

enum {REG_CONF, AUTO_CONF, ADV_CONF, READ_CONF, EXIT}; 


sdla_exec_t *exec;
ft1_exec_cmd_t *ctrl;
void *data;

/*
 * Strings & message tables.
 */
char progname[] = "ft1_config";
char wandev_dir[] = "/proc/net/wanrouter";         /* location of WAN devices */

char banner[] =
	"WANPIPE FT1 Configuration Utility"
	"(c) 1999 Sangoma Technologies Inc."
;
char helptext[] =
	"Usage:\tft1_config [{switches}] {device} \n"
	"\nWhere:"
	"\t{device}\tWANPIPE device name from /proc/net/router directory\n"
	"\t{switches}\tone of the following:\n"
	"\t\t\t-v\tverbose mode\n"
	"\t\t\t-h|?\tshow this help\n"
;

char* err_messages[] =          /* Error messages */
{
	"Invalid command line syntax",  /* ERR_SYNTAX */
	"Unknown error code",           /* ERR_LIMIT */
};


/*------------------------- MAIN PROGRAM --------------------------*/


/******************************************************************
* Main 
*
*
*
******************************************************************/

int main (int argc, char *argv[])
{
	int err=0;
	char filename[sizeof(wandev_dir) + WAN_DRVNAME_SZ + 2];

	if (argc < 3){
		printf("Prog name is %s %i\n", argv[0],WAN_DRVNAME_SZ );
		show_error(ERR_SYNTAX);
		return ERR_SYNTAX;
	}

	sprintf(filename, "%s/%s", wandev_dir, argv[1]);
	dev = open(filename, O_RDONLY);
	if (dev < 0){
		return -EINVAL;
	}
	
	exec = malloc(sizeof(sdla_exec_t));
	ctrl = malloc(sizeof(ft1_exec_cmd_t));
	data = malloc(SIZEOF_MB_DATA_BFR);
	exec->magic = WANPIPE_MAGIC;
	exec->cmd = ctrl;
	exec->data = data; 
	memset(&ctrl->command, 0, sizeof(ft1_exec_cmd_t));
	memset(data, 0, SIZEOF_MB_DATA_BFR);

	switch (atoi(argv[2])){
	
       		case REG_CONF: 
			if (argc != 9){
				printf("Illegal number of arguments\n");
				break;
			}
			if ((err=standard_config(dev,&argv[3])) == 0){
				sleep(1);
				err=read_configuration(dev);
			}
                    	break;
	      	case AUTO_CONF: 
			err=auto_config(dev);
		      	break;
	      	case ADV_CONF: 
			printf("Advanced Configurator Activated, Press ESC to Exit!\n\n");
			err=advanced_config(dev);
			break;
		case READ_CONF:
			err=read_configuration(dev);
			break;
		case EXIT:
			break;
              	default:
                     	printf("%s: Error Illegal Option\n",progname); 
                      	break;
       	}

	free(exec);             
	free(data);
	
	if (dev >= 0) close(dev);

	return err;
}


/******************************************************************
* Standard Configuration
*
*
*
******************************************************************/

int standard_config (int dev, char *argv[])
{
	
	ft1_config_t *ft1_config;
	int len;
	char buf[100];

	memset(buf,0,sizeof(buf));
	
	ft1_config = (ft1_config_t *)data;

	ft1_config->framing_mode   = atoi(argv[0]);
	ft1_config->encoding_mode  = atoi(argv[1]);
	ft1_config->line_build_out = atoi(argv[2]);  
	ft1_config->channel_base   = atoi(argv[3]);
	ft1_config->baud_rate_kbps = atoi(argv[4]);
	ft1_config->clock_mode     = atoi(argv[5]);

	ctrl->command = SET_FT1_CONFIGURATION;
	ctrl->buffer_length = sizeof(ft1_config_t);

	len = ctrl->buffer_length/2;

	if ((dev < 0) || (ioctl(dev, WANPIPE_EXEC, exec ) < 0)){
	       	return -EIO;
        }                               	

	return ctrl->return_code;
}

/******************************************************************
* Read Configuration 
*
*
*
******************************************************************/

int read_configuration (int dev)
{
	int i;

	ctrl->command = READ_FT1_CONFIGURATION;
	ctrl->buffer_length = 0;

	memset(data,0,SIZEOF_MB_DATA_BFR);

	if ((dev < 0) || (ioctl(dev, WANPIPE_EXEC, exec ) < 0)){
	       	return -EIO;
        }                               	

	/* Check for errors */
	if (ctrl->return_code){
		printf("Read Rc is %i\n",ctrl->return_code);
		return ctrl->return_code;
	}
	for (i=0;i<(ctrl->buffer_length/2);i++){
		printf("%i\n",*((unsigned short*)data + i));
	}
	return COMMAND_OK;
}

/******************************************************************
* Advanced Configuration
*
*
*
******************************************************************/

int advanced_config (int dev)
{

	int key_int, i;
	char key_ch;

	memset(data,0,SIZEOF_MB_DATA_BFR);

	for (;;){
		if(kbdhit(&key_int)){
			key_ch = key_int;

			if (key_ch == 0x1b){
				break;
			}

			ctrl->command = TRANSMIT_ASYNC_DATA_TO_FT1;
			ctrl->buffer_length = sizeof(key_ch);
			*((unsigned char *)data) = key_ch;

			if ((dev < 0) || (ioctl(dev, WANPIPE_EXEC, exec ) < 0)){
				printf("Write ioctl error\n");
	       			return -EIO;
        		}
     
			if (ctrl->return_code != 0){
			        printf("Write return code %i\n",ctrl->return_code);
				return ctrl->return_code;
			}
        	}else{       
			ctrl->command = RECEIVE_ASYNC_DATA_FROM_FT1;
			ctrl->buffer_length = 0;

			if ((dev < 0) || (ioctl(dev, WANPIPE_EXEC, exec ) < 0)){
				printf("Read ioctl error\n");
       				return -EIO;
       			}
		
			if (ctrl->return_code != COMMAND_OK && 
			    ctrl->return_code != TRY_AGAIN){
				printf("Read return code %i\n",ctrl->return_code);
				return ctrl->return_code;
			}
			
			if (ctrl->buffer_length){
				for (i=0;i<ctrl->buffer_length;i++){
					putchar(*((unsigned char *)data + i));
					fflush(stdout);
				}
			}
		}
	
	}
	return 0;
}


/******************************************************************
* Automatic Configuration
*
*
*
******************************************************************/

int auto_config (int dev)
{

	volatile ft1_config_t *ft1_config;
	int i,key;
	char buf[100];
	unsigned short current_FT1_baud_rate=0;

	memset(buf,0,sizeof(buf));
	
	ft1_config = (ft1_config_t *)data;

	ft1_config->framing_mode   = ESF_FRAMING;
	ft1_config->encoding_mode  = B8ZS_ENCODING;		
	ft1_config->line_build_out = LN_BLD_CSU_0dB_DSX1_0_to_133;  
	ft1_config->channel_base   = 0x01;	
	ft1_config->baud_rate_kbps = BAUD_RATE_FT1_AUTO_CONFIG;
	ft1_config->clock_mode     = CLOCK_MODE_NORMAL;	

	ctrl->command = SET_FT1_CONFIGURATION;
	ctrl->buffer_length = sizeof(ft1_config_t);

	if ((dev < 0) || (ioctl(dev, WANPIPE_EXEC, exec ) < 0)){
	       	return -EIO;
        }                               	

	if (ctrl->return_code){
		printf("Automatic, Write Command Failed\n");
		fflush(stdout);
		return ctrl->return_code;
	}

	if (ft1_config->baud_rate_kbps == 0xFFFF){
		printf("\nPerforming FT1 auto configuration ... Press any key to abort\n");
		fflush(stdout);	

		do{
			ctrl->command = READ_FT1_CONFIGURATION;
			ctrl->buffer_length = 0;

			if ((dev < 0) || (ioctl(dev, WANPIPE_EXEC, exec ) < 0)){
	       			return -EIO;
        		} 			

			switch(ctrl->return_code){

			case COMMAND_OK:
				printf("The FT1 configuration has been successfully completed. Baud rate is %u Kbps\n", ft1_config->baud_rate_kbps);			
				for (i=0;i<(ctrl->buffer_length/2);i++){
					printf("%i\n",*((unsigned short*)data + i));
				}
				return COMMAND_OK;
			
			case AUTO_FT1_CONFIG_NOT_COMPLETE:
				if (ft1_config->baud_rate_kbps != current_FT1_baud_rate){
					current_FT1_baud_rate = ft1_config->baud_rate_kbps;
					printf("Testing line for baud rate %i Kbps\n",
						current_FT1_baud_rate);
					fflush(stdout);
				}
				break;

			case AUTO_FT1_CFG_FAIL_OP_MODE:
				return ctrl->return_code;
	
			case AUTO_FT1_CFG_FAIL_INVALID_LINE:
				return ctrl->return_code;

			default:
				printf("Unknown return code 0x%02X\n", ctrl->return_code);
				fflush(stdout);
				return ctrl->return_code;
			}
 
		}while (!kbdhit(&key));


	}
	return ctrl->return_code;

}

/*******************************************************************
* Show error message.
*
*
*
*******************************************************************/
void show_error (int err)
{
	if (err == ERR_SYSTEM) fprintf(stderr, "%s: SYSTEM ERROR %d: %s!\n",
		progname, errno, strerror(errno))
	;
	else fprintf(stderr, "%s: %s!\n", progname,
		err_messages[min(err, ERR_LIMIT) - 2])
	;
}

/****** End *****************************************************************/
