/*************************************************************
 * File: mon/stty.c
 * Purpose: Part of core Monitor
 * 	This module contains the stty command.
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	970707	Fixed prob with not being able to set the baud rate
 *	970805	Fixed prob of incorrect con baud after boot w nvram overide
 *	980618	Changed baud list to use info from driver.
 */

#include <stdio.h>
#include <termio.h>
#include <string.h>
#include <mon.h>

struct brate {
	char *str;
	int value;
	};

struct brate btable[] = {
	{"0",B0},
	{"50",B50},
	{"75",B75},
	{"110",B110},
	{"134",B134},
	{"200",B200},
	{"150",B150},
	{"300",B300},
	{"600",B600},
	{"1200",B1200},
	{"1800",B1800},
	{"2400",B2400},
	{"4800",B4800},
	{"9600",B9600},
	{"19200",B19200},
	{"38400",B38400},
	{"57600",B57600},
	{"76800",B76800},
	{"115200",B115200},
	{"153600",B153600},
	{"230400",B230400},
	{"307200",B307200},
	{"460800",B460800},
	{"921600",B921600},
	{0}};

Optdesc stty_opts[] = {
	{"[opt][<baud>][<term>]","Set/display terminal options"},
	{"-v","list possible baud rates and terminal types"},
	{"-a","list all settings"},
	{"<baud>","set baud rate"},
	{"<term>","set terminal type"},
	{"sane","set sane settings"},
	{"ixany","allow any char to restart output"},
	{"-ixany","allow only <start> to restart output"},
	{"ixoff","enable tandem mode"},
	{"-ixoff","disable tandem mode"},
	{"echo","enable echo"},
	{"-echo","disable echo"},
	{0}};

/*************************************************************
*  stty(ac,av)
*	the 'stty' command 
*/
stty(ac,av)
int ac;
char *av[];
{
int fd,i,flag,aflag,vflag,j;
struct termio tbuf;
char buf[16];
Ulong bauds;

fd = STDIN;
ioctl_getsaved(fd,&tbuf);
flag = aflag = vflag = 0;
if (ioctl(fd,BAUDRATES,&bauds) == -1) {
	printf("warning: unable to get valid baudrates.\n");
	bauds = 0;;
	}
for (i=1;i<ac;i++) {
	if (av[i][0] == '-') { /* option */
		if (strequ(av[i],"-ixoff")) {
			flag = 1;
			tbuf.c_iflag &= ~IXOFF;
			}
		else if (strequ(av[i],"-ixany")) {
			flag = 1;
			tbuf.c_iflag &= ~IXANY;
			}
		else if (strequ(av[i],"-ixon")) {
			flag = 1;
			tbuf.c_iflag &= ~IXON;
			}
		else if (strequ(av[i],"-echo")) {
			flag = 1;
			noecho = 1;
			}
		else for (j=1;av[i][j] != 0;j++) {
			if (av[i][j] == 'a') aflag = 1;
			else if (av[i][j] == 'v') vflag = 1;
			}
		}
	else if (isdigit(av[i][0])) { /* set baud rate */
		if (bauds==0) {
			printf("baudrate is fixed on this device.\n");
			continue;
			}
		flag = 1;
		for (j=0;btable[j].str != 0;j++) {
			if (!(bauds&(1<<j))) continue; 
			if (strequ(av[i],btable[j].str)) break;
			}
		if (btable[j].str == 0) {
			printf("%s: unsupported baud rate\n",av[i]);
			continue;
			}
		tbuf.c_cflag &= ~CBAUD;
		tbuf.c_cflag |= btable[j].value;
		}
	else if (strequ(av[i],"ixoff")) {
		flag = 1;
		tbuf.c_iflag |= IXOFF;
		}
	else if (strequ(av[i],"ixon")) {
		flag = 1;
		tbuf.c_iflag |= IXON;
		}
	else if (strequ(av[i],"ixany")) {
		flag = 1;
		tbuf.c_iflag |= IXANY;
		}
	else if (strequ(av[i],"echo")) {
		flag = 1;
		noecho = 0;
		}
	else if (strequ(av[i],"sane")) {
		flag = 1;
		ioctl(fd,SETSANE);
		}
	else if (strpat(av[i],"tty?")) { /* select device */
		if (strequ(av[i],"tty0")) continue; /* 970805 */
		fd = open(av[i],0);
		if (fd == -1) {
			printf("can't open %s\n",av[i]);
			return;
			}
		if (ioctl(fd,TCGETA,&tbuf) == -1) {
			printf("stty: ioctl TCGETA failure\n");
			if (fd != STDIN) close(fd);
			return;
			}
		if (ioctl(fd,BAUDRATES,&bauds) == -1) {
			printf("warning: unable to get valid baudrates.\n");
			bauds = 0;;
			}
		}
	else { /* set term type */
		flag = 1;
		if (ioctl(fd,SETTERM,av[i]) == -1) {
			printf("term type %s not found\n",av[i]);
			continue;
			}
		}
	}

if (vflag) {
	if (bauds) {
		printf("Baud rates:\n\t");
		for (i=0;btable[i].str != 0;i++) {
			if (!(bauds&(1<<i))) continue; 
			printf("%6s ",btable[i].str);
			if (i == 8) printf("\n\t");
			}
		printf("\n");
		}
	printf("Terminal types:\n");
	for (i=0;(i = ioctl(i,TERMTYPE,buf)) != -1;) {
		if ((i%6) == 0) printf("\n");
		printf("%10s ",buf);
		}
	printf("\n");
	}
	
if (flag) {
	if (fd == STDIN) ioctl_putsaved(fd,&tbuf);
	if (ioctl(fd,TCSETAF,&tbuf) == -1) {
		fprintf(stderr,"stty: ioctl TCSETAF failure\n");
		}
	if (fd != STDIN) close(fd);
	return;
	}

flag = 0;
if (ioctl(fd,GETTERM,buf) != -1) {
	printf("term=%s ",buf);
	flag = 1;
	}
ioctl(fd,TCGETA,&tbuf);
for (i=0;btable[i].str != 0;i++) {
	if (tbuf.c_cflag == btable[i].value) break;
	}
if (bauds && btable[i].str != 0) {
	printf("baud=%s",btable[i].str);
	flag = 1;
	}
if (flag) printf("\n");

if (fd != STDIN) close(fd);
if (!aflag) return;

printf("%sistrip ",((tbuf.c_iflag&ISTRIP)?"":"-")); 
printf("%sixon ",((tbuf.c_iflag&IXON)?"":"-")); 
printf("%sixany ",((tbuf.c_iflag&IXANY)?"":"-"));
printf("%sixoff ",((tbuf.c_iflag&IXOFF)?"":"-"));
printf("%sicanon ",((tbuf.c_lflag&ICANON)?"":"-"));
printf("%secho ",((tbuf.c_lflag&ECHO)?"":"-"));
printf("%sechoe ",((tbuf.c_lflag&ECHOE)?"":"-"));
printf("%sicrnl ",((tbuf.c_iflag&ICRNL)?"":"-")); 
printf("%sonlcr ",((tbuf.c_oflag&ONLCR)?"":"-"));
printf("\n");

printf("erase=%s ",cc2str(buf,tbuf.c_cc[VERASE]));
printf("stop=%s ",cc2str(buf,tbuf.c_cc[V_STOP]));
printf("start=%s ",cc2str(buf,tbuf.c_cc[V_START]));
printf("eol=%s ",cc2str(buf,tbuf.c_cc[VEOL]));
printf("eol2=%s ",cc2str(buf,tbuf.c_cc[VEOL2]));
printf("vintr=%s ",cc2str(buf,tbuf.c_cc[VINTR]));
printf("\n");
}

/*************************************************************
*  getbaudrate(p)
*	convert string to baudrate
*/
getbaudrate(fd,p)
int fd;
char *p;
{
int j;
Ulong bauds;

if (ioctl(fd,BAUDRATES,&bauds) == -1) {
	printf("warning: unable to get valid baudrates.\n");
	bauds = 0;;
	}
for (j=0;btable[j].str != 0;j++) {
	if (!(bauds&(1<<j))) continue; 
	if (strequ(p,btable[j].str)) return(btable[j].value);
	}
return(0);
}

/*************************************************************
*  getstty(dst,dev)
*	get stty settings for specified device.
*	This is used for nvram settings.
*/
getstty(dst,dev)
char *dst;
int dev;
{
int fd,i;
char tmp[20];
struct termio tbuf;
Ulong bauds;

sprintf(tmp,"/dev/tty%d",dev);
fd = open(tmp,0);
if (fd == EOF) return(1);
ioctl(fd,TCGETA,&tbuf);
ioctl(fd,GETTERM,tmp);
close(fd);
if (ioctl(fd,BAUDRATES,&bauds) == -1) {
	printf("warning: unable to get valid baudrates.\n");
	bauds = 0;;
	}
for (i=0;btable[i].str;i++) {
	if (tbuf.c_cflag == btable[i].value) break;
	}
if (btable[i].str == 0) return(1);
sprintf(dst,"stty tty%d %s %s %sixany %sixoff\n",dev,
	(bauds)?btable[i].str:"",tmp,
	((tbuf.c_iflag&IXANY)?"":"-"), ((tbuf.c_iflag&IXOFF)?"":"-"));
return(0);
}

