/*************************************************************
 * File: mon/go.c
 * Purpose: code for execution control
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970216	Changed type name Cmd to CmdRec.
 *	970227	setTrigger: Added cleanup code on setbp error.
 *	970331	Used BPTYPE_ codes for setbp_target
 *	970822	Added clrhndlrs() to go.
 *	980802	Added count=0 to constReduc().
 *	981023	cont() updated to support -w for imon95.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <mon.h>

#define PREGS	/* enable pseudo regs */

char clientcmd[LINESZ];
extern int clkdat;
int bptTmpId = -1;
Bps Bpt[MAX_BPT];       /* user break points                    */
int bptCount;

#define PCHISTSZ  200
unsigned long pchist_d[PCHISTSZ+1];
int pchist_ip,pchist_op;

WhenRec *when_list,*when_ptr;
char *oprs[] = {"&&","||","==","!=",">=","<=",">","<",
		"+","-","*","/","&","|",0};


Optdesc g_opts[] = {
	{"[-st] [-p progname ] [adr [bptadr]] [-c args]","start execution (go)"},
	{"-p","specify program name progname in argv[0]"},
	{"-s","don't set client sp"},
	{"-t","time execution"},
	{"<adr>","start address"},
	{"<bptadr>","temporary breakpoint"},
	{"-c <args>","args to be passed to client"},
	{0}};

Optdesc t_opts[] = {
	{"[-v] [cnt]","trace (single-step)"},
	{"-v","verbose, list each step"},
	{"<cnt>","execute <cnt> instructions"},
	{0}};

Optdesc to_opts[] = {
	{"[-v] [cnt]","trace over (single-step)"},
	{"-v","verbose, list each step"},
	{"<cnt>","execute <cnt> instructions"},
	{0}};

Optdesc b_opts[] = {
	{"[[-s cmdstr]brkadr]..","set breakpoint at address"},
	{"-s <cmdstr>","execute command string when bpt is encountered"},
	{"<brkadr>","stop when PC is <brkadr>"},
	{"-d","display detailed break info"},
	{"-T","TinyRISC address"},
	{0}};

Optdesc w_opts[] = {
	{"<cond> <action>","complex breakpoint"},
	{"<cond>","expression defining condition"},
	{"<action>","a command string defining the action"},
	{0}};

Optdesc db_opts[] = {
	{"[numb|*]..","delete breakpoint"},
	{0}};

/*************************************************************
*  go(ac,av,info), the 'g' command
*/
go(ac,av,info)
int ac;
char *av[];
CmdRec *info;
{
Ulong adr;
int i,j,n,flags,tbpt;
char *p,tmp[16];

n = flags = 0;
bptTmpId = -1;
strcpy(clientcmd,av[0]);
strcat(clientcmd," ");

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

for (i=1;i<ac;i++) {
	if (av[i][0] == '-') {
		for (j=1;av[i][j] != 0;j++) {
			if (av[i][j] == 's') flags |= T_SP;
			else if (av[i][j] == 'p') flags |= T_PN;
			else if (av[i][j] == 't') clientcmd[0] = 't';
			else if (av[i][j] == 'c') {
				for (i++;i<ac;i++) {
					strcat(clientcmd,av[i]);
					strcat(clientcmd," ");
					}
				break;
				}
			else {
				printf("%c: unknown option\n",av[i][j]);
				printf("usage: %s %s\n",info->name,info->opts);
				return(1);
				}
			}
		}
	else {
		if (flags & T_PN) {					/* get program name */
		  strcpy(clientcmd, av[i]);
		  strcat(clientcmd, " ");
		  flags &= ~T_PN;
		}
		else if (n == 0) {
			n++;
			if (av[i][0] != '.') {
			  if(!get_rsa(&adr,av[i])) return(1);
			  putPc(adr);
			}
		}
		else if (n == 1) {
			n++;
			if(!get_rsa(&adr,av[i])) return(1);
			tbpt = setbp_target(-1,BPTYPE_ITMP,adr,0,0);
			if (tbpt < 0) {
				printf("error: unable to set bpt\n");
				return(1);
				}
			tbpt &= 0xff; /* 970520 mask out warnings */
			}
		else {
			if (tbpt != -1) clrbp_target(tbpt);
			printf("Too many arguments.\n");
			printf("usage: %s %s\n",info->name,info->opts);
			return(1);
			}
		}
	}
clrhndlrs(); /* 970822 */
swlst(2); /* switch to clilst */
run_target(0,flags,clientcmd);
}

Optdesc c_opts[] = {
	{"[bptadr]","continue execution"},
	{"bptadr","a temporary breakpoint"},
	{0}};

/*************************************************************
*  cont(ac,av,info) 
*	The continue command
*	981023 Updated to support -w for imon95.
*/
cont(ac,av,info)
int ac;
char *av[];
CmdRec *info;
{
Ulong adr;
char *p,tmp[16];
int i,n,tbpt,flags;

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

n = flags = 0;
for (i=1;i<ac;i++) {
	if (n==0) {
		if(!get_rsa(&adr,av[i])) return(1);
		n++;
		}
	else {
		printf("Too many arguments.\n");
		printf("usage: %s %s\n",info->name,info->opts);
		return(1);
		}
	}


bptTmpId = -1;
if (n == 1) {
	tbpt = setbp_target(-1,BPTYPE_ITMP,adr,0,0);
	if (tbpt < 0) {
		printf("error: unable to set bpt\n");
		return(1);
		}
	tbpt &= 0xff; /* 970520 mask out warnings */
	}
swlst(2); /* switch to clilst */
run_target(1,flags);
}


/*************************************************************
*  setTrigger(char *cond,char *action)
*	This function sets up a trigger event.
*	It converts cond from a string to an RPN expression.
*	It does this by calling parseCond(). parseCond() is
*	called twice, once to set the when_ptr which indicates
*	the amount of storage required, and once to generate the actual
*	expression. The expression is placed in the array pointed
*	to be when_list. when_list and when_ptr are then saved in the
*	cond and csz members of a Bps structure. The string that
*	describes the condition is saved in the member cstr.
*	The action string (if specified) is placed in the member
*	cmdstr. The function setbp_target() is called in order to
*	set the actual breakpoint. Only this target dependent routine
*	'knows' how this might best be achieved for a given set of
*	hardware.
*	970227	Added cleanup code on setbp error.
*/
setTrigger(cond,action)
char *cond,*action;
{
char *p;
int j,r;
Ulong v;

/* find an empty slot */
for(j=0;j < MAX_BPT && Bpt[j].cond;j++) ;
if(j >= MAX_BPT) {
	printf("too many breakpoints\n");
	return(-1);
	}

when_list = when_ptr = 0;
p = strdup(cond);
parseCond(p);
strcpy(p,cond);
when_list = (WhenRec *)malloc((unsigned int)when_ptr);
when_ptr = when_list;
parseCond(p);
free(p);
constReduc();

Bpt[j].cond = when_list;
Bpt[j].csz = when_ptr;
Bpt[j].cstr = strdup(cond);
if (action) Bpt[j].cmdstr = strdup(action);
else Bpt[j].cmdstr = 0;
bptCount++;
if ((v=pcBpt(Bpt[j].cond,Bpt[j].csz)) != -1) {
	/*  @pc CONST ==  				bpc   */
	/*  @pc CONST == ......... && 			bpc   */
	r = setbp_target(j,BPTYPE_PC,v,0,0); /* instr */
	}
else if ((v=singleDref(Bpt[j].cond,Bpt[j].csz)) != -1) {
	/*	ADDR CONST OPR				bda */
	/*	ADDR CONST OPR	........... &&		bda */
	r = setbp_target(j,0x20|BPTYPE_DATA,v,0,0); /* data */
	/* 0x20 is brk on write only */
	}
else {
	/*  ...................... ||			nonrt */
	r = setbp_target(j,BPTYPE_NONRT,0,0,0); /* nonrt execution */
	}
if (r >= 0) return(j); /* all ok */

/* 970227 error case */
Bpt[j].cond = 0;
free(Bpt[j].cstr);
if (Bpt[j].cmdstr) free(Bpt[j].cmdstr);
if (when_list) free(when_list);
return(-1);
}

#if 1
/*************************************************************
*/
setHwdbpt(atype,addr,mask)
int atype;
Ulong addr,mask;
{
char *p,buf[100];
int j;
char *cmd = "(@cause&0x3c)==0x1c"; /* look for a DBE exception */

/* find an empty slot */
for(j=0;j < MAX_BPT && Bpt[j].cond;j++) ;
if(j >= MAX_BPT) {
	printf("too many breakpoints\n");
	return(-1);
	}

when_list = when_ptr = 0;
p = strdup(cmd);
parseCond(p);
strcpy(p,cmd);
when_list = (WhenRec *)malloc((unsigned int)when_ptr);
when_ptr = when_list;
parseCond(p);
free(p);
constReduc();

Bpt[j].cond = when_list;
Bpt[j].csz = when_ptr;
sprintf(buf,"ab -%s%s %08x %08x", (atype&1)?"r":"", (atype&2)?"w":"",
	addr,mask);
Bpt[j].cstr = strdup(buf);
Bpt[j].cmdstr = strdup("stop");
bptCount++;
atype |= 8; /* indicate that addr2 is a mask value */
atype <<= 4;
if (setbp_target(j,atype|BPTYPE_DATA,addr,mask,0) >= 0) {
	printf("Bpt%d has been set (%s)\n",j,Bpt[j].cstr);
	return(j); /* ok */
	}

/* 970227 error case */
printf("Unable to set Bpt%d\n",j);
Bpt[j].cond = 0;
free(Bpt[j].cstr);
if (Bpt[j].cmdstr) free(Bpt[j].cmdstr);
if (when_list) free(when_list);
return(-1);
}

Optdesc ab_opts[] = {
	{"[-rw] addr1 [mask]","set access (data) bpt"},
	{"addr1","start address"},
	{"mask","address mask"},
	{"-r","break on read operations only"},
	{"-w","break on write operations only"},
	{0}};

/*************************************************************
*/
int ab_cmd(ac,av)
int ac;
char *av[];
{
int got_addr1,got_mask,mode;
Ulong addr1,mask;
int i,j,rtn,n;
char *cstr;

got_addr1 = got_mask = mode = 0;

for (i=1;i<ac;i++) {
	if (av[i][0] == '-') {
		for (j=1;av[i][j];j++) {
			if (av[i][j] == 'r') mode |= 1;
			else if (av[i][j] == 'w') mode |= 2;
			else {
				printf("%c: bad option\n",av[i][j]);
				return(0);
				}
			}
		}
	else if (!got_addr1) {
		if (!get_rsa(&addr1,av[i])) return(0);
		got_addr1 = 1;
		}
	else if (!got_mask) {
		if (!get_rsa(&mask,av[i])) return(0);
		got_mask = 1;
		}
	else {
		printf("%s: too many args.\n",av[i]);
		return(0);
		}
	}
if (!mode) mode = 3;
if (!got_addr1) {
	printf("Requires at least one address.\n");
	return(0);
	}
if (!got_mask) mask = 0xffffffff;
setHwdbpt(mode,addr1,mask);
}
#endif

/*************************************************************
*  pcBpt(cond,csz)
* 	@pc CONST ==
* 	@pc CONST == ...... &&
*	If this a PC-only breakpoint, return the bpt addr; else return -1.
*/
pcBpt(cond,csz)
WhenRec *cond,*csz;
{
if (cond[0].tag == WHEN_REG  && 
    strequ(((RegRec *)cond[0].val)->name,"PC") &&
    cond[1].tag == WHEN_CONST &&
    cond[2].tag == WHEN_OPR && cond[2].val == WHEN_OPR_EQ) {
    	if (csz-cond == 3) return(cond[1].val);
    	else if (csz[-1].tag == WHEN_OPR && csz[-1].val == WHEN_OPR_LAND) 
		return(cond[1].val);
    }
return(-1);
}

/*************************************************************
*  singleDref(cond,csz)
*	If this a singleDref breakpoint, return the bpt addr; else return -1.
*/
singleDref(cond,csz)
WhenRec *cond,*csz;
{
if (cond[0].tag == WHEN_CONST  && cond[1].tag == WHEN_MEM &&
    cond[2].tag == WHEN_CONST && cond[3].tag == WHEN_OPR) {
    	if (csz-cond == 4) return(cond[0].val);
    	else if (csz[-1].tag == WHEN_OPR && csz[-1].val == WHEN_OPR_LAND) 
		return(cond[0].val);
    }
return(-1);
}

/*************************************************************
*  when(ac,av,info)
*	The 'when' (set trigger event) command.
*/
when(ac,av,info)
int ac;
char *av[];
CmdRec *info;
{
int j;
char tmp[LINESZ];

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

if (ac != 3) {
	printf("usage: %s %s\n",info->name,info->opts);
	return(1);
	}

strnspc(av[1]); /* remove whitespace */
j = setTrigger(av[1],av[2]);
if (j != -1) {
	sprintf(prnbuf,"Bpt %2d: when '%s' ",j,Bpt[j].cstr);
	if (Bpt[j].cmdstr) {
		sprintf(tmp," \"%s\"",Bpt[j].cmdstr);
		strcat(prnbuf,tmp);
		}
	printf("%s\n",prnbuf);
	}
else printf("Unable to set specified breakpoint.\n");
}

/*************************************************************
*  trace(ac,av) 
*	The 't' and 'to' (single-step) commands
*/
trace(ac,av)
int ac;
char *av[];
{
int i,j,n,flags,count;
Ulong target;
unsigned long adr,val;

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

flags = 0;
if (strequ(av[0],"to")) flags |= T_O;

n = 0;
count = 1;
for (i=1;i<ac;i++) {
	if (av[i][0] == '-') {
		for (j=1;av[i][j] != 0;j++) {
			if (av[i][j] == 'v') flags |= (T_V|T_M);
			else if (av[i][j] == 'b') flags |= T_B;
			else if (av[i][j] == 'c') flags |= T_C;
			else if (av[i][j] == 'i') flags |= (T_I|T_M);
			else {
				printf("%c: unrecognized option\n",av[i][j]);
				return(1);
				}
			}
		}
	else {
		if (n == 0) {
			if (!get_rsa(&count,av[i])) return(1);
			flags |= T_M;
			}
		else {
			printf("%s: unrecognized argument\n",av[i]);
			return(1);
			}
		n++;
		}
	}

clrpchist();
swlst(2); /* switch to clilst */
run_target(2,flags,count);
}

/*************************************************************
*  setbp(ac,av) 
*	The 'b' (set breakpoint) command.
*	Set one or more breakpoint(s).
*	When invoked without args, this command displays a list
*	of all breakpoints currently set.
*/
setbp(ac,av)
int ac;
char *av[];
{
Ulong adr,msk,i,j,w;
char *str,*p,tmp[LINESZ];
int Tflag;
int flag = 0;

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

if(ac == 1) {
	dspbpts();
	return(0);
	}

w = 0; str = 0; Tflag = 0;
for(i=1;i<ac;i++) {
	if (av[i][0] == '-') {
		if (av[i][1] == 's') {
			i++;
			if (i >= ac) {
				printf("bad arg count\n");
				return(1);
				}
			str = av[i];
			}
		else if (av[i][1] == 'T') Tflag = 1;
		else if (av[i][1] == 'd') {
			for (j=0;j<MAX_BPT;j++) {
				if (Bpt[j].cond==0) continue;
				printf("Bpt %d:\n",j);
				printOps(Bpt[j].cond,Bpt[j].csz);
				}
			return(0);
			}
		else {
			printf("%s: unrecognized option\n",av[i]);
			return(1);
			}
		}
	else {
		flag = 1;
		if(!get_rsa(&adr,av[i])) return(1);
		if (!str) str = "stop";
		if (Tflag) adr |= 1; /* set LSB */
		sprintf(tmp,"@pc==0x%08x",adr);
		j = setTrigger(tmp,str);
		if (j != -1) {
			sprintf(prnbuf,"Bpt %2d: when (%s) ",j,Bpt[j].cstr);
			if (Bpt[j].cmdstr) {
				sprintf(tmp," \"%s\"",Bpt[j].cmdstr);
				strcat(prnbuf,tmp);
				}
			printf("%s\n",prnbuf);
			}
		else printf("Unable to set specified breakpoint.\n");
		}
	}
if (!flag) printf("break address not specified\n");
return(0);
}

/*************************************************************
*  int getBpid(void)
*	Returns the breakpoint id of a breakpoint that requests a stop
*	or -1 if none.
*/
getBpid()
{
int i,j,n;
char *p;

n = -1;
for (i=0;i<MAX_BPT;i++) {
	if (Bpt[i].cond == 0) continue;
	if (evalBpt(Bpt[i].cond,Bpt[i].csz)) {
		p = strdup(Bpt[i].cmdstr);
		printf("brkpt %d \"%s\"\n",i,p);
		if (do_cmd(p)) {
			if (n == -1) n = i;
			}
		free(p);
		}
	}
return(n);
}

/*************************************************************
*  dspbpts() display all breakpoints
*/
dspbpts()
{
int i,j,ln,siz;
char tmp[64];

if (!atob(&siz,getMonEnv("moresz"),10)) {
	printf("%s: bad moresz value\n",getMonEnv("moresz"));
	return(1);
	}

ioctl_cbreak(0L);
ln = siz;
for (i=0;i<MAX_BPT;i++) {
	if(!Bpt[i].cond) continue;
	sprintf(prnbuf,"Bpt %2d: when (%s) ",i,Bpt[i].cstr);
	if (Bpt[i].cmdstr) {
		sprintf(tmp," \"%s\"",Bpt[i].cmdstr);
		strcat(prnbuf,tmp);
		}
	if (more(prnbuf,&ln,siz)) break;
	}
}

/*************************************************************
*  clrbp(ac,av)
*	The 'db' command. Clear one or more selected breakpoint(s).
*	When invoked without args, this command displays a list
*	of all breakpoints currently set.
*/
clrbp(ac,av)
int ac;
char *av[];
{
Ulong adr;
int i,j,k;

if (!regChain) {
	printf("Target Description Driver not loaded\n");
	return(1);
	}

if(ac > 1){
	for(i = j = 0; j < ac - 1; j++){
		if (strequ(av[1+j],"*")) {
			clrbpt(-1);
			continue;
			}
		if (!atob(&i,av[1+j],10)) {
			printf("%s: decimal number expected\n",av[1+j]);
			continue;
			}
		clrbpt(i);
		}
	}
else dspbpts();
return(0);
}

/*************************************************************
*  void clrbpt(i)
*	Clear specified breakpoint. If i==-1 clear all bpts.
*/
void clrbpt(i)
int i;
{

if (i == -1) {
	for (i=0;i<MAX_BPT;i++) {
		if (Bpt[i].cond) {
			clrbp_target(i);
			free(Bpt[i].cond);
			Bpt[i].cond = 0;
			if (Bpt[i].cmdstr) free(Bpt[i].cmdstr);
			if (Bpt[i].cstr) free(Bpt[i].cstr);
			}
		}
	bptCount = 0;
	return;
	}

if (i < MAX_BPT) {
	if (Bpt[i].cond == 0) printf("Bpt %2d not set\n",i);
	else {
		clrbp_target(i);
		free(Bpt[i].cond);
		Bpt[i].cond = 0;
		if (Bpt[i].cmdstr) free(Bpt[i].cmdstr);
		if (Bpt[i].cstr) free(Bpt[i].cstr);
		}
	bptCount--;
	}
else printf("%d: breakpoint number too large\n",i);
}

/*************************************************************
*  is_validpc(adr)
*/
is_validpc(adr)
unsigned long adr;
{
int i,ac;
char *av[10],tmp[80];
unsigned long a1,a2;

strcpy(tmp,getMonEnv("validpc"));
ac = argvize(av,tmp);
if (ac%2 != 0) {
	printf("validpc variable must have even number of values\n");
	return(0);
	}
for (i=0;i<ac;i++) {
	if (!get_rsa(&a1,av[i])) return(0);
	i++;
	if (!get_rsa(&a2,av[i])) return(0);
	if (adr >= a1 && adr <= a2) return(1);
	}
return(0);
}

/*************************************************************
*  addpchist(adr)
*/
addpchist(adr)
unsigned long adr;
{
pchist_d[pchist_ip] = adr;
pchist_ip = incmod(pchist_ip,PCHISTSZ);
if (pchist_ip == pchist_op) pchist_op = incmod(pchist_op,PCHISTSZ);
}

/*************************************************************
*  clrpchist()
*/
clrpchist()
{
pchist_ip = pchist_op = 0;
}

/*************************************************************
*  unsigned long getpchist(n)
*/
unsigned long getpchist(n)
int n;
{
int i;

i = pchist_ip-n-1;
if (i < 0) i += PCHISTSZ+1;
if (incmod(i,PCHISTSZ) == pchist_op) return(0);
return(pchist_d[i]);
}

char *strrpstr();

/*************************************************************
*  constReduc()
*	optimize by performing constant reduction. 
*	eg. change 80023450 00000018 + to 80023468
*	It would be better to integrate this with parseCond().
*/
constReduc()
{
WhenRec *p,*q;
int count;
Ulong v1,v2;

count = 0;
for (p=when_list;p<when_ptr;p++) {
	switch (p->tag) {
		case WHEN_CONST : count++; break;
		case WHEN_OPR : 
		if (count < 2) {
			count = 0; /* 980802 */
			break;
			}
		v1 = (p-2)->val;
		v2 = (p-1)->val;
		/* printf("opt: %08x %08x %s\n",v1,v2,oprs[p->val]); */
		switch (p->val) {
			case WHEN_OPR_LAND : /* logical AND */
				if (v1 && v2) (p-2)->val = ~0;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_LOR :
				if (v1 || v2) (p-2)->val = ~0;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_EQ :
				if (v1 == v2) (p-2)->val = ~0; 
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_NE :
				if (v1 != v2)  (p-2)->val = ~0;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_GE :
				if (v1 >= v2)  (p-2)->val = ~0;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_LE :
				if (v1 <= v2)  (p-2)->val = ~0;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_GT :
				if (v1 > v2)  (p-2)->val = ~0;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_LT :
				if (v1 < v2) (p-2)->val = 1;
				else (p-2)->val = 0;
				break;
			case WHEN_OPR_ADD :
				(p-2)->val = v1 + v2;
				break;
			case WHEN_OPR_SUB :
				(p-2)->val = v1 - v2;
				break;
			case WHEN_OPR_MUL :
				(p-2)->val = v1 * v2;
				break;
			case WHEN_OPR_DIV :
				(p-2)->val = v1 / v2;
				break;
			case WHEN_OPR_BAND :  /* Bitwise AND */
				(p-2)->val = (v1 & v2);
				break;
			case WHEN_OPR_BOR : 
				(p-2)->val = (v1 | v2);
				break;
			default : count = 0;
			}
			if (count==2) {
				for (q=p+1;q<when_ptr;q++) *(q-2) = *q;
				when_ptr -= 2;
				count = 1;
				p = p-3;
				}
		default : count = 0;
		}
	}
}

/*************************************************************
*  parseCond(p)
*	Parse condition. 
*	This function evaluates expressions on the command line
*	using recursive decent. It places the result in an RPN
*	stack for faster run-time evaluation. It understands expressions
*	containing any combination of register names, symbols,
*	constants, perens, the operators ~!|&+/-* && ||, and the relational
*	operators > >= < <= == !=. In addition	it also permits 
*	the '^' operator to dereference any address. 
*/
parseCond(p)
char *p;
{
int r,inbase,inalpha,i;
char *q,subexpr[LINESZ];
extern char *badhexsym;
Ulong val;

/* strip enclosing parens */
while (*p == '(' && strbalp(p) == p+strlen(p)-1) {
	strdchr(p);
	p[strlen(p)-1] = 0;
	}

for (i=0;oprs[i];i++) {
	if (q=strrpstr(p,oprs[i])) {
		strNcpy(subexpr,p,q-p);
		r = parseCond(subexpr);
		if (r == 0) return(r);
		r = parseCond(q+strlen(oprs[i]));
		if (r == 0) return(r);
		if (when_list) {
			when_ptr->tag = WHEN_OPR;
			when_ptr->val = i;
			}
		when_ptr++;
		return(1);
		}
	}

/* 
 * having taken care of all the usual stuff, we handle the unary
 * operators and special cases.
 */
if (*p == '^') {
	r = parseCond(p+1);
	if (r == 0) return(r);
	if (when_list) {
		when_ptr->tag = WHEN_MEM;
		when_ptr->val = 0;
		}
	when_ptr++;
	return(1);
	}

if (*p == '!') {
	r = parseCond(p+1);
	if (r == 0) return(r);
	if (when_list) {
		when_ptr->tag = WHEN_NOT;
		when_ptr->val = 0;
		}
	when_ptr++;
	return(1);
	}

if (*p == '~') {
	r = parseCond(p+1);
	if (r == 0) return(r);
	if (when_list) {
		when_ptr->tag = WHEN_INV;
		when_ptr->val = 0;
		}
	when_ptr++;
	return(1);
	}

if (*p == '@') {
#ifdef PREGS
	if (ispreg(&p[1])) {
		val = (Ulong)(p[1]-'a');
		if (when_list) {
			when_ptr->tag = WHEN_PREG;
			when_ptr->val = val;
			}
		when_ptr++;
		r = 1;
		}
	else {
		val = (Ulong)findRegRec(&p[1],0);
		if (val) {
			if (when_list) {
				when_ptr->tag = WHEN_REG;
				when_ptr->val = val;
				}
			when_ptr++;
			r = 1;
			}
		else printf("%s: reg not found\n",&p[1]);
		}
#else
	val = (Ulong)findRegRec(&p[1],0);
	if (val) {
		if (when_list) {
			when_ptr->tag = WHEN_REG;
			when_ptr->val = val;
			}
		when_ptr++;
		r = 1;
		}
	else printf("%s: reg not found\n",&p[1]);
#endif
	}
else if (strequ(p,".")) {
	if (when_list) {
		when_ptr->tag = WHEN_REG;
		when_ptr->val = (Ulong)findRegRec("pc",0);
		}
	when_ptr++;
	r = 1;
	}
else if (isdigit(*p)) {
	inbase = matchenv("inbase");
	if (inbase == IB_TEN) r = atob(&val,p,10);
	else if (inbase == IB_SIXTEEN) r = atob(&val,p,16);
	else if (inbase == IB_EIGHT) r = atob(&val,p,8);
	else if (inbase == IB_AUTO) r = atob(&val,p,0);
	else {
		printf("%s: bad inbase value\n",getMonEnv("inbase"));
		return(0);
		}
	if (r == 0) {
		r = atob(&val,p,0);
		if (r == 0) printf("%s: bad base %s value\n",
						p,getMonEnv("inbase"));
		}
	if (r) {
		if (when_list) {
			when_ptr->tag = WHEN_CONST;
			when_ptr->val = val;
			}
		when_ptr++;
		}
	}
else if (isxdigit(*p)) {
	r = sym2adr(&val,p);
	if (r == 0) {
		r = atob(&val,p,16);
		if (r == 0) printf(badhexsym,p);
		}
	if (r) {
		if (when_list) {
			when_ptr->tag = WHEN_CONST;
			when_ptr->val = val;
			}
		when_ptr++;
		}
	}
else {
	r = sym2adr(&val,p);
	if (r == 0) {
		if (*p == '(') printf("unbalanced parens\n");
		else printf("%s: bad symbol name\n",p);
		}
	if (r) {
		if (when_list) {
			when_ptr->tag = WHEN_CONST;
			when_ptr->val = val;
			}
		when_ptr++;
		}
	}
return(r);
}

Ulong when_evalstk[8];
Ulong *when_evalptr;

/*************************************************************
*  evalBpt(p,e)
*/
evalBpt(s,e)
WhenRec *s,*e;
{
WhenRec *p;
Ulong *t,v;

when_evalptr = when_evalstk;
for (p=s;p<e;p++) evalOp(p);
#if 0
for (t=when_evalstk;t<when_evalptr;t++) printf("%08x\n",*t);
#endif
v = *--when_evalptr;
if (when_evalptr != when_evalstk) printf("stack corruption\n");
return(v);
}

/*************************************************************
*  evalOp(p)
*/
evalOp(p)
WhenRec *p;
{
Ulong v1,v2;

switch (p->tag) {
	case WHEN_CONST : *when_evalptr++ = p->val; break;
	case WHEN_MEM : 
		v1 = *--when_evalptr;
		*when_evalptr++ = read_target32(v1); 
		break;
	case WHEN_REG : *when_evalptr++ = getRegVal32(p->val); break;
#ifdef PREGS
	case WHEN_PREG : *when_evalptr++ = getPreg(p->val); break;
#endif
	case WHEN_NOT : 
		v1 = *--when_evalptr; 
		if (v1) *when_evalptr++ = 0;
		else *when_evalptr = 1;
		break;
	case WHEN_INV : v1 = *--when_evalptr; *when_evalptr++ = ~v1; break;
	case WHEN_OPR :
		switch (p->val) {
			case WHEN_OPR_LAND :
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				if (v1 && v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_LOR :
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				if (v1 || v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_EQ :
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				if (v1 == v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_NE :
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				if (v1 != v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_GE :
				v2 = *--when_evalptr;
				v1 = *--when_evalptr;
				if (v1 >= v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_LE :
				v2 = *--when_evalptr;
				v1 = *--when_evalptr;
				if (v1 <= v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_GT :
				v2 = *--when_evalptr;
				v1 = *--when_evalptr;
				if (v1 > v2) *when_evalptr++ = ~0;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_LT :
				v2 = *--when_evalptr;
				v1 = *--when_evalptr;
				if (v1 < v2) *when_evalptr++ = 1;
				else *when_evalptr++ = 0;
				break;
			case WHEN_OPR_ADD :
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				*when_evalptr++ = v1 + v2;
				break;
			case WHEN_OPR_SUB :
				v2 = *--when_evalptr;
				v1 = *--when_evalptr;
				*when_evalptr++ = v1 - v2;
				break;
			case WHEN_OPR_MUL :
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				*when_evalptr++ = v1 * v2;
				break;
			case WHEN_OPR_DIV :
				v2 = *--when_evalptr;
				v1 = *--when_evalptr;
				*when_evalptr++ = v1 / v2;
				break;
			case WHEN_OPR_BAND : 
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				*when_evalptr++ = (v1 & v2);
				break;
			case WHEN_OPR_BOR : 
				v1 = *--when_evalptr;
				v2 = *--when_evalptr;
				*when_evalptr++ = (v1 | v2);
				break;
			default : printf("%d: bad opr\n",p->val);
		}
		break;
	default : printf("%d: bad tag\n",p->tag);
	}
}

/*************************************************************
*  printOps(cond,csz)
*/
printOps(cond,csz)
WhenRec *cond,*csz;
{
WhenRec *p;
for (p=cond;p<csz;p++) {
	switch (p->tag) {
		case WHEN_OPR : 
			printf("opr %s\n",oprs[p->val]); 
			break;
		case WHEN_CONST : 
			printf("const %08x\n",p->val);
			break;
		case WHEN_MEM :
			printf("memref\n");
			break;
#ifdef PREGS
		case WHEN_PREG :
			printf("preg %c\n",p->val+'a');
			break;
#endif
		case WHEN_REG :
			printf("reg %s\n",((RegRec *)p->val)->name);
			break;
		case WHEN_NOT :
			printf("NOT\n");
			break;
		case WHEN_INV :
			printf("INV\n");
			break;
		default : printf("bad tag: %08x %s\n",p->val,p->tag);
		}
	}
}
