/*************************************************************
 * File: mon/bsocmds.c
 * Purpose: Part of core Monitor
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970304	Start of revision history
 *	970311	Added dummy for "sc message"
 *	971130	Added nice error msg to gmx, smx, and do_init
 */

#include <termio.h>
#include <mon.h>

#define GPREGLIST

/*************************************************************
*  commands to support BSO/Tasking's crossview
*/

Optdesc crossview_opts[] = {
	{"","Commands for Crossview"},
	{"gc","get config info"},
	{"sc","set config info"},
	{"sr","set register"},
	{"mr","display modified registers"},
	{"cp","copy memory range"},
	{"gmb","get memory bytes"},
	{"gmh","get memory half-words"},
	{"gmw","get memory words"},
	{"smb","set memory bytes"},
	{"smh","set memory half-words"},
	{"smw","set memory words"},
	{"init","reboot"},
	{"lo","binary download"},
	{0}};

/*************************************************************
*  do_echo(ac,av)
*	echo the arguments
*/
do_echo(ac,av)
int ac;
char *av[];
{
int i;

for (i=1;i<ac;i++) {
	if (i > 1) printf(" ");
	printf("%s",av[i]);
	}
printf("\n");
}

/*************************************************************
*  do_gc(ac,av)
*	get config info
*/
do_gc(ac,av)
int ac;
char *av[];
{
struct termio tbuf;

if (ac == 1) {
	printf("error: missing keyword\n");
	goto do_gc_1;
	}
if (strequ("ttyecho",av[1])) printf("ttyecho %s\n",(noecho)?"off":"on");
else if (strequ("ttyixon",av[1])) {
	ioctl(STDIN,TCGETA,&tbuf);
	printf("ttyixon %s\n",(tbuf.c_iflag&IXOFF)?"on":"off");
	}
else if (strequ("ttyixoff",av[1])) {
	ioctl(STDIN,TCGETA,&tbuf);
	printf("ttyixon %s\n",(tbuf.c_iflag&IXON)?"on":"off");
	}
else if (strequ("more",av[1])) 
	printf("more %s\n",(strequ(getMonEnv("moresz"),"9999999"))?"off":"on");
else if (strequ("endian",av[1])) {
#ifdef MIPSEB
	printf("endian big\n");
#else
	printf("endian little\n");
#endif
	}
else if (strequ("cpu",av[1])) {
	if (cpuType() == 64360) printf("cpu ATMizer\n");
	else printf("cpu LR%d\n",cpuType());
	}
else if (strequ("fpu",av[1])) {
	printf("fpu %s\n",((cpuType()==33050) || cp1ok)?"yes":"no");
	}
else if (strequ("xvw",av[1])) {
	printf("xvw %s\n",(xvwmode)?"on":"off");
	}
else {
	printf("%s: unknown keyword\n",av[1]);
    do_gc_1:
	printf("keywords: ttyecho ttyixon ttyixoff mod more endian cpu fpu\n");
	}
}

/*************************************************************
*  do_sc(ac,av)
*	set config info
*/
do_sc(ac,av)
int ac;
char *av[];
{
struct termio tbuf;
Ulong value;
int i,flags;
RegRec *p;

if (strequ("ttyecho",av[1])) {
	if (strequ("on",av[2])) noecho = 0;
	else if (strequ("off",av[2])) noecho = 1;
	else printf("%s: unknown option\n",av[2]);
	}
else if (strequ("ttyixon",av[1])) {
	ioctl(STDIN,TCGETA,&tbuf);
	if (strequ("on",av[2])) tbuf.c_iflag |= IXON;
	else if (strequ("off",av[2])) tbuf.c_iflag &= ~IXON;
	else printf("%s: unknown option\n",av[2]);
	ioctl(STDIN,TCSETAF,&tbuf);
	}
else if (strequ("ttyixoff",av[1])) {
	ioctl(STDIN,TCGETA,&tbuf);
	if (strequ("on",av[2])) tbuf.c_iflag |= IXOFF;
	else if (strequ("off",av[2])) tbuf.c_iflag &= ~IXOFF;
	else printf("%s: unknown option\n",av[2]);
	ioctl(STDIN,TCSETAF,&tbuf);
	}
else if (strequ("mod",av[1])) {
	if (strequ("on",av[2])) {
        	for (p=0;p=findRegRec("*",p);) {
			if (!(p->flags&F_MIPS)) continue;
			value = (* p->func)(0,p->regnum);
			p->prev = ~value; 
			p->modified = 1;
			}
		}
	else printf("%s: unknown option\n",av[2]);
	}
else if (strequ("more",av[1])) {
	if (strequ("on",av[2])) setenv("moresz","10");
	else if (strequ("off",av[2])) setenv("moresz","9999999");
	else printf("%s: unknown option\n",av[2]);
	}
else if (strequ("xvw",av[1])) {
	if (strequ("on",av[2])) xvwmode = 1;
	else if (strequ("off",av[2])) xvwmode = 0;
	else printf("%s: unknown option\n",av[2]);
	}
else if (strequ("message",av[1])) {
#if 0 /* 970311 I have no idea what this is supposed to do */
	if (strequ("full",av[2])) 
#endif
	}
else {
	printf("%s: unknown keyword\n",av[1]);
	printf("keywords: ttyecho ttyixon ttyixoff mod more\n");
	printf("options: on off\n");
	}
}

/*************************************************************
*  set register
*	sr <reg> <value>
*/
do_sr(ac,av)
int ac;
char *av[];
{
Ulong value;
int i,flags;
char tmp[40];
RegRec *p;

if (sscanf(av[2],"%x",&value) != 1) {
	printf("%s: bad value\n",av[2]);
	return(0);
	}
for (p=0;p=findRegRec("*",p);) {
	flags = p->flags;
	if (!(flags&F_MIPS)) continue;

	if (flags&(F_GPR|F_CP0)) {
		strcpy(tmp,p->aname);
		strtolower(tmp);
		if (strequ(tmp,"epc")) strcpy(tmp,"pc");
		}
	else strcpy(tmp,p->name);
	if (strequ(tmp,"C1_FRID")) strcpy(tmp,"fcr0");
	if (strequ(tmp,"C1_CSR")) strcpy(tmp,"fcr31");

	if (strequ(av[1],tmp)) {
		(* p->func)(1,p->regnum,p->size,value);
		return(0);
		}
	}
printf("%s: unknown register\n",av[1]);
}

/*************************************************************
*  do_mr(ac,av)
*	display modified registers
* keywords
*   gen : general purpose registers
*   fgr : floating point general registers ( f0-f31)
*   cp0 : (bpc, bda, dcic, bada, sr, cause, epc, prid )
*   cp1 : fcr0, fcr31
*   cpu : pc hi lo
*   any : any register name
*/
do_mr(ac,av)
int ac;
char *av[];
{
Ulong key,flags,i,value;
char tmp[20];
RegRec *p;

if (ac == 1) key = (F_GPR|F_CP0|F_FCR|F_CPU|F_FPR);
else {
	if (strequ("gen",av[1])) key = F_GPR;
	else if (strequ("fgr",av[1])) key = F_FPR;
	else if (strequ("f",av[1])) key = F_FPR;
	else if (strequ("cp0",av[1])) key = F_CP0;
	else if (strequ("cp1",av[1])) key = F_FCR;
	else if (strequ("cpu",av[1])) key = F_CPU;
	else {
		for (p=0;p=findRegRec("*",p);) {
			flags = p->flags;
			if (!(flags&F_MIPS)) continue;

			if (strequ(p->name,"C1_FRID")) 
				strcpy(tmp,"fcr0");
			else if (strequ(p->name,"C1_CSR")) 
				strcpy(tmp,"fcr31");
			else strcpy(tmp,p->name);

			if (striequ(av[1],tmp) || 
			    striequ(av[1],p->aname)) {
				value = (* p->func)(0,p->regnum,p->size);
				if ((p->prev != value) || p->modified) {
					printf("%s %08x\n",av[1],value);
					p->prev = value;
					p->modified = 0;
					}
				return(0);
				}
			}
		printf("%s: unknown reg name\n",av[1]);
		return(0);
		}
	}

for (p=0;p=findRegRec("*",p);) {
	flags = p->flags;
	if (!(flags&F_MIPS)) continue;
	if ((flags&key)==0) continue;
	value = (* p->func)(0,p->regnum,p->size);

	if (p->prev != value || p->modified) {
			if (flags&(F_CP0|F_FPR|F_CPU)) {
				strcpy(tmp,p->aname);
				strtolower(tmp);
				}
			else strcpy(tmp,p->name);
			if (strequ(tmp,"C1_FRID")) strcpy(tmp,"fcr0");
			if (strequ(tmp,"C1_CSR")) strcpy(tmp,"fcr31");
			printf("%s %08x\n",tmp,value);
			p->prev = value;
			p->modified = 0;
			}
	}
}

/*************************************************************
*  do_cp( ac, av )
*	copy memory range
*/
do_cp( ac, av )
int	ac;
char	*av[];
{
Ulong addrl;
Ulong addrh;
Ulong addrt;
Ulong val;

	if ( !strchr(av[1], ':') ||
	     (sscanf(av[1],"%x:%x",&addrl,&addrh) != 2) ||
	     (sscanf(av[2],"%x",&addrt ) != 1))	{
		printf("%s: bad arg\n",av[1]);
		return(0);
	}
	while ( addrl <= addrh )
	{
		val = read_target(XT_MEM, addrl, 1 );
		write_target(XT_MEM,addrt,val,1);
		addrl++; addrt++;
	}
}

/*************************************************************
*  gmx(ac,av)
*	get memory as (bytes/half-words/words)
*	gmx addr#cnt
*	gmx start:end
*	gmx addr
*/
gmx(ac,av)
int ac;
char *av[];
{
int i,cnt,mod,siz;
Ulong addr;
Ulong uaddr;
char *fmt;

if (ac != 2) {
	printf("usage: gm[bhw] addr|start:end|addr#cnt\n");
	return(0);
	}

switch (av[0][2]) {
	case 'b' : fmt = " %02x"; siz = 1; mod = 16; break;
	case 'h' : fmt = " %04x"; siz = 2; mod = 8;  break;
	case 'w' : fmt = " %08x"; siz = 4; mod = 4;  break;
	}
if (strchr(av[1],'#')) {
	if (sscanf(av[1],"%x#%x",&addr,&cnt) != 2) {
		printf("%s: bad arg\n",av[1]);
		return(0);
		}
	}
else if ( strchr(av[1], ':')) {
	if ( sscanf( av[1], "%x:%x", &addr, &uaddr) != 2) {
		printf("%s: bad arg\n",av[1]);
		return(0);
		}
	cnt = uaddr - addr;
	if ( av[0][2] == 'h' )
		cnt = cnt / 2;
	if ( av[0][2] == 'w' )
		cnt = cnt / 4;
	cnt++;
	}
else {
	if (sscanf(av[1],"%x",&addr) != 1) {
		printf("%s: bad arg\n",av[1]);
		return(0);
		}
	cnt = 1;
	}
printf("%08x",addr);
for (i=0;i<cnt;i++) {
	if (i && (i%mod)==0) printf("\n%08x",addr);
	printf(fmt,read_target(XT_MEM,addr,siz));
	addr += siz;
	}
printf("\n");
}

/*************************************************************
*  smx(ac,av)
*	set memory as (bytes/half-words/words)
*	smx addr val..
*/
smx(ac,av)
int ac;
char *av[];
{
int i,siz;
Ulong addr,value;

if (ac < 3) {
	printf("usage: sm[bhw] addr val\n");
	return(0);
	}

switch (av[0][2]) {
	case 'b' : siz = 1; break;
	case 'h' : siz = 2; break;
	case 'w' : siz = 4; break;
	}
if (sscanf(av[1],"%x",&addr) != 1) {
	printf("%s: bad addr\n",av[1]);
	return(0);
	}

for (i=2;i<ac;i++) {
	if (sscanf(av[i],"%x",&value) != 1) {
		printf("%s: bad value\n",av[i]);
		return(0);
		}
	write_target(XT_MEM,addr,value,siz);
	addr += siz;
	}
}

/*************************************************************
*  do_init(ac,av)
*	warm (cold) reboot
*	-c cold reboot
*/
do_init(ac,av)
int ac;
char *av[];
{

if (ac > 1) {
	if (strequ(av[1],"-c")) _start();
	else {
		printf("%s: bad option\n",av[1]);
		printf("usage: init [-c]\n");
		}
	}
else clrhndlrs();
}

