/*************************************************************
 * File: lib/tmr1.c
 * Purpose: clock using tmr1 on the 4102
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	981014	Moved from lib/c4102.c
 *	981106	Works with "-board bdmr4102".
 */

/* This module can only be used on the 4102, because the 4101, 4001 etc
 * did not allow tmr1 to generate interrupts.
 */

#ifndef LR4102
#define LR4102
#endif

#include <mips.h>
#include <utypes.h>

#define inw(a)		(*((volatile Ulong *)(a)))
#define outw(a,v)	(*((volatile Ulong *)(a))=(v))

/* 4102 cpu 32-bit timer */
#define TMRI    (M_TMR4001+O_TIC1)  /* initial value */
#define TMRC    (M_TMR4001+O_TCC1)  /* current value */
#define TMODE   (M_TMR4001+O_TMODE) /* control */
#define TSTAT   (M_TMR4001+O_TSTAT) /* status */

static Ulong _time,_ticks;

/*************************************************************
*/
static int clkfunc(int op,long value)
{
switch (op) {
    case 1 : _time = value; return(value);
    case 2 : return(_time);
    case 3 : return(_time*1000000)+(_ticks*10000);
    }
}

/*************************************************************
*/
static clkisr()
{

outw(TSTAT,inw(TSTAT)&~TSTAT_IN1); /* ack timer int */
_ticks++;
if (_ticks >= 100) {
	_time++;
	_ticks=0;
	}
return(1);
}

/*************************************************************
*/
Func *clkinit_tmr1_4102()
{
Ulong t8,t0,t1,intmask,msk,tmode,tstat;
int i,intnum,cf;

t8 = mfc0(C0_SR);
mtc0(C0_SR,t8&~SR_IEC); /* disable ints in SR */

/* disable clkints */
tmode = inw(TMODE)&~TMODE_MASK1;
outw(TMODE,tmode); /* stop timer */
tstat = inw(TSTAT)&~TSTAT_MASK1;
outw(TSTAT,tstat); /* no ints */
outw(M_SCR2,inw(M_SCR2)&~SCR2_TMRIE1);

t0 = mfc0(C0_CAUSE); /* pre-intr cause value */

/* program timer for short period */
outw(TMODE,tmode|TMODE_I1);  /* invert */
outw(M_SCR2,inw(M_SCR2)|SCR2_TMRIE1);
outw(TMRI,3);
outw(TSTAT,tstat|TSTAT_IE1); /* int enable */
outw(TMODE,tmode|TMODE_I1|TMODE_E1);  /* invert+enable */

for (i=0;i<100000;i++) { /* timeout value */
	t1 = mfc0(C0_CAUSE); /* new cause value */
	if (t1 != t0) break;
	}
intmask = SR_IMASK&(t0^t1);
for (intnum=0,msk=0x100;intnum<8;intnum++,msk<<=1) if (intmask&msk) break;

if (intnum > 7) {
	mtc0(C0_SR,t8);
	return(0);
	}

outw(M_SCR2,inw(M_SCR2)&~SCR2_TMRIE1);
outw(TMODE,tmode); /* stop timer */
outw(TSTAT,tstat); /* no ints */

/* attach isr */
IRQInstall(intnum,clkisr);

/* reprogram timer for 10ms */
sscanf(getMonEnv("clkfreq"),"%d",&cf);
cf *= 1000000;
outw(TMRI,cf/100); /* 10ms based on CPU clock frequency */
outw(TMODE,tmode|TMODE_I1);  /* invert */
outw(M_SCR2,inw(M_SCR2)|SCR2_TMRIE1);
outw(TSTAT,tstat|TSTAT_IE1); /* int enable */
outw(TMODE,tmode|TMODE_I1|TMODE_E1);  /* invert+enable */

return(clkfunc);
}

