/*************************************************************
 * File: lib/p90084.c
 * Purpose: Polled I/O routines for the LSI L9A0084 on-chip UART
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	980616	Start of revision history
 *	980618	Added OP_BAUDRATES
 */

#ifndef L9A0084
#define L9A0084
#endif
#include <mips.h>
#include <terms.h>


/* This is the pseudo device for the L64008 */
L90084rec l0084_sio[] = {
	{M_084_SOD0,M_084_SOC0,M_084_SID0,M_084_SIC0,M_084_ACLKDIV0},
	{M_084_SOD2,M_084_SOC2,M_084_SID2,M_084_SIC2,M_084_ACLKDIV2},
	{M_084_SOD1,M_084_SOC1,M_084_SID1,M_084_SIC1,M_084_ACLKDIV1},
	{0}};


/* Baudrate calculations:
 * 	Assumes ACLKDIV0:CSEL = 0 (13.5MHz)
 *	Assumes SOC0=SIC0:CSEL = 1 (iaclkp0)
 */
static Ulong btab[] = {
   /*   div    db4    */
	-1| 	0, /* B0 */
	-1|	0, /* B50 */
	-1|	0, /* B75 */
	-1|	0, /* B110 */
	-1|	0, /* B134 */
	-1|	0, /* B150 */
	-1|	0, /* B200 */
	-1|	0, /* B300 */
	-1|	0, /* B600 */
	176|	ACLKDIV_DB4, /* B1200 */
	132|	ACLKDIV_DB4, /* B1800 */
	88|	ACLKDIV_DB4, /* B2400 */
	176|	0, /* B4800 */
	88|	0, /* B9600 */
	44|	0, /* B19200 */
	22|	0, /* B38400 */
	0	   /* end */
	};

#define DEFRATE		13 /* default baudrate (9600) */

#define inb(a)         (*((volatile Uchar *)(a)))
#define inh(a)         (*((volatile Ushort *)(a)))
#define inw(a)         (*((volatile Ulong *)(a)))
#define outb(a,v)      (*((volatile Uchar *)(a))=(v))
#define outh(a,v)      (*((volatile Ushort *)(a))=(v))
#define outw(a,v)      (*((volatile Ulong *)(a))=(v))

#define RXRDY	SIC_SFNE
#define TXRDY	SOC_SFNF

/*************************************************************
*  p90084(op,siodat,chan,ch)
*/
p90084(op,siodat,chan,ch)
int op,chan,ch;
L90084rec *siodat;
{
int i,brate;
Ulong csel,w;
L90084rec *p;
Ulong m,v;

p = &siodat[chan];
switch (op) {
	case OP_RXRDY : 
		if (inw(p->sic)&RXRDY) return(1);
		break;

	case OP_RX :
		return inw(p->sid)&0xff;

	case OP_TXRDY :
		if (inw(p->soc)&TXRDY) return(1);
		break;

	case OP_TX :
		outw(p->sod,ch);
		break;

	case OP_INIT : 
		p = &siodat[0];
		for (i=0;p[i].sod;i++) {
			outw(p[i].baud,0);
			outw(p[i].baud,ACLKDIV_EN);
			outw(p[i].baud,btab[DEFRATE]|ACLKDIV_EN);
			/* cmd: enable rx & tx */
			/* mode: 1 stopbits, no parity, 8 bitsdata, 16x */
			csel = (i+1)<<SOC_CSELSHFT;
			outw(p[i].soc,SOC_EN|SOC_SD|csel); 
			outw(p[i].sic,SIC_EN|SIC_SD|csel); 
			}
		break;
	case OP_BAUD : 
		brate = ch;
		/* check brate <= 15 */
		if (brate > 15) return(1); /* baud rate too large */
		/* return if unsupported baud rate */
		if (btab[brate] == -1) return(1);  
		w = inw(p->baud)&~(ACLKDIV_DB4|ACLKDIV_DIV);
		outw(p->baud,btab[brate]|w);
		break;
	case OP_CLKINIT : break;
	case OP_DELAY : return(0); break;
	case OP_BAUDRATES :
		m = 1;
		for (i=v=0;btab[i];i++) {
			if (btab[i] != -1) v |= m;
			m <<= 1;
			}
		return(v);
		break;
	}
return(0);
}

