/*************************************************************
* File: k64388.s
* Author: Phil Bunce (pjb@carmel.com) Curtis Dahl (cdahl@lsil.com)
*
* Revision History:
*	970129	Initial release
*	970303	Brought ice port defs upto date. Added RXC etc.
*	970306	Changed to word ld and st ops to iceport.
*	970306	Added _start entry. Changed old _start to cstartup
*	970306	Added IBS to save area
*	970311	Changed bne to beq in test of xxRDY
*	970311	Added "and" with TXRDY to send_ack
*	970421	Change from Int5 to Int4
*	970829	Removed call to CpuInit in cstartup.
*	971113	Added wakeup banner (ABCD)
*	971114	Changed ICEPORT addr to b0ff.0000
*	971114	Added USE_NO_INTS switch
*	980312	Switched to version2 savearea format
*   	981110  Built for RAP
*
* This file contains the PROM resident code (the IceKernel) necessary to
* permit programs to be debugged using LSI Logic's SerialIce.  This
* implementation is for the 64388 (RAP) with the SerialIce Port.
*
* The code in this module executes in kseg1 (non cacheable), leaves
* BEV=1, and does not initialize the caches. 
*
* This module should be linked with .text section to start at bfc00000,
* and with data to start at some out-of-the-way place in RAM. I suggest
* 80000100. That way the downloaded programs can start at 80000200 and
* still be able to copy an exception handler to 80000080 for use once BEV
* has been cleared. Example compile/link command for this file:
*
*     pmcc -crt0 -prom -syms -T bfc00000 -D 80000100 -o k64388 k64388.s
*
* To download your application into RAM you should compile/link your 
* program using the following command.
*
*     pmcc -crt0 -T 80000200 -o myprog myprog_files...
*
* This file contains the following blocks of code:
*	reset_vector - The start of execution
*	utlb_vector  - UTLB exception handler
*	gen_vector   - Handler for all other exceptions
*	ice_loop     - Main loop of ICE
*	get_word     - Get one word from serial interface
*	put_word     - Put one word to serial interface
*	cpu_init     - Perform CPU-specific initialization
*
* This module requires the following areas of RAM:
*	instr_buffer - This is where the instructions that have been
*		       received from the host are saved.
*	savearea     - This is where I save the context of the
*		       downloaded program.
*	
*/

#include <mips.h>

/* Commands that are sent by the IceController */
#define SENDA0		0x12345678	/* execute instr_buffer */
#define RUN_MODE	0x87654321	/* run application */
#define SENDSAP		0xDEADBEEF	/* send savearea pointer */
#define ATTN		0x55		/* transfer control to IceKernel */
#define ACK		    0xaa		/* reply to ATTN */

/* Offsets into the savearea */
#define SA_VERS 2
#define ICE_SAV	0	/* save area version */
#define ICE_SAH 1	/* save area header size */
#define ICE_MAP	2	/* bit-map for savearea */
#define ICE_IBS 3	/* size of instr buffer */
#define ICE_GWP 4	/* pointer to get_word routine */
#define ICE_PWP 5	/* pointer to put_word routine */
#define ICE_EPC	6	/* saved so that it can be set */
#define ICE_LE  7	/* set if little endian */
#define ICE_SAHSIZE 8	/* size of save area header */
/* end of header. The remainder is kernel-specific */
#define ICE_AT	(ICE_SAHSIZE+0)	/* v0 is used to hold the value received */
#define ICE_V0	(ICE_SAHSIZE+1)	/* v0 is used to hold the value received */
#define ICE_A0	(ICE_SAHSIZE+2)	/* a0 is used to hold the value to be sent */
#define ICE_A1	(ICE_SAHSIZE+3)	/* a1 is used as a temp */
#define ICE_A2	(ICE_SAHSIZE+4)	/* a1 is used as a temp */
#define ICE_A3	(ICE_SAHSIZE+5)	/* a1 is used as a temp */
#define ICE_T0	(ICE_SAHSIZE+6)	/* t0 is used by the host as a temp */
#define ICE_T1	(ICE_SAHSIZE+7)	/* t1 is used by the host as a temp */
#define ICE_T2	(ICE_SAHSIZE+8)	/* t2 temp */
#define ICE_T3	(ICE_SAHSIZE+9)	/* t3 temp */
#define ICE_T4	(ICE_SAHSIZE+10)	/* t4 temp */
#define ICE_S0  (ICE_SAHSIZE+11)	/* pointer to instr_buffer */
#define ICE_RA	(ICE_SAHSIZE+12)	/* ra is needed for bal/jal instrs */
#define ICE_SIZE (ICE_SAHSIZE+13)
#define REG_MAP	0x80011ff6	/* gp regs in savearea */

#define UART_BASE                  0xb0008600
#define OFFSET_config              0x00 
#define OFFSET_divisor             0x08 
#define OFFSET_reset               0x10 
#define OFFSET_status              0x18 
#define OFFSET_rxbuff              0x20 
#define OFFSET_txbuff              0x28 
#define OFFSET_interval            0x30 
#define OFFSET_abrd                0x38 
#define OFFSET_channel             0x40 
#define UART_INTBIT	               SR_INT4


#define STATUS_DRDY_INT            0x00080000
#define STATUS_RDY                 0x00008000
#define STATUS_TXEMPTY             0x00000002
#define STATUS_TXALMOSTFULL        0x01000000
#define BRG_PS                     0x00100000

/***********
*  GPIO
***********/
#define GPIO_BASE 	                0xb000ad00
#define GPIO_CONFIG                 GPIO_BASE  		# Configuration Register
#define GPIO_ODR                    0xb000ad10		# Output Data Register
#define GPIO_IDR	                0xb000ad18		# Data Input Register

/*****************************
* GPIO Config Register Bits
*****************************/
#define GPIO_LB      0x00000100
#define GPIO_GPEN    0x00000200
#define GPIO_RST     0x00000400

/*****************************
* GPIO EN/OUT LED Bits
*****************************/
#define GPIO_LED_1      0x00000800
#define GPIO_LED_2      0x00000400
#define GPIO_LED_3      0x00000200
#define GPIO_LED_4      0x00000100
#define GPIO_LED_5      0x00000080
#define GPIO_LED_ALL    GPIO_LED_1|GPIO_LED_2|GPIO_LED_3|GPIO_LED_4|GPIO_LED_5


/*******************************
* GPIO Data Input Register Bits
*******************************/
#define	GPIO_DIP_1	0x00000008	# GPIO 3 is Dipswitch 1
#define	GPIO_DIP_2	0x00000010	# GPIO 4 is Dipswitch 2
#define	GPIO_DIP_3	0x00000020	# GPIO 5 is Dipswitch 3
#define	GPIO_DIP_4	0x00000040	# GPIO 6 is Dipswitch 4

#define J_RA_INSTR	0x03e00008
#define IBUFSIZE	100

	.comm instr_buffer,IBUFSIZE*4
	.comm savearea,ICE_SIZE*4

/*************************************************************
*  reset_vector:
*	This is where execution starts.
*	A maximum of 64 instructions allowed in this section
*/
	.globl _start
_start:
reset_vector: # bfc00000
    b       cpu_init

    .align  4
    .ascii  "Serial Ice 1B 12/22/1998"

/*************************************************************
* 		Start of interrupt-level code		     *
*************************************************************/
	.set noat
    .align 8
    	addiu	zero,zero,0	# nop to make the 2nd align work
    .align 7

	.globl gen_vector
	.ent gen_vector
gen_vector: # bfc00180
	# save regs

    la      k0, 1f
    li      k1, 0xa0000000
    or      k0,k1
    j       k0

1:

	la	    k0,savearea
	sw	    AT,ICE_AT*4(k0)
	.set at
	sw	    v0,ICE_V0*4(k0)
	sw	    a0,ICE_A0*4(k0)
	sw	    a1,ICE_A1*4(k0)
	sw	    a2,ICE_A2*4(k0)
	sw	    a3,ICE_A3*4(k0)
	sw	    t0,ICE_T0*4(k0)
	sw	    t1,ICE_T1*4(k0)
	sw	    t2,ICE_T2*4(k0)
	sw	    t3,ICE_T3*4(k0)
	sw	    t4,ICE_T4*4(k0)
	sw	    s0,ICE_S0*4(k0)
	sw	    ra,ICE_RA*4(k0)
	.set noreorder
	mfc0	t0,C0_EPC
	nop
	.set reorder
	sw	    t0,ICE_EPC*4(k0)

	# init s0 (KSEG1)
	la	    s0,instr_buffer

	# Is it my int?
	# read the CAUSE register
	.set noreorder
	mfc0	a0,C0_CAUSE
	nop
	.set reorder

	# hw int?
	and	    t0,a0,CAUSE_EXCMASK
	bne     t0,zero,send_ack	# br if not hw int

	and	    t0,a0,UART_INTBIT
	beq     t0,zero,send_ack	# br if not Uart hw int


	# make sure that this is a *real* attn byte
	# read the byte

	li	    k1,UART_BASE
    lw      k0,OFFSET_status(k1)

	and	    k0,STATUS_RDY
    beq     k0,zero,send_ack

	lw 	    k0,OFFSET_rxbuff(k1)

	li	    k1,ATTN
	bne	    k0,k1,restore_rfe	# br if not an attn byte

	# fall thru
	.end gen_vector

/*************************************************************
*/
	.globl send_ack
	.ent send_ack
send_ack:
	li	    k1,UART_BASE

	# make sure that the tx is ready
1:	lw 	    k0,OFFSET_status(k1)
	and	    k0,STATUS_TXALMOSTFULL
	bne	    k0,zero,1b

	li	    k0,ACK
	sw	    k0,OFFSET_txbuff(k1)
    
	# make sure that r8 and r9 are zero.
	li	    t0,0
	li	    t1,0
	# fall thru
	.end send_ack
	
/*************************************************************
*  ice_loop:
*	This is the main loop. We get words and process them.
*	There are 3 special types of word. 
*		1. RUN_MODE - transfer control to the customer's program.
*		2. SENDSAP  - Send the address of the savearea
*		3. SENDA0   - Execute the code in instr_buffer and send
*		              the value of register a0.
*	All other values are added to the instr_buffer.
*/
	.globl ice_loop
	.ent ice_loop
ice_loop:
	bal	    get_cmd

   	# check for SENDA0
	li	    k1,SENDA0
	bne	    k1,v0,1f

	# It is SENDA0. Execute the code in instr_buffer and send 
	# the value of register a0.
	# Make sure that the routine ends with a "j ra".
	sw	    zero,(s0)
	li	    k0,J_RA_INSTR
	sw	    k0,4(s0)
	sw	    zero,8(s0)
	# Make sure that the writes complete before the jal.
	.set noreorder
	nop
	nop
	nop
	.set reorder
	# Reset s0 to point to start of instr_buffer.
	la	    s0,instr_buffer
	jal	    s0		# execute instr_buffer

	bal	    put_word	# send A0
	b	    ice_loop

1:	# check for RUN_MODE
	li	    k1,RUN_MODE
	bne	    k1,v0,1f

restore_rfe:
	# It is RUN_MODE. Transfer control to the client.
	# restore regs

	la	    k0,savearea
	.set noat
	lw	    AT,ICE_AT*4(k0)
	lw	    v0,ICE_V0*4(k0)
	lw	    a0,ICE_A0*4(k0)
	lw	    a1,ICE_A1*4(k0)
	lw	    a2,ICE_A2*4(k0)
	lw	    a3,ICE_A3*4(k0)
	lw	    t0,ICE_T0*4(k0)
	lw	    t1,ICE_T1*4(k0)
	lw	    t2,ICE_T2*4(k0)
	lw	    t3,ICE_T3*4(k0)
	lw	    t4,ICE_T4*4(k0)
	lw	    s0,ICE_S0*4(k0)
	lw	    ra,ICE_RA*4(k0)
	.set noreorder
	lw	    k0,ICE_EPC*4(k0)
	nop
	j	    k0		# jump to client
	rfe
	.set reorder
	.set at

1:	# check for SENDSAP
	li	    k1,SENDSAP
	bne	    k1,v0,1f

	# It is SENDSAP. Send address of savearea.
	la	    a0,savearea
	or	    a0,1		# indicate new format
	bal	    put_word
	b	    ice_loop

1:	# else. Not a special word.
	sw	    v0,(s0)		# save word in instr_buffer
	addu	s0,4		# ready for next word
	b	    ice_loop
	.end ice_loop

	.set at

/*************************************************************
*  get_cmd()
*	Get one word from the serial interface. The result goes
*	in v0.
*/
	.globl get_cmd
	.ent get_cmd
get_cmd:
	li	    k1,UART_BASE
	li	    a1,4			# get 4 bytes

	# wait for rxrdy
3:  lw      k0,OFFSET_status(k1)
	and	    k0,STATUS_RDY
    beq     k0,zero,3b

	# get the byte
    lw      k0,OFFSET_rxbuff(k1)

	# first byte?
	bne	    a1,4,2f			# br if not first byte

	# is the byte a wakeup?
	bne	    k0,ATTN,2f		# br if not a wakeup

	# wait for txrdy
1:  lw      k0,OFFSET_status(k1)
	and	    k0,STATUS_TXALMOSTFULL
    bne     k0,zero,1b

	# send an ack
	li	    k0,ACK
    sw      k0,OFFSET_txbuff(k1)
	b	    3b

2:	sll	    v0,8			# move word into position
	or	    v0,k0			# merge byte with word
	subu	a1,1			# bytecount--
	bne	    a1,zero,3b		# do next byte

	j	    ra
	.end get_cmd

/*************************************************************
*  get_word()
*       Get one word from the serial interface. The result goes
*       in v0.
*/
        .globl get_word
        .ent get_word
get_word:
    li      k1,UART_BASE
    li      a1,4

1:  lw      k0,OFFSET_status(k1)
    and	    k0,STATUS_RDY
    beq     k0,zero,1b

    lw      k0,OFFSET_rxbuff(k1)
    sll     v0,8
    or      v0,k0
    subu    a1,1
    bne     a1,zero,1b

    j       ra
    .end get_word

/*************************************************************
*  put_word()
*       Put one word to the serial interface. The word to be sent
*       comes from a0.
*/
        .globl put_word
        .ent put_word
put_word:
    li      k1,UART_BASE
    li      a1,4

1:  lw      k0,OFFSET_status(k1)
	and	    k0,STATUS_TXALMOSTFULL
    bne     k0,zero,1b

    sw      a0,OFFSET_txbuff(k1)
    srl     a0,8
    subu    a1,1
    bne     a1,zero,1b

    j       ra
    .end put_word

/*************************************************************
* 		End of interrupt-level code		     *
*************************************************************/


    .globl cpu_init
    .ent cpu_init
cpu_init:
/******************************************************************************
* Set the following in the Status Register (Coprocessor 0):                  
*   BEV      :	Exceptions in boot region (beginning of EEPROM - see below) 
*   CU0      :	Enable access to CP0 in user mode                            
******************************************************************************/
    .set noreorder
					                # clear the SW interrupt bits in the Cause Register
	mtc0	zero, C0_CAUSE			# only SW interrupt bits are writeable
	nop
    nop
	
	li	    a0, SR_CU0 | SR_BEV		# setup the Status register
	mtc0	a0, C0_SR			# load the CP0 register
	nop
	nop

    li	    a0, CCC_BGE | CCC_MUL | CCC_MAD | CCC_CMP
    mtc0    a0, C0_CCC		# load the CP0 configuration register
	nop
	nop
    .set reorder

/************************************************************
* enable PCI access to RAP registers
* this is necessary to avoid PC hangs when dipswitch 8 is off
*************************************************************/
    la	    t0,0xb0008160		# PCI Reg_Acc_En Register
	lw	    t1,0(t0)
	or	    t1,0x80
	sw	    t1,0(t0)		# set to allow snooping over PCI

/********************************************************************
* SDRAM power up sequence (via Memory Controller)
********************************************************************/

/****************************************
* Write Delay Config Register          *
****************************************/

	# Setup Delay Config Register to:	
	#	Sys/PLL Clock Delay    = 3.5 ns
	#       Read Clock Delay       = 0.0 ns
	#	Write Data Clock Delay = 0.0 ns

	#	SDRAM Clock Delay      = 0.5 ns
	
	li	t0,0x00000e01	# Fill t0 with Delay Config values
	li	t3,0xb0008204	# Delay Config Reg Addr -> t0
	sw	t0,0(t3)	# Set delay Config Register
	
	/*******************************************************************
	* Setup EEPROM, PHY, and SDRAM					   *
	*******************************************************************/
	
eeprom_pup:

/*********************************************************************
 * EEPROM Configuration (via Memory Controller)
 *********************************************************************/

	# Configure EPROM to:
                    #	Write High Time     [21:20] = 3 cycles
	#	Write Low Time      [19:16] = 4 cycles
	#	Read Recovery Time  [6:4]   = 3 cycle
	#	Read Time           [3:0]   = 4 cycle

	li	t0,0x00340034	# settings -> t0
	li	t1,0xb0008228	# EPROM Control Register -> t1
	sw	t0,0(t1)	# Configure EPROM

phy_pup:	

/*********************************************************************
 * PHY Configuration (via Memory Controller)
 *********************************************************************/

	# Configure PHY to:
	#	Mux                  [31] = 0  disabled
	#	Address Latch Enable [30:27] = 7 cycles
	#	Write High Time      [21:20] = 3 cycles
	#	Write Low Time       [19:16] = 5 cycles
	#	Read Recovery Time   [6:4] = 6 cycles
	#	Read Time            [3:0] = 10 cycles

	li	t0,0x3835006a	# Settings -> t0
	li	t1,0xb0008230	# PHY Control Register -> t1
	sw	t0,0(t1)	    # Configure PHY

sdram_pup:	

	# Setup for 80 MHz to 100 MHz
	# Set the SDRAM Control Reg to:
	#	LP   = OFF
	#	CL   = 3
	#       RC   = 10
	#       DP2  = 0
	#       RBS  = 1
    #	    BBD2 = 0
	# Set refresh rate to: 1250 cycles


	/****************************************
	* Write SDRAM control in Mem Controler *
	****************************************/

	li	t1,0x0030a100	# Fill t1 with SDRAM control
	li	t3,0xb0008208	# SDRAM Control Reg Addr -> t1
	sw	t1,0(t3)	# Store preferences -> SDRAM Control Reg
	
				# Precharge the SDRAM	
	li	t1,0x00000001	# 0x1 -> t0
	li	t3,0xb0008210	# Precharge Cmd Reg Addr -> t1
	sw	t1,0(t3)	# Do a precharge

	/***************************************
	* Write refresh rate to Mem Controler *
	***************************************/
	
	li	t2,0x000004e2	# Fill t2 with refresh rate
	li	t3,0xb0008200	# SDRAM Refresh Reg Addr -> t1
	sw	t2,0(t3)	# Set refresh rate

	/*********************************************************************
	* Have Mem Controler configure SDRAM to agree with above parameters *
	*********************************************************************/

	# Set the Mode Reg to:
	#	Burst Length = 8
	#	Wrap Type	= 0
	#	Latency Mode	= 3

	li	t0,0x00000033	# t0 <- 0x33 
	li	t1,0xb0008218	# SDRAM Mode Set Reg -> t1
	sw	t0,0(t1)	    # Set Mode Reg (really doesn't matter what this value is)

	# Refer to tech spec info on Mem Controller and NEC
	#  SDRAM part uPD4516821

	li	t0,0x00000033	# t0 <- 0x33
	li	t1,0xa0019800	# You must write to this addr in order to set the Mode reg to 0x33
	sw	t0,0(t1)	# Set Mode Reg

	/**********************************
	* Refresh all SDRAM banks twice  *         
	**********************************/
	
	# Do first refresh of SDRAM

	li	t0,0x00000001	# t0 <- 0x1
	li	t1,0xb0008220	# SDRAM Refresh Cycle Cmd Reg Addr -> t1
	sw	t0,0(t1)	    # Refresh SDRAM

    # Do second refresh of SDRAM
	#  Note: must read value first to clear write back buffer

	lw	t3,0(t1)	# SDRAM Refresh Cycle Cmd Reg -> t3
	li	t0,0x00000002	# t0 <- 0x2
	sw	t0,0(t1)	# Refresh SDRAM


    /* 
        Setup Serial Ice Port
    */

    li      t1, UART_BASE

    li      t0, 0xfffffffe              #Take Uart out of reset
    sw      t0, OFFSET_reset(t1)

    li      t0, 0x00000008              #Turn loopback off, Select system clock
    sw      t0, OFFSET_config(t1)

    li      t0, STATUS_DRDY_INT
    sw      t0, OFFSET_status(t1)

    li      t0, 0x000000e3              #1 stopbit, no parity 8 data bits, ignor cts, ignor dsr, ignor dcd
    sw      t0, OFFSET_channel(t1)

    li      t0, 43                      # Select 115200 Baud
    sw      t0, OFFSET_divisor(t1)

    sw      zero, OFFSET_reset(t1)      # Take out of Tx and Rx Halt

	la	    k0,savearea
	la	    t0,get_word
	sw	    t0,ICE_GWP*4(k0)
	la	    t0,put_word
	sw	    t0,ICE_PWP*4(k0)
	li	    t0,IBUFSIZE
	sw	    t0,ICE_IBS*4(k0)
	li	    t0,REG_MAP
	sw	    t0,ICE_MAP*4(k0)
	li	    t0,SA_VERS
	sw	    t0,ICE_SAV*4(k0)
	li	    t0,ICE_SAHSIZE
	sw	    t0,ICE_SAH*4(k0)
	sw	    zero,ICE_LE*4(k0)

	# enable ints in SR MASK+IEC
	li	    k0,(SR_CU0 | SR_BEV | SR_IEC|UART_INTBIT)
	.set noreorder
	mtc0	k0,C0_SR
    nop
    nop
	.set reorder
	# print banner

    li      a0,0x44434241   # DCBA
    bal     put_word

1:
    b       1b

    .end cpu_init

