/*************************************************************
* File: k4011.s
* Author: Phil Bunce (pjb@carmel.com)
*
* Revision History:
*	970129	Initial release
*	970303	Brought ice port defs upto date. Added RXC etc.
*	970306	Changed to word ld and st ops to iceport.
*	970306	Added _start entry. Changed old _start to cstartup
*	970306	Added IBS to save area
*	970311	Changed bne to beq in test of xxRDY
*	970311	Added "and" with TXRDY to send_ack
*	970421	Change from Int5 to Int4
*	970829	Removed call to CpuInit in cstartup.
*	971113	Added wakeup banner (ABCD)
*	971114	Changed ICEPORT addr to b0ff.0000
*	971114	Added USE_NO_INTS switch
*	980312	Switched to version2 SAVEAREA format
*	980130	Removed savearea and instr_buffer from bss.
*
* This file contains the PROM resident code (the IceKernel) necessary to
* permit programs to be debugged using LSI Logic's SerialIce.  This
* implementation is for the 4011 with the SerialIce Port.
*
* The code in this module executes in kseg1 (non cacheable), leaves
* BEV=1, and does not initialize the caches.
*
* Example compile/link command for this file:
*
*     pmcc -crt0 -prom -syms -T bfc00000 -o k4011 k4011.s
*
* You can either merge your application with this file and include it in
* the PROM, or download your application into RAM and execute it there.
*
* To merge this file with you application. Use the following command line.
*
*     pmcc -prom -board bdmr4011 -o myprog myprog_files...
*
* To download your application into RAM you should compile/link your 
* program using the following command.
*
*     pmcc -board bdmr4011 -o myprog myprog_files...
*
* This file contains the following blocks of code:
*	reset_vector - The start of execution
*	utlb_vector  - UTLB exception handler
*	gen_vector   - Handler for all other exceptions
*	ice_loop     - Main loop of ICE
*	get_word     - Get one word from serial interface
*	put_word     - Put one word to serial interface
*	cpu_init     - Perform CPU-specific initialization
*
* This module requires the following areas of RAM:
*	INSTR_BUFFER - This is where the instructions that have been
*		       received from the host are saved.
*	SAVEAREA     - This is where I save the context of the
*		       downloaded program.
*	
*/
/*#define RUN_APPLICATION	/* run application on power-up */
/*#define USE_NO_INTS	/* don't use ints for kernel wakeup */

#ifndef LR4010
#define LR4010
#endif

#include <mips.h>

/* needed for r4000 mode. I could use -mips2 -- but why bother */
#define eret	.word 0x42000018	/* not known by cc2.11.2 */

/* Commands that are sent by the IceController */
#define SENDA0		0x12345678	/* execute INSTR_BUFFER */
#define RUN_MODE	0x87654321	/* run application */
#define SENDSAP		0xDEADBEEF	/* send SAVEAREA pointer */
#define ATTN		0x55		/* transfer control to IceKernel */
#define ACK		0xaa		/* reply to ATTN */

/* Offsets into the SAVEAREA */
#define SA_VERS 2
#define ICE_SAV	0	/* save area version */
#define ICE_SAH 1	/* save area header size */
#define ICE_MAP	2	/* bit-map for SAVEAREA */
#define ICE_IBS 3	/* size of instr buffer */
#define ICE_GWP 4	/* pointer to get_word routine */
#define ICE_PWP 5	/* pointer to put_word routine */
#define ICE_EPC	6	/* saved so that it can be set */
#define ICE_LE  7	/* set if little endian */
#define ICE_SAHSIZE 8	/* size of save area header */
/* end of header. The remainder is kernel-specific */
#define ICE_AT	(ICE_SAHSIZE+0)	/* v0 is used to hold the value received */
#define ICE_V0	(ICE_SAHSIZE+1)	/* v0 is used to hold the value received */
#define ICE_A0	(ICE_SAHSIZE+2)	/* a0 is used to hold the value to be sent */
#define ICE_A1	(ICE_SAHSIZE+3)	/* a1 is used as a temp */
#define ICE_A2	(ICE_SAHSIZE+4)	/* a1 is used as a temp */
#define ICE_A3	(ICE_SAHSIZE+5)	/* a1 is used as a temp */
#define ICE_T0	(ICE_SAHSIZE+6)	/* t0 is used by the host as a temp */
#define ICE_T1	(ICE_SAHSIZE+7)	/* t1 is used by the host as a temp */
#define ICE_T2	(ICE_SAHSIZE+8)	/* t2 temp */
#define ICE_T3	(ICE_SAHSIZE+9)	/* t3 temp */
#define ICE_T4	(ICE_SAHSIZE+10)	/* t4 temp */
#define ICE_S0  (ICE_SAHSIZE+11)	/* pointer to INSTR_BUFFER */
#define ICE_RA	(ICE_SAHSIZE+12)	/* ra is needed for bal/jal instrs */
#define ICE_SIZE (ICE_SAHSIZE+13)
#define REG_MAP	0x80011ff6	/* gp regs in SAVEAREA */

#define UART_BASE 	0xb0ff0000
#define UART_RXS	0x0	/* rx status */
#define UART_RXC	0x0	/* rx control */
#define UART_RXHR	0x4	/* tx holding reg */
#define UART_TXS	0x8	/* tx status */
#define UART_TXHR	0xc	/* tx holding reg */
#define UART_INTBIT	SR_INT4	/* 971115 */

#define RXS_RXRDY	(1<<0)	/* rx ready */
#define RXS_OVR		(1<<1)	/* rx overrun */
#define RXC_IE		(1<<0)	/* interrupt enable */
#define TXS_TXRDY	(1<<0)	/* tx ready */

#define J_RA_INSTR	0x03e00008
#define SAVEAREA	0xa0000000
#define INSTR_BUFFER	0xa0000280
#define IBUFSIZE	(0xa0000400-INSTR_BUFFER)


/*************************************************************
*  reset_vector:
*	This is where execution starts.
*	A maximum of 64 instructions allowed in this section
*/
	.globl _start
_start:
reset_vector: # bfc00000
	bal	cpu_init

	# make sure the sw bits of the CAUSE register are zero
	.set noreorder
	mtc0	zero,C0_CAUSE		
	.set reorder

	# enable ints in SR MASK+IEC
	li	k0,(SR_BEV|SR_IEC|UART_INTBIT)
	.set noreorder
	mtc0	k0,C0_SR
	.set reorder

	la	k0,SAVEAREA
	la	t0,get_word
	sw	t0,ICE_GWP*4(k0)
	la	t0,put_word
	sw	t0,ICE_PWP*4(k0)
	li	t0,IBUFSIZE
	sw	t0,ICE_IBS*4(k0)
	li	t0,REG_MAP
	sw	t0,ICE_MAP*4(k0)
	li	t0,SA_VERS
	sw	t0,ICE_SAV*4(k0)
	li	t0,ICE_SAHSIZE
	sw	t0,ICE_SAH*4(k0)
#ifdef MIPSEB
	li	t0,0
#else
	li	t0,1
#endif
	sw	t0,ICE_LE*4(k0)

	# print banner
        li      a0,0x44434241   # DCBA
        bal     put_word

	# wait here for the host to speak to me
#ifdef USE_NO_INTS 
   2:
	li	k1,UART_BASE

	# wait for rxrdy
   1:	lw 	k0,UART_RXS(k1)	
	and	k0,RXS_RXRDY
	beq	k0,zero,1b

	# make sure that this is a *real* attn byte
	# read the byte
	lw 	k0,UART_RXHR(k1)

#if 0   /* debug */
	not	k0

	# write the byte
	sw	k0,UART_TXHR(k1)

	b 	2b
#endif
	li	k1,ATTN
	bne	k0,k1,2b	# brif not an attn byte

	# init s0 (KSEG1)
	li	s0,INSTR_BUFFER
	b	send_ack
#else
#ifdef RUN_APP
	j	bspstart
#else
   1:	b	1b
#endif
#endif


/*************************************************************
* 		Start of interrupt-level code		     *
*************************************************************/
	.set noat

#ifdef R4KEXCEPTIONS
/*************************************************************
*  tlbrefill_vector:
*	We should never get one of these. But just in case.
*/
	.align 9
tlbrefill_vector: # bfc00200
	b	gen_vector

/*************************************************************
*  debug_vector:
*	We should never get one of these. But just in case.
*/
	.align 6
debug_vector: # bfc00240
	b	gen_vector
	.align 8
	# bfc00300
	.set noreorder; nop; .set reorder

#else /* R3000 mode */
/*************************************************************
*  utlb_vector:
*	We should never get one of these. But just in case.
*/
	.align 8
utlb_vector: # bfc00100
	b	gen_vector
#endif

/*************************************************************
*  gen_vector:
*	All the exceptions come through here.
*/
	.align 7
	.globl gen_vector
	.ent gen_vector
gen_vector: # bfc00180
	# make sure that we are in kseg1
	la	k0,1f
	li	k1,K1BASE
	or	k0,k1
	j	k0
   1:
	# save regs
	la	k0,SAVEAREA
	sw	AT,ICE_AT*4(k0)
	sw	v0,ICE_V0*4(k0)
	sw	a0,ICE_A0*4(k0)
	sw	a1,ICE_A1*4(k0)
	sw	a2,ICE_A2*4(k0)
	sw	a3,ICE_A3*4(k0)
	sw	t0,ICE_T0*4(k0)
	sw	t1,ICE_T1*4(k0)
	sw	t2,ICE_T2*4(k0)
	sw	t3,ICE_T3*4(k0)
	sw	t4,ICE_T4*4(k0)
	sw	s0,ICE_S0*4(k0)
	sw	ra,ICE_RA*4(k0)
	.set noreorder
	mfc0	t0,C0_EPC
	nop
	.set reorder
	sw	t0,ICE_EPC*4(k0)

	# init s0 (KSEG1)
	li	s0,INSTR_BUFFER

	# read the CAUSE register
	.set noreorder
	mfc0	a0,C0_CAUSE
	nop
	.set reorder

	# hw int?
	and	t0,a0,CAUSE_EXCMASK
	bne	t0,zero,send_ack	# brif not a hw int

	# It is a hw int. But is it my int?
	.set noreorder
	mfc0	t0,C0_SR
	nop
	.set reorder
	and	t0,a0		# qualify the CAUSE bits
	and	t0,UART_INTBIT
	beq	t0,zero,send_ack	# brif not mine

	# make sure that this is a *real* attn byte
	# read the byte
	li	k1,UART_BASE
	lw 	k0,UART_RXHR(k1)

	li	k1,ATTN
	bne	k0,k1,restore_rfe	# brif not an attn byte

	# fall thru
	.end gen_vector

/*************************************************************
*/
	.globl send_ack
	.ent send_ack
send_ack:
	li	k1,UART_BASE

	# make sure that the tx is ready
   1:	lw 	k0,UART_TXS(k1)
	and	k0,TXS_TXRDY
	beq	k0,zero,1b

	li	k0,ACK
	sw	k0,UART_TXHR(k1)

	# make sure that r8 and r9 are zero.
	li	t0,0
	li	t1,0
	# fall thru
	.end send_ack
	
/*************************************************************
*  ice_loop:
*	This is the main loop. We get words and process them.
*	There are 3 special types of word. 
*		1. RUN_MODE - transfer control to the customer's program.
*		2. SENDSAP  - Send the address of the SAVEAREA
*		3. SENDA0   - Execute the code in INSTR_BUFFER and send
*		              the value of register a0.
*	All other values are added to the INSTR_BUFFER.
*/
	.globl ice_loop
	.ent ice_loop
ice_loop:
	bal	get_cmd

#if 0 /* echo the input. Useful for debug */
	move	a0,v0
	bal	put_word
	b	ice_loop
#endif
   	# check for SENDA0
	li	k1,SENDA0
	bne	k1,v0,1f

	# It is SENDA0. Execute the code in INSTR_BUFFER and send 
	# the value of register a0.
	# Make sure that the routine ends with a "j ra".
	sw	zero,(s0)
	li	k0,J_RA_INSTR
	sw	k0,4(s0)
	sw	zero,8(s0)
	# Make sure that the writes complete before the jal.
	.set noreorder
	nop
	nop
	nop
	.set reorder
	# Reset s0 to point to start of INSTR_BUFFER.
	li	s0,INSTR_BUFFER
	jal	s0		# execute INSTR_BUFFER
	bal	put_word	# send A0
	b	ice_loop

   1:	# check for RUN_MODE
	li	k1,RUN_MODE
	bne	k1,v0,1f

restore_rfe:
	# It is RUN_MODE. Transfer control to the client.
	# restore regs
	la	k0,SAVEAREA
	lw	AT,ICE_AT*4(k0)
	lw	v0,ICE_V0*4(k0)
	lw	a0,ICE_A0*4(k0)
	lw	a1,ICE_A1*4(k0)
	lw	a2,ICE_A2*4(k0)
	lw	a3,ICE_A3*4(k0)
	lw	t0,ICE_T0*4(k0)
	lw	t1,ICE_T1*4(k0)
	lw	t2,ICE_T2*4(k0)
	lw	t3,ICE_T3*4(k0)
	lw	t4,ICE_T4*4(k0)
	lw	s0,ICE_S0*4(k0)
	lw	ra,ICE_RA*4(k0)
	.set noreorder
	lw	k0,ICE_EPC*4(k0)
	nop
#ifdef R4KEXCEPTIONS
	mtc0	k0,C0_EPC
	nop
	nop
	nop
	eret
	nop
#else
	j	k0		# jump to client
	rfe
#endif
	.set reorder

   1:	# check for SENDSAP
	li	k1,SENDSAP
	bne	k1,v0,1f

	# It is SENDSAP. Send address of SAVEAREA.
	la	a0,SAVEAREA
	or	a0,1		# indicate new format
	bal	put_word
	b	ice_loop

   1:	# else. Not a special word.
	sw	v0,(s0)		# save word in INSTR_BUFFER
	addu	s0,4		# ready for next word
	b	ice_loop
	.end ice_loop

	.set at

/*************************************************************
*  get_cmd()
*	Get one word from the serial interface. The result goes
*	in v0.
*/
	.globl get_cmd
	.ent get_cmd
get_cmd:
	li	k1,UART_BASE
	li	a1,4			# get 4 bytes

	# wait for rxrdy
   3:   lw      k0,UART_RXS(k1)
	and	k0,RXS_RXRDY
        beq     k0,zero,3b

	# get the byte
        lw      k0,UART_RXHR(k1)

	# first byte?
	bne	a1,4,2f			# brif not first byte

	# is the byte a wakeup?
	bne	k0,ATTN,2f		# brif not a wakeup

	# wait for txrdy
   1:   lw      k0,UART_TXS(k1)
	and	k0,TXS_TXRDY
        beq     k0,zero,1b

	# send an ack
	li	k0,ACK
        sw      k0,UART_TXHR(k1)
	b	3b

   2:	sll	v0,8			# move word into position
	or	v0,k0			# merge byte with word
	subu	a1,1			# bytecount--
	bne	a1,zero,3b		# do next byte

	j	ra
	.end get_cmd

/*************************************************************
*  get_word()
*       Get one word from the serial interface. The result goes
*       in v0.
*/
        .globl get_word
        .ent get_word
get_word:
        li      k1,UART_BASE
        li      a1,4

   1:   lw      k0,UART_RXS(k1)
	and	k0,RXS_RXRDY
        beq     k0,zero,1b

        lw      k0,UART_RXHR(k1)
        sll     v0,8
        or      v0,k0
        subu    a1,1
        bne     a1,zero,1b

        j       ra
        .end get_word

/*************************************************************
*  put_word()
*       Put one word to the serial interface. The word to be sent
*       comes from a0.
*/
        .globl put_word
        .ent put_word
put_word:
        li      k1,UART_BASE
        li      a1,4

   1:   lw      k0,UART_TXS(k1)
	and	k0,TXS_TXRDY
        beq     k0,zero,1b

        sw      a0,UART_TXHR(k1)
        srl     a0,8
        subu    a1,1
        bne     a1,zero,1b

        j       ra
        .end put_word

/*************************************************************
* 		End of interrupt-level code		     *
*************************************************************/
	.set at


/*************************************************************
*  cpu_init()
*	This is where the CPU-specific init code lives.
*	This implementation is for the Nitro (4010/11 eval board).
*	This example is designed to use ChanA of the DUART for connection
*	to the IceController.
*/
	.globl cpu_init
	.ent cpu_init
cpu_init:
	# setup the CCC configuration register
	# enable: CMP, IIE, DIE, MUL, MAD, BGE, IPWE(1K), WB
	# Icache: 2 way set assoc, 8K set size
	# Dcache: 2 way set assoc, 8K set size
	# CCC <- 0000 0001 1111 0111 1111 1110 0010 0000
#ifdef R4KEXCEPTIONS
	li	t0, 0x0037fe20		# w/o IIE,DIE
#else
	li	t0, 0x0137fe20		# w/o IIE,DIE
#endif
	.set noreorder
	mtc0	t0, C0_CCC		# load the CP0 configuration register
	.set reorder

	# Initialize the Synchronous DRAM controller
	li 	t0, M_DRAM_CFG
	li 	t1, 0x7033A753
	sw	t1, (t0)

	# set the CAS Latency
        li      t0, M_CL3		# CAS latency = 3
        sw      zero, (t0)

	# set Refresh rate
	li 	t0, M_DRAM_REFRESH
	li 	t1, 0x00000271		# at 40 Mhz 
	sw	t1, (t0)

	# enable scbus bus error detection
	li	t0,M_SCBUS_ST
	li	t1,3
	sw	t1,(t0)

#ifndef USE_NO_INTS
	# enable the RXINTS
        li      t1,UART_BASE
	li	t0,RXC_IE
	sw	t0,UART_RXC(t1)
#endif

	 j	ra
	.end cpu_init


#ifdef RUN_APPLICATION

#define RAM_GENVECT	0x80000080
#define STACKSIZE	(8*1024)
	.comm	stack,STACKSIZE
	.comm   flush_cache_ptr,4

/*************************************************************
*/
	.globl _exit
	.globl cstartup
	.ent cstartup
cstartup:
        # identify CPU and flush both caches
	la	s0,r4010_flush
        or      s0,K1BASE
        li      a0,ICACHEI
        jal     s0
        li      a0,DCACHEI
        jal     s0

#if 1	/* necessary for ROM-resident code */
	# copy the data to RAM
	li	a0,FDATA
	jal	cpdata
#endif

	# note that clrbss must not use a3
	jal	clrbss

	# set the global data pointer
	la	gp,_gp

	# save the address of the cache flush routine
	sw	s0,flush_cache_ptr

	# set sp
	la	sp,stack+STACKSIZE-24

#if 1	/* use the RAM-resident exception vectors */
	# copy exception handler to RAM vector
	la	t1,ehandler
	la	t2,ehandler_end
	li	t3,RAM_GENVECT	
   1:	lw	t0,(t1)
	sw	t0,(t3)
	addu	t1,4
	addu	t3,4
	bne	t1,t2,1b

	# flush the Icache
        li      a0,ICACHE
        jal     flush_cache

	# clear BEV
	.set noreorder
	mfc0	t0,C0_SR
	nop
	and	t0,~SR_BEV
	mtc0	t0,C0_SR
	.set reorder
#endif

 	# call the main C routine
	la	t0,main
	jal	t0
_exit:
	b	_exit
	.end cstartup


/*************************************************************
*  getmachtype()
*	Return the CPU type.
*/
	.globl getmachtype
	.ent getmachtype
getmachtype:
	li	v0,4011
	j	ra
	.end getmachtype


/*************************************************************
*  ehandler:
*	This is the exception handler that gets copied to RAM.
*	If the application uses exceptions, it will use this
*	code to transfer control to the IceKernel for all 
*	non-application exceptions.
*/
	.ent ehandler
ehandler:
	la	k0,gen_vector
	j	k0
ehandler_end:
	.end ehandler

/*************************************************************
*  flush_cache(type)
*	A C-callable routine to flush the caches.
*/
	.globl flush_cache
	.ent flush_cache
flush_cache:
	lw	t0,flush_cache_ptr
	j	t0
	.end flush_cache

#endif /*  RUN_APPLICATION */
