/******************************************************************/
/* Copyright (c) 2002,2003 BRECIS Communications                  */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/
/*************************************************************
 * File: lib/features.c
 * Purpose: Identify BRECIS chip features.
 *          Creates an environment variable with the list of features.
 * Revision History:
 */
#ifndef BRECIS5000
#define BRECIS5000
#endif

#include <mips.h>
#include <utypes.h>
#include <brecis.h>

static char macGroup0[4] = { ENET_MII, ENET_RMII, ENET_MII, ENET_RMII };

static char macGroup1[4] = {
  ENET_MII, ENET_RMII, ENET_RMII, ENET_RMII };

static char macGroup2[4] = {
  ENET_MII, ENET_MII, ENET_MII, ENET_RMII };

static char macGroup3[4] = {
  ENET_MII, ENET_MII, ENET_RMII, ENET_RMII };

static char macGroup4[4] = {
  FEATURE_NOEXIST, FEATURE_NOEXIST, ENET_RMII, ENET_RMII };

#define macC2007		macB2005

static void addFeature(char *features, char key, char value)
{
  int i = strlen(features);

  features[i++] = key;
  features[i++] = value;
  features[i] = '\0';
}

static void addZeusFeatures(char *features, unsigned long deviceid)
{
	unsigned long	miiIndex = (deviceid & CPU_DEVID_MIIMACMODE) >> 27;

	if (!(deviceid & (1<<17)))	/* If PCI available */
	{
		addFeature(features, PCI_KEY, PCI_HOST);
	}
	addFeature(features, ENET_KEY, macGroup0[miiIndex]);
	addFeature(features, ENET_KEY, FEATURE_NOEXIST);
	addFeature(features, ENET_KEY, macGroup3[miiIndex]);
	addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	addFeature(features, ENETTXD_KEY, FEATURE_NOEXIST);
	addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
}


static void addPoloFeatures(char *features, unsigned long deviceid, int isFPGA)
{
  char *macGroup;
  unsigned long subfamily = (deviceid & CPU_DEVID_SUBFAMILY) >> 16;
  unsigned long miiIndex = (deviceid & CPU_DEVID_MIIMACMODE) >> 27;

  switch(subfamily)
	{
	case POLO_SUBFAMILY_2005:
	  addFeature(features, SPAD_KEY, SPAD_POLO);
	  addFeature(features, ENET_KEY, macGroup1[miiIndex]);
	  addFeature(features, ENET_KEY, macGroup2[miiIndex]);
	  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  break;

	case POLO_SUBFAMILY_2007:
	  if (!(deviceid & CPU_DEVID_MIIPCIDISABLED))
		{
		  if (deviceid & CPU_DEVID_PCIHOST)
			{
			  addFeature(features, PCI_KEY, PCI_HOST);
			}
		  else
			{
			  addFeature(features, PCI_KEY, PCI_PERIPHERAL);
			}
		  addFeature(features, PCIMUX_KEY, 
					 (deviceid & CPU_DEVID_PCIMUXMODE) ? 
					 PCIMUX_SINGLE : PCIMUX_FULL);
		}
	  addFeature(features, SPAD_KEY, SPAD_POLO);
	  addFeature(features, ENET_KEY, macGroup1[miiIndex]);
	  addFeature(features, ENET_KEY, FEATURE_NOEXIST);
#if 0
	  if (! isFPGA)
#endif
	  addFeature(features, ENET_KEY, macGroup2[miiIndex]);
	  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  addFeature(features, ENETTXD_KEY, FEATURE_NOEXIST);
#if 0
	  if (! isFPGA)
#endif
		  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  break;

	case POLO_SUBFAMILY_2010:
	case POLO_SUBFAMILY_2020:
	case POLO_SUBFAMILY_2100:
	  if (deviceid & CPU_DEVID_MIIPCIDISABLED)
		{
		  macGroup = macGroup3;
		}
	  else
		{
		  if (deviceid & CPU_DEVID_PCIHOST)
			{
			  addFeature(features, PCI_KEY, PCI_HOST);
			}
		  else
			{
			  addFeature(features, PCI_KEY, PCI_PERIPHERAL);
			}
		  addFeature(features, PCIMUX_KEY, 
					 (deviceid & CPU_DEVID_PCIMUXMODE) ? 
					 PCIMUX_SINGLE : PCIMUX_FULL);
		  macGroup = macGroup4;
		}

	  switch (subfamily)
		{
		case POLO_SUBFAMILY_2010:
		  addFeature(features, SEC_KEY, SEC_SLOW);
		  break;

		case POLO_SUBFAMILY_2020:
		  addFeature(features, SEC_KEY, SEC_POLO);
		  addFeature(features, TDM_KEY, TDM_POLO);
		  break;

		case POLO_SUBFAMILY_2100:
		  addFeature(features, SEC_KEY, SEC_POLO);
		  break;

		}

	  addFeature(features, SPAD_KEY, SPAD_POLO);
	  addFeature(features, ENET_KEY, macGroup1[miiIndex]);
	  addFeature(features, ENET_KEY, macGroup[miiIndex]);
#if 0
	  if (! isFPGA)
#endif
		  addFeature(features, ENET_KEY, macGroup2[miiIndex]);
	  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
#if 0
	  if (! isFPGA)
#endif
		  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  break;

	default:
	  addFeature(features, SPAD_KEY, SPAD_POLO);
	  addFeature(features, ENET_KEY, macGroup1[miiIndex]);
	  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	}
}

void identify_features(void)
{
  char features[256];
  unsigned long deviceid = deviceID();
  unsigned long family = deviceid & CPU_DEVID_FAMILY;
  unsigned long revision = deviceid & CPU_DEVID_REVISION;

  features[0] = '\0';

  switch (family)
	{
	case FAMILY_FPGA:
	  if (FPGA_IS_ZEUS(revision))
	  	{
	  	addZeusFeatures(features, deviceid);
	  	}
	  else if (FPGA_IS_POLO(revision))
		{
		addPoloFeatures(features, deviceid, 1);
		}
	  else if (FPGA_IS_5000(revision))
		{
		  addFeature(features, ENET_KEY, ENET_MII);
		  addFeature(features, ENET_KEY, ENET_MII);
		  addFeature(features, SEC_KEY, SEC_TRIAD);
		  addFeature(features, TDM_KEY, TDM_TRIAD);
		  addFeature(features, ZSP_KEY, ZSP_TRIAD);
		}
	  else if (FPGA_IS_DUET(revision))
		{
		  addFeature(features, ENET_KEY, ENET_MII);
		  addFeature(features, ENET_KEY, ENET_MII);
		  addFeature(features, SEC_KEY, SEC_DUET);
		  addFeature(features, TDM_KEY, TDM_DUET);
		  addFeature(features, ZSP_KEY, ZSP_DUET);
		}
	  else
		{
		  addFeature(features, ENET_KEY, ENET_MII);
		}
	  break;

	case FAMILY_ZEUS:
	  addZeusFeatures(features, deviceid);
	  break;

	case FAMILY_POLO:
	  addPoloFeatures(features, deviceid, 0);
	  break;

	case FAMILY_DUET:
	  addFeature(features, ENET_KEY, ENET_MII);
	  addFeature(features, ENET_KEY, ENET_MII);
	  addFeature(features, ENET_KEY, ENET_MII);
	  addFeature(features, SEC_KEY, SEC_DUET);

	  if (revision == DUET_REVA)
	  {
			addFeature(features, TDM_KEY, TDM_DUET);
			addFeature(features, ZSP_KEY, ZSP_DUET);
	  }
#if 0
	  else
	  {
		  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
		  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
		  addFeature(features, ENETTXD_KEY, ENETTXD_RISING);
	  }
#endif
	  break;

	case FAMILY_TRIAD:
	  addFeature(features, ENET_KEY, ENET_MII);
	  addFeature(features, ENET_KEY, ENET_MII);
	  addFeature(features, SEC_KEY, SEC_TRIAD);
	  addFeature(features, TDM_KEY, TDM_TRIAD);
	  addFeature(features, ZSP_KEY, ZSP_TRIAD);
	  break;

	default:
	  addFeature(features, ENET_KEY, ENET_MII);
	}

  setenv(FEATURES, features);
}



