/*************************************************************
 * File: include/mon.h
 * Purpose: Principle header file for mon
 * Author: Phil Bunce (pjb@carmel.com)
 * Revision History:
 *	970216	Changed type Cmd to CmdRec, also added next.
 *	970217	Added externs for full_vers_info, monvers and libvers.
 *	970217	Added BREAKCODES. Removed spare1 and spare2 from CmdRec.
 *	970222	Added extern char toolsvers[] and incvers[].
 *	970303	Changed toolsvers to toolvers
 *	970304	Move NV_WRITE.. here from lib/nvram.c
 *	970311	Changed dbxmode to gdbmode
 *	970331	Moved BPTYPE_ code here from iceif.h (were ICE_TYPE_)
 *	970520	Added BP_W_XX and BP_E_XX codes
 *	971002	Changed incmod to >=
 *	971002	Changed decmod to mod-1
 *	980404	Added DL_MFLAG
 *	980725	Added compare command
 *	981214	Removed dcs from BrkList.
 */

#ifndef _MON_H_
#define _MON_H_

#if !defined(LANGUAGE_C) && !defined(LANGUAGE_ASSEMBLY)
#define LANGUAGE_C
#endif

#ifdef IMON95
#define printf xprintf
#define sprintf xsprintf
#endif

#include "mips.h"

#ifdef LANGUAGE_C
#include <string.h>
#include <setjmp.h>
#endif

#define MAX_BPT  32     /* max number of user breakpoints */
#define LINESZ  200	/* max command line length */
#define MAXREC  550	/* max length download rec (frecs are long!) */
#define MAX_AC	100	/* max args on command line */
#define PATSZ	100	/* max size of patterns */
#define HISTMAX 200	/* size of command history */

#ifndef CNTRL
#define CNTRL(x) 	(x & 0x1f)
#endif

#ifndef STDIN
#define STDIN	0	/* fd for stdin */
#endif

/* inbase values */
#define IB_AUTO 	0
#define IB_EIGHT 	1
#define IB_TEN 		2
#define IB_SIXTEEN 	3

/* inalpha values */
#define IA_HEX		0
#define IA_SYMBOL	1

/* trace_mode states */
#define TRACE_DC	4

#define WHEN_CONST  0
#define WHEN_MEM    1
#define WHEN_REG    2
#define WHEN_OPR    3
#define WHEN_NOT    4
#define WHEN_INV    5
#define WHEN_PREG   6
#define WHEN_OPR_LAND 	0
#define WHEN_OPR_LOR 	1
#define WHEN_OPR_EQ 	2
#define WHEN_OPR_NE 	3
#define WHEN_OPR_GE 	4
#define WHEN_OPR_LE 	5
#define WHEN_OPR_GT 	6
#define WHEN_OPR_LT 	7
#define WHEN_OPR_ADD 	8
#define WHEN_OPR_SUB 	9
#define WHEN_OPR_MUL 	10
#define WHEN_OPR_DIV 	11
#define WHEN_OPR_BAND   12
#define WHEN_OPR_BOR    13

/* for cpuOp(opc) */
#define CPU_CLKFREQ	1
#define CPU_REGLIST	2
#define CPU_NVINFO	3
#define CPU_REFRESH	4

/* type for read_target(TYPE,...)  and write_target(TYPE,...) */
#define XT_PC		0
#define XT_MEM		1
#define XT_GPR		2
#define XT_CP0		3
#define XT_CP1		4
#define XT_CP1C		5
#define XT_CP2		6
#define XT_CP2C		7
#define XT_CP3		8
#define XT_CP3C		9
#define XT_DBX		10  /* dbx core */

/* 970331 bpt types */
#define BPTYPE_PC	1
#define BPTYPE_DATA	2
#define BPTYPE_ITMP	3
#define BPTYPE_TRACE	4
#define BPTYPE_NONRT	5

/* 970401 bpt methods */
#define BRK_METHOD_RAM  1
#define BRK_METHOD_ROM  2
#define BRK_METHOD_HW   3

/* 970520 setbp_target return codes.
 * Errors are returned as (0-BP_E_XX) warnings are returned 
 * as ((BP_W_XX<<16)|bpid).
 */
/* warnings */
#define BP_W_MSK	0x101	/* Range feature is supported via mask */
#define BP_W_VAL	0x102	/* Value check is not supported in hardware */
#define BP_W_QAL	0x104	/* Requested quals are not supported in hw */
/* errors */
#define BP_E_BPT	0x200	/* No such breakpoint number */
#define BP_E_RGE	0x201	/* Range is not supported */
#define BP_E_QAL	0x202	/* The requested qualifiers can not be used */
#define BP_E_OUT	0x203	/* Out of hardware resources */
#define BP_E_NON	0x204	/* Hardware breakpoint not supported */
#define BP_E_VAL	0x205	/* value feature not supported */
#define BP_E_ERR	0x206	/* requested bpt can not be set */

#ifdef LANGUAGE_C

#include "utypes.h"

/* short forms */
#define getPc()       read_target(XT_PC,0,0)
#define putPc(v)      write_target(XT_PC,0,v,0)
#define getGpr32(n)   read_target(XT_GPR,n,0)
#define read_target8(a)	read_target(XT_MEM,a,1)
#define read_target16(a)	read_target(XT_MEM,a,2)
#define read_target32(a)	read_target(XT_MEM,a,4)
#define getRegVal32(p)	getRegVal(p)

#define getGpr(n)     read_target(XT_GPR,n,0)
#define putGpr(n,v)   write_target(XT_GPR,n,v,0)


#ifndef NOANSI
#define swap_word(v)	(\
	((((Ulong)(v))>>24)&0x000000ff) | \
	((((Ulong)(v))>> 8)&0x0000ff00) | \
	((((Ulong)(v))<< 8)&0x00ff0000) | \
	((((Ulong)(v))<<24)&0xff000000))
#define swap_half(v)	(\
	((((Ulong)(v))>>8)&0x00ff) | \
	((((Ulong)(v))<<8)&0xff00))
#endif


#define ICACHETAG       1
#define ICACHERAM       2
#define DCACHETAG       3
#define DCACHERAM       4


/* 981214 removed dcs member. Info now stored in aux[0]. */
/* Needed to make setbp_target less dbx-specific */
typedef struct BrkList {
        Uchar type;
        Uchar isset;
        Uchar method;
        Ulong addr;
        Ulong val;
	Ulong mask;
        Ulong aux[2];
        } BrkList;

/* nvRam info */
#define MAX_ERASE_BLOCKS	256	/* value could be up to 256 */

typedef struct NvEraseInfo {
	int	num_erase_blocks;
	struct {
		unsigned long sector_size;
		int num_sectors;
	} erase_block[MAX_ERASE_BLOCKS];
	} NvEraseInfo;

typedef struct NvType {
	int famcode;	/* device family code */
	char *name; 	/* device name */
	int size; 	/* total size in bytes */
	lFunc *driver;	/* start address of nv device driver */
	vFunc *edriver;	/* end address of nv device driver */
    NvEraseInfo *eraseInfo; /* erase info if not dynamically learned */
	} NvType; 

typedef struct NvInfo {
	char *name;	/* device name */
	Ulong start; 	/* base address of device */
	int width; 	/* 1,2 or 4 */
	int size; 	/* in bytes */
	int standard;	/* standard CFI/JEDEC */
	lFunc *driver;	/* address of nv device driver */
	Ulong dvrsa; 	/* start of driver save area */
	NvType *type;	/* pointer to nvType entry */
	} NvInfo;

#define NV_CFI			1		/* follow CFI standard */
#define NV_JEDEC		2		/* follow JEDEC standard */

#define NV_WRITE  1
#define NV_READ   2
#define NV_ERASE  3
#define NV_SERASE 4
#define NV_BLOCKWRITE 5
#define NV_SECTORSTATUS 6

#define NV_FLASHPROTECTED		0x100	/* read identifier protect bit */

#ifdef BRECIS5000
#define MAXFLASH 2
#else
#define MAXFLASH 1
#endif

#define MAXSYM 12
typedef struct Sym {
  struct Sym *aparent;
  struct Sym *alchild;
  struct Sym *archild;
  struct Sym *nparent;
  struct Sym *nlchild;
  struct Sym *nrchild;
  char *lname;
  unsigned int value;
  char name[MAXSYM];
} Sym;

/* A structure used for describing commands and their options. */
typedef struct Optdesc {
	char *name;	/* option list or option name */
	char *desc;	/* descriptive text */
	short flags; 	
	} Optdesc;

typedef struct CmdRec {
	char *name;
	Optdesc *opts;
	Func *func;
	struct CmdRec *next;
	} CmdRec;

typedef struct WhenRec {
	int tag;
	Ulong val;
	} WhenRec;

/* The array "Bps Bpt[]" is used to hold logical info about breakpoints.
 * The physical info is held in "BrkList brkList[]".
 * For more info see mon/go.c/setTrigger().
 */
typedef struct Bps {
	WhenRec *cond;	/* The condition list */
	WhenRec *csz;	/* The size of the condition list */
	char *cstr;	/* A string describing the condition */
	char *cmdstr;	/* The action list */
	} Bps;

typedef struct RegSpec { /* register field info */
	Uchar size;	/* size (width) of field */
	Uchar lsb;	/* position of field's lsb */
	char *name;	/* field name */
	Uchar base;	/* display base e.g. 2 8 10 16 */
	char **values; 	/* use this if base=0 */
	Uchar ro;	/* read only */
	} RegSpec;

typedef struct RegRec { /* register info */
	RvFunc *func;	/* function used to access register */
	RegSpec *spec;	/* field info */
	char *name;	/* primary name */
	char *aname;	/* alternate name */
	Ulong regnum;	/* register number */
	Ulong flags; 	
	int size;	/* size of the register (8,16,32) */
	RegVal prev;	/* to support the mr command */
	char modified;	/* flag a register modified */
	struct RegRec *next;
	} RegRec;

/* defs for RegRec flags field */
/* bits 0 thru 7 are architecture independent */
#define F_ANAME (1<<0)	/* print aname */
#define F_GPR	(1<<1)	/* gp reg */
#define F_WO	(1<<2)	/* write only */
#define F_RO	(1<<3)	/* read only */
#define F_FPR	(1<<4)	/* floating-point gen reg */
#define F_CPU	(1<<5)	/* pc hi & lo */
#define F_CP0	(1<<6)	/* basic cp0 regs */
#define F_FCR	(1<<7)	/* cp1 cntrl regs */
#define F_MIPS	(1<<8)	/* part of the basic mips core */

typedef struct EnvRec {
	char *name;
	char *init;
	char *values;
	Func *func;
	struct EnvRec *next;
	char *value;
	} EnvRec;

typedef struct OcmRec {
        struct OcmRec *next;
        Ulong size;	/* size of ocm */
        Ulong addr;	/* base address of ocm */
        Ulong *func;	/* pointer to function storage area */
	int funclen;	/* length of function (in words) */
	int regmap_ok;	/* no extra saves are necessary */
	int readonly;   /* the ocm is readonly */
	Ulong map;	/* the ocm's regmap */
        } OcmRec;

/* external data declarations */
extern int histno;	/* current history number */
extern char *searching;
extern CmdRec CmdTable[];
extern char line[];
extern char date[];
extern char vers[];
extern char prnbuf[];
extern Ulong topClientMem;
extern int vflag;
extern int noecho;
extern RegSpec mips_cause_def[];
extern RegSpec mips_sr_def[];
extern RegSpec mips_prid_def[];
extern RegSpec mips_config_def[];
extern int xvwmode;
extern int gdbmode;
extern struct envstr envvar[];
extern int bptTmpId;
extern int bptCount;
extern Bps Bpt[]; 
extern int cp1ok;
extern int tot;
extern char recbuf[];
extern int blksz,blkinx;
extern jmp_buf intrbuf;
extern int trace_mode;
extern char *excodes[];
extern RegRec gpreglist[];
extern RegRec *regChain;
extern int verbose;

#include <stdio.h>
extern FILE *dfp;

extern Ulong hostType, FormerPC;
extern NvInfo nvInfo[MAXFLASH];
extern char **regnames;
extern char **c0regNames;
extern char **c1dRegNames, **c1cRegNames;
extern char **c2dRegNames, **c2cRegNames;
extern char **c3dRegNames, **c3cRegNames;
extern char *regs_hw[], *regs_sw[];
extern Func *read_target_mem_ptr;
extern Func *write_target_mem_ptr;
extern int clkFrequency;
extern char full_vers_info[];
extern char monvers[];
extern char libvers[];
extern int mode_64bit;

/* function declarations */
Uchar *scan_byte();
Sym *findvalue();
char *getMonEnv();
char *findbang();
char *gethistn();
char *getexcname();
unsigned long getpchist();
Ulong cpuType();
Ulong dispmem();
Ulong dispfp();
RegRec *findRegRec();
RegVal mXmem(),mXgpr(),mXpc(),mXdbx();
RegVal mXc1(),cXc1();
RegVal cXc2(),mXc2();
RegVal mXc0();
int l4010op(),l33020op(),l33050op();
RegVal getRegVal();

#ifndef NOANSI
int adr2symoff(char *dst,Ulong value,int width);
Ulong dis16(char *prnbuf,Ulong addr,Ulong instr);
char *strccat(char *dst,char c);
int instr_size(Ulong adr,Ulong inst);
RegVal read_target(int type,Ulong addr,int sz);
#endif

int setbp(),copy(),dump(),fill(),cont(),go(),help(),dis(),modify();
int registers(),search(),trace(),clrbp(),load(), ping(), setup();
int dohi(),do_sym(),do_ls(),do_set(),no_cmd(),when();
int transp(), sdump(), debug(), compare_cmd();

extern Optdesc copy_opts[];	/* copy */
extern Optdesc search_opts[];	/* search */
extern Optdesc hi_opts[];	/* history */
extern Optdesc set_opts[];	/* set env var */
extern Optdesc sym_opts[];	/* define symbol */
extern Optdesc w_opts[];	/* when */
extern Optdesc b_opts[];	/* bpt */
extern Optdesc db_opts[];	/* bpt */
extern Optdesc g_opts[];	/* go */
extern Optdesc c_opts[];	/* cont */
extern Optdesc t_opts[];	/* trace */
extern Optdesc to_opts[];	/* trace over */
extern Optdesc m_opts[];	/* modify */
extern Optdesc d_opts[];	/* dump */
extern Optdesc ls_opts[];	/* list symbols */
extern Optdesc l_opts[];	/* list (disassemble) */
extern Optdesc r_opts[];	/* register */
extern Optdesc h_opts[];	/* help */
extern Optdesc fill_opts[];	/* fill */
extern Optdesc compare_opts[];	/* compare */
extern Optdesc tr_opts[];	/* tr */
extern Optdesc sdump_opts[];	/* dump */
extern Optdesc sh_opts[];
extern Optdesc more_opts[];	
extern Optdesc crossview_opts[]; /* for BSO's crossview */	
extern Optdesc debug_opts[];	/* for dbx and gdb */

#define getfield(w,s,p) ((((unsigned long)w)&(((1<<s)-1)<<p))>>p)

/* a well behaved strncpy (always null terminates the dest) */
#define strNcpy(x,y,z)	strncpy(x,y,z),(x)[z]=0

#define strequ(x,y)	((strcmp(x,y)==0)?1:0)

/* macros to increment and decrement x, modulus mod */
/* 971002 changed incmod to >= */
#define incmod(x,mod) (((x+1) >= mod)?0:x+1)
#define decmod(x,mod) (((x-1) < 0)?mod-1:x-1)

/* These definitions should be move to a new file.. string2.h */
/* definitions for fmt parameter of str_fmt(p,width,fmt) */
#define FMT_RJUST 0
#define FMT_LJUST 1
#define FMT_RJUST0 2
#define FMT_CENTER 3
char *strichr(),*strdchr(),*strposn(),*getword(), *cc2str();
char *strset(),*strrset(), *strbalp(),*strrpset();
int strlequ(),strpat();

#ifdef NOANSI
void mkcomment();
Ulong mips16_actual_branch_target_address();
#else
void mkcomment(char *p,char *fmt,Ulong v);
int mips16_is_extended(Ulong inst);
int mips16_needs_shift(Ulong inst);
int mips16_is_conditional_branch(Ulong addr,Ulong inst);
Ulong mips16_actual_branch_target_address(Ulong adr,Ulong inst);
int mips16_is_jal(Ulong addr,Ulong inst);
int mips16_is_jr(Ulong addr,Ulong inst);
int mips16_is_branch(Ulong addr,Ulong inst);
#endif

void dotik();
void addsym(),doval();
void clrbpt();

extern vpFunc *ether_driver_ptr;
void *sonic_driver();
void *am79970_driver();

/* The following is used when relocating pmon from flash to ram */

#endif /* LANGUAGE_C */

/* trace flags */
#define T_O	0x0001	/* trace over */
#define T_V	0x0002	/* verbose */
#define T_M	0x0004	/* multi */
#define T_B	0x0008	/* branches only in trace buffer */
#define T_C	0x0010	/* calls only in trace buffer */
#define T_I	0x0020	/* until pc invalid */
#define T_SP	0x0040	/* don't set sp */
#define T_W	0x0080	/* don't wait */
#define T_PN	0x0100	/* specify program name for argv[0] */


/* defs for RegRec flags field */
/* bits 8 thru 31 are available for architecture-specific info */
#define F_ALL	(F_33xxx|F_333xx|F_400x|F_401x)
#define F_33xxx (F_33000|F_33050|F_33x20)
#define F_333xx (F_33300|F_33310)
#define F_33x20 (F_33020|F_33120)
#define F_400x	(F_4001|F_4002)
#define F_401x	(F_4010)
#define F_3000  (1<<8)
#define F_33000 (1<<9)
#define F_33020 (1<<10)
#define F_33120 (1<<11)
#define F_33050 (1<<12)
#define F_33300 (1<<13)
#define F_33310 (1<<14)
#define F_4001  (1<<15)
#define F_4002  (1<<17)
#define F_4010  (1<<18)
#define F_64360 (1<<19)

/* dlecho values */
#define OFF		0
#define ON		1
#define LFEED		2

/* dlproto values */
#define NONE		0
#define XONXOFF		1
#define ETXACK		2

/* inbase values */
#define AUTO 		0
#define EIGHT 		1
#define TEN 		2
#define SIXTEEN 	3

/* inalpha values */
#define HEX		0
#define SYMBOL		1

/* download flags */
#define DL_sFLAG 0x01
#define DL_bFLAG 0x02
#define DL_eFLAG 0x04
#define DL_aFLAG 0x08
#define DL_TFLAG 0x10
#define DL_IFLAG 0x20
#define DL_BFLAG 0x40
#define DL_dFLAG 0x80
#define DL_MFLAG 0x100
#define DL_SFLAG 0x200

#define BPT_CODE	0x0000000d  /* this is what I use */
#define XVWBREAKCODE	0x03ffffcd
#define DBXBREAKCODE	0x0005000d /* not verified */
#define GDBBREAKCODE	0x0005000d
#define TINY_BPT_CODE	0xe805


#endif /* _MON_H_ */
