/******************************************************************/
/* Copyright (c) 2002 BRECIS Communications                       */
/*      This software is the property of BRECIS Communications    */
/*      and may not be copied or distributed in any form without  */
/*      a prior licensing arrangement.                            */
/*                                                                */
/* BRECIS COMMUNICATIONS DISCLAIMS ANY LIABILITY OF ANY KIND      */
/* FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS      */
/* SOFTWARE.                                                      */
/*                                                                */
/******************************************************************/

#ifndef _FLASHCONFIG_
#define _FLASHCONFIG_

#ifndef NULL
#define NULL	0
#endif

/* Format of pmon/bbload manufacturing flash configuration information */

typedef struct _flashConfigPtr {
  signed short offset;	/* Location of actual configuration information.
						 * This is a signed value, negative means the 
						 * actual configuration information is before
						 * the flashConfigPtr structure
						 */
  signed short length;	/* Length of actual configuration information */
  unsigned long crc;	/* CRC ISO 3309 [ISO-3309] or ITU-T V.42 [ITU-V42]:
						 * x^32+x^26+x^23+x^22+x^16+x^12+
						 * x^11+x^10+x^8+x^7+x^5+x^4+x^2+x+1
						 * and computed as follows:
						 * a) a buffer is constructed consisting of the
						 *    following:
						 *    1) flashConfigPtr.offset
						 *    2) flashConfigPtr.length
						 *    3) actual configuration information
						 * b) the CRC algorithm is run over this buffer
						 *    producing flashConfigPtr.crc
						 */
} flashConfigPtr;

/* Actual configuration information structure consists of key-length-value
 * multiples with the last key being the FLASHCONFIG_EOF key.
 * The key and the length are unsigned char.
 * The length does NOT include the key or length byte, but only the value.
 * A length of zero signifies no value.
 * A length must follow a key, even the FLASHCONFIG_EOF key must have a
 * length field of zero.
 */
#define	FLASHCONFIG_ZERO		0x00	/* this key is illegal */
#define FLASHCONFIG_SERIAL		0x01	/* board serial number */
#define FLASHCONFIG_ETHERMAC0	0x02	/* MAC0 ethernet MAC address */
#define FLASHCONFIG_ETHERMAC1	0x03	/* MAC1 ethernet MAC address */
#define FLASHCONFIG_ETHERMAC2	0x04	/* MAC2 ethernet MAC address */
#define FLASHCONFIG_ETHERPHY0	0x05	/* PHY0 address */
#define FLASHCONFIG_ETHERPHY1	0x06	/* PHY1 address */
#define FLASHCONFIG_ETHERPHY2	0x07	/* PHY2 address */
#define	FLASHCONFIG_EOF			0xff	/* end of configuration */

typedef unsigned char flashConfigKey;
typedef	unsigned char flashConfigLength;
typedef unsigned char flashConfigValue;

/* The following is based on rfc 2823, "8.1 CRC Generation" */

#define POLY32	0x4C11DB7

static __inline__ unsigned long crc32(unsigned long crcval, unsigned char cval)
{
  int i;

  crcval ^= cval << 24;

  for (i = 8; i--; )
	crcval = crcval & 0x80000000 ? (crcval << 1) ^ POLY32 : crcval << 1;
  return crcval;
}

static __inline__ unsigned long crc32_payload_begin(unsigned char *buffer, 
													int len)
{
  unsigned long crc;

  crc = 0xFFFFFFFFul;
  while (--len >= 0)
	crc = crc32(crc,*buffer++);
  return crc;
}

static __inline__ unsigned long crc32_payload_middle(unsigned long crc,
													 unsigned char *buffer, 
													 int len)
{
  while (--len >= 0)
	crc = crc32(crc,*buffer++);
  return crc;
}

static __inline__ unsigned long crc32_payload_end(unsigned long crc,
												  unsigned char *buffer, 
												  int len)
{
  while (--len >= 0)
	crc = crc32(crc,*buffer++);
  return crc ^ 0xFFFFFFFFul;
}

static __inline__ unsigned long crc32_payload(unsigned char * buffer, int len)
{
  return crc32_payload_end(crc32_payload_begin(buffer, 0), NULL, 0);
}

static __inline__ flashConfigPtr *validateManufConfig(flashConfigPtr *cptr)
{
		unsigned char *ptr;
		unsigned long crc;

		ptr = (unsigned char *) cptr;

		if (cptr->offset == ~0 && cptr->length == ~0)
		  return NULL;					/* if area erased */

		ptr += cptr->offset;

		crc = crc32_payload_begin((unsigned char *) cptr,
								  sizeof(signed short) +
								  sizeof(unsigned short));

		crc = crc32_payload_end(crc, (unsigned char *) ptr, cptr->length);

		if (crc == cptr->crc)
				return cptr;

		return NULL;
}

static flashConfigPtr *validateFlashConfig(unsigned char *flashConfigAddress)
{
		return validateManufConfig((flashConfigPtr *) flashConfigAddress);
}

static __inline__ int getFlashConfigValue(flashConfigPtr *flashConfig,
										  flashConfigKey key,
										  flashConfigValue **valuep,
										  flashConfigLength *valuelength)
{
		unsigned char *ptr;
		int index;

		ptr = (unsigned char *) flashConfig;

		ptr += flashConfig->offset;

		for (index = 0; index < flashConfig->length; index += ptr[index+1] + 2)
		{
				if (ptr[index] == FLASHCONFIG_ZERO ||
					ptr[index] == FLASHCONFIG_EOF)
				{
						break;	/* EOF or bad key */
				}

				if (ptr[index] == key)
				{
						*valuelength = ptr[index+1];
						*valuep = &ptr[index+2];

						return 0;		/* found */
				}
		}

		return 1;						/* not found */
} 

#endif /* _FLASHCONFIG_ */
