
#include <mips.h>
#include <defines.h>

typedef unsigned long U32;

#define	PREG_BASE	0xB8400000	/* Base address to Peripheral Reg  */
#define PER_BASE 	PREG_BASE

/**********************************************************/
/* GPIO registers										  */
/**********************************************************/
#define GPIO_DATA_REG	((volatile U32 *)(PER_BASE + 0xE0))	/* GPIO Data reg */
#define GPIO_CFG1_REG	((volatile U32 *)(PER_BASE + 0xE4)) /* GPIO CFG1 reg */
#define GPIO_CFG2_REG	((volatile U32 *)(PER_BASE + 0xE8))	/* GPIO CFG2 reg */

#define ERRORLEDMASK	0xff
#define ERRORDELAYCOUNT 100000000

/* nanodelay - Routine to perform short-duration delays using the MIPS
 *      count register.
 *
 * Written by: Mark Rustad, Brecis Communications, 07/31/01.
 */


/* In-line macro for COP0 count register access */

#define GetCOP0Count()  \
({ unsigned long __value;       \
        __asm__ volatile ("mfc0 %0, $9" : "=r" (__value));    \
        __value; })

#define OVERHEAD    3   /* Partial overhead adjustment */

static struct
{
    unsigned long   numerator;
    unsigned long   denominator;
}   nanoperiod;

/***************************************************************************
*
* NanoDelay - Delay for a short time.
*
* Input:
*  nanos - Number of nanoseconds to delay
*
* Notes:
*  NanoInit must have been called to establish the CPU clock rate
*  before calling this function.
*
*  The caller probably should have interrupts disabled before calling
*  this to have any expectation of real accuracy.
*
* RETURNS: N/A
*
*/

static inline void NanoDelay
	(
	unsigned int nanos
	)
	{
	unsigned long   count;
	long    difference;

	count = GetCOP0Count(); /* Get the count before doing calculations */
	count += ((nanos * nanoperiod.numerator) / nanoperiod.denominator)
		- OVERHEAD;
	do
		difference = count - GetCOP0Count();
	while (difference > 0);
	}



/***************************************************************************
*
* NanoInit - Set timing parameters for NanoDelay.
*
*
* Input:
*  freq - CPU clock frequency in Hz.
*
* Notes:
*  This routine takes the 2:1 ratio between the cpu clock and count
*  register into account, so the caller uses only the actual cpu clock
*  period to initialialize this routine.
*
* RETURNS: N/A
*
*/
static inline void NanoInit
	(
	)
	{
	unsigned long   denominator;
	unsigned int freq = cpuclockrate();

	/* The following provides for the 2:1 ratio between the cpu clock
	 * and the count register rate.
	 */
	denominator = 2000000000;

	/*
	 * Reduce the numbers. Note that because the denominator
	 * is a constant and only has prime factors of two
	 * and 5, we only need to divide them out. Cool, eh?
	 *
	 * Look at the object code to see how GCC gets all the quotients
	 * and remainders without using any divide instructions. Ah,
	 * just like on the Cyber...
	 */

	while ((denominator & 1) == 0 && (freq & 1) == 0)
		denominator >>= 1, freq >>= 1;

	while ((denominator % 5) == 0 && (freq % 5) == 0)
		denominator /= 5, freq /= 5;

	nanoperiod.numerator = freq;
	nanoperiod.denominator = denominator;
	}

static void initLED(int gpioPin)
{
    /* set the appropriate GPIO pins to be outputs */
    /* *GPIO_CFG1_REG controls GPIO 0-7 */
    /* *GPIO_CFG2_REG controls GPIO 8-15 */
    if (gpioPin < 8)
    {
        *GPIO_CFG1_REG &= ~(0xF << (gpioPin * 4));
        *GPIO_CFG1_REG |= 0x8 << (gpioPin * 4);
    }
    else
    {
        *GPIO_CFG2_REG &= ~(0xF << (gpioPin * 4));
        *GPIO_CFG2_REG |= 0x8 << (gpioPin * 4);
    }
}

static void turnOnLED(int gpioPin)
{
    *GPIO_DATA_REG |= 1 << gpioPin;    /* turns LED on */
}

static void turnOffLED (int gpioPin)
{
    *GPIO_DATA_REG &= ~(1 << gpioPin);    /* turns LED off */
}

static void dofunction(void (*func)(int gpioPin), U32 ledmask)
{
	int i;

	for (i = 0; i < 32; i++)
	{
		if (ledmask & (1<<i))
		{
			func(i);
			ledmask &= ~(1<<i);

			if (ledmask == 0)
				return;
		}
	}
}

/*************************************************************
*  exception()
*	An exception has been generated within PMON
*/
void exception()
{
	int i;
	int j;

	NanoInit();

	/* use minimal routines for printing */

	a5000StrOut("Exception occurred - register dump from DBGREG area ");
	a5000PrintWordReg((unsigned long) &DBGREG);
	a5000StrOut(CRLF);

	for (i = 0; i < NREGS; i = i + 4)
	{
		a5000PrintWordReg(i);
		a5000StrOut(": ");

		for (j = 0; j < 4; j++)
		{
			a5000PrintWordReg(DBGREG[i + j]);
			a5000StrOut(" ");
		}

		a5000StrOut(CRLF);
	}

	dofunction(initLED, ERRORLEDMASK);

	while (1)
	{
		dofunction(turnOnLED, ERRORLEDMASK);

		NanoDelay(ERRORDELAYCOUNT);

		dofunction(turnOffLED, ERRORLEDMASK);

		NanoDelay(ERRORDELAYCOUNT);
	}
}

