/**
 * @file csmsock.c
 *
 * Socket Routines for CSMan Agent -- Unix Domain Socket version.
 *
 * This file provides the following interfaces for CSMan Agent
 * to connect CSMan Server via Unix domain socket.
 *
 *	int csm_sock_init(char *svr_dir);
 *	int csm_sock_ask(int fd, char *qbuf, int qlen, char *abuf, int size);
 *	int csm_sock_term(int fd);
 *
 * Since it is used for CSMan Agent only, sanity checking is not so
 * necessary. Turn on DEBUG to do sanity checking for debugging.
 */

#include <stdio.h>
#include <unistd.h>		//for unlink
#include <stdlib.h>		//for mktemp
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <errno.h>

#include "csmprot.h"	
#include "csmsock.h"

#define MY_ADDR_PATH	"/tmp/csm-XXXXXX"

#ifndef DEBUG
#define DEBUG		0
#endif

/* session related data */
typedef struct {
	int csm_fd;	//fd to access unix domain socket (agent to socket)
	pid_t pid;
	int svr_addr_len;
	struct sockaddr_un svr_addr;
	char my_addr_path[sizeof(MY_ADDR_PATH)];	//file to be unlinked
} SESS_TAB;

static SESS_TAB sess[MAX_CONNECTION];

/* fd (agent to client) & index (sess tab) translation */
#define INDEX_OF_FD(x)		((x)-1)
#define FD_OF_INDEX(x)		((x)+1)
#define MARK_INDEX_FREE(x)	sess[x].svr_addr_len = 0
#define TEST_FREE_INDEX(x)	sess[x].svr_addr_len == 0

/*  Return value:
 *	-1	(MUST BE -1) error occured, error code is "errno"
 *	other value is sockfd for csm_sock_ask() and csm_sock_term()
 */
int csm_sock_init(const char *svr_dir)
{
	/* find a free entry */

	int idx;
	for (idx = 0; idx < MAX_CONNECTION; idx++)
		if (TEST_FREE_INDEX(idx))
			break;

	if (idx >= MAX_CONNECTION) {
		//excess maximum number of connection
		return -1;
	}

	/* open the socket */

	int sockfd = socket(PF_LOCAL, SOCK_DGRAM, 0);

	if (sockfd == -1) {

	#if DEBUG == 1
		fprintf(stderr, "socket err=%d\n", errno);
	#endif
		return -1;
	}

	/* bind our address */

	struct sockaddr_un my_addr;
	my_addr.sun_family = AF_LOCAL;
	mktemp(strcpy(sess[idx].my_addr_path, MY_ADDR_PATH));
	strcpy(my_addr.sun_path, sess[idx].my_addr_path);
	int rc = bind(sockfd, (struct sockaddr*)&my_addr, SUN_LEN(&my_addr));

	if (rc != 0) {

	#if DEBUG == 1
		fprintf(stderr, "bind err=%d\n", errno);
	#endif
		return -1;
	}

	/* prepare server address */

	sess[idx].csm_fd = sockfd;
	sess[idx].svr_addr.sun_family = AF_LOCAL;
	sprintf(sess[idx].svr_addr.sun_path, "%s/s",
			svr_dir ? svr_dir : DEFAULT_ROOT);
	sess[idx].svr_addr_len = SUN_LEN(&sess[idx].svr_addr);
	sess[idx].pid = getpid();

	return FD_OF_INDEX(idx);
}

/*  Return value:
 *	-1	(MUST BE -1) error occured, error code is "errno"
 *	other value is byte count of CSMan Server response
 */
int csm_sock_ask(int fd, const char *qbuf, int qlen, char *abuf, int size)
{
	int idx = INDEX_OF_FD(fd);

	register cmhdr_t *q = (cmhdr_t*)qbuf;
	 //"id.fd" and "id.pid" are not used now
	if (q->func_id == FUNC_ID_OPEN) q->id.pid = sess[idx].pid;
	else q->id.fd = sess[idx].csm_fd;

	int rc = sendto(sess[idx].csm_fd, qbuf, qlen, 0, 
		(struct sockaddr*)&sess[idx].svr_addr, sess[idx].svr_addr_len);

	if (rc == -1) {

	#if DEBUG == 1
		fprintf(stderr, "sendto err=%d\n", errno);
	#endif
		return -1;
	}

	rc = recvfrom(sess[idx].csm_fd, abuf, size, 0, NULL, NULL);

#if DEBUG == 1
	if (rc == -1) fprintf(stderr, "recvfrom err=%d\n", errno);
#endif

	return rc;
}

int csm_sock_term(int fd)
{
	int idx = INDEX_OF_FD(fd);

	close(sess[idx].csm_fd);
	unlink(sess[idx].my_addr_path);

	// mark index to free
	MARK_INDEX_FREE(idx);

	return 0;
}
