/**
 * @file csman.c
 *
 * CSMan API -- Unix Domain Socket Implementation.
 *
 * CSMan agent provides CSMan clients with standard APIs. For the details,
 * refer Bazaar:Concept/CSMan/Layering.
 *
 * This module translates CSMan API calls into messages that will be passed
 * to CSMan Server via Unix domian socket. Kind of RPC :)
 */

#include <string.h>

#include "unilog.h"

#include "csmprot.h"
#include "csmsock.h"

/*  Note! The handle returned and used by the following APIs is "sockfd" which is
 *  used to connect to CSMan Server. We do not use the handle that returned from 
 *  CSMan Server.
 */
static int fd_tab[MAX_CONNECTION];

/* fd (agent to client) & index (sess tab) translation */
#define INDEX_OF_FD(x)		index_of_fd(x)
#define FD_OF_INDEX(x)		fd_of_index(x)
#define MARK_INDEX_FREE(x)	fd_tab[x] = 0
#define TEST_FREE_INDEX(x)	fd_tab[x] == 0

/* map index of fd_tab to fd number (should be > 0) */
static int fd_of_index(int idx)
{
	return (idx << 8) | ((unsigned char)~idx);
}

/* map fd number to index of fd_tab */
static int index_of_fd(int fd)
{
	int idx = fd >> 8;
	if ((idx >= MAX_CONNECTION) || ((unsigned char)fd != (unsigned char)~idx))
		return CSM_RC_BADFD;
	return idx;
}

int open_csman(const char *config, int flags)
{
#if __CYGWIN__
	//in Cygwin, if the case of the pathname is not matched, 
	//the system has no response without errors.
	//to solve this problem, forcing it to be lower case always
	char temp[256];
	if (config) {
		config = strlwr(strcpy(temp, config));
	}
#endif

	/* find a free entry in fd_tab */

	int idx;
	for (idx = 0; idx < MAX_CONNECTION; idx++)
		if (TEST_FREE_INDEX(idx))
			break;

	if (idx >= MAX_CONNECTION) {
		//excess maximum number of connection
		return CSM_RC_OPENED;
	}

	int sockfd = csm_sock_init(config);
	if (sockfd == -1) {
		//fail to connect to CSMan Server
		return CSM_RC_OTHERS;
	}

	char msg[MAX_MESG_SIZE];

	register open_csman_req_t *q = (open_csman_req_t*)msg;

	q->header.func_id = FUNC_ID_OPEN;
	q->flags = flags;
	strcpy(q->config, config ? config : "");
	int len = sizeof(q->header) + sizeof(q->flags) + 1 + strlen(q->config);

	int rc = csm_sock_ask(sockfd, msg, len, msg, sizeof(msg));
	if (rc < 0)
	{
		csm_sock_term(sockfd);
		return CSM_RC_OTHERS;
	}

	register open_csman_res_t *a = (open_csman_res_t*)msg;

	//if (a->func_id != FUNC_ID_OPEN) return CSM_RC_OTHERS;

	if (a->rc < 0) {
		//CSMan Server reported open error
		return a->rc;
	}

	//successfully opened, log in fd_tab
	fd_tab[idx] = sockfd;

	return FD_OF_INDEX(idx);
}

ssize_t read_csman(int fd, CSID_T csid, void *buf, size_t count, int flags)
{
	int rc;

	int idx = INDEX_OF_FD(fd);
	if (idx < 0) {
		rc = CSM_RC_BADFD;
		goto end;
	}

	/* filter out bad csid */
	if (csid == 0) {
		rc = CSM_RC_NOID;
		goto end;
	}

	char msg[MAX_MESG_SIZE];

	register read_csman_req_t *q = (read_csman_req_t*)msg;

	q->header.func_id = FUNC_ID_READ;
	q->count = count;
	q->flags = flags;
	q->csid = csid;
	int len = sizeof(q->header) + sizeof(q->count) + sizeof(q->flags) + sizeof(q->csid);

	rc = csm_sock_ask(fd_tab[idx], msg, len, msg, sizeof(msg));
	if (rc < 0) {
		rc = CSM_RC_OTHERS;
		goto end;
	}

	register read_csman_res_t *a = (read_csman_res_t*)msg;
	rc = a->rc;

	//if (a->func_id != FUNC_ID_READ) return CSM_RC_OTHERS;

	if (rc > 0) {
		memcpy(buf, a->buf, rc);
	}

end:

	if ((rc < 0) && !(flags & CSM_R_NOLOG)) {
		LPRINTF(PRI_E, "CSID%08lX read err %d\n", (unsigned long)csid, rc);
	}

	return rc;
}

ssize_t write_csman(int fd, CSID_T csid, const void *buf, size_t count, int flags)
{
	int rc;

	int idx = INDEX_OF_FD(fd);
	if (idx < 0) {
		rc = CSM_RC_BADFD;
		goto end;
	}

	/* filter out bad csid */

	if (csid == 0) {
		rc = CSM_RC_NOID;
		goto end;
	}

	if ((count > MAX_ITEM_SIZE) || (count < 0)) {
		rc = CSM_RC_BUFFER;
		goto end;
	}

	char msg[MAX_MESG_SIZE];

	register write_csman_req_t *q = (write_csman_req_t*)msg;

	q->header.func_id = FUNC_ID_WRITE;
	q->count = count;
	q->flags = flags;
	q->csid = csid;
	memcpy(q->buf, buf, count);
	int len = sizeof(q->header) + sizeof(q->count) 
		+ sizeof(q->flags) + sizeof(q->csid) + count;

	rc = csm_sock_ask(fd_tab[idx], msg, len, msg, sizeof(msg));
	if (rc < 0) {
		rc = CSM_RC_OTHERS;
		goto end;
	}

	register write_csman_res_t *a = (write_csman_res_t*)msg;
	rc = a->rc;

	//if (a->func_id != FUNC_ID_WRITE) return CSM_RC_OTHERS;

end:

	if ((rc < 0) && !(flags & CSM_W_NOLOG)) {
		LPRINTF(PRI_E, "CSID%08lX write err %d\n", (unsigned long)csid, rc);
	}

	return rc;
}

int chkdirty_csman(int fd, CSID_T *pcsid, int flags)
{
	int idx = INDEX_OF_FD(fd);
	if (idx < 0)
		return CSM_RC_BADFD;

	return 0;
}

ssize_t dump_csman(int fd, CSID_T *pcsid, void *buf, size_t count)
{
	int idx = INDEX_OF_FD(fd);
	if (idx < 0)
		return CSM_RC_BADFD;

	if (count < 0) return CSM_RC_BUFFER;

	char msg[MAX_MESG_SIZE];

	register dump_csman_req_t *q = (dump_csman_req_t*)msg;

	q->header.func_id = FUNC_ID_DUMP;
	q->count = count;
	q->csid = *pcsid;
	int len = sizeof(q->header) + sizeof(q->count) + sizeof(q->csid);

	int rc = csm_sock_ask(fd_tab[idx], msg, len, msg, sizeof(msg));
	if (rc < 0) return CSM_RC_OTHERS;

	register dump_csman_res_t *a = (dump_csman_res_t*)msg;

	//if (a->func_id != FUNC_ID_DUMP) return CSM_RC_OTHERS;

	if (a->rc >= 0) {
		*pcsid = a->csid;
		memcpy(buf, a->buf, count < a->rc ? count : a->rc);
	}

	return a->rc;
}

int close_csman(int fd)
{
	int idx = INDEX_OF_FD(fd);
	if (idx < 0)
		return CSM_RC_BADFD;

	char msg[MAX_MESG_SIZE];

	register close_csman_req_t *q = (close_csman_req_t*)msg;

	q->header.func_id = FUNC_ID_CLOSE;
	int len = sizeof(q->header);

	int rc = csm_sock_ask(fd_tab[idx], msg, len, msg, sizeof(msg));
	if (rc < 0) return CSM_RC_OTHERS;

	register close_csman_res_t *a = (close_csman_res_t*)msg;

	//if (a->func_id != FUNC_ID_CLOSE) return CSM_RC_OTHERS;

	csm_sock_term(fd_tab[idx]);

	//successfully closed, log out fd_tab
	MARK_INDEX_FREE(idx);

	return a->rc;
}
