/*
 *	Handle incoming frames
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	$Id: br_input.c,v 1.5 2008/06/19 10:08:03 bradhuang Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_bridge.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"

// for fast-nat module, 2005-12-23
#ifdef NAT_SPEEDUP		
static int br_nat_speedup=1;
#endif
#ifdef IGMP_SNOOPING
#include <linux/ip.h>
#include <linux/in.h>
#include <linux/igmp.h>
#include <net/checksum.h>
extern int igmpsnoopenabled;
#endif
unsigned char bridge_ula[6] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x00 };
#ifdef IGMP_SNOOPING

//plus add ;2008-0609
#ifdef	MCAST_TO_UNICAST
extern int IGMPProxyOpened;
#endif

static char igmp_type_check(struct sk_buff *skb, unsigned char *gmac);
static void br_update_igmp_snoop_fdb(unsigned char op, struct net_bridge *br, struct net_bridge_port *p, unsigned char *gmac 
#ifdef	MCAST_TO_UNICAST
	, struct sk_buff *skb
#endif	
);

#endif
static int br_pass_frame_up_finish(struct sk_buff *skb)
{
	netif_rx(skb);

	return 0;
}

static void br_pass_frame_up(struct net_bridge *br, struct sk_buff *skb)
{
	struct net_device *indev;
#ifdef IGMP_SNOOPING
        unsigned char *dest;
        struct net_bridge_port *p;
        unsigned char mac[6];
        unsigned char operation;
        struct iphdr *iph;
	unsigned char proto=0;  
                          
       iph = skb->nh.iph;
	proto =  iph->protocol;                                                                                                                            
        dest = skb->mac.ethernet->h_dest;
        p = skb->dev->br_port;
        if (igmpsnoopenabled && MULTICAST_MAC(dest) && (skb->mac.ethernet->h_proto == ETH_P_IP)){
                if (proto== IPPROTO_IGMP){
			if ((operation=igmp_type_check(skb, mac)) > 0) {

#ifdef	MCAST_TO_UNICAST
               	            br_update_igmp_snoop_fdb(operation, br, p, mac,skb);
#else
                             br_update_igmp_snoop_fdb(operation, br, p, mac);
#endif
			}	
		}
        }
#endif

	br->statistics.rx_packets++;
	br->statistics.rx_bytes += skb->len;

	indev = skb->dev;
	skb->dev = &br->dev;
	skb->pkt_type = PACKET_HOST;
	skb_push(skb, ETH_HLEN);
	skb->protocol = eth_type_trans(skb, &br->dev);
#ifdef NAT_SPEEDUP
	// Directly call the function to enque skb
	if(br_nat_speedup) {
		br_pass_frame_up_finish(skb);
		return;
	}
#endif
#if 0
	if (!(list_empty(&nf_hooks[(PF_BRIDGE)][(NF_BR_LOCAL_IN)])))
		printk("netfilter no empty in PF_BRIDGE\n");
#endif	

	NF_HOOK(PF_BRIDGE, NF_BR_LOCAL_IN, skb, indev, NULL,
			br_pass_frame_up_finish);
}

static int br_handle_frame_finish(struct sk_buff *skb)
{
	struct net_bridge *br;
	unsigned char *dest;
	struct net_bridge_fdb_entry *dst;
	struct net_bridge_port *p;
	int passedup;

	dest = skb->mac.ethernet->h_dest;

	p = skb->dev->br_port;
	if (p == NULL)
		goto err_nolock;

	br = p->br;
	read_lock(&br->lock);
	if (skb->dev->br_port == NULL)
		goto err;

	passedup = 0;
	if (br->dev.flags & IFF_PROMISC) {
		struct sk_buff *skb2;

		skb2 = skb_clone(skb, GFP_ATOMIC);
		if (skb2 != NULL) {
			passedup = 1;
			br_pass_frame_up(br, skb2);
		}
	}

	if (dest[0] & 1) {
#ifdef IGMP_SNOOPING
		struct iphdr *iph;
		unsigned int ipaddr=0;
		unsigned char proto=0;
		unsigned char reserved=0;
		if (!(br->dev.flags & IFF_PROMISC) &&
                    MULTICAST_MAC(dest) && (skb->mac.ethernet->h_proto == ETH_P_IP)){
			unsigned char mac[6];
			unsigned char operation;
			
			iph = skb->nh.iph;
			ipaddr =  iph->daddr;
			proto =  iph->protocol;  
			if (proto == IPPROTO_IGMP) {
				if ((operation=igmp_type_check(skb, mac)) > 0) {
#ifdef	MCAST_TO_UNICAST
					br_update_igmp_snoop_fdb(operation, br, p, mac,skb);
#else
					br_update_igmp_snoop_fdb(operation, br, p, mac);
#endif
               }
            }
		}
	        //Brad disable 20080619
	        /*
		if((ipaddr&0xFFFFFF00)==0xE0000000)
	   		reserved=1;
	        */	
		if(igmpsnoopenabled && MULTICAST_MAC(dest) && proto !=IPPROTO_IGMP && ipaddr != 0xEFFFFFFA && reserved ==0) {	
			if ((dst = br_fdb_get(br, dest)) != NULL){
				br_multicast_forward(br, dst, skb, !passedup);
				br_fdb_put(dst);
			}else{
				br_flood_forward(br, skb, !passedup);
			}
		}
		else { // broadcast
			br_flood_forward(br, skb, !passedup);
 		}
#else

		br_flood_forward(br, skb, !passedup);
#endif
		if (!passedup)
#ifdef MULTICAST_FILTER
		{
			if ((br->fltr_portlist_num) &&
				!memcmp(dest, "\x01\x00\x5e", 3))
			{
				int i, pass_up = 1;
				unsigned short frag_offset = *((unsigned short *)&(skb->data[6]));
				unsigned short port = *((unsigned short *)&(skb->data[22]));
				unsigned long x;

				if ((frag_offset & 0x1fff) == 0) {	// check fragment offset
					for (i=0; i<br->fltr_portlist_num; i++) {
						if (port == br->fltr_portlist[i]) {
							pass_up = 0;
							break;
						}
					}
				}

				x = dest[3] ^ dest[4] ^ dest[5];
				x = x & (MLCST_MAC_ENTRY - 1);

				if (pass_up) {
					if ((br->fltr_maclist[x][3] != 0) &&
						!memcmp(&(br->fltr_maclist[x][0]), &(dest[3]), 3))
						kfree_skb(skb);
					else
						br_pass_frame_up(br, skb);
				}
				else {
					kfree_skb(skb);
					if (br->fltr_maclist[x][3] == 0) {
						memcpy(&(br->fltr_maclist[x][0]), &(dest[3]), 3);
						br->fltr_maclist[x][3] = 1;
					}
				}
			}
			else
				br_pass_frame_up(br, skb);
		}
#else
			br_pass_frame_up(br, skb);
#endif
		goto out;
	}

	dst = br_fdb_get(br, dest);
	if (dst != NULL && dst->is_local) {
		if (!passedup)
			br_pass_frame_up(br, skb);
		else
			kfree_skb(skb);
		br_fdb_put(dst);
		goto out;
	}

	if (dst != NULL) {
		br_forward(dst->dst, skb);
		br_fdb_put(dst);
		goto out;
	}

	br_flood_forward(br, skb, 0);

out:
	read_unlock(&br->lock);
	return 0;

err:
	read_unlock(&br->lock);
err_nolock:
	kfree_skb(skb);
	return 0;
}

void br_handle_frame(struct sk_buff *skb)
{
	struct net_bridge *br;
	unsigned char *dest;
	struct net_bridge_port *p;

	dest = skb->mac.ethernet->h_dest;

	p = skb->dev->br_port;
	if (p == NULL)
		goto err_nolock;

	br = p->br;
	read_lock(&br->lock);
	if (skb->dev->br_port == NULL)
		goto err;

	if (!(br->dev.flags & IFF_UP) ||
	    p->state == BR_STATE_DISABLED)
		goto err;

	if (skb->mac.ethernet->h_source[0] & 1)
		goto err;

#ifdef RTL_BRIDGE_MAC_CLONE
	if (br_mac_clone_handle_frame(br, p, skb) == -1)
		goto err;
#endif

	if (p->state == BR_STATE_LEARNING ||
	    p->state == BR_STATE_FORWARDING)
		br_fdb_insert(br, p, skb->mac.ethernet->h_source, 0);

	if (br->stp_enabled &&
	    !memcmp(dest, bridge_ula, 5) &&
	    !(dest[5] & 0xF0))
		goto handle_special_frame;

	if (p->state == BR_STATE_FORWARDING) {
		NF_HOOK(PF_BRIDGE, NF_BR_PRE_ROUTING, skb, skb->dev, NULL,
			br_handle_frame_finish);
		read_unlock(&br->lock);
		return;
	}

err:
	read_unlock(&br->lock);
err_nolock:
	kfree_skb(skb);
	return;

handle_special_frame:
	if (!dest[5]) {
		br_stp_handle_bpdu(skb);
		return;
	}

	kfree_skb(skb);
}
#ifdef IGMP_SNOOPING

static void ConvertMulticatIPtoMacAddr(__u32 group, unsigned char *gmac)
{
	__u32 u32tmp, tmp;
	int i;

	u32tmp = group & 0x007FFFFF;
	gmac[0]=0x01; gmac[1]=0x00; gmac[2]=0x5e;
	for (i=5; i>=3; i--) {
		tmp=u32tmp&0xFF;
		gmac[i]=tmp;
		u32tmp >>= 8;
	}
}
static char igmp_type_check(struct sk_buff *skb, unsigned char *gmac)
{
        struct iphdr *iph;
	__u8 hdrlen;
	struct igmphdr *igmph;

#ifdef	MCAST_TO_UNICAST	
	unsigned int IGMP_Group;// add  for fit igmp v3
#endif
	
	/* check IP header information */
	iph = skb->nh.iph;
	hdrlen = iph->ihl << 2;
	if ((iph->version != 4) &&  (hdrlen < 20))
		return -1;
	if (ip_fast_csum((u8 *)iph, iph->ihl) != 0)
		return -1;
	{ /* check the length */
	__u32 len = ntohs(iph->tot_len);
	if (skb->len < len || len < hdrlen)
		return -1; 
	}
	/* parsing the igmp packet */
	igmph = (struct igmphdr *)((u8*)iph+hdrlen);
	
#ifdef	MCAST_TO_UNICAST
	/*IGMP-V3 type Report*/
	if(igmph->type == IGMPV3_HOST_MEMBERSHIP_REPORT)
	{
		//printk("rec v3 report 1\n");
		/*in 11n seem need no care igmpProxy is opened or not,plus 2008-0612*/
		#if	0
		if(IGMPProxyOpened==0){
			IGMP_Group = *(unsigned int*)((unsigned int*)igmph + 3);
			
			//printk("v3_group:%02X:%02X:%02X:%02X\n",
			//IGMP_Group>>24,(IGMP_Group<<8)>>24,(IGMP_Group<<16)>>24,(IGMP_Group<<24)>>24);
		}else{
			return -1;//don't care v3 report
		}
		#else
			IGMP_Group = *(unsigned int*)((unsigned int*)igmph + 3);
		#endif
		
	}else{	//4 V2 or V1
		//printk("igmph->group:%04X\n",igmph->group);	
		IGMP_Group = igmph->group;
	}
#endif
#ifdef	MCAST_TO_UNICAST

	/*check if it's protocol reserved group */
	if(!IN_MULTICAST(IGMP_Group))
	{			
			return -1;
	}
	//Brad disable 20080619
	/*
	if((IGMP_Group&0xFFFFFF00)==0xE0000000){			
			return -1;
	}
	*/
	ConvertMulticatIPtoMacAddr(IGMP_Group, gmac);
	
#else
	/*check if it's protocol reserved group */
	if(!IN_MULTICAST(igmph->group))
	{
			return -1;
	}
	//Brad disable 20080619
	/*
	if((igmph->group&0xFFFFFF00)==0xE0000000){			
			return -1;
	}
	*/
	ConvertMulticatIPtoMacAddr(igmph->group, gmac);
#endif



				
	if (
#ifdef	MCAST_TO_UNICAST		
		(igmph->type==IGMPV3_HOST_MEMBERSHIP_REPORT)||	//for support igmp v3
#endif		
		(igmph->type==IGMP_HOST_MEMBERSHIP_REPORT) ||
	    (igmph->type==IGMP_HOST_NEW_MEMBERSHIP_REPORT)) 
	{

		return 1; /* report and add it */
	}else if (igmph->type==IGMP_HOST_LEAVE_MESSAGE){
		return 2; /* leave and delete it */
	}	
	
	
	return -1;
}


static void br_update_igmp_snoop_fdb(unsigned char op, struct net_bridge *br, struct net_bridge_port *p, unsigned char *dest 
#ifdef	MCAST_TO_UNICAST
, struct sk_buff *skb
#endif
)
{
#ifdef	MCAST_TO_UNICAST
	if(!MULTICAST_MAC(dest)|| dest==NULL){		
		return;
	}
#endif
	struct net_bridge_fdb_entry *dst;
	unsigned char i=0;
	unsigned char exist=0;
	unsigned short del_group_src=0;
	/* check whether entry exist */
	dst = br_fdb_get(br, dest);
	if (dst != NULL) {
		if((dst->group_src & (1<< p->port_no))== (1 << p->port_no)){
				exist = 1;
		}		
	
	}
	if (op == 1) /* add */
	{	

#ifdef	MCAST_TO_UNICAST
		struct net_device *dev = __dev_get_by_name("wlan0");	
		if (dev) {			
			unsigned char StaMacAndGroup[20];
			memcpy(StaMacAndGroup, dest, 6);
			memcpy(StaMacAndGroup+6, skb->mac.ethernet->h_source, 6);				
			if (dev->do_ioctl != NULL)
				dev->do_ioctl(dev, (struct ifreq*)StaMacAndGroup, 0x8B80);				
		}
#endif

		if (dst) {
			if (exist == 0)	{
	        	        dst->group_src = dst->group_src | (1 << p->port_no);
			}
			// update the timer to prevent from timeout
			dst->ageing_timer = jiffies;
		}
		else {
			/* insert one fdb entry */
			br_fdb_insert(br, p, dest, 0);
			dst = br_fdb_get(br, dest);
			if(dst !=NULL){
				dst->group_src = dst->group_src | (1 << p->port_no);
			}
		}
	}
	else if (op == 2) /* delete */
	{
		if (dst && exist){
			del_group_src = ~(1 << p->port_no);
			dst->group_src = dst->group_src & del_group_src;

#ifdef	MCAST_TO_UNICAST
			struct net_device *dev = __dev_get_by_name("wlan0");	
			if (dev) {			
				unsigned char StaMacAndGroup[12];
				memcpy(StaMacAndGroup, dest , 6);
				memcpy(StaMacAndGroup+6, skb->mac.ethernet->h_source, 6);
				if (dev->do_ioctl != NULL) 
					dev->do_ioctl(dev, (struct ifreq*)StaMacAndGroup, 0x8B81);							
			}
#endif	
			
		}
	}
}

#endif // IGMP_SNOOPING
