/*
 *	Device handling code
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 *
 *	$Id: br_device.c,v 1.3 2008/06/19 10:08:20 bradhuang Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 */

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/if_bridge.h>
#include <asm/uaccess.h>
#include "br_private.h"
#ifdef IGMP_SNOOPING
#include <linux/ip.h>
#include <linux/in.h>
#include <linux/igmp.h>
#include <net/checksum.h>
#endif

static int br_dev_do_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	unsigned long args[4];
	unsigned long *data;

	if (cmd != SIOCDEVPRIVATE)
		return -EOPNOTSUPP;

	data = (unsigned long *)rq->ifr_data;
	if (copy_from_user(args, data, 4*sizeof(unsigned long)))
		return -EFAULT;

	return br_ioctl(dev->priv, args[0], args[1], args[2], args[3]);
}

static struct net_device_stats *br_dev_get_stats(struct net_device *dev)
{
	struct net_bridge *br;

	br = dev->priv;

	return &br->statistics;
}

#ifdef NAT_SPEEDUP
#include <net/dst.h>
#endif

static int __br_dev_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct net_bridge *br;
	unsigned char *dest;
	struct net_bridge_fdb_entry *dst;
#ifdef IGMP_SNOOPING
	struct iphdr *iph;
	unsigned int ipaddr=0;
	unsigned char proto=0;
	unsigned char reserved=0;
	extern int igmpsnoopenabled;

	iph = skb->nh.iph;
	ipaddr = iph->daddr;
	proto = iph->protocol;
	//Brad disable 20080619
	/*
	if ((ipaddr&0xFFFFFF00) == 0xE0000000)
		reserved=1;
	*/	
#endif

	br = dev->priv;
	br->statistics.tx_packets++;
	br->statistics.tx_bytes += skb->len;

	dest = skb->mac.raw = skb->data;
	skb_pull(skb, ETH_HLEN);

	if (dest[0] & 1) {
#ifdef IGMP_SNOOPING
		if(igmpsnoopenabled && MULTICAST_MAC(dest) && (proto != IPPROTO_IGMP) && (ipaddr != 0xEFFFFFFA) && (reserved == 0)) {
			if ((dst = br_fdb_get(br, dest)) != NULL) {
				br_multicast_deliver(br, dst, skb, 0);
				br_fdb_put(dst);
			}
			else {
				br_flood_deliver(br, skb, 0);
			}
		}
		else { // broadcast
			br_flood_deliver(br, skb, 0);
		}
		return 0;
#else
		br_flood_deliver(br, skb, 0);
		return 0;
#endif
	}

	if ((dst = br_fdb_get(br, dest)) != NULL) {
#ifdef NAT_SPEEDUP
		if (skb->dst && skb->dst->hh && skb->dst->hh->fdb_cache==NULL) {
			skb->dst->hh->fdb_cache = (void*)dst;
			dst->hh_ptr = skb->dst->hh;
		}
#endif
		br_deliver(dst->dst, skb);
		br_fdb_put(dst);
		return 0;
	}

	br_flood_deliver(br, skb, 0);
	return 0;
}

int br_dev_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct net_bridge *br;
	int ret;

	br = dev->priv;
	read_lock(&br->lock);
	ret = __br_dev_xmit(skb, dev);
	read_unlock(&br->lock);

	return ret;
}

static int br_dev_open(struct net_device *dev)
{
	struct net_bridge *br;

	netif_start_queue(dev);

	br = dev->priv;
	read_lock(&br->lock);
	br_stp_enable_bridge(br);
	read_unlock(&br->lock);

	return 0;
}

static void br_dev_set_multicast_list(struct net_device *dev)
{
}

static int br_dev_stop(struct net_device *dev)
{
	struct net_bridge *br;

	br = dev->priv;
	read_lock(&br->lock);
	br_stp_disable_bridge(br);
	read_unlock(&br->lock);

	netif_stop_queue(dev);

	return 0;
}

int br_dev_set_mac_address(struct net_device *dev, void *addr)
{
	struct net_bridge *br;
	unsigned char *p;

	br = dev->priv;
	p = ((struct sockaddr *)addr)->sa_data;
	read_lock(&br->lock);
	br_stp_assign_bridge_id(br, p);
	read_unlock(&br->lock);

	return 0;
}

static int br_dev_accept_fastpath(struct net_device *dev, struct dst_entry *dst)
{
	return -1;
}

void br_dev_setup(struct net_device *dev)
{
	memset(dev->dev_addr, 0, ETH_ALEN);

	dev->do_ioctl = br_dev_do_ioctl;
	dev->get_stats = br_dev_get_stats;
	dev->hard_start_xmit = br_dev_xmit;
	dev->open = br_dev_open;
	dev->set_multicast_list = br_dev_set_multicast_list;
	dev->stop = br_dev_stop;
	dev->accept_fastpath = br_dev_accept_fastpath;
	dev->tx_queue_len = 0;
	//dev->set_mac_address = NULL;
	dev->set_mac_address = br_dev_set_mac_address;
}
