// vim: set et sw=4 sts=4 cindent:
/*
 * @File: wlanif.h
 *
 * @Abstract: Load balancing daemon WLAN interface
 *
 * @Notes:
 *
 * @@-COPYRIGHT-START-@@
 *
 * Copyright (c) 2014 Qualcomm Atheros, Inc.
 * All Rights Reserved.
 * Qualcomm Atheros Confidential and Proprietary.
 *
 * @@-COPYRIGHT-END-@@
 */

#ifndef wlanif__h
#define wlanif__h

#include "lbd.h"

// ====================================================================
// Types exposed purely for logging purposes
// ====================================================================

/**
 * @brief The diagnostic logging message IDs generated by this module.
 */
typedef enum wlanif_msgId_e {
    /// Raw utilization measurement: 8-bit band + 8-bit utilization
    wlanif_msgId_rawChanUtilization,

    /// Raw RSSI measurement: MAC + 8-bit band + 8-bit RSSI
    wlanif_msgId_rawRSSI
} wlanif_msgId_e;

// ====================================================================
// Common types exported to other modules
// ====================================================================

/**
 * @brief Type that denotes the Wi-Fi band.
 *
 * This type is only for use within the Load Balancing Daemon.
 */
typedef enum wlanif_band_e {
    wlanif_band_24g,   ///< 2.4 GHz
    wlanif_band_5g,    ///< 5 GHz
    wlanif_band_invalid,  ///< band is not known or is invalid
} wlanif_band_e;

/**
 * @brief IDs of events that are generated from this module.
 */
typedef enum wlanif_event_e {
    wlanif_event_probereq,    ///< RX'ed probe request
    wlanif_event_authrej,     ///< TX'ed authentication message with failure

    wlanif_event_act_change,  ///< RX'ed activity change

    wlanif_event_assoc,       ///< A client associated
    wlanif_event_disassoc,    ///< A client disassociated

    wlanif_event_chan_util,   ///< Channel utilization measurement
    wlanif_event_vap_restart, ///< VAP was restarted (eg. due to channel
                              ///< change or a down/up)

    wlanif_event_rssi_xing,   ///< RSSI crossing threshold

    wlanif_event_rssi_measurement, ///< RSSI measurement

    wlanif_event_band_steering_state,  ///< Band steering on/off change
    wlanif_event_maxnum
} wlanif_event_e;

/**
 * @brief The format of the wlanif_event_probereq event.
 */
typedef struct wlanif_probeReqEvent_t {
    /// Address of the STA that sent the probe request.
    struct ether_addr sta_addr;

    /// The band on which the probe request was sent.
    wlanif_band_e band;

    /// The measured RSSI of the probe request.
    u_int8_t rssi;
} wlanif_probeReqEvent_t;

/**
 * @brief The format of the wlanif_event_authrej event.
 */
typedef struct wlanif_authRejEvent_t {
    /// Address of the STA that sent the authentication message and that
    /// is being refused admission due to an ACL.
    struct ether_addr sta_addr;

    /// The band on which the message was sent.
    wlanif_band_e band;

    /// The measured RSSI of the received authentication message.
    u_int8_t rssi;
} wlanif_authRejEvent_t;

/**
 * @brief The format of the wlanif_event_act_change event.
 */
typedef struct wlanif_actChangeEvent_t {
    /// Address of the STA whose activity status changed.
    struct ether_addr sta_addr;

    /// The band on which the STA is associated.
    wlanif_band_e band;

    /// The activity status
    LBD_BOOL active;
} wlanif_actChangeEvent_t;

/**
 * @brief The format of the wlanif_event_assoc and wlanif_event_disassoc
 *        events.
 *
 * These events carry the same payload and thus share the same event
 * structure.
 */
typedef struct wlanif_assocEvent_t {
    /// Address of the STA that associated/disassociated.
    struct ether_addr sta_addr;

    /// The band on which the change occurred.
    wlanif_band_e band;
} wlanif_assocEvent_t;

/**
 * @brief The format of the wlanif_event_chan_util event.
 */
typedef struct wlanif_chanUtilEvent_t {
    /// The band on which the STA is associated.
    wlanif_band_e band;

    /// The channel utilization as a percentage.
    u_int8_t utilization;
} wlanif_chanUtilEvent_t;

/**
 * @brief The format of the wlanif_vap_restart event.
 */
typedef struct wlanif_vapRestartEvent_t {
    /// The band on which the VAP was restarted.
    wlanif_band_e band;
} wlanif_vapRestartEvent_t;

/**
 * @brief Enum types denote the RSSI crossing direction
 */
typedef enum {
    wlanif_rssi_unchanged = 0,
    wlanif_rssi_up = 1,
    wlanif_rssi_down = 2,

    wlanif_rssi_xing_invalid
} wlanif_rssiXingDirection_e;

/**
 * @brief The format of the wlanif_event_rssi_xing event
 */
typedef struct wlanif_rssiXingEvent_t {
    /// Address of the STA whose RSSI is reported.
    struct ether_addr sta_addr;

    /// The band on which the STA is associated.
    wlanif_band_e band;

    /// The RSSI measurement
    u_int8_t rssi;

    /// Flag indicating if it crossed inactivity RSSi threshold
    wlanif_rssiXingDirection_e inactRSSIXing;

    /// Flag indicating if it crossed low RSSi threshold
    wlanif_rssiXingDirection_e lowRSSIXing;
} wlanif_rssiXingEvent_t;

/**
 * @brief The format of the wlanif_event_rssi_measurement event
 */
typedef struct wlanif_rssiMeasurementEvent_t {
    /// Address of the STA whose RSSI is reported.
    struct ether_addr sta_addr;

    /// The band on which the STA is associated.
    wlanif_band_e band;

    /// The RSSI measurement
    u_int8_t rssi;
} wlanif_rssiMeasurementEvent_t;

/**
 * @brief The format of the wlanif_event_band_steering_state event.
 */
typedef struct wlanif_bandSteeringStateEvent_t {
    /// Whether band steering is enabled.
    LBD_BOOL enabled;
} wlanif_bandSteeringStateEvent_t;

/**
 * @brief Callback function when dumping the associated STAs.
 *
 * @param [in] addr  the MAC address of the associated STA
 * @param [in] band  the band of the associated STA
 * @param [in] cookie  the parameter provided in the dump call
 */
typedef void (*wlanif_associatedSTAsCB)(const struct ether_addr *addr,
                                        wlanif_band_e band, void *cookie);

// ====================================================================
// Public API
// ====================================================================

/**
 * @brief Initialize the library.
 *
 * Note that asynchronous events will not be enabled until the listen init
 * callbacks are invoked.
 *
 * @return LBD_OK on successful init; otherwise LBD_NOK
 */
LBD_STATUS wlanif_init(void);

/**
 * @brief Set overload status on a band
 *
 * @param [in] band  The band on which to set overload status
 * @param [in] overload  LBD_TRUE for overload, LBD_FALSE for not overload
 *
 * @return LBD_OK on successfully set overload; otherwise LBD_NOK
 */
LBD_STATUS wlanif_setOverload(wlanif_band_e band, LBD_BOOL overload);

/**
 * @brief For each of the VAPs, dump the associated STAs and invoke the
 *        callback with each STA MAC address and the band on which it
 *        is associated.
 *
 * @param [in] callback  the callback to invoke with the associated STA
 *                       information
 * @param [in] cookie  the parameter to provide in the callback (in addition
 *                     to the STA information) for use by the caller of this
 *                     function
 *
 *
 * @return LBD_OK on success; otherwise LBD_NOK
 */
LBD_STATUS wlanif_dumpAssociatedSTAs(wlanif_associatedSTAsCB callback,
                                     void *cookie);

/**
 * @brief Request real-time RSSI measurement of a specific station
 *
 * The RSSI measurement will be reported back in wlanif_event_rssi_measurement.
 *
 * If the previous request has not completed in driver, this request will fail.
 *
 * @param [in] band  the band that the client is associated with
 * @param [in] staAddr  the MAC address of the specific station
 * @param [in] numSamples  number of RSSI samples to average before reporting RSSI back
 *
 * @return  LBD_OK if the request is sent successfully; otherwise LBD_NOK
 */
LBD_STATUS wlanif_requestStaRSSI(wlanif_band_e band,
                                 const struct ether_addr * staAddr,
                                 u_int8_t numSamples);

/**
 * @brief Set whether probe responses should be withheld for the given
 *        STA or not.
 *
 * @pre The band must have been disabled for the STA first before this is
 *      called.
 *
 * @param [in] band  the band on which to disable probe responses
 * @param [in] staAddr  the MAC address of the specific station
 * @param [in] withholdProbes  whether to withhold probe responses for
 *                             this STA
 *
 * @return  LBD_OK if the request is sent successfully; otherwise LBD_NOK
 */
LBD_STATUS wlanif_setProbeRespWHForSTA(wlanif_band_e band,
                                       const struct ether_addr *staAddr,
                                       LBD_BOOL withholdProbes);

/**
 * @brief Record that the STA should not be allowed to associated on the
 *        provided band.
 *
 * @param [in] band  the band on which the STA should not be allowed to
 *                   associate
 * @param [in] staAddr  the MAC address of the STA
 *
 * @return LBD_OK if the prohibition for the STA on the band was added;
 *         otherwise LBD_NOK
 */
LBD_STATUS wlanif_disableBandForSTA(wlanif_band_e band,
                                    const struct ether_addr *staAddr);

/**
 * @brief Record that the STA should be allowed to associated on the
 *        provided band.
 *
 * @param [in] band  the band on which the STA should be allowed to
 *                   associate
 * @param [in] staAddr  the MAC address of the STA
 *
 * @return LBD_OK if the band was updated to allow the STA to associated;
 *         otherwise LBD_NOK
 */
LBD_STATUS wlanif_enableBandForSTA(wlanif_band_e band,
                                   const struct ether_addr *staAddr);

/**
 * @brief Kick the STA out of the provided band, forcing disassociation.
 *
 * @param [in] band  the band on which the STA should be disassociated
 * @param [in] staAddr the MAC address of the STA to disassociate
 *
 * @return LBD_OK if the request to disassociate was successfully handled;
 *         otherwise LBD_NOK
 */
LBD_STATUS wlanif_disassociateStaOnBand(wlanif_band_e band,
                                        const struct ether_addr *staAddr);

/**
 * @brief Terminate the library.
 *
 * @return LBD_OK on successful termination; otherwise LBD_NOK
 */
LBD_STATUS wlanif_fini(void);

#if defined(LBD_DBG_MENU) && defined(GMOCK_UNIT_TESTS)
struct cmdContext;

/**
 * @brief Perform an operation on a given MAC address with respect to
 *        the ACL.
 *
 * @param [in] context  the output context
 * @param [in] cmd  the command in the debug CLI
 */
void wlanifMenuAclHandler(struct cmdContext *context,
                          const char *cmd);

#endif /* LBD_DBG_MENU && GMOCK_UNIT_TESTS */

#endif
