/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File in accordance with the terms and conditions of the General 
Public License Version 2, June 1991 (the "GPL License"), a copy of which is 
available along with the File in the license.txt file or by writing to the Free 
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or 
on the worldwide web at http://www.gnu.org/licenses/gpl.txt. 

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED 
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY 
DISCLAIMED.  The GPL License provides additional details about this warranty 
disclaimer.

*******************************************************************************/

#include <common.h>
#include <net.h>
#include <malloc.h>
#include "mvOs.h"
#include "mvSysHwConfig.h"
#include "mvEthDrv.h"
#include "mvEthPhy.h"
#include "mvBoardEnvLib.h"

#undef MV_DEBUG

#ifdef MV_DEBUG
#define DB(x) x
#else
#define DB(x)
#endif

/****************************************************** 
 * driver internal definitions --                     *
 ******************************************************/ 
/* use only tx-queue0 and rx-queue0 */
#define EGIGA_DEF_TXQ 0
#define EGIGA_DEF_RXQ 0

/* rx buffer size */ 
#define ETH_HLEN       14
#define WRAP           (2 + ETH_HLEN + 4 + 32)  /* 2(HW hdr) 14(MAC hdr) 4(CRC) 32(extra for cache prefetch)*/
#define MTU            1500
#define RX_BUFFER_SIZE (MTU + WRAP)

/* rings length */
#define EGIGA_TXQ_LEN   20
#define EGIGA_RXQ_LEN   20

#define RX_POLL_TIMEOUT 0x1000
#define TX_POLL_TIMEOUT 0x1000

typedef struct _efastPriv
{
	int port;
	ETH_PORT_INIT *port_init;
	MV_U32 htpr_addr;
	MV_U32 rxqCount;
	MV_U32 txqCount;
	MV_BOOL devInit;
} efastPriv; 


/****************************************************** 
 * functions prototype --                             *
 ******************************************************/
static int mvEgigaLoad( int port, char *name, char *enet_addr );

static int mvEgigaInit( struct eth_device *dev, bd_t *p );
static int mvEgigaHalt( struct eth_device *dev );
static int mvEgigaTx( struct eth_device *dev, volatile MV_VOID *packet, int len );
static int mvEgigaRx( struct eth_device *dev );

MV_VOID mvEgigaStrToMac( char *source , char *dest );
static unsigned int mvEgigaStrToHex( char ch );


void init_phy(int port)
{
	MV_U16 reg;
	
	mvEthPhyRegRead(mvBoardPhyAddrGet(port), ETH_PHY_CTRL_REG,&reg);

	/* set speed to 100M */
	reg &= ~(ETH_PHY_CTRL_SPEED_LSB_MASK | ETH_PHY_CTRL_SPEED_MSB_MASK);
	reg |= ETH_PHY_CTRL_SPEED_LSB_MASK;
	/* disable AN */
	reg &= ~ETH_PHY_CTRL_AN_ENABLE_MASK;
	/* Set FD */
	reg |= ETH_PHY_CTRL_DUPLEX_MASK;

	mvEthPhyRegWrite(mvBoardPhyAddrGet(port), ETH_PHY_CTRL_REG, reg);

	/* reset the phy */
	mvEthPhyRegRead(mvBoardPhyAddrGet(port), ETH_PHY_CTRL_REG, &reg);
	reg |= ETH_PHY_CTRL_RESET_MASK;
	mvEthPhyRegWrite(mvBoardPhyAddrGet(port), ETH_PHY_CTRL_REG, reg);	
	
	return;
}

extern MV_STATUS mvEthWinInit(void);
/*********************************************************** 
 * mv_eth_initialize --                                    *
 *   main driver initialization. loading the interfaces.   *
 ***********************************************************/
int mv_eth_initialize( bd_t *bis ) 
{

	int port,i,j;
	MV_8 enet_addr[6*2+1];
	MV_8 name[NAMESIZE+1];
	MV_8 enetvar[9];
	MV_8 *tmp;

	if(MV_OK != mvEthWinInit()) {
		printf("Error in window initialization \n");
		return 1;
	}

	/* load interface(s) */
	for( port=0; port < mvCtrlEthMaxPortGet(); port++ ) {
		/* interface name */
		sprintf( name, "efast%d", port );
		/* interface MAC addr extract */
		sprintf( enetvar, port ? "eth%daddr" : "ethaddr", port );
		tmp = getenv( enetvar );
		for( i = 0, j = 0;i < 6*2+1 ; j++ ){ 
			if( tmp[j] != ':' ) {
				enet_addr[i] = tmp[j];
				i++;
			}
		}		
			
		mvEgigaLoad( port, name, enet_addr );
		/* Init phy */
		if ( DB_88W8660_DDR2 == mvBoardIdGet() ) {
			init_phy(port);
		}	
	}

	return 0;
}


/*********************************************************** 
 * mvEgigaLoad --                                          *
 *   load a network interface into uboot network core.     *
 *   initialize sw structures e.g. private, rings, etc.    *
 ***********************************************************/
static int mvEgigaLoad( int port, char *name, char *enet_addr ) 
{
	struct eth_device *dev = NULL;
	efastPriv *priv = NULL;
	ETH_PORT_INIT *port_init;
	unsigned int addr_table_size = 0;

	DB( printf( "%s: %s load - ", __FUNCTION__, name ) );

	dev = malloc( sizeof(struct eth_device) );
	priv = malloc( sizeof(efastPriv) );

	if( !dev ) {
		DB( printf( "%s: %s falied to alloc eth_device (error)\n", __FUNCTION__, name ) );
		goto error;
	}

	if( !priv ) {
		DB( printf( "%s: %s falied to alloc efast_priv (error)\n", __FUNCTION__, name ) );
		goto error;
	}
	memset( priv, 0, sizeof(efastPriv) );

	/* allocate space for MAC table */
   	if (PORT_CONFIG_VALUE & ETH_HASH_SIZE_500) {
      		/* 1/2KB table size */
          	addr_table_size = 0x4000;
   	}
   	else {
   		/* 8KB table size */
      		addr_table_size = 0x40000;
     	}

	priv->htpr_addr = (MV_U32)memalign((1 << ADDRESS_TABLE_ALIGNMENT), addr_table_size);
	if( !priv->htpr_addr ) {
		DB( printf( "%s: %s falied to alloc MAC address table\n", __FUNCTION__, name ));
		goto error;
	}
	memset( (void*)priv->htpr_addr, 0, addr_table_size);

	/* allocate structure for HAL */
	port_init = malloc(sizeof(ETH_PORT_INIT));
	if( !port_init ) {
		DB( printf( "%s: %s falied to alloc HAL port init structure \n", __FUNCTION__, name ));
		goto error;
	}
	memset(port_init, 0, sizeof(ETH_PORT_INIT));

	/* init device methods */
	memcpy( dev->name, name, NAMESIZE );
	mvEgigaStrToMac( enet_addr, dev->enetaddr );
	
	dev->init = (void *)mvEgigaInit;
	dev->halt = (void *)mvEgigaHalt;
	dev->send = (void *)mvEgigaTx;
	dev->recv = (void *)mvEgigaRx;
	dev->priv = priv;
	dev->iobase = 0;
	priv->port = port;

	/* init Hal structure */
	memcpy(port_init->portMacAddr, dev->enetaddr, MV_MAC_ADDR_SIZE);
	port_init->portPhyAddr = mvBoardPhyAddrGet(priv->port);
	port_init->portSdmaConfig = PORT_SDMA_CONFIG_VALUE;
	port_init->portConfig = PORT_CONFIG_VALUE;
	port_init->portConfigExtend = PORT_CONFIG_EXTEND_VALUE ;
	port_init->portAddrTblBase = priv->htpr_addr;

    	port_init->portTxQueue[EGIGA_DEF_TXQ].descNum = EGIGA_TXQ_LEN;
    	port_init->portTxQueue[EGIGA_DEF_TXQ].buffSize = 0;
    	port_init->portTxQueue[EGIGA_DEF_TXQ].descBaseAddr = (MV_U32)memalign(TX_DESC_ALIGNED_SIZE, EGIGA_TXQ_LEN * TX_DESC_ALIGNED_SIZE);
	if(!port_init->portTxQueue[EGIGA_DEF_TXQ].descBaseAddr) {
		DB( printf( "falied to allocate Tx desc\n" ) );
		goto error;
	}
    	port_init->portTxQueue[EGIGA_DEF_TXQ].buffBaseAddr = 0;

    	port_init->portRxQueue[EGIGA_DEF_RXQ].descNum = EGIGA_RXQ_LEN;
    	port_init->portRxQueue[EGIGA_DEF_RXQ].buffSize= RX_BUFFER_SIZE;
    	port_init->portRxQueue[EGIGA_DEF_RXQ].descBaseAddr = (MV_U32)memalign(RX_DESC_ALIGNED_SIZE, EGIGA_RXQ_LEN * RX_DESC_ALIGNED_SIZE);

	if(!port_init->portRxQueue[EGIGA_DEF_RXQ].descBaseAddr) {
		DB( printf( "falied to allocate Rx desc\n" ) );
		goto error;
	}

    	port_init->portRxQueue[EGIGA_DEF_RXQ].buffBaseAddr = (MV_U32)memalign(32, EGIGA_RXQ_LEN * RX_BUFFER_SIZE);
	if(!port_init->portRxQueue[EGIGA_DEF_RXQ].buffBaseAddr) {
		DB( printf( "falied to allocate Rx buffers\n" ) );
		goto error;
	}

	priv->port_init = port_init;

	/* register the interface */
	eth_register(dev);


	DB( printf( "%s: %s load ok\n", __FUNCTION__, name ) );
	return 0;

	error:
	printf( "%s: %s load failed\n", __FUNCTION__, name );
	if( priv->htpr_addr ) free( (void*)priv->htpr_addr );
	if( priv->port_init ) free( priv->port_init );
	if( priv ) free( dev->priv );
	if( dev ) free( dev );
	return -1;
}

unsigned int efast_init=0;

static int mvEgigaInit( struct eth_device *dev, bd_t *p )
{
	efastPriv *priv = dev->priv;

	DB( printf( "%s: %s init - ", __FUNCTION__, dev->name ) );

	/* Check Link status on phy */
	if ( DB_88W8660_DDR2 == mvBoardIdGet() ) {
		if( mvEthPhyCheckLink( mvBoardPhyAddrGet(priv->port) ) == MV_FALSE ) {
			printf( "%s no link\n", dev->name );
			return 0;
		}
		else DB( printf( "link up\n" ) );
	}


	efast_init = 1;

	/* init the hal -- create internal port control structure and descriptor rings, */
	/* open address decode windows, disable rx and tx operations. mask interrupts.  */
	if(MV_OK != mvEthPortInit( priv->port, priv->port_init )) {
		DB( printf( "falied to init eth port (error)\n" ) );
		goto error;
	}

	priv->devInit = MV_TRUE;

	/* start the hal - rx/tx activity */
        /* Set some flags to default values */
	if( mvEthPortStart( priv->port ) != MV_OK ) {
		DB(printf( "%s: %s mvEthPortEnalbe failed (error)\n", __FUNCTION__, dev->name ));
		goto error;
	}

	DB( printf( "%s: %s complete ok\n", __FUNCTION__, dev->name ) );
	return 1;

	error:
	if( priv->devInit )
		mvEgigaHalt( dev );
	printf( "%s: %s failed\n", __FUNCTION__, dev->name );
	return 0;
}

static int mvEgigaHalt( struct eth_device *dev )
{
	efastPriv *priv = dev->priv;

	DB( printf( "%s: %s halt - ", __FUNCTION__, dev->name ) );

	if( priv->devInit == MV_TRUE ) {
		/* stop the port activity, mask all interrupts */
		mvEthPortStop( priv->port );
		priv->devInit = MV_FALSE;
	}

	efast_init = 0;

	DB( printf( "%s: %s complete\n", __FUNCTION__, dev->name ) );
	return 0;
}

static int mvEgigaTx( struct eth_device *dev, volatile void *buf, int len )
{
	efastPriv *priv = dev->priv;
	BUF_INFO bufInfo;
	MV_STATUS status;
	unsigned int txPollTimeout = 0;

	/* if no link exist */
	if(!efast_init) return 0;

	bufInfo.userInfo1 = 0x44CAFE44; /* dummy*/
	bufInfo.userInfo2 = 0x33CAFE33; /* dummy*/
	bufInfo.cmdSts = 0;
	bufInfo.bufSize = len;
	bufInfo.byteCnt = len;
	bufInfo.pBuff = (MV_U8*)buf;
	bufInfo.pData = (MV_U8*)buf;
	bufInfo.pNextBufInfo = NULL;

	/* send the packet */
	status = mvEthPortSend(priv->port, EGIGA_DEF_TXQ, &bufInfo);

	if( status != MV_OK ) {
		if( status == MV_NO_RESOURCE )
			DB( printf( "ring is full (error)\n" ) );
		else if( status == MV_ERROR )
			DB( printf( "error\n" ) );
		else
			DB( printf( "unrecognize status (error) ethPortSend\n" ) );
		goto error;
	} 
	else DB( printf( "ok\n" ) );

	priv->txqCount++;
	
	/* release the transmitted packet(s) */
	while( 1 ) {

		DB( printf( "%s: %s tx-done - ", __FUNCTION__, dev->name ) );

		/* get a packet */
		status = mvEthPortSendDone(priv->port, EGIGA_DEF_TXQ, &bufInfo);

		if( status == MV_OK ) {

			priv->txqCount--;

			/* validate skb */
			if( bufInfo.userInfo1 != 0x44CAFE44 ) {
				DB( printf( "error\n" ) );
				goto error;
			}
			if( bufInfo.userInfo2 != 0x33CAFE33 ) {
				DB( printf( "error\n" ) );
				goto error;
			}

			/* handle tx error */
			if( bufInfo.cmdSts & (ETH_ERROR_SUMMARY) ) {
				DB( printf( "bad status (error)\n" ) );
				goto error;
			}
			DB( printf( "ok\n" ) );
		}
		else if( status == MV_EMPTY ) {
			/* no more work */
			DB( printf( "no more work\n" ) );
			break;
		}
		else if( status == MV_NO_MORE ) {
			/* hw still in tx */
			DB( printf( "hw still in tx\n" ) );
			txPollTimeout++;
			if(txPollTimeout > TX_POLL_TIMEOUT)
				goto error;
			continue; 		
		}
		else {
			DB( printf( "unrecognize status (error) ethTxReturnDesc\n" ) );
			goto error;
		}
	}

	DB( printf( "%s: %s complete ok\n", __FUNCTION__, dev->name ) );

	return 0;

	error:
	DB( printf( "%s: %s failed\n", __FUNCTION__, dev->name ) );
	return 1;
}


static int mvEgigaRx( struct eth_device *dev )
{
	efastPriv*  	priv = dev->priv;
    	BUF_INFO 	bufInfo;
	MV_STATUS   	status;
	unsigned int 	rxPollTimeout = 0;

	/* if no link exist */
	if(!efast_init) return 0;

	while( 1 ) {

		if (rxPollTimeout++ == RX_POLL_TIMEOUT) {
                	return 0;
        	}

		/* get rx packet from hal */
		status = mvEthPortReceive(priv->port, EGIGA_DEF_RXQ, &bufInfo);

		if( status == MV_OK ) {
			/*DB( printf( "good rx\n" ) );*/
			priv->rxqCount--;

			/* check rx error status */
			if( bufInfo.cmdSts & (ETH_ERROR_SUMMARY) ) {
				DB( printf( "bad rx status: %08x \n", bufInfo.cmdSts ) );
				goto error;
			}
			else {

				DB( printf( "%s: %s calling NetRecieve ", __FUNCTION__, dev->name) );
				/* good rx - push the packet up (skip on two first empty bytes) */
				NetReceive( (MV_U8 *)bufInfo.pBuff , (int)bufInfo.byteCnt );
			}


			DB( printf( "%s: %s refill rx buffer - ", __FUNCTION__, dev->name) );

			/* give the buffer back to hal (re-init the buffer address) */
			bufInfo.bufSize = RX_BUFFER_SIZE;
			bufInfo.byteCnt = RX_BUFFER_SIZE;
			bufInfo.pNextBufInfo = NULL;

			status =  mvEthPortReceiveDone(priv->port, EGIGA_DEF_RXQ, &bufInfo);
	
			if( status == MV_OK ) {
				priv->rxqCount++;
			}
			else if( status == MV_FULL ) {
				/* this buffer made the ring full */
				priv->rxqCount++;
				DB( printf( "ring full\n" ) );
				break;
			} 
			else {
				printf( "error\n" );
				goto error;
			}

		} else if( status == MV_NO_MORE ) {
			/* no more rx packets ready */
			/* DB( printf( "no more work\n" ) );*/
			break;
		} else if( status == MV_NO_RESOURCE ) {
			/* no buffers for rx */
			DB( printf( "no resource (error)\n" ) );
			goto error;
		} else {
			DB( printf( "unrecognize status (error) ethRxReturnBuff\n" ) );
			goto error;
		}
	}

	/*DB( printf( "%s: %s complete ok\n", __FUNCTION__, dev->name ) );*/
	return 0;

	error:
	DB( printf( "%s: %s failed\n", __FUNCTION__, dev->name ) );
	return 1;
}


/*********************************************************** 
 * string helpers for mac address setting                  *
 ***********************************************************/
MV_VOID mvEgigaStrToMac( MV_8 *source , MV_8 *dest ) 
{
	dest[0] = (mvEgigaStrToHex( source[0] ) << 4) + mvEgigaStrToHex( source[1] );
	dest[1] = (mvEgigaStrToHex( source[2] ) << 4) + mvEgigaStrToHex( source[3] );
	dest[2] = (mvEgigaStrToHex( source[4] ) << 4) + mvEgigaStrToHex( source[5] );
	dest[3] = (mvEgigaStrToHex( source[6] ) << 4) + mvEgigaStrToHex( source[7] );
	dest[4] = (mvEgigaStrToHex( source[8] ) << 4) + mvEgigaStrToHex( source[9] );
	dest[5] = (mvEgigaStrToHex( source[10] ) << 4) + mvEgigaStrToHex( source[11] );
}

static MV_U32 mvEgigaStrToHex( MV_8 ch ) 
{
	if( (ch >= '0') && (ch <= '9') ) return( ch - '0' );
	if( (ch >= 'a') && (ch <= 'f') ) return( ch - 'a' + 10 );
	if( (ch >= 'A') && (ch <= 'F') ) return( ch - 'A' + 10 );

	return 0;
}


