/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

This software file (the "File") is owned and distributed by Marvell 
International Ltd. and/or its affiliates ("Marvell") under the following
alternative licensing terms.  Once you have made an election to distribute the
File under one of the following license alternatives, please (i) delete this
introductory statement regarding license alternatives, (ii) delete the two
license alternatives that you have not elected to use and (iii) preserve the
Marvell copyright notice above.

********************************************************************************
Marvell Commercial License Option

If you received this File from Marvell and you have entered into a commercial
license agreement (a "Commercial License") with Marvell, the File is licensed
to you under the terms of the applicable Commercial License.

********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File in accordance with the terms and conditions of the General 
Public License Version 2, June 1991 (the "GPL License"), a copy of which is 
available along with the File in the license.txt file or by writing to the Free 
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or 
on the worldwide web at http://www.gnu.org/licenses/gpl.txt. 

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED 
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY 
DISCLAIMED.  The GPL License provides additional details about this warranty 
disclaimer.
********************************************************************************
Marvell BSD License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File under the following licensing terms. 
Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    *   Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer. 

    *   Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution. 

    *   Neither the name of Marvell nor the names of its contributors may be 
        used to endorse or promote products derived from this software without 
        specific prior written permission. 
    
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

/*******************************************************************************
*
* ethernet.c - Marvell Fast Ethernte Controller driver
*
* DESCRIPTION:
*       This file introduce low level API to Marvell's Fast Ethernet 
*		Controller. This Fast Ethernet Controller driver API controls
*		1) Operations (i.e. port init, start, reset etc').
*		2) Data flow (i.e. port send, receive etc').
*		This driver is designed to support multiple Fast Ethernet Media Access 
*		Controllers.
*       The Fast Ethernet port is controlled via ETH_PORT_CTRL struct.
*		This struct includes user configuration information as well as driver 
*		internal data needed for its operations.
*		                                                                       
*		Supported Features:
*		- This low level driver is OS independent. Allocating memory for the 
*		  descriptor rings and buffers are not within the scope of this driver.
*		- The user is free from Rx/Tx queue managing.
*		- Tx data flow operation is on packet basis (support for transmitted
*		  packet spanned over multiple buffers).
*		- Rx data flow operation in on buffer basis (no support for received 
*		  packet spanned over multiple buffers).
*		- Simple operation API.
*		- Simple data flow API.
*		- Support cached descriptors for better performance.
*		- Port control register configuration API.
*																			   
*		Operation flow:
*
*		Initialization phase
*		This phase initialize the following

 
*		This phase complete the initialization of the ETH_PORT_INFO struct. 
*		User information regarding port configuration has to be set prior to 
*		calling the port initialization routine. For example, the user has to 
*		assign the portPhyAddr field which is board depended parameter.
*		In this phase any port Tx/Rx activity is halted, MIB counters are 
*		cleared, PHY address is set according to user parameter and access to
*		DRAM and internal SRAM memory spaces.
*
*		Driver ring initialization
*		Allocating memory for the descriptor rings and buffers is not 
*		within the scope of this driver. Thus, the user is required to allocate 
*		memory for the descriptors ring and buffers. Those memory parameters 
*		are used by the Rx and Tx ring initialization routines in order to 
*		curve the descriptor linked list in a form of a ring. 
*		Note: Pay special attention to alignment issues when using cached 
*		descriptors/buffers. In this phase the driver stores information in the 
*		ETH_PORT_INFO struct regarding each queue ring.
*
*		Driver start 
*		This phase prepares the Ethernet port for Rx and Tx activity. It uses 
*		the information stored in the ETH_PORT_INFO struct to initialize the 
*		various port registers.
*
*		Data flow:
*		All packet references to/from the driver are done using BUF_INFO struct.
*       This struct is a unified struct used with Rx and Tx operations. 
*       This way the user is not required to be fully familiar with neither Tx  
*       nor Rx descriptors structures.
*		The driver's descriptors rings are management by indexes. Those indexes
*		controls the ring resources and used to indicate a SW resource error:
*		'current' 
*			This index points to the current available resource for use. For 
*			example in Rx process this index will point to the descriptor  
*			that will be passed to the user upon calling the receive routine.
*			In Tx process, this index will point to the descriptor
*			that will be assigned with the user packet info and transmitted.
*		'used'    
*			This index points to the descriptor that need to restore its 
*			resources. For example in Rx process, using the Rx buffer return
*			API will attach the buffer returned in packet info to the descriptor 
*			pointed by 'used'. In Tx process, using the Tx descriptor return 
*			will merely return the user packet info with the command status of  
*			the transmitted buffer pointed by the 'used' index. Nevertheless, it 
*			is essential to use this routine to update the 'used' index.
*		'first'
*			This index supports Tx Scatter-Gather. It points to the first 
*			descriptor of a packet assembled of multiple buffers. For example 
*			when in middle of such packet we have a Tx resource error the 
*			'curr' index get the value of 'first' to indicate that the ring 
*			returned to its state before trying to transmit this packet.
*																			   
*		Receive operation:
*		The ethPortReceive API set the packet information struct, passed by the 
*       caller, with received information from the 'current' SDMA descriptor. 
*		It is the user responsibility to return this resource back to the Rx 
*       descriptor ring to enable the reuse of this source. Return Rx resource 
*       is done using the ethRxReturnBuff API.
*		                                                                       
*		Transmit operation:
*		The ethPortSend API supports Scatter-Gather which enables to send a 
*		packet spanned over multiple buffers. This means that for each 
*		packet info structure given by the user and put into the Tx descriptors 
*		ring, will be transmitted only if the 'LAST' bit will be set in the 
*		packet info command status field. This API also consider restriction 
*       regarding buffer alignments and sizes.
*		The user must return a Tx resource after ensuring the buffer has been 
*		transmitted to enable the Tx ring indexes to update.
*		                                                                       
*		BOARD LAYOUT                                                           
*		This device is on-board.  No jumper diagram is necessary.              
*		                                                                       
*		EXTERNAL INTERFACE                                                     
*		                                                                       
*       Prior to calling the initialization routine ethPortInit() the user must
*       set the following fields under ETH_PORT_INFO struct:		                                                                       
*       portNum             User Ethernet port number.
*       portBase		    User PHY address of Ethernet port.
*       portPhyAddr		    User PHY address of Ethernet port.
*       portMacAddr[6]	    User defined port MAC address.
*       portConfig          User port configuration value.
*       portConfigExtend    User port config extend value.
*       portSdmaConfig      User port SDMA config value.
*       *portPrivate        User scratch pad for user specific data structures.
*		                                                                       
*       This driver introduce a set of default values to use prior to calling
*       the port initialization routine:
*       PORT_CONFIG_VALUE           Default port configuration value
*       PORT_CONFIG_EXTEND_VALUE    Default port extend configuration value
*       PORT_SDMA_CONFIG_VALUE      Default sdma control value
*
*		This driver data flow is done using the BUF_INFO struct which is a 
* 		unified struct for Rx and Tx operations:
*		byteCnt				Tx/Rx descriptor buffer byte count.
*		cmdSts				Tx/Rx descriptor command status.
*		pBuff				Tx/Rx descriptor buffer pointer.
*		pData				Tx/Rx descriptor data pointer.
*		returnInfo			Tx/Rx user resource return information.
*		                                                                       
*
*		EXTERNAL SUPPORT REQUIREMENTS                                          
*		                                                                       
*		This driver requires the following external support:
*		                                                                       
*		D_CACHE_FLUSH_LINE (address, address offset)
*		                                                                       
*		This macro applies assembly code to flush and invalidate cache line.
*		address        - address base.                                 
*		address offset - address offset                   
*		                                                                       
*			                                                                   
*		CPU_PIPE_FLUSH
*		                                                                       
*		This macro applies assembly code to flush the CPU pipeline.
*		                                                                       
*******************************************************************************/
/* includes */
#include "mvOs.h"
#include "mvEthDrv.h"
#include "mvEthPhy.h"
#include "mvEthAddrFilter.h"
#include "mvUniMacRegs.h"

/* defines */
#undef DRV_DEBUG

/* Driver debug control */
#ifdef DRV_DEBUG

#define DRV_DEBUG_OFF		0x0000
#define DRV_DEBUG_RX		0x0001
#define DRV_DEBUG_TX		0x0002
#define DRV_DEBUG_POLL		(DRV_DEBUG_POLL_RX | DRV_DEBUG_POLL_TX)
#define DRV_DEBUG_POLL_RX   	0x0004
#define DRV_DEBUG_POLL_TX   	0x0008
#define DRV_DEBUG_IOCTL		0x0020
#define DRV_DEBUG_INIT		0x0040
#define DRV_DEBUG_START    	0x0080
#define DRV_DEBUG_DUMP    	0x0100
#define DRV_DEBUG_MEM    	0x0200
#define DRV_DEBUG_RX_ERR   	0x0400
#define DRV_DEBUG_ALL    	0xffff

int	ethPortDebug = DRV_DEBUG_ALL; 

#define DRV_PRINT(FLG, X)										\
    {															\
    if (ethPortDebug & FLG) printf X;							\
    }

#else /* DRV_DEBUG */

#define DRV_PRINT(FLG, X)

#endif /* DRV_DEBUG */


#define MIN_TX_BUFF_LOAD    8

	
/* Macros that save access to desc in order to find next desc pointer  */
#define RX_NEXT_DESC_PTR(pRxDescr, pQueueCtrl)						        \
        ((pRxDescr) == (pQueueCtrl)->pLastDescr) ?                          \
               (ETH_RX_DESC*)((pQueueCtrl)->pFirstDescr) :                 \
               (ETH_RX_DESC*)(((MV_U32)(pRxDescr)) + RX_DESC_ALIGNED_SIZE)

#define TX_NEXT_DESC_PTR(pTxDescr, pQueueCtrl)						  \
        ((pTxDescr) == (pQueueCtrl)->pLastDescr) ?                          \
               (ETH_TX_DESC*)((pQueueCtrl)->pFirstDescr) :                 \
               (ETH_TX_DESC*)(((MV_U32)(pTxDescr)) + TX_DESC_ALIGNED_SIZE)



/* An offest in Tx descriptors to store data for buffers less than 8 Bytes */
#define TX_BUF_OFFSET_IN_DESC       0x18


#define D_CACHE_FLUSH_LINE(desc, dummy)	mvOsCacheLineFlushInv((MV_ULONG)(desc))	
#define D_CACHE_INVALIDATE_LINE(desc, dummy) 	mvOsCacheLineInv((MV_ULONG)(desc))
#define PHY_MEM(addr) 				mvOsIoVirtToPhy(NULL, (void*)addr)

/* typedefs */
typedef struct _ethRxDesc
{
	MV_U32 cmdSts; 
#ifdef MV_CPU_LE
	MV_U16 byteCnt;
	MV_U16 bufSize;
#else /* BE */
	MV_U16 bufSize;
	MV_U16 byteCnt;
#endif	
	MV_U32 bufPhyPtr;
	MV_U32 nextDescPtr;
    	MV_U32 bufPtr;
    	MV_U32 userInfo1;
    	MV_U32 userInfo2;

} ETH_RX_DESC;


typedef struct _ethTxDesc
{
	MV_U32 cmdSts;
#ifdef MV_CPU_LE
	MV_U16 shadow;
	MV_U16 byteCnt;
#else /* BE */
	MV_U16 byteCnt;
	MV_U16 shadow;
#endif	
	MV_U32 bufPhyPtr;
    	MV_U32 nextDescPtr; 
    	MV_U32 userInfo1;
    	MV_U32 userInfo2;
    
} ETH_TX_DESC;

typedef struct 
{
    void* pFirstDescr;
    void* pLastDescr;
    void* pCurrentDescr;
    void* pUsedDescr;
    int   resource;
} ETH_QUEUE_CTRL;

/* Ethernet port specific infomation */
typedef struct _ethPortCtrl
{
    int     portPhyAddr;      /* port phy address     */
    MV_U32  portBase;     /* Port registers base offset */
    MV_U8   portMacAddr[6];   /* Port MAC address.    */
    MV_U32  portConfig;
    MV_U32  portConfigExtend;
    MV_U32  portSdmaConfig;
    MV_U32  portAddrTblBase;

    /* User scratch pad for user specific data structures */
    void *portPrivate;

    ETH_QUEUE_CTRL rxQueue[ETH_MAX_RX_QUEUE_NUM]; /* Rx ring resource  */
    ETH_QUEUE_CTRL txQueue[ETH_MAX_TX_QUEUE_NUM]; /* Tx ring resource  */

} ETH_PORT_CTRL; 

/* locals */
/* This array holds the control structure of each port */
ETH_PORT_CTRL portCtrl[ETH_PORT_NUM];

/* SDMA routines */
static void ethPortInitRxMemory (ETH_QUEUE_CTRL* pQueueCtrl,
								 int   	rxDescNum,
								 int	rxBuffSize,
								 MV_U32	rxDescBaseAddr,
								 MV_U32	rxBuffBaseAddr);

static void ethPortInitTxMemory (ETH_QUEUE_CTRL* pQueueCtrl,
								 int	txDescNum,
								 int	txBuffSize,
								 MV_U32	txDescBaseAddr,
								 MV_U32	txBuffBaseAddr);

static MV_VOID ethBCopy(MV_U32 srcAddr, MV_U32 dstAddr, int byteCount);


/*******************************************************************************
* mvEthPortInit - Initialize the Ethernet port driver
*
* DESCRIPTION:
*       This function initialize the ethernet port:
*       1) Initialize internal SW struct according to input struct.
*       2) Initialize Rx/Tx queue memory resources according to input struct.
*		3) Stop all data Rx/Tx activity. Disable port.
*       4) Initiate address filtering.
*		5) Add unicast address to address filtering according to input struct.
*       5) Initialize the SMI unit underlying the Ethenret port according to 
*		   input struct
*       Note: Call this routine prior to ethPortStart routine and after setting
*       user values in the user fields of Ethernet port control struct (i.e.
*       portPhyAddr).
*
* INPUT:
*       *pEthPortInit - Ethernet port initialization struct.
*
* OUTPUT:
*       See description.
*
* RETURN:
*       MV_BAD_PARAM - Bad alignment of a initialization struct parameter.
*		MV_ERROR     - Initialization of SMI unit underlying the Ethenret port
*                      Failed.
*
*******************************************************************************/
MV_STATUS mvEthPortInit(int ethPortNum, ETH_PORT_INIT *pEthPortInit)
{
    MV_U32          addrTblPhyAddr, addrTblAddr; 
    MV_32           queue; 
    MV_U32	    qDescNum, qBuffSize, qDescBaseAddr, qBuffBaseAddr;
    ETH_PORT_CTRL*  pPortCtrl;
    ETH_QUEUE_CTRL* pQueueCtrl;
    
    DRV_PRINT (DRV_DEBUG_INIT, ("In mvEthPortInit\n"));
	
    pPortCtrl = &portCtrl[ethPortNum];

    pPortCtrl->portPhyAddr = pEthPortInit->portPhyAddr;      
    pPortCtrl->portBase    = pEthPortInit->portBase;     
    pPortCtrl->portMacAddr[0] = pEthPortInit->portMacAddr[0];   
    pPortCtrl->portMacAddr[1] = pEthPortInit->portMacAddr[1];   
    pPortCtrl->portMacAddr[2] = pEthPortInit->portMacAddr[2];   
    pPortCtrl->portMacAddr[3] = pEthPortInit->portMacAddr[3];   
    pPortCtrl->portMacAddr[4] = pEthPortInit->portMacAddr[4];   
    pPortCtrl->portMacAddr[5] = pEthPortInit->portMacAddr[5];   
    pPortCtrl->portMacAddr[6] = pEthPortInit->portMacAddr[6];   
    
    pPortCtrl->portPrivate = pEthPortInit->portPrivate;
    
    /* Check address table memory area alignment */
    if ((pEthPortInit->portAddrTblBase % (1 << ADDRESS_TABLE_ALIGNMENT)) != 0)
	{
		DRV_PRINT (DRV_DEBUG_INIT,("Port Address table bad alignment for 0x%x",
												pEthPortInit->portAddrTblBase));
        return MV_BAD_PARAM;
	}
    else
        pPortCtrl->portAddrTblBase = pEthPortInit->portAddrTblBase;


    if (pEthPortInit->portConfig == 0x0)
        pPortCtrl->portConfig = PORT_CONFIG_VALUE;
    else
        pPortCtrl->portConfig = pEthPortInit->portConfig;       
    
    if (pEthPortInit->portConfigExtend == 0x0)
        pPortCtrl->portConfigExtend = PORT_CONFIG_EXTEND_VALUE;
    else
        pPortCtrl->portConfigExtend = pEthPortInit->portConfigExtend; 

    if (pEthPortInit->portSdmaConfig == 0x0)
        pPortCtrl->portSdmaConfig = PORT_SDMA_CONFIG_VALUE;   
    else
        pPortCtrl->portSdmaConfig = pEthPortInit->portSdmaConfig;   

    /* Zero out Rx SW structs */ 
    /* Initialize port Rx memory resources for each user initiated queue */
    for(queue = 0; queue < ETH_MAX_RX_QUEUE_NUM; queue++)
    {    
		qDescNum 	  = pEthPortInit->portRxQueue[queue].descNum;
		qBuffSize 	  = pEthPortInit->portRxQueue[queue].buffSize;
		qDescBaseAddr = pEthPortInit->portRxQueue[queue].descBaseAddr;
		qBuffBaseAddr = pEthPortInit->portRxQueue[queue].buffBaseAddr;
        
	if (qDescBaseAddr != 0)
        {
            /* Check memory parameter alignment */
            /* Rx desc Must be 4LW aligned (i.e. Descriptor_Address[3:0]=0000)*/
            if ((qDescBaseAddr % (1 << RX_DESC_ALIGNMENT)) != 0) 
			{
				DRV_PRINT (DRV_DEBUG_INIT,("portRxQueue %d bad alignment for \
									descBaseAddr 0x%x\n",queue, qDescBaseAddr));
                return MV_BAD_PARAM; 
            }
                        
            /* Rx buffers must be 64-bit aligned.       */
            if ((qBuffBaseAddr % (1 << RX_BUFF_ALIGNMENT)) != 0)
			{
				DRV_PRINT (DRV_DEBUG_INIT,("portRxQueue %d bad alignment for \
									buffBaseAddr 0x%x\n",queue, qBuffBaseAddr));
                return MV_BAD_PARAM; 
			}


            /* Rx buffers are limited to 64K bytes and Minimum size is 8 bytes*/
            if ((qBuffSize < 8) || (qBuffSize > ETH_RX_BUFFER_MAX_SIZE)) 
			{
				DRV_PRINT (DRV_DEBUG_INIT,("portRxQueue %d bad buffer size for \
									        buffSize 0x%x\n",queue, qBuffSize));
                return MV_BAD_PARAM; 
			}
        
            pQueueCtrl = &pPortCtrl->rxQueue[queue];
            pQueueCtrl->pCurrentDescr = 0x00000000;
            pQueueCtrl->pUsedDescr = 0x00000000;

			/* Init resource available for this queue */
			pQueueCtrl->resource = qDescNum;


			ethPortInitRxMemory(pQueueCtrl,
                                qDescNum,
                                qBuffSize,
                                qDescBaseAddr,
                                qBuffBaseAddr);
        }                                          
    }
	

    /* Initialize port Tx memory resources for each user initiated queue */
    for(queue = 0; queue < ETH_MAX_TX_QUEUE_NUM; queue++)
	{
		qDescNum 	  = pEthPortInit->portTxQueue[queue].descNum;
		qBuffSize 	  = pEthPortInit->portTxQueue[queue].buffSize;
		qDescBaseAddr = pEthPortInit->portTxQueue[queue].descBaseAddr;
		qBuffBaseAddr = pEthPortInit->portTxQueue[queue].buffBaseAddr;

        if (qDescBaseAddr != 0)
        {
            /* Check memory parameter alignment Note that Tx buffer alignment */
            /* is on buffer payload < 8 bytes                                 */

			/* Tx desc Must be 4LW aligned (i.e. Descriptor_Address[3:0]=0000)*/
			if ((qDescBaseAddr % (1 << TX_DESC_ALIGNMENT)) != 0) 
			{
				DRV_PRINT (DRV_DEBUG_INIT,("portTxQueue %d bad alignment for \
									descBaseAddr 0x%x\n",queue, qDescBaseAddr));
                return MV_BAD_PARAM;    
			}

			/* Tx buffers are limited to 64K bytes */
			if (qBuffSize > ETH_TX_BUFFER_MAX_SIZE)
			{
				DRV_PRINT (DRV_DEBUG_INIT,("portTxQueue %d bad buffer size for \
											buffSize 0x%x\n",queue, qBuffSize));
                return MV_BAD_PARAM;    
			}
			
            /* Zero out Tx SW structs */ 
            pQueueCtrl = &pPortCtrl->txQueue[queue];
            pQueueCtrl->pCurrentDescr = 0x00000000;
            pQueueCtrl->pUsedDescr = 0x00000000;

			/* Init resource available for this queue */
			pQueueCtrl->resource = qDescNum;
            
			ethPortInitTxMemory(pQueueCtrl,
                                qDescNum,
                                qBuffSize,
                                qDescBaseAddr,
                                qBuffBaseAddr);
        }
    }

    mvEthPortStop(ethPortNum); 
    DRV_PRINT (DRV_DEBUG_START, ("mvEthPortStop OK...\n"));
    
    /* Extract the Physical address of the address filter table */
    addrTblAddr = portCtrl[ethPortNum].portAddrTblBase;
    addrTblPhyAddr = PHY_MEM(addrTblAddr);

    mvEthAddrFilterInit(ethPortNum, 
                        addrTblAddr, 
                        addrTblPhyAddr, 
                        portCtrl[ethPortNum].portConfig );

    /* Add the assigned Ethernet address to the port's address table */	
    mvEthPortMacAddr(ethPortNum, pPortCtrl->portMacAddr,ADD_MAC_ADDR);

    DRV_PRINT (DRV_DEBUG_INIT, ("mvEthPortMacAddr OK...\n"));
	
    DRV_PRINT (DRV_DEBUG_INIT, ("mvEthPortInit OK...\n"));

    return MV_OK;

}

/*******************************************************************************
* mvEthPortStart - Start the Ethernet port activity.
*
* DESCRIPTION:
*       This routine prepares the Ethernet port for Rx and Tx activity:
*       1. Initialize Tx and Rx Current Descriptor Pointer for each queue that
*          has been initialized.
*       2. Initialize and enable the Ethernet configuration port by writing to 
*          the port's configuration and command registers.
*       3. Initialize and enable SDMA Rx and Tx activities.
*       After completing these steps, the ethernet port SDMA can starts to 
*       perform Rx and Tx activities.
*
*       Note: Each Rx and Tx queue descriptor's list must be initialized prior  
*       to calling this function.
*
* INPUT:
*		ethPortNum - Ethernet port number.
*
* OUTPUT:
*       Ethernet port is ready to receive and transmit.
*
* RETURN:
*       MV_ERROR - Port PHY Link is not up.
*       MV_OK  - otherwise.
*
*******************************************************************************/
MV_STATUS mvEthPortStart(int ethPortNum)
{
    MV_32           queue;
    ETH_PORT_CTRL   *pPortCtrl;
    ETH_QUEUE_CTRL  *pQueueCtrl;
    ETH_TX_DESC     *pTxCurrDesc;
    ETH_RX_DESC     *pRxCurrDesc;

    DRV_PRINT (DRV_DEBUG_START, ("In mvEthPortStart\n"));

    pPortCtrl = &portCtrl[ethPortNum];

	/* Assignment of Tx CTRP of given queue */
	for(queue = 0; queue < ETH_MAX_TX_QUEUE_NUM; queue++)
	{
        pQueueCtrl = &pPortCtrl->txQueue[queue];
        pTxCurrDesc = pQueueCtrl->pCurrentDescr;

		ETH_REG_WRITE( (CURR_TX_DESC_PTR0(ethPortNum)
					  + (4 * queue ) ), 
					  PHY_MEM((MV_U32)pTxCurrDesc));
    }
	
    /* Assignment of Rx CRDP of given queue */
	for(queue = 0; queue < ETH_MAX_RX_QUEUE_NUM; queue++)
	{
        pQueueCtrl = &pPortCtrl->rxQueue[queue];
        pRxCurrDesc = pQueueCtrl->pCurrentDescr;
		
		ETH_REG_WRITE( (FIRST_RX_DESC_PTR0(ethPortNum) 
					    + (4 * queue)), 
					  PHY_MEM(pRxCurrDesc));
		
		ETH_REG_WRITE( (CURR_RX_DESC_PTR0(ethPortNum) 
					    + (4 * queue)),
					  PHY_MEM(pRxCurrDesc));
	}

    /* Assign port configuration and command. */
    ETH_REG_WRITE(PORT_CONFIG_REG(ethPortNum), pPortCtrl->portConfig);
    ETH_REG_WRITE(PORT_CONFIG_EXTEND_REG(ethPortNum), 
				   pPortCtrl->portConfigExtend);
    ETH_REG_WRITE(SDMA_CONFIG_REGISTER(ethPortNum), pPortCtrl->portSdmaConfig);

    /* Enable port Rx. */
    ETH_REG_WRITE(SDMA_COMMAND_REGISTER(ethPortNum), ETH_ENABLE_RX_DMA);

   return MV_OK;
}

/*******************************************************************************
* mvEthPortStop - Start the Ethernet port activity.
*
* DESCRIPTION:
* 		This routine resets the Ethernet port by aboarting any SDMA activity 
*		and disables the port.
*		The Receiver And the Transmit Unit are in idle state after this 
*		command is issued.
*
* INPUT:
*		ethPortNum - Ethernet port number.
*
* OUTPUT:
*       Ethernet port is in idle state.
*
* RETURN:
*		N/A
*
*******************************************************************************/
void mvEthPortStop(int ethPortNum)
{
    MV_U32  regData;
	

	/* Stop Tx port activity. Check port Tx activity. */
	ETH_REG_READ(SDMA_COMMAND_REGISTER(ethPortNum), &regData);

	/* Extract channel activity information */
	regData &= (ETH_START_TX_HIGH | ETH_START_TX_LOW);
	regData = (regData >> 23);	

    if(regData)
	{
		/* Issue stop command for active channels only */
		ETH_REG_WRITE(SDMA_COMMAND_REGISTER(ethPortNum), (regData << 16));

		/* Wait for all Tx activity to terminate. */
		do
		{
			/* Check port cause register that Tx high and low stopped */
			ETH_REG_READ(SDMA_COMMAND_REGISTER(ethPortNum), &regData);
		}
		while(regData & (ETH_START_TX_HIGH | ETH_START_TX_LOW));
	}
	
    /* Stop Rx port activity */
    ETH_REG_WRITE(SDMA_COMMAND_REGISTER(ethPortNum), ETH_ABORT_RECEIVE);
	
	/* Wait for all Rx activity to terminate */
	do
	{
		ETH_REG_READ(SDMA_COMMAND_REGISTER(ethPortNum), &regData);
	}
	while(regData & ETH_ABORT_RECEIVE);
	
	/* Reset the Enable bit in the Configuration Register */
  	ETH_REG_READ(PORT_CONFIG_REG(ethPortNum), &regData);
	regData &= ~PORT_ENABLE;
    ETH_REG_WRITE(PORT_CONFIG_REG(ethPortNum), regData);
	
	return;
}

/*******************************************************************************
* ethPortInitRxMemory - Curve an Rx chain desc list and buffer in memory.
*
* DESCRIPTION:
*       This function prepares an Rx chained list of descriptors and packet 
*       buffers in a form of a ring. The routine is called during port 
*       initialization routine and before port start routine. 
*       The Ethernet SDMA engine uses CPU bus addresses to access the various 
*       devices in the system (i.e. DRAM). 
*
* INPUT:
*       pQueueCtrl     - Pointer to Queue control structure
*       rxDescNum      - Number of Rx descriptors
*       rxBuffSize     - Size of Rx buffer
*       rxDescBaseAddr - Rx descriptors memory area base addr.
*       rxBuffBaseAddr - Rx buffer memory area base addr.
*
* OUTPUT:
*       The routine updates the Ethernet port control struct with information 
*       regarding the Rx descriptors and buffers.
*
* RETURN:
*		N/A.
*
*******************************************************************************/
static void ethPortInitRxMemory(ETH_QUEUE_CTRL* pQueueCtrl,
                                int    	rxDescNum,
				    	        int    	rxBuffSize,
						        MV_U32 	rxDescBaseAddr,
						        MV_U32 	rxBuffBaseAddr)
{
	ETH_RX_DESC	 *pRxDesc;
	ETH_RX_DESC  *pRxPrevDesc; /* pointer to link with the last descriptor */
	MV_U32 		 bufferAddr; 
    int			 ix;				/* a counter */

    DRV_PRINT(DRV_DEBUG_MEM, ("mvEthenret: In ethPortInitRxMemory\n"));
    DRV_PRINT(DRV_DEBUG_MEM, ("Rx Descriptor number 0x%x\n", rxDescNum));
	DRV_PRINT(DRV_DEBUG_MEM, ("Rx Buffer size 0x%x\n", rxBuffSize));
	DRV_PRINT(DRV_DEBUG_MEM, ("Rx Descriptor base addr 0x%x\n",rxDescBaseAddr));
	DRV_PRINT(DRV_DEBUG_MEM, ("Rx Buffer base addr 0x%0x\n", rxBuffBaseAddr));

    pRxDesc		 = (ETH_RX_DESC*)rxDescBaseAddr;
	pRxPrevDesc	 = pRxDesc;
	bufferAddr   = rxBuffBaseAddr;

	/* initialize the Rx descriptors ring */
    for (ix = 0; ix < rxDescNum; ix++)
	{
        pRxDesc->bufSize     = rxBuffSize;
        pRxDesc->byteCnt     = 0x0000;
        pRxDesc->cmdSts      = ETH_BUFFER_OWNED_BY_DMA | ETH_ENABLE_INTERRUPT;
        pRxDesc->nextDescPtr = PHY_MEM(pRxDesc) + RX_DESC_ALIGNED_SIZE;
        pRxDesc->bufPhyPtr 	 = PHY_MEM(bufferAddr);
        pRxDesc->bufPtr      = bufferAddr;
        pRxDesc->userInfo1   = 0x00000000;
        pRxDesc->userInfo2   = 0x00000000;
        D_CACHE_FLUSH_LINE (pRxDesc, 0);
		bufferAddr += rxBuffSize;
        pRxPrevDesc = pRxDesc;
		pRxDesc = (ETH_RX_DESC*)((MV_U32)pRxDesc + RX_DESC_ALIGNED_SIZE);
    }

	/* Closing RX descriptors ring */
    pRxPrevDesc->nextDescPtr = PHY_MEM(rxDescBaseAddr);
    D_CACHE_FLUSH_LINE (pRxPrevDesc, 0);

	/* Save Rx desc pointer to driver struct. */
	pQueueCtrl->pCurrentDescr = (void*)rxDescBaseAddr;
	pQueueCtrl->pUsedDescr = (void*)rxDescBaseAddr;
	
	pQueueCtrl->pFirstDescr = (void*)rxDescBaseAddr;
	pQueueCtrl->pLastDescr = (void*)(rxDescBaseAddr + 
                            		((rxDescNum-1)*RX_DESC_ALIGNED_SIZE));

    return;
}
/*******************************************************************************
* ethPortInitTxMemory - Curve a Tx chain desc list and buffer in memory.
*
* DESCRIPTION:
*       This function prepares a Tx chained list of descriptors and packet 
*       buffers in a form of a ring. The routine is called during port 
*       initialization routine and before port start routine. 
*       The Ethernet SDMA engine uses CPU bus addresses to access the various 
*       devices in the system (i.e. DRAM).
*
* INPUT:
*		ethPortNum 	   - Ethernet port number.
*	    txQueue        - Number of Tx queue.
*       txDescNum      - Number of Tx descriptors
*       txBuffSize     - Size of Tx buffer
*       txDescBaseAddr - Tx descriptors memory area base addr.
*       txBuffBaseAddr - Tx buffer memory area base addr.
*
* OUTPUT:
*       The routine updates the Ethernet port control struct with information 
*       regarding the Tx descriptors and buffers.
*
* RETURN:
*		N/A.
*
*******************************************************************************/
static void ethPortInitTxMemory(ETH_QUEUE_CTRL* pQueueCtrl,
                                int 	txDescNum,
                                int		txBuffSize,
                                MV_U32 	txDescBaseAddr,
                                MV_U32 	txBuffBaseAddr)
{

	ETH_TX_DESC	 *pTxDesc;
	ETH_TX_DESC  *pTxPrevDesc;
	MV_U32 bufferAddr; 
    int			 ix;				/* a counter */

	DRV_PRINT(DRV_DEBUG_MEM, ("mvEthenret: In ethPortInitTxMemory\n"));
	DRV_PRINT(DRV_DEBUG_MEM, ("Tx Descriptor number 0x%x\n", txDescNum));
	DRV_PRINT(DRV_DEBUG_MEM, ("Tx Buffer size 0x%x\n", txBuffSize));
	DRV_PRINT(DRV_DEBUG_MEM, ("Tx Descriptor base addr 0x%x\n",txDescBaseAddr));
	DRV_PRINT(DRV_DEBUG_MEM, ("Tx Buffer base addr 0x%0x\n", txBuffBaseAddr));


    /* save the first desc pointer to link with the last descriptor */
    pTxDesc		 = (ETH_TX_DESC*)txDescBaseAddr;
	pTxPrevDesc	 = pTxDesc;
	bufferAddr   = txBuffBaseAddr;

	/* Initialize the Tx descriptors ring */
    for (ix = 0; ix < txDescNum; ix++)
    {
        pTxDesc->byteCnt     = 0x0000;
        pTxDesc->shadow      = 0x0000;
        pTxDesc->cmdSts      = 0x00000000;
        pTxDesc->nextDescPtr = PHY_MEM(pTxDesc) + TX_DESC_ALIGNED_SIZE;  
        pTxDesc->bufPhyPtr   = PHY_MEM(bufferAddr);
        pTxDesc->userInfo1   = 0x00000000;
        pTxDesc->userInfo2   = 0x00000000;
        D_CACHE_FLUSH_LINE (pTxDesc, 0);
		bufferAddr += txBuffSize;
        pTxPrevDesc = pTxDesc;
		pTxDesc = (ETH_TX_DESC*)((MV_U32)pTxDesc + TX_DESC_ALIGNED_SIZE);

    }    
	/* Closing Tx descriptors ring */
    pTxPrevDesc->nextDescPtr = PHY_MEM(txDescBaseAddr);
    D_CACHE_FLUSH_LINE (pTxPrevDesc, 0);

	/* Set Tx desc pointer in driver struct. */
	pQueueCtrl->pCurrentDescr = (void*)txDescBaseAddr;
	pQueueCtrl->pUsedDescr = (void*)txDescBaseAddr;
    
    /* Init Tx ring base and size parameters */
    pQueueCtrl->pFirstDescr = (void*)txDescBaseAddr;
	pQueueCtrl->pLastDescr = (void*)(txDescBaseAddr + 
                            ((txDescNum-1)*TX_DESC_ALIGNED_SIZE));
	  
    return;
}

/*******************************************************************************
* mvEthPortSend - Send an Ethernet packet
*
* DESCRIPTION:
*		This routine send a given packet described by pBufinfo parameter. It 
*       supports transmitting of a packet spaned over multiple buffers. The 
*       routine updates 'curr' indexe according to the packet passed to the 
*		routine.
*		Note: The transmit operation is on packet basis. It supports Tx of
*		      packet consist of multiple buffers.  
*
* INPUT:
*		ethPortNum - Ethernet port number.
*	    txQueue    - Number of Tx queue.
*	    *pBufInfo  - Pointer to packet sent by user.
*
* OUTPUT:
*		Tx ring 'curr' indexe is updated. Tx resource counter updated.
*
* RETURN:
*       MV_NO_RESOURCE  - Tx resource error.
*		MV_ERROR   		- Routine can not access Tx desc ring.
*		MV_NO_MORE		- Packet placed into port HW but no resource left.
*       MV_OK      		- Packet placed into port HW.
*
*******************************************************************************/
MV_STATUS mvEthPortSend(int ethPortNum, int txQueue, BUF_INFO *pBufInfo)
{
	ETH_TX_DESC* 	pTxCurrDesc;
	ETH_TX_DESC* 	pTxUsedDesc;
    ETH_TX_DESC*	pTxFirstDesc = 0;
    ETH_TX_DESC* 	pTxNextCurr;
	MV_U32 			commandStatus;
    ETH_QUEUE_CTRL* pQueueCtrl;
	int             packetBufCount = 1;
	
    pQueueCtrl = &portCtrl[ethPortNum].txQueue[txQueue];

	/* Do not process Tx ring in case of Tx ring resource error */
	if(pQueueCtrl->resource == 0)
		return MV_NO_RESOURCE;
    
	/* Get the Tx Desc ring indexes */
    pTxCurrDesc = pQueueCtrl->pCurrentDescr;
    pTxUsedDesc = pQueueCtrl->pUsedDescr;

	if(pTxCurrDesc == 0)
        return MV_ERROR;
	
	while(MV_TRUE)
	{	
		/* The following parameters are used to save readings from memory */
		pTxNextCurr = TX_NEXT_DESC_PTR(pTxCurrDesc, pQueueCtrl);
		commandStatus   = pBufInfo->cmdSts;
		
		if (packetBufCount == 1)
		{
			/* Update first desc */
			pTxFirstDesc  = pTxCurrDesc;
			commandStatus |= ETH_FIRST_DESC;
		}
		else
		{
			commandStatus |= ETH_BUFFER_OWNED_BY_DMA;
		}
	
        pTxCurrDesc->userInfo1 = pBufInfo->userInfo1;	
        pTxCurrDesc->userInfo2 = pBufInfo->userInfo2;	

		/* Buffers with a payload smaller than 8 bytes must be aligned to  	*/
		/* 64-bit boundary. We use the memory allocated for Tx descriptor. 	*/
		/* This memory located in TX_BUF_OFFSET_IN_DESC offset within the  	*/
		/* Tx descriptor.													*/
		if(pBufInfo->byteCnt <= MIN_TX_BUFF_LOAD)
		{
			pTxCurrDesc->bufPhyPtr = (MV_U32)pTxCurrDesc+ TX_BUF_OFFSET_IN_DESC;
			ethBCopy((MV_U32)pBufInfo->pData, 
					 pTxCurrDesc->bufPhyPtr, 
					 pBufInfo->byteCnt);
			pTxCurrDesc->bufPhyPtr = PHY_MEM(pTxCurrDesc->bufPhyPtr);
		}
		else
		{
			pTxCurrDesc->bufPhyPtr = PHY_MEM(pBufInfo->pData);
		}
		
		pTxCurrDesc->byteCnt     = pBufInfo->byteCnt;	
	
        
		/* Is this last (or first and last) buffer in packet */
		if( (pBufInfo->pNextBufInfo == 0) ||
            	(pBufInfo->pNextBufInfo->pData == 0) )			 
		{        
			/* Set last desc with DMA ownership and interrupt enable. */
			pTxCurrDesc->cmdSts = commandStatus            	| 
								  ETH_BUFFER_OWNED_BY_DMA	| 
								  ETH_ENABLE_INTERRUPT		|
								  ETH_GEN_CRC			|
								  ETH_ZERO_PADDING		|
								  ETH_LAST_DESC;
			D_CACHE_FLUSH_LINE ((MV_U32)pTxCurrDesc,  0);
	
			/* Update first buffer in multiple buffer per packet */
			if(packetBufCount != 1)	
			{
				pTxFirstDesc->cmdSts |= ETH_BUFFER_OWNED_BY_DMA;
				D_CACHE_FLUSH_LINE ((MV_U32)pTxFirstDesc, 0);
			}
			/* Flush CPU pipe */		
			CPU_PIPE_FLUSH;
	
			/* Apply send command */
			ETH_REG_WRITE(SDMA_COMMAND_REGISTER(ethPortNum),						    
						   (1 << (23 + ((txQueue + 1) %2) ) ) );
			
			/* Finish Tx packet. */
            		pQueueCtrl->pCurrentDescr = pTxNextCurr;
			
			pQueueCtrl->resource -= packetBufCount;

			/* Check Tx resources availability for next packets */
			if(pQueueCtrl->resource == 0)
			{
				/* Packet transmitted but there are no more resources */
				return MV_NO_MORE; 
			}
			else
			{
				/* Update the current descriptor */
				return MV_OK;
			}
	
		}
		/* We are in a middle of a packet consists of multiple buffers */
		else
		{
			/* Check Tx resources availability for next buffers in packet */
			if(pQueueCtrl->resource == 0)
			{
				/* No more buffers for this packet. Abort this packet 	*/
				/* Return 'current' index back to 'first' index 		*/

				ETH_TX_DESC* 	pTxAbortDesc;

				for(pTxAbortDesc = pTxFirstDesc; pTxAbortDesc == pTxUsedDesc;)
				{
					pTxAbortDesc = TX_NEXT_DESC_PTR(pTxAbortDesc, pQueueCtrl);
					pTxAbortDesc->cmdSts = 0;
				
				}
				
				pQueueCtrl->pCurrentDescr = pTxFirstDesc;
				return MV_NO_RESOURCE; 
			}
			else
			{
				/* Update the current descriptor. Go to next buffer */
				
				pTxCurrDesc->cmdSts	= commandStatus;
				D_CACHE_FLUSH_LINE ((MV_U32)pTxCurrDesc,  0);
	
				pTxCurrDesc = pTxNextCurr;
				pBufInfo = pBufInfo->pNextBufInfo;
				packetBufCount++;
            }
		}
	}
}

/*******************************************************************************
* mvEthPortSendDone - Free all used Tx descriptors.
*
* DESCRIPTION:
*		This routine returns the transmitted packet information to the caller
*		(on a packet basis. No buffer basis).
*       It uses 'first' index to support Tx desc return in case a transmit 
*       of a packet spanned over multiple buffer still in process.
*       In case the Tx queue was in "resource error" condition, where there are 
*       no available Tx resources, the function resets the resource error state.
*		Note: The transmit operation is on packet basis. It supports Tx of
*		      packet consist of multiple buffers.  
*
* INPUT:
*		ethPortNum - Ethernet port number.
*	    txQueue    - Number of Tx queue.
*	    *pBufInfo  - Output buffer where the driver places packet Tx status.
*
* OUTPUT:
*		Tx ring 'used' indexe is updated. Tx resource counter updated.
*
* RETURN:
*		MV_ERROR   - Routine can not access Tx desc ring.
*		MV_NO_MORE - Nothing to release.
*       MV_OK      - Resource of transmitted packet returned successfuly.
*
*******************************************************************************/
MV_STATUS mvEthPortSendDone(int ethPortNum, int txQueue, BUF_INFO *pBufInfo)
{
    ETH_TX_DESC*	pTxUsedDesc;
    ETH_TX_DESC*	pTxCurrDesc;
    ETH_TX_DESC*    pTxFirstDesc = 0;
    ETH_TX_DESC* 	pTxNextUsed;
	MV_U32 	        commandStatus;
    ETH_QUEUE_CTRL* pQueueCtrl;
	int             packetBufCount = 1;
	
    pQueueCtrl = &portCtrl[ethPortNum].txQueue[txQueue];

	/* Get the Tx Desc ring indexes */
    pTxCurrDesc = pQueueCtrl->pCurrentDescr;
    pTxUsedDesc = pQueueCtrl->pUsedDescr;

    /* Sanity check */
    if(pTxUsedDesc == 0)
		return MV_ERROR;
			
	/* Nothing to release */
	if((pTxUsedDesc == pTxCurrDesc) && 
	   (pQueueCtrl->resource != 0))
		return MV_EMPTY;	
	
    pTxFirstDesc = pTxUsedDesc;
	
	while(MV_TRUE)
	{
		commandStatus = pTxUsedDesc->cmdSts;
	
		/* Still transmitting... */
		if (commandStatus & (ETH_BUFFER_OWNED_BY_DMA)) 
		{         
			/* Undo 'current' index. */
			pQueueCtrl->pUsedDescr = pTxFirstDesc;

			return MV_NO_MORE;
		}                            
			
		D_CACHE_INVALIDATE_LINE((MV_U32)pTxUsedDesc, 0);
					
		if (commandStatus & (ETH_FIRST_DESC)) 
		{   
			/* Pass the packet information to the caller */
			pBufInfo->cmdSts     = commandStatus;
			pBufInfo->userInfo1  = pTxUsedDesc->userInfo1;
			pBufInfo->userInfo2  = pTxUsedDesc->userInfo2;
		}
		pTxNextUsed = TX_NEXT_DESC_PTR(pTxUsedDesc, pQueueCtrl);
		
        if (commandStatus & (ETH_LAST_DESC)) 
		{
			
			/* Update the next descriptor to release. */
			pQueueCtrl->pUsedDescr = pTxNextUsed;
					
			/* Any Tx return cancels the Tx resource error status */
			pQueueCtrl->resource += packetBufCount;

			return MV_OK;
		}
		else
		{
			pTxUsedDesc = pTxNextUsed;
			packetBufCount++;
		}
	}
}

/*******************************************************************************
* mvEthPortGetTxResource - Get Ethenret port Tx resource counter.
*
* DESCRIPTION:
*		This function retrieves the current Tx resource counter.
*
* INPUT:
*		ethPortNum - Ethernet port number.
*	    txQueue    - Number of Tx queue.
*
* OUTPUT:
*		N/A.
*
* RETURN:
*		Number of Tx resources.
*
*******************************************************************************/
MV_32 mvEthPortGetTxResource(MV_32 ethPortNum, MV_32 txQueue)
{
	return (portCtrl[ethPortNum].txQueue[txQueue].resource);
		
}

/*******************************************************************************
* mvEthPortReceive - Get received information from Rx ring.
*
* DESCRIPTION:
* 		This routine returns to the caller the pointer to recieved buffer. 
*		There is no data copying during routine operation. All information 
*		is returned using pointer to buffer information struct passed from 
*		the caller. 
*		In order to support virtual addressing, the driver returns the virtual 
*		buffer address stored in the descriptor.
*		Note: The receive operation is on buffer basis. It does not support
*		      multiple buffer per Rx packet.  
*
* INPUT:
*		ethPortNum - Ethernet port number.
*	    rxQueue    - Number of Rx queue.
*	    *pBufInfo  - Output buffer where the driver places Rx buffer info.
*
* OUTPUT:
*		Rx ring current indexe is updated. Rx resource counter updated.
*
* RETURN:
*       MV_NO_RESOURCE  - Rx resource error.
*		MV_ERROR   		- Routine can not access Rx desc ring.
*		MV_NO_MORE		- Buffer still in HW receive process.
*       MV_OK      		- Received buffer information is available to caller.
*
*******************************************************************************/
MV_STATUS mvEthPortReceive(int ethPortNum, int rxQueue, BUF_INFO *pBufInfo)
{
    ETH_RX_DESC     *pRxCurrDesc;
	MV_U32          commandStatus;
    ETH_QUEUE_CTRL* pQueueCtrl;

    pQueueCtrl = &(portCtrl[ethPortNum].rxQueue[rxQueue]);

	/* Do not process Rx ring in case of Rx ring resource error */
	if(pQueueCtrl->resource == 0)
		return MV_NO_RESOURCE;

	/* Get the Rx Desc ring 'curr and 'used' indexes */
	pRxCurrDesc = pQueueCtrl->pCurrentDescr;
    
	/* Sanity check */
	if (pRxCurrDesc == 0)
		return MV_ERROR;
		
	/* The following parameters are used to save readings from memory */
	commandStatus	= pRxCurrDesc->cmdSts;

	/* Still receiving... Undo receive for this packet. */
	if (commandStatus & (ETH_BUFFER_OWNED_BY_DMA))
	{
        D_CACHE_INVALIDATE_LINE ((MV_U32)pRxCurrDesc, 0);
			
		return MV_NO_MORE;
	}                            
		
	pBufInfo->byteCnt   = pRxCurrDesc->byteCnt;
	pBufInfo->cmdSts    = commandStatus;
	pBufInfo->pBuff		= (MV_U8*)pRxCurrDesc->bufPtr;
	pBufInfo->pData		= (MV_U8*)pRxCurrDesc->bufPtr;
    pBufInfo->pNextBufInfo = NULL;
	pBufInfo->userInfo1 = pRxCurrDesc->userInfo1;
	pBufInfo->userInfo2 = pRxCurrDesc->userInfo2;
	
	/* Clean the return info field to indicate that the packet has been */
	/* moved to the upper layers 										*/
	/*pRxCurrDesc->returnInfo = 0; */
			
	pQueueCtrl->resource--;

	/* Update 'curr' in data structure */
	pQueueCtrl->pCurrentDescr = RX_NEXT_DESC_PTR(pRxCurrDesc, pQueueCtrl);
			
	/* Invalidate descriptor in cache. Next access will be from memory */
	D_CACHE_INVALIDATE_LINE ((MV_U32)pRxCurrDesc, 0);

	return MV_OK;

}

/*******************************************************************************
* mvEthPortReceiveDone - Returns an Rx buffer back to the Rx ring.
*
* DESCRIPTION:
*		This routine returns an Rx buffer back to the Rx ring. It retrieves the 
*       next 'used' descriptor and attached the returned buffer to it.
*		In order to support virtual addressing, the driver stores the virtual 
*		buffer address in descriptor.
*		Note: The receive operation is on buffer basis. It does not support
*		      multiple buffer per Rx packet.  
*
* INPUT:
*		ethPortNum - Ethernet port number.
*	    rxQueue    - Number of Rx queue.
*	    *pBufInfo  - Input buffer where the caller places Rx buffer to return.
*
* OUTPUT:
*		New available Rx resource in Rx descriptor ring.
*
* RETURN:
*		MV_ERROR in case the routine can not access Rx desc ring.
*       MV_OK otherwise.
*
*******************************************************************************/
MV_STATUS mvEthPortReceiveDone(int ethPortNum, int rxQueue, BUF_INFO *pBufInfo)
{
	ETH_RX_DESC*    pRxUsedDesc;	/* Where to return Rx resource */
    ETH_QUEUE_CTRL* pQueueCtrl;
            
    pQueueCtrl = &portCtrl[ethPortNum].rxQueue[rxQueue];

	/* Get 'used' Rx descriptor */
	pRxUsedDesc = pQueueCtrl->pUsedDescr;
	
    /* Sanity check */
	if (pRxUsedDesc == 0)
		return MV_ERROR;
	
	pRxUsedDesc->bufSize   = pBufInfo->byteCnt;
	pRxUsedDesc->bufPhyPtr = PHY_MEM(pBufInfo->pBuff);
	pRxUsedDesc->bufPtr    = (MV_U32)pBufInfo->pBuff;
	pRxUsedDesc->userInfo1 = pBufInfo->userInfo1;
	pRxUsedDesc->userInfo2 = pBufInfo->userInfo2;
		
    /* Return the descriptor to DMA ownership */
	pRxUsedDesc->cmdSts = ETH_BUFFER_OWNED_BY_DMA | ETH_ENABLE_INTERRUPT;
		
	/* Flush descriptor and CPU pipe */
	D_CACHE_FLUSH_LINE ((MV_U32)pRxUsedDesc, 0);
	CPU_PIPE_FLUSH;
	
	pQueueCtrl->resource++;

    /* Move the used descriptor pointer to the next descriptor */
	pQueueCtrl->pUsedDescr = RX_NEXT_DESC_PTR(pRxUsedDesc, pQueueCtrl);
				
	return MV_OK;
}

/*******************************************************************************
* mvEthPortGetRxResource - Get Ethenret port Rx resource counter.
*
* DESCRIPTION:
*		This function retrieves the current Rx resource counter.
*
* INPUT:
*		ethPortNum - Ethernet port number.
*	    rxQueue    - Number of Rx queue.
*
* OUTPUT:
*		N/A.
*
* RETURN:
*		Number of Rx resources.
*
*******************************************************************************/
MV_32 mvEthPortGetRxResource(MV_32 ethPortNum, MV_32 rxQueue)
{
	return (portCtrl[ethPortNum].rxQueue[rxQueue].resource);		
}

/*******************************************************************************
* ethBCopy - Copy bytes from source to destination
*
* DESCRIPTION:
*       This function supports the eight bytes limitation on Tx buffer size. 
*       The routine will zero eight bytes starting from the destination address
*       followed by copying bytes from the source address to the destination.
*
* INPUT:
*       srcAddr   - 32 bit source address.
*       dstAddr   - 32 bit destination address.
*       byteCount - Number of bytes to copy.
*
* OUTPUT:
*       See description.
*
* RETURN:
*       None.
*
*******************************************************************************/
void ethBCopy(MV_U32 srcAddr, MV_U32 dstAddr, int byteCount)
{
	while(byteCount != 0)
	{
		*(char*)dstAddr = *(char*)srcAddr;
		dstAddr++;
		srcAddr++;
		byteCount--;
	}
}

void  ethPortRegs(int port)  
{
    MV_U32  regValue;

    mvOsPrintf("Ethernet port #%d registers:\n\n", port);


    ETH_REG_READ( PORT_CONFIG_REG(port), &regValue);
  	mvOsPrintf("PORT_CONFIG_REG         : 0x%X = 0x%08lx\n", 
				PORT_CONFIG_REG(port), regValue);

    ETH_REG_READ( PORT_CONFIG_EXTEND_REG(port), &regValue);
  	mvOsPrintf("PORT_CONFIG_EXTEND_REG  : 0x%X = 0x%08lx\n", 
				PORT_CONFIG_EXTEND_REG(port), regValue);

    ETH_REG_READ( PORT_COMMAND_REGISTER(port), &regValue);
  	mvOsPrintf("PORT_COMMAND_REGISTER   : 0x%X = 0x%08lx\n", 
				PORT_COMMAND_REGISTER(port), regValue);

    ETH_REG_READ( PORT_STATUS_REGISTER(port), &regValue);
  	mvOsPrintf("PORT_STATUS_REGISTER    : 0x%X = 0x%08lx\n", 
				PORT_STATUS_REGISTER(port), regValue);

    ETH_REG_READ( SDMA_CONFIG_REGISTER(port), &regValue);
  	mvOsPrintf("SDMA_CONFIG_REGISTER    : 0x%X = 0x%08lx\n", 
				SDMA_CONFIG_REGISTER(port), regValue);

    ETH_REG_READ( SDMA_COMMAND_REGISTER(port), &regValue);
  	mvOsPrintf("SDMA_COMMAND_REGISTER   : 0x%X = 0x%08lx\n", 
				SDMA_COMMAND_REGISTER(port), regValue);


    ETH_REG_READ( ETH_CAUSE_REG(port), &regValue);
  	mvOsPrintf("ETH_CAUSE_REG           : 0x%X = 0x%08lx\n", 
				ETH_CAUSE_REG(port), regValue);

    ETH_REG_READ( ETH_RESET_SELECT_REG(port), &regValue);
  	mvOsPrintf("ETH_RESET_SELECT_REG    : 0x%X = 0x%08lx\n", 
				ETH_RESET_SELECT_REG(port), regValue);

    ETH_REG_READ( ETH_MASK_REG(port), &regValue);
  	mvOsPrintf("ETH_MASK_REG            : 0x%X = 0x%08lx\n", 
				ETH_MASK_REG(port), regValue);
}
    
    
void  ethPortQueues(int port, int rxQueue, int txQueue)  
{
    ETH_QUEUE_CTRL  *pQueueCtrl;
    MV_U32          regValue;
    ETH_RX_DESC     *pRxDescr;
    ETH_TX_DESC     *pTxDescr;
	int				i;

    if( (rxQueue >=0) && (rxQueue < ETH_MAX_RX_QUEUE_NUM) )
    {
        pQueueCtrl = &portCtrl[port].rxQueue[rxQueue];
        mvOsPrintf("Port #%d, RX Queue #%d\n\n", port, rxQueue);

        ETH_REG_READ( FIRST_RX_DESC_PTR(port, rxQueue), &regValue);
  	    mvOsPrintf("FIRST_RX_DESC_PTR       : 0x%X = 0x%08lx\n", 
				    FIRST_RX_DESC_PTR(port, rxQueue), regValue);

        ETH_REG_READ( CURR_RX_DESC_PTR(port, rxQueue), &regValue);
  	    mvOsPrintf("CURR_RX_DESC_PTR        : 0x%X = 0x%08lx\n", 
				    CURR_RX_DESC_PTR(port, rxQueue), regValue);

        if(pQueueCtrl->pFirstDescr != NULL)
        {
            mvOsPrintf("pFirstDescr=%p, pLastDescr=%p, numOfResources=%d\n",
                        pQueueCtrl->pFirstDescr, pQueueCtrl->pLastDescr, 
                        pQueueCtrl->resource);
            mvOsPrintf("pCurrDescr: %p, pUsedDescr: %p\n",
                        pQueueCtrl->pCurrentDescr, pQueueCtrl->pUsedDescr);

            pRxDescr = (ETH_RX_DESC*)pQueueCtrl->pFirstDescr;
            i = 0; 
            do {
                mvOsPrintf("%2d. pDescr=%p, cmdSts=0x%08lx, byteCnt=%4d, bufSize=%4d, bufPtr=0x%08lx\n", 
                           i, pRxDescr, pRxDescr->cmdSts, pRxDescr->byteCnt, 
                           pRxDescr->bufSize, pRxDescr->bufPhyPtr);
/*
                mvOsPrintf("\tbufPhyPtr=%p, nextDescPtr=%p, bufPtr=%p\n", 
                            pRxDescr->bufPhyPtr, pRxDescr->nextDescPtr, pRxDescr->bufPtr);
                mvOsPrintf("\tuserInfo1=0x%08x, userInfo2=0x%08x\n", 
                   	    pRxDescr->userInfo1, pRxDescr->userInfo2);
*/
                pRxDescr = RX_NEXT_DESC_PTR(pRxDescr, pQueueCtrl);
                i++;
            } while (pRxDescr != pQueueCtrl->pFirstDescr);
        }
        else
            mvOsPrintf("RX Queue #%d is NOT CREATED\n", rxQueue);
    }

    if( (txQueue >=0) && (txQueue < ETH_MAX_TX_QUEUE_NUM) )
    {
        pQueueCtrl = &portCtrl[port].txQueue[txQueue];
        mvOsPrintf("Port #%d, TX Queue #%d\n\n", port, txQueue);

        ETH_REG_READ( CURR_TX_DESC_PTR(port, (ETH_MAX_TX_QUEUE_NUM-1)-txQueue), 
                            &regValue);
  	    mvOsPrintf("CURR_TX_DESC_PTR        : 0x%X = 0x%08lx\n", 
				    CURR_TX_DESC_PTR(port, (ETH_MAX_TX_QUEUE_NUM-1)-txQueue), regValue);

        if(pQueueCtrl->pFirstDescr != NULL)
        {
            mvOsPrintf("pFirstDescr=%p, pLastDescr=%p, numOfResources=%d\n",
                        pQueueCtrl->pFirstDescr, pQueueCtrl->pLastDescr, 
                        pQueueCtrl->resource);
            mvOsPrintf("pCurrDescr: %p, pUsedDescr: %p\n",
                        pQueueCtrl->pCurrentDescr, pQueueCtrl->pUsedDescr);

            pTxDescr = (ETH_TX_DESC*)pQueueCtrl->pFirstDescr;
            i = 0; 
            do {
                mvOsPrintf("%2d. pDescr=%p, cmdSts=0x%08lx, byteCnt=%4d, bufSize=%4d, bufPtr=0x%08lx\n", 
                           i, pTxDescr, pTxDescr->cmdSts, pTxDescr->byteCnt, 
                           pTxDescr->shadow, pTxDescr->bufPhyPtr);
/*
                mvOsPrintf("\tbufPhyPtr=0x%08x, nextDescPtr=0x%08x\n", 
                            pTxDescr->bufPhyPtr, pTxDescr->nextDescPtr);
                mvOsPrintf("\tuserInfo1=0x%08x, userInfo2=0x%08x\n", 
                       pTxDescr->userInfo1, pTxDescr->userInfo2);
*/
                pTxDescr = TX_NEXT_DESC_PTR(pTxDescr, pQueueCtrl);
                i++;
            } while (pTxDescr != pQueueCtrl->pFirstDescr);
        }
        else
            mvOsPrintf("TX Queue #%d is NOT CREATED\n", txQueue);
    }
}
