/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

This software file (the "File") is owned and distributed by Marvell 
International Ltd. and/or its affiliates ("Marvell") under the following
alternative licensing terms.  Once you have made an election to distribute the
File under one of the following license alternatives, please (i) delete this
introductory statement regarding license alternatives, (ii) delete the two
license alternatives that you have not elected to use and (iii) preserve the
Marvell copyright notice above.

********************************************************************************
Marvell Commercial License Option

If you received this File from Marvell and you have entered into a commercial
license agreement (a "Commercial License") with Marvell, the File is licensed
to you under the terms of the applicable Commercial License.

********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File in accordance with the terms and conditions of the General 
Public License Version 2, June 1991 (the "GPL License"), a copy of which is 
available along with the File in the license.txt file or by writing to the Free 
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or 
on the worldwide web at http://www.gnu.org/licenses/gpl.txt. 

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED 
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY 
DISCLAIMED.  The GPL License provides additional details about this warranty 
disclaimer.
********************************************************************************
Marvell BSD License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File under the following licensing terms. 
Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    *   Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer. 

    *   Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution. 

    *   Neither the name of Marvell nor the names of its contributors may be 
        used to endorse or promote products derived from this software without 
        specific prior written permission. 
    
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*******************************************************************************/

/*******************************************************************************
* addressTable.c - this file has all the functions of the address table
*
* DESCRIPTION:
*       None.
*
* DEPENDENCIES:
*       None.
*
*******************************************************************************/

/* includes */
#include "mvUniMacRegs.h"
#include "mvEthDrv.h"
#include "mvEthAddrFilter.h"


/* defines  */

#define MAC_ENTRY_SIZE                      8
#define MAC_ADDRESS_STRING_SIZE             12
#define _8K_TABLE                           0

#define HASH_DEFUALT_MODE                   14
#define HASH_MODE                           13
#define HASH_SIZE                           12
#define PROMISCUOUS_MODE                    0

#define _8K_TABLE                           0

#define VALID_BIT							0
#define VALID_MASK							(1<<VALID_BIT)

#define SKIP_BIT                            1
#define SKIP_MASK                           (1<<SKIP_BIT)

#define RD_BIT								2
#define RD_MASK								(1<<RD_BIT)

#define PRIO_OFFSET							(51-32)
#define PRIO_MASK							(3<<PRIO_OFFSET)

#define HOP_NUMBER                          2
#define EIGHT_K                             0x8000
#define HALF_K                              0x8000/16

#define MAX_NUMBER_OF_ADDRESSES_TO_STORE    1000



          
#define MV_NIBBLE(X)          \
        (((X&0x1) << 3 ) +    \
         ((X&0x2) << 1 ) +    \
         ((X&0x4) >> 1 ) +    \
         ((X&0x8) >> 3 ) )

#define FLIP_6_BITS(X)        \
        (((X&0x1) << 5 ) +    \
         ((X&0x2) << 3 ) +    \
         ((X&0x4) << 1 ) +    \
         ((X&0x8) >> 1 ) +    \
         ((X&0x10) >> 3) +    \
         ((X&0x20) >> 5) )
        
#define FLIP_9_BITS(X)        \
        (((X&0x1 ) << 8) +    \
         ((X&0x2 ) << 6) +    \
         ((X&0x4 ) << 4) +    \
         ((X&0x8 ) << 2) +    \
         ((X&0x10) << 0) +    \
         ((X&0x20) >> 2) +    \
         ((X&0x40) >> 4) +    \
         ((X&0x80) >> 6) +    \
         ((X&0x100) >> 8) )

/* typedefs */

/* Describes address table entry command */
typedef enum _mvAddrTblCmd
{
	MV_ADDR_ENTRY_DISCARD = 0,
	MV_ADDR_ENTRY_RECEIVE = 1,

}MV_ADDR_TBL_CMD;

/* This struct describes address filter table entry */
typedef struct _mvAddrTableEntry
{
    MV_BOOL			isValid;
    MV_BOOL			isSkip;
    MV_ADDR_TBL_CMD		action;
    MV_U8			macAddr[MV_MAC_ADDR_SIZE];
    MV_U16			prio;

} MV_ADDR_TABLE_ENTRY;


/* locals   */

unsigned int addressTableHashMode[ETH_PORT_NUM];     
unsigned int addressTableHashSize[ETH_PORT_NUM];     
unsigned int addressTableBase[ETH_PORT_NUM];

static MV_STATUS initAddressTable(int ethPort, 
                            int hashMode,
					        int hashSize,
                            int hashDefaultMode,
							MV_U32 virtBaseAddr,
							MV_U32 phyBaseAddr);

static MV_32 hashTableFunction(unsigned int macH,
							unsigned int macL,
							int HashSize,
							int hash_mode);

static MV_STATUS addAddressTableEntry(int ethPort,
                            unsigned int macH,
                            unsigned int macL,
                            unsigned int rd,
                            unsigned int skip);

static MV_STATUS findFirstAddressTableEntry(int port, 
							unsigned int* startEntry,
                            MV_ADDR_TABLE_ENTRY* pAddrEntry);

/*******************************************************************************
* mvEthAddrFilterInit - Initialize Ethenret port address filter table
*
* DESCRIPTION:
*		This function initialize Ethenret port MAC address filtering table
*		driver structures according to port configuration.
*
* INPUT:
*		ethPortNum		- Ethernet port number.
*       addrTblVirtAddr - Address table memory area virtual address.
*		addrTblPhyAddr	- Address table memory area physical address.
*		portConfig		- Port configuration.
*
* OUTPUT:
*		Calls driver internal function initAddressTable() with parameters 
*		according to port configuration.
*
* RETURN:
*		initAddressTable return value.
*
*******************************************************************************/
MV_STATUS mvEthAddrFilterInit(MV_32  ethPortNum, 
                              MV_U32 addrTblVirtAddr,
                              MV_U32 addrTblPhyAddr,
                              MV_U32 portConfig)
{
    int hashTblSize, hashFuncMode, hashdefaultMode;
	MV_STATUS	 retVal;

    if (portConfig & ETH_HASH_SIZE_500)
        hashTblSize = 1;
    else
        hashTblSize = 0;
    
    if (portConfig & ETH_HASH_FUNCTION_1)
        hashFuncMode = 1;
    else
        hashFuncMode = 0;
    
    if (portConfig & ETH_HASH_PASS_ADDRS_NOT_FOUND)
        hashdefaultMode = 1;
    else
        hashdefaultMode = 0;

    retVal = initAddressTable(ethPortNum, 
                              hashFuncMode, 
                              hashTblSize, 
                              hashdefaultMode, 
                              addrTblVirtAddr,
					          addrTblPhyAddr );    
	return retVal;
}

/*******************************************************************************
* mvEthPortMacAddr - Manipulate address filter table.
*
* DESCRIPTION:
*		This function add/remove MAC addresses from the port MAC address.
*
* INPUT:
*		ethPortNum	- Ethernet port number.
*		*pAddr		- Address to be added/removed.
*		opFlag		- Operation option flag. Use ADD_MAC_ADDR, REMOVE_MAC_ADDR.
*
* OUTPUT:
*		See description.
*
* RETURN:
*		MV_OK is output succeeded.
*		MV_ERROR if addAddressTableEntry( ) failed.
*
*******************************************************************************/
MV_STATUS mvEthPortMacAddr(int  ethPortNum, unsigned char *pAddr, int  opFlag)
{
    MV_U32 macH;
    MV_U32 macL;
	MV_STATUS	 retVal;
				
	if (opFlag == CLEAR_MAC_ADDR)
    {
        /* AR: clear MAC address */
    }
	macH =  (pAddr[1] << 0) |
            (pAddr[0] << 8) ;
	
    macL =  (pAddr[5] << 0) | 
            (pAddr[4] << 8) |
			(pAddr[3] << 16)| 
            (pAddr[2] << 24);

	retVal = addAddressTableEntry(ethPortNum,
								  macH,
								  macL,
								  1,
								  opFlag);
    return retVal;
}

/*******************************************************************************
* initAddressTable - Initialize address table.
*
* DESCRIPTION:
*		This function initialize Ethenret port MAC address filtering table
*		driver structures.
*
* INPUT:
*		ethPortNum		- Ethernet port number.
* 		hashMode 		- hash mode 0 or hash mode 1.
* 		hashSize 		- Size of hash table (0=8K ,1=1/2K)
* 		hashDefaultMode - 0 = Discard addresses not found in the address table
* 						  1 = Pass addresses not found in the address table.
*       virtBaseAddr	- Address table memory area virtual address.
*		phyBaseAddr		- Address table memory area physical address.
*
* OUTPUT:
*		See description.
*
* RETURN:
*		MV_OK 
*
*******************************************************************************/
MV_STATUS initAddressTable(int ethPortNum, 
					  int hashMode,
					  int hashSize,
					  int hashDefaultMode,
                      MV_U32 virtBaseAddr,
                      MV_U32 phyBaseAddr)
{
	unsigned int hashLengh[ETH_PORT_NUM];

    hashLengh[0] = EIGHT_K; 
    hashLengh[1] = HALF_K;
    addressTableHashMode[ethPortNum] = hashMode;
    addressTableHashSize[ethPortNum] = hashSize;
    addressTableBase[ethPortNum] = virtBaseAddr;


    ETH_REG_WRITE(HASH_TABLE_BASE_ADDR(ethPortNum), phyBaseAddr);
    
    return MV_OK;
}

/*******************************************************************************
* addAddressTableEntry - Initialize address table.
*
* DESCRIPTION:
*		This function will add an entry to the address table depends on 
*		the hash mode and hash size that was initialized.
*
* INPUT:
*		ethPortNum	- Ethernet port number.
* 		macH 		- 2 most significant bytes of the MAC address.
* 		macL 		- 4 least significant bytes of the MAC address.
* 		rd 			- RD field in the address table.
* 		skip 		- If 1 , skip this address.
*
* OUTPUT:
*		See description.
*
* RETURN:
*		MV_FULL in case the address table can not include the required address.
*		MV_OK otherwise.
*
*******************************************************************************/
MV_STATUS addAddressTableEntry(int ethPortNum,
                               unsigned int macH,
                               unsigned int macL,
                               unsigned int rd,
                               unsigned int skip)
{
    unsigned int addressHighValue;
    unsigned int addressLowValue;
    void* entryNumber;
    int hashBase = addressTableBase[ethPortNum]; 
    int i;
    unsigned int addressLowRead;
    unsigned int addressHighRead;

    entryNumber = (void*)(hashBase + MAC_ENTRY_SIZE*hashTableFunction(macH,macL,
                                        (int)addressTableHashSize[ethPortNum],
                                        (int)addressTableHashMode[ethPortNum]));

    addressLowValue = VALID_MASK | (skip<<SKIP_BIT) | (rd<<RD_BIT) |
                      (((macH>>8)&0xf)<<3)   |
                      (((macH>>12)&0xf)<<7)  |
                      (((macH>>0)&0xf)<<11)  | 
                      (((macH>>4)&0xf)<<15)  | 
                      (((macL>>24)&0xf)<<19) | 
                      (((macL>>28)&0xf)<<23) |
                      (((macL>>16)&0xf)<<27) |
                      ((((macL>>20)&0x1)<<31));

    addressHighValue = ((macL>>21)&0x7)      | 
                       (((macL>>8)&0xf)<<3)  |
                       (((macL>>12)&0xf)<<7) | 
                       (((macL>>0)&0xf)<<11) | 
                       (((macL>>4)&0xf)<<15);

/*
	gtOsPrintf("High = 0x%08x, Low = 0x%08x\n", addressHighValue, 
												addressLowValue);
*/

    /* find a free place */
    for(i = 0 ; i < HOP_NUMBER ; i++)
    {
        addressLowRead = *(unsigned int*)(entryNumber+i*MAC_ENTRY_SIZE+4);
        if((!(addressLowRead & VALID_MASK))/* || (addressLowRead & SKIP)*/)
		{
            entryNumber = entryNumber+i*MAC_ENTRY_SIZE;
            break;
        }
        else /* if the address is the same locate it at the same position */
        {
            addressHighRead = *(unsigned int*)(entryNumber+i*MAC_ENTRY_SIZE);
            if(((addressLowRead>>3)&0x1fffffff)==
               ((addressLowValue>>3)&0x1fffffff)
               && ((addressHighRead)==(addressHighValue)))
            {
                entryNumber = entryNumber+i*MAC_ENTRY_SIZE;
                break;
            }
        }

    }

    if(i == HOP_NUMBER)
    {
        return MV_FULL;
    }


    /* write the address to the address table */
    *(unsigned int*)(entryNumber) = addressLowValue;
    *(unsigned int*)(entryNumber+4) = addressHighValue;

    return MV_OK;
}

/*******************************************************************************
* hashTableFunction - Calculate hash function.
*
* DESCRIPTION:
* 		This function will calculate the hash function of the address.
* 		Calculation depends on the hash mode and hash size.
*
* INPUT:
* 		macH 	 - 2 most significant bytes of the MAC address.
* 		macL	 - 4 least significant bytes of the MAC address.
* 		hashSize - Size of the hash table (0=8K ,1=1/2K).
* 		hashMode - Hash mode 0 or hash mode 1.
*
* OUTPUT:
*		N/A.
*
* RETURN:
*		Hash function calculation.
*
*******************************************************************************/
MV_32 hashTableFunction(unsigned int macH,
						unsigned int macL,
                        int HashSize,
						int hash_mode)
{
    unsigned int hashResult;
    unsigned int ethernetAddH;
    unsigned int ethernetAddL;
    unsigned int ethernetAdd0;
    unsigned int ethernetAdd1;
    unsigned int ethernetAdd2;
    unsigned int ethernetAdd3;
    unsigned int ethernetAddHSwapped = 0;
    unsigned int ethernetAddLSwapped = 0;

    ethernetAddH = MV_NIBBLE_SWAP_16BIT(macH);
    ethernetAddL = MV_NIBBLE_SWAP_32BIT(macL);

    ethernetAddHSwapped = MV_NIBBLE(ethernetAddH&0xf)+
                          ((MV_NIBBLE((ethernetAddH>>4)&0xf))<<4)+
                          ((MV_NIBBLE((ethernetAddH>>8)&0xf))<<8)+
                          ((MV_NIBBLE((ethernetAddH>>12)&0xf))<<12);


    ethernetAddLSwapped = MV_NIBBLE(ethernetAddL&0xf)+
                          ((MV_NIBBLE((ethernetAddL>>4)&0xf))<<4)+
                          ((MV_NIBBLE((ethernetAddL>>8)&0xf))<<8)+
                          ((MV_NIBBLE((ethernetAddL>>12)&0xf))<<12)+
                          ((MV_NIBBLE((ethernetAddL>>16)&0xf))<<16)+
                          ((MV_NIBBLE((ethernetAddL>>20)&0xf))<<20)+
                          ((MV_NIBBLE((ethernetAddL>>24)&0xf))<<24)+
                          ((MV_NIBBLE((ethernetAddL>>28)&0xf))<<28);



    ethernetAddH = ethernetAddHSwapped;
    ethernetAddL = ethernetAddLSwapped;

    if(hash_mode == 0)
    {
        ethernetAdd0 = (ethernetAddL>>2) & 0x3f;
        ethernetAdd1 = (ethernetAddL & 0x3) | ((ethernetAddL>>8) & 0x7f)<<2;
        ethernetAdd2 = (ethernetAddL>>15) & 0x1ff;
        ethernetAdd3 = ((ethernetAddL>>24) & 0xff) | ((ethernetAddH & 0x1)<<8);

    }
    else
    {
        ethernetAdd0 = FLIP_6_BITS((ethernetAddL) & 0x3f);
        ethernetAdd1 = FLIP_9_BITS(((ethernetAddL>>6) & 0x1ff));
        ethernetAdd2 = FLIP_9_BITS((ethernetAddL>>15) & 0x1ff);
        ethernetAdd3 = FLIP_9_BITS((((ethernetAddL>>24) & 0xff) | 
                                    ((ethernetAddH & 0x1)<<8)));

    }

    hashResult = (ethernetAdd0<<9) | (ethernetAdd1^ethernetAdd2^ethernetAdd3);

    if(HashSize == _8K_TABLE)
    {
        hashResult = hashResult & 0xffff;
    }
    else
    {
        hashResult = hashResult & 0x7ff;
    }

    return hashResult;
}

/*******************************************************************************
* mvEthPortFilterEnable - Enable/Disable address filtering
*
* DESCRIPTION:
*       This function enables or disables address filtering.
*
* INPUT:
*       ethPortNum  - Ethernet port number.
*       enabler		- Enabler parameter where 1 enables filtering and 0 disables
*                 	  filtering (promiscuous mode)
*
* OUTPUT:
*       Address filtering enabled or disabled.
*
* RETURN:
*       MV_OK.
*
*******************************************************************************/
MV_STATUS  mvEthPortFilterEnable(int ethPortNum, MV_BOOL enabler)
{
	MV_U32  regValue;
    
    ETH_REG_READ(PORT_CONFIG_REG(ethPortNum), &regValue);
    
    if (enabler)
    {
        regValue &= ~ETH_PROMISCUOUS_MODE;  /* Enable filtering */
    }
    else
    {
        regValue |= ETH_PROMISCUOUS_MODE;   /* Disable filtering */
    }
    
    ETH_REG_WRITE(PORT_CONFIG_REG(ethPortNum), regValue);

    return MV_OK;
}

/*******************************************************************************
* scanAddressTable - Scan address filter table
*
* DESCRIPTION:
* 		This function will scan and print all the address table valid entries.
*
* INPUT:
*       ethPortNum  - Ethernet port number.
*
* OUTPUT:
*       N/A.
*
* RETURN:
*       Number of valid entries.
*
*******************************************************************************/
MV_32 scanAddressTable(int ethPortNum)
{
    unsigned int		entry = 0;
	int					count = 0;
	MV_ADDR_TABLE_ENTRY	addrEntry;
	char				macStr[100];

    while(findFirstAddressTableEntry(ethPortNum, &entry, &addrEntry) == MV_OK)
	{
		sprintf(macStr, "%02x:%02x:%02x:%02x:%02x:%02x", 
							addrEntry.macAddr[0],
							addrEntry.macAddr[1],
							addrEntry.macAddr[2],
							addrEntry.macAddr[3],
							addrEntry.macAddr[4],
							addrEntry.macAddr[5]);			

		printf("%d. MAC=%s, HashEntry=0x%x, SKIP=%d, RD=%d, Prio=%d\n",
							count, macStr, entry, addrEntry.isSkip, 
							addrEntry.action, addrEntry.prio);

		count++;
	}

    return count;
}

/* MV_STATUS findFirstAddressTableEntry(int port,unsigned int* startEntry)
* 
* This function will scan and print the first address table valid entry.
* Inputs
* port - ETHERNET port number.
* MV_OK if success.
* MV_FAIL if fail to make the assignment.
*/
/*******************************************************************************
* findFirstAddressTableEntry - Find first address filter table valid entry
*
* DESCRIPTION:
* 		This function will find first address filter table valid entry.
*
* INPUT:
*       ethPortNum  - Ethernet port number.
*		*startEntry - Address filter table entry number output buffer.
*       *pAddrEntry - Address output buffer.
*
* OUTPUT:
*       Output buffers are updated with rlevent data.
*
* RETURN:
*
*******************************************************************************/
MV_STATUS findFirstAddressTableEntry(int port,
									 unsigned int* startEntry,
                                     MV_ADDR_TABLE_ENTRY* pAddrEntry)
{
    unsigned int currentEntry;
    unsigned int hashLengh[ETH_PORT_NUM];
    unsigned int addressLowRead, addressHighRead;
    unsigned int endTable, firstEntry;
    int			 hashSize;

    hashLengh[0] = EIGHT_K; 
    hashLengh[1] = HALF_K;

    hashSize = addressTableHashSize[port];
    endTable = (addressTableBase[port]+hashLengh[hashSize]*MAC_ENTRY_SIZE);
    firstEntry = *startEntry;

    currentEntry = addressTableBase[port]+((firstEntry)*MAC_ENTRY_SIZE);
    while(currentEntry < endTable)
    {
        addressLowRead = *(unsigned int*)(currentEntry);
        if(addressLowRead & VALID_MASK)
        {
            addressHighRead = *(unsigned int*)(currentEntry+4);
/*
			gtOsPrintf("High = 0x%08x, Low = 0x%08x\n", addressHighRead , 
														addressLowRead);
*/
			pAddrEntry->macAddr[5] = (addressHighRead>>(43-32))&0xff;
			pAddrEntry->macAddr[4] = (addressHighRead>>(35-32))&0xff;
			pAddrEntry->macAddr[3] = ((addressLowRead>>27)&0x1f) |
											 ((addressHighRead&0x7)<<5);
			pAddrEntry->macAddr[2] = (addressLowRead>>19)&0xff;
			pAddrEntry->macAddr[1] = (addressLowRead>>11)&0xff;
			pAddrEntry->macAddr[0] = (addressLowRead>>3)&0xff;

			pAddrEntry->action = (addressLowRead & RD_MASK) ? MV_ADDR_ENTRY_RECEIVE :
															  MV_ADDR_ENTRY_DISCARD;

			pAddrEntry->isSkip = (addressLowRead & SKIP_MASK) ? MV_TRUE : MV_FALSE;

			pAddrEntry->isValid = MV_TRUE;
			
			pAddrEntry->prio = (addressHighRead & PRIO_MASK) >> PRIO_OFFSET;

            firstEntry++;
            *startEntry = firstEntry;
            return MV_OK;
        }
        currentEntry += MAC_ENTRY_SIZE;
        firstEntry++;

    }

    return MV_ERROR;
}
