/******************************************************************
 * Copyright 2005 Mentor Graphics Corporation
 * Copyright (C) 2005-2006 by Texas Instruments
 *
 * This file is part of the Inventra Controller Driver for Linux.
 *
 * The Inventra Controller Driver for Linux is free software; you
 * can redistribute it and/or modify it under the terms of the GNU
 * General Public License version 2 as published by the Free Software
 * Foundation.
 *
 * The Inventra Controller Driver for Linux is distributed in
 * the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with The Inventra Controller Driver for Linux ; if not,
 * write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA  02111-1307  USA
 *
 * ANY DOWNLOAD, USE, REPRODUCTION, MODIFICATION OR DISTRIBUTION
 * OF THIS DRIVER INDICATES YOUR COMPLETE AND UNCONDITIONAL ACCEPTANCE
 * OF THOSE TERMS.THIS DRIVER IS PROVIDED "AS IS" AND MENTOR GRAPHICS
 * MAKES NO WARRANTIES, EXPRESS OR IMPLIED, RELATED TO THIS DRIVER.
 * MENTOR GRAPHICS SPECIFICALLY DISCLAIMS ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY; FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT.  MENTOR GRAPHICS DOES NOT PROVIDE SUPPORT
 * SERVICES OR UPDATES FOR THIS DRIVER, EVEN IF YOU ARE A MENTOR
 * GRAPHICS SUPPORT CUSTOMER.
 ******************************************************************/

#ifndef __MUSB_MUSBDEFS_H__
#define __MUSB_MUSBDEFS_H__

#include <linux/slab.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/smp_lock.h>
#include <linux/errno.h>
#include <linux/device.h>
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#include <linux/usb.h>
#include <linux/usb_otg.h>
#include <linux/usb/musb.h>

struct musb;
struct musb_hw_ep;
struct musb_ep;


#include "debug.h"
#include "dma.h"

#ifdef CONFIG_USB_MUSB_SOC
/*
 * Get core configuration from a header converted (by cfg_conv)
 * from the Verilog config file generated by the core config utility
 *
 * For now we assume that header is provided along with other
 * arch-specific files.  Discrete chips will need a build tweak.
 * So will using AHB IDs from silicon that provides them.
 */
#include <asm/arch/hdrc_cnf.h>
#endif

#include "plat_arc.h"
#include "musbhdrc.h"

#include "musb_gadget.h"
#include "../core/hcd.h"
#include "musb_host.h"
#include "otg.h"


/* REVISIT tune this */
#define	MIN_DMA_REQUEST		1	/* use PIO below this xfer size */


#ifdef CONFIG_USB_MUSB_OTG

#define	is_peripheral_enabled(musb)	((musb)->board_mode != MUSB_HOST)
#define	is_host_enabled(musb)		((musb)->board_mode != MUSB_PERIPHERAL)
#define	is_otg_enabled(musb)		((musb)->board_mode == MUSB_OTG)

/* NOTE:  otg and peripheral-only state machines start at B_IDLE.
 * OTG or host-only go to A_IDLE when ID is sensed.
 */
#define is_peripheral_active(m)		(!(m)->bIsHost)
#define is_host_active(m)		((m)->bIsHost)

#else
#define	is_peripheral_enabled(musb)	is_peripheral_capable()
#define	is_host_enabled(musb)		is_host_capable()
#define	is_otg_enabled(musb)		0

#define	is_peripheral_active(musb)	is_peripheral_capable()
#define	is_host_active(musb)		is_host_capable()
#endif

#if defined(CONFIG_USB_MUSB_OTG) || defined(CONFIG_USB_MUSB_PERIPHERAL)
/* for some reason, the "select USB_GADGET_MUSB_HDRC" doesn't always
 * override that choice selection (often USB_GADGET_DUMMY_HCD).
 */
#ifndef CONFIG_USB_GADGET_MUSB_HDRC
#error bogus Kconfig output ... select CONFIG_USB_GADGET_MUSB_HDRC
#endif
#endif	/* need MUSB gadget selection */


#ifdef CONFIG_PROC_FS
#include <linux/fs.h>
#define MUSB_CONFIG_PROC_FS
#endif

/****************************** PERIPHERAL ROLE *****************************/

#ifdef CONFIG_USB_GADGET_MUSB_HDRC

#define	is_peripheral_capable()	(1)

extern irqreturn_t musb_g_ep0_irq(struct musb *);
extern void musb_g_tx(struct musb *, u8);
extern void musb_g_rx(struct musb *, u8);
extern void musb_g_reset(struct musb *);
extern void musb_g_suspend(struct musb *);
extern void musb_g_resume(struct musb *);
extern void musb_g_disconnect(struct musb *);

#else

#define	is_peripheral_capable()	(0)

static inline irqreturn_t musb_g_ep0_irq(struct musb *m) { return IRQ_NONE; }
static inline void musb_g_reset(struct musb *m) {}
static inline void musb_g_suspend(struct musb *m) {}
static inline void musb_g_resume(struct musb *m) {}
static inline void musb_g_disconnect(struct musb *m) {}

#endif

/****************************** HOST ROLE ***********************************/

#ifdef CONFIG_USB_MUSB_HDRC_HCD

#define	is_host_capable()	(1)

extern irqreturn_t musb_h_ep0_irq(struct musb *);
extern void musb_host_tx(struct musb *, u8);
extern void musb_host_rx(struct musb *, u8);

#else

#define	is_host_capable()	(0)

static inline irqreturn_t musb_h_ep0_irq(struct musb *m) { return IRQ_NONE; }
static inline void musb_host_tx(struct musb *m, u8 e) {}
static inline void musb_host_rx(struct musb *m, u8 e) {}

#endif


/****************************** CONSTANTS ********************************/

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#ifndef MUSB_C_NUM_EPS
#define MUSB_C_NUM_EPS ((u8)16)
#endif

#ifndef MUSB_MAX_END0_PACKET
#define MUSB_MAX_END0_PACKET ((u16)MGC_END0_FIFOSIZE)
#endif

/* host side ep0 states */
#define MGC_END0_START  0x0
#define MGC_END0_OUT    0x2
#define MGC_END0_IN     0x4
#define MGC_END0_STATUS 0x8

/* peripheral side ep0 states */
enum musb_g_ep0_state {
	MGC_END0_STAGE_SETUP,		/* idle, waiting for setup */
	MGC_END0_STAGE_TX,		/* IN data */
	MGC_END0_STAGE_RX,		/* OUT data */
	MGC_END0_STAGE_STATUSIN,	/* (after OUT data) */
	MGC_END0_STAGE_STATUSOUT,	/* (after IN data) */
	MGC_END0_STAGE_ACKWAIT,		/* after zlp, before statusin */
} __attribute__ ((packed));

/* OTG protocol constants */
#define OTG_TIME_A_WAIT_VRISE	100		/* msec (max) */
#define OTG_TIME_A_WAIT_BCON	0		/* 0=infinite; min 1000 msec */
#define OTG_TIME_A_IDLE_BDIS	200		/* msec (min) */

/*************************** REGISTER ACCESS ********************************/

/* Endpoint registers (other than dynfifo setup) can be accessed either
 * directly with the "flat" model, or after setting up an index register.
 */

#if defined(CONFIG_ARCH_DAVINCI) || defined(CONFIG_ARCH_OMAP243X)
/* REVISIT "flat" takes about 1% more object code space and can't be very
 * noticeable for speed differences.  But for now indexed access seems to
 * misbehave (on DaVinci) for at least peripheral IN ...
 */
#define	MUSB_FLAT_REG
#endif

/* TUSB mapping: "flat" plus ep0 special cases */
#if	defined(CONFIG_USB_TUSB6010)
#define MGC_SelectEnd(_pBase, _bEnd) \
	musb_writeb((_pBase), MGC_O_HDRC_INDEX, (_bEnd))
#define	MGC_END_OFFSET			MGC_TUSB_OFFSET

/* "flat" mapping: each endpoint has its own i/o address */
#elif	defined(MUSB_FLAT_REG)
#define MGC_SelectEnd(_pBase, _bEnd)	(((void)(_pBase)),((void)(_bEnd)))
#define	MGC_END_OFFSET			MGC_FLAT_OFFSET

/* "indexed" mapping: INDEX register controls register bank select */
#else
#define MGC_SelectEnd(_pBase, _bEnd) \
	musb_writeb((_pBase), MGC_O_HDRC_INDEX, (_bEnd))
#define	MGC_END_OFFSET			MGC_INDEXED_OFFSET
#endif

/* FIXME: replace with musb_readcsr(hw_ep *, REGNAME), etc
 * using hw_ep->regs, for all access except writing INDEX
 */
#ifdef	MUSB_FLAT_REG
#define MGC_ReadCsr8(_pBase, _bOffset, _bEnd) \
	musb_readb((_pBase), MGC_END_OFFSET((_bEnd), (_bOffset)))
#define MGC_ReadCsr16(_pBase, _bOffset, _bEnd) \
	musb_readw((_pBase), MGC_END_OFFSET((_bEnd), (_bOffset)))
#define MGC_WriteCsr8(_pBase, _bOffset, _bEnd, _bData) \
	musb_writeb((_pBase), MGC_END_OFFSET((_bEnd), (_bOffset)), (_bData))
#define MGC_WriteCsr16(_pBase, _bOffset, _bEnd, _bData) \
	musb_writew((_pBase), MGC_END_OFFSET((_bEnd), (_bOffset)), (_bData))
#else
#define MGC_ReadCsr8(_pBase, _bOffset, _bEnd) \
	    musb_readb(_pBase, (_bOffset + 0x10))
#define MGC_ReadCsr16(_pBase, _bOffset, _bEnd) \
	musb_readw(_pBase, (_bOffset + 0x10))
#define MGC_WriteCsr8(_pBase, _bOffset, _bEnd, _bData) \
	musb_writeb(_pBase, (_bOffset + 0x10), _bData)
#define MGC_WriteCsr16(_pBase, _bOffset, _bEnd, _bData) \
	musb_writew(_pBase, (_bOffset + 0x10), _bData)
#endif

/****************************** FUNCTIONS ********************************/

#define MUSB_HST_MODE(_pthis)\
	{ (_pthis)->bIsHost=TRUE; (_pthis)->bIsDevice=FALSE; }
#define MUSB_DEV_MODE(_pthis) \
	{ (_pthis)->bIsHost=FALSE; (_pthis)->bIsDevice=TRUE; }
#define MUSB_OTG_MODE(_pthis) \
	{ (_pthis)->bIsHost=FALSE; (_pthis)->bIsDevice=FALSE; }

#define MUSB_IS_HST(_x) ((_x)->bIsHost && !(_x)->bIsDevice)
#define MUSB_IS_DEV(_x) (!(_x)->bIsHost && (_x)->bIsDevice)
#define MUSB_IS_OTG(_x) (!(_x)->bIsHost && !(_x)->bIsDevice)

#define test_devctl_hst_mode(_x) \
	(musb_readb((_x)->pRegs, MGC_O_HDRC_DEVCTL)&MGC_M_DEVCTL_HM)

/* REVISIT OTG isn't a third non-error mode... */
#define MUSB_MODE(_x) ( MUSB_IS_HST(_x)?"HOST" \
		:(MUSB_IS_DEV(_x)?"PERIPHERAL" \
		:(MUSB_IS_OTG(_x)?"UNCONNECTED" \
		:"ERROR")) )

/************************** Ep Configuration ********************************/

/** The End point descriptor */
struct MUSB_EpFifoDescriptor {
	u8 bType;		/* 0 for autoconfig, CNTR, ISOC, BULK, INTR */
	u8 bDir;		/* 0 for autoconfig, INOUT, IN, OUT */
	int wSize;		/* 0 for autoconfig, or the size */
};

#define MUSB_EPD_AUTOCONFIG	0

#define MUSB_EPD_T_CNTRL	1
#define MUSB_EPD_T_ISOC		2
#define MUSB_EPD_T_BULK		3
#define MUSB_EPD_T_INTR		4

#define MUSB_EPD_D_INOUT	0
#define MUSB_EPD_D_TX		1
#define MUSB_EPD_D_RX		2

/******************************** TYPES *************************************/

/*
 * struct musb_hw_ep - endpoint hardware (bidirectional)
 *
 * Ordered slightly for better cacheline locality.
 */
struct musb_hw_ep {
	struct musb		*musb;
	void __iomem		*fifo;
	void __iomem		*regs;

#ifdef CONFIG_USB_TUSB6010
	void __iomem		*conf;
#endif

	/* index in musb->aLocalEnd[]  */
	u8			bLocalEnd;

	/* hardware configuration, possibly dynamic */
	u8			bIsSharedFifo;
	u8			tx_double_buffered;
	u8			rx_double_buffered;
	u16			wMaxPacketSizeTx;
	u16			wMaxPacketSizeRx;

	struct dma_channel	*tx_channel;
	struct dma_channel	*rx_channel;

#ifdef CONFIG_USB_TUSB6010
	/* TUSB has "asynchronous" and "synchronous" dma modes */
	dma_addr_t		fifo_async;
	dma_addr_t		fifo_sync;
#endif

#ifdef CONFIG_USB_MUSB_HDRC_HCD
	void __iomem		*target_regs;

	/* currently scheduled peripheral endpoint */
	struct musb_qh		*in_qh;
	struct musb_qh		*out_qh;

	u8			rx_reinit;
	u8			tx_reinit;
#endif

#ifdef CONFIG_USB_GADGET_MUSB_HDRC
	/* peripheral side */
	struct musb_ep		ep_in;			/* TX */
	struct musb_ep		ep_out;			/* RX */
#endif
};

static inline struct usb_request *next_in_request(struct musb_hw_ep *hw_ep)
{
#ifdef CONFIG_USB_GADGET_MUSB_HDRC
	return next_request(&hw_ep->ep_in);
#else
	return NULL;
#endif
}

static inline struct usb_request *next_out_request(struct musb_hw_ep *hw_ep)
{
#ifdef CONFIG_USB_GADGET_MUSB_HDRC
	return next_request(&hw_ep->ep_out);
#else
	return NULL;
#endif
}

/*
 * struct musb - Driver instance data.
 */
struct musb {
	spinlock_t		Lock;
	struct clk		*clock;
	irqreturn_t		(*isr)(int, void *, struct pt_regs *);
	struct work_struct	irq_work;

#ifdef CONFIG_USB_MUSB_HDRC_HCD

	u32			port1_status;
	unsigned long		rh_timer;

	u8 bEnd0Stage;		/* end0 stage while in host */

	/* bulk traffic normally dedicates endpoint hardware, and each
	 * direction has its own ring of host side endpoints.
	 * we try to progress the transfer at the head of each endpoint's
	 * queue until it completes or NAKs too much; then we try the next
	 * endpoint.
	 */
	struct musb_hw_ep	*bulk_ep;

	struct list_head	control;	/* of musb_qh */
	struct list_head	in_bulk;	/* of musb_qh */
	struct list_head	out_bulk;	/* of musb_qh */
	struct musb_qh		*periodic[32];	/* tree of interrupt+iso */
#endif

	/* called with IRQs blocked; ON/nonzero implies starting a session,
	 * and waiting at least a_wait_vrise_tmout.
	 */
	void			(*board_set_vbus)(struct musb *, int is_on);

	struct dma_controller	*pDmaController;

	struct device		*controller;
	void __iomem		*ctrl_base;
	void __iomem		*pRegs;

#ifdef CONFIG_USB_TUSB6010
	dma_addr_t		async;
	dma_addr_t		sync;
#endif

	/* passed down from chip/board specific irq handlers */
	u8			int_usb;
	u16			int_rx;
	u16			int_tx;
	struct pt_regs		*int_regs;

	struct otg_transceiver	xceiv;

	int nIrq;

	struct musb_hw_ep	 aLocalEnd[MUSB_C_NUM_EPS];
#define control_ep		aLocalEnd

	u16			vbuserr_retry;
	u16 wEndMask;
	u8 bEndCount;

	u8 board_mode;		/* enum musb_mode */
	int			(*board_set_power)(int state);

	u8			min_power;	/* vbus for periph, in mA/2 */

	/* active means connected and not suspended */
	unsigned is_active:1;

	unsigned bIsMultipoint:1;
	unsigned bIsDevice:1;
	unsigned bIsHost:1;
	unsigned bIgnoreDisconnect:1;	/* during bus resets */

#ifdef C_MP_TX
	unsigned bBulkSplit:1;
#define	can_bulk_split(musb,type) \
		(((type) == USB_ENDPOINT_XFER_BULK) && (musb)->bBulkSplit)
#else
#define	can_bulk_split(musb,type)	0
#endif

#ifdef C_MP_RX
	unsigned bBulkCombine:1;
	/* REVISIT allegedly doesn't work reliably */
#if 0
#define	can_bulk_combine(musb,type) \
		(((type) == USB_ENDPOINT_XFER_BULK) && (musb)->bBulkCombine)
#else
#define	can_bulk_combine(musb,type)	0
#endif
#else
#define	can_bulk_combine(musb,type)	0
#endif

#ifdef CONFIG_USB_GADGET_MUSB_HDRC
	unsigned bIsSelfPowered:1;
	unsigned bMayWakeup:1;
	unsigned bSetAddress:1;
	unsigned bTestMode:1;
	unsigned softconnect:1;

	enum musb_g_ep0_state	ep0_state;
	u8			bAddress;
	u8			bTestModeValue;
	u16			ackpend;		/* ep0 */
	struct usb_gadget	g;			/* the gadget */
	struct usb_gadget_driver *pGadgetDriver;	/* its driver */
#endif

#ifdef CONFIG_USB_MUSB_OTG
	struct otg_machine	OtgMachine;
	u8 bDelayPortPowerOff;
#endif

#ifdef MUSB_CONFIG_PROC_FS
	struct proc_dir_entry *pProcEntry;
#endif
};

static inline void musb_set_vbus(struct musb *musb, int is_on)
{
	musb->board_set_vbus(musb, is_on);
}

#ifdef CONFIG_USB_GADGET_MUSB_HDRC
static inline struct musb *gadget_to_musb(struct usb_gadget *g)
{
	return container_of(g, struct musb, g);
}
#endif


/***************************** Glue it together *****************************/

extern const char musb_driver_name[];

extern void musb_start(struct musb *pThis);
extern void musb_stop(struct musb *pThis);

extern void musb_write_fifo(struct musb_hw_ep *ep,
			     u16 wCount, const u8 * pSource);
extern void musb_read_fifo(struct musb_hw_ep *ep,
			       u16 wCount, u8 * pDest);

extern void musb_load_testpacket(struct musb *);

extern irqreturn_t musb_interrupt(struct musb *);

extern void musb_platform_enable(struct musb *musb);
extern void musb_platform_disable(struct musb *musb);

#ifdef CONFIG_USB_TUSB6010
extern void musb_platform_try_idle(struct musb *musb);
extern int musb_platform_get_vbus_status(struct musb *musb);
#else
#define musb_platform_try_idle(x)		do {} while (0)
#define musb_platform_get_vbus_status(x)	0
#endif

extern int __devinit musb_platform_init(struct musb *musb);
extern int musb_platform_exit(struct musb *musb);

/*-------------------------- ProcFS definitions ---------------------*/

struct proc_dir_entry;

#if (MUSB_DEBUG > 0) && defined(MUSB_CONFIG_PROC_FS)
extern struct proc_dir_entry *musb_debug_create(char *name,
						    struct musb *data);
extern void musb_debug_delete(char *name, struct musb *data);

#else
static inline struct proc_dir_entry *musb_debug_create(char *name,
							   struct musb *data)
{
	return NULL;
}
static inline void musb_debug_delete(char *name, struct musb *data)
{
}
#endif

#endif	/* __MUSB_MUSBDEFS_H__ */
