
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/string.h>
#include <linux/rtc.h>
#include <linux/bcd.h>



static unsigned short normal_i2c[] =  { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 
                                        0x36, 0x37, I2C_CLIENT_END };

I2C_CLIENT_INSMOD;

#define S35390A_CMD_STATUS1	0
#define S35390A_CMD_STATUS2	1
#define S35390A_CMD_TIME1	2

#define S35390A_BYTE_YEAR	0
#define S35390A_BYTE_MONTH	1
#define S35390A_BYTE_DAY	2
#define S35390A_BYTE_WDAY	3
#define S35390A_BYTE_HOURS	4
#define S35390A_BYTE_MINS	5
#define S35390A_BYTE_SECS	6

#define S35390A_FLAG_POC	0x01
#define S35390A_FLAG_BLD	0x02
#define S35390A_FLAG_24H	0x40
#define S35390A_FLAG_RESET	0x80
#define S35390A_FLAG_TEST	0x01

struct s35390a {
	struct i2c_client *client[8];
	struct rtc_device *rtc;
	int twentyfourhour;
};


struct s35390a {
	u8			reg_addr;
	u8			regs[8];
	struct i2c_msg		msg[2];
	struct i2c_client	client;
	struct rtc_device	*rtc;
};


static int s35390a_get_time(struct device *dev, struct rtc_time *t)
{
	struct s35390a	*s35390a = dev_get_drvdata(dev);
	int		tmp;

	/* read the RTC registers all at once */
	s35390a->msg[1].flags = I2C_M_RD;
	s35390a->msg[1].len = 7;

	tmp = i2c_transfer(s35390a->client.adapter, s35390a->msg, 2);
	if (tmp != 2) {
		dev_err(dev, "%s error %d\n", "read", tmp);
		return -EIO;
	}

	dev_dbg(dev, "%s: %02x %02x %02x %02x %02x %02x %02x\n",
			"read",
			s35390a->regs[0], s35390a->regs[1],
			s35390a->regs[2], s35390a->regs[3],
			s35390a->regs[4], s35390a->regs[5],
			s35390a->regs[6]);

	t->tm_sec = BCD2BIN(s35390a->regs[S35390A_REG_SECS] & 0x7f);
	t->tm_min = BCD2BIN(s35390a->regs[S35390A_REG_MIN] & 0x7f);
	tmp = s35390a->regs[S35390A_REG_HOUR] & 0x3f;
	t->tm_hour = BCD2BIN(tmp);
	t->tm_wday = BCD2BIN(s35390a->regs[S35390A_REG_WDAY] & 0x07) - 1;
	t->tm_mday = BCD2BIN(s35390a->regs[S35390A_REG_MDAY] & 0x3f);
	tmp = s35390a->regs[S35390A_REG_MONTH] & 0x1f;
	t->tm_mon = BCD2BIN(tmp) - 1;

	/* assume 20YY not 19YY, and ignore DS1337_BIT_CENTURY */
	t->tm_year = BCD2BIN(s35390a->regs[S35390A_REG_YEAR]) + 100;

	dev_dbg(dev, "%s secs=%d, mins=%d, "
		"hours=%d, mday=%d, mon=%d, year=%d, wday=%d\n",
		"read", t->tm_sec, t->tm_min,
		t->tm_hour, t->tm_mday,
		t->tm_mon, t->tm_year, t->tm_wday);

	return 0;
}

static int s35390a_set_time(struct device *dev, struct rtc_time *t)
{
	struct s35390a	*s35390a = dev_get_drvdata(dev);
	int		result;
	int		tmp;
	u8		*buf = s35390a->regs;

	dev_dbg(dev, "%s secs=%d, mins=%d, "
		"hours=%d, mday=%d, mon=%d, year=%d, wday=%d\n",
		"write", dt->tm_sec, dt->tm_min,
		dt->tm_hour, dt->tm_mday,
		dt->tm_mon, dt->tm_year, dt->tm_wday);

	*buf++ = 0;		/* first register addr */
	buf[S35390A_REG_SECS] = BIN2BCD(t->tm_sec);
	buf[S35390A_REG_MIN] = BIN2BCD(t->tm_min);
	buf[S35390A_REG_HOUR] = BIN2BCD(t->tm_hour);
	buf[S35390A_REG_WDAY] = BIN2BCD(t->tm_wday + 1);
	buf[S35390A_REG_MDAY] = BIN2BCD(t->tm_mday);
	buf[S35390A_REG_MONTH] = BIN2BCD(t->tm_mon + 1);

	/* assume 20YY not 19YY */
	tmp = t->tm_year - 100;
	buf[S35390A_REG_YEAR] = BIN2BCD(tmp);

	if (s35390a->type == ds_1337)
		buf[S35390A_REG_MONTH] |= DS1337_BIT_CENTURY;
	else if (s35390a->type == ds_1340)
		buf[S35390A_REG_HOUR] |= DS1340_BIT_CENTURY_EN
				| DS1340_BIT_CENTURY;

	s35390a->msg[1].flags = 0;
	s35390a->msg[1].len = 8;

	dev_dbg(dev, "%s: %02x %02x %02x %02x %02x %02x %02x\n",
		"write", buf[0], buf[1], buf[2], buf[3],
		buf[4], buf[5], buf[6]);

	result = i2c_transfer(s35390a->client.adapter, &s35390a->msg[1], 1);
	if (result != 1) {
		dev_err(dev, "%s error %d\n", "write", tmp);
		return -EIO;
	}
	return 0;
}

static struct rtc_class_ops ds13xx_rtc_ops = {
	.read_time	= s35390a_get_time,
	.set_time	= s35390a_set_time,
};

static struct i2c_driver s35390a_driver;

static int __devinit
s35390a_detect(struct i2c_adapter *adapter, int address, int kind)
{
	struct s35390a		*s35390a;
	int			err = -ENODEV;
	struct i2c_client	*client;
	int			tmp;

	if (!(s35390a = kzalloc(sizeof(struct s35390a), GFP_KERNEL))) {
		err = -ENOMEM;
		goto exit;
	}

	client = &s35390a->client;
	client->addr = address;
	client->adapter = adapter;
	client->driver = &s35390a_driver;
	client->flags = 0;

	i2c_set_clientdata(client, s35390a);

	s35390a->msg[0].addr = client->addr;
	s35390a->msg[0].flags = 0;
	s35390a->msg[0].len = 1;
	s35390a->msg[0].buf = &s35390a->reg_addr;

	s35390a->msg[1].addr = client->addr;
	s35390a->msg[1].flags = I2C_M_RD;
	s35390a->msg[1].len = sizeof(s35390a->regs);
	s35390a->msg[1].buf = s35390a->regs;

	/* HACK: "force" implies "needs ds1337-style-oscillator setup" */
	if (kind >= 0) {
		s35390a->type = ds_1337;

		s35390a->reg_addr = DS1337_REG_CONTROL;
		s35390a->msg[1].len = 2;

		tmp = i2c_transfer(client->adapter, s35390a->msg, 2);
		if (tmp != 2) {
			pr_debug("read error %d\n", tmp);
			err = -EIO;
			goto exit_free;
		}

		s35390a->reg_addr = 0;
		s35390a->msg[1].len = sizeof(s35390a->regs);

		/* oscillator is off; need to turn it on */
		if ((s35390a->regs[0] & DS1337_BIT_nEOSC)
				|| (s35390a->regs[1] & DS1337_BIT_OSF)) {
			printk(KERN_ERR "no ds1337 oscillator code\n");
			goto exit_free;
		}
	} else
		s35390a->type = ds_1307;

read_rtc:
	/* read RTC registers */

	tmp = i2c_transfer(client->adapter, s35390a->msg, 2);
	if (tmp != 2) {
		pr_debug("read error %d\n", tmp);
		err = -EIO;
		goto exit_free;
	}

	/* minimal sanity checking; some chips (like DS1340) don't
	 * specify the extra bits as must-be-zero, but there are
	 * still a few values that are clearly out-of-range.
	 */
	tmp = s35390a->regs[S35390A_REG_SECS];
	if (tmp & S35390A_BIT_CH) {
		if (s35390a->type && s35390a->type != ds_1307) {
			pr_debug("not a s35390a?\n");
			goto exit_free;
		}
		s35390a->type = ds_1307;

		/* this partial initialization should work for s35390a,
		 * ds1338, ds1340, st m41t00, and more.
		 */
		dev_warn(&client->dev, "oscillator started; SET TIME!\n");
		i2c_smbus_write_byte_data(client, 0, 0);
		goto read_rtc;
	}
	tmp = BCD2BIN(tmp & 0x7f);
	if (tmp > 60)
		goto exit_free;
	tmp = BCD2BIN(s35390a->regs[S35390A_REG_MIN] & 0x7f);
	if (tmp > 60)
		goto exit_free;

	tmp = BCD2BIN(s35390a->regs[S35390A_REG_MDAY] & 0x3f);
	if (tmp == 0 || tmp > 31)
		goto exit_free;

	tmp = BCD2BIN(s35390a->regs[S35390A_REG_MONTH] & 0x1f);
	if (tmp == 0 || tmp > 12)
		goto exit_free;

	/* force into in 24 hour mode (most chips) or
	 * disable century bit (ds1340)
	 */
	tmp = s35390a->regs[S35390A_REG_HOUR];
	if (tmp & (1 << 6)) {
		if (tmp & (1 << 5))
			tmp = BCD2BIN(tmp & 0x1f) + 12;
		else
			tmp = BCD2BIN(tmp);
		i2c_smbus_write_byte_data(client,
				S35390A_REG_HOUR,
				BIN2BCD(tmp));
	}

	/* FIXME chips like 1337 can generate alarm irqs too; those are
	 * worth exposing through the API (especially when the irq is
	 * wakeup-capable).
	 */

	switch (s35390a->type) {
	case unknown:
		strlcpy(client->name, "unknown", I2C_NAME_SIZE);
		break;
	case ds_1307:
		strlcpy(client->name, "s35390a", I2C_NAME_SIZE);
		break;
	case ds_1337:
		strlcpy(client->name, "ds1337", I2C_NAME_SIZE);
		break;
	case ds_1340:
		strlcpy(client->name, "ds1340", I2C_NAME_SIZE);
		break;
	}

	/* Tell the I2C layer a new client has arrived */
	if ((err = i2c_attach_client(client)))
		goto exit_free;

	s35390a->rtc = rtc_device_register(client->name, &client->dev,
				&ds13xx_rtc_ops, THIS_MODULE);
	if (IS_ERR(s35390a->rtc)) {
		err = PTR_ERR(s35390a->rtc);
		dev_err(&client->dev,
			"unable to register the class device\n");
		goto exit_detach;
	}

	return 0;

exit_detach:
	i2c_detach_client(client);
exit_free:
	kfree(s35390a);
exit:
	return err;
}

static int __devinit
s35390a_attach_adapter(struct i2c_adapter *adapter)
{
	if (!i2c_check_functionality(adapter, I2C_FUNC_I2C))
		return 0;
	return i2c_probe(adapter, &addr_data, s35390a_detect);
}

static int __devexit s35390a_detach_client(struct i2c_client *client)
{
	int		err;
	struct s35390a	*s35390a = i2c_get_clientdata(client);

	rtc_device_unregister(s35390a->rtc);
	if ((err = i2c_detach_client(client)))
		return err;
	kfree(s35390a);
	return 0;
}

static struct i2c_driver s35390a_driver = {
	.driver = {
		.name	= "s35390a",
		.owner	= THIS_MODULE,
	},
	.attach_adapter	= s35390a_attach_adapter,
	.detach_client	= __devexit_p(s35390a_detach_client),
};

static int __init s35390a_init(void)
{
	return i2c_add_driver(&s35390a_driver);
}
module_init(s35390a_init);

static void __exit s35390a_exit(void)
{
	i2c_del_driver(&s35390a_driver);
}
module_exit(s35390a_exit);

MODULE_DESCRIPTION("RTC driver for S35390A and similar chips");
MODULE_LICENSE("GPL");
