/*
 * Sandpoint specific fixups.
 *
 * Author: Mark A. Greer <mgreer@mvista.com>
 *
 * 2006 (c) MontaVista, Software, Inc.  This file is licensed under
 * the terms of the GNU General Public License version 2.  This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <stdarg.h>
#include <stddef.h>
#include "types.h"
#include "elf.h"
#include "page.h"
#include "string.h"
#include "stdio.h"
#include "io.h"
#include "ops.h"

extern char _end[];

#define	KB	1024
#define	MB	(1024*1024)

#define	CPU_824X	0
#define	CPU_7XX		1
#define	CPU_7457	2
#define	CPU_NUM		3

static u32 cpu_pll[CPU_NUM][32] = {
	[CPU_824X] = { /* 824x */
		5, 6, 9, 4, 4, 5, 2, 6, 6, 4, 9, 6, 5, 7, 6, 7,
		4, 5, 4, 6, 7, 8, 8, 4, 6, 5, 8, 6, 6, 5, 7, 0
	},
	[CPU_7XX] = { /* 750/755 */
		0, 15, 14, 2, 4, 13, 20, 9, 6, 11, 8, 10, 16, 12, 7, 0,
		0,  0,  0, 0, 0,  0,  0, 0, 0,  0, 0,  0,  0,  0, 0, 0

	},
	[CPU_7457] = { /* 7457 */
		23, 34, 15, 30, 14, 36,  2, 40,  4, 42, 13, 26, 17, 48, 19, 18,
		 6, 21, 11, 22,  8, 20, 10, 24, 16, 28, 12, 32, 27, 56,  0, 25
	}
};

static struct processor_info {
	u32	pvr;
	u32	mask;
	u32	bus_freq;
	u32	hid1_shift;
	u32	hid1_mask;
	u32	pll_tbl_idx;
	u32	max_mem;	/* DINK still sets up mem ctlr wrong */
} processor_info_tbl[] = { /* From cputable -- MHz are only guesses */
	/* 824x */
	{ 0x00810000, 0x7fff0000, 100000000, 27, 0x1f, CPU_824X, 0x08000000 },
	/* 750 */
	{ 0x00084202, 0xffffffff, 100000000, 28, 0xf, CPU_7XX, 0x02000000 },
	/* 745/755 */
	{ 0x00083000, 0xfffff000, 100000000, 28, 0xf, CPU_7XX, 0x02000000 },
	/* 7447/7457 Rev 1.0 */
	{ 0x80020100, 0xffffffff, 100000000, 12, 0x1f, CPU_7457, 0x04000000 },
	/* 7447/7457 Rev 1.1 */
	{ 0x80020101, 0xffffffff, 100000000, 12, 0x1f, CPU_7457, 0x04000000 },
	/* 7447/7457 Rev 1.2 & up*/
	{ 0x80020000, 0xffff0000, 100000000, 12, 0x1f, CPU_7457, 0x04000000 },
	/* 7447A */
	{ 0x80030000, 0xffff0000, 100000000, 12, 0x1f, CPU_7457, 0x80000000 },
};

static struct processor_info *get_processor_info(u32 pvr)
{
	struct processor_info *pit = processor_info_tbl;
	u32 i;

	for (i=0; i<ARRAY_SIZE(processor_info_tbl); i++, pit++)
		if (pit->pvr == (pvr & pit->mask))
			return pit;
	return NULL;
}

#define	__stringify_1(x)	#x
#define	__stringify(x)		__stringify_1(x)

#define SPRN_PVR	0x11F	/* Processor Version Register */
#define SPRN_HID1	0x3F1	/* Hardware Implementation Register 1 */
#define mfspr(rn)	({unsigned long rval; \
			asm volatile("mfspr %0," __stringify(rn) \
				: "=r" (rval)); rval;})

static void sandpoint_fixups(void)
{
	u32 i, v[2], hid1, max_mem = 0xffffffff;
	void *devp;
	struct processor_info *pit;
	extern u32 mpc10x_get_mem_size(void);

	/* Update cpu's clock-frequency & timebase-frequency in fdt */
	if ((pit = get_processor_info(mfspr(SPRN_PVR)))) {
		if ((devp = finddevice("/cpus/PowerPC,603e"))) {
			max_mem = pit->max_mem;

			hid1 = (mfspr(SPRN_HID1) >> pit->hid1_shift)
				& pit->hid1_mask;
			v[0] = pit->bus_freq
				* cpu_pll[pit->pll_tbl_idx][hid1]/2;
			setprop(devp, "clock-frequency", v, sizeof(v[0]));

			v[0] = pit->bus_freq / 4;
			setprop(devp, "timebase-frequency", v, sizeof(v[0]));
		}
		if ((devp = finddevice("/soc10x@fc000000")))
			setprop(devp, "clock-frequency", &pit->bus_freq,
					sizeof(u32));
	}

	/* Get the RAM size from the memory controller & update fdt */
	if ((devp = finddevice("/memory"))) {
		i = mpc10x_get_mem_size();
		v[0] = 0;
		v[1] = min(i, max_mem);
		setprop(devp, "reg", v, sizeof(v));
	}

	/* XXXX stuff from platforms/.../sandpoint.c should be here */
}

static void sandpoint_reset(void)
{
	void _nmask_and_or_msr(unsigned long nmask, unsigned long or_val);
	_nmask_and_or_msr(0, (1<<6)); /* Set exception prefix high - firmware */

	/* Reset system via Port 92 */
	out_8((volatile unsigned char *)0xfe000092, 0x00);
	out_8((volatile unsigned char *)0xfe000092, 0x01);

	for(;;);	/* Spin until reset happens */
}

#define	HEAP_SIZE	(16*MB)

int platform_init(void *promptr, const void *elfhdr, char *dt_blob_start,
		char *dt_blob_end)
{
	const Elf32_Ehdr *elf32 = elfhdr;
	Elf32_Phdr *elf32ph = (Elf32_Phdr *)((unsigned long)elf32
		+ elf32->e_phoff);
	char *heap_start, *dtb;
	int dt_size = dt_blob_end - dt_blob_start;

	if (dt_size <= 0) /* No fdt */
		goto err_out;

	if (elf32->e_ident[EI_CLASS] != ELFCLASS32)
		goto err_out;

	/* Start heap after end of the zImage or end of the kernel,
	 * whichever is higher.  That's so things allocated by
	 * simple_alloc won't overwrite any part of the zImage or
	 * get overwritten when the early kernel code relocates
	 * the kernel to 0.
	 */
	heap_start = (char *)_ALIGN(elf32ph->p_memsz + elf32ph->p_offset, 4096);
	heap_start = max(heap_start, (char *)_end);

	if ((unsigned)simple_alloc_init(heap_start, HEAP_SIZE, 2*KB, 16)
			> (32*MB))
		goto err_out;

	/* Relocate dtb to safe area past end of zImage & kernel */
	dtb = malloc(dt_size);
	if (!dtb)
		goto err_out;
	memmove(dtb, dt_blob_start, dt_size);
	if (ft_init(dtb, dt_size, 16))
		goto err_out;

	platform_ops.fixups = sandpoint_fixups;
	platform_ops.exit = sandpoint_reset;

	return serial_console_init();

err_out:
	return -1;
}
