#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/poll.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* file ops */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/seq_file.h>
#include <linux/cdev.h>
#include <linux/miscdevice.h>

#include <asm/system.h>		/* cli(), *_flags */
#include <asm/uaccess.h>	/* copy_*_user */

#include <mvTwsi.h>

#define FRNT_PNL_DEBUG
#include <asm/arch/frontpanel.h>

/*
 * parameters which can be set at load time.
 */
int frnt_pnl_major =   FRNT_PNL_MAJOR;
int frnt_pnl_minor =   FRNT_PNL_MINOR;
int frnt_pnl_nr_devs = FRNT_PNL_NR_DEVS;
int frnt_pnl_slave_addr = FRNT_PNL_SLAVE_ADDR;

module_param(frnt_pnl_major, int, S_IRUGO);
module_param(frnt_pnl_minor, int, S_IRUGO);
module_param(frnt_pnl_nr_devs, int, S_IRUGO);
module_param(frnt_pnl_slave_addr, int, S_IRUGO);

static DECLARE_WAIT_QUEUE_HEAD(frnt_pnl_wait);
static struct fasync_struct *frnt_pnl_async_queue;

/*
 * frnt_pnl_lock protects frnt_pnl_irq_data
 */
static DEFINE_SPINLOCK(frnt_pnl_lock);
static unsigned long frnt_pnl_irq_data;

/*
 * frnt_pnl_mutex protects frnt_pnl_inuse
 */
static DEFINE_MUTEX(frnt_pnl_mutex);
static unsigned long frnt_pnl_inuse;

static unsigned char ExReg[2];

/* actual write to frnt_pnl */
static int frnt_pnl_twsi_write(unsigned offset, unsigned char *buf, int cnt)
{
    MV_TWSI_SLAVE   twsiSlave;

    twsiSlave.slaveAddr.type = ADDR7_BIT;
    twsiSlave.slaveAddr.address = frnt_pnl_slave_addr;
    twsiSlave.validOffset = MV_TRUE;
    twsiSlave.offset = offset;
    twsiSlave.moreThen256 = MV_FALSE;
    return mvTwsiWrite (&twsiSlave, buf, cnt);
}

/* actual read from frnt_pnl */
static int frnt_pnl_twsi_read(unsigned offset, unsigned char *buf, int cnt)
{
    MV_TWSI_SLAVE   twsiSlave;

    twsiSlave.slaveAddr.type = ADDR7_BIT;
    twsiSlave.slaveAddr.address = frnt_pnl_slave_addr;
    twsiSlave.validOffset = MV_TRUE;
    twsiSlave.offset = offset;
    twsiSlave.moreThen256 = MV_FALSE;
    return mvTwsiRead (&twsiSlave, buf, cnt);
}

static int frnt_pnl_clr_reg_bits(int index, int bitmask)
{
    static unsigned char val[2];

    frnt_pnl_twsi_read(index, val, 2);
    val[0] &= (~bitmask >> 8);
    val[1] &= (~bitmask & 0xff);
    return frnt_pnl_twsi_write(index, val, 2);
}

static int frnt_pnl_set_reg_bits(int index, int bitmask)
{
    static unsigned char val[2];

    frnt_pnl_twsi_read(index, val, 2);
    val[0] |= bitmask >> 8;
    val[1] |= bitmask & 0xff;
    return frnt_pnl_twsi_write(index, val, 2);
}

/* IRQ Handlers */
static irqreturn_t frnt_pnl_irq(int irq, void *id, struct pt_regs *r)
{
	spin_lock(&frnt_pnl_lock);
	frnt_pnl_irq_data = frnt_pnl_irq_data + 1;
    /* disable the frontpanel interrupt */
    frnt_pnl_twsi_read(REG_INTR_STATUS, ExReg, 2);
    ExReg[1] &= ~(INTR_ENABLE_FLAG);
    frnt_pnl_twsi_write(REG_INTR_STATUS, ExReg, 2);
	spin_unlock(&frnt_pnl_lock);

	wake_up_interruptible(&frnt_pnl_wait);
	kill_fasync(&frnt_pnl_async_queue, SIGIO, POLL_IN);
	return IRQ_HANDLED;
}

static int frnt_pnl_fasync(int fd, struct file *file, int on)
{
	return fasync_helper(fd, file, on, &frnt_pnl_async_queue);
}

static ssize_t frnt_pnl_read(struct file *file, char __user *buf, size_t count,
        loff_t *ppos)
{
	DECLARE_WAITQUEUE(wait, current);
	unsigned long data;
	ssize_t ret;

	if (count < sizeof(unsigned long))
		return -EINVAL;

	add_wait_queue(&frnt_pnl_wait, &wait);
	do {
		__set_current_state(TASK_INTERRUPTIBLE);

		spin_lock_irq(&frnt_pnl_lock);
		data = frnt_pnl_irq_data;
		frnt_pnl_irq_data = 0;
		spin_unlock_irq(&frnt_pnl_lock);

		if (data != 0) {
			ret = 0;
			break;
		}
		if (file->f_flags & O_NONBLOCK) {
			ret = -EAGAIN;
			break;
		}
		if (signal_pending(current)) {
			ret = -ERESTARTSYS;
			break;
		}
		schedule();
	} while (1);
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&frnt_pnl_wait, &wait);

	if (ret == 0) {
		ret = put_user(data, (unsigned long __user *)buf);
		if (ret == 0)
			ret = sizeof(unsigned long);
	}
	return ret;
}

static unsigned int frnt_pnl_poll(struct file *file, poll_table *wait)
{
	unsigned long data;

	poll_wait(file, &frnt_pnl_wait, wait);

	spin_lock_irq(&frnt_pnl_lock);
	data = frnt_pnl_irq_data;
	spin_unlock_irq(&frnt_pnl_lock);

	return data != 0 ? POLLIN | POLLRDNORM : 0;
}

static int frnt_pnl_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
        unsigned long arg)
{
    int err = 0;
    int ret = 0;
    unsigned char value[2];
    frontPanelReg reg;
	void __user *uarg = (void __user *)arg;
    unsigned char exreg[2];

    if (_IOC_TYPE(cmd) != FRNT_PNL_IOC_MAGIC) return -ENOTTY;
    if (_IOC_NR(cmd) > FRNT_PNL_IOC_MAXNR) return -ENOTTY;

    if (_IOC_DIR(cmd) & _IOC_READ)
        err = !access_ok(VERIFY_WRITE, (void __user *)arg, _IOC_SIZE(cmd));
    else if (_IOC_DIR(cmd) & _IOC_WRITE)
        err =  !access_ok(VERIFY_READ, (void __user *)arg, _IOC_SIZE(cmd));
    if (err) return -EFAULT;

    switch(cmd) {
        case FRNT_PNL_IOCGETREG:
            ret = copy_from_user(&reg, uarg, sizeof(frontPanelReg));
            if (ret) {
                ret = -EFAULT;
                break;
            }
            ret = frnt_pnl_twsi_read(reg.index, value, 2);
            if (ret) {
                mvTwsiReset();
                ret = -EFAULT;
                break;
            }
            reg.value = ((value[0] << 8) | value[1]);
            ret = copy_to_user(uarg, &reg, sizeof(frontPanelReg));
            if (ret)
                ret = -EFAULT;
            break;

        case FRNT_PNL_IOCSETREG:
            ret = copy_from_user(&reg, uarg, sizeof(frontPanelReg));
            if (ret) {
                ret = -EFAULT;
                break;
            }
            value[0] = ((reg.value >> 8) & 0xff);
            value[1] = ((reg.value) & 0xff);
            ret = frnt_pnl_twsi_write(reg.index, value, 2);
            if (ret) {
                mvTwsiReset();
                ret = -EFAULT;
            }
            break;

        case FRNT_PNL_IOCGETEXREG:
            reg.index = REG_INTR_STATUS;
            reg.value = ((ExReg[0] << 8) | ExReg[1]);
            ret = copy_to_user(uarg, &reg, sizeof(frontPanelReg));
            if (ret)
                ret = -EFAULT;
            exreg[0] = ExReg[0];
            exreg[1] = ExReg[1];
            exreg[1] |= INTR_ENABLE_FLAG;
            exreg[1] &= ~(BUTTON_MASK);
            ret = frnt_pnl_twsi_write(reg.index, exreg, 2);
            if (ret) {
                mvTwsiReset();
                ret = -EFAULT;
            }
            break;

        case FRNT_PNL_IOCSETADDR:
            ret = __get_user(frnt_pnl_slave_addr, (int __user *)arg);
            break;
    }
    return ret;
}

static int frnt_pnl_open(struct inode *inode, struct file *file)
{
    int ret;

    mutex_lock(&frnt_pnl_mutex);

    spin_lock_irq(&frnt_pnl_lock);
    frnt_pnl_irq_data = 0;
    spin_unlock_irq(&frnt_pnl_lock);

    /* Enable interrupt */
    ret = frnt_pnl_set_reg_bits(REG_INTR_STATUS, INTR_ENABLE_FLAG);
    if (ret) {
        ret = -ENXIO;
        goto out;
    }

    /* Initialize button trigger value */
    ret = frnt_pnl_set_reg_bits(REG_RSTKEY_TRIGGER, RSTKEY_TRIGGER_VAL);
    if (ret) {
        ret = -ENXIO;
        goto out;
    }
    ret = frnt_pnl_set_reg_bits(REG_PWRKEY_TRIGGER, PWRKEY_TRIGGER_VAL);
    if (ret) {
        ret = -ENXIO;
        goto out;
    }
    ret = frnt_pnl_set_reg_bits(REG_ALRMKEY_TRIGGER, PWRKEY_TRIGGER_VAL);
    if (ret) {
        ret = -ENXIO;
        goto out;
    }

    if (!frnt_pnl_inuse) {
        ret = request_irq(IRQ_GPP_11, frnt_pnl_irq, IRQF_DISABLED, "frontpanel", NULL);
        if (ret)
            printk(KERN_ERR "frontpanel: IRQ%d already in use\n", IRQ_GPP_11);
    }

    frnt_pnl_inuse += 1;
    ret = 0;
out:
    mutex_unlock(&frnt_pnl_mutex);

    return ret;
}

static int frnt_pnl_release(struct inode *inode, struct file *file)
{
	spin_lock_irq(&frnt_pnl_lock);
	frnt_pnl_irq_data = 0;
	spin_unlock_irq(&frnt_pnl_lock);

    mutex_lock(&frnt_pnl_mutex);
    frnt_pnl_inuse -= 1;
    mutex_unlock(&frnt_pnl_mutex);

    if (!frnt_pnl_inuse) {
        free_irq(IRQ_GPP_11, NULL);
    }
    frnt_pnl_fasync(-1, file, 0);
    return 0;
}

static struct file_operations frnt_pnl_fops = {
    .owner		= THIS_MODULE,
    .llseek		= no_llseek,
    .read		= frnt_pnl_read,
	.poll		= frnt_pnl_poll,
    .ioctl		= frnt_pnl_ioctl,
    .open		= frnt_pnl_open,
    .release	= frnt_pnl_release,
	.fasync		= frnt_pnl_fasync,
};

static struct miscdevice frnt_pnl_miscdev = {
    .minor		= MISC_DYNAMIC_MINOR,
    .name		= "frnt_pnl",
    .fops		= &frnt_pnl_fops,
};

#if defined(FRNT_PNL_DEBUG)
static int frnt_pnl_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    char *p = page;

    p += sprintf(p, "frnt_pnl in use = %lu\n",frnt_pnl_inuse);
    p += sprintf(p, "frnt_pnl irq = %lu\n",frnt_pnl_irq_data);
    p += sprintf(p, "\n");
    return p - page;
}
#endif

void frnt_pnl_cleanup_module(void)
{
    mutex_lock(&frnt_pnl_mutex);
    remove_proc_entry("driver/frnt_pnl", NULL);
    misc_deregister(&frnt_pnl_miscdev);
    mutex_unlock(&frnt_pnl_mutex);
}

int frnt_pnl_init_module(void)
{
    int ret = -EBUSY;

    mutex_lock(&frnt_pnl_mutex);
    ret = misc_register(&frnt_pnl_miscdev);
#if defined(FRNT_PNL_DEBUG)
    if (ret == 0)
        create_proc_read_entry("driver/frnt_pnl", 0, NULL,
                frnt_pnl_read_proc, NULL);
#endif
    mutex_unlock(&frnt_pnl_mutex);

    return ret;
}

module_init(frnt_pnl_init_module);
module_exit(frnt_pnl_cleanup_module);


MODULE_AUTHOR("Ajith B");
MODULE_LICENSE("Seenergy Corp.");



