#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/poll.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* file ops */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/seq_file.h>
#include <linux/cdev.h>
#include <linux/miscdevice.h>
#include <asm/system.h>		/* cli(), *_flags */
#include <asm/uaccess.h>	/* copy_*_user */
#include <asm/io.h>
#include <linux/ioport.h>

#include <asm/arch/cpld.h>

unsigned long cpld_base_addr = CPLD_BASE_ADDR;

int cpld_open (struct inode *inode, struct file *filp)
{
	return 0;
}


int cpld_release (struct inode *inode, struct file *filp)
{
	return 0;
}

ssize_t cpld_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
    int i;
    unsigned char buffer[CPLD_MAX_PORTS];

    count = (count > CPLD_MAX_PORTS)?CPLD_MAX_PORTS:count;

    for (i = 0; i < count; i++) {
        buffer[i] = ioread8(cpld_base_addr+i);
        rmb();
    }
    if (copy_to_user(buf, buffer, count))
        return -EFAULT;
    return count;
}

ssize_t cpld_write(struct file *filp, const char __user *buf, size_t count,
		loff_t *f_pos)
{
    int i;
    unsigned char buffer[CPLD_MAX_PORTS];

    count = (count > CPLD_MAX_PORTS)?CPLD_MAX_PORTS:count;

    if (copy_from_user(buffer, buf, count))
        return -EFAULT;

    for (i = 0; i < count; i++) {
        iowrite8(buffer[i], cpld_base_addr+i);
        wmb();
    }
    return count;
}

unsigned int cpld_poll(struct file *filp, poll_table *wait)
{
	return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
}


struct file_operations cpld_fops = {
	.owner	 = THIS_MODULE,
	.read	 = cpld_read,
	.write	 = cpld_write,
	.poll	 = cpld_poll,
	.open	 = cpld_open,
	.release = cpld_release,
};

static struct miscdevice cpld_miscdev = {
    .minor		= MISC_DYNAMIC_MINOR,
    .name		= "cpld",
    .fops		= &cpld_fops,
};

int cpld_init(void)
{
    int ret;
    if (!request_mem_region(cpld_base_addr, CPLD_MAX_PORTS, "cpld")) {
        printk(KERN_INFO "cpld: Can't get I/O mem address 0x%lx\n",cpld_base_addr);
        return -ENODEV;
    }
    cpld_base_addr = (unsigned long) ioremap(cpld_base_addr, CPLD_MAX_PORTS);
    ret = misc_register(&cpld_miscdev);
#if 0
	ret = register_chrdev(0, "cpld", &cpld_fops);
#endif
	if (ret < 0) {
		printk(KERN_INFO "cpld: Can't get major number\n");
        iounmap((void __iomem *)cpld_base_addr);
		release_mem_region(cpld_base_addr, CPLD_MAX_PORTS);
		return ret;
	}
    return 0;
}

void cpld_cleanup(void)
{
    iounmap((void __iomem *)cpld_base_addr);
    release_mem_region(cpld_base_addr, CPLD_MAX_PORTS);
}

module_init(cpld_init);
module_exit(cpld_cleanup);

MODULE_AUTHOR("Ajith B");
MODULE_LICENSE("Seenergy Corp.");

