/* script.c
 *
 * Functions to call the DHCP client notification scripts 
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include "options.h"
#include "dhcpd.h"
#include "dhcpc.h"
#include "packet.h"
#include "options.h"
#include "debug.h"
#include "PRO_file.h"

#define CGI_DS_CONF "/etc/CGI_ds.conf"

static int run_bound(char **msg);
extern int in_auto_ip;

static int run_deconfig(void);

/* get a rough idea of how long an option will be (rounding up...) */
static int max_option_length[] = {
	[OPTION_IP] =		sizeof("255.255.255.255 "),
	[OPTION_IP_PAIR] =	sizeof("255.255.255.255 ") * 2,
	[OPTION_STRING] =	1,
	[OPTION_BOOLEAN] =	sizeof("yes "),
	[OPTION_U8] =		sizeof("255 "),
	[OPTION_U16] =		sizeof("65535 "),
	[OPTION_S16] =		sizeof("-32768 "),
	[OPTION_U32] =		sizeof("4294967295 "),
	[OPTION_S32] =		sizeof("-2147483684 "),
};


static int upper_length(int length, struct dhcp_option *option)
{
	return max_option_length[option->flags & TYPE_MASK] *
	       (length / option_lengths[option->flags & TYPE_MASK]);
}


static int sprintip(char *dest, char *pre, unsigned char *ip) {
	return sprintf(dest, "%s%d.%d.%d.%d ", pre, ip[0], ip[1], ip[2], ip[3]);
}
#define TASK_PID_FOLDER		"/var/run/down"

void ReloadAllTasks(void)
{
	DIR *dir=NULL;
	struct dirent *dirp=NULL;
	char *p=0;
	int pid=0;
	
	dir=opendir(TASK_PID_FOLDER);
	if(dir!=NULL){	
		while( (dirp=readdir(dir)) !=NULL ){
			if(!strcmp(dirp->d_name,".") || !strcmp(dirp->d_name,".."))
				continue;

			p=strrchr(dirp->d_name,'.');
			if(!p){
				continue;				
			}
			p++;
			pid=-1;
			pid=atoi(p);
			if(pid>1)
				kill(pid, SIGHUP);
		}
		closedir(dir);
	}
		
	return;	
}


#define	MEDIA_SERVER_PID		"/var/run/media_server.pid"
void ReloadUPnPAVServerIP(void)
{
	FILE *fp=NULL;
	int pid=-1;
	
	fp=fopen(MEDIA_SERVER_PID,"rt");
	if(fp){
		fscanf(fp,"%d", &pid);	
		fclose(fp);
	}
	if(pid>0)
		kill(pid, SIGHUP);		
	return;
}
/* Fill dest with the text of option 'option'. */
static void fill_options(char *dest, unsigned char *option, struct dhcp_option *type_p)
{
	int type, optlen;
	u_int16_t val_u16;
	int16_t val_s16;
	u_int32_t val_u32;
	int32_t val_s32;
	int len = option[OPT_LEN - 2];

	dest += sprintf(dest, "%s=", type_p->name);

	type = type_p->flags & TYPE_MASK;
	optlen = option_lengths[type];
	for(;;) {
		switch (type) {
		case OPTION_IP_PAIR:
			dest += sprintip(dest, "", option);
			*(dest++) = '/';
			option += 4;
			optlen = 4;
		case OPTION_IP:	/* Works regardless of host byte order. */
			dest += sprintip(dest, "", option);
 			break;
		case OPTION_BOOLEAN:
			dest += sprintf(dest, *option ? "yes " : "no ");
			break;
		case OPTION_U8:
			dest += sprintf(dest, "%u ", *option);
			break;
		case OPTION_U16:
			memcpy(&val_u16, option, 2);
			dest += sprintf(dest, "%u ", ntohs(val_u16));
			break;
		case OPTION_S16:
			memcpy(&val_s16, option, 2);
			dest += sprintf(dest, "%d ", ntohs(val_s16));
			break;
		case OPTION_U32:
			memcpy(&val_u32, option, 4);
			dest += sprintf(dest, "%lu ", (unsigned long) ntohl(val_u32));
			break;
		case OPTION_S32:
			memcpy(&val_s32, option, 4);
			dest += sprintf(dest, "%ld ", (long) ntohl(val_s32));
			break;
		case OPTION_STRING:
			memcpy(dest, option, len);
			dest[len] = '\0';
			return;	 /* Short circuit this case */
		}
		option += optlen;
		len -= optlen;
		if (len <= 0) break;
	}
}


static char *find_env(const char *prefix, char *defaultstr)
{
	extern char **environ;
	char **ptr;
	const int len = strlen(prefix);

	for (ptr = environ; *ptr != NULL; ptr++) {
		if (strncmp(prefix, *ptr, len) == 0)
			return *ptr;
	}
	return defaultstr;
}


/* put all the paramaters into an environment */
static char **fill_envp(struct dhcpMessage *packet)
{
	int num_options = 0;
	int i, j;
	char **envp;
	unsigned char *temp;
	char over = 0;

	if (packet == NULL)
		num_options = 0;
	else {
		for (i = 0; options[i].code; i++)
			if (get_option(packet, options[i].code))
				num_options++;
		if (packet->siaddr) num_options++;
		if ((temp = get_option(packet, DHCP_OPTION_OVER)))
			over = *temp;
		if (!(over & FILE_FIELD) && packet->file[0]) num_options++;
		if (!(over & SNAME_FIELD) && packet->sname[0]) num_options++;		
	}
	
	envp = xmalloc((num_options + 5) * sizeof(char *));
	envp[0] = xmalloc(sizeof("interface=") + strlen(client_config.interface));
	sprintf(envp[0], "interface=%s", client_config.interface);
	envp[1] = find_env("PATH", "PATH=/bin:/usr/bin:/sbin:/usr/sbin");
	envp[2] = find_env("HOME", "HOME=/");

	if (packet == NULL) {
		envp[3] = NULL;
		return envp;
	}

	envp[3] = xmalloc(sizeof("ip=255.255.255.255"));
	sprintip(envp[3], "ip=", (unsigned char *) &packet->yiaddr);
	for (i = 0, j = 4; options[i].code; i++) {
		if ((temp = get_option(packet, options[i].code))) {
			envp[j] = xmalloc(upper_length(temp[OPT_LEN - 2], &options[i]) + strlen(options[i].name) + 2);
			fill_options(envp[j], temp, &options[i]);
			j++;
		}
	}
	if (packet->siaddr) {
		envp[j] = xmalloc(sizeof("siaddr=255.255.255.255"));
		sprintip(envp[j++], "siaddr=", (unsigned char *) &packet->siaddr);
	}
	if (!(over & FILE_FIELD) && packet->file[0]) {
		/* watch out for invalid packets */
		packet->file[sizeof(packet->file) - 1] = '\0';
		envp[j] = xmalloc(sizeof("boot_file=") + strlen(packet->file));
		sprintf(envp[j++], "boot_file=%s", packet->file);
	}
	if (!(over & SNAME_FIELD) && packet->sname[0]) {
		/* watch out for invalid packets */
		packet->sname[sizeof(packet->sname) - 1] = '\0';
		envp[j] = xmalloc(sizeof("sname=") + strlen(packet->sname));
		sprintf(envp[j++], "sname=%s", packet->sname);
	}	
	envp[j] = NULL;
	return envp;
}


/* Call a script with a par file and env vars */
void run_script(struct dhcpMessage *packet)
{
	int pid;
	char **envp;

	/* call script */
	if(packet==NULL)  {
		run_deconfig();
		return ;
	}
	pid = fork();
	if (pid) {
		waitpid(pid, NULL, 0);
		return;
	} else if (pid == 0) {
		envp = fill_envp(packet);
		
		/*  */

		/*ifconfig interface */
		run_bound(envp);
		exit(1);
	}			
}
#define MDNS_CONF_F		"/etc/howl/mDNSResponder.conf"

int ResetmDNSConf(void)
{
	FILE *fh=NULL;
	char hostname[32+1]={0}, port[12]="80", mac_addr_str[24]={0};
	int ret=0;
	char afp_en[16]={0};
	
	system("/bin/killall -SIGINT mDNSResponder >/dev/null 2>&1");
	sleep(3);
	system("/bin/killall -SIGINT mDNSResponder >/dev/null 2>&1");
	fh=fopen(CGI_DS_CONF,"rt");
	if(!fh)
		return -1;
	ret=PRO_GetStr("network","disk_server_name",hostname,32,fh);
	ret=PRO_GetStr("network","hw_addr",mac_addr_str,24,fh);
	ret=PRO_GetStr("network", "port", port, 11, fh);
	ret=PRO_GetStr("network","apple_enable",afp_en,16,fh);
	fclose(fh);	
	if(ret || !strlen(hostname))
		return -1;
	truncate(MDNS_CONF_F, 0);
	fh=fopen(MDNS_CONF_F, "wt");
	if(!fh){
		return -1;
	}
	fprintf(fh,"\"%s\"\t_http._tcp\tlocal.\t%s\n", hostname,port);
	if(!strcmp(afp_en, "yes")){
		fprintf(fh, "\"%s (AFP)\"\t_afpovertcp._tcp\tlocal.\t548\n", hostname);
	}
	fclose(fh);
	system("/sbin/mDNSResponder >/dev/null 2>&1");
	return 0;
}
#ifdef AUTO_IP

int ResetUPnP(void)
{
	int pid=0;
	FILE *fh;
	char upnp[6];

	fh=fopen(CGI_DS_CONF,"rt");
	if(!fh)
		return 1;
	PRO_GetStr("network","upnp",upnp,5,fh);
	fclose(fh);
	fh=fopen("/var/run/upnpd.pid", "rt");
	if(fh){
		fscanf(fh,"%d",&pid);
		fclose(fh);
		if(pid>1)
			kill(pid,SIGINT);
	}
	else
		system("/bin/killall upnpd 2>/dev/null");
	if(!strcmp(upnp,"yes")){
		if(fork()==0){
			setsid();
			//for(i=0;i<64;i++)
				//close(i);
			if(fork()>0){
				exit(0);
			}
			chdir("/");
			umask(0);
			sleep(8);
			execlp("/usr/sbin/upnpd","upnpd","&>/dev/null",NULL);
			exit(0);
		}
	}
	return 0;
}
void config_auto_ip(char *ip_addr, char *netmask)
{
	char *msg[3]={NULL},ip_str[24], mask_str[32];
	
	sprintf(ip_str, "ip=%s",ip_addr);
	sprintf(mask_str, "subnet=%s", netmask);
	msg[0]=ip_str;
	msg[1]=mask_str;
	msg[2]=NULL;
	run_bound(msg);
	system("/sbin/route add -net default egiga0 2>/dev/null"); 
	system("/sbin/route add -net 239.0.0.0 netmask 255.0.0.0 egiga0 2>/dev/null");
	system("/etc/rc.d/rc.samba 2>/dev/null"); 
	ReloadUPnPAVServerIP();
	ResetUPnP();
	ReloadAllTasks();
	ResetmDNSConf();
}
#endif


static int run_bound(char ** msg)
{
	char buf[128],buf1[100];
	char temp[128] = "";
	int num;
	char *p0,*p1;
	int ipflag=0,maskflag=0,routeflag=0;
	int tmp0,tmp1,tmp2,tmp3;
	unsigned char ip[5],nmask[5],bcast[5];
	FILE *fp;
	int mtu;
	char dns_type[12]="dhcp";
 	char wins_enable[8]="no";
    
	fp=fopen(CGI_DS_CONF,"rt");
	if(!fp)
		mtu=1500;
	else{
		if(PRO_GetInt("network","mtu",&mtu,fp))
			mtu=1500;
		PRO_GetStr("network","dns_type",dns_type, sizeof(dns_type)-1, fp);
		PRO_GetStr("network","wins_enable",wins_enable, sizeof(wins_enable)-1, fp);    
		fclose(fp);
	}
	
	for(num=0;msg[num]!=NULL;num++)	{
		if((p0 = strstr(msg[num],"ip=" ))!=NULL) {
			p0 += strlen("ip=");
			strcpy(temp,p0);
			ipflag=1;
			sscanf(temp,"%d.%d.%d.%d",&tmp0,&tmp1,&tmp2,&tmp3);
			ip[0]=(unsigned char)tmp0;ip[1]=(unsigned char)tmp1;
			ip[2]=(unsigned char)tmp2;ip[3]=(unsigned char)tmp3;
			ip[4]='\0';
			p1=strchr(temp, ' ');
			if(p1)	
				*p1=0;
			sprintf(buf,"/sbin/ifconfig %s %s mtu %d 2> /dev/null",client_config.interface,temp,mtu);
			system(buf);
			PRO_SetStr("network","ip_addr",temp,CGI_DS_CONF);
			continue;
		}
		if((p0 = strstr(msg[num],"subnet="))!=NULL) {
			p0 += strlen("subnet=");
			strcpy(temp,p0);
			maskflag=1;
			sscanf(temp,"%d.%d.%d.%d",&tmp0,&tmp1,&tmp2,&tmp3);
			nmask[0]=(unsigned char)tmp0;nmask[1]=(unsigned char)tmp1;
			nmask[2]=(unsigned char)tmp2;nmask[3]=(unsigned char)tmp3;
			nmask[4]='\0';
			p1=strchr(temp, ' ');
			if(p1)	
				*p1=0;			
			sprintf(buf,"%s %s %s %s 2> /dev/null","/sbin/ifconfig",client_config.interface,"netmask",temp);
			system(buf);
			PRO_SetStr("network","netmask",temp,CGI_DS_CONF);
			continue;
		}
		if((strcmp(wins_enable,"yes")==0)&&((p0 = strstr(msg[num],"wins="))!=NULL)) {
			p0 += strlen("wins=");
			strcpy(temp,p0);
			p1=strchr(temp, ' ');
			if(!p1){
				PRO_SetStr("network","wins_server",temp,CGI_DS_CONF);
				continue;
			}
			*p1=0;
			p1++;
			PRO_SetStr("network","wins_server",temp,CGI_DS_CONF);
			p0=strchr(p1,' ');
			if(!p0){	
				if(strlen(p1))			
					PRO_SetStr("network","wins_server2",p1,CGI_DS_CONF);
				else
					continue;
			}
			*p0=0;
			PRO_SetStr("network","wins_server2",p1,CGI_DS_CONF);
			p0++;
			p1=strchr(p0, ' ');
			if(!p1){
				if(strlen(p0))						
					PRO_SetStr("network","wins_server3",p0,CGI_DS_CONF);
				else
					continue;
			}	
			*p1=0;
			PRO_SetStr("network","wins_server3",p0,CGI_DS_CONF);
			continue;
		}
		if((p0 = strstr(msg[num],"router="))!=NULL) {
			p0 += strlen("router=");
			strcpy(temp,p0);
			p1=strchr(temp, ' ');
			if(p1)	
				*p1=0;			
			sprintf(buf,"/sbin/route del default gw 0.0.0.0 dev %s 2> /dev/null ",client_config.interface);
			system(buf);
			sprintf(buf,"/sbin/route add default gw %s  dev %s 2> /dev/null ",temp,client_config.interface);
			system(buf);
			strcpy(buf1,buf);
			routeflag=1;
			PRO_SetStr("network", "gateway", temp,   CGI_DS_CONF);
			continue;
		}
		if((p0 =strstr(msg[num],"dns="))!=NULL) {
			if(!strcmp(dns_type, "dhcp")){
				truncate("/etc/resolv.conf", 0);
				PRO_SetStr("network","dns_server1","",CGI_DS_CONF);
				PRO_SetStr("network","dns_server2","",CGI_DS_CONF);
				PRO_SetStr("network","dns_server3","",CGI_DS_CONF);				
				p0 += strlen("dns=");
				if((p1 = strchr(p0,' '))==NULL) {
					strcpy(temp,p0);
					if(strlen(temp)>5){
						sprintf(buf,"echo nameserver %s >> %s",temp,"/etc/resolv.conf");
						system(buf);
						PRO_SetStr("network","dns_server1",temp,CGI_DS_CONF);
					}
					continue;
				}
				*p1=0;
				strcpy(temp,p0);
				if(strlen(temp)>5){
					sprintf(buf,"echo nameserver %s >> %s",temp,"/etc/resolv.conf");
					system(buf);
					PRO_SetStr("network","dns_server1",temp,CGI_DS_CONF);
				}
				p0 = p1 +1;
				/*dns2*/
				if((p1 = strchr(p0,' '))==NULL) {
						strcpy(temp,p0);
						if(strlen(temp)>5){
							sprintf(buf,"echo nameserver %s >> %s",temp,"/etc/resolv.conf");
							system(buf);
							PRO_SetStr("network","dns_server2",temp,CGI_DS_CONF);
						}
						continue;
					}
					*p1=0;
					strcpy(temp,p0);
					if(strlen(temp)>5){
						sprintf(buf,"echo nameserver %s >> %s",temp,"/etc/resolv.conf");
						system(buf);
						PRO_SetStr("network","dns_server2",temp,CGI_DS_CONF);
					}
					p0 = p1 +1;
					/*dns3*/
					p1=strchr(p0,' ');
					if(p1)
						*p1=0;
					strcpy(temp,p0);
					if(strlen(temp)>5){
						sprintf(buf,"echo nameserver %s >> %s",temp,"/etc/resolv.conf");
						system(buf);
						PRO_SetStr("network","dns_server3",temp,CGI_DS_CONF);
					}
					if(p1)
						p0 = p1 +1;
			}
			continue;
		}
	}	
	if(maskflag&&ipflag)	{
		bcast[0]=(~nmask[0])|ip[0];bcast[1]=(~nmask[1])|ip[1];
		bcast[2]=(~nmask[2])|ip[2];bcast[3]=(~nmask[3])|ip[3];	
		sprintf(buf,"/sbin/ifconfig %s broadcast %d.%d.%d.%d 2> /dev/null",
		client_config.interface,bcast[0],bcast[1],bcast[2],bcast[3]);
		system(buf);
		if(routeflag){
			sprintf(buf,"/sbin/route del default gw 0.0.0.0 dev %s 2> /dev/null ",client_config.interface);
			system(buf);				
			system(buf1);
		}
		else {
			PRO_SetStr("network", "gateway", "",   CGI_DS_CONF);		    
		}
	}
	system("/sbin/route add -net 239.0.0.0 netmask 255.0.0.0 egiga0 2>/dev/null");
	ReloadUPnPAVServerIP();
	ResetmDNSConf();
	if(in_auto_ip)
		ResetUPnP();
	return 0;
}

static int run_deconfig(void)
{
	char buf[50];
	int ret;
	
	sprintf(buf,"%s %s %s >/dev/null","/sbin/ifconfig" ,client_config.interface,"0.0.0.0");
	ret=system(buf);
	
	return ret;
}


