/*   ____  __  _ _____ ____     _ _            _   
**  / ___||  \/ |_   _|  _ \___| (_) ___ _ __ | |_ 
**  \___ \| |\/| || | | |_)/ __| | |/ _ \ '_ \| __|
**   ___) | |  | || | |  _| (__| | |  __/ | | | |_ 
**  |____/|_|  |_||_| |_|  \___|_|_|\___|_| |_|\__|
**   
**  SMTPclient -- simple SMTP client
**
**  This program is a minimal SMTP client that takes an email
**  message body and passes it on to a SMTP server (default is the
**  MTA on the local host). Since it is completely self-supporting,
**  it is especially suitable for use in restricted environments.
**
**  ======================================================================
**
**  Copyright (c) 1997 Ralf S. Engelschall, All rights reserved.
**
**  This program is free software; it may be redistributed and/or modified
**  only under the terms of either the Artistic License or the GNU General
**  Public License, which may be found in the SMTP source distribution.
**  Look at the file COPYING. 
**
**  This program is distributed in the hope that it will be useful, but
**  WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  ======================================================================
**
**  smtpclient_main.c -- program source
**
**  Based on smtp.c as of August 11, 1995 from
**      W.Z. Venema,
**      Eindhoven University of Technology,
**      Department of Mathematics and Computer Science,
**      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <syslog.h>
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <signal.h>
#include "dns.h"

#include "smtpclient_getopt.h"
#include "smtpclient_errno.h"
#include "smtpclient_vers.h"
#include "PRO_file.h"
#include "constant.h"

static char *cc_addr = NULL;
static char *err_addr = NULL;
static char *from_addr = NULL;
char mailhost[64];
static int mailport = 25;
static char *reply_addr = NULL;
static char *subject = NULL;
static int mime_style = 1;
static int verbose = 0;
static char *msg_file = NULL;
static int usesyslog = 0;
static int rflag = 0;
#define	TMP_INFO_FILE	"/etc/test_mail.info"
#define	STATUS_FILE		"/tmp/mail.status"

static FILE *sfp;
static FILE *rfp;
static FILE *msgfp;
static int chk_ehlo = 0;		//check ehlo response from smtp server
									// 1: server supports auth mechanism
									// 0: no auth mechanism

#define dprintf  if (verbose) printf
#define dvprintf if (verbose) vprintf

/* hack for Ultrix */
#ifndef LOG_DAEMON
#define LOG_DAEMON 0
#endif
int change_space_to_underline(char *host_name)
{
	int i=0;
	
	if (NULL == host_name)
	{
		return -1;
	}
	while(host_name[i]!=0x00)
	{
		if (0x20==host_name[i])	{
			host_name[i] = 0x5f;
		}
		i++;
	}
	return 0;
}
void WriteSMTPStatus(char *format, ...)
{
    va_list args;
	FILE *fp=NULL;

	fp=fopen(STATUS_FILE, "at");
	if(!fp)
		return;
	va_start(args, format);
    vfprintf(fp, format, args);
    fputs("\n",fp);
    va_end(args);		
	fclose(fp);
}
#define	MAX_KEY_LEN			128
#define	MAX_SECTION_LEN		64
#define	MESSAGE_FILE	"/etc/cgi_msg"

int GetLineFromFile(char **lbuf,FILE *fp)
{
	int	c,sz=128;
	char *p,*buf;

	buf=(char *) malloc(sz);
	if(!buf){
		*lbuf=NULL;
		return(-1);
	}
	p=buf;
	while ((c=fgetc(fp)) != '\n' && c!=EOF) {
		if (p-buf ==sz-2) {
			buf=(char *) realloc(buf, sz*2);
			if(!buf){
				*lbuf=NULL;
				return 1;
			}
			p=buf+sz-2;
			sz*=2;
		}
		*p++=c;
	}
	if(c=='\n')
	     *p++=c;
	if (c==EOF &&p==buf) {
		free(buf);
		*lbuf=NULL;
		return(-1);
	}
	*p++='\0';
	*lbuf=buf;
	return(0);
}

int PRO_GetStr(char *sect, char *key, char *val, int size, FILE * fp) 
{
    char *buf;
    char item[MAX_KEY_LEN + 2], section[MAX_SECTION_LEN + 3];
    int find_sect = 0, len = 0, len1 = 0;

    if (val == NULL || fp == NULL || key == NULL)
		return -1;
    if (sect != NULL) {
		if (sect[0] != '[' || sect[strlen(sect) - 1] != ']')
		    sprintf(section, "[%s]", sect);
		else
		    strcpy(section, sect);
		rewind(fp);
		len = strlen(section);
		while (!GetLineFromFile(&buf, fp)) {
		    if (!strncmp(buf, section, len)) {
				find_sect = 1;
				free(buf);
				break;
		    }
		    free(buf);
		}
    }
    else
		find_sect = 0;
    if (!find_sect)
		rewind(fp);
    sprintf(item, "%s=", key);
    len = strlen(item);
    while (!GetLineFromFile(&buf, fp)) {
		if (buf[0] == '[') {
		    free(buf);
		    break;		// end of section
		}
		if (!strncmp(buf, item, len)) {
		    len1 = strlen(buf);
		    len1--;
		    while (buf[len1] == 0x0d || buf[len1] == 0x0a)
				buf[len1--] = '\0';
		    if (strlen(buf + len) > size)
				buf[len + size] = 0;
		    strcpy(val, buf + len);
		    free(buf);
		    return 0;
		}
		free(buf);
    }
    return -1;
}

#define	SMTP_UNKNOWN_HOST		1
#define	SMTP_CONN_FAIL			2
#define	SMTP_MAIL_SERVER_OK		3
#define	SMTP_MAIL_ADDR_OK		4
#define	SMTP_MAIL_SERVER_FAIL	5
#define	SMTP_MAIL_ADDR_FAIL		6

int ReadErrMsgFromFile(int index, char *pMsg)
{
	FILE *ff=NULL;
	char index_str[4]={0};
	int ret=-1;
	        	
	sprintf(index_str, "%d",index);
	ff=fopen(MESSAGE_FILE, "rt");
	if(ff){
		ret=PRO_GetStr("MAIL", index_str, pMsg, 128, ff);
		fclose(ff);
	}	

	return 	ret;
}

/*
**  logging support
*/
void
LogMsg (char *str, ...)
{
    va_list ap;
    char buf[1024];

    va_start (ap, str);
    vsprintf (buf, str, ap);
    if (usesyslog)
        syslog (LOG_ERR, "%s", buf);
    else
        fprintf (stdout, "%s\n", buf);
    va_end (ap);
    return;
}

/*
**  usage page
*/
void
usage (void)
{
    fprintf (stderr, "Usage: smtp [options] recipients ...\n");
    fprintf (stderr, "\n");
    fprintf (stderr, "Message Header Options:\n");
    fprintf (stderr, "  -s, --subject=STR      subject line of message\n");
    fprintf (stderr, "  -f, --from=ADDR        address of the sender\n");
    fprintf (stderr,
             "  -r, --reply-to=ADDR    address of the sender for replies\n");
    fprintf (stderr,
             "  -e, --errors-to=ADDR   address to send delivery errors to\n");
    fprintf (stderr,
             "  -c, --carbon-copy=ADDR address to send copy of message to\n");
    fprintf (stderr, "\n");
#if 0    
    fprintf (stderr, "Processing Options:\n");
    fprintf (stderr,
             "  -S, --smtp-host=HOST   host where MTA can be contacted via SMTP\n");
    fprintf (stderr,
             "  -P, --smtp-port=NUM    port where MTA can be contacted via SMTP\n");
    fprintf (stderr,
             "  -M, --mime-encode      use MIME-style translation to quoted-printable\n");
    fprintf (stderr,
             "  -L, --use-syslog       log errors to syslog facility instead of stderr\n");
    fprintf (stderr, "\n");
#endif    
    fprintf (stderr, "Giving Feedback:\n");
    fprintf (stderr,
             "  -v, --verbose          enable verbose logging messages\n");
    fprintf (stderr, "  -V, --version          display version string\n");
    fprintf (stderr, "  -h, --help             display this page\n");
    fprintf (stderr, "\n");
    return;
}

/*
**  version page
*/
void
version (void)
{
    fprintf (stdout, "%s\n", SMTPclient_Hello);
    fprintf (stdout, "\n");
    fprintf (stdout,
             "Copyright (c) 1997 Ralf S. Engelschall, All rights reserved.\n");
    fprintf (stdout, "\n");
    fprintf (stdout,
             "This program is distributed in the hope that it will be useful,\n");
    fprintf (stdout,
             "but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
    fprintf (stdout,
             "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    fprintf (stdout, "the GNU General Public License for more details.\n");
    fprintf (stdout, "\n");
    return;
}

void alarm_handle(int sig)
{
	LogMsg ("Timeout\n");
	exit(1);
}

/*
**  examine message from server 
*/
int
get_response (void)
{
    char buf[BUFSIZ];

    while (fgets (buf, sizeof (buf), rfp)) {
        buf[strlen (buf) - 1] = 0;
        dprintf ("%s --> %s\n", mailhost, buf);
        if (!isdigit (buf[0]) || buf[0] > '3') {
            LogMsg ("%s", buf);
            return (1);
        }
		if (!(strncmp(&buf[4],"AUTH", 4)))
	            chk_ehlo=1;  
        if (buf[3] != '-')
            break;	            
    }    
    return 0;
}

/*
**  say something to server and check the response
*/
int
chat (char *fmt, ...)
{
    va_list ap;

    va_start (ap, fmt);
    vfprintf (sfp, fmt, ap);
    va_end (ap);

#if 1
    va_start (ap, fmt);
    dprintf ("%s <-- ", mailhost);
    dvprintf (fmt, ap);
    va_end (ap);
#endif
    fflush (sfp);
    return (get_response ());
}

/*
**  transform to MIME-style quoted printable
**
**  Extracted from the METAMAIL version 2.7 source code (codes.c)
**  and modified to emit \r\n at line boundaries.
*/

static char basis_hex[] = "0123456789ABCDEF";

void
toqp (FILE * infile, FILE * outfile)
{
    int c;
    int ct = 0;
    int prevc = 255;

    while ((c = getc (infile)) != EOF) {
        if ((c < 32 && (c != '\n' && c != '\t'))
            || (c == '=') || (c >= 127) || (ct == 0 && c == '.')) {
            putc ('=', outfile);
            putc (basis_hex[c >> 4], outfile);
            putc (basis_hex[c & 0xF], outfile);
            ct += 3;
            prevc = 'A';        /* close enough */
        }
        else if (c == '\n') {
            if (prevc == ' ' || prevc == '\t') {
                putc ('=', outfile);    /* soft & hard lines */
                putc (c, outfile);
            }
            putc (c, outfile);
            ct = 0;
            prevc = c;
        }
        else {
            if (c == 'F' && prevc == '\n') {
                /*
                 * HORRIBLE but clever hack suggested by MTR for
                 * sendmail-avoidance
                 */
                c = getc (infile);
                if (c == 'r') {
                    c = getc (infile);
                    if (c == 'o') {
                        c = getc (infile);
                        if (c == 'm') {
                            c = getc (infile);
                            if (c == ' ') {
                                /* This is the case we are looking for */
                                fputs ("=46rom", outfile);
                                ct += 6;
                            }
                            else {
                                fputs ("From", outfile);
                                ct += 4;
                            }
                        }
                        else {
                            fputs ("Fro", outfile);
                            ct += 3;
                        }
                    }
                    else {
                        fputs ("Fr", outfile);
                        ct += 2;
                    }
                }
                else {
                    putc ('F', outfile);
                    ++ct;
                }
                ungetc (c, infile);
                prevc = 'x';    /* close enough -- printable */
            }
            else {
                putc (c, outfile);
                ++ct;
                prevc = c;
            }
        }
        if (ct > 72) {
            putc ('=', outfile);
            putc ('\r', outfile);
            putc ('\n', outfile);
            ct = 0;
            prevc = '\n';
        }
    }
    if (ct) {
        putc ('=', outfile);
        putc ('\r', outfile);
        putc ('\n', outfile);
    }
    return;
}
/*
**  base64 encoder
**
**  encode username and password for one of smtp authentication "LOGIN"
*/
static char table64[]=
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
void base64Encode(char *intext, char *output)
{
  unsigned char ibuf[3];
  unsigned char obuf[4];
  int i;
  int inputparts;

  while(*intext) {
    for (i = inputparts = 0; i < 3; i++) {
      if(*intext) {
        inputparts++;
        ibuf[i] = *intext;
        intext++;
      }
      else
        ibuf[i] = 0;
      }

    obuf [0] = (ibuf [0] & 0xFC) >> 2;
    obuf [1] = ((ibuf [0] & 0x03) << 4) | ((ibuf [1] & 0xF0) >> 4);
    obuf [2] = ((ibuf [1] & 0x0F) << 2) | ((ibuf [2] & 0xC0) >> 6);
    obuf [3] = ibuf [2] & 0x3F;

    switch(inputparts) {
      case 1: /* only one byte read */
        sprintf(output, "%c%c==",
            table64[obuf[0]],
            table64[obuf[1]]);
        break;
      case 2: /* two bytes read */
        sprintf(output, "%c%c%c=",
            table64[obuf[0]],
            table64[obuf[1]],
            table64[obuf[2]]);
        break;
      default:
        sprintf(output, "%c%c%c%c",
            table64[obuf[0]],
            table64[obuf[1]],
            table64[obuf[2]],
            table64[obuf[3]] );
        break;
    }
    output += 4;
  }
  *output=0;
}

/*
**  main procedure
*/

struct option options[] = {
    {"subject", 1, NULL, 's'},
    {"from", 1, NULL, 'f'},
    {"replay-to", 1, NULL, 'r'},
    {"errors-to", 1, NULL, 'e'},
    {"carbon-copy", 1, NULL, 'c'},
    {"smtp-host", 1, NULL, 'h'},
    {"smtp-port", 1, NULL, 'p'},
    {"mime-encode", 0, NULL, 'M'},
    {"use-syslog", 0, NULL, 'L'},
    {"verbose", 0, NULL, 'v'},
    {"version", 0, NULL, 'V'},
    {"help", 0, NULL, 'h'}
};

int
main (int argc, char **argv)
{
    char buf[BUFSIZ];
    char my_name[BUFSIZ];
    char tmp_buf[256], mail_addr[128]={0}, err_msg[128]={0};
	char enc_user[128]={0}, enc_pw[128]={0};
    char smtp_server[128]={0},smtp_user[64]={0}, smtp_pass[64]={0};
    int use_smtp_server=0,smtp_auth=0;
    struct sockaddr_in sin;
    struct hostent *hp;
    //struct servent *sp;
    int c;
    int s;
    int r;
    int i=0, j=0;
    struct passwd *pwd;
    char  *p;
    //char srcchar;
    struct MX *host;
    int ret, num;
	FILE *fh=NULL;
	int send_ok=0;
	int native_mime=0;
    /* 
     *  Go away when something gets stuck.
     */
    signal(SIGALRM,alarm_handle);
    alarm (60);
	remove(STATUS_FILE);
    /*
     *  Parse options
     */
    while ((c =
            getopt_long (argc, argv, ":s:f:r:e:c:m:P:MLnvVh", options,
                         NULL)) != EOF) {
        switch (c) {
        case 's':
            subject = optarg;
            break;
        case 'f':
            from_addr = optarg;
            break;
        case 'r':
            reply_addr = optarg;
            break;
        case 'e':
            err_addr = optarg;
            break;
        case 'c':
            cc_addr = optarg;
            break;
        case 'm':
            msg_file = optarg;
            break;
        case 'P':
            mailport = atoi (optarg);
            break;
        case 'M':
            mime_style = 1;
            break;
        case 'n':
            native_mime = 1;
            break;            
        case 'L':
            usesyslog = 1;
            break;
        case 'v':
            verbose = 1;
            break;
        case 'V':
            version ();
            exit (0);
        case 'h':
            usage ();
            exit (0);
        default:
            fprintf (stderr, "SMTP: invalid option `%c'\n", optopt);
            fprintf (stderr, "Try `%s --help' for more information.\n",
                     argv[0]);
            exit (1);
        }
    }
    if (argc == optind) {
        fprintf (stderr, "SMTP: wrong number of arguments\n");
        fprintf (stderr, "Try `%s --help' for more information.\n", argv[0]);
        exit (1);
    }

    if(verbose)
 		fh=fopen(TMP_INFO_FILE,"rt");
	else
		fh=fopen(FSH_DS_CONF_FILE,"rt");
	if(fh){
		int smtp_port=25;
			
		PRO_GetInt("LOG", "smtp_specify", &use_smtp_server, fh);
		PRO_GetStr("LOG", "smtp_server", smtp_server,sizeof(smtp_server)-1, fh);
		PRO_GetInt("LOG", "smtp_auth", &smtp_auth, fh);
		PRO_GetStr("LOG", "smtp_user", smtp_user,sizeof(smtp_user)-1, fh);
		PRO_GetStr("LOG", "smtp_pass", smtp_pass,sizeof(smtp_pass)-1, fh);
		PRO_GetInt("LOG", "smtp_port", &smtp_port, fh);
		fclose(fh);
		if(smtp_port>=1024)
			mailport=smtp_port;
	}
    /*  
     *  Open Syslog facility
     */
    if (usesyslog)
        openlog (argv[0], LOG_PID, LOG_DAEMON);
    /*
     *  Find out my own host name for HELO; 
     *  if possible, get the FQDN.
     */
    if (gethostname (my_name, sizeof (my_name) - 1) < 0) {
        LogMsg ("gethostname: %s", errorstr (errno));
        exit (1);
    }
    change_space_to_underline(my_name);
	
    /*
     *  Determine from address.
     */
    if (from_addr == NULL) {
	    if ((pwd = getpwuid (getuid ())) == 0) {
            sprintf (buf, "userid-%d@%s", getuid (), my_name);
        }
        else {
            //sprintf (buf, "%s@%s", pwd->pw_name, my_name);
            sprintf (buf, "%s@%s", "admin", my_name);
        }
        from_addr = strdup (buf);
    }
	if(use_smtp_server){
		 for (j = optind; j < argc; j++) {
	        p = strchr (argv[j], '@');
	        if (!p)
	            continue;		 	
			strcpy(mail_addr, smtp_server);
		    if ((hp = gethostbyname (smtp_server)) == NULL) {
				goto out;
		    }
		    if (hp->h_addrtype != AF_INET) {
		        LogMsg("unknown address family: %d", hp->h_addrtype);
				goto out;
		    }	    
			memset ((char *) &sin, 0, sizeof (sin));
			memcpy ((char *) &sin.sin_addr, hp->h_addr, hp->h_length);
			sin.sin_family = hp->h_addrtype;
			sin.sin_port = htons (mailport);
			if ((s = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
				LogMsg ("socket: %s", errorstr (errno));
				goto out;
			}
			if (connect (s, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
				LogMsg ("connect: %s", errorstr (errno));
				goto out;
			}
			if ((r = dup (s)) < 0) {
				LogMsg ("dup: %s", errorstr (errno));
				goto out;
			}
			if ((sfp = fdopen (s, "w")) == 0) {
				LogMsg ("fdopen: %s", errorstr (errno));
				goto out;
			}
			if ((rfp = fdopen (r, "r")) == 0) {
				LogMsg ("fdopen: %s", errorstr (errno));
				exit (1);
			}
			/* 
			 *  Give out SMTP headers.
			 */
			ret = get_response ();  /* banner */
			if (ret)
				goto out;
			chat("EHLO %s\r\n", my_name);
			if(smtp_auth && chk_ehlo){	//smtp authentication
				chat ("AUTH LOGIN\r\n");
				base64Encode(smtp_user, enc_user);
				chat ("%s\r\n", enc_user);
				base64Encode(smtp_pass, enc_pw);
				ret = chat ("%s\r\n", enc_pw);
				if (ret > 0)
					goto out;
			}
		    ret = chat ("MAIL FROM: <%s>\r\n", from_addr);
			if (ret)
				goto out;
			ret = chat ("RCPT TO: <%s>\r\n", argv[j]);
			if (ret)
				goto out;
			ret = chat ("DATA\r\n");
			if (ret)
				goto out;
			send_ok=1;               
			/* 
			*  Give out Message header. 
			*/
			fprintf (sfp, "From: %s\r\n", from_addr);
			if (subject)
				fprintf (sfp, "Subject: %s\r\n", subject);
			if (reply_addr)
				fprintf (sfp, "Reply-To: %s\r\n", reply_addr);
			if (err_addr)
				fprintf (sfp, "Errors-To: %s\r\n", err_addr);
			if ((pwd = getpwuid (getuid ())) == 0) {
				fprintf (sfp, "Sender: userid-%d@%s\r\n", getuid (), my_name);
			}
			else {
				//fprintf (sfp, "Sender: %s@%s\r\n", pwd->pw_name, my_name);
				fprintf (sfp, "Sender: %s\r\n", from_addr);
			}
			fprintf (sfp, "To: %s", argv[j]);
			fprintf (sfp, "\r\n");
			if (cc_addr)
				fprintf (sfp, "Cc: %s\r\n", cc_addr);
		
			if (mime_style) {
				fprintf (sfp, "MIME-Version: 1.0\r\n");
				if(native_mime)
					fprintf (sfp, "Content-Type: text/plain; charset=Shift_JIS\r\n");
				else
					fprintf (sfp, "Content-Type: text/plain; charset=UTF-8\r\n");
				fprintf (sfp, "Content-Transfer-Encoding: quoted-printable\r\n");
			}
			fprintf (sfp, "\r\n");
			if (rflag == 1) 
				fprintf (sfp, "%s\r\n", tmp_buf);
			else{
				if(!msg_file){
					while (fgets (buf, sizeof (buf), stdin)) {
						if (rflag == 0)
							sprintf (tmp_buf, "%s", buf);
						buf[strlen (buf) - 1] = 0;
						if (strcmp (buf, ".") == 0)   /* quote alone dots */
							fprintf (sfp, "..\r\n");
						else      /* pass thru mode */
							fprintf (sfp, "%s\r\n", buf);
					}
				}
				else{
					if ((msgfp = fopen (msg_file, "r")) != 0) {
						while (fgets (buf, sizeof (buf), msgfp)) {
							if (rflag == 0)
								sprintf (tmp_buf, "%s", buf);
							buf[strlen (buf) - 1] = 0;
							if (strcmp (buf, ".") == 0)   /* quote alone dots */
								fprintf (sfp, "..\r\n");
							else      /* pass thru mode */
								fprintf (sfp, "%s\r\n", buf);
						}
						fclose(msgfp);
					}
					fprintf (sfp, "\r\n");  /* terminate the text to tell MTA the end of messages */
					rflag = 1;            
				}
				ret = chat (".\r\n");
				if (ret)
					goto out;
				ret = chat ("QUIT\r\n");
				if (ret)
					goto out;
				fclose (sfp);
				fclose (rfp);
			}
		}
	}
	else{
	    /*
	     *  Connect to smtp daemon on mailhost.
	     */
	    for (j = optind; j < argc; j++) {
	        p = strchr (argv[j], '@');
	        if (!p)
	            continue;
	        else
	            sprintf (mailhost, "%s", p + 1);
			strcpy(mail_addr, argv[j]);            
	        ret = dns_mxip (mailhost, &host, &num);
	        if (ret){
				ret=ReadErrMsgFromFile(SMTP_UNKNOWN_HOST, err_msg);
				if(!ret && strlen(err_msg))
					WriteSMTPStatus(err_msg, mailhost);
				else
					WriteSMTPStatus("%s: Unknown Host.", mailhost);
	            continue;
			}
	        for (i = 0; i < num; i++) {
	            if ((hp = gethostbyname (host[i].name)) == NULL) {
	                if (i == num - 1) {
	                    LogMsg ("%s: unknown host\n", mailhost);
						ret=ReadErrMsgFromFile(SMTP_UNKNOWN_HOST, err_msg);
						if(!ret && strlen(err_msg))
							WriteSMTPStatus(err_msg, mailhost);
						else
							WriteSMTPStatus("%s: Unknown Host.", mailhost);	                    
	                    goto out;
	                }
	                else
	                    continue;
	            }
	            if (hp->h_addrtype != AF_INET) {
	                LogMsg ("unknown address family: %d", hp->h_addrtype);
	                //WriteSMTPStatus("%s: Unknown Address Family.\n", hp->h_addrtype);
	               	goto out;
	            }
	            memset ((char *) &sin, 0, sizeof (sin));
	            memcpy ((char *) &sin.sin_addr, hp->h_addr, hp->h_length);
	            sin.sin_family = hp->h_addrtype;
	            sin.sin_port = htons (25);
	            if ((s = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
	                LogMsg ("socket: %s", errorstr (errno));
	                //WriteSMTPStatus("Socket Error: %s", errorstr (errno));
	                goto out;
	            }
	            if (connect (s, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
	                LogMsg ("connect: %s", errorstr (errno));
					ret=ReadErrMsgFromFile(SMTP_CONN_FAIL, err_msg);
					if(!ret && strlen(err_msg))
						WriteSMTPStatus(err_msg, errorstr (errno));
					else
						WriteSMTPStatus("Connection Failed: %s", errorstr (errno));	                
	                goto out;
	            }
	            if ((r = dup (s)) < 0) {
	                LogMsg ("dup: %s", errorstr (errno));
	                goto out;
	            }
	            if ((sfp = fdopen (s, "w")) == 0) {
	                LogMsg ("fdopen: %s", errorstr (errno));
	                goto out;
	            }
	            if ((rfp = fdopen (r, "r")) == 0) {
	                LogMsg ("fdopen: %s", errorstr (errno));
	                goto out;
	            }
	            /* 
	             *  Give out SMTP headers.
	             */
	            ret = get_response ();  /* banner */
	            if (ret)
	                continue;
	            ret = chat ("HELO %s\r\n", my_name);
	            if (ret)
	                continue;
	
	            ret = chat ("MAIL FROM: <%s>\r\n", from_addr);
		    if (ret)
	                continue;
	            ret = chat ("RCPT TO: <%s>\r\n", argv[j]);
	            if (ret)
	                continue;
	            ret = chat ("DATA\r\n");
	            if (ret)
	                continue;
				send_ok=1;               
	            /* 
	             *  Give out Message header. 
	             */
	            fprintf (sfp, "From: %s\r\n", from_addr);
	            if (subject)
	                fprintf (sfp, "Subject: %s\r\n", subject);
	
	            if (reply_addr)
	                fprintf (sfp, "Reply-To: %s\r\n", reply_addr);
	            if (err_addr)
	                fprintf (sfp, "Errors-To: %s\r\n", err_addr);
	            if ((pwd = getpwuid (getuid ())) == 0) {
	                fprintf (sfp, "Sender: userid-%d@%s\r\n", getuid (), my_name);
	            }
	            else {
	                //fprintf (sfp, "Sender: %s@%s\r\n", pwd->pw_name, my_name);
	                fprintf (sfp, "Sender: %s\r\n", from_addr);
	            }
	            fprintf (sfp, "To: %s", argv[j]);
	            fprintf (sfp, "\r\n");
	            if (cc_addr)
	                fprintf (sfp, "Cc: %s\r\n", cc_addr);
	
	            if (mime_style) {
	                fprintf (sfp, "MIME-Version: 1.0\r\n");
	                if(native_mime)
	                	fprintf (sfp, "Content-Type: text/plain; charset=Shift_JIS\r\n");
	                else
	                	fprintf (sfp, "Content-Type: text/plain; charset=UTF-8\r\n");
	                fprintf (sfp, "Content-Transfer-Encoding: quoted-printable\r\n");
	            }
	            fprintf (sfp, "\r\n");
	            if (rflag == 1) 
	                fprintf (sfp, "%s\r\n", tmp_buf);
	            else{
	                if(!msg_file)
	                    while (fgets (buf, sizeof (buf), stdin)) {
	                        if (rflag == 0)
	                            sprintf (tmp_buf, "%s", buf);
	                   		buf[strlen (buf) - 1] = 0;
	                        if (strcmp (buf, ".") == 0)   /* quote alone dots */
	                            fprintf (sfp, "..\r\n");
	                    	else      /* pass thru mode */
	                        	fprintf (sfp, "%s\r\n", buf);
	                    }
	                else{
	            		if ((msgfp = fopen (msg_file, "r"))){
	                        while (fgets (buf, sizeof (buf), msgfp)) {
	                            if (rflag == 0)
	                                sprintf (tmp_buf, "%s", buf);
	                            buf[strlen (buf) - 1] = 0;
	                            if (strcmp (buf, ".") == 0)   /* quote alone dots */
	                                fprintf (sfp, "..\r\n");
	                            else      /* pass thru mode */
	                                fprintf (sfp, "%s\r\n", buf);
	                        }
	                        fclose(msgfp);
						}	                       
	                }
	                fprintf (sfp, "\r\n");  /* terminate the text to tell MTA the end of messages */
	                rflag = 1;            
	            }
	            ret = chat (".\r\n");
	            if (ret)
	                continue;
	            ret = chat ("QUIT\r\n");
	            if (ret)
	                continue;
	            fclose (sfp);
	            fclose (rfp);
				break;
	        }
	    }
    }
out:
    if(send_ok){
    	if(use_smtp_server){
    		printf("Test mail was sent to SMTP Mail Server '%s'.\n",mail_addr);
			ret=ReadErrMsgFromFile(SMTP_MAIL_SERVER_OK, err_msg);
			if(!ret && strlen(err_msg))
				WriteSMTPStatus(err_msg,mail_addr);
			else
				WriteSMTPStatus("Test mail was sent to SMTP Mail Server '%s'.",mail_addr);	    		
    	}
    	else{
    		printf("Test mail was sent to '%s'.\n",mail_addr);
			ret=ReadErrMsgFromFile(SMTP_MAIL_ADDR_OK, err_msg);
			if(!ret && strlen(err_msg))
				WriteSMTPStatus(err_msg,mail_addr);
			else
				WriteSMTPStatus("Test mail was sent to '%s'.",mail_addr);	
    	}
    	return 0;
	}    	
	if(use_smtp_server){
		printf("Test mail failed to be sent to SMTP Mail Server '%s'.\n",mail_addr);
		ret=ReadErrMsgFromFile(SMTP_MAIL_SERVER_FAIL, err_msg);
		if(!ret && strlen(err_msg))
			WriteSMTPStatus(err_msg,mail_addr);
		else
			WriteSMTPStatus("Test mail failed to be sent to SMTP Mail Server '%s'.",mail_addr);			
	}
	else{
		printf("Test mail failed to be sent to '%s'.\n",mail_addr);
		ret=ReadErrMsgFromFile(SMTP_MAIL_ADDR_FAIL, err_msg);
		if(!ret && strlen(err_msg))
			WriteSMTPStatus(err_msg,mail_addr);
		else
			WriteSMTPStatus("Test mail failed to be sent to '%s'.",mail_addr);			
		
	}
	while (fgets (buf, sizeof (buf), stdin)) 
		continue;
    return -1;
}
